/*
 * $Id: AdjRowCol.c,v 1.14 1995/03/21 19:47:23 gerd Exp $
 */

/* Copyright (C) 1995 Gerd Rausch
   Please send comments and suggestions to:
   <gerd@alf.gun.de>

AdjRowCol is free software; you can redistribute it and/or modify it under
the terms of the GNU Library General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

AdjRowCol is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU Library General Public
License along with this software; see the file COPYING.LIB.  If not,
write to the Free Software Foundation, 675 Mass Ave, Cambridge, MA
02139, USA.  */

#include <stdlib.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "AdjRowColP.h"

#ifndef XtX
#define XtX(w)	   ((w)->core.x)
#endif

#ifndef XtY
#define XtY(w)	   ((w)->core.y)
#endif

#ifndef XtWidth
#define XtWidth(w) ((w)->core.width)
#endif

#ifndef XtHeight
#define XtHeight(w) ((w)->core.height)
#endif

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif

static void ClassInitialize();
static void Initialize();
static void Release();
static void ChangeManaged();
static XtGeometryResult GeometryManager();
static void Resize();
static void Redisplay();
static Boolean SetValues();

static void CheckCursorOrDragAction();
static void CursorNormalAction();
static void GripDragStartAction();
static void GripDragEndAction();

static void AdjustLayout();
static void Layout();
static void ClearGrips();
static void DrawGrips();
static void PreferredSize();

static XtResource resources[]= {
    {
	XtNthreeDimensional, XtCThreeDimensional,
	XtRBoolean, sizeof(Boolean),
	XtOffset(AdjRowColWidget, adjRowCol.threeD),
	XtRImmediate, (XtPointer)True
    },
    {
	XtNorientation, XtCOrientation,
	XtROrientation, sizeof(XtOrientation),
	XtOffset(AdjRowColWidget, adjRowCol.orientation),
	XtRImmediate, (XtPointer)XtorientHorizontal
    },
    {
	XtNcolumns, XtCColumns,
	XtRInt, sizeof(int),
	XtOffset(AdjRowColWidget, adjRowCol.columns),
	XtRImmediate, (XtPointer)0
    },
    { 
	XtNrowsAdjustable, XtCAdjustable,
	XtRBoolean, sizeof(Boolean),
	XtOffset(AdjRowColWidget, adjRowCol.adjustable[DirectionIdxRow]),
	XtRImmediate, (XtPointer)True
    },
    { 
	XtNcolsAdjustable, XtCAdjustable,
	XtRBoolean, sizeof(Boolean),
	XtOffset(AdjRowColWidget, adjRowCol.adjustable[DirectionIdxCol]),
	XtRImmediate, (XtPointer)True
    },
#ifndef USE_MOTIF
    {
	XtNforeground, XtCForeground,
	XtRPixel, sizeof(Pixel),
	XtOffset(AdjRowColWidget, adjRowCol.foreground),
	XtRString, (XtPointer)"Black"
    },
    {
	XtNshadowThickness, XtCShadowThickness,
	XtRDimension, sizeof(Dimension),
	XtOffset(AdjRowColWidget, adjRowCol.shadowThickness),
	XtRImmediate, (XtPointer)1
    },
    {
	XtNtopShadowColor, XtCTopShadowColor,
	XtRPixel, sizeof(Pixel),
	XtOffset(AdjRowColWidget, adjRowCol.topShadowColor),
	XtRString, (XtPointer)"gray80"
    },
    {
	XtNbottomShadowColor, XtCBottomShadowColor,
	XtRPixel, sizeof(Pixel),
	XtOffset(AdjRowColWidget, adjRowCol.bottomShadowColor),
	XtRString, (XtPointer)"gray60"
    },
#else /* USE_MOTIF */
    {
	XmNshadowThickness, XmCShadowThickness,
	XtRDimension, sizeof(Dimension),
	XtOffset(AdjRowColWidget, manager.shadow_thickness),
	XtRImmediate, (XtPointer)1
    },
#endif /* USE_MOTIF */
    {
	XtNgripThickness, XtCGripThickness,
	XtRDimension, sizeof(Dimension),
	XtOffset(AdjRowColWidget, adjRowCol.gripThickness),
	XtRImmediate, (XtPointer)2
    },
    {
	XtNgripBorder, XtCGripBorder,
	XtRDimension, sizeof(Dimension),
	XtOffset(AdjRowColWidget, adjRowCol.gripBorder),
	XtRImmediate, (XtPointer)2
    },
    {
	XtNgripHorizCursor, XtCCursor,
	XtRCursor, sizeof(Cursor),
	XtOffset(AdjRowColWidget, adjRowCol.gripHorizCursor),
	XtRString, (XtPointer)"sb_v_double_arrow"
    },
    {
	XtNgripVertCursor, XtCCursor,
	XtRCursor, sizeof(Cursor),
	XtOffset(AdjRowColWidget, adjRowCol.gripVertCursor),
	XtRString, (XtPointer)"sb_h_double_arrow"
    },
    {
	XtNgripCrossCursor, XtCCursor,
	XtRCursor, sizeof(Cursor),
	XtOffset(AdjRowColWidget, adjRowCol.gripCrossCursor),
	XtRString, (XtPointer)"fleur"
    },
};

static XtActionsRec actions[]={
    {"AdjCheckCursorOrDrag", CheckCursorOrDragAction},
    {"AdjCursorNormal", CursorNormalAction},
    {"AdjGripDragStart", GripDragStartAction},
    {"AdjGripDragEnd", GripDragEndAction},
};

static char translations[]="\
  <Leave>: AdjCursorNormal()\n\
  <Motion>: AdjCheckCursorOrDrag()\n\
  <Btn1Down>: AdjGripDragStart()\n\
  <Btn1Up>: AdjGripDragEnd()\n\
";

AdjRowColClassRec adjRowColClassRec = {
    {/* core_class members */
#ifdef USE_MOTIF
	(WidgetClass) &xmManagerClassRec,	/* superclass */
#else
	(WidgetClass) &compositeClassRec,	/* superclass */
#endif
	"AdjRowCol",				/* class_name */
	sizeof(AdjRowColRec),			/* widget_size */
	ClassInitialize,			/* class_initialize */
	NULL,					/* class_part_init */
	FALSE,					/* class_inited */
	Initialize,				/* initialize */
	NULL,					/* initialize_hook */
	XtInheritRealize,			/* realize */
	actions,				/* actions */
	XtNumber(actions),			/* num_actions */
	resources,				/* resources */
	XtNumber(resources),			/* num_resources */
	NULLQUARK,				/* xrm_class */
	TRUE,					/* compress_motion */
	TRUE,					/* compress_exposure */
	TRUE,					/* compress_enterleave */
	FALSE,					/* visible_interest */
	Release,				/* destroy */
	Resize,					/* resize */
	Redisplay,				/* expose */
	SetValues,				/* set_values */
	NULL,					/* set_values_hook */
	XtInheritSetValuesAlmost,		/* set_values_almost */
	NULL,					/* get_values_hook */
	NULL,					/* accept_focus */
	XtVersion,				/* version */
	NULL,					/* callback_private */
	translations,				/* tm_table */
	XtInheritQueryGeometry,			/* query_geometry */
	XtInheritDisplayAccelerator,		/* display_accelerator */
	NULL					/* extension */
    },{ /* composite_class members */
	GeometryManager,			/* geometry_manager */
	ChangeManaged,				/* change_managed */
	XtInheritInsertChild,			/* insert_child */
	XtInheritDeleteChild,			/* delete_child */
	NULL					/* extension */
    },{ /* adjRowCol_class members */
	0					/* empty */
    }
};

WidgetClass adjRowColWidgetClass=(WidgetClass)&adjRowColClassRec;

/* Table Access functions */
static void InitTable(t)
    TableInfo *t;
{
    int i,j;
    for(i=0;i<DimensionIdxLast;i++)
	for(j=0;j<DirectionIdxLast;j++){
	    t->dimensions[i][j].alloced=t->dimensions[i][j].n=0;
	    t->dimensions[i][j].values=NULL;
	}
    t->allocedManagedChildren=t->nManagedChildren=0;
    t->managedChildren=NULL;
    t->resortManagedChildren=False;
}
static void DisposeTable(t)
    TableInfo *t;
{
    int i,j;
    for(i=0;i<DimensionIdxLast;i++)
	for(j=0;j<DirectionIdxLast;j++)
	    if(t->dimensions[i][j].values)
		XtFree((char *)t->dimensions[i][j].values);
    if(t->managedChildren)
	XtFree((char *)t->managedChildren);
}
static void ClearTableManagedChildren(t)
    TableInfo *t;
{
    t->nManagedChildren=0;
    t->resortManagedChildren=False;
}
static void ClearTableRequired(t)
    TableInfo *t;
{
    int j;
    for(j=0;j<DirectionIdxLast;j++)
	t->dimensions[DimensionIdxRequired][j].n=0;
}
static void ClearTable(t)
    TableInfo *t;
{
    int i,j;
    for(i=0;i<DimensionIdxLast;i++)
	for(j=0;j<DirectionIdxLast;j++)
	    t->dimensions[i][j].n=0;
    t->nCols=t->nRows=0;
    ClearTableManagedChildren(t);
}
static void CopyTable(source,dest)
    TableInfo *source, *dest;
{
    int i,j;
    DisposeTable(dest);
    InitTable(dest);
    for(i=0;i<DimensionIdxLast;i++)
	for(j=0;j<DirectionIdxLast;j++){
	    DimensionArray *sourceP=&source->dimensions[i][j];
	    DimensionArray *destP=&dest->dimensions[i][j];
	    if(sourceP->n>0){
		destP->values=(Dimension *)XtMalloc(sizeof(Dimension)*sourceP->alloced);
		destP->alloced=sourceP->alloced;
		memcpy(destP->values,sourceP->values,sizeof(Dimension)*sourceP->n);
		destP->n=sourceP->n;
	    }
	}
    if(source->nManagedChildren>0){
	dest->managedChildren=(Widget *)XtMalloc(sizeof(Widget)*source->allocedManagedChildren);
	dest->allocedManagedChildren=source->allocedManagedChildren;
	memcpy(dest->managedChildren,source->managedChildren,sizeof(Widget)*source->nManagedChildren);
	dest->nManagedChildren=source->nManagedChildren;
	dest->resortManagedChildren=source->resortManagedChildren;
    }
}
static void CheckArray(ar,elemSize,alloced,n,idx)
    Dimension **ar;
    int elemSize,*alloced, *n, idx;
{
    int newSize;
    if(idx>=*alloced){
	newSize=idx+16;
	if(*ar)
	    *ar=(Dimension *)XtRealloc((char *)*ar,elemSize*newSize);
	else
	    *ar=(Dimension *)XtMalloc(elemSize*newSize);
	*alloced=newSize;
    }
    if(idx>=*n){
	memset(*ar+*n,0,elemSize*(idx-*n+1));
	*n=idx+1;
    }
}
static int GetTableDimensionN(t,dimension,direction)
    TableInfo *t;
    DimensionIdx dimension;
    DirectionIdx direction;
{
    return t->dimensions[dimension][direction].n;
}
static Dimension GetTableDimensionValue(t,dimension,direction,col)
    TableInfo *t;
    DimensionIdx dimension;
    DirectionIdx direction;
    int col;
{
    DimensionArray *dp=&t->dimensions[dimension][direction];
    CheckArray(&dp->values,sizeof(Dimension),&dp->alloced,&dp->n,col);
    return dp->values[col];
}
static void SetTableDimensionValue(t,dimension,direction,col,value)
    TableInfo *t;
    DimensionIdx dimension;
    DirectionIdx direction;
    int col;
    Dimension value;
{
    DimensionArray *dp=&t->dimensions[dimension][direction];
    CheckArray(&dp->values,sizeof(Dimension),&dp->alloced,&dp->n,col);
    dp->values[col]=value;
}
/* some shortcut functions */
static int GetTableCols(t)
    TableInfo *t;
{
    return GetTableDimensionN(t,DimensionIdxPreferred,DirectionIdxCol);
}
static int GetTableRows(t)
    TableInfo *t;
{
    return GetTableDimensionN(t,DimensionIdxPreferred,DirectionIdxRow);
}
static int GetTableWidth(t,col)
    TableInfo *t;
    int col;
{
    return GetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxCol,col);
}
static int GetTableHeight(t,col)
    TableInfo *t;
    int col;
{
    return GetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxRow,col);
}
static void SetTableWidth(t,col,value)
    TableInfo *t;
    int col;
    Dimension value;
{
    SetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxCol,col,value);
}
static void SetTableHeight(t,row,value)
    TableInfo *t;
    int row;
    Dimension value;
{
    SetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxRow,row,value);
}
static Dimension BetweenCellSpace(aw,direction)
    AdjRowColWidget aw;
    DirectionIdx direction;
{
#ifdef USE_MOTIF
    Dimension shadowThickness=aw->manager.shadow_thickness;
#else
    Dimension shadowThickness=aw->adjRowCol.shadowThickness;
#endif
    if(aw->adjRowCol.adjustable[direction])
	if(aw->adjRowCol.threeD)
	    return 2*shadowThickness+2*aw->adjRowCol.gripBorder;
	else
	    return aw->adjRowCol.gripThickness+2*aw->adjRowCol.gripBorder;
    else
	return aw->adjRowCol.gripBorder;
}
static Dimension GetTableCellWidth(aw,t,col)
    AdjRowColWidget aw;
    TableInfo *t;
    int col;
{
    Dimension width=GetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxCol,col);
    return width+BetweenCellSpace(aw,DirectionIdxCol);
}
static Dimension GetTableCellHeight(aw,t,row)
    AdjRowColWidget aw;
    TableInfo *t;
    int row;
{
    Dimension height=GetTableDimensionValue(t,DimensionIdxPreferred,DirectionIdxRow,row);
    return height+BetweenCellSpace(aw,DirectionIdxRow);
}
static int compareWidget(a,b)
    Widget *a,*b;
{
    return *b-*a;
}
static Boolean HasTableManagedChild(t,w)
    TableInfo *t;
    Widget w;
{
    Widget *wp;
    if(t->nManagedChildren<=0)
	return False;
    if(t->resortManagedChildren){
	qsort(t->managedChildren,t->nManagedChildren,sizeof(Widget),compareWidget);
	t->resortManagedChildren=False;
    }
    return !!bsearch(&w,t->managedChildren,t->nManagedChildren,sizeof(Widget),compareWidget);
}
static void AddTableManagedChild(t,w)
    TableInfo *t;
    Widget w;
{
    int idx=t->nManagedChildren;
    CheckArray(&t->managedChildren,sizeof(Widget),&t->allocedManagedChildren,&t->nManagedChildren,idx);
    t->managedChildren[idx]=w;
    t->resortManagedChildren=True;
}

static int CalculateCursorState(aw)
    AdjRowColWidget aw;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    int direction,state=0;
    for(direction=0;direction<DirectionIdxLast;direction++)
	state=state*2+(ap->dragInfo[direction].idx>=0);
    return state;
}
static void CheckDragGripPosition(aw,x,y)
    AdjRowColWidget aw;
    int x,y;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    int oldCursorState=CalculateCursorState(aw),newCursorState;
    int direction,coord,iCoord,idx;
    DimensionIdx dimension=DimensionIdxPreferred;
    for(direction=0;direction<DirectionIdxLast;direction++){
	coord=direction==DirectionIdxCol?x:y;
	ap->dragInfo[direction].idx=-1;
	if(ap->adjustable[direction] && coord>=0){
	    iCoord=0;
	    for(idx=0;idx<GetTableDimensionN(&ap->currentTable,dimension,direction)-1;idx++){
		iCoord+=GetTableDimensionValue(&ap->currentTable,dimension,direction,idx);
		if(coord>=iCoord && coord<iCoord+BetweenCellSpace(aw,direction)){
		    ap->dragInfo[direction].idx=idx;
		    ap->dragInfo[direction].start=coord;
		    ap->dragInfo[direction].last=-1;
		    break;
		}
		iCoord+=BetweenCellSpace(aw,direction);
	    }
	}
    }
    newCursorState=CalculateCursorState(aw);
    if(newCursorState!=oldCursorState){
	if(ap->dragInfo[DirectionIdxCol].idx>=0 &&
	   ap->dragInfo[DirectionIdxRow].idx>=0)
	    ap->draggingCursor=ap->gripCrossCursor;
	else
	    if(ap->dragInfo[DirectionIdxCol].idx>=0)
		ap->draggingCursor=ap->gripVertCursor;
	    else
		if(ap->dragInfo[DirectionIdxRow].idx>=0)
		    ap->draggingCursor=ap->gripHorizCursor;
		else
		    ap->draggingCursor=None;
	XDefineCursor(XtDisplay((Widget)aw),XtWindow((Widget)aw),ap->draggingCursor);
    }
}

static void DrawRubberBand(aw)
    AdjRowColWidget aw;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    Position x,y;
    if(ap->dragInfo[DirectionIdxCol].idx>=0){
	x=ap->dragInfo[DirectionIdxCol].last;
	XFillRectangle(XtDisplay((Widget)aw),XtWindow((Widget)aw),ap->rubberGC,
		       x+ap->gripBorder,0,
		       ap->gripThickness,XtHeight((Widget)aw));
    }
    if(ap->dragInfo[DirectionIdxRow].idx>=0){
	y=ap->dragInfo[DirectionIdxRow].last;
	XFillRectangle(XtDisplay((Widget)aw),XtWindow((Widget)aw),ap->rubberGC,
		       0,y+ap->gripBorder,
		       XtWidth((Widget)aw),ap->gripThickness);
    }
}

static void CheckCursorOrDragAction(w,event,params,num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    AdjRowColWidget aw;
    AdjRowColPart *ap;
    if(event->type!=MotionNotify || !XtIsSubclass(w,adjRowColWidgetClass))
	return;
    aw=(AdjRowColWidget)w;
    ap=&aw->adjRowCol;
    if(ap->dragging){
	DrawRubberBand(aw); /* clear it */
	ap->dragInfo[DirectionIdxCol].last=event->xmotion.x;
	ap->dragInfo[DirectionIdxRow].last=event->xmotion.y;
	DrawRubberBand(aw); /* draw a new one */
    }else
	CheckDragGripPosition(aw,event->xmotion.x,event->xmotion.y);
}
static void CursorNormalAction(w,event,params,num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    AdjRowColWidget aw;
    AdjRowColPart *ap;
    if(!XtIsSubclass(w,adjRowColWidgetClass))
	return;
    aw=(AdjRowColWidget)w;
    ap=&aw->adjRowCol;
    if(!aw->adjRowCol.dragging)
	CheckDragGripPosition(aw,-1,-1);
}
static void GripDragStartAction(w,event,params,num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    AdjRowColWidget aw;
    AdjRowColPart *ap;
    if(event->type!=ButtonPress || !XtIsSubclass(w,adjRowColWidgetClass))
	return;
    aw=(AdjRowColWidget)w;
    ap=&aw->adjRowCol;
    CheckDragGripPosition(aw,event->xbutton.x,event->xbutton.y);
    ap->dragging=
	ap->dragInfo[DirectionIdxCol].idx>=0 ||
	ap->dragInfo[DirectionIdxRow].idx>=0;
    if(ap->dragging){
	XGrabPointer(XtDisplay(w),XtWindow(w),False,ButtonReleaseMask|PointerMotionMask,GrabModeAsync,GrabModeAsync,None,aw->adjRowCol.draggingCursor,CurrentTime);
	ap->dragInfo[DirectionIdxCol].last=event->xbutton.x;
	ap->dragInfo[DirectionIdxRow].last=event->xbutton.y;
	DrawRubberBand(aw);
    }
}
static void GripDragEndAction(w,event,params,num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    AdjRowColWidget aw;
    AdjRowColPart *ap;
    Position newPos;
    Dimension width,height;
    XtGeometryResult result;
    Dimension replyWidth, replyHeight;
    DimensionIdx dimension=DimensionIdxPreferred;
    if(event->type!=ButtonRelease || !XtIsSubclass(w,adjRowColWidgetClass))
	return;
    aw=(AdjRowColWidget)w;
    ap=&aw->adjRowCol;
    if(ap->dragging){
	int direction,newSize,maxSize;
	DrawRubberBand(aw); /* clear rubberband */
	XUngrabPointer(XtDisplay(w),CurrentTime);
	ap->dragging=False;
	for(direction=0;direction<DirectionIdxLast;direction++){
	    if(ap->dragInfo[direction].idx>=0){
		if(direction==DirectionIdxCol){
		    newPos=event->xbutton.x;
		    maxSize=XtWidth(w);
		}else{
		    newPos=event->xbutton.y;
		    maxSize=XtHeight(w);
		}
		newSize=GetTableDimensionValue(&ap->intendedTable,dimension,direction,ap->dragInfo[direction].idx)+newPos-ap->dragInfo[direction].start;
		if(newSize<=0)
		    newSize=1;
		if(newSize>maxSize)
		    newSize=maxSize;
		SetTableDimensionValue(&ap->intendedTable,dimension,direction,ap->dragInfo[direction].idx,newSize);
		SetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,direction,ap->dragInfo[direction].idx,newSize);
	    }
	}
	PreferredSize(aw,&width,&height);
	result = XtMakeResizeRequest((Widget) aw, width, height,
				     &replyWidth,
				     &replyHeight);

	if (result == XtGeometryAlmost) 
	    XtMakeResizeRequest((Widget) aw,
				replyWidth, replyHeight, 
				NULL, NULL);
	if(result!=XtGeometryYes)
	    AdjustLayout(aw,True);
	
	Layout(aw);
    }
}

static void DrawGrips(aw,flatGC,topGC,bottomGC)
    AdjRowColWidget aw;
    GC flatGC,topGC,bottomGC;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    Widget w=(Widget)aw;
    int idx,coord,direction;
    DimensionIdx dimension=DimensionIdxPreferred;
#ifdef USE_MOTIF
    Dimension shadowThickness=aw->manager.shadow_thickness;
#else
    Dimension shadowThickness=aw->adjRowCol.shadowThickness;
#endif
    if(!XtIsRealized((Widget)aw))
	return;
    for(direction=0;direction<DirectionIdxLast;direction++){
	if(ap->adjustable[direction]){
	    coord=0;
	    for(idx=0;idx<GetTableDimensionN(&ap->currentTable,dimension,direction)-1;idx++){
		coord+=GetTableDimensionValue(&ap->currentTable,dimension,direction,idx);
		if(ap->threeD){
		    if(direction==DirectionIdxCol){
			XFillRectangle(XtDisplay(w),XtWindow(w),topGC,
				       coord+ap->gripBorder,0,
				       shadowThickness,XtHeight(w));
			XFillRectangle(XtDisplay(w),XtWindow(w),bottomGC,
				       coord+ap->gripBorder+shadowThickness,0,
				       shadowThickness,XtHeight(w));
		    }else{
			XFillRectangle(XtDisplay(w),XtWindow(w),topGC,
				       0,coord+ap->gripBorder,
				       XtWidth(w),shadowThickness);
			XFillRectangle(XtDisplay(w),XtWindow(w),bottomGC,
				       0,coord+ap->gripBorder+shadowThickness,
				       XtWidth(w),shadowThickness);
		    }
		}else
		    if(direction==DirectionIdxCol)
			XFillRectangle(XtDisplay(w),XtWindow(w),flatGC,
				       coord+ap->gripBorder,0,
				       ap->gripThickness,XtHeight(w));
		    else
			XFillRectangle(XtDisplay(w),XtWindow(w),flatGC,
				       0,coord+ap->gripBorder,
				       XtWidth(w),ap->gripThickness);
		    
		coord+=BetweenCellSpace(aw,direction);
	    }
	}
    }
}

static void ClearGrips(aw)
    AdjRowColWidget aw;
{
    if(aw->adjRowCol.gripsDisplayed){
	DrawGrips(aw,aw->adjRowCol.clearGC,aw->adjRowCol.clearGC,aw->adjRowCol.clearGC);
	aw->adjRowCol.gripsDisplayed=False;
    }
}    
static void DisplayGrips(aw)
    AdjRowColWidget aw;
{
    DrawGrips(aw,aw->adjRowCol.gripGC,aw->adjRowCol.topGC,aw->adjRowCol.bottomGC);
    aw->adjRowCol.gripsDisplayed=True;
}

static void NextChild(aw,col,row,x,y)
    AdjRowColWidget aw;
    int *col, *row;
    Position *x, *y;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    if(ap->orientation==XtorientVertical){
	if(y)
	    *y+=GetTableCellHeight(aw,&ap->intendedTable,*row);
	(*row)++;
	if(ap->columns>0 && *row>=ap->columns){
	    *row=0;
	    if(y)
		*y=0;
	    if(x)
		*x+=GetTableCellWidth(aw,&ap->intendedTable,*col);
	    (*col)++;
	}
    }else{
	if(x)
	    *x+=GetTableCellWidth(aw,&ap->intendedTable,*col);
	(*col)++;
	if(ap->columns>0 && *col>=ap->columns){
	    *col=0;
	    if(x)
		*x=0;
	    if(y)
		*y+=GetTableCellHeight(aw,&ap->intendedTable,*row);
	    (*row)++;
	}
    }
}

static void Layout(aw)
    AdjRowColWidget aw;
{
    Widget *childP;
    Position x=0,y=0;
    int row=0,col=0;
    Boolean gripsCleared=False;
    AdjRowColPart *ap=&aw->adjRowCol;
    ClearTableManagedChildren(&ap->intendedTable);
    ClearTableRequired(&ap->intendedTable);
    for(childP=aw->composite.children;
	childP<aw->composite.children+aw->composite.num_children;
	childP++){
	Widget child=*childP;
	if(child->core.managed){
	    AddTableManagedChild(&ap->intendedTable,child);
	    if(x!=XtX(child) || y!=XtY(child) ||
		GetTableWidth(&ap->intendedTable,col)!=XtWidth(child) ||
		GetTableHeight(&ap->intendedTable,row)!=XtHeight(child)){
		if(!gripsCleared){
		    ClearGrips(aw);
		    gripsCleared=True;
		}
		XtConfigureWidget(child,x,y,
				  GetTableWidth(&ap->intendedTable,col),
				  GetTableHeight(&ap->intendedTable,row),0);
	    }
	    NextChild(aw,&col,&row,&x,&y);
	}
    }
    CopyTable(&ap->intendedTable,&ap->currentTable);
    CheckDragGripPosition(aw,-1,-1);
}

static void RestoreTables(aw)
    AdjRowColWidget aw;
{
    Widget *childP;
    XtWidgetGeometry reply;
    Dimension width, height;
    int col=0,row=0,newSize;
    AdjRowColPart *ap=&aw->adjRowCol;

    ClearTable(&ap->intendedTable);

    for(childP=aw->composite.children;
	childP<aw->composite.children+aw->composite.num_children;
	childP++){
	Widget child=*childP;
	if(child->core.managed){
	    AddTableManagedChild(&ap->intendedTable,child);
	    XtQueryGeometry(child,NULL,&reply);
	    width=MAX(XtWidth(child),reply.width);
	    height=MAX(XtHeight(child),reply.height);
	    if(col>=GetTableCols(&ap->intendedTable) ||
	       width>GetTableWidth(&ap->intendedTable,col))
		SetTableWidth(&ap->intendedTable,col,width);
	    if(row>=GetTableRows(&ap->intendedTable) ||
	       height>GetTableHeight(&ap->intendedTable,row))
		SetTableHeight(&ap->intendedTable,row,height);
	    if(!HasTableManagedChild(&ap->currentTable,child)){
		if(width>GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,DirectionIdxCol,col))
		    SetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,DirectionIdxCol,col,width);
		if(height>GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,DirectionIdxRow,row))
		    SetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,DirectionIdxRow,row,height);
	    }
	    NextChild(aw,&col,&row,NULL,NULL);
	}
    }
    CopyTable(&ap->intendedTable,&ap->currentTable);
}

static void PreferredSize(aw,w,h)
    AdjRowColWidget aw;
    Dimension *w, *h;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    Dimension size;
    DimensionIdx dimension=DimensionIdxPreferred;
    int direction,idx;
    for(direction=0;direction<DirectionIdxLast;direction++){
	size=0;
	for(idx=0;idx<GetTableDimensionN(&ap->intendedTable,dimension,direction);idx++)
	    size+=GetTableDimensionValue(&ap->intendedTable,dimension,direction,idx);
	if(ap->adjustable[direction] && GetTableDimensionN(&ap->intendedTable,dimension,direction)>1)
	    size+=(GetTableDimensionN(&ap->intendedTable,dimension,direction)-1)*BetweenCellSpace(aw,direction);
	if(direction==DirectionIdxCol)
	    *w=size;
	else
	    *h=size;
    }
}    

static void SubtractGrips(aw,w,h)
    AdjRowColWidget aw;
    Dimension *w, *h;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    int width,height;
    width=*w;
    height=*h;

    if(ap->adjustable[DirectionIdxCol] && GetTableCols(&ap->intendedTable)>1)
	width-=(GetTableCols(&ap->intendedTable)-1)*BetweenCellSpace(aw,DirectionIdxCol);
    if(ap->adjustable[DirectionIdxRow] && GetTableRows(&ap->intendedTable)>1)
	height-=(GetTableRows(&ap->intendedTable)-1)*BetweenCellSpace(aw,DirectionIdxRow);
    if(width<=0)
	width=1;
    if(height<=0)
	height=1;
    *w=width;
    *h=height;
}

static void ResizeCellsDirection(aw,direction,useRequired,sizeNum,sizeDen)
    AdjRowColWidget aw;
    DirectionIdx direction;
    Boolean useRequired;
    int sizeNum,sizeDen;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    int idx;
    int required,requireFields=0,requirePointsLeft=0,wouldHaveBeen,newSize,notRequiredFields;
    if(useRequired){
	/* useRequire is only set for increasing sizes */
	for(idx=0;idx<GetTableDimensionN(&ap->intendedTable,DimensionIdxPreferred,direction);idx++){
	    required=GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,direction,idx);
	    if(required>0){
		wouldHaveBeen=GetTableDimensionValue(&ap->intendedTable,DimensionIdxPreferred,direction,idx)*sizeNum/sizeDen;
		if(wouldHaveBeen<=0)
		    wouldHaveBeen=1;
		requireFields++;
		requirePointsLeft+=wouldHaveBeen-required;
	    }
	}
    }
    notRequiredFields=GetTableDimensionN(&ap->intendedTable,DimensionIdxPreferred,direction)-requireFields;
    for(idx=0;idx<GetTableDimensionN(&ap->intendedTable,DimensionIdxPreferred,direction);idx++){
	if(useRequired){
	    required=GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,direction,idx);
	    if(required>0){
		SetTableDimensionValue(&ap->intendedTable,DimensionIdxPreferred,direction,idx,required);
		continue;
	    }
	}
	newSize=GetTableDimensionValue(&ap->intendedTable,DimensionIdxPreferred,direction,idx)*sizeNum/sizeDen+requirePointsLeft/notRequiredFields;
	if(newSize<=0)
	    newSize=1;
	SetTableDimensionValue(&ap->intendedTable,DimensionIdxPreferred,direction,idx,newSize);
    }
}

/* Adjust the layout to fit the current size
   We try to give new cells at least their requested
   value and resize the others instead
   */
static void AdjustLayout(aw,forceRequired)
    AdjRowColWidget aw;
    Boolean forceRequired;
{
    AdjRowColPart *ap=&aw->adjRowCol;
    Dimension newWidth,newHeight,newSize,
	preferredWidth,preferredHeight,preferredSize;
    int resizeableSize,decreaseSize;
    DimensionIdx dimension=DimensionIdxPreferred;
    int direction,idx;
    Widget *childP;

    newWidth=XtWidth((Widget)aw);
    newHeight=XtHeight((Widget)aw);
    SubtractGrips(aw,&newWidth,&newHeight);

    PreferredSize(aw,&preferredWidth,&preferredHeight);
    SubtractGrips(aw,&preferredWidth,&preferredHeight);

    for(direction=0;direction<DirectionIdxLast;direction++){
	if(direction==DirectionIdxCol){
	    newSize=newWidth;
	    preferredSize=preferredWidth;
	}else{
	    newSize=newHeight;
	    preferredSize=preferredHeight;
	}
	if(newSize<preferredSize){
	    /* calculate by how much we can resize our cells */
	    resizeableSize=0;
	    for(idx=0;idx<GetTableDimensionN(&ap->intendedTable,dimension,direction);idx++){
		int current,required;
		current=GetTableDimensionValue(&ap->intendedTable,dimension,direction,idx);
		required=GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,direction,idx);
		if(current>required)
		    resizeableSize+=current-required;
	    }
	    if(resizeableSize>=preferredSize-newSize){
		/* forced resizing X possible */
		decreaseSize=preferredSize-newSize;
		for(idx=0;idx<GetTableDimensionN(&ap->intendedTable,dimension,direction);idx++){
		    int current,required,possible;
		    current=GetTableDimensionValue(&ap->intendedTable,dimension,direction,idx);
		    required=GetTableDimensionValue(&ap->intendedTable,DimensionIdxRequired,direction,idx);
		    if(current>required){
			possible=current-required;
			current-=decreaseSize*possible/resizeableSize;
			if(current<=0)
			    current=1;
			SetTableDimensionValue(&ap->intendedTable,dimension,direction,idx,current);
		    }
		}
	    }else
		ResizeCellsDirection(aw,direction,False,newSize,preferredSize);
	}else
	    ResizeCellsDirection(aw,direction,forceRequired,newSize,preferredSize);
    }
}

static void GetGCs(aw)
    AdjRowColWidget aw;
{
    XtGCMask valuemask;
    XGCValues values;

#ifdef USE_MOTIF
    values.foreground=aw->manager.foreground;
#else
    values.foreground=aw->adjRowCol.foreground;
#endif
    values.background=aw->core.background_pixel;
    valuemask=GCForeground|GCBackground;
    aw->adjRowCol.gripGC=XtGetGC((Widget)aw,valuemask,&values);

#ifdef USE_MOTIF
    values.foreground=aw->manager.top_shadow_color;
#else
    values.foreground=aw->adjRowCol.topShadowColor;
#endif
    values.background=aw->core.background_pixel;
    valuemask=GCForeground|GCBackground;
    aw->adjRowCol.topGC=XtGetGC((Widget)aw,valuemask,&values);

#ifdef USE_MOTIF
    values.foreground=aw->manager.bottom_shadow_color;
#else
    values.foreground=aw->adjRowCol.bottomShadowColor;
#endif
    values.background=aw->core.background_pixel;
    valuemask=GCForeground|GCBackground;
    aw->adjRowCol.bottomGC=XtGetGC((Widget)aw,valuemask,&values);

    values.foreground=aw->core.background_pixel;
    values.background=aw->core.background_pixel;
    valuemask=GCForeground|GCBackground;
    aw->adjRowCol.clearGC=XtGetGC((Widget)aw,valuemask,&values);

    values.function=GXxor;
    values.foreground=~0;
    values.subwindow_mode=IncludeInferiors;
    valuemask=GCFunction|GCForeground|GCSubwindowMode;
    aw->adjRowCol.rubberGC=XtGetGC((Widget)aw,valuemask,&values);
}

static void ReleaseGCs(aw)
    AdjRowColWidget aw;
{
    XtReleaseGC( (Widget)aw, aw->adjRowCol.gripGC );
    XtReleaseGC( (Widget)aw, aw->adjRowCol.topGC );
    XtReleaseGC( (Widget)aw, aw->adjRowCol.bottomGC );
    XtReleaseGC( (Widget)aw, aw->adjRowCol.clearGC );
    XtReleaseGC( (Widget)aw, aw->adjRowCol.rubberGC );
}

static void
ClassInitialize()
{
    XtAddConverter( XtRString, XtROrientation, XmuCvtStringToOrientation,
		    (XtConvertArgList)NULL, (Cardinal)0 );
}

static void Initialize(req, new, args, num_args)
    Widget req, new;
    ArgList args;
    Cardinal *num_args;
{
    AdjRowColWidget aw=(AdjRowColWidget)new;
    int direction;

    aw->adjRowCol.gripsDisplayed=False;
    InitTable(&aw->adjRowCol.currentTable);
    InitTable(&aw->adjRowCol.intendedTable);
    aw->adjRowCol.dragging=False;
    for(direction=0;direction<DirectionIdxLast;direction++)
	aw->adjRowCol.dragInfo[direction].idx=-1;

    if (XtWidth(req) <= 0) 
	XtWidth(new) = 16;
    
    if (XtHeight(req) <= 0) 
	XtHeight(new) = 16;

    GetGCs(aw);
}

static void 
Release(w)
    Widget w;
{
    AdjRowColWidget aw=(AdjRowColWidget)w;
    ReleaseGCs(aw);
    DisposeTable(&aw->adjRowCol.currentTable);
    DisposeTable(&aw->adjRowCol.intendedTable);
}

static void ChangeManaged(w)
   Widget w;
{
    AdjRowColWidget aw=(AdjRowColWidget)w;
    Dimension width,height;
    XtGeometryResult result;
    Dimension replyWidth, replyHeight;
    Widget *childP;

    RestoreTables(aw);
    PreferredSize(aw,&width,&height);
    result=XtMakeResizeRequest((Widget)aw,width,height,&replyWidth,&replyHeight);
    if(result==XtGeometryAlmost)
	XtMakeResizeRequest((Widget)aw,replyWidth,replyHeight,NULL,NULL);
    if(result!=XtGeometryYes)
	AdjustLayout(aw,False);

    Layout(aw);
}

static XtGeometryResult GeometryManager(w, request, reply)
    Widget w;
    XtWidgetGeometry *request, *reply;
{
    return XtGeometryNo;
}

static void Resize(w)
    Widget w;
{
    AdjRowColWidget aw=(AdjRowColWidget)w;

    AdjustLayout(aw,False);

    Layout(aw);
}

static void
Redisplay(w, event, region)
    Widget w;
    XEvent * event;
    Region region;
{
    AdjRowColWidget aw=(AdjRowColWidget)w;
    DisplayGrips(aw);
}

static Boolean SetValues(old,request,new,args,num_args)
    Widget old,request,new;
    ArgList args;
    Cardinal *num_args;
{
    AdjRowColWidget old_aw=(AdjRowColWidget)old;
    AdjRowColWidget new_aw=(AdjRowColWidget)new;
    Boolean redisplay;
#ifdef USE_MOTIF
    if(old_aw->manager.foreground!=new_aw->manager.foreground ||
       old_aw->manager.top_shadow_color!=new_aw->manager.top_shadow_color ||
       old_aw->manager.bottom_shadow_color!=new_aw->manager.bottom_shadow_color){
#else
    if(old_aw->adjRowCol.foreground!=new_aw->adjRowCol.foreground ||
       old_aw->adjRowCol.topShadowColor!=new_aw->adjRowCol.topShadowColor ||
       old_aw->adjRowCol.bottomShadowColor!=new_aw->adjRowCol.bottomShadowColor){
#endif
	ReleaseGCs(old_aw);
	GetGCs(new_aw);
	redisplay=True;
    }
    if(old_aw->adjRowCol.gripHorizCursor!=new_aw->adjRowCol.gripHorizCursor ||
       old_aw->adjRowCol.gripVertCursor!=new_aw->adjRowCol.gripVertCursor ||
       old_aw->adjRowCol.gripCrossCursor!=new_aw->adjRowCol.gripCrossCursor){
	CheckDragGripPosition(old_aw,-1,-1);
    }
    if(old_aw->adjRowCol.orientation!=new_aw->adjRowCol.orientation ||
       old_aw->adjRowCol.columns!=new_aw->adjRowCol.columns ||
       old_aw->adjRowCol.gripThickness!=new_aw->adjRowCol.gripThickness ||
       old_aw->adjRowCol.gripBorder!=new_aw->adjRowCol.gripBorder ||
#ifdef USE_MOTIF
       old_aw->manager.shadow_thickness!=new_aw->manager.shadow_thickness
#else
       old_aw->adjRowCol.shadowThickness!=new_aw->adjRowCol.shadowThickness
#endif
	){
	AdjustLayout(new_aw,False);
	Layout(new_aw);
	redisplay=True;
    }
    return redisplay;
}
