/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "earthpos.h"

void	sun_relative_position(epi,epo,lat_ret, lon_ret)
     t_earth_pos_input *epi;
     t_earth_pos_output	*epo;
     double	*lat_ret;           /* (return) latitude  */
     double	*lon_ret;           /* (return) longitude */
{
  double lat, lon;

  lat = epo->sun_lat + epi->view_lat;
  lon = epo->sun_lon + epi->view_lon;

  /* sanity check */
  assert((lat >= -180) && (lat <= 180));
  assert((lon >= -360) && (lon <= 360));

  if (lat > 90)
  {
    lat  = 180 - lat;
    lon += 180;
  }
  else if (lat < -90)
  {
    lat  = -180 - lat;
    lon += 180;
  }

  if (lon > 180)
  {
    do
      lon -= 360;
    while (lon > 180);
  }
  else if (lon < -180)
  {
    do
      lon += 360;
    while (lon < -180);
  }

  *lat_ret = lat;
  *lon_ret = lon;
}

void simple_orbit(epi,ssue, lat, lon)
     t_earth_pos_input *epi;
     time_t  ssue;              /* seconds since unix epoch */
     double *lat;               /* (return) latitude        */
     double *lon;               /* (return) longitude       */
{
  double x, y, z;
  double a, c, s;
  double t1, t2;

  /* start at 0 N 0 E */
  x = 0;
  y = 0;
  z = 1;

  /* rotate in about y axis (from z towards x) according to the number
   * of orbits we've completed
   */
  a  = ((double) ssue / epi->period) * (2*M_PI);
  c  = cos(a);
  s  = sin(a);
  t1 = c*z - s*x;
  t2 = s*z + c*x;
  z  = t1;
  x  = t2;

  /* rotate about z axis (from x towards y) according to the
   * inclination of the orbit
   */
  a  = epi->inclin * (M_PI/180);
  c  = cos(a);
  s  = sin(a);
  t1 = c*x - s*y;
  t2 = s*x + c*y;
  x  = t1;
  y  = t2;

  /* rotate about y axis (from x towards z) according to the number of
   * rotations the earth has made
   */
  a  = ((double) ssue / EarthPeriod) * (2*M_PI);
  c  = cos(a);
  s  = sin(a);
  t1 = c*x - s*z;
  t2 = s*x + c*z;
  x  = t1;
  z  = t2;

  *lat = asin(y) * (180/M_PI);
  *lon = atan2(x, z) * (180/M_PI);
}

void			compute_positions(epi,epo)
t_earth_pos_input	*epi;
t_earth_pos_output	*epo;
{
  /* determine position on earth's surface where sun is directly
   * overhead
   */
  if (epi->compute_sun_pos)
    sun_position(epi->fixed_time, &(epo->sun_lat), &(epo->sun_lon));

  /* determine viewing position
   */
  if (epi->view_pos_type == ViewPosTypeFixed)
    {
      epo->view_lat = epi->view_lat;
      epo->view_lon = epi->view_lon;
    }
  else
    if (epi->view_pos_type == ViewPosTypeSunRel)
      {
	sun_relative_position(epi,epo,&(epo->view_lat), &(epo->view_lon));
      }
    else if (epi->view_pos_type == ViewPosTypeOrbit)
      {
	simple_orbit(epi,epi->fixed_time, &(epo->view_lat), &(epo->view_lon));
      }
}
