/*
** MamLoad.c
**
**	This module contains the implementation of MamLoadMessage.
**
**	*WARNING!*
**	THE PARSING IS BASED ON ASSUMPTION THAT THE INPUT IS CORRECTLY
**	FORMED. THE LOAD WILL NOT CRASH ON INCORRECT INPUT, BUT THE
**	INFORMATION LOADED INTO MEMORY MAY NOT BE VERY USEFUL.
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#define UNFOLD_RFC822	1 /* Parse included messages also */

#include "common.h"
#include "Mam.h"
#include "MamCommon.h"
#include "message.h"

#undef tables_h
#define ENUM_TABLE_GENERATION
#include "tables.h"

typedef void (*HeaderHandler)(MsgBody, char *, char *, char *, void *);
typedef char *(*ParseParameter)(MsgBody, char *, char *, char *);

typedef struct HeaderControl
    {
	char *name;		/* Header field name in lower case */
	HeaderHandler handler;	/* Handler function for the value */
	/*
	** The location of the value:
	*/
#	define LOC_MESSAGE 0	/* Not stored or global to Message */
#	define LOC_BODY 1	/* Current MsgBody */
#	define LOC_HEADER 2	/* Current MsgHeader */
	int location;
	int offset;		/* ..to value in selected structure */
    } HeaderControl;

typedef struct ParameterControl
    {
	char *name;
	ParseParameter parse;
    } ParameterControl;

typedef struct Keyword
    {
	char *name;
	int value;
    } Keyword;

static Keyword state_token[] = 
    {
	{"new",		Msg_New},
	{"read",	Msg_Read},
	{"draf",	Msg_Draft},
	{"sent",	Msg_Sent},
	{"modified",	Msg_Modified},
	{"replied-to",	Msg_Replied_To},
	{"is-archived",	Msg_Is_Archived},
	{"awaiting_reply", Msg_Awaiting_Reply},
	{"tagged",	Msg_Tagged},
	{NULL,		Msg_No_State},
    };

static int LookKeyword(char *token, Keyword *key)
    {
	while (key->name && strcmp(key->name, token))
		++key;
	return key->value;
    }

static int StringToEnum(char *token, char **list)
    {
	int i = 0;
	char *s;

	for (i = 0; (s = *list++) != NULL; i++)
		if (strcmp(s, token) == 0)
			return i;
	return 0;
    }

static char *StringValue(char *v, char *e)
    {
	register int len;
	char *s = NULL;

	/* Eliminate White space around the value */

	while (v < e && (*v == ' ' || *v == '\t'))
		v += 1;
	while (e > v && (e[-1] == ' ' || e[-1] == '\t'))
		e -= 1;
	if ((len = e - v) > 0)
	    {
		s = MamMalloc(len+1);
		memcpy(s, v, len);
		s[len] = '\0';
	    }
	return s;
    }

static Message current_message = NULL;	/* Message Structure being loaded */
static char *current_boundary = NULL;	/* Lastest Boundary string */
static char *current_buffer = NULL;	/* Current Buffer start */

/*
** StringList
**	In various places the parser must collect a set of strings, the
**	number of which is not known in advance. StringList is a temporary
**	holding place for the strings while they are being collected.
**	StringList is usually permanently allocated, and the list is just
**	grown (see GrowList) if the limit is reached.
*/
typedef struct StringList
    {
	char **list;	/* Current List In use */
	int size;	/* Number of slots in list */
	int count;	/* Number of slots used */
    } StringList;

/*
** GrowList
**	Change the buffer allocation in StringList (lst). The new
**	allocation will have at least 'min' entries.
*/
static void GrowList(StringList *lst, int min)
    {
	int oldsize = lst->size;
	char **new;

	if (lst->size)
		lst->size *= 2;	/* Just double the previous allocation! */
	else
		lst->size = 100;/* Start from something reasonable */
	if (lst->size < min)
		lst->size = min;
	new = (char **)MamMalloc(lst->size * sizeof(*new));
	if (oldsize && lst->list)
	    {
		memcpy((char *)new, (char *)lst->list, oldsize * sizeof(*new));
		free(lst->list);
	    }
	lst->list = new;
    }

/*
** GrabList
**	Allocate the real space for the strings stored in StringList and
**	return a pointer to it. The returned list is terminated by one
**	extra NULL pointer. Returns NULL, if the StringList was empty.
*/
static char **GrabList(StringList *list)
    {
	char **new;

	if (list->count > 0)
	    {
		new = (char **)MamMalloc((list->count+1) * sizeof(*new));
		memcpy((char *)new, (char *)list->list,
		       list->count * sizeof(*new));
		new[list->count] = NULL;
		list->count = 0;
		return new;
	    }
	return NULL;
    }

/*
** EmptyList
**	Clean out the stringlist from any old content. Usually the list
**	is emptied via GrabList, thus this function is in most cases NO-OP.
*/
static void EmptyList(StringList *list)
    {
	while (list->count > 0)
		if (list->list[--list->count])
			free((void *)list->list[list->count]);
    }

static char **StringValueList(char *v, char *e)
    {
	static StringList list;

	int c;
	char *p, *q;

	EmptyList(&list);
	while (v < e)
	    {
		for (p = q = v; v < e; ++q)
			if ((c = *v++) == ',')
				break;
		if (list.count == list.size)
			GrowList(&list, list.count+1);
		if ((list.list[list.count] = StringValue(p, q)) != NULL)
			list.count += 1;
	    }
	return GrabList(&list);
    }

/*
** tokenchars
**	is used in determining the valid characters within token (MIME)
**	and also their conversion to lowercase. Non-zero is valid token
**	character, and zero is terminating character for token.
*/
static unsigned char tokenchars[256] = 
    {
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 ,'!','"','#','$','%','&','\'',0 , 0 ,'*','+', 0 ,'-','.', 0 ,
	'0','1','2','3','4','5','6','7','8','9', 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 ,'a','b','c','d','e','f','g','h','i','j','k','l','m','n','o',
	'p','q','r','s','t','u','v','w','x','y','z', 0 , 0 , 0 ,'^','_',
	'`','a','b','c','d','e','f','g','h','i','j','k','l','m','n','o',
	'p','q','r','s','t','u','v','w','x','y','z','{','|','}','~', 0 ,

	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,
     };


/*
** ParseTokenLower
**	Extract a token from input and make it lower case. token is set to
**	point a NUL terminated token in static area, which is preserved
**	until the next call to this function. Returned pointer points to
**	character following the token in source. Does not eat the
**	terminating character.
*/
static char *ParseTokenLower(char **token, char *v, char *e)
    {
	static char buf[80];
	
	char *r;
	unsigned char c;
	for ( ; v < e && (*v == ' ' || *v == '\t'); ++v);
	r = buf;
	if (v < e && *v == '"')
		for (++v; v < e; )
		    {
			if ((c = *v++) == '"')
				break;
			else if (c == '\\' && v < e)
				c = *v++;
			if (r < &buf[sizeof(buf)-1])
				*r++ = tokenchars[c];
		    }
	else
		for (;v < e && (c=tokenchars[*(unsigned char *)v])!= 0; v++)
			if (r < &buf[sizeof(buf)-1])
				*r++ = c;
	*r++ = '\0';
	*token = buf;
	for ( ; v < e && (*v == ' ' || *v == '\t'); ++v);
	return v;
    }

/*
** ParseValue
**	Extract a value from input. Handle quoted strings too.
**	Does not eat the terminating character. This function only
**	skips the parameter value, if the 'value' pointer is NULL.
*/
static char *ParseValue(char **value, char *v, char *e)
    {
	char *p, *r;
	int c;

	for (p = v; v < e && ((c = *v++) == ' ' || c == '\t'); p = v);
	if (c != '"')
	    {
		for (v = p; v < e && tokenchars[*(unsigned char *)v]; ++v);
		r = v;
	    }
	else for (p = r = v; v < e; *r++ = c)
	    {
		if ((c = *v++) == '"')
			break;
		else if (c == '\\' && v < e)
			c = *v++;
	    }
	if (value)
		*value = StringValue(p, r);
	for ( ; v < e && (*v == ' ' || *v == '\t'); ++v);
	return v;
    }


static StringList unknown_parameters;

char *Parameter_unknown(MsgBody body, char *p, char *v, char *e)
    {
	if (unknown_parameters.count+2 > unknown_parameters.size)
		GrowList(&unknown_parameters, unknown_parameters.count+2);
	unknown_parameters.list[unknown_parameters.count] =
		(char *)MamMalloc(strlen(p)+1);
	strcpy(unknown_parameters.list[unknown_parameters.count], p);
	v=ParseValue(&unknown_parameters.list[++unknown_parameters.count],v,e);
	unknown_parameters.count += 1;
	return v;
    }

static void AddUnknownParameter(char *p, char *v)
    {
	int i = unknown_parameters.count;

	if (i+2 > unknown_parameters.size)
		GrowList(&unknown_parameters, i+2);
	unknown_parameters.list[i] = (char *)MamMalloc(strlen(p)+1);
	strcpy(unknown_parameters.list[i++], p);
	unknown_parameters.list[i] = (char *)MamMalloc(strlen(v)+1);
	strcpy(unknown_parameters.list[i++], v);
	unknown_parameters.count = i;
    }

char *Parameter_ignore(MsgBody body, char *p, char *v, char *e)
    {
	return ParseValue((char **)NULL, v, e); /* Skip the value */
    }

char *Parameter_transfer_type(MsgBody body, char *p, char *v, char *e)
    {
	v = ParseTokenLower(&p, v, e);
	body->transfer_type = (ContType)StringToEnum(p, string_ContType);
	if (body->transfer_type == CT_unknown)
		AddUnknownParameter("x-eb-transfer-type", p);
	return v;
    }

char *Parameter_transfer_subtype(MsgBody body, char *p, char *v, char *e)
    {
	v = ParseTokenLower(&p, v, e);
	body->transfer_subtype = (ContType)StringToEnum(p, string_SubType);
	if (body->transfer_type == ST_unknown)
		AddUnknownParameter("x-eb-transfer-subtype", p);
	return v;
    }


char *Parameter_boundary(MsgBody body, char *p, char *v, char *e)
    {
	if (current_boundary != NULL)
		free(current_boundary);
	return ParseValue(&current_boundary, v, e);
    }

char *Parameter_charset(MsgBody body, char *p, char *v, char *e)
    {
	v = ParseTokenLower(&p, v, e);
	body->charset = (Charset)StringToEnum(p, string_Charset);
	if (body->charset == CS_unknown)
		AddUnknownParameter("x-eb-charset", p);
	return v;
    }

char *Parameter_filename(MsgBody body, char *p, char *v, char *e)
    {
	if (body->content)
	    {
		v = ParseValue(&body->content->filename, v, e);
		body->content->start_offset = 0;
		body->content->length = -1;
	    }
	else
		v = Parameter_unknown(body, p, v, e);
	return v;
    }

static ParameterControl parameter_field[] = 
    {
	{"boundary",	Parameter_boundary },
	{"charset",	Parameter_charset },
	{"filename",	Parameter_filename },
	{"x-eb-transfer-type",	Parameter_transfer_type },
	{"x-eb-transfer-subtype", Parameter_transfer_subtype },
	{"x-eb-type",	Parameter_ignore }, /* Should never exist in file */
	{"x-eb-subtype",Parameter_ignore }, /* Should never exist in file */
	{NULL,		Parameter_unknown },
    };

static StringList unknown_headers;

static void RFC822_Unknown(MsgBody body, char *f, char *v, char *e, void *data)
    {
	int i = unknown_headers.count;

	if (i+2 > unknown_headers.size)
		GrowList(&unknown_headers, i+2);
	unknown_headers.list[i] = (char *)MamMalloc(strlen(f)+1);
	strcpy(unknown_headers.list[i], f);
	unknown_headers.list[++i] = StringValue(v, e);
	unknown_headers.count = ++i;
    }

static void RFC822_State(MsgBody body, char *f, char *v, char *e, void *data)
    {
	int c;
	char *p, *q;

	if (current_message->state != Msg_No_State)
		return; /* State already set. Only first one is effective */
	while (v < e)
	    {
		for (p = q = v; v < e; ++q)
			if ((c = *v++) == ' ')
				break;
		*q = '\0';
		c = LookKeyword(p, state_token);
		current_message->state |= c;
		if (c & Msg_Tagged)
		    {
			if (current_message->tag)
				free(current_message->tag);
			v = ParseValue(&current_message->tag, v, e);
		    }
	    }
    }

/*
** MamCreateMsgBody
**	Create and Initialize a MsgBodyStructure
*/
static MsgBody MamCreateMsgBody(MsgBody parent)
    {
	MsgBody body = (MsgBody)MamCalloc(1,sizeof(*body));

	body->parent = parent;
	if (parent && parent->subtype != ST_digest)
	    {
		body->type = CT_text;
		body->subtype = ST_plain;
	    }
	else
	    {
		body->type = CT_message;
		body->subtype = ST_rfc822;
	    }
	body->charset = CS_us_ascii;
	body->encoding = ENC_sevenBit;
	return body;
    }

/*
*/
static void RFC822_ContentType(MsgBody body, char *f, char *v, char *e,
			       void *data)
    {
	char *token;

	EmptyList(&unknown_parameters);
	v = ParseTokenLower(&token, v, e);
	body->type = (ContType)StringToEnum(token, string_ContType);
	if (body->type == CT_unknown &&
	    strcmp(token, string_ContType[(int)CT_unknown]) != 0)
		AddUnknownParameter("x-eb-type", token);
	if (v < e && *v == '/')
	    {
		v = ParseTokenLower(&token, v+1, e);
		body->subtype = (SubType)StringToEnum(token, string_SubType);
		if (body->subtype == ST_unknown &&
		    strcmp(token, string_SubType[(int)ST_unknown]) != 0)
			AddUnknownParameter("x-eb-subtype", token);
	    }
	else
		body->subtype = ST_unknown;
	/*
	** We have to decide and allocate content at this point. Parameters
	** of the content type need to have a location to store.
	*/
	if ((body->type == CT_message && body->subtype != ST_partial) ||
	    (body->type == CT_multipart))
	    {
		if (body->child == NULL)
			body->child = MamCreateMsgBody(body);
	    }
	else
		body->content=(BodyContent)MamCalloc(1,sizeof(*body->content));
	while (v < e && *v == ';')
	    {
		v = ParseTokenLower(&token, v+1, e);
		if (v < e && *v == '=')
		    {
			ParameterControl *h;

			for (h = &parameter_field[0]; h->name != NULL; ++h)
				if (strcmp(token, h->name) == 0)
					break;
			v = (*h->parse)(body, token, v+1, e);
		    }
	    }
	body->params = GrabList(&unknown_parameters);
    }

static void RFC822_ContentTransferEncoding(MsgBody body, char *f, char *v,
					   char *e, void *data)
    {
	ParseTokenLower(&f, v, e);
	body->encoding = (Encoding)StringToEnum(f, string_Encoding);
    }

static void RFC822_Importance(MsgBody body, char *f, char *v, char *e,
			      void *data)
    {
	ParseTokenLower(&f, v, e);
	*((Importance *)data) = (Importance)StringToEnum(f, string_Importance);
    }

static void RFC822_Sensitivity(MsgBody body, char *f, char *v, char *e,
			       void *data)
    {
	ParseTokenLower(&f, v, e);
        *((Sensitivity *)data)=(Sensitivity)StringToEnum(f,string_Sensitivity);
    }

static void RFC822_Time(MsgBody body, char *f, char *v, char *e, void *data)
    {
    }

static void RFC822_List(MsgBody body, char *f, char *v, char *e, void *data)
    {
	char ***list = (char ***)data;

	if (*list == NULL)
		*list = StringValueList(v, e);
	else
		RFC822_Unknown(body, f, v, e, (void *)NULL);
    }

static void RFC822_String(MsgBody body, char *f, char *v, char *e, void *data)
    {
	char **string = (char **)data;

	if (*string == NULL)
		*string = StringValue(v, e);
	else
		RFC822_Unknown(body, f, v, e, (void *)NULL);
    }

static void RFC822_Ignore(MsgBody body, char *f, char *v, char *e, void *data)
    {
    }


/*
** Specially processed RFC-822 Headers
*/
#define TO_MESSAGE(x)	LOC_MESSAGE,offsetof(struct _Message, x)
#define TO_BODY(x)	LOC_BODY,offsetof(struct _MsgBody, x)
#define TO_HEADER(x)	LOC_HEADER,offsetof(struct _MsgHeader, x)

static HeaderControl rfc822_header[] =
    {
	{"x-eb-state",	RFC822_State,	TO_MESSAGE(state)},
	{"from",	RFC822_String,	TO_HEADER(from)},
	{"date",	RFC822_String,	TO_HEADER(date)},
	{"subject",	RFC822_String,	TO_HEADER(subject)},
	{"sender",	RFC822_String,	TO_HEADER(sender)},
	{"to",		RFC822_List,	TO_HEADER(to)},
	{"cc",		RFC822_List,	TO_HEADER(cc)},
	{"bcc",		RFC822_List,	TO_HEADER(bcc)},
	{"comments",	RFC822_String,	TO_HEADER(comments)},
	{"keywords",	RFC822_String,	TO_HEADER(keywords)},
	{"in-reply-to",	RFC822_String,	TO_HEADER(in_reply_to)},
	{"expiry-date",	RFC822_Time,	TO_HEADER(expires_at)},
	{"reply-by",	RFC822_Time,	TO_HEADER(reply_by)},
	{"importance",	RFC822_Importance, TO_HEADER(importance)},
	{"sensitivity",	RFC822_Sensitivity, TO_HEADER(sensitivity)},
	{"references",	RFC822_List,	TO_HEADER(related_to)},
	{"obsoletes",	RFC822_List,	TO_HEADER(obsoletes)},
	{"mime-version",RFC822_Ignore,	TO_MESSAGE(state)},
	{"message-id",	RFC822_String,	TO_HEADER(id)},
	{"content-type",RFC822_ContentType, TO_BODY(type)},
	{"content-id",	RFC822_String,	TO_BODY(id)},
	{"content-transfer-encoding",
			RFC822_ContentTransferEncoding, TO_BODY(encoding)},
	{"content-description",
			RFC822_String,	TO_BODY(description)},
	{NULL,		RFC822_Unknown,	TO_HEADER(unknownfields)},
    };

/*
** ParseField
*/
static void ParseField(MsgBody body, char *f, char *v, char *e)
    {
	HeaderControl *h;
	char *data;

	for (h = &rfc822_header[0]; h->name != NULL; ++h)
		if (strcmp(f, h->name) == 0)
			break;
	switch (h->location)
	    {
	    case LOC_MESSAGE:
		data = (char *)current_message;
		break;
	    case LOC_BODY:
		data = (char *)body;
		break;
	    case LOC_HEADER:
		if (body->header == NULL)
			body->header = (MsgHeader)MamCalloc(1, MsgHeaderSize);
		data = (char *)body->header;
		break;
	    default:
		return; /* corruption!! */
	    }
	(*h->handler)(body, f, v, e, (void *)(data + h->offset));
    }

static void ParseMimeMultipart(MsgBody, char *, char *, char *);

/*
** ParseMessage
*/
static void ParseMessage(MsgBody body, char *s, char *e)
    {
	char *start = s;
	int c;
	char *p, *q, *r;
	/*
	** Parse RFC-822 headers.
	*/
	EmptyList(&unknown_headers);
	while (p = s, s < e && (c = *s++) != '\n')
	    {
		/* extract field name up to ':' */
		for (r = p; s < e && c != ':'; c = *(unsigned char *)s++)
			if (c > ' ' && c != 127)
				*r++ = isalpha(c) ? tolower(c) : c;
			else if (c == '\n')
			    {
				/*
				** Obviously, this is not a mail message,
				** just try to cope. This is by no means
				** a clean recovery--for that we should clean
				** out all possible header info parsed to
				** this point... --msa
				*/
				s = start;
				goto bad_headers;
			    }
		*r = '\0';
		/* extract field value */
		for (r = q = s; s < e; *r++ = c)
			if ((c = *s++) == '\n')
				if (s < e && (*s == ' ' || *s == '\t'))
				    {
					*r++ = ' ';
					do
						c = *s++;
					while (s < e && (c==' ' || c=='\t'));
				    }
				else
					break;
		ParseField(body, p, q, r);
	    }
    bad_headers:
	/*
	** NOTE:
	**	If there are unknown fields, the structures for MsgHeader
	**	has already been allocated (no need to test for existence
	**	here!)
	*/
	if (unknown_headers.count)
		body->header->unknownfields =
			GrabList(&unknown_headers);
#ifdef UNFOLD_RFC822
	/*
	** If body is of type 'message', child is required (except for
	** partial).
	*/
	if (body->child == NULL &&
	    body->type == CT_message && body->subtype != ST_partial)
		body->child = MamCreateMsgBody(body);
#endif
	/*
	** Add content structure for all MsgBodies. If this is not a
	** a basic (leaf) body, then the content indicated here covers
	** the subtree. [Note: if content in message is zero length, the
	** content structure is not touched here. If it exists, it might
	** be a reference to an external file containing the real body
	** content.]
	*/
	if (e > s)
	    {
		if (body->content == NULL)
			body->content=(BodyContent)
				MamCalloc(1,sizeof(*body->content));
		else if (body->content->filename)
		    {
			/*
			** Make sure the filename == NULL, if the content
			** is taken from the message header file --msa
			*/
			free(body->content->filename);
			body->content->filename = 0;
		    }
		body->content->start_offset = s - current_buffer;
		body->content->length = e - s;
	    }
	/*
	** Parse RFC 822 content
	*/
	if (body->child)
	    {
		if (body->type != CT_multipart || current_boundary == NULL)
		    {
			body->child->type = CT_text;
			body->child->subtype = ST_plain;
			body->child->charset = CS_us_ascii;
			body->child->encoding = ENC_sevenBit;
			ParseMessage(body->child, s, e);
		    }
		else
		    {
			p = current_boundary;
			current_boundary = NULL;
			ParseMimeMultipart(body, p, s, e);
			free(p);
		    }
	    }
    }

/*
** TrailingWhiteSpace
**	Return a pointer after the next NEWLINE, if only white space
**	characters occur before it. Return NULL otherwise.
**
** *NOTE*
**	A special case: if end is reached before the NEWLINE, then
**	an implicit NEWLINE is assumed at end.
*/
static char *TrailingWhiteSpace(char *s, char *e)
    {
	while (s < e)
		if (*s++ == '\n')
			break;
		else if (s[-1] != ' ' || s[-1] != '\t')
			return NULL;
	return s;
    }

static void ParseMimeMultipart(MsgBody parent, char *boundary, char *s,char *e)
    {
	int len = strlen(boundary);
	int last = 0;
	char *a = NULL, *skip, *begin = s;
	MsgBody *body = &parent->child;
	while (s < e && !last)
	    {
		if (s[0] == '-' && s[1] == '-' &&
		    &s[len+2] <= e && memcmp(s+2, boundary, len) == 0 &&
		    (
		     (last = (&s[len+4] <= e &&
			      s[len+2] == '-' && s[len+3] == '-' &&
			      (skip=TrailingWhiteSpace(&s[len+4],e)) != NULL))
		     ||
		     ((skip = TrailingWhiteSpace(&s[len+2], e)) != NULL)
		     ))
		    {
			char *b = s - 1;

			s = skip;
			if (a != NULL)
			    {
				if (*body == NULL)
					*body = MamCreateMsgBody(parent);
				ParseMessage(*body, a, b);
				body = &(*body)->next;
			    }
			a = s;
		    }
		else while (s < e && *s++ != '\n');
	    }
	if (!last)
	    {
		/*
		** A mangled multipart message, definitely not MIME, just
		** dump the last unterminated part (or the whole thing)
		** into a message body.
		*/
		if (*body == NULL)
			*body = MamCreateMsgBody(parent);
		ParseMessage(*body, a ? a : begin, e);
	    }
    }
/*
** MamLoadMessage
*/
Message MamLoadMessage(char *name)
    {
	long len;
	int result;

	current_message = NULL;
	/*
	** Setup general error exit handling
	*/
	if ((result = setjmp(mam_error_exit)))
	    {
		if (current_buffer != NULL)
			free(current_buffer);
		current_buffer = NULL;
		FreeMessage(current_message);
		current_message = NULL;
		return NULL;
	    }
	/*
	** The actual loading of message begin here...
	*/
	if ((current_message = NewMessage()) == NULL)
		longjmp(mam_error_exit, MamError_MEMORY);
	if (name == NULL)
		longjmp(mam_error_exit, MamError_NOPATH);
	if ((result = MamReadHeader(name, &current_buffer, &len)) != 0)
		longjmp(mam_error_exit, result);
	if (current_message->name == NULL)
	    {
		current_message->name = MamMalloc(strlen(name)+1);
		strcpy(current_message->name, name);
	    }
	if (current_message->body == NULL)
		current_message->body =
			MamCalloc(1, sizeof(*current_message->body));
	current_message->body->type = CT_text;
	current_message->body->subtype = ST_plain;
	current_message->body->transfer_type = CT_unknown;
	current_message->body->transfer_subtype = ST_unknown;
	current_message->body->charset = CS_us_ascii;
	current_message->body->encoding = ENC_sevenBit;
	ParseMessage(current_message->body,
		     current_buffer, &current_buffer[len]);
	free(current_buffer);
	return current_message;
    }

/*
** MamPeekMessage
*/
MsgState MamPeekMessage(char *name)
    {
	FILE *fp;
	long offset, length;
	int c;
	char buf[300];
	char *p, *q, *v, *e;
	MsgState state = Msg_No_State;

	if (MamHeaderInfo(name, &p, &offset, &length, (time_t *)NULL))
		return state;
	if ((fp = fopen(p, "rb")) == NULL)
		return state;
	if (offset && fseek(fp, offset, 0))
		return state;
	length = fread(buf, 1, sizeof(buf)-1, fp);
	fclose(fp);
	if (length <= 0)
		return state;
	if (strncmp(buf, "x-eb-state:", 11) != 0)
		return state;
	for (v = buf+11, e = &buf[length]; v < e; )
	    {
		for (p = q = v; v < e; ++q)
			if ((c = *v++) == ' ')
				break;
		*q = '\0';
		c = LookKeyword(p, state_token);
		state |= c;
		if (c & Msg_Tagged)
			v = ParseValue((char **)NULL,v,e); /* Skip tagvalue */
	    }
	return state;
    }
