/*
** display.c
**
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#include <ctype.h>

#include <stdlib.h>
#if SYSV_INCLUDES
#	include <malloc.h>
#	include <memory.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#endif

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xew/Basic.h>
#include <X11/Xew/Text.h>
#include <X11/Xew/TextEd.h>
#include <X11/Xew/Raster.h>
#include <X11/Xew/Audio.h>
#include <X11/Xew/Video.h>
#include <X11/Xew/Frame.h>

#if SYSV_INCLUDES || ANSI_INCLUDES
#	include <limits.h>
#else
#ifndef LONG_MAX
#define LONG_MAX (~((unsigned long)0L)>>1)
#endif
#endif

#include "Mam.h"
#include "HeaderExt.h"
#include "typedefs.h"
#include "message.h"
#include "display.h"

#undef tables_h		/* Force re-include */
#define ENUM_TABLE_GENERATION
#include "tables.h"

#define ESC	0x1b
#define FF	0x0c

typedef struct CharMapping
    {
	Charset charset;
	char *initial;
    } CharMapping;

static CharMapping char_mapping[] =
    {
	{CS_iso8859_2, "\033.B"},
	{CS_iso8859_5, "\033.L"},
	{CS_iso8859_6, "\033.G"},
	{CS_iso8859_7, "\033.F"},
	{CS_iso8859_8, "\033.H"},
	{CS_iso8859_9, "\033.M"},
	{0, NULL},
    };

static char default_charset[] = "\33(B\33)B\33.A\33/A\17\33\175";

typedef struct RTMapping
    {
	char *tag;
	char *sgr;
    } RTMapping;

static RTMapping mapping[] =
    {
	{ "bold",	"\033[1m" },
	{ "italic",	"\033[3m" },
	{ "underline",	"\033[4m" },
	{ "excerpt",	"\033[34m" },
	{ "fixed",	"\033[19m" }, /* Assuming alternate font 9!! */
	{ "/bold",	"\033[22m" },
	{ "/italic",	"\033[23m" },
	{ "/underline",	"\033[24m" },
	{ "/excerpt",	"\033[39m" },
	{ "/fixed",	"\033[10m" }, /* Assuming primary font (0)!! */
	{ "bigger",	"\033[150 B" },
	{ "smaller",	"\033[70 B" },
	{ "/bigger",	"\033[100 B" },
	{ "/smaller",	"\033[100 B" },

	{ "flushleft",	"\033[1;5 F" },
	{ "flushright",	"\033[1;7 F" },
	{ "flushboth",	"\033[1;2 F" },
	{ "center",	"\033[1;6 F" },
	{ "nofill",	"\033[0 F" },
	{ "verbatim",	"\033[0 F" },

	{ "/flushleft",	"\033[1 F" },
	{ "/flushright","\033[1 F" },
	{ "/flushboth",	"\033[1 F" },
	{ "/center",	"\033[1 F" },
	{ "/fill",	"\033[1 F" },
	{ "/verbatim",	"\033[1 F" },

	{ "nl",		"\r\n" },
	{ "np",		"\r\n\r\n" },
	{ "lt",		"<" },
	{ NULL,		NULL },
    };

/*
** The locked modes for richtext, enriched and plain text
**
** (THESE ARE NOT SUPPOSED TO BE HARD CODED, BUT SHOULD DEFINE APPLICATION
** RESOURCES FOR THESE! --msa)
*/
static char plain_prefix[] = "\033[0 F\033[0;19m"; /* Use alternate font 9 */
static char enriched_prefix[] = "\033[1 F\033[0;10m"; /* Use primary font(0) */
/*
** .. quick and dirty converter... (or mongrel of richtext and enriched)
** (and far from complete!)
**
**	NOTE!	The content buffer is modified in the process (all tags
**		are converted to lower case and a NUL byte terminated).
*/
static void RichToPlain(Widget w, char *s, long length,  int enriched)
    {
	int newline = 0;
	char *e = s + length;
	char *mark;

	for (mark = s; s < e; )
	    {
		register int c = *s++;

		if( c == '<')
		    {
			char *p, *r;
			RTMapping *h;

			if (s - mark > 1)
				XeTextInsert(w, mark, (s - mark) - 1);
			for (p = r = s; s < e && (c = *s++) != '>'; *r++ = c)
				if (isupper(c))
					c = tolower(c);
			for (h = mapping; h->tag; ++h)
				if (memcmp(p, h->tag, r - p) == 0)
				    {
					XeTextInsert(w,h->sgr,strlen(h->sgr));
					break;
				    }
			mark = s;
		    }
		else if (c == '\n')
		    {
			if (!enriched || !newline)
			    {
				if (s - mark > 1)
					XeTextInsert(w, mark, (s - mark) - 1);
				mark = s;
			    }
			newline = 1;
		    }
		else
			newline = 0;
	    }
	if (s - mark > 0)
		XeTextInsert(w, mark, (s - mark));
    }

/*
** RelaseContent
**	Assuming that the contentString is a dynamically allocated memory
**	array. A simple callback that removes the contentString from
**	the widget and releases the memory.
*/
static void ReleaseContent(Widget w, XtPointer client_data,XtPointer call_data)
    {
	String content = NULL;

	XtVaGetValues(w, XtNcontentString, &content, (XtPointer)NULL);
	if (content)
	    {
		XtVaSetValues(w, XtNcontentString, NULL, (XtPointer)NULL);
		free((void *)content);
	    }
    }

static char *Pick(char **list, char *def, char *name)
    {
	char *s, *v;

	if (list != NULL) 
		while ((s = *list++) && (v = *list++))
			if (strcmp(s, name) == 0)
				return v;
	return def;
    }

/*
** MakeContent
**
** content != NULL, if the content is malloced from memory.
**
** This function is a quick "hack". Should really consult /etc/mailcap or
** other user preferences...
*/
static Widget MakeContent
	(Widget root, MsgBody body, char *content, long length,
	 XtCallbackProc callback, XtPointer data)
    {
	static char filter_xwd[] = "xwdtopnm %s";
	static char filter_xbm[] = "xbmtopbm %s";
	static char filter_xpm[] = "xpmtoppm %s";
	
	Widget w = None;
	CharMapping *cm;
	int free_content = 0;
	char *tmp_name;
	FILE *tmp_file;
	char *filter = NULL;
	int subtype;
	Arg args[10];
	int n = 0;
	int enriched;
	
	/*
        ** Start inserting the content of this body part. Make sure the
	** insert point will be tagged with the 'body' pointer.
	*/
	XeTextSetInsertionTag(root, (XeTextTag)body);
	subtype = body->subtype;
	if (body->type == CT_rasterImage && subtype == ST_unknown)
	    {
		/*
		** Make some wild guesses here
		*/
		char *subtype_name = string_SubType[(int)subtype];

		subtype_name = Pick(body->params, subtype_name,
				    "x-eb-subtype");
		if (strstr(subtype_name, "xwd") ||
		    strcmp(subtype_name, "x-windowdump") == 0)
			subtype = ST_xwd;
		else if (strstr(subtype_name, "x-portable-"))
			subtype = ST_pbm;
		else if (strstr(subtype_name, "xbm") ||
			 strstr(subtype_name, "xbitmap"))
			subtype = ST_xwd, filter = filter_xbm; /* ugh.. */
		else if (strstr(subtype_name, "xpm"))
			subtype = ST_xwd, filter = filter_xpm; /* ugh.. */
	    }
	enriched = 0;
	switch (subtype)
	    {
	    default:
		if (body->type != CT_text)
		    {
			char *type_name = string_ContType[(int)body->type];
			char *subtype_name = string_SubType[(int)subtype];
			char buf[300];
			
			if (body->type == CT_unknown)
				type_name = Pick(body->params, type_name,
						 "x-eb-type");
			if (subtype == ST_unknown)
				subtype_name = Pick(body->params, subtype_name,
						    "x-eb-subtype");
			free_content = True;
			if (sizeof(buf)-2 <
			    strlen(type_name) + strlen(subtype_name))
				break; /* Blah! */
			strcpy(buf, type_name);
			strcat(buf, "/");
			strcat(buf, subtype_name);
			XtSetArg(args[n], XtNcontentString, buf); ++n;
			XtSetArg(args[n], XtNcontentLength, strlen(buf)); ++n;
			w = XtCreateManagedWidget
				("messageUnknown",xeTextWidgetClass,
				 root,args,n);
			break;
		    }
		/* FALL THROUGH TO text/plain!! */
	    case ST_plain:
		if (content == NULL)
			break;
		for (cm = char_mapping; cm->initial; ++cm)
			if (body->charset == cm->charset)
			    {
				XeTextInsert(root, cm->initial,
					     strlen(cm->initial));
				break;
			    }
		XeTextInsert(root,plain_prefix,sizeof(plain_prefix)-1);
		XeTextInsert(root, content, length);
		XeTextInsert(root,default_charset,sizeof(default_charset)-1);
		free_content = True;
		break;
	    case ST_enriched:
		enriched = 1;
		/* FALL THROUGH */
	    case ST_richtext:
		if (content == NULL)
			break;
		for (cm = char_mapping; cm->initial; ++cm)
			if (body->charset == cm->charset)
			    {
				XeTextInsert(root, cm->initial,
					     strlen(cm->initial));
				break;
			    }
		XeTextInsert(root, enriched_prefix, sizeof(enriched_prefix)-1);
		RichToPlain(root, content, length, enriched);
		XeTextInsert(root,default_charset,sizeof(default_charset)-1);
		free_content = True;
		break;
	    case ST_tiff:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_TIFF); ++n;
		w = XtCreateManagedWidget
			("messageRaster", xeRasterWidgetClass, root, args, n);
		free_content = True;
		break;
	    case ST_gif:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_GIF); ++n;
		w = XtCreateManagedWidget
			("messageRaster", xeRasterWidgetClass, root, args, n);
		free_content = True;
		break;
	    case ST_pbm:
	    case ST_pgm:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_PBM); ++n;
		w = XtCreateManagedWidget
			("messageRaster", xeRasterWidgetClass, root, args, n);
		free_content = True;
		break;
	    case ST_xwd:
		if (!content)
			break;
		if (filter == NULL)
			filter = filter_xwd;
		tmp_name = tempnam("./", "img");
		tmp_file = fopen(tmp_name, "wb");
		fwrite(content, 1, length, tmp_file);
		fclose(tmp_file);
		free(content);
		length = strlen(tmp_name) + strlen(filter_xwd) + 1;
		content = (char *)malloc(length);
		sprintf(content, filter, tmp_name);
		XtSetArg(args[n], XtNcontentFilter, content); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_PBM); ++n;
		w = XtCreateManagedWidget
			("messageRaster", xeRasterWidgetClass, root, args, n);
		unlink(tmp_name);
		free(tmp_name);
		free_content = True;
		break;
	    case ST_jpeg:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_JPEG); ++n;
		w = XtCreateManagedWidget
			("messageRaster", xeRasterWidgetClass, root, args, n);
		free_content = True;
		break;
	    case ST_mpeg:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNcontentFormat, XeContentFormat_MPEG); ++n;
		w = XtCreateManagedWidget
			("messageVideo", xeVideoWidgetClass, root, args, n);
		break;
	    case ST_basic:
		XtSetArg(args[n], XtNcontentString, content); ++n;
		XtSetArg(args[n], XtNcontentLength, length); ++n;
		XtSetArg(args[n], XtNplay, XePlay_FORWARD); ++n;
		w = XtCreateManagedWidget("messageAudio", xeAudioWidgetClass,
					  root, args, n);
		break;
	    }
	if (content)
		if (free_content)
			free(content);
		else
			XtAddCallback(w, XtNdestroyCallback,
				      ReleaseContent, (XtPointer)content);
	if (w != None)
		XtAddCallback(w, XtNactivateCallback, callback, data);
	/*
	** Leave insert point tagged with the parent body pointer
	*/
	XeTextSetInsertionTag(root, (XeTextTag)body->parent);
	return w;
    }

	 
static Widget DisplayBasicContent
	(Widget root, Message msg, MsgBody body,
	 XtCallbackProc callback, XtPointer data)
    {
	char *content;
	long length;

	content = MamGetContent(msg, body, &length);
	if (content == NULL)
		return None;
	if (length <= 0)
	    {
		free(content);
		return None;
	    }
	return MakeContent(root, body, content, length, callback, data);
    }

/*
** GetParamValue
*/
static char *GetParamValue(char **list, char *name)
    {
	char *s, *v;
	if (list && name)
		while ((s = *list++) != NULL)
			if ((v = *list++) != NULL && strcmp(name, s) == 0)
				return v;
	return NULL;
    }

static int CanDo(MsgBody body)
    {
	MsgBody child;

	if (body->type == CT_multipart)
		if (body->subtype == ST_alternative)
		    {
			for (child = body->child; child; child = child->next)
				if (CanDo(child))
					return 1;
			return 0;
		    }
		else
		    {
			/*
			** multipart/mixed. Completely ad hoc selection
			** algorithm: assume can do, if the parts that
			** can be shown outnumber the parts that cannot
			** be shown.
			*/
			int i = 0;

			for (child = body->child; child; child = child->next)
				i += CanDo(child) ? 1 : -1;
			return i > 0;
		    }
	else switch (body->subtype)
	    {
	    case ST_external:
		return CanDo(body->child);
	    case ST_plain:
	    case ST_richtext:
	    case ST_tiff:
	    case ST_gif:
	    case ST_jpeg:
	    case ST_basic:
	    case ST_octet_stream:
	    case ST_rfc822:
		return 1;
	    default:
		return 0;
	    }
    }

static Widget PresentExternalBody
	(Widget root, Message msg, MsgBody body,
	 XtCallbackProc callback, XtPointer data)
    {
	char *scheme = NULL, *site= NULL, *path = NULL, *name = NULL;

	char **params = body->params;
	char buf[1000];
	Arg args[10];
	int n;
	Widget w;

	/*
	** Construct something like URL (very ad hoc)
	*/
	scheme = GetParamValue(params, string_Params[Param_AccessType]);
	site = GetParamValue(params, string_Params[Param_SiteAddress]);
	if (site == NULL)
		site = GetParamValue(params, "server");
	path = GetParamValue(params, string_Params[Param_DirName]);
	name = GetParamValue(params, string_Params[Param_FileName]);
	buf[sizeof(buf)-1] = 0;
	strcpy(buf, scheme ? scheme : "unknown");
	strcat(buf, ":");
	if (site)
	    {
		strcat(buf, "//");
		strcat(buf, site);
		strcat(buf, "/");
	    }
	if (path)
	    {
		strcat(buf, path);
		strcat(buf, "/");
	    }
	if (name)
		strcat(buf, name);
	if (buf[sizeof(buf)-1])
	    {
		printf("Fatal:Overflowed buf in PresentExternalBody\n");
		exit(1);
	    }
	XeTextSetInsertionTag(root, (XeTextTag)body);
	n = 0;
	XtSetArg(args[n], XtNcontentString, buf); ++n;
	XtSetArg(args[n], XtNcontentLength, strlen(buf)); ++n;
	w = XtCreateManagedWidget("commandButton",
				  xeTextWidgetClass, root, args,n);
	XtAddCallback(w, XtNactivateCallback, callback, data);
	XeTextSetInsertionTag(root, (XeTextTag)body->parent);
	return w;
    }

Widget DisplayContentPart(Widget root, Message msg, MsgBody body,
			  XtCallbackProc callback, XtPointer data,
			  HeaderDisplayProc header)
    {
	MsgBody child;
	Widget w;

	if (body == NULL)
		return None;
	if (body->type == CT_multipart && body->subtype == ST_alternative)
		for (child = body->child ; child; child = child->next)
			if (CanDo(child))
				body = child;
	if (body->subtype == ST_external)
		w = PresentExternalBody(root, msg, body, callback, data);
	else if (body->child)
	    {
		if (body->subtype == ST_rfc822 && body->child)
		    {
			Widget parent = root;
			
			XeTextSetInsertionTag(parent, (XeTextTag)body);
			root = XtCreateManagedWidget
				("messageMessage", xeTextEdWidgetClass, parent,
				 (Arg *)NULL, 0);
			XtAddCallback(root,XtNactivateCallback,callback,data);
			XeTextDisableDisplay(root);
			(*header)(root, body->child);
			/*
			** Add implicit Newline after complex bodies
			*/
			XeTextInsert(parent, "\n", 1L);
			XeTextEnableDisplay(root);
		    }
		XeTextDisableDisplay(root);
		for (child = body->child; child; child = child->next)
			w = DisplayContentPart
				(root,msg,child,callback,data,header);
		XeTextEnableDisplay(root);
	    }
	else if (body->content != NULL)
		w = DisplayBasicContent(root, msg, body, callback, data);
	return w;
    }

void DisplayContent(Widget root, Message msg,
		    XtCallbackProc callback, XtPointer data,
		    HeaderDisplayProc header)
    {
	if (msg)
		DisplayContentPart(root, msg, msg->body, callback,data,header);
    }
