/*
** filename.c
**	contains miscellaneous filename manipulation functions
**
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#if SYSV_INCLUDES
#	include <malloc.h>
#	include <memory.h>
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#endif

#include <sys/stat.h>
#include <pwd.h>

#include "filename.h"

/*
** FileName
**	process optional "~"-contstructs and return resolved filename. The
**	filename can be input in two parts.
**
**	s, e	- delimits the first part
**	fs, fe	- delimits the second part
**
** *NOTE*
**	the parts are simply assumed concatenated (whether the second part
**	look like "absolute" path or not, is not checked!). On the other
**	hand, if first and second part are non-empty, and if first part
**	does not end with '/' or second part does not start with '/', a
**	'/' inserted between the parts.
*/
char *FileName(char *s, char *e, char *fs, char *fe)
    {
	static char buf[512];

	int n = sizeof(buf) - 1, i, k;

	if (s == e)
	    {
		s = fs;
		e = fe;
		fs = fe;
	    }
	i = 0;
	if (s < e && *s == '~')
	    {
		char *home;

		if (!*++s || *s == '/')
			home = getenv("HOME");
		else
		    {
			struct passwd *pw;
			char *user = s;

			while (s < e && *s != '/')
				++s;
			if (s - user > n)
				return NULL;
			memcpy(buf, user, s - user);
			buf[s - user] = 0;
			if ((pw = getpwnam(buf)) == NULL)
				return NULL;
			home = pw->pw_dir;
		    }
		if (home == NULL || n < (i = strlen(home)))
			return NULL;
		strcpy(buf, home);
	    }
	k = e - s;
	if (k > 0)
	    {
		if (n < i + k)
			return NULL;
		memcpy(buf + i, s, k);
		i += k;
	    }
	k = fe - fs;
	if (k > 0)
	    {
		if (i > 0 && buf[i-1] != '/' && *fs != '/')
			buf[i++] = '/';
		if (n < i + k)
			return NULL;
		memcpy(buf + i, fs, k);
		i += k;
	    }
	buf[i] = 0;
	return buf;
    }

/*
** FileNameFromPath
**	look for the specified file from the libraries and return a
**	full pathname of the file.
*/
char *FileNameFromPath(char *s, char *e, char *libs)
    {
	char *r, *name;
	struct stat info;

	if (libs == NULL || (s < e && (*s == '~' || *s == '/')))
		return FileName(s, e, e, e);
	for (r = libs; *r; libs = ++r)
	    {
		if (!(r = strchr(libs, ':')))
			r = libs + strlen(libs);
		name = FileName(libs, r, s, e);
		if (name && stat(name, &info) == 0)
			return name;
	    }
	return NULL;
    }

