/*
** BasicColor
**
**	A collection of widget class independent color manipulation
**	functions.
**
**      The color allocationg algorithm has been modified from xv 2.21
**	xvcolor.c - color allocation/sorting/freeing code John Bradley,
**	University of Pennsylvania (bradley@cis.upenn.edu)
**
**
** Copyright 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if SYSV_INCLUDES
#	include <memory.h>
#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <assert.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/BasicP.h>
#include "BasicColor.h"

static void FreeColors(w, cm, pixels, npixels, planes)
XeBasicWidget w;
Colormap cm;
unsigned long pixels[];
int npixels;
unsigned long planes;
    {
	Visual *visual = w->basic.visual;

	if (visual->class != TrueColor && visual->class != DirectColor)
		XFreeColors(XtDisplay(w), cm, pixels, npixels, planes);
    }


void _XeFreeColors(w)
XeBasicWidget w;
    {
	if (w->basic.cmap && w->basic.cmap != w->core.colormap)
	    {
		/*
		** Widget has a temporary private colormap. This private one
		** has been stored into core.colormap, and the original is
		** saved into basic.cmap.
		*/
		XFreeColormap(XtDisplay(w), w->core.colormap);
		w->core.colormap = w->basic.cmap;
		w->basic.cmap = 0;
	    }
	else if (w->basic.alloc)
	    {
		/*
		** Widget is either using the original inherited colormap
		** or a colormap allocated for the non-default visual.
		*/
		FreeColors(w, w->core.colormap, w->basic.alloc,
			   w->basic.num_allocated, 0L);
	    }
	XFlush(XtDisplay(w));
	if (w->basic.alloc)
	    {
		XtFree((void *)w->basic.alloc);
		w->basic.alloc = NULL;
	    }
	w->basic.num_allocated = 0;
    }

/*
** _XeResetColormap
**	Release all previous color allocations and start anew
**	from the default colormap or from a private new colormap
**	(if parameter private = True).
*/
Colormap _XeResetColormap(w, private)
XeBasicWidget w;
int private;
    {
	_XeFreeColors(w);
	w->basic.alloc =(unsigned long *)
		XtMalloc(w->basic.num_colors * sizeof(unsigned long));
	if (private)
	    {
		if (w->basic.cmap == w->core.colormap)
			XFreeColormap(XtDisplay(w), w->core.colormap);
		else
			w->basic.cmap = w->core.colormap;
		w->core.colormap = XCreateColormap(XtDisplay(w), XtWindow(w),
						   w->basic.visual, AllocNone);
		XSetWindowColormap(XtDisplay(w),XtWindow(w),w->core.colormap);
	    }
	return w->core.colormap;
    }
/*
** structure and routine used in SortColormap()
*/
typedef unsigned char byte;
typedef struct thing
    {
	byte r,g,b;		/* actual value of color  */
	int oldindex;           /* its index in the old colormap */
	int use;                /* # of pixels of this color */
	int mindist;            /* min distance to a selected color */
    } CMAPENT;


static void SortColormap(w, size, r, g, b, data, width, height, scanline)
XeBasicWidget w;
int size;
byte *r, *g, *b, *data;
int width, height, scanline;
    {
	byte *p, *q;

	int   i, j, mdist, entry, d, hist[256], trans[256];
	static CMAPENT c[256], c1[256], *cp, *cj, *ck;

	/* initialize histogram and compute it */

	memset((void *)hist, 0, sizeof(hist));
	for (p = data, i = height; i > 0; i--, p += scanline)
		for (j = width, q = p; j > 0; j--, q++)
			hist[*q]++;
	/*
	** Put the actually-used colors into the 'c' array in the order
	** they occur also, while we're at it, number of colors and close
	** up gaps in colortable
	*/
	for (i = w->basic.num_colors = 0, cp = c; i < size; i++) 
		if (hist[i]) 
		    { 
			trans[i] = i;
			cp->r = r[i];
			cp->g = g[i];
			cp->b = b[i];
			cp->use = hist[i]; 
			cp->oldindex = i;
			cp->mindist = 200000; /* 255^2 * 3 = 195075 */
			w->basic.num_colors += 1;
			cp += 1;
		    }
	/*
	** find most-used color, put that in c1[0]
	*/
	for (entry = -1, mdist = -1, i = 0; i < w->basic.num_colors; i++)
		if (c[i].use > mdist)
		    {
			mdist = c[i].use; 
			entry=i; 
		    }
	c1[0] = c[entry];
	c[entry].use = 0;   /* and mark it dealt with */
	/*
	** sort rest of colormap.  Half of the entries are allocated on the
	** basis of distance from already allocated colors, and half on the
	** basis of usage.  (NB: 'taxicab' distance is used throughout this
	** file.)
	**
	** Mod:  pick first 10 colors based on maximum distance.  pick
	** remaining colors half by distance and half by usage   -- JHB
	**
	** To obtain O(n^2) performance, we keep each unselected color
	** (in c[], with use>0) marked with the minimum distance to any of
	** the selected colors (in c1[]).  Each time we select a color, we
	** can update the minimum distances in O(n) time. 
	**
	** mod by Tom Lane   Tom.Lane@g.gp.cs.cmu.edu
	**
	** (above comment was in xvraster.c --msa)
	*/
	for (i = 1; i < w->basic.num_colors; i++)
	    {
		int ckR, ckG, ckB;
		/*
		** Get RGB of color last selected  and choose
		** selection method
		*/
		ck = &c1[i-1];            /* point to just-selected color */
		ckR = ck->r;
		ckG = ck->g;
		ckB = ck->b;
		entry = mdist = -1;
		if (i&1 || i<10) 
			/*
			** Now find the i'th most different color.
			** we want to select the unused color that has
			** the greatest mindist
			*/
			for (j = 0, cj = c; j < w->basic.num_colors; j++,cj++)
			    {
				if (cj->use)
				    { 
					/* this color has not been marked
					   already. update mindist */
					d = (cj->r - ckR) * (cj->r - ckR) +
					    (cj->g - ckG) * (cj->g - ckG) + 
					    (cj->b - ckB) * (cj->b - ckB);
					if (cj->mindist > d)
						cj->mindist = d;
					if (cj->mindist > mdist)
					    {
						mdist = cj->mindist;
						entry = j;
					    }
				    }
			    }
		else 
			/* Now find the i'th most different color */
			/* we want to select the unused color that has
			   the greatest usage */
			for (j=0, cj=c; j < w->basic.num_colors; j++,cj++)
			    {
				if (cj->use)
				    {  /* this color has not been marked
					  already. update mindist */
					d = (cj->r - ckR) * (cj->r - ckR) +
					    (cj->g - ckG) * (cj->g - ckG) + 
					    (cj->b - ckB) * (cj->b - ckB);
					if (cj->mindist > d)
						cj->mindist = d;
					if (cj->use > mdist) 
					    {
						mdist = cj->use; 
						entry = j;
					    }
				    }
			    }
		/* c[entry] is the next color to put in the map.  do so */
		c1[i] = c[entry];
		c[entry].use = 0;
	    }
	for ( i = 0; i < w->basic.num_colors; i++)
	    {
		trans[c1[i].oldindex] = i;
		r[i] = c1[i].r;
		g[i] = c1[i].g;
		b[i] = c1[i].b;
	    }
	/*
	** modify image to reflect new colormap
	*/
	for (p = data, i = height; i; i--, p += scanline)
		for (j = width, q = p; j; j--, q++)
			*q = trans[*q];
    }

#define NOPIX 0xffffffff    

/*
** _XeAllocColors
**	Remap and allocate colors for a map.
**	This function is only designed to be called with PseudoColor
**	and GrayScale visuals. Calling with others may produce incorrect
**	results.
**
**	Return the number of exact colors actually allocated (do not
**	count "close colors").
**
**  WARNING!
**	THIS FUNCTION GLOBBERS THE INPUT COLORMAP AND IMAGE. DO NOT
**	CALL FOR IMAGE AND COLORMAP WHICH YOU WANT TO KEEP!
*/
int _XeAllocColors(w, size, r, g, b, data, width, height, scanline)
XeBasicWidget w;
int size;			/* Size of the input RGB color maps */
unsigned char *r, *g, *b;	/* RGB colormaps */
unsigned char *data;		/* Optional input data matrix */
int width, height, scanline;	/* (data paramters) */
    {
	int j, c, n, unique, p1alloc, p2alloc, p3alloc;
	Colormap cmap;
	XColor defs[256], ctab[256];
	int fc2pcol[256];	/* maps freecols into pic pixel values */
	unsigned long cols[256];
	int dc;
	unsigned char *p, *q;

	if (size > 256)
		XeWidgetErrorMsg
			((Widget)w, "bugBasicColorMapTooLarge",
			 "Cannot handle larger than 256 colormaps",
			 (String *)NULL,0);
	unique = p2alloc = p3alloc = 0;
	SortColormap(w, size, r, g, b, data, width, height, scanline);
	cmap =_XeResetColormap
		(w, w->basic.colormap_use == XeColormapUse_PRIVATE);
	for (c = 0; c < w->basic.num_colors; c++)
	    {
		cols[c] = NOPIX;
		defs[c].red   = r[c] << 8;
		defs[c].green = g[c] << 8;
		defs[c].blue  = b[c] << 8;
		defs[c].flags = DoRed | DoGreen | DoBlue;
	    }
	/*
	** FIRST PASS COLOR ALLOCATION:  
	** for each color in the 'desired colormap', try to get it via
	** AllocColor().  If for any reason it fails, mark that pixel
	** 'unallocated' and worry about it later.  Repeat.
	**
	** attempt to allocate first max_colors entries in colormap 
	** note: On displays with less than 8 bits per RGB gun, it's quite
	** possible that different colors in the original picture will be
	** mapped to the same color on the screen.  X does this for you
	** silently.  However, this is not-desirable for this application, 
	** because when I say 'allocate me 32 colors' I want it to allocate
	** 32 different colors, not 32 instances of the same 4 shades...
	*/
	for (c=0; c<w->basic.num_colors && unique < w->basic.max_colors; c++)
	    {
		if (XAllocColor(XtDisplay(w), cmap, &defs[c]))
		    { 
			unsigned long pixel, *fcptr;

			pixel = cols[c] = defs[c].pixel;
			/*
			** see if the newly allocated color is new and
			** different
			*/
			for (j=0, fcptr = w->basic.alloc;;)
				if (j++ == w->basic.num_allocated)
				    {
					unique++;
					break;
				    }
				else if (*fcptr++ == pixel)
					break;
			fc2pcol[w->basic.num_allocated] = c;
			w->basic.alloc[w->basic.num_allocated++] = pixel;
		    }
		else if (w->basic.colormap_use == XeColormapUse_OPTIONAL &&
			 !w->basic.cmap)
		    {
			/*
			** the allocation failed.  If we want 'perfect' color,
			** and we haven't already created our own colormap,
			** we'll want to do so
			*/
			cmap = _XeResetColormap((XeBasicWidget)w, True);
			for (c=0; c < w->basic.num_colors; c++)
				cols[c] = NOPIX;
			unique = 0;
			c = -1;
		    }
		else
			/*
			** either we don't care about perfect color,
			** or we do care, have allocated our own
			** colormap, and have STILL run out of colors
			** (possible, even on an 8 bit display), just
			** mark pixel as unallocated.  We'll deal with
			** it later
			*/
			cols[c] = NOPIX;
	    }  /* FIRST PASS */
	p1alloc = w->basic.num_allocated;
	if (w->basic.num_allocated == w->basic.num_colors)
		goto colorallocdone;
	/*
	** SECOND PASS COLOR ALLOCATION:
	** Allocating 'exact' colors failed.  Now try to allocate 'closest'
	** colors.
	** 
	** Read entire X colormap (or first 256 entries) in from display.
	** for each unallocated pixel, find the closest color that actually
	** is in the X colormap.  Try to allocate that color (read only).
	** If that fails, the THIRD PASS will deal with it
	*/
	dc = (w->basic.visual->map_entries < 256) ? 
		w->basic.visual->map_entries : 256;
	for (c = 0; c < dc; c++)
		ctab[c].pixel = (unsigned long)c;
	XQueryColors(XtDisplay(w), cmap, ctab, dc);
	for (c=0; c < w->basic.num_colors && unique < w->basic.max_colors; c++)
	    {
		int d, mdist, close;
		unsigned long ri,gi,bi;
			
		if (cols[c] != NOPIX)
			continue;

		/* an unallocated pixel */

		mdist = 100000;
		close = -1;
		ri = r[c]; 
		gi = g[c];
		bi = b[c];
		for (j = 0; j < dc; j++)
		    {
			d = abs(ri - (ctab[j].red   >> 8)) +
			    abs(gi - (ctab[j].green >> 8)) +
			    abs(bi - (ctab[j].blue  >> 8));
			if (d < mdist)
			    {
				mdist=d;
				close=j;
			    }
		    }
		assert(close >= 0);
		if (XAllocColor(XtDisplay(w), cmap, &ctab[close]))
		    {
			defs[c] = ctab[close];
			cols[c]  = ctab[close].pixel;
			fc2pcol[w->basic.num_allocated] = c;
			w->basic.alloc[w->basic.num_allocated++] = cols[c];
			p2alloc++;
			unique++;
		    }
	    }
	if (w->basic.num_allocated == w->basic.num_colors)
		goto colorallocdone;
	/*
	** THIRD PASS COLOR ALLOCATION:
	** We've alloc'ed all the colors we can.  Now, we have to map any
	** remaining unalloced pixels into either A) the colors that we DID get
	** (noglob), or B) the colors found in the X colormap
	*/
	for (c = 0, n = w->basic.num_allocated; c < w->basic.num_colors; c++)
	    {
		int d, k, mdist, close;
		unsigned long ri,gi,bi;

		if (cols[c] != NOPIX)
			continue;
		/* an unallocated pixel */
			
		mdist = 100000;   close = -1;
		ri = r[c];  gi = g[c];  bi = b[c];
		if (w->basic.colormap_use != XeColormapUse_SHAREDOWN)
		    {   /* search the entire X colormap */
			for (j = 0; j < dc; j++)
			    {
				d = abs(ri - (ctab[j].red >> 8)) +
				    abs(gi - (ctab[j].green >> 8)) +
				    abs(bi - (ctab[j].blue >> 8));
				if (d < mdist)
				    {
					mdist = d;
					close = j;
				    }
			    }
			p3alloc++;
			assert(close >= 0);
			defs[c] = ctab[close];
		    }
		else
		    { 
			/* only search the alloc'd colors */
			for (j = 0; j < w->basic.num_allocated; j++)
			    {
				k = fc2pcol[j];
				d = abs(ri - (defs[k].red >> 8)) +
				    abs(gi - (defs[k].green >> 8)) +
				    abs(bi - (defs[k].blue >> 8));
				if (d < mdist)
				    { 
					mdist = d;
					close = k;
				    }
			    }
			assert(close >= 0);
			defs[c] = defs[close];
		    }
		w->basic.alloc[n++] = cols[c] = defs[c].pixel;
	    }  /* THIRD PASS */
    colorallocdone:
	for (c = 0; c < w->basic.num_colors; c++)
	    {
		r[cols[c]] = defs[c].red >> 8;
		g[cols[c]] = defs[c].green >> 8;
		b[cols[c]] = defs[c].blue >> 8;
	    }
	/*
	** Change pixel values in the image to match the allocated
	** colors.
	*/
	for (p = data, n = height; n > 0; n--, p += scanline)
		for (j = width, q = p; j > 0; j--, q++)
			*q = cols[*q];
	return p1alloc;
    }

