/*
** Font Handling Routines of the Text Widget
**
** Copyright 1992, 1993, 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>
#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <ctype.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xew/XewInit.h>
#include <X11/Xew/TextP.h>
#include "TextFont.h"

#define XLISTFONTS_MAX 2000	/* Max number of fonts to query from server */
#define XLISTFONTS_MSK "*"	/* Font mask to use in query */

/*
** XeFontListItem
*/
typedef struct XeFontListItem
    {
	char *mask;		/* X Font name mask */
	char *cset;		/* Character Set to be used with this font */
    } XeFontListItem;

/*
** XeFontListRec
**	Font List structure. The allocation size for this is
**
**		XtOffset(list) + count * sizeof(XeFontListItem)
*/
typedef struct XeFontListRec
    {
	unsigned int count;	/* Number of entries in the list */
	XeFontListItem list[1];	/* The space is allocated dynamically */
    } XeFontListRec;

/*
** XeDisplayFont
**	Each font actually being used is represented by XeDisplayFont
**	structure.
*/
typedef struct XeDisplayFont
    {
	struct XeDisplayFont *next; /* Link to next size of this same font */
	unsigned int size;	/* pixel size of the font */
	unsigned int refs;	/* reference count */
	XFontStruct *info;	/* Font information */
    } XeDisplayFont;

/*
** XeFontDataBase
**	As application may use multiple display connections with
**	different fonts in each, each display used needs to have
**	own private font database.
*/
typedef struct XeFontDataBase
    {
	struct XeFontDataBase *next; /* Link to next font database */
	Display *display;	/* Identification of the display */
	XFontStruct *default_font;/* Default Font "fixed" */
	int count;		/* actual number of font names */
	char **names;		/* array of font names (XListFonts) */
	XeDisplayFont **fonts;	/* array of display fonts (count) entries */
    } XeFontDataBase;

/*
** SplitFontName
**
*/
typedef struct XeFontField
    {
	int length;
	char *name;
    } XeFontField;

/*
** SplitFontName
**	Split fontname into array of (length, field) delimited
**	by '-' character. The max number of fields is given by
**	'n', which must be > 0. If there are more fields than
**	specified by 'n', all of the remaining string will be
**	placed into f[n-1]. If there are less fields than 'n',
**	all extra fields are initialized to "*", if the name
**	begins with "-" or "+" (assuming XLFD name); or to NULL
**	otherwise.
**
** Note:If the name conforms to X Logical Font Description, the
**	first field will always be empty or is a string starting
**	with '+' character.
*/
static void SplitFontName(name, f, n)
char *name;
XeFontField *f;
int n;
    {
	char *s;
	if (n <= 0 || name == NULL)
		return; /* ..should really not happen --msa */

	if (*name == '+' || *name == '-')
	    {
		/* XLFD type of name */

		for ( ; --n > 0 && (s = strchr(name, '-')) != NULL; f++)
		    {
			f->name = name;
			f->length = s - name;
			name = s + 1;
		    }
		f->name = name;
		f->length = strlen(name);
		if (f->length > 0)
			/*
			** Take the last field only, if it is non-empty.
			** This makes endings "-" and "-*" equivalent.
			*/
			f += 1;
		else
			n++;

		/* Fill in the unused fields with "*" */

		for ( ; --n >= 0; f++)
		    {
			f->name = "*";
			f->length = 1;
		    }
	    }
	else
	    {
		f->name = name;
		f->length = strlen(name);
		--n;

		/* Fill in the possible unused fields with NULL */
		for (f += 1; --n >= 0; f++)
		    {
			f->name = NULL;
			f->length = 0;
		    }
	    }
    }

#define	XLFD_FOUNDRY		1
#define XLFD_FAMILY_NAME	2
#define XLFD_WEIGHT_NAME	3
#define XLFD_SLANT		4
#define XLFD_SETWIDTH_NAME	5
#define XLFD_ADD_STYLE_NAME	6
#define XLFD_PIXEL_SIZE		7
#define XLFD_POINT_SIZE		8
#define XLFD_RESOLUTION_X	9
#define XLFD_RESOLUTION_Y	10
#define XLFD_SPACING		11
#define XLFD_AVERAGE_WIDTH	12
/*
** By using only 14 fields in 'fields', CHARSET_REGISTRY and
** CHARSET_ENCODING will get clumped into single field in
** SplitFontName, which exactly what is wanted.
*/
#define XLFD_CHARSET		13

/*
** FindFontDataBase
**	Find font data base for a widget. If the display doesn't yet
**	have a font database, it will be created.
*/
static XeFontDataBase *FindFontDataBase(w)
Widget w;
    {
	static XeFontDataBase *FontData = NULL;

	Display *display = XtDisplay(w);
	XeFontDataBase *fdb;

	for (fdb = FontData; ; fdb = fdb->next)
		if (fdb == NULL)
		    {
			int i;

			fdb = (XeFontDataBase *) XtMalloc(sizeof(*fdb));
			fdb->next = FontData;
			FontData = fdb;
			fdb->display = display;
			fdb->names =
				XListFonts(display, XLISTFONTS_MSK,
					   XLISTFONTS_MAX, &fdb->count);
			fdb->fonts = (XeDisplayFont **)
				XtMalloc(fdb->count * sizeof(XeDisplayFont *));
			for (i = 0; i < fdb->count; i++)
				fdb->fonts[i] = NULL;
			fdb->default_font = XLoadQueryFont(display, "fixed");
			if (fdb->default_font == NULL)
			    {
				XtAppError(XtWidgetToApplicationContext(w),
			       "XQueryFont for default font 'fixed' failed.");
			    }
			break;
		    }
		else if (fdb->display == display)
			break;
	return fdb;
    }

/*
**  Compare if a given string (name) matches the given
**  mask (which can contain wild cards: '*' - match any
**  number of chars, '?' - match any single character.
**
**	return	0, if match
**		1, if no match
**
** Iterative matching function, rather than recursive.
** Written by Douglas A Lewis (dalewis@acsu.buffalo.edu)
*/
static int matches(ma, na)
char *ma, *na;
    {
	int	wild = 0;
	unsigned char *m = (unsigned char *)ma, *n = (unsigned char *)na;
	unsigned char *mask = (unsigned char *)ma;
	
	while (1)
	    {
		if (!*m)
		    {
	  		if (!*n)
				return 0;
			while (m > mask && *--m == '?')
				;
			if (*m == '*')
				return 0;
			if (wild) 
			    {
				m = (unsigned char *)ma;
				n = (unsigned char *)++na;
			    }
			else
				return 1;
		    }
		else if (!*n)
		    {
			while(*m == '*')
				m++;
			return (*m != 0);
		    }
		if (*m == '*')
		    {
			while (*m == '*')
				m++;
			wild = 1;
			ma = (char *)m;
			na = (char *)n;
		    }
	        if (*m != *n && *m != '?')
		    {
			if (wild)
			    {
				m = (unsigned char *)ma;
				n = (unsigned char *)++na;
			    }
			else
				return 1;
		    }
		else
		    {
			if (*m)
				m++;
			if (*n)
				n++;
		    }
	    }
    }

/*
** FindMatchingFont
**	Search FontDataBase starting for the first font name that
**	matches the specified mask. Return index of the matching
**	font or (-1), if none matches. The search starts from the
**	index specified by start. If (start < 0) or (start >= count),
**	search fails immediate.
*/
static int FindMatchingFont(mask, start, fdb)
char *mask;
int start;
XeFontDataBase *fdb;
    {
	char **ptr;

	if (start >= 0)
		for (ptr = &fdb->names[start]; start < fdb->count;
		     start++, ptr++)
			if (*ptr && matches(mask, *ptr) == 0)
				return start;
	return -1;
    }

/*
** LoadFont
*/
static XeDisplayFont *LoadFont(dpy, name)
Display *dpy;
char *name;
    {
	XeFontField fields[14];
	int scaled;
	XeDisplayFont *df = (XeDisplayFont *)XtMalloc(sizeof(*df));
	
	SplitFontName(name, fields, XtNumber(fields));
	scaled = fields[XLFD_PIXEL_SIZE].length == 1 &&
		 fields[XLFD_PIXEL_SIZE].name[0] == '0';
	df->refs = 0;
	df->next = NULL;
	if (scaled)
	    {
		df->size = 0;
		df->info = NULL;
	    }
	else
	    {
		unsigned long p = 0;
		Atom ps;
		ps = XInternAtom(dpy, "PIXEL_SIZE", False);
		/* Unfortunately, cannot just store atom value into
		   a static. Application maybe using multiple displays,
		   and each might have different atom value for the
		   same string? Check this! --msa */

		if ((df->info = XLoadQueryFont(dpy, name)) != NULL)
		    {
			if (XGetFontProperty(df->info, ps, &p))
				df->size = (int)p;
			else
				df->size = -1;
		    }
	    }
	return df;
    }

/*
** LoadScaledFont
*/
static XeDisplayFont *LoadScaledFont(dpy, name, size)
Display *dpy;
char *name;
int size;
    {
	XeFontField fields[14];
	XeDisplayFont *df = (XeDisplayFont *)XtMalloc(sizeof(*df));
	char tmp[512], *s;
	int i;

	SplitFontName(name, fields, XtNumber(fields));

	df->refs = 0;
	df->next = NULL;
	df->size = size;
	for (i = 0, s = tmp;;)
	    {
		if (i == XLFD_PIXEL_SIZE)
		    {
			sprintf(s, "%d", (int)size);
			s += strlen(s);
		    }
		else
		    {
			memcpy(s, fields[i].name, fields[i].length);
			s += fields[i].length;
		    }
		i += 1;
		if (i == XtNumber(fields) || fields[i].name == NULL)
			break; /* Last field */
		*s++ = '-';
	    }
	*s = '\0';
	df->info = XLoadQueryFont(dpy, tmp);
	return df;
    }

/*
** GetFontMask
**	Locate a font name mask that matches the given characterset. This
**	function will always succeed.
*/
static char *CopyString(s, n)
char *s; int n;
    {
	char *p = NULL;

	if (n > 0)
	    {
		p = (char *)XtMalloc(n+1);
		memcpy(p, s, n);
		p[n] = '\0';
	    }
	return p;
    }

/*
** XeMakeFontList
**	Parse Font List Specification and allocate a Font List
**	entry for the parsed result.
*/
XeFontList XeMakeFontList(fls, len)
char *fls;
int len;
    {
	XeFontList fl;
	char *s, *r;
	int i;
	char *q = fls + len; /* End marker */

	if (len <= 0)
		return NULL;
	/* Find out the number of entries in list */
	for (i = 0, s = fls; ; s++)
	    {
		i += 1;
		s = strchr(s, ',');
		if (s == NULL || s >= q)
			break;
	    }
	fl = (XeFontList)XtMalloc
		(XtOffsetOf(XeFontListRec,list[0]) + i*sizeof(XeFontListItem));
	fl->count = i;
	for (i = 0, s = fls; i < fl->count; i++, s = r + 1)
	    {
		char *mask = NULL, *cset = NULL;

		while (isspace(*s))
			s++;
		for (r = s; r < q && *r != ',';)
		    {
			if (*r++ == '=')
			    {
				if (mask)
					XtFree(mask);
				mask = CopyString(s, r - s - 1);
				s = r;
			    }
		    }
		if (mask)
			cset = CopyString(s, r - s);
		else
			mask = CopyString(s, r - s);
		fl->list[i].mask = mask;
		fl->list[i].cset = cset;
	    }
	return fl;
    }

static char *GetFontMask(font, charset)
XeFontList font;
char *charset;
    {
	int i;
	char *mask = "-*-*-*-*-*-*-*-*-*-*-*-*-*-*";

	if (font != NULL)
		for (i = 0; i < font->count; i++)
		    {
			if (font->list[i].cset == NULL)
				mask = font->list[i].mask;
			else if (strcmp(charset, font->list[i].cset) == 0)
				return font->list[i].mask;
		    }
	return mask;
    }

/*
** FindFont
**	Find a font that matches the requested properties closest
**
**	weight	Weight_NORMAL, _FAINT or _BOLD
**	posture	FALSE=not italiced, TRUE=Italized
**	size	Requested pixel height of the font
**
** NOTE: This function is a CPU hog due to relying on 'matches' function.
**	 Something should be done about it --msa
*/
XFontStruct *XeFindFont(w, charset, font, weight, posture, size)
XeTextWidget w;
char *charset;
XeFontList font;
int weight, posture, size;
    {
	XeFontField fields[14];
	char mask[500], *s;
	XeDisplayFont *best, *df = NULL;
	int diff, bestdiff, i;
	XeFontDataBase *fdb;
	int is_XLFD = FALSE; /* TRUE, if XLFD fontname syntax */
	int oblique = FALSE; /* TRUE, if alternate search with
				Slant 'o' should be done also. */
	fdb = FindFontDataBase((Widget)w);
	best = NULL;
	bestdiff = 100000;

	SplitFontName(GetFontMask(font,charset), fields, XtNumber(fields));
	if (fields[0].length == 0 || fields[0].name[0] == '+')
	    {
		/*
		** The font mask follows the XLFD syntax, can actually
		** substitute the requested search criterias.
		*/
		/*
		** Build search name
		*/
		is_XLFD = TRUE;

		if (fields[XLFD_WEIGHT_NAME].length == 1 &&
		    fields[XLFD_WEIGHT_NAME].name[0] == '*')
		    {
			/* Substitute weight name */
			if (weight == Weight_BOLD)
			    {
				fields[XLFD_WEIGHT_NAME].name = "bold";
				fields[XLFD_WEIGHT_NAME].length = 4;
			    }
			else
			    {
				fields[XLFD_WEIGHT_NAME].name = "medium";
				fields[XLFD_WEIGHT_NAME].length = 6;
			    }
		    }
		if (fields[XLFD_SLANT].length == 1 &&
		    fields[XLFD_SLANT].name[0] == '*')
		    {
			/* Substitute slant, note length is already 1 */
			fields[XLFD_SLANT].name = posture ? "i" : "r";
			oblique = posture; /* ..sigh.. --msa */
		    }
		/* ..the cryptic test below is the penalty for forcing
		   the last two fields together when calling SplitFontName.
		   Depending on whether fontname had all fields or not, this
		   last field may have value "*", "*-*" or something the user
		   explicitly specified. --msa */

		if (fields[XLFD_CHARSET].length <= 3 &&
		    memcmp(fields[XLFD_CHARSET].name, "*-*",
			   fields[XLFD_CHARSET].length) == 0)
		    {
			/* substitue charset */
			fields[XLFD_CHARSET].name = charset;
			fields[XLFD_CHARSET].length = strlen(charset);
		    }
	    }

	for (;;)
	    {
		/* Build the actual search mask */

		for (i = 0, s = mask;;)
		    {
			memcpy(s, fields[i].name, fields[i].length);
			s += fields[i].length;
			i += 1;
			if (i == XtNumber(fields) || fields[i].name == NULL)
				break; /* Last field */
			*s++ = '-';
		    }
		*s = '\0';

		for (i = 0; (i = FindMatchingFont(mask, i, fdb)) >= 0; i++)
		    {
			XeDisplayFont **prev;
			
			df = fdb->fonts[i];
			if (df == NULL)
				df = fdb->fonts[i] =
					LoadFont(XtDisplay(w), fdb->names[i]);
			/*
			** Scan through the list of loaded fonts associated
			** with this one fontname. This list can have more than
			** entry only if the font is scalable.
			*/
			for (prev = &fdb->fonts[i] ; df != NULL;
			     prev = &df->next, df = *prev)
			    {
				diff = size - df->size;
				if (diff == 0)
					goto havefont;
				else if (diff < 0)
					diff = -diff;
				if (diff > bestdiff)
					continue;
				best = df;
				bestdiff = diff;
			    }
			if (fdb->fonts[i]->size == 0)
			    {
				/*
				** Scalable font, load a font with exactly
				** the requested size and assume it is the
				** one we want. Append this entry to the list
				** so that the next time we need this same size
				** we already find it in above loop and it wont
				** get loaded multiple times.
				*/
				*prev = df = LoadScaledFont
					(XtDisplay(w), fdb->names[i], size);
				goto havefont;
			    }
		    }
		if (oblique)
		    {
			/*
			** Because Slanted fonts may appear as '-o-' or
			** '-i-' and we want to find at least something
			** closely matching the the appearance, we search
			** all through again with 'o'.
			*/
			fields[XLFD_SLANT].name = "o";
			oblique = FALSE;
			continue;
		    }
		if (best || !is_XLFD)
		    {
			df = best;
			break;
		    }			
		/*
		** No font matched at all. Try to drop off some special
		** requirements and repeat the process. This can be done
		** only if the font selection follows XLFD syntax.
		*/
		if (fields[XLFD_SLANT].length != 1 ||
		    memcmp(fields[XLFD_SLANT].name,"r", 1) != 0)
		    {
			/* Change slant to upright */
			fields[XLFD_SLANT].name = "r";
			fields[XLFD_SLANT].length = 1;
			continue;
		    }
		if (fields[XLFD_WEIGHT_NAME].length != 6 ||
		    memcmp(fields[XLFD_WEIGHT_NAME].name,"medium", 6) != 0)
		    {
			/* Change weigth to "medium" */
			fields[XLFD_WEIGHT_NAME].name = "medium";
			fields[XLFD_WEIGHT_NAME].length = 6;
			continue;
		    }
		break; /* EXIT THE OUTER LOOP */
	    }

    havefont:
	if (df == NULL || df->info == NULL)
		return fdb->default_font;
	else
	    {
		df->refs += 1;
		return df->info;
	    }
    }

