/*
** ccfilter.c
**
**	Character Content Filter interprets the character stream
**	according to various standards (ISO 2022, ISO 649, ISO6429,
**	ISO 6937).
**
** *NOTE*
**	In various places this module uses hex values for byte
**	codes. This is *intentional* and *required* so that the
**	generated code will be independent of the characterset
**	the C compiler is using.
**
** Copyright 1992, 1993 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if ANSI_INCLUDES
#	include <stddef.h>
#endif
#include <X11/IntrinsicP.h>
#include "ccfilter.h"

#define TRUE 1
#define FALSE 0

#define MAX_MULTIPLE_BYTES (4)

typedef enum
    {
	NUL,	SOH,	STX,	ETX,	EOT,	ENQ,	ACK,	BEL,
	BS ,	HT ,	LF ,	VT ,	FF ,	CR ,	SO ,	SI ,
	DLE,	DC1,	DC2,	DC3,	DC4,	NAK,	SYN,	ETB,
	CAN,	EM ,	SUB,	ESC,	FS ,	GS ,	RS ,	US
    } C0_Code;

typedef enum
    {
	X80,	X81,	BPH,	NBH,	IND,	NEL,	SSA,	ESA,
	HTS,	HTJ,	VTS,	PLD,	PLU,	RI ,	SS2,	SS3,
	DCS,	PU1,	PU2,	STS,	CCH,	MW ,	SPA,	EPA,
	SOS,	X99,	X9A,	CSI,	ST ,	OSC,	PM ,	APC
    } C1_Code;

#if NeedFunctionPrototypes
typedef void (*CodeHandler)(ccf_Context, int Code);

static void Handler_CC(ccf_Context, int);/* General Complete Code Handler */
static void Handler_C0(ccf_Context, int);/* General C0 Handler */
static void Handler_C1(ccf_Context, int);/* General C1 Handler */
static void Handler_GC(ccf_Context, int);/* General GL/GR Handler */
static void Handler_GC_Switch(ccf_Context, int); /* General GL/GR switch handler */
static void Handler_GLSS2(ccf_Context, int);/* Single Shift G2 Handler */
static void Handler_GLSS3(ccf_Context, int);/* Single Shift G3 Handler */
static void Handler_ESC(ccf_Context, int);/* General ESC handler */
static void Handler_CSI(ccf_Context, int);/* General CSI handler */
#else
typedef void (*CodeHandler)();

static void Handler_CC();	/* General Complete Code Handler */
static void Handler_C0();	/* General C0 Handler */
static void Handler_C1();	/* General C1 Handler */
static void Handler_GC();	/* General GL/GR Handler */
static void Handler_GC_Switch(); /* General GL/GR switch handler */
static void Handler_GLSS2();	/* Single Shift G2 Handler */
static void Handler_GLSS3();	/* Single Shift G3 Handler */
static void Handler_ESC();	/* General ESC handler */
static void Handler_CSI();	/* General CSI handler */
#endif

typedef struct GC_type
    {
	CodeHandler handler;
	ccf_Gs Gn;
    } GC_type;


/*
**  CSI Interpreter State
*/
typedef struct CSI_State
    {
#	define CSI_STATE_INITIAL (0)
#	define CSI_STATE_PARAMETER (1)
#	define CSI_STATE_INTERMEDIATE (2)

	int state;
	int private;		/* Private sequence indicator */
	int parameter[30];	/* Parameters of a CSI sequence */
	int current;		/* Current Parameter being interpreted */
	char intermediate[4];	/* Intermediate Codes */
	int count;		/* Number of Intermediate characters */
    } CSI_State;

/*
** ESC Interpreter State
*/
typedef struct ESC_State
    {
	char intermediate[9];	/* Intermediate Codes */
	int count;		/* Number of Intermediate characters */
    } ESC_State;

typedef struct ccf_ContextRec
    {
	void *client_data;
	ccf_Gn call_Gn;
	ccf_Cn call_Cn;
	ccf_ESC call_ESC;
	ccf_CSI call_CSI;
	ccf_DG call_DG;
	ccf_DC call_DC;
	int multiple_byte;	/* Bytes remaining from next character */
	int GC_multiple_byte;	/* Bytes in one character in current GC */
	int run_length;		/* Length of contiguous bytes in same set */
	int run_characters;	/* Number of completed contiguous characters */
	char *run_start;	/* First byte of the run */
	CodeHandler C0;		/* C0 Control Handler */
	CodeHandler C1;		/* C1 Control Handler */
	GC_type GL;		/* Current GL state */
	GC_type GR;		/* Current GR state */
	GC_type *current;	/* Current GC (GL or GR) */
	GC_type *other;		/* Other GC (GL or GR) */
	int saveS;		/* Temporary for interpreting Gn designation */
	ccf_Gs saveGn;		/* Temporary for interpreting Gn designation */
	ccf_Cs saveCn;		/* Temporary for interpreting Cn designation */
	int bytes[5];		/* Bytes in sets in G0, G1, G2, G3 and CC */
	char hold_over[MAX_MULTIPLE_BYTES];
	char *holding;
	union 
	    {
		CSI_State csi;
		ESC_State esc;
	    } i;
    } ccf_ContextRec;

/*
** ExitControl
**	Must be called when the last byte of a control sequence
**	has been processed. This prepares the system for the next
**	run of normal graphic characters.
*/
static void ExitControl(cc)
ccf_Context cc;
    {
	cc->run_start += cc->run_length; /* Skips over the control sequence */
	cc->run_length = 0;
	cc->current = &cc->GL;
	cc->other = &cc->GR;
	cc->GL.handler = Handler_GC;
	cc->GR.handler = Handler_GC_Switch;
	cc->multiple_byte = cc->GC_multiple_byte = cc->bytes[cc->GL.Gn];
    }

/*
** EnterControl
**	Must be called when a control sequence starts (does not cause
**	any harm if called repeatedly within the control sequence).
**	Flush out a sequence of graphic characters preceding the
**	control sequence (if any). Any incomplete multi-byte character
**	will be scratched by this.
*/
static void EnterControl(cc)
ccf_Context cc;
    {
	if (cc->run_characters > 0)
	    {
		(*cc->call_Gn)(cc->client_data,  cc->current->Gn,
			       cc->run_start, cc->run_characters);
		cc->run_characters = 0;
	    }
    }

/*
** Handler_CSI
**	Stay in this handler until a Final Character is reached or
**	an error is detected (in which case the whole sequence up
**	to this point is discarded).
*/
static void Handler_CSI(cc, Code)
ccf_Context cc;
int Code;
    {
	switch (Code)
	    {
	    case 0x30: /* 0 */
	    case 0x31: /* 1 */
	    case 0x32: /* 2 */
	    case 0x33: /* 3 */
	    case 0x34: /* 4 */
	    case 0x35: /* 5 */
	    case 0x36: /* 6 */
	    case 0x37: /* 7 */
	    case 0x38: /* 8 */
	    case 0x39: /* 9 */
		if (cc->i.csi.state > CSI_STATE_PARAMETER)
			break;
		cc->i.csi.state = CSI_STATE_PARAMETER;
		cc->i.csi.parameter[cc->i.csi.current] =
			cc->i.csi.parameter[cc->i.csi.current]*10 + Code-0x30;
		return;	/* Continue CSI */

	    case 0x3A: /* : */
	    case 0x3B: /* ; */
		if (cc->i.csi.state > CSI_STATE_PARAMETER)
			break;
		cc->i.csi.state = CSI_STATE_PARAMETER;
		if (cc->i.csi.current < XtNumber(cc->i.csi.parameter)-1)
			++cc->i.csi.current;
		cc->i.csi.parameter[cc->i.csi.current] = 0;
		return;	/* Continue CSI */
	    case 0x3C: /* < */
	    case 0x3D: /* = */
	    case 0x3E: /* > */
	    case 0x3F: /* ? */
		if (cc->i.csi.state != CSI_STATE_INITIAL)
			break; /* -- Error! */
		cc->i.csi.private = Code;
		cc->i.csi.state = CSI_STATE_PARAMETER;
		return;	/* Continue CSI */
	    case 0x20: /*   */
	    case 0x21: /* ! */
	    case 0x22: /* " */
	    case 0x23: /* # */
	    case 0x24: /* $ */
	    case 0x25: /* % */
	    case 0x26: /* & */
	    case 0x27: /* ' */
	    case 0x28: /* ( */
	    case 0x29: /* ) */
	    case 0x2A: /* * */
	    case 0x2B: /* + */
	    case 0x2C: /* , */
	    case 0x2D: /* - */
	    case 0x2E: /* . */
	    case 0x2F: /* / */
		/*
		**  Intermediate Codes
		*/
		cc->i.csi.state = CSI_STATE_INTERMEDIATE;
		if (cc->i.csi.count == XtNumber(cc->i.csi.intermediate))
			break; /* -- ERROR, abort sequence */
		cc->i.csi.intermediate[cc->i.csi.count++] = Code;
		return;
	    default:
		/*
		** Call different functions depending on what kind of
		** sequence we had...
		*/
		cc->i.csi.current += 1;
		(*cc->call_CSI)(cc->client_data,
				cc->i.csi.private,
				cc->i.csi.parameter,
				cc->i.csi.current,
				cc->i.csi.intermediate,
				cc->i.csi.count,
				Code);
		break;
	    }
	ExitControl(cc);
    }

/*
** Handler_C1
**	Handle C1 control codes.
*/
static void Handler_C1(cc, Code)
ccf_Context cc;
int Code;
    {
	EnterControl(cc);
	switch (Code)
	    {
	    case SS2:	/* (N) Single Shift Two (Introducer) */
		cc->multiple_byte = cc->bytes[ccf_G2];
		cc->run_start += cc->run_length;
		cc->run_length = 0;
		cc->GL.handler = Handler_GLSS2;
		cc->GR.handler = Handler_GLSS2;
		break;
	    case SS3:	/* (O) Single Shift Three (Introducer) */
		cc->multiple_byte = cc->bytes[ccf_G3];
		cc->run_start += cc->run_length;
		cc->run_length = 0;
		cc->GL.handler = Handler_GLSS3;
		cc->GR.handler = Handler_GLSS3;
		break;
	    case CSI:	/* ([) Control Sequence Introducer */
		cc->i.csi.state = CSI_STATE_INITIAL;
		cc->i.csi.current = 0;
		cc->i.csi.count = 0;
		cc->i.csi.private = 0;
		cc->i.csi.parameter[0] = 0;
		cc->GL.handler = Handler_CSI;
		cc->GR.handler = Handler_CSI;
		break;
	    default:
		(*cc->call_Cn)(cc->client_data, ccf_C1, Code);
		ExitControl(cc);
		break;
	    }
    }

/*
** Handler_LoadCx
**	Changing of Control Repertoire is not really implemented.
*/
static void Handler_LoadCx(cc, Code)
ccf_Context cc;
int Code;
    {
	ExitControl(cc);
    }

/*
** Handler_LoadGx
**	Handle designations of single-byte character sets
**
**	94	G0	ESC 2/8 F
**		G1	ESC 2/9 F
**		G2	ESC 2/10 F
**		G3	ESC 2/11 F
**
**	96	G1	ESC 2/13 F
**		G2	ESC 2/14 F
**		G3	ESC 2/15 F
**			     ^
**			  (this & 7) = SaveGx
**
**	Complete Code	ESC 2/5 F  (SaveGx = -1)
*/
static void Handler_LoadGx(cc, Code)
ccf_Context cc;
int Code;
    {
	ccf_Gs save_Gn = cc->saveGn; /* Just in case  */
	int save_S = cc->saveS;

	Code = (*cc->call_DG)(cc->client_data, save_Gn, save_S, FALSE, Code);
	if (Code == 0)
		;
	else if (save_Gn == ccf_CC)
	    {
		/*
		** Escape from ISO 2022
		*/
		cc->GL.handler = cc->C0 = cc->C1 = Handler_CC;
		cc->GL.Gn = ccf_CC;
		cc->GR = cc->GL;
	    }
	else
		cc->bytes[save_Gn] =
			Code > MAX_MULTIPLE_BYTES ? MAX_MULTIPLE_BYTES : Code;
	ExitControl(cc);
    }

/*
** Handler_LoadMGx
**	Handle designations of multiple-byte character sets
**
**	n x 94	G0	ESC 2/4 F
**		G1	ESC 2/4 2/9 F
**		G2	ESC 2/4 2/10 F
**		G3	ESC 2/4 2/11 F
**
**	n x 96	G1	ESC 2/4 2/13 F
**		G2	ESC 2/4 2/14 F
**		G3	ESC 2/4 2/15 F
**				  ^
**				(this & 7) = SaveGx
** *NOTE*
**	this function *leaks*, it passes any number of
**	intermediate codes and remembers only the last.
*/
static void Handler_LoadMGx(cc, Code)
ccf_Context cc;
int Code;
    {
	ccf_Gs save_Gn = cc->saveGn; /* (Paranoid programming ;) */
	int save_S = cc->saveS;

	if (Code == 0x29 || Code == 0x2A || Code == 0x2B ||
	    Code == 0x2D || Code == 0x2E || Code == 0x2F)
	    {
		cc->saveGn = (ccf_Gs)(Code & 3);
		cc->saveS = (Code & 4) != 0;
	    }
	else if (Code > 0x3F)
	    {
		Code=(*cc->call_DG)(cc->client_data,save_Gn,save_S,TRUE,Code);
		if (Code)
			cc->bytes[cc->saveGn] =
				Code > MAX_MULTIPLE_BYTES ?
					MAX_MULTIPLE_BYTES : Code;
		ExitControl(cc);
	    }
    }

static void Handler_ESC_Intermediate(cc, Code)
ccf_Context cc;
int Code;
    {
	if (Code < 0x30)
	    {
		if (cc->i.esc.count < XtNumber(cc->i.esc.intermediate))
			cc->i.esc.intermediate[cc->i.esc.count++] = Code;
	    }
	else
	    {
		(cc->call_ESC)(cc->client_data,
			       cc->i.esc.intermediate,
			       cc->i.esc.count,
			       Code);
		ExitControl(cc);
	    }
    }

static void Handler_ESC(cc, Code)
ccf_Context cc;
int Code;
    {
	if (Code < 0x30)
		/*
		** Intermediate character
		*/
		switch (Code)
		    {
		    case 0x21:
		    case 0x22:
			cc->saveCn = (ccf_Cs)(Code & 1);
			cc->GL.handler = Handler_LoadCx;
			cc->GR.handler = Handler_LoadCx;
			break;
		    case 0x24:
			cc->saveGn = ccf_G0; /* Default to n x 94, G0 */
			cc->saveS = 0;
			cc->GL.handler = Handler_LoadMGx;
			cc->GR.handler = Handler_LoadMGx;
			break;
		    case 0x25: /* CC, 256 (Complete Code) */
			cc->saveGn = ccf_CC;
			cc->GL.handler = Handler_LoadGx;
			cc->GR.handler = Handler_LoadGx;
			break;
		    case 0x28: /* G0, 94 */
		    case 0x29: /* G1, 94 */
		    case 0x2A: /* G2, 94 */
		    case 0x2B: /* G3, 94 */

		    case 0x2D: /* G1, 96 */
		    case 0x2E: /* G2, 96 */
		    case 0x2F: /* G3, 96 */
			cc->saveGn = (ccf_Gs)(Code & 3);
			cc->saveS = (Code & 4) != 0;
			cc->GL.handler = Handler_LoadGx;
			cc->GR.handler = Handler_LoadGx;
			break;
		    default:
			cc->GL.handler = Handler_ESC_Intermediate;
			cc->GR.handler = Handler_ESC_Intermediate;
			cc->i.esc.count = 1;
			cc->i.esc.intermediate[0] = Code;
			break;
		    }
	else if (Code < 0x60)
		/*
		** 7 bit representation of C1 control
		*/
		Handler_C1(cc, Code & 0x1F);
	else
	    {
		if (Code == 0x7E) /* LS1R */
			cc->GR.Gn = ccf_G1;
		else if (Code == 0x6E) /* LS2 */
			cc->GL.Gn = ccf_G2;
		else if (Code == 0x7D) /* LS2R */
			cc->GR.Gn = ccf_G2;
		else if (Code == 0x6F) /* LS3 */
			cc->GL.Gn = ccf_G3;
		else if (Code == 0x7C) /* LS3R */
			cc->GR.Gn = ccf_G3;
		else
			(*cc->call_ESC)(cc->client_data,
					(char *)NULL, 0, Code);
		ExitControl(cc);
    	    }
    }

/*
** Handler_C0
*/
static void Handler_C0(cc, Code)
ccf_Context cc;
int Code;
    {
	EnterControl(cc);
	switch (Code)
	    {
	    case ESC:
		cc->GL.handler = Handler_ESC;
		cc->GR.handler = Handler_ESC;
		return;
	    case SI:	/* LS0, lock shift G0 */
		cc->GL.Gn = ccf_G0;
		break;
	    case SO:	/* LS1, lock shift G1 */
		cc->GL.Gn = ccf_G1;
		break;
	    case CAN:
		/* Will abort CSI/ESC sequence if in progress */
	    default:
		(*cc->call_Cn)(cc->client_data, ccf_C0, Code);
		break;
	    }
	ExitControl(cc);
    }


/*
** Handler_CC
*/
static void Handler_CC(cc, Code)
ccf_Context cc;
int Code;
    {
    }

/*
** Handler_GC
**	Passes consecutive bytes belonging to the same set.
*/
static void Handler_GC(cc, Code)
ccf_Context cc;
int Code;
    {
	if (--cc->multiple_byte == 0)
	    {
		/*
		** One complete multiple-byte or single-byte character
		** completed.
		*/
		cc->multiple_byte = cc->GC_multiple_byte;
		cc->run_characters += 1;
	    }
    }

/*
** Handler_GC_Switch
**	GC_Switch is called when after a sequence of characters
**	in GL or GR set, a byte belonging to another set arrives.
*/
static void Handler_GC_Switch(cc, Code)
ccf_Context cc;
int Code;
    {
	GC_type *temp;
	/*
	** If a multiple-byte sequence within GL or GR is interrupted by
	** a byte belonging to the other set, just include this byte into
	** the multiple byte character and continue using the same GL/GR set.
	** (This is an error condition not specified in the standard and
	** this way of handling it is the usual practise in the existing
	** implementations --msa)
	*/
	if (cc->multiple_byte == cc->GC_multiple_byte)
	    {
		/*
		** Starting a Run on the other set. Flush out any accumulated
		** characters and initialize for new alternate run.
		*/
		if (cc->run_characters > 0)
			(*cc->call_Gn)(cc->client_data,
				       cc->current->Gn,
				       cc->run_start,
				       cc->run_characters);
		cc->run_start += cc->run_length - 1;
		cc->run_length = 1;
		cc->run_characters = 0;
		temp = cc->current;
		cc->current = cc->other;
		cc->other = temp;
		cc->other->handler = Handler_GC_Switch;
		cc->current->handler = Handler_GC;
		cc->multiple_byte =
			cc->GC_multiple_byte = cc->bytes[cc->current->Gn];
	    }
	Handler_GC(cc, Code);
    }

static void Handler_GLSS2(cc, Code)
ccf_Context cc;
int Code;
    {
	if (--cc->multiple_byte == 0)
	    {
		(*cc->call_Gn)(cc->client_data, ccf_G2, cc->run_start, 1);
		ExitControl(cc);
	    }
    }

static void Handler_GLSS3(cc, Code)
ccf_Context cc;
int Code;
    {
	if (--cc->multiple_byte == 0)
	    {
		(*cc->call_Gn)(cc->client_data, ccf_G3, cc->run_start, 1);
		ExitControl(cc);
	    }
    }

/*
** ccf_Feed
**	Decode a portion of the stream.
*/
int ccf_Feed(cc, s, n)
ccf_Context cc;
char *s;
int n;
    {
	register int Code;
	register char *h;
	if (cc->multiple_byte < cc->GC_multiple_byte)
	    {
		/*
		** A multiple-byte character was divided across portions.
		** Assuming that this will not really be a common occurrence,
		** we do it by rather brute force method.
		*/
		h = cc->holding;
		cc->run_start = cc->hold_over;
		cc->run_length = h - cc->hold_over;
		cc->run_characters = 0;
		do
		    {
			if (n == 0)
			    {
				/*
				** This portion didn't complete the character.
				*/
				cc->holding = h;
				return 0;
			    }
			Code = *(unsigned char *)s;
			/*
			** Occurrence of any control code will abort
			** the multibyte sequence.
			*/
			if ((0x60 & Code) == 0)
				break;
			--n;
			cc->run_length += 1;
			*h++ = Code;
			s++;
			if (Code & ~(0x7F))
				(*cc->GR.handler)(cc, Code & 0x7F);
			else
				(*cc->GL.handler)(cc, Code);
		    } while (cc->multiple_byte < cc->GC_multiple_byte);
		if (cc->run_characters > 0)
			(*cc->call_Gn)(cc->client_data,
				       cc->current->Gn,
				       cc->hold_over, 1);
	    }
	cc->run_start = s;
	cc->run_length = 0;
	cc->run_characters = 0;
	while (--n >= 0)
	    {
		cc->run_length += 1;
		Code = *(unsigned char *)s++;
	 	if (Code & ~(0x7F))
			(*(Code&0x60 ? cc->GR.handler:cc->C1))(cc,Code & 0x7F);
		else	
			(*(Code&0x60 ? cc->GL.handler:cc->C0))(cc,Code);
	    }
	/*
	** Flush out all remaining full characters from this portion
	*/
	if (cc->run_characters > 0)
		(*cc->call_Gn)(cc->client_data,
			       cc->current->Gn,
			       cc->run_start,
			       cc->run_characters);
	/*
	** If multiple-byte character is being devided across portions,
	** copy the first part into HoldOver.
	*/
	if (cc->multiple_byte < cc->GC_multiple_byte)
	    {
		s = cc->run_start + cc->run_characters * cc->GC_multiple_byte;
		n = cc->GC_multiple_byte - cc->multiple_byte;
		for (h = cc->hold_over; --n >= 0;)
			*h++ = *s++;
		cc->holding = h;
	    }
	return 0;
    }

ccf_Context ccf_Open(client_data, Gn, Cn, ESC, CSI, DG, DC)
void *client_data;
ccf_Gn Gn;
ccf_Cn Cn;
ccf_ESC ESC;
ccf_CSI CSI;
ccf_DG DG;
ccf_DC DC;
    {
	ccf_Context cc = (ccf_Context)XtCalloc(1, sizeof(ccf_ContextRec));
	int i;

	if (cc)
	    {
		cc->client_data = client_data;
		cc->call_Gn = Gn;
		cc->call_Cn = Cn;
		cc->call_ESC = ESC;
		cc->call_CSI = CSI;
		cc->call_DG = DG;
		cc->call_DC = DC;
		cc->C0 = Handler_C0;
		cc->C1 = Handler_C1;
		cc->GL.handler = Handler_GC;
		cc->GL.Gn = ccf_G0;
		cc->GR.handler = Handler_GC_Switch;
		cc->GR.Gn = ccf_G2;
		for (i = 0; i < XtNumber(cc->bytes); ++i)
			cc->bytes[i] = 1;
		cc->multiple_byte=cc->GC_multiple_byte=cc->bytes[cc->GL.Gn];
		cc->current = &cc->GL;
		cc->other = &cc->GR;
	    }
	return cc;
    }

void ccf_Close(cc)
ccf_Context cc;
    {
	if (cc)
		XtFree((char *)cc);
    }
