/*-
|> File
|| drawcmd.c
||
|> Description
||
|> Comments
||
|> History
|| 950jj _mf Created.
|| Oct/Nov 1995 modified by msa
||
||==================================================================|>
*/
#ifndef _NOIDENT
#ident "$Id: drawcmd.c,v 2.0 1995/06/05 16:52:11 reynolds Exp mreynold $"
#endif

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <X11/Xproto.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "printer.h"
#include "printlib.h"
#include "drawcmd.h"
#include "global.h"
#include "afm.h"

extern struct PrintClient PCS[];

/*-=================================================================|>
||
|> Public functions
||
||==================================================================|>
||
*/

/*
** BuildFont postscript program has 6 parameters
**
**	base fontname
**	new fontname
**	new encoding
**	unique id offset
**	widths array (in application pixel units)
**	size (in application pixel units)
*/
static char *ps_BuildFont[] =
    {
	"/buildfontdict 15 dict def",
	"/BuildFont",
	"{ buildfontdict begin",
	"  /size exch def",
	"  /widths exch def",
	"  /uniqueid exch def",
	"  /newencoding exch def",
	"  /newfontname exch def",
	"  /basefontname exch def",
	"  /basefontdict basefontname findfont def",
	"  /newfont basefontdict maxlength  2 add dict def",
	"  /newmetrics basefontdict /CharStrings get length dict def",
	"  /uniqueid basefontdict /UniqueId known",
	"  { basefontdict /UniqueId get} { 0 } ifelse uniqueid add def",
	"  /charcount 0 def",
	"  newencoding",
	"  { /charname exch def",
	"    charname /.notdef ne",
	"    { /newwidth widths charcount get 1000 mul size div round cvi def",
	"      newmetrics charname newwidth put",
	"    } if",
	"    /charcount charcount 1 add def",
	"  } forall",
	"  basefontdict",
	"  { exch dup dup /FID ne exch /Encoding ne and",
	"    { exch newfont 3 1 roll put }",
	"    { pop pop }",
	"  ifelse",
	"  } forall",
	"  newfont /FontName newfontname put",
	"  newfont /Encoding newencoding put",
	"  newfont /Metrics newmetrics put",
	"  newfont /UniqueId uniqueid put",
	"  newfontname newfont definefont pop",
	"  end",
	"} def",
	NULL,
    };

/* round up to next multiple of 4 */
long
pad (const long n)
{
  return ((n + 3) & ~0x3);
}

/* set up the initial definitions in the postscript file */

void initprocs (struct PrintClient *pc, int fd)
{
  FILE                   *fp = pc->printfp;
  int                    width, height;
  char                   **s;

/* postscript has images reverse from real world */

  fprintf (fp, "%%!PS-Adobe-2.0\n");
  fprintf (fp, "%%%%Creator: X to PostScript server\n");
  fprintf (fp, "%%%%Pages: 1\n");
  if (!pc->landscape)
  {
    fprintf (fp, "%%%%Orientation: Portrait\n");
  }
  else
  {
    fprintf (fp, "%%%%Orientation: Landscape\n");
  }
  fprintf (fp, "%%%%EndComments\n");

/* definition for image drawing function */

  fprintf (fp, "/imagedraw\n");
  fprintf (fp, "{\n");
  fprintf (fp, "/matrix exch cvlit def\n");
  fprintf (fp, "/depth exch cvlit def\n");
  fprintf (fp, "/height exch cvlit def\n");
  fprintf (fp, "/width exch cvlit def\n");
  fprintf (fp, "/picstr width string def\n");
  fprintf (fp, "width height depth matrix\n");
  fprintf (fp, "    { currentfile picstr readhexstring pop }\n");
  fprintf (fp, "    false 3 colorimage\n");
  fprintf (fp, "} def\n");

/* definition of arc drawing function */

  fprintf (fp, "/elliparc\n");
  fprintf (fp, "{\n");
  fprintf (fp, "/angle2 exch cvlit def\n");
  fprintf (fp, "/angle1 exch cvlit def\n");
  fprintf (fp, "/height exch cvlit def\n");
  fprintf (fp, "/width exch cvlit def\n");
  fprintf (fp, "/y exch cvlit def\n");
  fprintf (fp, "/x exch cvlit def\n");
  fprintf (fp, "1 height width div scale\n");
  fprintf (fp, "x width 2 div add\n");  /* x: center */
  fprintf (fp, "y height 2 div add\n");   /* y: center */
  fprintf (fp, "width 2 div\n");  /* radius */
  fprintf (fp, "angle1 angle2 arc\n");  /* ang1 ang2 arc */
  fprintf (fp, "} def\n");

/* define font build function */

  for (s = ps_BuildFont; *s; ++s)
	  fprintf(fp, "%s\n", *s);
	  
/* create ISO 8859-1 Encoding array */

  fprintf(fp, "/ISO-8859-1 [\n");
  for (s = GetEncoding_ISO8859_1(); *s; ++s)
	  fprintf(fp, "/%s\n", *s);
  fprintf(fp, "] def\n");

  if (pc->landscape)
      {
	  fprintf (fp, "90 rotate\n");
	  width = pc->pageHeight;
	  height = pc->pageWidth;
      }
  else
      {
	  width = pc->pageWidth;
	  height = pc->pageHeight;
      }
  fprintf (fp, "6 dict begin\nclippath pathbbox\n");
  fprintf (fp, "/ury exch def /urx exch def\n");
  fprintf (fp, "/lly exch def /llx exch def\n");
  fprintf (fp, "/pagewidth urx llx sub def\n");
  fprintf (fp, "/pageheight ury lly sub def\n");
  fprintf (fp, "llx ury translate\n");
  switch (pc->preserveAspect)
      {
      case XpScale_FITAREA:
	  /* scale client page to printer page, may change aspect ratio */
	  fprintf(fp, "pagewidth %d div\n", width);
	  fprintf(fp, "pageheight %d div\n", height);
	  break;
      default:
      case XpScale_MAXASPECT:
	  /* fit and scale client page to printer page, preserve aspect */
	  fprintf(fp, "pageheight %d div pagewidth %d div 2 copy\n",
		  height, width);
	  fprintf(fp, "lt {pop} {exch pop} ifelse dup\n");
	  break;
      case XpScale_WIDTH:
	  /* match client width to printer width, preserve aspect */
	  fprintf(fp, "pagewidth %d div\n", width);
	  fprintf(fp, "dup\n");
	  break;
      case XpScale_HEIGHT:
	  /* match client height to printer height, preserve aspect */
	  fprintf(fp, "pageheight %d div\n", height);
	  fprintf(fp, "dup\n");
	  break;
      case XpScale_NONE:
	  /* no scaling at all */
	  fprintf(fp, "1 1\n");
      }
  fprintf(fp, "neg scale\n");
  fprintf(fp, "end\n");
  fprintf (fp, "0 setlinewidth\n");

/* set a basic default font */

  fprintf (fp, "/Times-Roman findfont 7 scalefont setfont\n");

}

/* set the foreground color according to the pixel and the current colormap */

static void setcolor (FD fd, FILE * fp, unsigned long pixel, Colormap cmap)
{
  double                  red,
                          green,
                          blue;
  cmapinfoPtr             cmapinfo;
  XColor                  corrColor;

  cmapinfo = getcmapinfo (cmap);
  if (cmapinfo == NULL)
  {
    fprintf (stderr, "failed to find cmap %ld\n", cmap);
    return;
  }
  if (pixel < NUM_COLORS)
  {

/* apply color correction if there's a color correction file */

/* ..the code below fails for TrueColor visual! Should it work? -- msa */

    if (PCS[fd].correctFile != NULL)
    {
      corrColor.red = cmapinfo->colors[pixel].red;
      corrColor.green = cmapinfo->colors[pixel].green;
      corrColor.blue = cmapinfo->colors[pixel].blue;
      correctColor (&corrColor, PCS[fd].numCorrect, PCS[fd].corrects);
      red = (double) corrColor.red / 65536.0;
      green = (double) corrColor.green / 65535.0;
      blue = (double) corrColor.blue / 65535.0;
    }
    else
    {
      red = (double) cmapinfo->colors[pixel].red / 65536.0;
      green = (double) cmapinfo->colors[pixel].green / 65535.0;
      blue = (double) cmapinfo->colors[pixel].blue / 65535.0;
    }
    fprintf (fp, "%.3f %.3f %.3f setrgbcolor\n", red, green, blue);
  }
}

/* set the line width of the current line */

static void setlinewidth (FILE * fp, int line_width)
{
  fprintf (fp, "%d setlinewidth\n", line_width);
}

/* set the line cap parameter for the graphics state */

static void setlinecap (FILE * fp, int cap_style)
{
  int                     style = 0;

  switch (cap_style) {
  case CapNotLast:
  case CapButt:
    style = 0;
    break;
  case CapRound:
    style = 1;
    break;
  case CapProjecting:
    style = 2;
    break;
  default:
    break;
  }
  fprintf (fp, "%d setlinecap\n", style);
}

/* set the line join style */

void
setlinejoin (FILE * fp,
             int join_style)
{
  int                     style = 0;

  switch (join_style)
  {
  case JoinMiter:
    style = 0;
    break;
  case JoinRound:
    style = 1;
    break;
  case JoinBevel:
    style = 2;
    break;
  default:
    break;
  }
  fprintf (fp, "%d setlinejoin\n", style);
}


static void setdash(FILE *fp, int num, unsigned char *dashes, int offset)
    {
	fprintf(fp, "[");
	while (--num >= 0)
	    {
		fprintf(fp, " %d", (int)*dashes);
		++dashes;
	    }
	fprintf(fp, "] %d setdash\n", offset);
    }

/* find font by Font ID */

static fontPtr findfont(Font fontId)
    {
	int i;
	fontPtr font;

	if (fontId == initial_GC.font)
		return NULL;

	for (i = 0, font = fontTable->fontTable; i < fontTable->numFonts;
	     i++, font++)
		if (font->fid == fontId)
			return font;
	return NULL;
    }


/*
** definefont
**	create a font to be used. Because the printserver needs to
**	use a specific encodings and explicit width values, each font
**	used must be reencoded and widhts modified.
*/
static fontPtr addfont(FILE *fp, int fd, Font fontId)
    {
	static int unique = 0;

	fontPtr	font;
	
	if (fontId == initial_GC.font)
		return NULL;
	font = findfont(fontId);
	if (font && font->font_data && font->font_data->font_info)
	    {
		FontLoad *load;
		int i;
		FontData *data = font->font_data;
		XFontStruct *info = data->font_info;

		for (load = PCS[fd].fonts; load; load = load->next)
			if (load->font == font->font_data &&
			    load->size == font->size)
				return font; /* Already loaded! Nothing to do! */
		/*
		** Font with this specific size has not yet been initialized
		*/
		if ((load = (FontLoad *)malloc(sizeof(FontLoad))) == NULL)
			return NULL;
		load->next = PCS[fd].fonts;
		load->size = font->size;
		load->font = ReferenceFont(data);
		PCS[fd].fonts = load;

		/*
		** Build a size specific new font
		*/
		fprintf(fp, "/%s\n", data->font_name);
		if (data->iso_8859_1)
		    {
			fprintf(fp, "/%s-%d-ISO-8859-1\n",
				data->font_name, font->size);
			fprintf(fp, "ISO-8859-1\n");
		    }
		else
		    {
			fprintf(fp, "/%s-%d-ADOBE-FONTSPECIFIC\n",
				data->font_name, font->size);
			fprintf(fp, "/%s findfont /Encoding get\n",
				data->font_name);
		    }
		fprintf(fp, "%d\n", ++unique);
		fprintf(fp, "[ ");
		for (i = -info->min_char_or_byte2;
		     i<= (int)(info->max_char_or_byte2-info->min_char_or_byte2);
		     ++i)
		    {
			if (i < 0)
				fprintf(fp, "0 ");
			else
				fprintf(fp, "%d ",
					PIXELS(data,
					       info->per_char[i].width,
					       font->size));
			if ((i & 0xF) == 0)
				fprintf(fp, "\n");
		    }
		fprintf(fp, "\n]\n");
		fprintf(fp, "%d BuildFont\n", font->size);
	    }
	return font;
    }

/* set an appropriate postscript font from an X bitmap font */

static int setfont (FILE * fp, int fd, Font fontId)
{
  fontPtr                 font;

/* first, look in the font cache to see if there's a matching font */

  if (fontId == initial_GC.font)
	  return FALSE;	/* No font defined for this GC! */
  font = addfont(fp, fd, fontId);
  if (font && font->font_data)
    {
	char *font_name = font->font_data->font_name;
	char *encoding = "";

        if (font_name != NULL && font->font_data->font_info != NULL)
	    {
		if (font->font_data->iso_8859_1)
			encoding = "ISO-8859-1";
		else
			encoding = "ADOBE-FONTSPECIFIC";
		fprintf (fp, "/%s-%d-%s findfont %d scalefont setfont\n",
			 font_name, font->size, encoding, font->size);
	    }
        return TRUE;
    }
  fprintf (stderr, "server font id %ld not found\n", fontId);
  return FALSE;
}

/* fill a list of rectangles */

void
polyfillrectangle (FD fd,
		   FILE * fp,
                   Window win,
                   gcinfoPtr gc,
                   int nrect,
                   XRectangle *rectangles)
{
  /*
  ** Postscript and X rectangles are slightly different, fill and clip
  ** out the parts that would be painted outside the X rectangle (half
  ** of the linewidth all around)
  */
  while (nrect-- > 0)
  {
    int width = rectangles->width - 1;
    int height = rectangles->height - 1;

    if (width <= 0)
	    width = 1;
    if (height <= 0)
	    height = 1;
    fprintf (fp, "gsave\n");
    fprintf (fp, "newpath\n");
    fprintf (fp, "%d %d moveto\n", rectangles->x, rectangles->y);
    fprintf (fp, "0 %d rlineto\n", height);
    fprintf (fp, "%d 0 rlineto\n", width);
    fprintf (fp, "0 %d rlineto\n", - height);
    fprintf (fp, "closepath clip fill\n");   /* closepath does last vertex */
    fprintf (fp, "grestore\n");

/* next rectangle */

    rectangles++;
  }
}

/* fill a list of rectangles */

void
polyrectangle (FD fd,
	       FILE * fp,
               Window win,
               gcinfoPtr gc,
               int nrect,
               XRectangle *rectangles)
{
  /*
  ** Postscript and X rectangles are slightly different, need to trick
  ** a bit: double the linewidth and clip the half that goes outside the
  ** rectangle.
  */
  while (nrect-- > 0)
  {
    fprintf (fp, "gsave\n");
    setlinewidth(fp, 2 * gc->attr.line_width);
    fprintf (fp, "newpath\n");
    fprintf (fp, "%d %d moveto\n", rectangles->x, rectangles->y);
    fprintf (fp, "0 %d rlineto\n", rectangles->height);
    fprintf (fp, "%d 0 rlineto\n", rectangles->width);
    fprintf (fp, "0 %d rlineto\n", -rectangles->height);
    fprintf (fp, "closepath clip stroke\n");   /* closepath does last vertex */
    fprintf (fp, "grestore\n");

/* next rectangle */

    rectangles++;
  }
}

/* draw a filled polygon */

void
fillpoly (FD fd,
	  FILE * fp,
          Window win,
          gcinfoPtr gc,
          int npoint,
          int coordmode,
          XPoint *points)
{
  fprintf (fp, "newpath\n");
  fprintf (fp, "%d %d moveto\n", (int) points->x, (int) points->y);
  points++;
  while (--npoint > 0)
  {
    if (coordmode == CoordModePrevious)
    {
      fprintf (fp, "%d %d rlineto\n", (int) points->x, (int) points->y);
    }
    else
    {
      fprintf (fp, "%d %d lineto\n", (int) points->x, (int) points->y);
    }
    points++;
  }
  fprintf (fp, "closepath fill\n");
}

/* draw a polyline */

void
polyline (FD fd,
	  FILE * fp,
          Window win,
          gcinfoPtr gc,
          int npoint,
          int coordmode,
          XPoint *points)
{
  fprintf (fp, "newpath\n");
  fprintf (fp, "%d %d moveto\n", (int) points->x, (int) points->y);
  points++;
  while (--npoint > 0)
  {
    if (coordmode == CoordModePrevious)
    {
      fprintf (fp, "%d %d rlineto\n", (int) points->x, (int) points->y);
    }
    else
    {
      fprintf (fp, "%d %d lineto\n", (int) points->x, (int) points->y);
    }
    points++;
  }
  fprintf (fp, "stroke\n");
}

/* draw a polypoint */

void
polypoint (FD fd,
	   FILE * fp,
           Window win,
           gcinfoPtr gc,
           int npoint,
           int coordmode,
           XPoint *points)
{
  fprintf (fp, "newpath\n");
  if (coordmode == CoordModePrevious)
  {
    fprintf (fp, "%d %d moveto\n", (int) points->x, (int) points->y);
    npoint--;
    points++;
  }

/* draw the points as very, very short lines */

  while (npoint-- > 0)
  {
    if (coordmode == CoordModeOrigin)
    {
      fprintf (fp, "%d %d moveto\n", (int) points->x, (int) points->y);
      fprintf (fp, "1 1 rlineto\n");
    }
    else
    {
      fprintf (fp, "%d %d rmoveto\n", (int) points->x, (int) points->y);
      fprintf (fp, "1 1 rlineto\n");
    }
    points++;
  }
  fprintf (fp, "stroke\n");
}

/* arc draw function */

void
polyarc (FD fd,
	 FILE * fp,
         Window win,
         gcinfoPtr gc,
         int narc,
         xArc * arcs)
{
  double                  angle1,
                          angle2;

  fprintf (fp, "newpath\n");
  while (narc--)
  {

/* X arcs are relative, PostScript Arcs are not */

    angle1 = ((double) arcs->angle1) / 64.0;
    angle2 = ((double) arcs->angle2) / 64.0 + angle1;
    fprintf (fp, "%d %d %d %d %f %f elliparc\n",
             arcs->x, arcs->y, arcs->width, arcs->height, -angle2, -angle1);
  }
  fprintf (fp, "stroke\n");
}

/* draw a polysegment set */

void
polysegment (FD fd,
	     FILE * fp,
             Window win,
             gcinfoPtr gc,
             int nseg,
             XSegment *segments)
{
  fprintf (fp, "newpath\n");
  while (nseg-- > 0)
  {
    fprintf (fp, "%d %d moveto\n", segments->x1, segments->y1);
    fprintf (fp, "%d %d lineto\n", segments->x2, segments->y2);
    segments++;
  }
  fprintf (fp, "stroke\n");
}

/*
** showtext
**	Generates "( text ) show" (with safe quoting of characters)
*/
static void showtext(FILE *fp, unsigned char *text, int length)
    {
	unsigned char buf[1024], *p, *limit, *end;

	limit = &buf[sizeof(buf)-12]; /* ..such that there is always room
					 for *any* character and the trailing
					 ") show\n" */
	p = buf;
	end = text + length;
	*p++ = '(';
	while (text < end)
	    {
		if (p >= limit)
		    {
			fwrite(buf, 1, p - buf, fp);
			p = buf;
		    }
		if (*text < 32 || *text > 127)
		    {
			*p++ = '\\';
			*p++ = ((*text >> 6) & 0x7) + '0';
			*p++ = ((*text >> 3) & 0x7) + '0';
			*p++ = (*text++ & 0x7) + '0';
		    }
		else
		    {
			if (*text == '(' || *text == ')' || *text == '\\')
				*p++ = '\\';
			*p++ = *text++;
		    }
	    }
	strcpy((char *)p, ") show\n");
	p += 7;
	fwrite(buf, 1, p - buf, fp);
    }

/* draw text, which is set up in chunks, and may be interspersed with
   font information */

void
polytext8 (FD fd,
	   FILE * fp,
           Window win,
           gcinfoPtr gc,
           int x,
           int y,
           int textlen,
           unsigned char *buffer)
{
  int                     size,
                          delta;

  fprintf (fp, "%d %d moveto\n", x, y);
  fprintf (fp, "gsave\n");
  fprintf (fp, "1 -1 scale\n");
  while (textlen > 0)
  {
    if (buffer[0] == 255)
    {
      gc->attr.font = (buffer[0] << 24) | (buffer[1] << 16) |
	      (buffer[2] << 8) | buffer[0];
      setfont(fp, fd, gc->attr.font);
      buffer += 5;
      textlen -= 5;
    }
    else
    {
      size = *buffer++;
      textlen -= size;
      delta = *buffer++;
      if (delta)
	      fprintf (fp, "0 %d rmoveto\n", delta);
      showtext(fp, buffer, size);
      buffer += size;
      break;
    }
  }
  fprintf (fp, "grestore\n");
}

/* draw text, which is set up in chunks, and may be interspersed with
   font information */

void
polytext16 (FD fd,
	    FILE * fp,
            Window win,
            gcinfoPtr gc,
            int x,
            int y,
            int ntext,
            unsigned char *buffer)
{

    /* stubbed out, because I'm not sure how to do this */

}

/* draw text, which is set up in chunks, and may be interspersed with
   font information */

void
imagetext8 (FD fd,
	    FILE * fp,
            Window win,
            gcinfoPtr gc,
            int x,
            int y,
            int textlen,
            unsigned char *buffer)
{
  if (textlen > 0)
  {
      fontPtr font = findfont(gc->attr.font);

      if (font && font->font_data && font->font_data->font_info)
	  {
	      int dir, ascent, descent, height;
	      XCharStruct overall;
	      winfoPtr winfo;
	      FontData *f = font->font_data;
	      winfo = getwinfo(win, winfoTable);

	      XTextExtents(font->font_data->font_info,
			   (char *)buffer, textlen,
			   &dir, &ascent, &descent,
			   &overall);
	      fprintf (fp, "gsave\n");
	      if (winfo)
		      setcolor(winfo->fd, fp, gc->attr.background,
			       winfo->attr.colormap);
	      height = ascent + descent;
	      ascent = PIXELS(f, ascent, font->size);
	      descent = PIXELS(f, height, font->size) - ascent;
	      overall.width = PIXELS(f, overall.width, font->size);
	      fprintf (fp, "newpath\n");
	      fprintf (fp, "%d %d moveto\n", (int) x, (int)y-ascent);
	      fprintf (fp, "0 %d rlineto\n", (int) ascent+descent);
	      fprintf (fp, "%d 0 rlineto\n", (int) overall.width);
	      fprintf (fp, "0 %d rlineto\n", (int) -ascent-descent);
	      fprintf (fp, "closepath fill\n");
	      fprintf (fp, "grestore\n");
	  }

      fprintf (fp, "%d %d moveto\n", x, y);
      fprintf (fp, "gsave\n");
      fprintf (fp, "1 -1 scale\n");
      showtext(fp, buffer, textlen);
      fprintf (fp, "grestore\n");
  }
}

/* draw text, which is set up in chunks, and may be interspersed with
   font information */

void
imagetext16 (FD fd,
	     FILE * fp,
             Window win,
             gcinfoPtr gc,
             int x,
             int y,
             int ntext,
             unsigned char *buffer)
{

/* stubbed out, because I'm not sure how to do this */

#if 0
  char                    textbuf[512];
  int                     size;

  fprintf (fp, "%d %d moveto\n", x, y);
  while (ntext-- > 0)
  {

/* we'll skip over the font information for the moment */

    if (buffer[0] == 255)
    {
      buffer += 5;
    }
    else
    {
      size = *buffer++;
      strncpy (textbuf, buffer, size);
      fprintf (fp, "(%s) show\n", textbuf);
      buffer += size;
    }
  }
#endif
}

/* clear an area of a window ( draw a rectangle in the window's background 
   color). */

void
cleararea (FD fd,
	   FILE * fp,
           Window win,
           XWindowAttributes *attr,
           int x,
           int y,
           int width,
           int height)
{
/* actually, we don't want to do anything */

/* could just fill the rectangle with current gc... -- msa */

#if 0
  fprintf (fp, "%d %d moveto\n", x, y);
  fprintf (fp, "%d 0 rlineto\n", width);
  fprintf (fp, "0 %d rlineto\n", height);
  fprintf (fp, "%d 0 rlineto\n", -width);
  fprintf (fp, "closepath fill\n");
#endif
}

/* put an image */

void
putimage (FD fd,
	  FILE * fp,
          Window win,
          gcinfoPtr gc,
          int depth,
          int width,
          int height,
          int x,
          int y,
          int leftpad,
          int format,
          int numColors,
          XColor *colors,
          unsigned char *data)
{
    XColor                 *corrColors;
    int                     i;

    fprintf (fp, "gsave\n");

  /* if there's a color correction file, apply it to the colors in the image */


    if ( depth == 1)
	{
	    /* Use the current foreground/background colors */

	    XColor realColor[2];

	    /* ..will crash on TrueColor, and should also check the values
	       for validity? -- fix later, msa */

	    realColor[0] = colors[gc->attr.foreground];
	    realColor[1] = colors[gc->attr.background];

	    printImageGrey (data, x, y,
			    width, height, depth,
			    realColor, 2, fp,
			    PRINT_OPTIONS.indexCspaceSupport);
	}
    else if (depth == 8)
	{    	
	    if (PCS[fd].correctFile != NULL)
		{
		    corrColors = (XColor *) malloc(sizeof(XColor) * numColors);
		    for (i = 0; i < numColors; i++)
			{
			    corrColors[i] = colors[i];
			    correctColor (&corrColors[i], PCS[fd].numCorrect,
					  PCS[fd].corrects);
			}
		    printImageGrey (data, x, y,
				    width, height, depth,
				    corrColors, numColors, fp,
				    PRINT_OPTIONS.indexCspaceSupport);
		    free (corrColors);
		}
	    else
		{
		    printImageGrey (data, x, y,
				    width, height, depth,
				    colors, numColors, fp,
				    PRINT_OPTIONS.indexCspaceSupport);
		}
    }
    else if ( depth == 24)
	{
	    printImage24(data, x, y, width, height, depth, fp);
	}
    fprintf (fp, "grestore\n");
}

/* if the current GC pointer changes, reset it and set a new graphics state */

void
setCurrGc (FD fd,
           FILE * fp,
           gcinfoPtr gc,
           Colormap cmap)
{
  if (gc != PCS[fd].gc)
  {

/* plane mask, background and function haven't been addressed yet */

    setcolor(fd, fp, gc->attr.foreground, cmap);
    setlinewidth(fp, gc->attr.line_width);
    setlinecap(fp, gc->attr.cap_style);
    setlinejoin(fp, gc->attr.join_style);
    if (gc->attr.line_style == LineSolid)
	    setdash(fp, 0, (unsigned char *)gc->dashes, 0);
    else
	    setdash(fp, gc->numDashes, (unsigned char *)gc->dashes,
		    gc->attr.dash_offset);
    setfont(fp, fd, gc->attr.font);
    PCS[fd].gc = gc;
  }
}

/* set the values in a GC given the flags and the data buffer from a request,
   and if it's the current GC, set the value immediately */

void
setGcvalues (FD fd,
             FILE * fp,
             gcinfoPtr gc,
             unsigned long value_mask,
             unsigned long *request,  /* each val is a long */
             Colormap cmap)
{
  if (value_mask & GCFunction)
  {
    gc->attr.function = *request++;
  }
  if (value_mask & GCPlaneMask)
  {
    gc->attr.plane_mask = *request++;
  }

/* set the foreground color */

  if (value_mask & GCForeground)
  {
    gc->attr.foreground = *request++;
    if (gc == PCS[fd].gc)
    {
      if (gc->attr.foreground < NUM_COLORS)
      {
        setcolor (fd, fp, gc->attr.foreground, cmap);
      }
      else
      {
        fprintf (stderr, "pixel %ld out of range\n", gc->attr.foreground);
      }
    }
  }
  if (value_mask & GCBackground)
  {
    gc->attr.background = *request++;
  }
  if (value_mask & GCLineWidth)
  {
    gc->attr.line_width = *request++;
    if (gc->attr.line_width == 0) /* Force always 1 for 0! */
	gc->attr.line_width = 1;
  }
  if (value_mask & GCLineStyle)
  {
    gc->attr.line_style = *request++;
  }
  if (value_mask & GCCapStyle)
  {
    gc->attr.cap_style = *request++;
  }
  if (value_mask & GCJoinStyle)
  {
    gc->attr.join_style = *request++;
  }
  if (value_mask & GCFillStyle)
  {
    gc->attr.fill_style = *request++;
  }
  if (value_mask & GCFillRule)
  {
    gc->attr.fill_rule = *request++;
  }
  if (value_mask & GCTile)
  {
    gc->attr.tile = *request++;
  }
  if (value_mask & GCStipple)
  {
    gc->attr.stipple = *request++;
  }
  if (value_mask & GCTileStipXOrigin)
  {
    gc->attr.ts_x_origin = *request++;
  }
  if (value_mask & GCTileStipYOrigin)
  {
    gc->attr.ts_y_origin = *request++;
  }
  if (value_mask & GCFont)
  {
    gc->attr.font = *request++;
    setfont (fp, fd, gc->attr.font);
  }
  if (value_mask & GCSubwindowMode)
  {
    gc->attr.subwindow_mode = *request++;
  }
  if (value_mask & GCGraphicsExposures)
  {
    gc->attr.graphics_exposures = *request++;
  }
  if (value_mask & GCClipXOrigin)
  {
    gc->attr.clip_x_origin = *request++;
  }
  if (value_mask & GCClipYOrigin)
  {
    gc->attr.clip_y_origin = *request++;
  }
  if (value_mask & GCClipMask)
  {
    gc->attr.clip_mask = *request++;
  }
  if (value_mask & GCDashOffset)
  {
    gc->attr.dash_offset = *request++;
    /* changing dash_offset will implicitly reset the dashes too!! */
    if (gc->dashes)
	    free((char *)gc->dashes);
    gc->numDashes = 0;
  }
  if (value_mask & GCDashList)
  {
    gc->attr.dashes = *request++;
    if (gc->dashes)
	    free((char *)gc->dashes);
    if ((unsigned char)gc->attr.dashes > 0)
	{
	    gc->dashes = (char *)malloc(1);
	    gc->dashes[0] = gc->attr.dashes;
	    gc->numDashes = 1;
	}
    else
	{
	    gc->dashes = NULL;
	    gc->numDashes = 0;
	}
  }
  if (value_mask & GCArcMode)
  {
    gc->attr.arc_mode = *request++;
  }

/* if it's the current GC, then set the appropriate attributes */

  if (gc == PCS[fd].gc)
  {
    setcolor(fd, fp, gc->attr.foreground, cmap);
    setlinewidth(fp, gc->attr.line_width);
    setlinecap(fp, gc->attr.cap_style);
    setlinejoin (fp, gc->attr.join_style);
    if (gc->attr.line_style == LineSolid)
	    setdash(fp, 0, (unsigned char *)gc->dashes, 0);
    else
	    setdash(fp, gc->numDashes, (unsigned char *)gc->dashes,
		    gc->attr.dash_offset);
    setfont(fp, fd, gc->attr.font);
  }
}

/* set the values in a GC given the flags and the data buffer from a request,
   and if it's the current GC, set the value immediately */

void
copyGcvalues (gcinfoPtr gc,
              gcinfoPtr srcGc,
              unsigned long value_mask)
{
  if (value_mask & GCFunction)
  {
    gc->attr.function = srcGc->attr.function;
  }
  if (value_mask & GCPlaneMask)
  {
    gc->attr.plane_mask = srcGc->attr.plane_mask;
  }

/* set the foreground color */

  if (value_mask & GCForeground)
  {
    gc->attr.foreground = srcGc->attr.foreground;
  }
  if (value_mask & GCBackground)
  {
    gc->attr.background = srcGc->attr.background;
  }
  if (value_mask & GCLineWidth)
  {
    gc->attr.line_width = srcGc->attr.line_width;
  }
  if (value_mask & GCLineStyle)
  {
    gc->attr.line_style = srcGc->attr.line_style;
  }
  if (value_mask & GCCapStyle)
  {
    gc->attr.cap_style = srcGc->attr.cap_style;
  }
  if (value_mask & GCJoinStyle)
  {
    gc->attr.join_style = srcGc->attr.join_style;
  }
  if (value_mask & GCFillStyle)
  {
    gc->attr.fill_style = srcGc->attr.fill_style;
  }
  if (value_mask & GCFillRule)
  {
    gc->attr.fill_rule = srcGc->attr.fill_rule;
  }
  if (value_mask & GCTile)
  {
    gc->attr.tile = srcGc->attr.tile;
  }
  if (value_mask & GCStipple)
  {
    gc->attr.stipple = srcGc->attr.stipple;
  }
  if (value_mask & GCTileStipXOrigin)
  {
    gc->attr.ts_x_origin = srcGc->attr.ts_x_origin;
  }
  if (value_mask & GCTileStipYOrigin)
  {
    gc->attr.ts_y_origin = srcGc->attr.ts_y_origin;
  }
  if (value_mask & GCFont)
  {
    gc->attr.font = srcGc->attr.font;
  }
  if (value_mask & GCSubwindowMode)
  {
    gc->attr.subwindow_mode = srcGc->attr.subwindow_mode;
  }
  if (value_mask & GCGraphicsExposures)
  {
    gc->attr.graphics_exposures = srcGc->attr.graphics_exposures;
  }
  if (value_mask & GCClipXOrigin)
  {
    gc->attr.clip_x_origin = srcGc->attr.clip_x_origin;
  }
  if (value_mask & GCClipYOrigin)
  {
    gc->attr.clip_y_origin = srcGc->attr.clip_y_origin;
  }
  if (value_mask & GCClipMask)
  {
    gc->attr.clip_mask = srcGc->attr.clip_mask;
  }
  if (value_mask & GCDashOffset)
  {
    gc->attr.dash_offset = srcGc->attr.dash_offset;
  }
  if (value_mask & GCDashList)
  {
    if (gc->dashes)
	    free((char *)gc->dashes);
    gc->numDashes = srcGc->numDashes;
    if (srcGc->dashes && srcGc->numDashes > 0)
	{
	    gc->dashes = (char *)malloc(gc->numDashes);
	    memcpy((char *)gc->dashes, (char *)srcGc->dashes, gc->numDashes);
	}
    else
	    gc->dashes = NULL;
  }
  if (value_mask & GCArcMode)
  {
    gc->attr.arc_mode = srcGc->attr.arc_mode;
  }
}

/* copy the pixmap from the screen and draw it on the page */

void
copyarea (FD fd,
	  FILE * fp,
          Drawable src,
          Window win,
          int src_x,
          int src_y,
          int dstX,
          int dstY,
          int width,
          int height,
          int numColors,
          XColor *colors)
    {
    }

/* initialize the default values for the print clients */


void initClient( struct PrintClient *pcs)
{
    static PrintClient init_PrintClient;

    *pcs = init_PrintClient;

    /*
    ** Initialize specific non-zero/non-NULL values
    */
    pcs->fontScale = 1;		/* (not really used now -- msa) */
    pcs->pixmapScale = 1;	/* (not really used now -- msa) */
    pcs->pageWidth = 612;	/* 8 1/2 x 11 is default size */
    pcs->pageHeight = 792;
}

void
initPrintClients (
                   struct PrintClient pcs[],
                   const int numclients)

{
  int                     ii = 0;

  for (; ii < numclients; ii++) {
    initClient( &pcs[ ii]);
  }
}

/*- end drawcmd.c
*/
