#include <stdio.h>
#include <stdlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xew/Print.h>
#include <X11/Xew/Frame.h>
#include <X11/Xew/TextEd.h>
#include "printing.h"


#if !USING_PRINTSERVER && !USING_XPRINTER

/*
**	*****************************************************************
**	* A very basic testing tool. The printing is simply directed to	*
**	* the current display onto a separate simple window.		*
**	*****************************************************************
*/
#include "ui_tools.h"

static XePrintProcess pp;
static XePrinter printer;

static void PrintNext(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	XClearWindow(printer.printer, printer.page);
	if (pp)
		XePrintResume(pp);
	else
		UI_Popdown(w, NULL, NULL);
    }

static void ClosePrint(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	XePrintCancel(pp);
	UI_Popdown(w, NULL, NULL);
    }

static void CreatePrintWindow(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	UI_ElementDef *r = (UI_ElementDef *)call_data;

	r->widget = XtVaCreateManagedWidget
		("printWindow", coreWidgetClass, w,
		 XtNwidth, printer.page_width,
		 XtNheight, printer.page_height,
		 (XtPointer)NULL);
    }

extern UI_PanelDef popup_print_window;

static UI_ActionDef print_actions[] =
    {
	{"page",	PrintNext, (XtPointer)&popup_print_window},
	{"cancel",	ClosePrint, (XtPointer)&popup_print_window},
	{NULL},
    };

static UI_ElementDef print_window_pane[] =
    {
	UI_BLANK("print", CreatePrintWindow),
	UI_END(),
    };

UI_PanelDef popup_print_window =
    {
	"printWindow",
	print_window_pane,
	print_actions,
	"Print Window",
    };

static void PrintingPage(w, _pp, page, data)
Widget w;
XePrintProcess _pp;
int page;
XtPointer data;
    {
	char buf[100];

	if (page > 0)
	    {
		sprintf(buf, "Page %d", page);
		if (page > 1)
			XePrintPause(_pp);
	    }
	else
	    {
		pp = NULL;
		strcpy(buf, "Done");
	    }
	XtVaSetValues(print_actions[0].widget,
		      XtNlabel, buf,
		      (XtPointer)NULL);
    }

/*
** PrintToDisplay
**	a temporary test function for printing support. The client_data
**	is a *pointer* to Widget *pointer*!
*/
void PrintToDisplay(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	
	int c;
	Pixel black, white;
	Widget *wp, pane, panel;

	wp = (Widget *)client_data;
	if (!wp || !*wp)
		return;
	printer.printer = XtDisplay(*wp);
	printer.page_width = 600;
	printer.page_height = 600;
	black = BlackPixel(printer.printer, DefaultScreen(printer.printer));
	white = WhitePixel(printer.printer, DefaultScreen(printer.printer));
	UI_PopupPanel(*wp, (XtPointer)&popup_print_window,
		      (XtPointer)&popup_print_window);

	pane = XtNameToWidget(popup_print_window.shell, "*printWindow");
	XSync(printer.printer, False);	/* Wait for the print window kludge
					   to be ready... */
	printer.page = XtWindow(pane);

	if (!pp)
	    {
		XClearWindow(printer.printer, printer.page);
		pp = XePrint(*wp, &printer, PrintingPage, (XtPointer)NULL);
	    }
    }
#endif


#if USING_PRINTSERVER

#include "ui_tools.h"
#include "printlib.h"
#include <setjmp.h>

/*
**	*****************************************************************
**	* The printing is directed to a separate printing server that	*
**	* looks almost like another X Display				*
**	*****************************************************************
*/
extern UI_PanelDef popup_print_control;
static char popup_print_heading[300];
static XePrintProcess current_print_job;
static char *current_print_id;
static int print_page_count;
static XtWorkProcId current_work_proc;

static jmp_buf catch_dead_server;

/*
** PrintIOErrorHandler
**	try to handle the case where print server is lost...
*/
static int PrintIOErrorHandler(display, event)
Display *display;
XErrorEvent *event;
    {
	longjmp(catch_dead_server, 1);
    }

/*
** ResumePrinting
**	a XtWorkProc that will actually start the printing of each
**	page. The "work process" solution is required because otherwise
**	the user would have very little chance to abort the printing
**	process because X event loop may be totally "busy" with the
**	printing.
*/
static Boolean ResumePrinting(client_data)
XtPointer client_data;
    {
	XePrintProcess pp = (XePrintProcess)client_data;

	if (pp && pp == current_print_job)
	    {
		print_page_count += 1;
		XePrintResume(pp);
	    }
	current_work_proc = 0;
	return True;	/* One shot action! */
    }

static void PrintingPage(w, pp, page, data)
Widget w;
XePrintProcess pp;
int page;
XtPointer data;
    {
	XePrinter *p = (XePrinter *)data;
	char page_file[30];
	
	if (page != 1)
	    {
		XClosePrintWindow(p->printer, p->page);
		XFlush(p->printer);
	    }
	if (page > 0)
	    {
		sprintf(popup_print_heading,
			"Printing '%s', page %d ...", current_print_id, page);
		sprintf(page_file, "page-%d.ps", page);
		XOpenPrintWindow(p->printer, p->page, page_file);
	    }
	else
	    {
		current_print_job = NULL;
		sprintf(popup_print_heading,
			"Printing '%s' completed, %d %s",
			current_print_id, print_page_count,
			print_page_count == 1 ? "page" : "pages");
	    }
	if (current_work_proc)
		XtRemoveWorkProc(current_work_proc);
	/*
	** This UI_PopupPanel is mainly to get the changed heading updated
	*/
	UI_PopupPanel(NULL, /* Panel should exist! no parent needed! */
		      (XtPointer)&popup_print_control,
		      (XtPointer)&popup_print_control);
	XFlush(XtDisplay(w)); /* ..or perhaps should use XSynch()? */
	/*
	** Arrange a pause to allow user interaction and expose events
	** to complete in the popup_print_control. Printing will resume
	** when the work procedure gets called.
	*/
	if (page != 0)
	    {
		XePrintPause(pp);
		current_work_proc = XtAppAddWorkProc
			(XtWidgetToApplicationContext(w),ResumePrinting,
			 (XtPointer)pp);
	    }
    }


typedef enum 
    {
	Toggle_Portrait,
	Toggle_Landscape,
	Toggle_A4,
	Toggle_A5,
	Toggle_A3,
	Toggle_Letter,
	Toggle_Widget,
	Toggle_UserSize,
	Toggle_ExactFit,
	Toggle_MaxAspect,
	Toggle_FitWidth,
	Toggle_FitHeight,
	Toggle_FitNone,

	Toggle_Background,
	Toggle_Dither,
	Toggle_Mono,
	Toggle_Gray,
	Toggle_Color

    } ToggleIds;

/*
** SetInitialToggle
*/
static void SetInitialToggle(panel, id)
UI_PanelDef *panel;
int id;
    {
	UI_ElementDef *e;
	for (e = panel->elements; e && e->op != UI_EndOp; e++)
	    {
		if ((int)e->data == id &&
		    e->widget &&
		    (e->op == UI_ToggleOp || e->op == UI_RadioOp))
		    {
			XtVaSetValues(e->widget,
				      XtNstate, True,
				      (XtPointer)NULL);
		    }
	    }
	
    }
/*
** CancelPrint
*/
static void CancelPrint(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	if (current_work_proc)
	    {
		XtRemoveWorkProc(current_work_proc);
		current_work_proc = 0;
	    }
	if (!current_print_job)
		UI_WarningMessage(w, "No printing active!", (char **)NULL, 0);
	else
		XePrintCancel(current_print_job);
	/* Do not touch current_print_job here, the cancel will call
	   the callback with page==0 eventually... */
    }

/*
** StartPrint
*/
static void StartPrint(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	static XePrinter p;

	Display *display = XtDisplay(w);
	int screen = DefaultScreen(display);
	
	UI_PanelDef *panel = (UI_PanelDef *)client_data;
	UI_ElementDef *e;
	int landscape = 0, page_width = 210, page_height = 297, fitting = 1;
	int mm_width = DisplayWidthMM(display, screen);
	int mm_height = DisplayHeightMM(display, screen);
	int screen_width = DisplayWidth(display, screen);
	int screen_height = DisplayHeight(display, screen);
	int page_pixels = 0;
	
	if (!panel || !panel->target)
		return;
	if (current_print_job)
	    {
		UI_WarningMessage
			(w, "Previous printing has not completed yet!",
			 (char **)NULL, 0);
		return;
	    }
	/*
	** Set defaults for the cases where no toggles might be set
	*/
	p.imaging.dither = XeDither_NONE;
	p.background = False;
	p.imaging.color_mode = XeColorMode_COLOR;

	for (e = panel->elements; e && e->op != UI_EndOp; e++)
	    {
		Boolean state;
		char *s;

		if (e->op != UI_ToggleOp && e->op != UI_RadioOp)
			continue;
		if (!e->widget)
			continue;
		XtVaGetValues(e->widget, XtNstate, &state, (XtPointer)NULL);
		if (!state)
			continue;
		switch ((ToggleIds)e->data)
		    {
		    case Toggle_Portrait:
			landscape = 0;
			break;
		    case Toggle_Landscape:
			landscape = 1;
			break;
		    case Toggle_A4:
			page_width = 210;
			page_height = 297;
			break;
		    case Toggle_A5:
			page_width = 297 / 2;
			page_height = 210;
			break;
		    case Toggle_A3:
			page_width = 297;
			page_height = 2 * 210;
			break;
		    case Toggle_Letter:
			page_width = 216;
			page_height = 279;
			break;
		    case Toggle_Widget:
			    {
				Dimension ww, wh;
				XtVaGetValues(panel->target,
					      XtNwidth, &ww,
					      XtNheight, &wh,
					      (XtPointer)NULL);
				page_width = ww;
				page_height = wh;
				page_pixels = TRUE;
			    }
			break;
		    case Toggle_UserSize:
			/*
			** Just assume the next two elements are input
			** areas which contain the user defined dimensions
			*/
			if (e[1].op != UI_InputOp || e[2].op != UI_InputOp)
				break;
			if (!e[1].widget || !e[2].widget)
				break;
			s = XeTextGetString(e[1].widget, (long *)NULL);
			if (s)
				page_width = atoi(s);
			free(s);
			s = XeTextGetString(e[2].widget, (long *)NULL);
			if (s)
				page_height = atoi(s);
			break;
		    case Toggle_ExactFit:
			fitting = XpScale_FITAREA;
			break;
		    case Toggle_MaxAspect:
			fitting = XpScale_MAXASPECT;
			break;
		    case Toggle_FitWidth:
			fitting = XpScale_WIDTH;
			break;
		    case Toggle_FitHeight:
			fitting = XpScale_HEIGHT;
			break;
		    case Toggle_FitNone:
			fitting = XpScale_NONE;
			break;
		    case Toggle_Background:
			p.background = True;
			break;
		    case Toggle_Dither:
			p.imaging.dither = XeDither_FS4;
			break;
		    case Toggle_Mono:
			p.imaging.color_mode = XeColorMode_MONO;
			break;
		    case Toggle_Gray:
			p.imaging.color_mode = XeColorMode_GRAY;
			break;
		    case Toggle_Color:
			p.imaging.color_mode = XeColorMode_COLOR;
			break;
		    default:
			break;
		    }
	    }
	if (!page_pixels)
	    {
		/*
		** ..for now, just assume margins of 5mm all around on any page
		*/
		page_width -= 10;
		page_height -= 10;
		/*
		** If the physical dimensions of the screen are not known,
		** just assume 75dpi
		*/
		if (mm_width <= 0)
		    {
			screen_width = 75;
			mm_width = 25;
		    }
		if (mm_height <= 0)
		    {
			screen_height = 75;
			mm_height = 25;
		    }
		page_width =
			(page_width * screen_width + screen_width/2) /
				mm_width;
		page_height =
			(page_height * screen_height + screen_height/2) /
				mm_height;
	    }
	if (page_height <= 0 || page_width <= 0)
	    {
		char buf[2*sizeof(int)*3+10], *ptr;
		ptr = buf;

		sprintf(buf, "%d x %d", page_width, page_height);
		UI_WarningMessage(w, "Invalid page size %s", &ptr, 1);
		return;
	    }
	if (landscape)
	    {
		p.page_height = page_width;
		p.page_width = page_height;
	    }
	else
	    {
		p.page_width = page_width;
		p.page_height = page_height;
	    }
	if (!p.printer)
	    {
		char *pname = getenv("PRINTSERVER");
		if (!pname)
		    {
			UI_WarningMessage
				(w, "No PRINTSERVER environment definition",
				 (char **)NULL, 0);
			return;
		    }
		p.printer = XOpenDisplay(pname);
		if (!p.printer)
		    {
			UI_WarningMessage
				(w, "Cannot connect print server %s",&pname,1);
			return;
		    }
		p.page = DefaultRootWindow(p.printer);
	    }
	/*
	** Do some trickery with IO Error handler and try to detect
	** if print server has died since the last use..
	*/
	    {
		int (*handler)();


		if (setjmp(catch_dead_server))
		    {
			p.printer = NULL;
			UI_WarningMessage(w,"Lost connection to the print server?",
					  (char **)NULL, 0);
			return;
		    }
		handler = XSetIOErrorHandler(PrintIOErrorHandler);
		XSetPrintParams (p.printer, fitting, 1, 1,
				 page_width, page_height,
				 0, landscape, NULL);
		XSync(p.printer, False);
		XSetIOErrorHandler(handler);
	    }
	if (current_work_proc)
	    {
		XtRemoveWorkProc(current_work_proc);
		current_work_proc = 0;
	    }
	current_print_id = XtName(panel->target);
	print_page_count = 0;
	sprintf(popup_print_control.heading,
		"Printing '%s', initializing...", current_print_id);
	/*
	** This UI_PopupPanel is mainly to get the changed heading updated
	*/
	UI_PopupPanel(NULL, /* Panel should exist! no parent needed! */
		      (XtPointer)&popup_print_control,
		      (XtPointer)&popup_print_control);
	XFlush(XtDisplay(w)); /* ..or perhaps should use XSynch()? */
	current_print_job = XePrint
		(panel->target, &p, PrintingPage, (XtPointer)&p);
    }



static char command_name[] = "commandButton";
static char panel_name[] = "menuPanel";
static char toggle_name[] = "toggleBox";
static char input_name[] = "inputBox";
static char label_name[] = "labelBox";


static UI_ActionDef print_actions[] =
    {
	{"start print",	StartPrint, (XtPointer)&popup_print_control},
	{"cancel print", CancelPrint, (XtPointer)&popup_print_control},
	{"close"},
	{NULL},
    };


static UI_ElementDef print_window_pane[] =
    {
	UI_PANE(panel_name,	"Page definition"),
	UI_LABEL(label_name,	"orientation"),
	UI_TOGGLE(toggle_name,	"portrait", Toggle_Portrait, NULL),
	UI_RADIO(toggle_name,	"landscape", Toggle_Landscape, NULL),
	UI_LABEL(label_name,	"page size"),
	UI_TOGGLE(toggle_name,	"A4", Toggle_A4, NULL),
	UI_RADIO(toggle_name,	"A5", Toggle_A5, NULL),
	UI_RADIO(toggle_name,	"A3", Toggle_A3, NULL),
	UI_RADIO(toggle_name,	"Letter", Toggle_Letter, NULL),
	UI_RADIO(toggle_name,	"Widget to be printed", Toggle_Widget, NULL),
	UI_RADIO(toggle_name,	"user defined (below)", Toggle_UserSize, NULL),
	UI_INPUT(input_name,	" page width [mm]:", NULL),
	UI_INPUT(input_name,	" page height [mm]:", NULL),
	UI_PANE(panel_name,	"Page mapping to the printing area"),
	UI_TOGGLE(toggle_name,	"exact fit, change aspect ratio",
		  Toggle_ExactFit, NULL),
	UI_RADIO(toggle_name,	"fit preserving the aspect ratio",
		 Toggle_MaxAspect, NULL),
	UI_RADIO(toggle_name,	"fit by width", Toggle_FitWidth, NULL),
	UI_RADIO(toggle_name,	"fit by height", Toggle_FitHeight, NULL),
	UI_RADIO(toggle_name,	"absolute (1 pixel is 1/72 inch)",
		 Toggle_FitNone, NULL),
	UI_LABEL(label_name,	"Imaging parameters:"),
	UI_TOGGLE(toggle_name,	"print background", Toggle_Background, NULL),
	UI_TOGGLE(toggle_name,	"dither with FS4", Toggle_Dither, NULL),
	UI_TOGGLE(toggle_name,	"mono", Toggle_Mono, NULL),
	UI_RADIO(toggle_name,	"gray", Toggle_Gray, NULL),
	UI_RADIO(toggle_name,	"color", Toggle_Color, NULL),
	UI_END(),
    };

UI_PanelDef popup_print_control =
    {
	"printControl",
	print_window_pane,
	print_actions,
	popup_print_heading,
    };

/*
** RemovePanelTarget
**	a callback function that removes the Widget from the panel
**	target.
*/
static void RemovePanelTarget(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	UI_PanelDef *panel = (UI_PanelDef *)client_data;

	if (panel && panel->target == w)
		panel->target = None;
    }

/*
** ChangePanelTarget
**	Change the target widget of a panel. Primarily, this function
**	adds a Destroy Callback to the widget for removing the reference
**	from the panel, if widget gets destroyed.
*/
static void ChangePanelTarget(w, panel)
Widget w;
UI_PanelDef *panel;
    {
	if (panel->target)
		XtRemoveCallback(panel->target, XtNdestroyCallback,
				 RemovePanelTarget, (XtPointer)panel);
	if ((panel->target = w) != 0)
		XtAddCallback(w, XtNdestroyCallback, RemovePanelTarget,
			      (XtPointer)panel);
    }

/*
** PrintToDisplay
**	a temporary test function for printing support. The client_data
**	is a *pointer* to Widget *pointer*!
*/
void PrintToDisplay(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	Widget *wp;
	int first_time;
	wp = (Widget *)client_data;
	if (!wp || !*wp)
		return;
	ChangePanelTarget(*wp, &popup_print_control);
	first_time = popup_print_control.shell == 0;
	sprintf(popup_print_control.heading,
		"Printing '%s', choose options and then start",
		XtName(*wp));
	UI_PopupPanel(w, (XtPointer)&popup_print_control, (XtPointer)NULL);
	if (first_time)
	    {
		SetInitialToggle(&popup_print_control, Toggle_Portrait);
		SetInitialToggle(&popup_print_control, Toggle_A4);
		SetInitialToggle(&popup_print_control, Toggle_MaxAspect);
		SetInitialToggle(&popup_print_control, Toggle_Dither);
		SetInitialToggle(&popup_print_control, Toggle_Color);
	    }
    }
#endif



#if USING_XPRINTER
/*
**	*****************************************************************
**	* The printing is directed implemened using Bristol Incs	*
**	* XPRINTER library.						*
**	*****************************************************************
*/

#include <Xp.h>
#include <XpMacros.h>

static void PrintingPage(w, pp, page, data)
Widget w;
XePrintProcess pp;
int page;
XtPointer data;
    {
	XePrinter *p = (XePrinter *)data;

	if (page > 0)
	    {
		if (page != 1)
			XpPageEject(p->printer);
		printf("Printing page %d.\n", page);
	    }
	else
	    {
		XpEndPage(p->printer);
		XpEndDoc(p->printer);
		XpClosePrinter(p->printer);
		p->printer = NULL;
		printf("Print completed.\n");
	    }
    }

/*
** PrintToDisplay
**	a temporary test function for printing support. The client_data
**	is a *pointer* to Widget *pointer*!
*/
void PrintToDisplay(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
    {
	static XePrinter p;
	int c;
	Pixel black, white;
	Widget *wp;
	XpPageInfo page_info;
	double scale;

	wp = (Widget *)client_data;
	if (!wp || !*wp)
		return;

	if (!p.printer)
	    {
		p.printer = XpOpenPrinter();
		p.page = 0;
	    }
	XpMatch(XtDisplay(*wp), 0, p.printer);
	XpStartDoc(p.printer, "SAMPLE");
	XpStartPage(p.printer);
	XpQueryPageSize(p.printer, XpGetPageSize(p.printer), &page_info);
	scale = XpGetScale(p.printer);
	p.page_height = XpDisplayHeight(p.printer, DefaultScreen(p.printer));
	p.page_width = XpDisplayWidth(p.printer, DefaultScreen(p.printer));
	(void)XePrint(*wp, &p, PrintingPage, (XtPointer)&p);
    }
#endif


