.\"remove .ig hn for full docs
.de hi
.ig eh
..
.de eh
..
.TH "" 3 "" "Version 3.0" "Free Widget Foundation"
.SH NAME
XfwfAnsiTerm
.SH DESCRIPTION
The AnsiTerm widget emulates an ANSI terminal in an X Window.
It displays lines of text and scrolls them if necessary. It
displays a cursor, to indicate where the next text will appear
and it interprets the ANSI escape codes to position the cursor
or highlight the text.

Text is written to the widget by calling \fIXfwfAnsiWrite\fP (a
convenience function which calls the \fIwrite\fP method).

A callback is provided for reacting to key presses. The widget
also supports the selection mechanism, the same way the xterm
program does (select by dragging with mouse button 1 pressed,
paste by clicking with mouse button 2)

Another callback is provided to notify the application when the
number of rows or columns changes. If the application runs a
shell or some other program in the background, it can use this
callback to send a \fISIGWINCH\fP signal to the child process.

.SS "Public variables"

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
XfwfAnsiTerm
Name	Class	Type	Default
XtNrows	XtCRows	int 	24 
XtNcolumns	XtCColumns	int 	80 
XtNfont	XtCFont	String 	XtDefaultFont 
XtNboldfont	XtCBoldfont	XFontStruct *	infer_bold 
XtNmargin	XtCMargin	int 	10 
XtNforeground	XtCForeground	Pixel 	XtDefaultForeground 
XtNkeyCallback	XtCKeyCallback	Callback	NULL 
XtNresizeCallback	XtCResizeCallback	Callback	NULL 

.TE
.ps +2

.TP
.I "XtNrows"
The \fIrows\fP resource is used to set the number of lines of
text. The widget's \fIheight\fP is adjusted accordingly. If the
height changes for any other reason, the \fIrows\fP resource is
recomputed to reflect the actual number of lines, except that
the value will always be at least 1.

	

.hi
int  rows = 24 
.eh

.TP
.I "XtNcolumns"
The \fIcolumns\fP resource sets the width of the widget to a
certain number of characters. If the widget is later resized by
any other means, the value of \fIcolumns\fP is recomputed to reflect
the actual nummber of columns, but always at least 1.

	

.hi
int  columns = 80 
.eh

.TP
.I "XtNfont"
The widget uses two fonts: normal and bold. They should
usually be monospaced fonts. Note that \fIfont\fP is a string, not a
\fIXFontStruct\fP; this is to enable the widget to derive the name
of the bold version from this string.

	

.hi
String  font = XtDefaultFont 
.eh

.TP
.I "XtNboldfont"
The \fIboldfont\fP is by default computed on the basis of the
\fIfont\fP resource.

	

.hi
XFontStruct * boldfont = <CallProc>infer_bold 
.eh

.TP
.I "XtNmargin"
The margin between the frame and the text, on all sides.

	

.hi
int  margin = 10 
.eh

.TP
.I "XtNforeground"
The color of the text.

	

.hi
Pixel  foreground = <String>XtDefaultForeground 
.eh

.TP
.I "XtNkeyCallback"
The \fIkeyCallback\fP is called whenever the widget receives a key
press. The callback is called by the \fIkey\fP action procedure. The
\fIcall_data\fP argument contains a pointer to the character that
was pressed (type: \fIchar *\fP). (If the pressed key has no
representation in the form of one or more characters, the
callback is not called.)

	

.hi
<Callback> XtCallbackList  keyCallback = NULL 
.eh

.TP
.I "XtNresizeCallback"
The \fIresizeCallback\fP is invoked when the widget changes the
number of rows or columns. The \fIcall_data\fP holds a pointer to a
struct \fIXfwfResizeInfo\fP containing two ints. The first is
the number of rows, the second the number of columns

	

.hi
<Callback> XtCallbackList  resizeCallback = NULL 
.eh

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
XfwfBoard
Name	Class	Type	Default
XtNabs_x	XtCAbs_x	Position 	0 
XtNrel_x	XtCRel_x	Float 	"0.0"
XtNabs_y	XtCAbs_y	Position 	0 
XtNrel_y	XtCRel_y	Float 	"0.0"
XtNabs_width	XtCAbs_width	Position 	0 
XtNrel_width	XtCRel_width	Float 	"1.0"
XtNabs_height	XtCAbs_height	Position 	0 
XtNrel_height	XtCRel_height	Float 	"1.0"
XtNhunit	XtCHunit	Float 	"1.0"
XtNvunit	XtCVunit	Float 	"1.0"
XtNlocation	XtCLocation	String 	NULL 

.TE
.ps +2

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
XfwfFrame
Name	Class	Type	Default
XtNcursor	XtCCursor	Cursor 	None 
XtNframeType	XtCFrameType	FrameType 	XfwfRaised 
XtNframeWidth	XtCFrameWidth	Dimension 	0 
XtNouterOffset	XtCOuterOffset	Dimension 	0 
XtNinnerOffset	XtCInnerOffset	Dimension 	0 
XtNshadowScheme	XtCShadowScheme	ShadowScheme 	XfwfAuto 
XtNtopShadowColor	XtCTopShadowColor	Pixel 	compute_topcolor 
XtNbottomShadowColor	XtCBottomShadowColor	Pixel 	compute_bottomcolor 
XtNtopShadowStipple	XtCTopShadowStipple	Bitmap 	NULL 
XtNbottomShadowStipple	XtCBottomShadowStipple	Bitmap 	NULL 

.TE
.ps +2

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
XfwfCommon
Name	Class	Type	Default
XtNtraversalOn	XtCTraversalOn	Boolean 	True 
XtNhighlightThickness	XtCHighlightThickness	Dimension 	2 
XtNhighlightColor	XtCHighlightColor	Pixel 	XtDefaultForeground 
XtNhighlightPixmap	XtCHighlightPixmap	Pixmap 	None 
XtNnextTop	XtCNextTop	Callback	NULL 
XtNuserData	XtCUserData	Pointer	NULL 

.TE
.ps +2

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
Composite
Name	Class	Type	Default
XtNchildren	XtCChildren	WidgetList 	NULL 
insertPosition	XtCInsertPosition	XTOrderProc 	NULL 
numChildren	XtCNumChildren	Cardinal 	0 

.TE
.ps +2

.ps -2
.TS
center box;
cBsss
lB|lB|lB|lB
l|l|l|l.
Core
Name	Class	Type	Default
XtNx	XtCX	Position 	0 
XtNy	XtCY	Position 	0 
XtNwidth	XtCWidth	Dimension 	0 
XtNheight	XtCHeight	Dimension 	0 
borderWidth	XtCBorderWidth	Dimension 	0 
XtNcolormap	XtCColormap	Colormap 	NULL 
XtNdepth	XtCDepth	Int 	0 
destroyCallback	XtCDestroyCallback	XTCallbackList 	NULL 
XtNsensitive	XtCSensitive	Boolean 	True 
XtNtm	XtCTm	XTTMRec 	NULL 
ancestorSensitive	XtCAncestorSensitive	Boolean 	False 
accelerators	XtCAccelerators	XTTranslations 	NULL 
borderColor	XtCBorderColor	Pixel 	0 
borderPixmap	XtCBorderPixmap	Pixmap 	NULL 
background	XtCBackground	Pixel 	0 
backgroundPixmap	XtCBackgroundPixmap	Pixmap 	NULL 
mappedWhenManaged	XtCMappedWhenManaged	Boolean 	True 
XtNscreen	XtCScreen	Screen *	NULL 

.TE
.ps +2

.SS "Exports"

The \fIXfwfAnsiWrite\fP function sends characters to the AnsiTerm
widget. It is a convenience function that simply calls the \fIwrite\fP
method of the widget.

.nf
void  XfwfAnsiWrite( $, char * buf, int  nbytes)
.fi

.hi
{
    if (! XtIsSubclass($, xfwfAnsiTermWidgetClass))
	XtAppError(XtWidgetToApplicationContext($),
		   "XfwfAnsiWrite may only be called for AnsiTerm widgets.");
    $write($, buf, nbytes);
}
.eh

The \fIresizeCallback\fP is passed a pointer to an
\fIXfwfResizeInfo\fP struct.

	

.nf

.B type
 XfwfResizeInfo = struct {
	    int rows, columns;
	}
.fi

.SS "Translations"

.nf
<Key>: key() 
.fi

.nf
Shift<Btn1Down>: extend_selection() 
.fi

.nf
<Btn1Down>: start_selection() 
.fi

.nf
<Btn1Motion>: extend_selection() 
.fi

.nf
<Btn1Up>: end_selection() 
.fi

.nf
<Btn2Down>: paste_selection() 
.fi

.hi
.SS "Actions"

.TP
.I "key

The \fIkey\fP action procedure calls the \fIkeyCallback\fP callback
procedure with the pressed key as \fIcall_data\fP. Nothing happens
if the key doesn't have a representation as (a sequence of)
characters.

If \fIXLookupString\fP returns a character representation, that
representation is used, otherwise the keysym is checked for some
special keys (Return, Backspace, etc.). Finally, if the user
pressed Control + letter, the character returned is (letter -
64).

.hi

.nf
void key($, XEvent* event, String* params, Cardinal* num_params)
{
    char buf[15];
    KeySym keysym;
    int n, i;
    
    if (event->type != KeyPress  event->type != KeyRelease)
	XtError("key() action in AnsiTerm not associated with key events");

    n = XLookupString(event->xkey, buf, sizeof(buf), keysym, NULL);
    if (n == 0)
	switch (keysym) {
	case XK_BackSpace:  n = 1; buf[0] = '\\010';	    break;
	case XK_Home:	    n = 3; strcpy(buf, "\\033[H");   break;
	case XK_Down:	    n = 3; strcpy(buf, "\\033[B");   break;
	case XK_Left:	    n = 3; strcpy(buf, "\\033[D");   break;
	case XK_Right:	    n = 3; strcpy(buf, "\\033[C");   break;
	case XK_Up:	    n = 3; strcpy(buf, "\\033[A");   break;
	case XK_Return:
	case XK_KP_Enter:
	case XK_Linefeed:   n = 1; buf[0] = '\\015';	    break;
	case XK_Tab:	    n = 1; buf[0] = '\\011';	    break;
	case XK_Delete:     n = 1; buf[0] = '\\177';	    break;
	default:
	    if ((event->xkey.state  ControlMask)
		 65 <= keysym  keysym <= 95) {
		buf[n++] = keysym - 64;	/* Control + letter */
	    }
	}
    for (i = 0; i < n; i++) XtCallCallbackList($, $keyCallback, buf[i]);
}
.fi

.eh

.TP
.I "start_selection

The selection mechanism consist of three action procedures for
highlighting a selection and one for pasting text into the
widget.

\fIstart_selection\fP establishes the start of the highlighted
selection from the mouse position.  \fIextend_selection\fP
highlights the text between the start position and the current
mouse position. \fIend_selection\fP copies the highlighted part to a
buffer and notifies the X server that this widget wants to be
the current selection owner.

\fIpaste_selection\fP requests the current selection from whatever
application has it and processes it as if the user had typed it;
i.e., it calls the \fIkeyCallback\fP for every character.

.hi

.nf
void start_selection($, XEvent* event, String* params, Cardinal* num_params)
{
    find_cell($, event->xbutton.x, event->xbutton.y, $start_x, $start_y);
    $end_x = $start_x;
    $end_y = $start_y;
    $drag_started = FALSE;			/* No selection yet */
}
.fi

.eh

.TP
.I "extend_selection

\fIextend_selection\fP is called when the mouse is dragged over
the text. It finds the cell that the mouse is on and highlights
all cells from the one where the drag started to the current one
(both inclusive).

The function only draws the lines that the mouse passed between
the previous and the current event.

.hi

.nf
void extend_selection($, XEvent* event, String* params, Cardinal* num_params)
{
    int i, j, framewd, sx, sy, ex, ey, x, y;

    $drag_started = TRUE;

    find_cell($, event->xbutton.x, event->xbutton.y, x, y);

    if ($start_y < y
        || ($start_y == y  $start_x < x)) {
        sy = $start_y;  ey = y;
        sx = $start_x;  ex = x;
    } else {                                   /* Swap start and end */
        sy = y;  ey = $start_y;
        sx = x;  ex = $start_x;
    }    
    framewd = compute_framewd($);

    /* Draw the lines that the mouse passed over since last event */
    for (i = min(y, $end_y); i <= max(y, $end_y); i++) {

        /* Toggle REV attribute on cells within selected region */
        if (i == sy  i == ey)
            for (j = sx; j <= ex; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (i == sy)
            for (j = sx; j < $columns; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (i == ey)
            for (j = 0; j < ex; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (sy < i  i < ey)
            for (j = 0; j < $columns; j++) $attribs[i][j] ^= ATTRIB_REV;

        draw_line($, framewd, i, 0, $columns);	/* Draw the line */

        /* Toggle REV attribute on cells in region again */
        if (i == sy  i == ey)
            for (j = sx; j <= ex; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (i == sy)
            for (j = sx; j < $columns; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (i == ey)
            for (j = 0; j < ex; j++) $attribs[i][j] ^= ATTRIB_REV;
        else if (sy < i  i < ey)
            for (j = 0; j < $columns; j++) $attribs[i][j] ^= ATTRIB_REV;
    }

    $end_x = x;
    $end_y = y;
}
.fi

.eh

.TP
.I "end_selection

When the mouse button is released, the selection is complete
and the highlighted part is copied to a buffer. The widget then
tells the X server that it wants to become the selection owner
for the PRIMARY selection.

A simple click should not be taken as a selection. If there has
been no movement between the mouse press and release, the
\fIend_selection\fP action simply returns without doing anything.

The highlight is immediately removed from the screen. This is
easy to implement but it removes a visual indicator that some
people may want to leave there while the selection is active.
Something to reconsider for the next version?

.hi

.nf
void end_selection($, XEvent* event, String* params, Cardinal* num_params)
{
    int sx, sy, ex, ey, k, i, j, framewd;

    if (! $drag_started) return;		/* No movement since BtnDown */

    if ($start_y < $end_y
        || ($start_y == $end_y  $start_x < $end_x)) {
        sy = $start_y;  ey = $end_y;
        sx = $start_x;  ex = $end_x;
    } else {					/* Swap start  end */
        sy = $end_y;  ey = $start_y;
        sx = $end_x;  ex = $start_x;
    }

    /* Unhighlight the selection */
    framewd = compute_framewd($);
    if (sy == ey) {
        draw_line($, framewd, sy, sx, ex + 1);
    } else {
        draw_line($, framewd, sy, sx, $columns); /* First (partial) line */
        for (i = sy + 1; i <= ey - 1; i++)      /* Middle lines */
            draw_line($, framewd, i, 0, $columns);
        draw_line($, framewd, ey, 0, ex + 1);   /* Last (partial) line */
    }

    /* Copy selection to buffer */
    if (sy == ey) {

        $selection = XtRealloc($selection, ex - sx + 2);
        for (k = 0, j = sx; j <= ex; k++, j++)
            $selection[k] = $contents[sy][j];
        $selection[k] = '\\0';
        $selection_len = k;

    } else {

        $selection = XtRealloc($selection,
            $columns - sx + 1 + (ey - sy - 1) * ($columns + 1) + ex + 3);
        k = 0;
        for (j = sx; j < $columns; j++, k++)
            $selection[k] = $contents[sy][j];
        $selection[k++] = '\\n';
        for (i = sy + 1; i <= ey - 1; i++) {
            for (j = 0; j < $columns; j++, k++)
                $selection[k] = $contents[i][j];
            $selection[k++] = '\\n';
        }
        for (j = 0; j <= ex; j++, k++)
            $selection[k] = $contents[ey][j];
        $selection[k] = '\\0';
        $selection_len = k;
    }

    /* Now ask the X server for ownership of the selection */
    if (XtOwnSelection($, XA_PRIMARY, event->xbutton.time,
		       convert_proc, lose_ownership_proc, NULL) == FALSE)
        XtAppWarning(XtWidgetToApplicationContext($),
		     "Failed to become selection owner");
}
.fi

.eh

.TP
.I "paste_selection

.hi

.nf
void paste_selection($, XEvent* event, String* params, Cardinal* num_params)
{
    XtGetSelectionValue($, XA_PRIMARY, XA_STRING, paste_callback, NULL,
	event->xbutton.time);
}
.fi

.eh

.hi

.hi
.SH "Importss"

.nf

.B incl
 <assert.h>
.fi

.nf

.B incl
 <stdio.h>
.fi

.nf

.B incl
 <X11/Xatom.h>
.fi

.nf

.B incl
 <X11/keysym.h>
.fi

.nf

.B incl
 <X11/Xmu/Atoms.h>
.fi

.hi

.hi
.SS "Private variables"

The size of a character cell is assumed to be constant.
It is stored in \fIcellwidth\fP and \fIcellheight\fP.

	

.nf
int  cellwidth
.fi

.nf
int  cellheight
.fi

.nf
int  uline_pos
.fi

.nf
int  uline_thick
.fi

Four GC's are used for drawing the eight possible faces:
normal/bold, normal/reverse, with/without underline.

	

.nf
XFontStruct * fnt
.fi

.nf
GC  gc
.fi

.nf
GC  boldgc
.fi

.nf
GC  revgc
.fi

.nf
GC  revboldgc
.fi

The contents of the widget are held in an array of lines.
Each character also has a byte with attributes (bold,
reverse, underlined) in a separate array. \fIallocated_lines\fP
is the length of the array. It is used in the
\fIallocate_contents\fP function to know the number of lines to
deallocate.

The \fIATTRIB_DIRTY\fP attribute is used by the \fIwrite\fP method and
its utility functions to mark lines that have changed, so that
they can all be redrawn at the end of the \fIwrite\fP function. This
flag is only ever set on the first character of a line.

	

.nf
char ** contents
.fi

.nf
char ** attribs
.fi

.nf
int  allocated_rows
.fi

.nf
int  allocated_columns
.fi

.nf
 ATTRIB_BOLD
.fi

.nf
 ATTRIB_REV
.fi

.nf
 ATTRIB_ULINE
.fi

.nf
 ATTRIB_INVIS
.fi

.nf
 ATTRIB_DIRTY
.fi

The cursor position is held in \fIcursorx\fP and \fIcursory\fP.
Because the cursor might move between being drawn and being
removed, the timer procedure that blinks the cursor keeps
the position where the cursor is drawn in a separate pair
of coordinates.

Another pair of coordinates is used to implement the `ESC 7' or
`ESC [ s'(save cursor) and `ESC 8' or `ESC [ u' (restore cursor)
control sequences.

	

.nf
int  cursorx
.fi

.nf
int  cursory
.fi

.nf
int  old_cx
.fi

.nf
int  old_cy
.fi

.nf
int  savedx
.fi

.nf
int  savedy
.fi

.nf
Bool  cursor_on
.fi

The next text to be added at the cursor position has
attributes \fIcur_attrib\fP. The value of this variable can only be
changed through ANSI escapes, that are sent to the \fIwrite\fP
method. \fIinsert_mode\fP is likewise only set by escape sequences;
it determines whether new characters on a line push existing
characters to the right and newlines push lower lines down.
\fIlast_char\fP is the most recently written character.

	

.nf
char  cur_attrib
.fi

.nf
Bool  insert_mode
.fi

.nf
char  last_char
.fi

The \fIwrite\fP method interprets escape using an FSM, the
state of which is recorded in \fIstate\fP. Escape sequences can
have up to 5 numeric parameters, which are stored by
\fIwrite\fP in 5 registers.

	

.nf
enum {Init, Esc, EscLB, EscLBGT,
	    EscLBGT2, Register0, Register1, Register2,
	    Register3, Register4} State
.fi

.nf
State  state
.fi

.nf
int  escparm[5]
.fi

The \fItimer\fP variable holds the ID of the timer function
that handles the blinking of the cursor.

	

.nf
XtIntervalId  timer
.fi

The current selection is kept in a buffer.

	

.nf
int  start_x
.fi

.nf
int  start_y
.fi

.nf
int  end_x
.fi

.nf
int  end_y
.fi

.nf
char * selection
.fi

.nf
int  selection_len
.fi

A press of Button1 without a subsequent motion is not enough to
start a selection. The \fIdrag_started\fP variable is set to \fITRUE\fP
by \fIextend_selection\fP and is inspected by \fIend_selection\fP to
decide whether a selection has taken place. If \fIdrag_started\fP is
\fIFALSE\fP the event was a simple click.

        

.nf
Bool  drag_started
.fi

The AnsiTerm widget can `lock' a number of lines.
\fIlocked_lines\fP gives the number of lines that are locked,
starting from the top.

        

.nf
int  locked_lines
.fi

.hi

.hi
.SS "Methods"

The \fIinitialize\fP method initializes the private variables
and sets the geometry resources to the values indicated by
\fIrows\fP and \fIcolumns\fP.

An event handler for the Map and Unmap events is
registered, which will in turn install or remove a timer
that makes the cursor blink.

Note that \fImake_gc\fP uses \fIuline_thick\fP, which is set by
\fIcompute_cell_size\fP.

The strange call to \fIXmuInternAtom\fP is necessary to initialize the Xmu
atom caching mechanism. \fIXA_TARGETS\fP is used by the selection
procedures.

.nf
initialize(Widget  request, $, ArgList  args, Cardinal * num_args)
{
    int framewd = 2 * compute_framewd($);

    (void) XmuInternAtom(XtDisplay($), XmuMakeAtom("NULL"));
    $fnt = XLoadQueryFont(XtDisplay($), $font);
    if (! $fnt)
	XtAppError(XtWidgetToApplicationContext($),
		   "Failed to find font in AnsiTerm");
    compute_cell_size($);

    $gc = $boldgc = $revgc = $revboldgc = NULL;
    make_gc($);

    if ($columns < 1) $columns = 1;
    if ($rows < 1) $rows = 1;
    $width = framewd + 2 * $margin + $columns * $cellwidth;
    $height = framewd + 2 * $margin + $rows * $cellheight;

    $allocated_columns = $allocated_rows = 0;
    $contents = $attribs = NULL;
    allocate_contents($);

    XtAddEventHandler($, StructureNotifyMask, FALSE, map_handler, NULL);

    $savedx = $savedy = $cursorx = $cursory = 0;
    $cursor_on = FALSE;
    $cur_attrib = 0;
    $insert_mode = FALSE;
    $last_char = ' ';
    $state = Init;
    $selection = NULL;
    $locked_lines = 0;
}
.fi

If the font, the margin or the number of rows/columns
changes, the \fIset_values\fP method will resize the widget in
order to keep the requested number of rows/columns. Only if
\fIwidth\fP or \fIheight\fP is changed explicitly (and \fIrows\fP or
\fIcolumns\fP is not changed at the same time), will the \fIrows\fP
or \fIcolumns\fP be recomputed and changed.

.nf
Boolean  set_values(Widget  old, Widget  request, $, ArgList  args, Cardinal * num_args)
{
    Bool redraw = FALSE, need_gc = FALSE;
    int framewd = 2 * compute_framewd($);

    if ($font != $old$font || $boldfont != $old$boldfont) {
	if ($fnt) XFreeFont(XtDisplay($), $fnt);
	$fnt = XLoadQueryFont(XtDisplay($), $font);
	need_gc = TRUE;
	compute_cell_size($);
    }
    if ($foreground != $old$foreground
	|| $background_pixel != $old$background_pixel) {
	need_gc = TRUE;
    }
    if (need_gc) {
	make_gc($);
	redraw = TRUE;
    }

    if ($columns < 1) $columns = 1;		/* Silently increase */
    if ($rows < 1) $rows = 1;

    if ($columns != $old$columns)
	$width = framewd + 2 * $margin + $columns * $cellwidth;
    else if ($width != $old$width)
	$columns = max(1, ($width - framewd - 2 * $margin)/$cellwidth);
    else if ($margin != $old$margin || $cellwidth != $old$cellwidth)
	$width = framewd + 2 * $margin + $columns * $cellwidth;

    if ($rows != $old$rows)
	$height = framewd + 2 * $margin + $rows * $cellheight;
    else if ($height != $old$height)
	$rows = max(1, ($height - framewd - 2 * $margin)/$cellheight);
    else if ($margin != $old$margin || $cellheight != $old$cellheight)
	$height = framewd + 2 * $margin + $rows * $cellheight;

    if ($rows != $old$rows || $columns != $old$columns)
	allocate_contents($);
	
    if ($width != $old$width || $height != $old$height)
	redraw = FALSE;			/* Don't redraw if resizing */

    if ($cursorx >= $columns) $cursorx = $columns - 1;
    if ($cursory >= $rows) $cursory = $rows - 1;

    return redraw;
}
.fi

The \fIresize\fP method keeps the \fIrows\fP and \fIcolumns\fP resources
synchronized with the new \fIwidth\fP and \fIheight\fP.

.nf
resize($)
{
    XfwfResizeInfo cb_info;
    int oldcolumns = $columns, oldrows = $rows;
    int framewd = 2 * compute_framewd($);

    $columns = max(1, ($width - framewd - 2 * $margin)/$cellwidth);
    $rows = max(1, ($height - framewd - 2 * $margin)/$cellheight);

    if ($cursorx >= $columns) $cursorx = $columns - 1;
    if ($cursory >= $rows) $cursory = $rows - 1;

    if ($rows != oldrows || $columns != oldcolumns) {
	allocate_contents($);
	cb_info.rows = $rows;
	cb_info.columns = $columns;
	XtCallCallbackList($, $resizeCallback, cb_info);
    }
}
.fi

When the parent widget asks for the preferred size, the
AnsiTerm widget replies with the current size or the next
smaller size that can accomodate a whole number of
characters both horizontally and vertically.

.nf
XtGeometryResult  query_geometry($, XtWidgetGeometry * request, XtWidgetGeometry * reply)
{
    int framewd = compute_framewd($);

    /* Compute our preferred geometry */
    reply->request_mode = CWWidth | CWHeight;
    reply->width = framewd + 2 * $margin + $columns * $cellwidth;
    reply->height = framewd + 2 * $margin + $rows * $cellheight;

    /* Is our preferred geometry different from the requested? */
    if (((request->request_mode  CWWidth)
	     request->width != reply->width)
	|| ((request->request_mode  CWHeight)
	     request->height != reply->height))
	return XtGeometryAlmost;

    /* Accept, because either size is OK or size not requested at all */
    return XtGeometryYes;
}
.fi

The \fIdestroy\fP method frees the memory occupied by the
\fIcontents\fP and \fIattribs\fP arrays. It also frees the GCs and the
fonts.

.nf
destroy($)
{
    int i;

    if ($allocated_rows != 0) {
	for (i = 0; i < $allocated_rows; i++) {
	    XtFree($contents[i]);
	    XtFree($attribs[i]);
	}
	XtFree((char *) $contents);
	XtFree((char *) $attribs);
    }
    XtReleaseGC($, $gc);
    XtReleaseGC($, $boldgc);
    XtReleaseGC($, $revgc);
    XtReleaseGC($, $revboldgc);
    XFreeFont(XtDisplay($), $fnt);
    XFreeFont(XtDisplay($), $boldfont);
}
.fi

The \fIexpose\fP method draws only the characters in the exposed
region. (More precisely: in the smallest rectangle enclosing the
region, as given by the fields of the event structure.)

QUESTION: is it useful to maintain an off-screen copy of the
text in a Pixmap and redraw the screen from that? It might speed
up the redrawing, but it takes up server memory and some servers
may already have their own `save-under' optimizations.

.nf
expose($, XEvent * event, Region  region)
{
    int top, bot, lft, rgt, framewd, y, x, i;
    XExposeEvent *ev = (XExposeEvent *) event;
    
    if (! XtIsRealized($)) return;

    #expose($, event, region);			/* Superclass draws frame */

    if (! event) {
	top = 0;  bot = $rows - 1;
	lft = 0;  rgt = $columns - 1;
    } else {
	find_cell($, ev->x, ev->y, lft, top);
	find_cell($, ev->x + ev->width, ev->y + ev->height, rgt, bot);
    }
    framewd = compute_framewd($);
    if (lft < $columns  top < $rows)
	for (i = top; i <= bot; i++) draw_line($, framewd, i, lft, rgt + 1);
}
.fi

The \fIwrite\fP method is the means by which characters are added
to the display. The function adds characters to the \fIcontents\fP
array, wrapping and scrolling if necessary. It also interprets
ANSI escape codes. It implements a finite state machine to keep
track of its position in an escape sequence.

Actual drawing is postponed until all \fIn\fP characters have been
processed. At that point all lines with `dirty' flags on their
first character are redrawn. This is only efficient when \fIn\fP is
greater than one, it becomes more efficient with larger \fIn\fP.

The resoning lies in the expectation that programs generate
large numbers of characters at once, but one character at a time
is usually the result of a user typing. In the latter case the
small performance penalty is probably not noticable. This
hypothesis has not been tested in practice, however.

(Note that the `dirty' flag is not reset when the widget is not
realized. This shouldn't matter much.)

.nf
write($, char * text, int  n)
{
    int framewd, i;

    for (i = 0; i < n; i++) {
        switch ($state) {
        case Init:
            switch (text[i]) {
            case '\\007': XBell(XtDisplay($), 100); break; /* Bell */
            case '\\015': $cursorx = 0; break;	/* CR */
            case '\\011': next_tabstop($); break; /* Tab */
            case '\\012': cursor_down($); break;	/* LF */
            case '\\010': if ($cursorx) $cursorx--; break; /* Backspace */
            case '\\033': $state = Esc; break;	/* Esc */
            default: add_char($, text[i]);
            }
            break;
        case Esc:
            switch (text[i]) {
            case '[': $state = EscLB; break;
	    case '2': $state = Init; break;	/* Set tab stop ignored */
	    case '7':
		$savedx = $cursorx;
		$savedy = $cursory;
		$state = Init;
		break;
	    case '8': goto_xy($, $savedx, $savedy); $state = Init; break;
            default:                            /* Unrecognized sequence */
                add_char($, '\\033');
                add_char($, text[i]);
                $state = Init;
            }
	    break;
	case EscLB:
	    switch (text[i]) {
	    case 'H': $cursorx = $cursory = 0; $state = Init; break;
	    case 'J': clear_eos($); $state = Init; break;
	    case 'K': clear_eol($), $state = Init; break;
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[0] = text[i] - '0';
		$state = Register0;
		break;
	    case 'C': 
		if ($cursorx < $columns - 1) $cursorx++;
		$state = Init;
		break;
	    case 'A': if ($cursory) $cursory--; $state = Init; break;
	    case 'P': delete_char($); $state = Init; break;
	    case 'M': delete_line($); $state = Init; break;
	    case 'm': $cur_attrib = 0; $state = Init; break;
	    case '@': insert_chars($, 1); $state = Init; break;
	    case 'L': insert_line($); $state = Init; break;
	    case '>': $state = EscLBGT; break;
	    case 's':
		$savedx = $cursorx;
		$savedy = $cursory;
		$state = Init;
		break;
	    case 'u': goto_xy($, $savedx, $savedy); $state = Init; break;
	    case 'r': $state = Init; break;	/* Set scroll region ignored */
	    default:				/* Unrecognized sequence */
		add_char($, '\\033');
		add_char($, '[');
		add_char($, text[i]);
		$state = Init;
	    }
		break;
	case Register0:
	    switch (text[i]) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[0] = 10 * $escparm[0] + text[i] - '0';
		break;
	    case ';': $escparm[1] = 0; $state = Register1; break;
	    case 'm': set_attrib($, 1); $state = Init; break;
	    case 'h': $insert_mode = TRUE; $state = Init; break;
	    case 'l': $insert_mode = FALSE; $state = Init; break;
	    case '@': insert_chars($, $escparm[0]); $state = Init; break;
	    case 'b': repeat_char($, $escparm[0] - 1); $state = Init; break;
	    case 'H': case 'f':
                goto_xy($, 1, $escparm[0] - 1);
		$state = Init;
		break;
	    case 'n': report_cursor_pos($); $state = Init; break;
	    case 'j':
		$cursorx = $cursory = 0;
		clear_eos($);
		$state = Init;
		break;
	    case 'r': $state = Init; break;	/* Set scroll region ignored */
	    default: $state = Init;		/* Ignore unknown sequence */
	    }
	    break;
	case Register1:
	    switch (text[i]) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[1] = 10 * $escparm[1] + text[i] - '0';
		break;
	    case ';': $escparm[2] = 0; $state = Register2; break;
	    case 'm': set_attrib($, 2); $state = Init; break;
	    case 'H': case 'f':
                goto_xy($, $escparm[1] - 1, $escparm[0] - 1);
		$state = Init;
		break;
	    case 'r': $state = Init; break;	/* Set scroll region ignored */
	    default: $state = Init;		/* Ignore unknown sequence */
	    }
	    break;
	case Register2:
	    switch (text[i]) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[2] = 10 * $escparm[2] + text[i] - '0';
		break;
	    case ';': $escparm[3] = 0; $state = Register3; break;
	    case 'm': set_attrib($, 3); $state = Init; break;
	    default: $state = Init;		/* Ignore unknown sequence */
	    }
	    break;
	case Register3:
	    switch (text[i]) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[3] = 10 * $escparm[3] + text[i] - '0';
		break;
	    case ';': $escparm[4] = 0; $state = Register4; break;
	    case 'm': set_attrib($, 4); $state = Init; break;
	    default: $state = Init;		/* Ignore unknown sequence */
	    }
	    break;
	case Register4:
	    switch (text[i]) {
	    case '0': case '1': case '2': case '3': case '4':
	    case '5': case '6': case '7': case '8': case '9':
		$escparm[4] = 10 * $escparm[4] + text[i] - '0';
		break;
	    case 'm': set_attrib($, 5); $state = Init; break;
	    default: $state = Init;		/* Ignore unknown sequence */
	    }
	    break;
	case EscLBGT:				/* Seen \\E[> */
	    switch (text[i]) {
	    case '2': $state = EscLBGT2; break;
	    default:
		add_char($, '\\033');
		add_char($, '[');
		add_char($, '>');
		add_char($, text[i]);
		$state = Init;
	    }
	    break;
	case EscLBGT2:				/* Seen \\E[>2 */
	    switch (text[i]) {
	    case 'h': memory_lock($); $state = Init; break;
	    case 'l': memory_unlock($); $state = Init; break;
	    default:
		add_char($, '\\033');
		add_char($, '[');
		add_char($, '>');
		add_char($, '2');
		add_char($, text[i]);
		$state = Init;
	    }
		break;
	default:
	    assert(! "Cannot happen");
	}
    }

    /* Finally redraw all lines that are marked as `dirty' */
    if (XtIsRealized($)) {
	framewd = compute_framewd($);
	for (i = 0; i < $rows; i++)
	    if ($attribs[i][0]  ATTRIB_DIRTY) {
		$attribs[i][0] = ~ATTRIB_DIRTY;
		draw_line($, framewd, i, 0, $columns);
	    }
    }
}
.fi

.hi

.hi
.SH "Utilities"

\fBdef\fP max(a, b) =
((a )>(b )?(a ):(b ))

\fBdef\fP min(a, b) =
((a )<(b )?(a ):(b ))

The \fIdraw_line\fP function is called by \fIexpose\fP to draw the
characters in line \fIrow\fP between columns \fIlft\fP and \fIrgt\fP. It
draws runs of characters with the same attributes.

.nf
draw_line($, int  framewd, int  row, int  lft, int  rgt)
{
    Display *dpy = XtDisplay($);
    Window win = XtWindow($);
    int x, y, i, n;
    GC use_gc;

    x = framewd + $margin + lft * $cellwidth;
    y = framewd + $margin + row * $cellheight + $fnt->ascent;
    while (lft < rgt) {
	i = lft + 1;
	while (i < rgt  $attribs[row][i] == $attribs[row][lft]) i++;
        if ($attribs[row][lft]  ATTRIB_INVIS) {
            XClearArea(dpy, win, x, y - $fnt->ascent, n * $cellwidth,
		       $cellheight, FALSE);
        } else {
	    switch ($attribs[row][lft]  (ATTRIB_REV | ATTRIB_BOLD)) {
	    case 0: use_gc = $gc; break;
	    case ATTRIB_BOLD: use_gc = $boldgc; break;
	    case ATTRIB_REV: use_gc = $revgc; break;
	    case ATTRIB_REV | ATTRIB_BOLD: use_gc = $revboldgc; break;
	    }
	    n = i - lft;
	    XDrawImageString(dpy, win, use_gc, x, y, $contents[row][lft], n);
	    if ($attribs[row][lft]  ATTRIB_ULINE)
		XDrawLine(dpy, win, use_gc, x, y + $uline_pos,
			  x + n * $cellwidth, y + $uline_pos);
	}
	lft = i;
	x += n * $cellwidth;
    }
}
.fi

The \fIconvert_proc\fP function is a callback called by Xt when
some application requests the selection. It checks if the
requested type is \fIXA_STRING\fP, since it can only give data in
that format.

.nf
Boolean  convert_proc($, Atom * selection, Atom * target, Atom * type_return, XtPointer * value_return, unsigned  long * length_return, int * format_return)
{
    if (*target == XA_TARGETS(XtDisplay($))) {

	XSelectionRequestEvent *req;
	Atom *targetP;
	Atom *std_targets;
	unsigned long std_length;

	req = XtGetSelectionRequest($, *selection, NULL);
	XmuConvertStandardSelection($, req->time, selection, target,
	    type_return, (XtPointer *) std_targets, std_length,
	    format_return);
	*value_return = XtMalloc(sizeof(Atom) * (std_length + 1));
	targetP = *(Atom **) value_return;
	*length_return = std_length + 1;
	*targetP++ = XA_STRING;
	bcopy((char*)std_targets, (char*)targetP, sizeof(Atom) * std_length);
	XtFree((char *) std_targets);
	*type_return = XA_ATOM;
	*format_return = sizeof(Atom) * 8;
	return TRUE;

    } else if (*target == XA_STRING) {
	*value_return = $selection;
	*type_return = XA_STRING;
	*length_return = $selection_len;
	*format_return = 8;
	return TRUE;
	
    } else {

	if (XmuConvertStandardSelection($, CurrentTime, selection, target,
		type_return, value_return, length_return, format_return))
	    return TRUE;
	else
	    return FALSE;
    }
}
.fi

The \fIlose_ownership_proc\fP is a callback that is called when
the widget loses selection ownership. Since the AnsiTerm widget
already unhighlighted the selection when the mouse was released,
there is nothing for this function to do.

.nf
lose_ownership_proc($, Atom * selection)
{}
.fi

The \fIpaste_callback\fP is installed by the \fIpaste_selection\fP
action procedure. It receives the selection data and `pastes' it
into the AnsiTerm. In this case that means it is passed
character for character to the \fIkeyCallback\fP function.

.nf
paste_callback($, XtPointer  client_data, Atom * selection, Atom * type, XtPointer  value, unsigned  long * length, int * format)
{
    char *text = (char *) value;
    int i;

    assert(*selection == XA_PRIMARY  *type == XA_STRING  *format == 8);

    if (value == NULL  *length == 0) {
	XBell(XtDisplay($), 100);
	return;
    }
    for (i = 0; i < *length; i++)
	XtCallCallbackList($, $keyCallback, (XtPointer) text[i]);
    XtFree(value);
}
.fi

\fIcompute_cell_size\fP compares the character sizes from the
normal and the bold font and sets \fIcellwidth\fP and \fIcellheight\fP
to the character size of the larger of the two.

.nf
compute_cell_size($)
{
    unsigned long h1, h2;

    if (! XGetFontProperty($fnt, XA_QUAD_WIDTH, h1))
	h1 = XTextWidth($fnt, "M", 1);
    if (! XGetFontProperty($boldfont, XA_QUAD_WIDTH, h2))
	h2 = XTextWidth($boldfont, "M", 1);
    $cellwidth = max(h1, h2);

    h1 = $fnt->ascent + $fnt->descent;
    h2 = $boldfont->ascent + $boldfont->descent;
    $cellheight = max(h1, h2);

    if (! XGetFontProperty($fnt, XA_UNDERLINE_POSITION, h1)) h1 = 2;
    if (! XGetFontProperty($boldfont, XA_UNDERLINE_POSITION, h2)) h2 = 2;
    $uline_pos = max(h1, h2);

    if (! XGetFontProperty($fnt, XA_UNDERLINE_THICKNESS, h1)) h1 = 1;
    if (! XGetFontProperty($boldfont, XA_UNDERLINE_THICKNESS, h2)) h2 = 1;
    $uline_thick = max(h1, h2);
}
.fi

.nf
int  compute_framewd($)
{
    Position x, y;
    Dimension w, h;

    $compute_inside($, x, y, w, h);
    return x;
}
.fi

The \fIallocate_contents\fP function first deallocates the memory
for the lines that are no longer needed or allocates memory if
the number of lines has grown. It then lengthens or shortens
each line to the new number of columns. New lines and columns
are initialized to spaces, with default (=zero) attributes.

\fBdef\fP myrealloc(p, n) =
((XtPointer )XtRealloc ((XtPointer )(p ),(n )*sizeof (*(p ))))

.nf
allocate_contents($)
{
    int i, j;

    /* Remove superfluous lines */
    for (i = $rows; i < $allocated_rows; i++) {
	XtFree($contents[i]);
	XtFree($attribs[i]);
    }
    /* Allocate and initialize new lines */
    $contents = myrealloc($contents, $rows);
    $attribs = myrealloc($attribs, $rows);
    for (i = $allocated_rows; i < $rows; i++) {
	$contents[i] = XtMalloc($columns);
	$attribs[i] = XtMalloc($columns);
	for (j = 0; j < $columns; j++) {
	    $contents[i][j] = ' ';
	    $attribs[i][j] = 0;
	}
    }
    /* Lengthen or shorten existing lines */
    if ($allocated_columns != $columns) {
	for (i = 0; i < min($allocated_rows, $rows); i++) {
	    $contents[i] = myrealloc($contents[i], $columns);
	    $attribs[i] = myrealloc($attribs[i], $columns);
	    for (j = $allocated_columns; j < $columns; j++) {
		$contents[i][j] = ' ';
		$attribs[i][j] = 0;
	    }
	}
    }
    $allocated_rows = $rows;
    $allocated_columns = $columns;
}
.fi

The \fIblink_handler\fP function is called every 500 milliseconds
to blink the cursor. It draws the character cell under the
cursor alternately in its normal face and in reverse.
It reinstalls itself.

.nf
blink_handler(XtPointer  client_data, XtIntervalId * id)
{
    Widget $ = (Widget) client_data;

    if ($cursor_on) {
	/* Restore character to normal face */
	draw_line($, compute_framewd($), $old_cy, $old_cx, $old_cx + 1);
    } else {
	/* Draw in reverse */
	$old_cx = $cursorx;
	$old_cy = $cursory;
	$attribs[$old_cy][$old_cx] ^= ATTRIB_REV;
	draw_line($, compute_framewd($), $old_cy, $old_cx, $old_cx + 1);
	$attribs[$old_cy][$old_cx] ^= ATTRIB_REV;
    }
    $cursor_on = ! $cursor_on;
    $timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
	    500, blink_handler, $);
}
.fi

The \fImap_handler\fP function is installed by \fIinitialize\fP as an
event handler for Map and Unmap events. It will install (on Map)
or remove (on Unmap) the timer function that blinks the cursor.

TO DO: this can be made a little more efficient by checking the
widget's \fIvisibilty\fP field (after setting the class variable
\fIvisible_interest\fP to \fITRUE\fP) and installing a timer for
blinking only when the widget is visible and a timer for
periodically checking \fIvisible\fP otherwise.

.nf
map_handler($, XtPointer  client_data, XEvent * event, Boolean * cont)
{
    if (event->type == MapNotify)
	$timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
				 500, blink_handler, $);
    else if (event->type == UnmapNotify)
	XtRemoveTimeOut($timer);
}
.fi

The \fImake_gc\fP function is called by \fIinitialize\fP and
\fIset_values\fP every time the fonts or the colors change.

.nf
make_gc($)
{
    XGCValues values;
    XtGCMask mask;

    if ($gc) XtReleaseGC($, $gc);
    values.font = $fnt->fid;
    values.foreground = $foreground;
    values.background = $background_pixel;
    values.line_width = $uline_thick;
    mask = GCFont | GCForeground | GCBackground | GCLineWidth;
    $gc = XtGetGC($, mask, values);

    if ($boldgc) XtReleaseGC($, $boldgc);
    values.font = $boldfont->fid;
    values.foreground = $foreground;
    values.background = $background_pixel;
    values.line_width = $uline_thick;
    mask = GCFont | GCForeground | GCBackground | GCLineWidth;
    $boldgc = XtGetGC($, mask, values);

    if ($revgc) XtReleaseGC($, $revgc);
    values.font = $fnt->fid;
    values.foreground = $background_pixel;
    values.background = $foreground;
    values.line_width = $uline_thick;
    mask = GCFont | GCForeground | GCBackground | GCLineWidth;
    $revgc = XtGetGC($, mask, values);

    if ($revboldgc) XtReleaseGC($, $revboldgc);
    values.font = $boldfont->fid;
    values.foreground = $background_pixel;
    values.background = $foreground;
    values.line_width = $uline_thick;
    mask = GCFont | GCForeground | GCBackground | GCLineWidth;
    $revboldgc = XtGetGC($, mask, values);
}
.fi

The \fIadd_char\fP function is called by the \fIwrite\fP method to add
a character at the current cursor position. The cursor is then
moved to the right (unless it is already at the right margin).

When \fIinsert_mode\fP is on, the characters that are on the same
line to the right of the cursor are shifted to the right.

The first character of the modified line is marked with the
\fIATTRIB_DIRTY\fP flag, to notify the \fIwrite\fP that this line should
be redrawn.

.nf
add_char($, int  c)
{
    if ($insert_mode) {
        memmove($contents[$cursory][$cursorx+1],
                $contents[$cursory][$cursorx], $columns - $cursorx - 1);
        memmove($attribs[$cursory][$cursorx+1],
                $attribs[$cursory][$cursorx], $columns - $cursorx - 1);
        $contents[$cursory][$cursorx] = c;
        $attribs[$cursory][$cursorx] = $cur_attrib;
    } else {
        $contents[$cursory][$cursorx] = c;
        $attribs[$cursory][$cursorx] = $cur_attrib;
    }
    $attribs[$cursory][0] |= ATTRIB_DIRTY;
    if ($cursorx < $columns - 1)
        $cursorx++;
    else {					/* Margin reached */
        $cursorx = 0;				/* CR */
        cursor_down($);				/* LF */
    }
    $last_char = c;				/* Save for repeat_char() */
}
.fi

The \fIrepeat_char\fP functions inserts \fIn\fP copies of the most recently
inserted character.

.nf
repeat_char($, int  n)
{
    for (; n > 0; n--) add_char($, $last_char);
}
.fi

\fInext_tabstop\fP moves the cursor to the next tab stop. In
overwrite mode, the cursor is moved and the line is not changed;
in insert mode up to 7 spaces are inserted and the line is
marked for redrawing.

.nf
next_tabstop($)
{
    if (! $insert_mode) {
	$cursorx = (($cursorx + 8)/8) * 8;
    } else {
	int x = (($cursorx + 8)/8) * 8;
	if (x >= $columns) x = $columns - 1;
	while ($cursorx != x) add_char($, ' ');
	$attribs[$cursory][0] |= ATTRIB_DIRTY;
    }
}
.fi

The \fIdelete_line\fP function deletes the line on which the
cursor is, shifting all lower lines up. The cursor doesn't move.
All lines that changed are marked for redrawing. The deleted
line is reinserted at the bottom and cleared. (This way no
memory is lost.)

.nf
delete_line($)
{
    int i;
    char *swap_contents, *swap_attribs;

    swap_contents = $contents[$cursory];
    swap_attribs = $attribs[$cursory];
    for (i = $cursory; i < $rows - 1; i++) {
	$contents[i] = $contents[i+1];
	$attribs[i] = $attribs[i+1];
	$attribs[i][0] |= ATTRIB_DIRTY;
    }
    $contents[$rows-1] = swap_contents;
    $attribs[$rows-1] = swap_attribs;
    $attribs[$rows-1][0] |= ATTRIB_DIRTY;
    memset($contents[$rows-1], ' ', $columns);
    memset($attribs[$rows-1], 0, $columns);
}
.fi

The \fIcursor_down\fP function moves the cursor to the next lower
line or scrolls the text up.

.nf
cursor_down($)
{
    int framewd, save_y;
    
    if ($cursory < $rows - 1)
	$cursory++;
    else {
	save_y = $cursory;
        $cursory = $locked_lines;
	delete_line($);
	$cursory = save_y;
    }
}
.fi

The \fIclear_eol\fP function clears the text from the cursor to
the end of the line.

.nf
clear_eol($)
{
    int framewd = compute_framewd($);

    memset($contents[$cursory][$cursorx], ' ', $columns - $cursorx);
    memset($attribs[$cursory][$cursorx], 0, $columns - $cursorx);
    $attribs[$cursory][0] |= ATTRIB_DIRTY;
}
.fi

The \fIclear_eos\fP function clears the text from the cursor to
the end of the screen.

.nf
clear_eos($)
{
    int i, save_x, save_y;

    clear_eol($);
    save_x = $cursorx;
    save_y = $cursory;
    $cursorx = 0;
    for (i = save_y + 1; i < $rows; i++) {
	$cursory = i;
	clear_eol($);
    }
    $cursorx = save_x;
    $cursory = save_y;
}
.fi

\fIdelete_char\fP removes the character under the cursor and
shifts the rest of the line one position to the left.

.nf
delete_char($)
{
    memmove($contents[$cursory][$cursorx], $contents[$cursory][$cursorx+1],
	$columns - $cursorx - 1);
    memmove($attribs[$cursory][$cursorx], $attribs[$cursory][$cursorx+1],
	$columns - $cursorx - 1);
    $contents[$cursory][$columns-1] = ' ';
    $attribs[$cursory][$columns-1] = 0;
    $attribs[$cursory][0] |= ATTRIB_DIRTY;
}
.fi

\fIinsert_chars\fP inserts \fIn\fP spaces, shifting the rest of the
line to the right.

.nf
insert_chars($, int  n)
{
    if (n > $columns - $cursorx) n = $columns - $cursorx;
    memmove($contents[$cursory][$cursorx+n], $contents[$cursory][$cursorx],
	    $columns - $cursorx - n);
    memmove($attribs[$cursory][$cursorx+n], $attribs[$cursory][$cursorx],
	    $columns - $cursorx - n);
    memset($contents[$cursory][$cursorx], ' ', n);
    memset($attribs[$cursory][$cursorx], 0, n);
    $attribs[$cursory][0] |= ATTRIB_DIRTY;
}
.fi

\fIinsert_line\fP inserts a blank line at the cursor's line,
moving the existing line and all lines below it down. The line
that is pushed off the bottom is reinserted at the cursor and
cleared.

.nf
insert_line($)
{
    int framewd, i;
    char *swap_contents, *swap_attribs;

    swap_contents = $contents[$rows-1];
    swap_attribs = $attribs[$rows-1];
    for (i = $rows - 1; i > $cursory; i--) {
	$contents[i] = $contents[i-1];
	$attribs[i] = $attribs[i-1];
	$attribs[i][0] |= ATTRIB_DIRTY;
    }
    $contents[$cursory] = swap_contents;
    $attribs[$cursory] = swap_attribs;
    $attribs[$cursory][0] |= ATTRIB_DIRTY;
    memset($contents[$cursory], ' ', $columns);
    memset($attribs[$cursory], 0, $columns);
}
.fi

\fIset_attrib\fP sets the current text attributes. The parameter
\fIn\fP tells the function how many of the registers contain useful
codes.	Each of these registers is interpreted.

.nf
set_attrib($, int  n)
{
    int i;

    for (i = 0; i < n; i++) {
	switch($escparm[i]) {
	case 0: $cur_attrib = 0; break;
	case 1: $cur_attrib |= ATTRIB_BOLD; break;
	case 4: $cur_attrib |= ATTRIB_ULINE; break;
	case 5: $cur_attrib |= ATTRIB_BOLD; break; /* Should be: blinking */
	case 7: $cur_attrib |= ATTRIB_REV; break;
        case 8: $cur_attrib |= ATTRIB_INVIS; break;
	}
    }
}
.fi

\fImemory_lock\fP and \fImemory_unlock\fP lock and unlock lines from
the top of the screen. \fImemory_lock\fP locks all lines above the
cursor.

.nf
memory_lock($)
{
    $locked_lines = $cursory;                  /* Lock lines above cursor */
}
.fi

.nf
memory_unlock($)
{
    $locked_lines = 0;
}
.fi

\fIgoto_xy\fP positions the cursor at the given coordinates, making sure
they fall within the current extend.

.nf
goto_xy($, int  x, int  y)
{
    $cursorx = max(0, min($columns - 1, x));
    $cursory = max(0, min($rows - 1, y));
}
.fi

\fIreport_cursor_pos\fP sends back the current cursor position via
the \fIkeyCallback\fP callback (as if the user had typed it). The
returned data is in the format `ESC [ r ; c R' (where r = row +
1 and c = column + 1).

.nf
report_cursor_pos($)
{
    char s[25];
    int i, n;

    n = sprintf(s, "\\033[%d;%dR", $cursory + 1, $cursorx + 1);
    for (i = 0; i < n; i++) XtCallCallbackList($, $keyCallback, s[i]);
}
.fi

\fIfind_cell\fP computes the coordinates of the cell that contains
(cx,cy) or that is closest to it.

.nf
find_cell($, int  cx, int  cy, int * col, int * row)
{
    int i, j, x, y, framewd;

    framewd = compute_framewd($);

    for (x = framewd + $margin + $cellwidth, i = 0;
	i < $columns - 1  x < cx;
	i++, x += $cellwidth) ;

    for (y = framewd + $margin + $cellheight, j = 0;
	j < $rows - 1  y < cy;
	j++, y += $cellheight) ;

    *col = i;
    *row = j;
}
.fi

The function \fIparse_font_name\fP makes a copy of the font name into
\fItable[0]\fP (room is allocated on the heap). All hyphens are then
replaced by \fI\\0\fP and the other 14 entries in \fItable\fP are made to point
to the start of each part of the name.

\fBdef\fP F_FOUNDRY = 1 

\fBdef\fP F_FAMILY = 2 

\fBdef\fP F_WEIGHT = 3 

\fBdef\fP F_SLANT = 4 

\fBdef\fP F_SET_WIDTH = 5 

\fBdef\fP F_SANS = 6 

\fBdef\fP F_PIXELS = 7 

\fBdef\fP F_POINTS = 8 

\fBdef\fP F_HRESOLUTION = 9 

\fBdef\fP F_VRESOLUTION = 10 

\fBdef\fP F_SPACING = 11 

\fBdef\fP F_AVG_WITH = 12 

\fBdef\fP F_CHARSET = 13 

.nf
Boolean  parse_font_name(String  name, String * table)
{
    int i, field;

    if (name == NULL || name[0] != '-') return False;
    table[0] = XtNewString(name);
    for (i = field = 0; name[i] != '\\0'; i++) {
	if (name[i] == '-') {
	    field++;
	    table[0][i] = '\\0';
	    table[field] = table[0][i+1];
	    if (field == 13) break;		/* No more fields */
	}
    }
    return True;
}
.fi

\fIinfer_bold\fP tries to find a bold version of the \fIfont\fP
resource. It parses the \fIfont\fP name and tries to form a new name
out of it. If that doesn't result in a valid font name, it uses
the \fIfont\fP resource unchanged.

.nf
infer_bold($, int  offset, XrmValue * value)
{
    static XFontStruct *fs = NULL;
    String info[15];
    char s[250];

    if ($font == NULL)
	XtAppError(XtWidgetToApplicationContext($),
		   "Font resource may not be NULL in AnsiTerm widget");

    if (parse_font_name($font, info)) {
	sprintf(s, "-%s-%s-%s-%s-*-%s-*-%s-%s-%s-%s-*-%s",
		info[F_FOUNDRY], info[F_FAMILY], "bold",
		info[F_SLANT], /* info[F_SET_WIDTH], */ info[F_SANS],
		/* info[F_PIXELS], */ info[F_POINTS], info[F_HRESOLUTION],
		info[F_VRESOLUTION], info[F_SPACING],
		/* info[F_AVG_WIDTH], */ info[F_CHARSET]);
	fs = XLoadQueryFont(XtDisplay($), s);
    }
    if (! fs)					/* Try font unchanged */
	fs = XLoadQueryFont(XtDisplay($), $font);
    if (! fs)
	XtAppWarning(XtWidgetToApplicationContext($),
		   "Couldn't find font for AnsiTerm widget");
    value->addr = (XtPointer) fs;
}
.fi

.hi
