/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>
#include "XearthCvt.h"
#include "EarthP.h"
#include "XearthI.h"

static void		refresh();

static void             ClassInitialize();
static void             Initialize();
static void             Destroy();
static Boolean          SetValues();
static void		Resize();
static void             Redisplay();
static XtGeometryResult	QueryGeometry();

static XtGeometryResult GeometryManager();
static void             ChangeManaged();
static void		InsertChild();

static void             ConstraintInitialize();
static void             ConstraintDestroy();
static Boolean          ConstraintSetValues();

static XtResource	resources[] = 
{
  {XtNprojType,		XtCProjType,		XtRProjType,sizeof (ProjType),
   XtOffsetOf(EarthRec,earth.projType),
   XtRImmediate,	(XtPointer)ProjTypeOrthographic},
  {XtNviewPosType,	XtCViewPosType,	XtRViewPosType,sizeof (ViewPosType),
   XtOffsetOf(EarthRec,earth.viewPosType),
   XtRString,		"sunrel"},
  {XtNviewLat,		XtCViewLat,		XtRFloat,	sizeof (float),
   XtOffsetOf(EarthRec,earth.viewLat),
   XtRString,		"0.0"},
  {XtNviewLon,		XtCViewLon,		XtRFloat,	sizeof (float),
   XtOffsetOf(EarthRec,earth.viewLon),
   XtRString,		"0.0"},
  {XtNperiod,		XtCPeriod,		XtRFloat,	sizeof (float),
   XtOffsetOf(EarthRec,earth.period),
   XtRString,		"0.0"},
  {XtNinclin,		XtCInclin,		XtRFloat,	sizeof (float),
   XtOffsetOf(EarthRec,earth.inclin),
   XtRString,		"0.0"},
  {XtNviewRot,		XtCViewRot,		XtRFloat,	sizeof (float),
   XtOffsetOf(EarthRec,earth.viewRot),
   XtRString,		"0.0"},
  {XtNviewRot,		XtCViewRot,		XtRFloat, sizeof (float),
   XtOffsetOf(EarthRec,earth.viewRot),
   XtRString,		"0.0"},
  {XtNviewMag,		XtCViewMag,		XtRFloat, sizeof (float),
   XtOffsetOf(EarthRec,earth.viewMag),
   XtRString,		"1.0"},
  {XtNstarFreq,		XtCStarFreq,		XtRFloat, sizeof (float),
   XtOffsetOf(EarthRec,earth.starFreq),
   XtRString,		"0.002"},
  {XtNbigStars,		XtCBigStars,		XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.bigStars),
   XtRImmediate,	(XtPointer)0},
  {XtNday,		XtCDay,			XtRInt, sizeof (int),
   XtOffsetOf(EarthRec,earth.day),
   XtRImmediate,	(XtPointer)100},
  {XtNnight,		XtCNight,		XtRInt, sizeof (int),
   XtOffsetOf(EarthRec,earth.night),
   XtRImmediate,	(XtPointer)5},
  {XtNterminator,	XtCTerminator,		XtRInt, sizeof (int),
   XtOffsetOf(EarthRec,earth.terminator),
   XtRImmediate,	(XtPointer)1},
  {XtNmono,		XtCMono,		XtRBoolean,sizeof (Boolean),
   XtOffsetOf(EarthRec,earth.mono),
   XtRImmediate,	(XtPointer)False},
  {XtNnumColors,	XtCNumColors,		XtRInt, sizeof (int),
   XtOffsetOf(EarthRec,earth.numColors),
   XtRImmediate,	(XtPointer)64},
  {XtNxGamma,		XtCXGamma,		XtRFloat, sizeof (float),
   XtOffsetOf(EarthRec,earth.xGamma),
   XtRString,		"1.0"},
  {XtNtime,		XtCTime,    XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.time),
   XtRImmediate,	(XtPointer)0},
  {XtNshiftX,		XtCShiftX,  XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.shiftX),
   XtRImmediate,	(XtPointer)0},
  {XtNshiftY,		XtCShiftY,  XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.shiftY),
   XtRImmediate,	(XtPointer)0},
  {XtNcomputeSunPos,	XtCComputeSunPos,  XtRBoolean,	sizeof (Boolean),
   XtOffsetOf(EarthRec,earth.computeSunPos),
   XtRImmediate,	(XtPointer)True},
  {XtNdoShade,		XtCDoShade,	XtRBoolean,	sizeof (Boolean),
   XtOffsetOf(EarthRec,earth.doShade),
   XtRImmediate,	(XtPointer)True},
  {XtNdoStars,		XtCDoStars,	XtRBoolean,	sizeof (Boolean),
   XtOffsetOf(EarthRec,earth.doStars),
   XtRImmediate,	(XtPointer)True},
  {XtNdoGrid,		XtCDoGrid,	XtRBoolean,	sizeof (Boolean),
   XtOffsetOf(EarthRec,earth.doGrid),
   XtRImmediate,	(XtPointer)True},
  {XtNgridBig,		XtCGridBig,	XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.gridBig),
   XtRImmediate,	(XtPointer)6},
  {XtNgridSmall,	XtCGridSmall,	XtRInt,	sizeof (int),
   XtOffsetOf(EarthRec,earth.gridSmall),
   XtRImmediate,	(XtPointer)15},
};

static XtResource	earthConstraintResources[] = 
{
  {XtNlatitude,		XtCLatitude,	XtRFloat, sizeof (float),
   XtOffsetOf(EarthConstraintsRec,earth.latitude),
   XtRImmediate,	(XtPointer)0},
  {XtNlongitude,		XtCLongitude,	XtRFloat, sizeof (float),
   XtOffsetOf(EarthConstraintsRec,earth.longitude),
   XtRImmediate,	(XtPointer)0},
};

static void		grExp(self,event,params,num_params)
Widget			self;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  refresh(self,event);
}

static char		translations[] =
"<GrExp>:		grExp()";

static XtActionsRec	actions[] = 
{
  {"grExp",		grExp},
};

EarthClassRec		earthClassRec = 
{
  {
					/* core_class fields  */
    (WidgetClass)&constraintClassRec,	/* superclass         */
    "Earth",				/* class_name         */
    sizeof (EarthRec),			/* widget_size        */
    ClassInitialize,			/* class_init         */
    NULL,				/* class_part_init    */
    FALSE,				/* class_inited       */	
    Initialize,				/* initialize         */
    NULL,				/* initialize_hook    */	
    XtInheritRealize,			/* realize            */
    NULL,				/* actions            */
    0,					/* num_actions        */	
    resources,				/* resources          */
    XtNumber(resources),		/* num_resources      */
    NULLQUARK,				/* xrm_class          */
    TRUE,				/* compress_motion    */	
    TRUE,				/* compress_exposure  */	
    TRUE,				/* compress_enterleave*/	
    TRUE,				/* visible_interest   */
    Destroy,				/* destroy            */
    Resize,				/* resize             */
    Redisplay,				/* expose             */
    SetValues,				/* set_values         */
    NULL,				/* set_values_hook    */	
    XtInheritSetValuesAlmost,		/* set_values_almost  */
    NULL,				/* get_values_hook    */	
    NULL,				/* accept_focus       */
    XtVersion,				/* version            */	
    NULL,				/* callback_private   */
    NULL,				/* tm_table           */
    QueryGeometry,			/* query_geometry     */	
    NULL,				/* display_accelerator*/
    NULL,				/* extension          */
  },
  {
					/* composite_class fields */
    GeometryManager,			/* geometry_manager    */
    ChangeManaged,			/* change_managed      */
    InsertChild,			/* insert_child        */	
    XtInheritDeleteChild,		/* delete_child        */	
    NULL,				/* extension           */
  },
  { 
					/* constraint_class fields */
   earthConstraintResources,		/* subresources        */
   XtNumber(earthConstraintResources),	/* subresource_count   */
   sizeof(EarthConstraintsRec),		/* constraint_size     */
   ConstraintInitialize,		/* initialize          */
   ConstraintDestroy,			/* destroy             */
   ConstraintSetValues,			/* set_values          */
   NULL,				/* extension           */
   },
  {
					/* Earth class fields */
    0,					/* ignore              */	
  }
};

WidgetClass		earthWidgetClass = (WidgetClass)&earthClassRec;

#ifdef HAVE_STDARG_H
VOID_FUNC               XearthWarning(char *fmt,...)
#else
VOID_FUNC               XearthWarning(fmt,va_alist)
char                    *fmt;
va_dcl
#endif
{
  va_list               ap;
  char                  buf[BUFSIZ];
  
  assert(fmt);
#ifdef HAVE_STDARG_H
  va_start(ap,fmt);
#else
  va_start(ap);
  va_arg(ap,char *);
#endif
#ifdef HAVE_SNPRINTF
  vsnprintf(buf,sizeof (buf),fmt,ap);
#else
  vsprintf(buf,fmt,ap);
#endif
  XtWarning(buf);
  va_end(ap);
}

static void		init_msi_from_self(self,msi)
Widget			self;
t_map_scan_input	*msi;
{
  msi->proj_type = EARTH.projType;
  /*msi->view_lon = ;*/
  /*msi->view_lat = ;*/
  msi->view_rot = EARTH.viewRot;
  msi->view_mag = EARTH.viewMag;
  msi->wdth = CORE.width;
  msi->hght = CORE.height;
  msi->shift_x = EARTH.shiftX;
  msi->shift_y = EARTH.shiftY;
}	

static void		init_msi_from_epo(epo,msi)
t_earth_pos_output	*epo;
t_map_scan_input	*msi;
{
  /*msi->proj_type = ;*/
  msi->view_lon = epo->view_lon;
  msi->view_lat = epo->view_lat;
  /*msi->view_rot = ;*/
  /*msi->view_mag = ;*/
  /*msi->wdth = ;*/
  /*msi->hght = ;*/
  /*msi->shift_x = ;*/
  /*msi->shift_y = ;*/
}	

static void		init_epi_from_self(self,epi)
Widget			self;
t_earth_pos_input	*epi;
{
  epi->compute_sun_pos = EARTH.computeSunPos;
  epi->fixed_time = EARTH.time;
  epi->view_pos_type = EARTH.viewPosType;
  epi->view_lat = EARTH.viewLat;
  epi->view_lon = EARTH.viewLon;
  epi->period = EARTH.period;
  epi->inclin = EARTH.inclin;
}

static void		init_mri_from_self(self,mri)
Widget			self;
t_map_render_input	*mri;
{
  mri->do_shade = EARTH.doShade;
  /*mri->sun_lon = ;*/
  /*mri->sun_lat = ;*/
  mri->do_stars = EARTH.doStars;
  mri->star_freq = EARTH.starFreq;
  mri->big_stars = EARTH.bigStars;
  mri->do_grid = EARTH.doGrid;
  mri->grid_big = EARTH.gridBig;
  mri->grid_small = EARTH.gridSmall;
  mri->day = EARTH.day;
  mri->night = EARTH.night;
  mri->terminator = EARTH.terminator;
}

static void		init_mri_from_epo(epo,mri)
t_earth_pos_output	*epo;
t_map_render_input	*mri;
{
  /*mri->do_shade = ;*/
  mri->sun_lon = epo->sun_lon;
  mri->sun_lat = epo->sun_lat;
  /*mri->do_stars = ;*/
  /*mri->star_freq = ;*/
  /*mri->big_stars = ;*/
  /*mri->do_grid = ;*/
  /*mri->grid_big = ;*/
  /*mri->grid_small = ;*/
  /*mri->day = ;*/
  /*mri->night = ;*/
  /*mri->terminator = ;*/
  /*mri->dots = ;*/
}	

#define RENDER_TYPE_MONO_1   0
#define RENDER_TYPE_MONO_8   1
#define RENDER_TYPE_COLOR_8  2
#define RENDER_TYPE_MONO_16  3
#define RENDER_TYPE_COLOR_16 4
#define RENDER_TYPE_MONO_32  5
#define RENDER_TYPE_COLOR_32 6

static int		get_render_type(depth,mono)
int			depth;
Boolean			mono;
{
  switch (depth)
    {
    case 1:
      return (RENDER_TYPE_MONO_1);
    case 8:
      return (mono ? RENDER_TYPE_MONO_8 : RENDER_TYPE_COLOR_8);
    case 12:
    case 15:
    case 16:
      return (mono ? RENDER_TYPE_MONO_16 : RENDER_TYPE_COLOR_16);
    case 24:
      return (mono ? RENDER_TYPE_MONO_32 : RENDER_TYPE_COLOR_32);
    }
  return (-1);
}

static void		dith_cleanup(self)
Widget			self;
{
  XEARTH_FREE_PROC(EARTH.pels,
		   "xearth",
		   "*:pels");
  switch (EARTH.render_type)
    {
    case RENDER_TYPE_MONO_1:
    case RENDER_TYPE_MONO_8:
    case RENDER_TYPE_MONO_16:
    case RENDER_TYPE_MONO_32:
      mono_dither_cleanup(&EARTH.mdo);
      break ;
    case RENDER_TYPE_COLOR_8:
    case RENDER_TYPE_COLOR_16:
    case RENDER_TYPE_COLOR_32:
      dither_cleanup(&(EARTH.mdo));
      break ;
    }
}

static void		dith(self,render_type)
Widget			self;
int			render_type;
{
  t_status		status;

  switch (render_type)
    {
    case RENDER_TYPE_MONO_1:
    case RENDER_TYPE_MONO_8:
    case RENDER_TYPE_MONO_16:
    case RENDER_TYPE_MONO_32:
      mono_dither_setup(&(EARTH.msi),&(EARTH.mdo));
      if ((EARTH.pels = XEARTH_ALLOC_PROC(2 * sizeof (Pixel),
					  "xearth",
					  "dither:pels",
					  &status)) == NULL)
	{
	  XearthWarning("EarthWidget: XEARTH_ALLOC_PROC: %d",status);
	  exit(1);
	}
      EARTH.pels[0] = BlackPixel(XtDisplay(self),
				 DefaultScreen(XtDisplay(self)));
      EARTH.pels[1] = WhitePixel(XtDisplay(self),
				 DefaultScreen(XtDisplay(self)));
      break ;
    case RENDER_TYPE_COLOR_8:
    case RENDER_TYPE_COLOR_16:
    case RENDER_TYPE_COLOR_32:
      {
	u_char		*tmp;
	double		inv_xgamma;
	XColor		xc;
	int		i;

	dither_setup(&(EARTH.msi),&(EARTH.mdo),EARTH.numColors);
	if ((EARTH.pels =
	     XEARTH_ALLOC_PROC(EARTH.mdo.dither_ncolors * sizeof (Pixel),
			       "xearth",
			       "dither:pels",
			       &status)) == NULL)
	  {
	    XearthWarning("EarthWidget: XEARTH_ALLOC_PROC: %d",status);
	    exit(1);
	  }
	tmp = EARTH.mdo.dither_colormap;
	inv_xgamma = 1.0 / EARTH.xGamma;
	for (i = 0;i < EARTH.mdo.dither_ncolors;i++)
	  {
	    xc.red   = ((1<<16)-1) * pow(((double) tmp[0] / 255), inv_xgamma);
	    xc.green = ((1<<16)-1) * pow(((double) tmp[1] / 255), inv_xgamma);
	    xc.blue  = ((1<<16)-1) * pow(((double) tmp[2] / 255), inv_xgamma);
	    if (XAllocColor(XtDisplay(self),
			    DefaultColormap(XtDisplay(self),
					    DefaultScreen(XtDisplay(self))),
			    &xc) == 0)
	      {
 XearthWarning("EarthWidget: XAllocColor: unable to allocate enough colors");
                 exit(1);
	      }
	    EARTH.pels[i] = xc.pixel;
	    tmp += 3;
	  }
	break ;
      }
    }
}

static void		xim_cleanup(self)
Widget			self;
{
  XEARTH_FREE_PROC(EARTH.dith,
		   "xearth",
		   "*:dith");
#ifdef NOTDEF
  unmark(EARTH.xbuf, /* XBUF IS FREED BY XDESTROYIMAGE */
	 "xearth",
	 "*:xbuf");
#endif
  XDestroyImage(EARTH.xim);
}

static void		xim(self,msi,render_type)
Widget			self;
t_map_scan_input	*msi;
int			render_type;
{
  int			dith_size;
  int			xbuf_size;
  int			bits_per_pixel;
  t_status		status;

  switch (render_type)
    {
    case RENDER_TYPE_MONO_1:
      dith_size      = msi->wdth + 7;
      xbuf_size      = dith_size >> 3;
      bits_per_pixel = 1;
      break;
    case RENDER_TYPE_MONO_8:
    case RENDER_TYPE_COLOR_8:
      dith_size      = msi->wdth;
      xbuf_size      = dith_size;
      bits_per_pixel = 8;
      break;
    case RENDER_TYPE_MONO_16:
    case RENDER_TYPE_COLOR_16:
      dith_size      = msi->wdth;
      xbuf_size      = dith_size * 2;
      bits_per_pixel = 16;
      break;
    case RENDER_TYPE_MONO_32:
    case RENDER_TYPE_COLOR_32:
      dith_size      = msi->wdth;
      xbuf_size      = dith_size * 4;
      bits_per_pixel = 32;
      break;
    }
  if ((EARTH.dith = XEARTH_ALLOC_PROC(sizeof (t_u32) * dith_size,
				      "xearth",
				      "xim:dith",
				      &status)) == NULL)
    {
      XearthWarning("EarthWidget: XEARTH_ALLOC_PROC: %d",status);
      exit(1);
    }
  if ((EARTH.xbuf = XEARTH_ALLOC_PROC(xbuf_size * msi->hght,
				      "xearth",
				      "xim:xbuf",
				      &status)) == NULL)
    {
      XearthWarning("EarthWidget: XEARTH_ALLOC_PROC: %d",status);
      exit(1);
    }
  EARTH.xim = XCreateImage(XtDisplay(self),
			   DefaultVisual(XtDisplay(self),
					 DefaultScreen(XtDisplay(self))),
			   DefaultDepth(XtDisplay(self),
					DefaultScreen(XtDisplay(self))),
			   ZPixmap,
			   0,
			   (char *)(EARTH.xbuf),
			   msi->wdth,
			   msi->hght,
			   8,
			   xbuf_size);
  if (EARTH.xim->bits_per_pixel != bits_per_pixel)
    {
      XearthWarning("EarthWidget: unexpected bits/pixel for depth %d\n",
		    DefaultDepth(XtDisplay(self),
				 DefaultScreen(XtDisplay(self))));
      exit(1);
    }
  if (render_type == RENDER_TYPE_MONO_1)
    {
      /* force MSBFirst bitmap_bit_order and byte_order
       */
      EARTH.xim->bitmap_bit_order = MSBFirst;
      EARTH.xim->byte_order       = MSBFirst;
    }
}

/* pack pixels into ximage format (assuming bits_per_pixel == 1,
 * bitmap_bit_order == MSBFirst, and byte_order == MSBFirst)
 */
static void pack_mono_1(src, dst,self)
     t_u32 *src;
     u_char  *dst;
     Widget self;
{
  int      i, i_lim;
  unsigned val;

  i_lim = EARTH.msi.wdth;
  for (i=0; i<i_lim; i+=8)
  {
    val = ((src[0] << 7) | (src[1] << 6) | (src[2] << 5) |
           (src[3] << 4) | (src[4] << 3) | (src[5] << 2) |
           (src[6] << 1) | (src[7] << 0));

    /* if white is pixel 0, need to toggle the bits
     */
    dst[i>>3] =
      (WhitePixel(XtDisplay(self),
		  DefaultScreen(XtDisplay(self))) == 0) ? (~ val) : val;
    src += 8;
  }
}

/* pack pixels into ximage format (assuming bits_per_pixel == 8)
 */
static void pack_8(src, map, dst,self)
     t_u32 *src;
     Pixel   *map;
     u_char  *dst;
     Widget  self;
{
  int      i, i_lim;
  unsigned val;

  i_lim = EARTH.msi.wdth;
  for (i=0; i<i_lim; i++)
  {
    val = map[src[i]];
    dst[i] = val;
  }
}

/* pack pixels into ximage format (assuming bits_per_pixel == 16)
 */
static void pack_16(src, map, dst,self)
     t_u32 *src;
     Pixel   *map;
     u_char  *dst;
     Widget self;
{
  int      i, i_lim;
  unsigned val;

  i_lim = EARTH.msi.wdth;

  if (EARTH.xim->byte_order == MSBFirst)
  {
    for (i=0; i<i_lim; i++)
    {
      val    = map[src[i]];
      dst[0] = (val >> 8) & 0xff;
      dst[1] = val & 0xff;
      dst   += 2;
    }
  }
  else /* (xim->byte_order == LSBFirst) */
  {
    for (i=0; i<i_lim; i++)
    {
      val    = map[src[i]];
      dst[0] = val & 0xff;
      dst[1] = (val >> 8) & 0xff;
      dst   += 2;
    }
  }
}

/* pack pixels into ximage format (assuming bits_per_pixel == 32)
 */
static void pack_32(src, map, dst,self)
     t_u32 *src;
     Pixel   *map;
     u_char  *dst;
     Widget	self;
{
  int      i, i_lim;
  unsigned val;

  i_lim = EARTH.msi.wdth;

  if (EARTH.xim->byte_order == MSBFirst)
  {
    for (i=0; i<i_lim; i++)
    {
      val    = map[src[i]];
      dst[0] = (val >> 24) & 0xff;
      dst[1] = (val >> 16) & 0xff;
      dst[2] = (val >> 8) & 0xff;
      dst[3] = val & 0xff;
      dst   += 4;
    }
  }
  else /* (xim->byte_order == LSBFirst) */
  {
    for (i=0; i<i_lim; i++)
    {
      val    = map[src[i]];
      dst[0] = val & 0xff;
      dst[1] = (val >> 8) & 0xff;
      dst[2] = (val >> 16) & 0xff;
      dst[3] = (val >> 24) & 0xff;
      dst   += 4;
    }
  }
}

static void row(row,idx,self)
     u_char *row;
     int idx;
     Widget self;
{
  switch (EARTH.render_type)
  {
  case RENDER_TYPE_MONO_1:
    mono_dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_mono_1(EARTH.dith,
		EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
		self);
    break;

  case RENDER_TYPE_MONO_8:
    mono_dither_row(&(EARTH.msi),&(EARTH.mdo),row,EARTH.dith);
    pack_8(EARTH.dith,
	   EARTH.pels,
	   EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	   self);
    break;

  case RENDER_TYPE_MONO_16:
    mono_dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_16(EARTH.dith,
	    EARTH.pels,
	    EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	    self);
    break;

  case RENDER_TYPE_MONO_32:
    mono_dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_32(EARTH.dith,
	    EARTH.pels, 
	    EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	    self);
    break;

  case RENDER_TYPE_COLOR_8:
    dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_8(EARTH.dith,
	   EARTH.pels,
	   EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	   self);
    break;

  case RENDER_TYPE_COLOR_16:
    dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_16(EARTH.dith,
	    EARTH.pels,
	    EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	    self);
    break;

  case RENDER_TYPE_COLOR_32:
    dither_row(&(EARTH.msi),&(EARTH.mdo),row, EARTH.dith);
    pack_32(EARTH.dith,
	    EARTH.pels, 
	    EARTH.xbuf + idx * EARTH.xim->bytes_per_line,
	    self);
    break;

  }
}

#define VIEW_MAG_OK(ViewMag) ((ViewMag) > 0)
#define VIEW_ROT_OK(ViewRot) ((ViewRot) >= -180.0 && (ViewRot) <= 360.0)
#define VIEW_LAT_OK(ViewLat) ((ViewLat) >= -90.0 && (ViewLat) <= 90.0)
#define VIEW_LON_OK(ViewLon) ((ViewLon) >= -180.0 && (ViewLon) <= 180.0)

static void		configure_child(child,msi,mso)
Widget			child;
t_map_scan_input	*msi;
t_map_scan_output	*mso;
{
  Widget		self;
  Position		xret;
  Position		yret;
  
  self = child->core.parent;
  if (XearthGetXY(self,
		  EARTH_I_C(child).latitude,
		  EARTH_I_C(child).longitude,
		  &xret,
		  &yret))
    {
      XtMoveWidget(child,
		   xret,
		   yret);
      if (XtIsRealized(child))
	{
	  if (!EARTH_I_C(child).mapped)
	    {
	      XtMapWidget(child);
	      EARTH_I_C(child).mapped = True;
	    }
	}
    }
  else
    {
      if (XtIsRealized(child))
	{
	  XtUnmapWidget(child);
	  EARTH_I_C(child).mapped = False;
	}
    }
}

static void		configure_children(self)
Widget			self;
{
  int			i;

  i = 0;
  while (i < COMPOSITE.num_children)
    {
      configure_child(COMPOSITE.children[i],&(EARTH.msi),&(EARTH.mso));
      i++;
    }
}

static void		reput_image(self)
Widget			self;
{
  if (XtIsRealized(self))
    XPutImage(XtDisplay(self),
	      XtWindow(self),
	      EARTH.gc,
	      EARTH.xim,
	      0,
	      0,
	      0,
	      0,
	      CORE.width,
	      CORE.height);
}

static void		refresh(self,event)
Widget			self;
XEvent			*event;
{
  if (event->type != Expose && event->type != GraphicsExpose)
    {
      XearthWarning("EarthWidget: bad event type for exposure");
      return ;
    }
  XPutImage(XtDisplay(self),
	    XtWindow(self),
	    EARTH.gc,
	    EARTH.xim,
	    event->xexpose.x,
	    event->xexpose.y,
	    event->xexpose.x,
	    event->xexpose.y,
	    event->xexpose.width,
	    event->xexpose.height);
}

/*
 * methods
 */

static void		ClassInitialize()
{
  XearthRegisterConverters();
}

static void		Initialize(request,self)
Widget			request;
Widget			self;
{
#ifdef NOTDEF /* FILL WITH FF */
  bff((char *)(&(EARTH.msi)),sizeof (EARTH.msi));
  bff((char *)(&(EARTH.mso)),sizeof (EARTH.mso));
  bff((char *)(&(EARTH.mdo)),sizeof (EARTH.mdo));
  bff((char *)(&(EARTH.mri)),sizeof (EARTH.mri));
  bff((char *)(&(EARTH.mro)),sizeof (EARTH.mro));
  bff((char *)(&(EARTH.epi)),sizeof (EARTH.epi));
  bff((char *)(&(EARTH.epo)),sizeof (EARTH.epo));
#endif
  if (CORE.width == 0)
    CORE.width = 256;
  if (CORE.height == 0)
    CORE.height = 256;
  EARTH.gc = XCreateGC(XtDisplay(self),
		       DefaultRootWindow(XtDisplay(self)),
		       0,
		       NULL);
  XSetState(XtDisplay(self),
	    EARTH.gc,
	    WhitePixel(XtDisplay(self),
		       DefaultScreen(XtDisplay(self))),
	    BlackPixel(XtDisplay(self),
		       DefaultScreen(XtDisplay(self))),
	    GXcopy,
	    AllPlanes);
  if ((EARTH.render_type = get_render_type(CORE.depth,
					   EARTH.mono)) < 0)
    {
      XearthWarning("EarthWidget: unsupported depth %d",CORE.depth);
      exit(1);
    }
  init_msi_from_self(self,&(EARTH.msi));
  dith(self,EARTH.render_type);
  init_epi_from_self(self,&(EARTH.epi));
  compute_positions(&(EARTH.epi),&(EARTH.epo));
  init_msi_from_epo(&(EARTH.epo),&(EARTH.msi));
  mso_alloc(&(EARTH.msi),&(EARTH.mso));
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_self(self,&(EARTH.mri));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  mro_alloc(&(EARTH.mro));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  xim(self,&(EARTH.msi),EARTH.render_type);
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
} 

static Boolean		SetValues(old,request,self)
Widget			old;
Widget			request;
Widget			self;
{
  Boolean		redraw;
  Boolean		reputim;

  redraw = False;
  reputim = False;
  if (EARTH_I(request).time != EARTH_I(old).time)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      EARTH.epi.fixed_time = EARTH.time;
      compute_positions(&(EARTH.epi),&(EARTH.epo));
      init_msi_from_epo(&(EARTH.epo),&(EARTH.msi));
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (EARTH_I(request).shiftX != EARTH_I(old).shiftX)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      EARTH.msi.shift_x = EARTH.shiftX;
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (EARTH_I(request).shiftY != EARTH_I(old).shiftY)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      EARTH.msi.shift_y = EARTH.shiftY;
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (EARTH_I(request).doGrid != EARTH_I(old).doGrid)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      EARTH.mri.do_grid = EARTH.doGrid;
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (EARTH_I(request).doShade != EARTH_I(old).doShade)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      EARTH.mri.do_shade = EARTH.doShade;
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (EARTH_I(request).projType != EARTH_I(old).projType)
    {
      mso_reset(&(EARTH.msi),&(EARTH.mso));
      mro_reset(&(EARTH.mro));
      EARTH.msi.proj_type = EARTH.projType;
      scan_map(&(EARTH.msi),&(EARTH.mso));
      init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
      do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
      render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	     (t_row_proc)row,
	     self);
      configure_children(self);
      reputim = True;
    }
  if (reputim)
    reput_image(self);
  return (redraw);
}

static void		Destroy(self)
Widget			self;
{
  mso_cleanup(&(EARTH.msi),&(EARTH.mso));
  mro_cleanup(&(EARTH.mro));
  dith_cleanup(self);
  xim_cleanup(self);
  XFreeGC(XtDisplay(self),EARTH.gc);
}

static void		Resize(self)
Widget			self;
{
  mso_cleanup(&(EARTH.msi),&(EARTH.mso));
  mro_cleanup(&(EARTH.mro));
  dith_cleanup(self);
  xim_cleanup(self);
  EARTH.msi.wdth = CORE.width;
  EARTH.msi.hght = CORE.height;
  dith(self,EARTH.render_type);
  mso_alloc(&(EARTH.msi),&(EARTH.mso));
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  mro_alloc(&(EARTH.mro));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  xim(self,&(EARTH.msi),EARTH.render_type);
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
  configure_children(self);
}

static void		Redisplay(self,event,region)
Widget			self;
XEvent			*event;
Region			region;
{
  refresh(self,event);
}

static XtGeometryResult	QueryGeometry(self,intended,preferred)
Widget			self;
XtWidgetGeometry	*intended;
XtWidgetGeometry	*preferred;
{
  return (XtGeometryYes);
}

static XtGeometryResult	GeometryManager(child,request,reply)
Widget			child;
XtWidgetGeometry	*request;
XtWidgetGeometry	*reply;
{
  Widget		self;

  self = child->core.parent;
  if ((request->request_mode & CWX && request->x != child->core.x) ||
      (request->request_mode & CWY && request->y != child->core.y))
      return (XtGeometryNo);
  if (request->request_mode & CWWidth)
    child->core.width = request->width;
  if (request->request_mode & CWHeight)
    child->core.height = request->height;
  if (request->request_mode & CWBorderWidth)
    child->core.border_width = request->border_width;
  return (XtGeometryYes);
}

static void		ChangeManaged(self)
Widget			self;
{
  
}

static void		InsertChild(child)
Widget			child;
{
  Widget		self;
  
  constraintClassRec.composite_class.insert_child(child);
  self = child->core.parent;
  configure_child(child,&(EARTH.msi),&(EARTH.mso));
}

static void		ConstraintInitialize(request,new)
Widget			request;
Widget			new;
{
  EARTH_I_C(new).mapped = False;
} 

static Boolean		ConstraintSetValues(old,request,new,args,num_args)
Widget			old;
Widget			request;
Widget			new;
ArgList			args;
Cardinal		*num_args;
{
  Widget		self;
#ifdef DEBUG
  EarthConstraints	old_constraints;
  EarthConstraints	request_constraints;
  EarthConstraints	new_constraints;

  old_constraints = (EarthConstraints)(old->core.constraints);
  request_constraints = (EarthConstraints)(request->core.constraints);
  new_constraints = (EarthConstraints)(new->core.constraints);
#endif

  printf("%f %f - %f %f\n",
	 EARTH_I_C(request).latitude,
	 EARTH_I_C(request).longitude,
	 EARTH_I_C(old).latitude,
	 EARTH_I_C(old).longitude);
  self = new->core.parent;
  if ((EARTH_I_C(request).latitude != EARTH_I_C(old).latitude) ||
      (EARTH_I_C(request).longitude != EARTH_I_C(old).longitude))
    configure_child(new,&(EARTH.msi),&(EARTH.mso));
}

static void		ConstraintDestroy(self) 
Widget			self;
{ 
}

/*
 * public
 */

/*
 * SHOULD SET A PUBLIC PROC FOR ALL FLOAT RESOURCES
 */

void			XearthSetViewMag(self,view_mag)
Widget			self;
float			view_mag;
{
  if (!VIEW_MAG_OK(view_mag))
    {
      XearthWarning("EarthWidget: view mag must be strictly positive");
      return ;
    }
  mso_reset(&(EARTH.msi),&(EARTH.mso));
  mro_reset(&(EARTH.mro));
  EARTH.msi.view_mag = view_mag;
  EARTH.viewMag = view_mag;
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
  reput_image(self);
  configure_children(self);
}

void			XearthSetViewRot(self,view_rot)
Widget			self;
float			view_rot;
{
  if (!VIEW_ROT_OK(view_rot))
    {
      XearthWarning("EarthWidget: view rot must be strictly positive");
      return ;
    }
  mso_reset(&(EARTH.msi),&(EARTH.mso));
  mro_reset(&(EARTH.mro));
  EARTH.msi.view_rot = view_rot;
  EARTH.viewRot = view_rot;
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
  reput_image(self);
  configure_children(self);
}

void			XearthSetViewLat(self,view_lat)
Widget			self;
float			view_lat;
{
  if (!VIEW_LAT_OK(view_lat))
    {
      XearthWarning("EarthWidget: view lat must be strictly positive");
      return ;
    }
  mso_reset(&(EARTH.msi),&(EARTH.mso));
  mro_reset(&(EARTH.mro));
  EARTH.msi.view_lat = view_lat;
  EARTH.viewLat = view_lat;
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
  reput_image(self);
  configure_children(self);
}

void			XearthSetViewLon(self,view_lon)
Widget			self;
float			view_lon;
{
  if (!VIEW_LON_OK(view_lon))
    {
      XearthWarning("EarthWidget: view lon must be strictly positive");
      return ;
    }
  mso_reset(&(EARTH.msi),&(EARTH.mso));
  mro_reset(&(EARTH.mro));
  EARTH.msi.view_lon = view_lon;
  EARTH.viewLon = view_lon;
  scan_map(&(EARTH.msi),&(EARTH.mso));
  init_mri_from_epo(&(EARTH.epo),&(EARTH.mri));
  do_dots(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro));
  render(&(EARTH.msi),&(EARTH.mso),&(EARTH.mri),&(EARTH.mro),
	 (t_row_proc)row,
	 self);
  reput_image(self);
  configure_children(self);
}

void			XearthChildGetLatitude(child,latitude)
Widget			child;
#if NeedWidePrototypes
double			*latitude;
#else
float			*latitude;
#endif
{
  Widget		self;
  
  self = child->core.parent;
  *latitude = (EARTH_I_C(child).latitude);
}

void			XearthChildSetLatitude(child,latitude)
Widget			child;
float			latitude;
{
  Widget		self;

  self = child->core.parent;
  EARTH_I_C(child).latitude = latitude;
  configure_child(child,&(EARTH.msi),&(EARTH.mso));
}

void			XearthChildGetLongitude(child,longitude)
Widget			child;
#if NeedWidePrototypes
double			*longitude;
#else
float			*longitude;
#endif
{
  Widget		self;
  
  self = child->core.parent;
  *longitude = (EARTH_I_C(child).longitude);
}

void			XearthChildSetLongitude(child,longitude)
Widget			child;
float			longitude;
{
  Widget		self;
  
  self = child->core.parent;
  EARTH_I_C(child).longitude = longitude;
  configure_child(child,&(EARTH.msi),&(EARTH.mso));
}

int			XearthGetXY(self,
				    latitude,
				    longitude,
				    xret,
				    yret)
Widget			self;
float			latitude;
float			longitude;
Position		*xret;
Position		*yret;
{
  double		lat;
  double		lon;
  double		pos[3];
  t_map_scan_output	*mso;
  
  mso = &(EARTH.mso);
  lat = latitude * (M_PI/180);
  lon = longitude * (M_PI/180);
  pos[0] = sin(lon) * cos(lat);
  pos[1] = sin(lat);
  pos[2] = cos(lon) * cos(lat);
  XFORM_ROTATE(pos,EARTH.mso.view_pos_info);
  if (EARTH.msi.proj_type == ProjTypeOrthographic)
  {
    if (pos[2] <= 0)
      {
	return (0);
      }
  }
  else /* ProjTypeMercator */
  {
    pos[0] = MERCATOR_X(pos[0], pos[2]);
    pos[1] = MERCATOR_Y(pos[1]);
  }
  (*xret) = XPROJECT(pos[0]);
  (*yret) = YPROJECT(pos[1]);
  return (1);
}
