/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * mapdither.c
 * kirk johnson
 * july 1993
 *
 * RCS $Id: dither.c,v 1.18 1995/09/25 01:09:14 tuna Exp $
 *
 * Copyright (C) 1989, 1990, 1993, 1994, 1995 Kirk Lauritz Johnson
 *
 * Parts of the source code (as marked) are:
 *   Copyright (C) 1989, 1990, 1991 by Jim Frost
 *   Copyright (C) 1992 by Jamie Zawinski <jwz@lucid.com>
 *
 * Permission to use, copy, modify and freely distribute xearth for
 * non-commercial and not-for-profit purposes is hereby granted
 * without fee, provided that both the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.
 *
 * Unisys Corporation holds worldwide patent rights on the Lempel Zev
 * Welch (LZW) compression technique employed in the CompuServe GIF
 * image file format as well as in other formats. Unisys has made it
 * clear, however, that it does not require licensing or fees to be
 * paid for freely distributed, non-commercial applications (such as
 * xearth) that employ LZW/GIF technology. Those wishing further
 * information about licensing the LZW patent should contact Unisys
 * directly at (lzw_info@unisys.com) or by writing to
 *
 *   Unisys Corporation
 *   Welch Licensing Department
 *   M/S-C1SW19
 *   P.O. Box 500
 *   Blue Bell, PA 19424
 *
 * The author makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express or
 * implied warranty.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "earthmisc.h"
#include "mapdither.h"
#include "XearthI.h"

void dither_setup(msi,mdo,ncolors)
     t_map_scan_input *msi;
     t_map_dither_output *mdo;
     int ncolors;
{
  int      i;
  int      val;
  int      half;
  unsigned nbytes;
  t_status status;

  half = (ncolors - 2) / 2;
  ncolors = half*2 + 2;
  mdo->dither_ncolors = ncolors;

  if ((mdo->level = XEARTH_ALLOC_PROC(ncolors,
				      "xearth",
				      "dither_setup:level",
				      &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }

  nbytes = ncolors * 3;

  if ((mdo->dither_colormap = XEARTH_ALLOC_PROC(nbytes,
						"xearth",
						"dither_setup:dither_colormap",
						&status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  bzero((char *) mdo->dither_colormap, nbytes);

  mdo->level[0] = 0;
  for (i=1; i<=half; i++)
  {
    val = (i * 255) / half;

    mdo->dither_colormap[i*3+0] = 0;
    mdo->dither_colormap[i*3+1] = val;
    mdo->dither_colormap[i*3+2] = 0;
    mdo->level[i] = val;

    i += half;
    mdo->dither_colormap[i*3+0] = 0;
    mdo->dither_colormap[i*3+1] = 0;
    mdo->dither_colormap[i*3+2] = val;
    mdo->level[i] = val;
    i -= half;
  }

  mdo->dither_colormap[(ncolors-1)*3+0] = 255;
  mdo->dither_colormap[(ncolors-1)*3+1] = 255;
  mdo->dither_colormap[(ncolors-1)*3+2] = 255;
  mdo->level[ncolors-1] = 255;

  for (i=0; i<256; i++)
  {
    val = (i * half + 127) / 255;

    mdo->grn_idx[i] = val;
    
    if (val == 0)
      mdo->blu_idx[i] = val;
    else
      mdo->blu_idx[i] = val + half;
  }

  nbytes = (sizeof(t_s32) * 2) * (msi->wdth+2);

  if ((mdo->curr = XEARTH_ALLOC_PROC(nbytes,
				     "xearth",
				     "dither_setup:curr",
				     &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  bzero((char *) mdo->curr, nbytes);
  mdo->curr += 2;

  if ((mdo->next = XEARTH_ALLOC_PROC(nbytes,
				     "xearth",
				     "dither_setup:next",
				     &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  bzero((char *) mdo->next, nbytes);
  mdo->next += 2;

  mdo->even_row = 1;
}


void dither_row(msi,mdo,row, rslt)
     t_map_scan_input	*msi;
     t_map_dither_output *mdo;
     u_char  *row;
     t_u32 *rslt;
{
  if (mdo->even_row)
    dither_row_ltor(msi,mdo,row, rslt);
  else
    dither_row_rtol(msi,mdo,row, rslt);

  mdo->even_row = !mdo->even_row;
}


void			dither_cleanup(mdo)
t_map_dither_output	*mdo;
{
  XEARTH_FREE_PROC(mdo->curr - 2,
		   "xearth",
		   "*:*");
  XEARTH_FREE_PROC(mdo->next - 2,
		   "xearth",
		   "*:*");
  XEARTH_FREE_PROC(mdo->dither_colormap,
		   "xearth",
		   "*:dither_colormap");
  XEARTH_FREE_PROC(mdo->level,
		   "xearth",
		   "*:level");
}


void dither_row_ltor(msi,mdo,row, rslt)
     t_map_scan_input		*msi;
     t_map_dither_output	*mdo;
     u_char  *row;
     t_u32 *rslt;
{
  int      i, i_lim;
  int      grn, g_tmp;
  int      blu, b_tmp;
  int      idx;
  u_char  *rowtmp;
  t_s32 *currtmp;
  t_s32 *nexttmp;

  rowtmp  = row;
  currtmp = mdo->curr;
  nexttmp = mdo->next;

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    grn = rowtmp[1];
    blu = rowtmp[2];

    if ((grn == 0) && (blu == 0))
    {
      rslt[i] = 0;
    }
    else if ((grn == 255) && (blu == 255))
    {
      rslt[i] = mdo->dither_ncolors - 1;
    }
    else
    {
      grn += currtmp[0];
      blu += currtmp[1];

      if (grn > blu)
      {
	if (grn < 0)
	  grn = 0;
	else if (grn > 255)
	  grn = 255;

	idx  = mdo->grn_idx[grn];
	grn -= mdo->level[idx];
      }
      else
      {
	if (blu < 0)
	  blu = 0;
	else if (blu > 255)
	  blu = 255;

	idx  = mdo->blu_idx[blu];
	blu -= mdo->level[idx];
      }

      rslt[i] = idx;

      /* conceptually, what we want here is something like
       *
       *   g_tmp = (grn<0) ? 7 : 8;
       *   b_tmp = (blu<0) ? 7 : 8;
       *   currtmp[ 2] += ((grn * 7) + g_tmp) >> 4;
       *   currtmp[ 3] += ((blu * 7) + b_tmp) >> 4;
       *   nexttmp[ 2] += ((grn * 1) + g_tmp) >> 4;
       *   nexttmp[ 3] += ((blu * 1) + b_tmp) >> 4;
       *   nexttmp[ 0] += ((grn * 5) + g_tmp) >> 4;
       *   nexttmp[ 1] += ((blu * 5) + b_tmp) >> 4;
       *   nexttmp[-2] += ((grn * 3) + g_tmp) >> 4;
       *   nexttmp[-1] += ((blu * 3) + b_tmp) >> 4;
       *
       * but we can get tighter code by computing the product terms
       * via a sequence of additions into g_tmp and b_tmp
       */
      g_tmp = ((grn<0) ? 7 : 8) + grn;
      b_tmp = ((blu<0) ? 7 : 8) + blu;
      nexttmp[2] += (g_tmp >> 4);
      nexttmp[3] += (b_tmp >> 4);
      grn += grn;
      blu += blu;
      g_tmp += grn;
      b_tmp += blu;
      nexttmp[-2] += (g_tmp >> 4);
      nexttmp[-1] += (b_tmp >> 4);
      g_tmp += grn;
      b_tmp += blu;
      nexttmp[0] += (g_tmp >> 4);
      nexttmp[1] += (b_tmp >> 4);
      g_tmp += grn;
      b_tmp += blu;
      currtmp[2] += (g_tmp >> 4);
      currtmp[3] += (b_tmp >> 4);
    }

    rowtmp  += 3;
    currtmp += 2;
    nexttmp += 2;
  }

  currtmp = mdo->curr;
  mdo->curr    = mdo->next;
  mdo->next    = currtmp;
  bzero((char *) mdo->next, (unsigned) ((sizeof(t_s32) * 2) * msi->wdth));
}


void dither_row_rtol(msi,mdo,row, rslt)
     t_map_scan_input	*msi;
     t_map_dither_output	*mdo;
     u_char  *row;
     t_u32 *rslt;
{
  int      i;
  int      grn, g_tmp;
  int      blu, b_tmp;
  int      idx;
  u_char  *rowtmp;
  t_s32 *currtmp;
  t_s32 *nexttmp;

  rowtmp  = row  + 3*(msi->wdth-1);
  currtmp = mdo->curr + 2*(msi->wdth-1);
  nexttmp = mdo->next + 2*(msi->wdth-1);

  for (i=(msi->wdth-1); i>=0; i--)
  {
    grn = rowtmp[1];
    blu = rowtmp[2];

    if ((grn == 0) && (blu == 0))
    {
      rslt[i] = 0;
    }
    else if ((grn == 255) && (blu == 255))
    {
      rslt[i] = mdo->dither_ncolors - 1;
    }
    else
    {
      grn += currtmp[0];
      blu += currtmp[1];

      if (grn > blu)
      {
	if (grn < 0)
	  grn = 0;
	else if (grn > 255)
	  grn = 255;

	idx  = mdo->grn_idx[grn];
	grn -= mdo->level[idx];
      }
      else
      {
	if (blu < 0)
	  blu = 0;
	else if (blu > 255)
	  blu = 255;

	idx  = mdo->blu_idx[blu];
	blu -= mdo->level[idx];
      }

      rslt[i] = idx;

      /* conceptually, what we want here is something like
       *
       *   g_tmp = (grn<0) ? 7 : 8;
       *   b_tmp = (blu<0) ? 7 : 8;
       *   currtmp[-2] += ((grn * 7) + g_tmp) >> 4;
       *   currtmp[-1] += ((blu * 7) + b_tmp) >> 4;
       *   nexttmp[-2] += ((grn * 1) + g_tmp) >> 4;
       *   nexttmp[-1] += ((blu * 1) + b_tmp) >> 4;
       *   nexttmp[ 0] += ((grn * 5) + g_tmp) >> 4;
       *   nexttmp[ 1] += ((blu * 5) + b_tmp) >> 4;
       *   nexttmp[ 2] += ((grn * 3) + g_tmp) >> 4;
       *   nexttmp[ 3] += ((blu * 3) + b_tmp) >> 4;
       *
       * but we can get tighter code by computing the product terms
       * via a sequence of additions into g_tmp and b_tmp
       */
      g_tmp = ((grn<0) ? 7 : 8) + grn;
      b_tmp = ((blu<0) ? 7 : 8) + blu;
      nexttmp[-2] += (g_tmp >> 4);
      nexttmp[-1] += (b_tmp >> 4);
      grn += grn;
      blu += blu;
      g_tmp += grn;
      b_tmp += blu;
      nexttmp[2] += (g_tmp >> 4);
      nexttmp[3] += (b_tmp >> 4);
      g_tmp += grn;
      b_tmp += blu;
      nexttmp[0] += (g_tmp >> 4);
      nexttmp[1] += (b_tmp >> 4);
      g_tmp += grn;
      b_tmp += blu;
      currtmp[-2] += (g_tmp >> 4);
      currtmp[-1] += (b_tmp >> 4);
    }

    rowtmp  -= 3;
    currtmp -= 2;
    nexttmp -= 2;
  }

  currtmp = mdo->curr;
  mdo->curr    = mdo->next;
  mdo->next    = currtmp;
  bzero((char *) mdo->next, (unsigned) ((sizeof(t_s32) * 2) * msi->wdth));
}


void mono_dither_setup(msi,mdo)
t_map_scan_input	*msi;
t_map_dither_output	*mdo;
{
  int      i;
  unsigned nbytes;
  t_status status;

  nbytes = sizeof(t_s32) * (msi->wdth+2);

  if ((mdo->curr = XEARTH_ALLOC_PROC(nbytes,
				     "xearth",
				     "mono_dither_setup:curr",
				     &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  for (i=0; i<(msi->wdth+2); i++)
    mdo->curr[i] = (random() & ((1<<9)-1)) - (1<<8);
  mdo->curr += 1;

  if ((mdo->next = XEARTH_ALLOC_PROC(nbytes,
				     "xearth",
				     "mono_dither_setup:next",
				     &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  bzero((char *) mdo->next, nbytes);
  mdo->next += 1;

  mdo->even_row = 1;
}


void mono_dither_row(msi,mdo,row, rslt)
     t_map_scan_input	*msi;
     t_map_dither_output	*mdo;
     u_char  *row;
     t_u32 *rslt;
{
  int i, i_lim;

  /* convert row to gray scale (could save a few instructions per
   * pixel by integrating this into the mono_dither_row_* functions)
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    rslt[i] = (2 * row[0]) + (5 * row[1]) + row[2];
    row += 3;
  }

  /* dither to 0s (black) and 1s (white)
   */
  if (mdo->even_row)
    mono_dither_row_ltor(msi,mdo,rslt);
  else
    mono_dither_row_rtol(msi,mdo,rslt);

  mdo->even_row = !mdo->even_row;
}


void mono_dither_cleanup(mdo)
t_map_dither_output	*mdo;
{
  XEARTH_FREE_PROC(mdo->curr - 1,
		   "xearth",
		   "*:curr");
  XEARTH_FREE_PROC(mdo->next - 1,
		   "xearth",
		   "*:next");
}


void mono_dither_row_ltor(msi,mdo,row)
     t_map_scan_input *msi;
     t_map_dither_output *mdo;
     t_u32 *row;
{
  int      i, i_lim;
  int      val, tmp;
  t_u32 *rowtmp;
  t_s32 *currtmp;
  t_s32 *nexttmp;

  rowtmp  = row;
  currtmp = mdo->curr;
  nexttmp = mdo->next;

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    val = rowtmp[0];

    if (val == 0)
    {
      rowtmp[0] = 0;
    }
    else if (val == 2040)
    {
      rowtmp[0] = 1;
    }
    else
    {
      val += currtmp[0];

      if (val > 1020)
      {
	rowtmp[0] = 1;
	val -= 2040;
      }
      else
      {
	rowtmp[0] = 0;
      }

      /* conceptually, what we want here is something like
       *
       *   tmp = (val < 0) ? 7 : 8;
       *   currtmp[ 1] += ((val * 7) + tmp) >> 4;
       *   nexttmp[ 1] += ((val * 1) + tmp) >> 4;
       *   nexttmp[ 0] += ((val * 5) + tmp) >> 4;
       *   nexttmp[-1] += ((val * 3) + tmp) >> 4;
       *
       * but we can get tighter code by computing the product terms
       * via a sequence of additions into tmp
       */
      tmp = ((val < 0) ? 7 : 8) + val;
      nexttmp[1] += (tmp >> 4);
      val += val;
      tmp += val;
      nexttmp[-1] += (tmp >> 4);
      tmp += val;
      nexttmp[0] += (tmp >> 4);
      tmp += val;
      currtmp[1] += (tmp >> 4);
    }

    rowtmp  += 1;
    currtmp += 1;
    nexttmp += 1;
  }

  currtmp = mdo->curr;
  mdo->curr    = mdo->next;
  mdo->next    = currtmp;
  bzero((char *) mdo->next, (unsigned) (sizeof(t_s32) * msi->wdth));
}


void mono_dither_row_rtol(msi,mdo,row)
     t_map_scan_input *msi;
     t_map_dither_output *mdo;
     t_u32 *row;
{
  int      i;
  int      val, tmp;
  t_u32 *rowtmp;
  t_s32 *currtmp;
  t_s32 *nexttmp;

  rowtmp  = row  + (msi->wdth-1);
  currtmp = mdo->curr + (msi->wdth-1);
  nexttmp = mdo->next + (msi->wdth-1);

  for (i=(msi->wdth-1); i>=0; i--)
  {
    val = rowtmp[0];

    if (val == 0)
    {
      rowtmp[0] = 0;
    }
    else if (val == 2040)
    {
      rowtmp[0] = 1;
    }
    else
    {
      val += currtmp[0];

      if (val > 1020)
      {
	rowtmp[0] = 1;
	val -= 2040;
      }
      else
      {
	rowtmp[0] = 0;
      }

      /* conceptually, what we want here is something like
       *
       *   tmp = (val < 0) ? 7 : 8;
       *   currtmp[-1] += ((val * 7) + tmp) >> 4;
       *   nexttmp[-1] += ((val * 1) + tmp) >> 4;
       *   nexttmp[ 0] += ((val * 5) + tmp) >> 4;
       *   nexttmp[ 1] += ((val * 3) + tmp) >> 4;
       *
       * but we can get tighter code by computing the product terms
       * via a sequence of additions into tmp
       */
      tmp = ((val < 0) ? 7 : 8) + val;
      nexttmp[-1] += (tmp >> 4);
      val += val;
      tmp += val;
      nexttmp[1] += (tmp >> 4);
      tmp += val;
      nexttmp[0] += (tmp >> 4);
      tmp += val;
      currtmp[-1] += (tmp >> 4);
    }

    rowtmp  -= 1;
    currtmp -= 1;
    nexttmp -= 1;
  }

  currtmp = mdo->curr;
  mdo->curr    = mdo->next;
  mdo->next    = currtmp;
  bzero((char *) mdo->next, (unsigned) (sizeof(t_s32) * msi->wdth));
}
