/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * maprender.c
 * kirk johnson
 * october 1993
 *
 * RCS $Id: render.c,v 1.20 1995/09/25 01:10:37 tuna Exp $
 *
 * Copyright (C) 1989, 1990, 1993, 1994, 1995 Kirk Lauritz Johnson
 *
 * Parts of the source code (as marked) are:
 *   Copyright (C) 1989, 1990, 1991 by Jim Frost
 *   Copyright (C) 1992 by Jamie Zawinski <jwz@lucid.com>
 *
 * Permission to use, copy, modify and freely distribute xearth for
 * non-commercial and not-for-profit purposes is hereby granted
 * without fee, provided that both the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.
 *
 * Unisys Corporation holds worldwide patent rights on the Lempel Zev
 * Welch (LZW) compression technique employed in the CompuServe GIF
 * image file format as well as in other formats. Unisys has made it
 * clear, however, that it does not require licensing or fees to be
 * paid for freely distributed, non-commercial applications (such as
 * xearth) that employ LZW/GIF technology. Those wishing further
 * information about licensing the LZW patent should contact Unisys
 * directly at (lzw_info@unisys.com) or by writing to
 *
 *   Unisys Corporation
 *   Welch Licensing Department
 *   M/S-C1SW19
 *   P.O. Box 500
 *   Blue Bell, PA 19424
 *
 * The author makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express or
 * implied warranty.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "earthmisc.h"
#include "mapscan.h"
#include "maprender.h"
#include "XearthI.h"

int dot_comp(a, b)
     const void *a;
     const void *b;
{
  return (((const ScanDot *) a)->y - ((const ScanDot *) b)->y);
}

void render_rows_setup(mso,mri,mro)
t_map_scan_output	*mso;
t_map_render_input	*mri;
t_map_render_output	*mro;
{
  int i;

  mro->scanbitcnt = mso->scanbits->count;
  mro->scanbit    = (ScanBit *) mso->scanbits->elts;
  mro->dotcnt     = mro->dots->count;
  mro->dot        = (ScanDot *) mro->dots->elts;

  /* precompute table for translating between
   * scan buffer values and pixel types
   */
  for (i=0; i<256; i++)
    if (i == 0)
      mro->scan_to_pix[i] = PixTypeSpace;
    else if (i > 64)
      mro->scan_to_pix[i] = PixTypeLand;
    else
      mro->scan_to_pix[i] = PixTypeWater;
}


void render_next_row(msi,mro,buf, idx)
     t_map_scan_input *msi;
     t_map_render_output *mro;
     t_s32 *buf;
     int     idx;
{
  int      i, i_lim;
  int      tmp;
  int      _scanbitcnt;
  ScanBit *_scanbit;

  bzero((char *) buf, (unsigned) (sizeof(t_s32) * msi->wdth));

  /* explicitly copy mro->scanbitcnt and scanbit to local variables
   * to help compilers figure out that they can be registered
   */
  _scanbitcnt = mro->scanbitcnt;
  _scanbit    = mro->scanbit;

  while ((_scanbitcnt > 0) && (_scanbit->y == idx))
  {
    /* use i_lim to encourage compilers to register loop limit
     */
    i_lim = _scanbit->hi_x;
    tmp   = _scanbit->val;
    for (i=_scanbit->lo_x; i<=i_lim; i++)
      buf[i] += tmp;

    _scanbit    += 1;
    _scanbitcnt -= 1;
  }

  /* copy changes to mro->scanbitcnt and scanbit out to memory
   */
  mro->scanbitcnt = _scanbitcnt;
  mro->scanbit    = _scanbit;

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
    buf[i] = mro->scan_to_pix[(int) (buf[i] & 0xff)];

  while ((mro->dotcnt > 0) && (mro->dot->y == idx))
  {
    tmp = mro->dot->x;

    if (mro->dot->type == DotTypeStar)
    {
      if (buf[tmp] == PixTypeSpace)
        buf[tmp] = PixTypeStar;
    }
    else
    {
      switch (buf[tmp])
      {
      case PixTypeLand:
        buf[tmp] = PixTypeGridLand;
        break;

      case PixTypeWater:
        buf[tmp] = PixTypeGridWater;
        break;
      }
    }

    mro->dot    += 1;
    mro->dotcnt -= 1;
  }
}


void no_shade_row(msi,scanbuf, rslt)
     t_map_scan_input *msi;
     t_s32 *scanbuf;
     u_char *rslt;
{
  int i, i_lim;

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    switch (scanbuf[i])
    {
    case PixTypeSpace:        /* black */
      rslt[0] = 0;
      rslt[1] = 0;
      rslt[2] = 0;
      break;

    case PixTypeStar:         /* white */
    case PixTypeGridLand:
    case PixTypeGridWater:
      rslt[0] = 255;
      rslt[1] = 255;
      rslt[2] = 255;
      break;

    case PixTypeLand:         /* green */
      rslt[0] = 0;
      rslt[1] = 255;
      rslt[2] = 0;
      break;

    case PixTypeWater:        /* blue */
      rslt[0] = 0;
      rslt[1] = 0;
      rslt[2] = 255;
      break;

    default:
      assert(0);
    }

    rslt += 3;
  }
}


void compute_sun_vector(mso,mri,rslt)
     t_map_scan_output *mso;
     t_map_render_input *mri;
     double *rslt;
{
  rslt[0] = sin(mri->sun_lon * (M_PI/180)) * cos(mri->sun_lat * (M_PI/180));
  rslt[1] = sin(mri->sun_lat * (M_PI/180));
  rslt[2] = cos(mri->sun_lon * (M_PI/180)) * cos(mri->sun_lat * (M_PI/180));

  XFORM_ROTATE(rslt, mso->view_pos_info);
}


void orth_compute_inv_x(msi,mso,inv_x)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     double *inv_x;
{
  int i, i_lim;

  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
    inv_x[i] = INV_XPROJECT(i);
}


void orth_shade_row(msi,mso,mro,idx, scanbuf, sol, inv_x, rslt)
     t_map_scan_input	*msi;
     t_map_scan_output	*mso;
     t_map_render_output *mro;
     int     idx;
     t_s32 *scanbuf;
     double *sol;
     double *inv_x;
     u_char *rslt;
{
  int    i, i_lim;
  int    val;
  double x, y, z;
  double scale;
  double tmp;

  y = INV_YPROJECT(idx);

  /* save a little computation in the inner loop
   */
  tmp = 1 - (y*y);

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    switch (scanbuf[i])
    {
    case PixTypeSpace:        /* black */
      rslt[0] = 0;
      rslt[1] = 0;
      rslt[2] = 0;
      break;

    case PixTypeStar:         /* white */
    case PixTypeGridLand:
    case PixTypeGridWater:
      rslt[0] = 255;
      rslt[1] = 255;
      rslt[2] = 255;
      break;

    case PixTypeLand:         /* green, blue */
    case PixTypeWater:
      x = inv_x[i];
      z = tmp - (x*x);
      z = SQRT(z);
      scale = x*sol[0] + y*sol[1] + z*sol[2];
      if (scale < 0)
      {
	val = mro->night_val;
      }
      else
      {
	val = mro->day_val_base + (scale * mro->day_val_delta);
	if (val > 255)
	  val = 255;
	else
	  assert(val >= 0);
      }

      if (scanbuf[i] == PixTypeLand)
      {
	/* land (green) */
	rslt[0] = 0;
	rslt[1] = val;
	rslt[2] = 0;
      }
      else
      {
	/* water (blue) */
	rslt[0] = 0;
	rslt[1] = 0;
	rslt[2] = val;
      }
      break;

    default:
      assert(0);
    }

    rslt += 3;
  }
}


void merc_shade_row(msi,mso,mro,idx, scanbuf, sol, rslt)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     t_map_render_output *mro;
     int     idx;
     t_s32 *scanbuf;
     double *sol;
     u_char *rslt;
{
  int    i, i_lim;
  int    val;
  double x, y, z;
  double sin_theta;
  double cos_theta;
  double scale;
  double tmp;

  y = INV_YPROJECT(idx);
  y = INV_MERCATOR_Y(y);

  /* conceptually, on each iteration of the i loop, we want:
   *
   *   x = sin(INV_XPROJECT(i)) * sqrt(1 - (y*y));
   *   z = cos(INV_XPROJECT(i)) * sqrt(1 - (y*y));
   *
   * computing this directly is rather expensive, however, so we only
   * compute the first (i=0) pair of values directly; all other pairs
   * (i>0) are obtained through successive rotations of the original
   * pair (by inv_proj_scale radians).
   */

  /* compute initial (x, z) values
   */
  tmp = sqrt(1 - (y*y));
  x   = sin(INV_XPROJECT(0)) * tmp;
  z   = cos(INV_XPROJECT(0)) * tmp;

  /* compute rotation coefficients used
   * to find subsequent (x, z) values
   */
  tmp = mso->inv_proj_scale;
  sin_theta = sin(tmp);
  cos_theta = cos(tmp);

  /* use i_lim to encourage compilers to register loop limit
   */
  i_lim = msi->wdth;
  for (i=0; i<i_lim; i++)
  {
    switch (scanbuf[i])
    {
    case PixTypeSpace:        /* black */
      rslt[0] = 0;
      rslt[1] = 0;
      rslt[2] = 0;
      break;

    case PixTypeStar:         /* white */
    case PixTypeGridLand:
    case PixTypeGridWater:
      rslt[0] = 255;
      rslt[1] = 255;
      rslt[2] = 255;
      break;

    case PixTypeLand:         /* green, blue */
    case PixTypeWater:
      scale = x*sol[0] + y*sol[1] + z*sol[2];
      if (scale < 0)
      {
	val = mro->night_val;
      }
      else
      {
	val = mro->day_val_base + (scale * mro->day_val_delta);
	if (val > 255)
	  val = 255;
	else
	  assert(val >= 0);
      }

      if (scanbuf[i] == PixTypeLand)
      {
	/* land (green) */
	rslt[0] = 0;
	rslt[1] = val;
	rslt[2] = 0;
      }
      else
      {
	/* water (blue) */
	rslt[0] = 0;
	rslt[1] = 0;
	rslt[2] = val;
      }
      break;

    default:
      assert(0);
    }

    /* compute next (x, z) values via 2-d rotation
     */
    tmp = (cos_theta * z) - (sin_theta * x);
    x   = (sin_theta * z) + (cos_theta * x);
    z   = tmp;

    rslt += 3;
  }
}


void render(msi,mso,mri,mro,rowfunc,data)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     t_map_render_input *mri;
     t_map_render_output *mro;
     t_row_proc rowfunc;
     void *data;
{
  int     i, i_lim;
  t_s32 *scanbuf;
  u_char *row;
  double *inv_x;
  double  sol[3];
  double  tmp;
  t_status status;

  if ((scanbuf = XEARTH_ALLOC_PROC(sizeof (t_s32) * msi->wdth,
				   "xearth",
				   "render:scanbuf",
				   &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  if ((row = XEARTH_ALLOC_PROC(msi->wdth * 3,
			       "xearth",
			       "render:row",
			       &status)) == NULL)
    {
      XearthWarning("XEARTH_ALLOC_PROC");
      exit(1);
    }
  
  inv_x = NULL;
  render_rows_setup(mso,mri,mro);

  if (mri->do_shade)
  {
    /* inv_x[] only gets used with orthographic projection
     */
    if (msi->proj_type == ProjTypeOrthographic)
    {
      if ((inv_x = XEARTH_ALLOC_PROC(sizeof (double) * msi->wdth,
				     "xearth",
				     "render:inv_x",
				     &status)) == NULL)
	{
	  XearthWarning("XEARTH_ALLOC_PROC");
	  exit(1);
	}
      orth_compute_inv_x(msi,mso,inv_x);
    }

    compute_sun_vector(mso,mri,sol);

    /* precompute shading parameters
     */
    mro->night_val     = mri->night * (255.99/100.0);
    tmp           = mri->terminator / 100.0;
    mro->day_val_base  = ((tmp * mri->day) + ((1-tmp) * mri->night))  * (255.99/100.0);
    mro->day_val_delta = (mri->day * (255.99/100.0)) - mro->day_val_base;
  }

  /* main render loop
   * (use i_lim to encourage compilers to register loop limit)
   */
  i_lim = msi->hght;
  for (i=0; i<i_lim; i++)
  {
    render_next_row(msi,mro,scanbuf, i);

    if (!mri->do_shade)
      no_shade_row(msi,scanbuf, row);
    else if (msi->proj_type == ProjTypeOrthographic)
      orth_shade_row(msi,mso,mro,i, scanbuf, sol, inv_x, row);
    else
      merc_shade_row(msi,mso,mro,i, scanbuf, sol, row);

    rowfunc(row,i,data);
  }

  XEARTH_FREE_PROC(scanbuf,
		   "xearth",
		   "*:scanbuf");
  XEARTH_FREE_PROC(row,
		   "xearth",
		   "*:row");

  if (inv_x != NULL) 
    {
      XEARTH_FREE_PROC(inv_x,
		       "xearth",
		       "*:inv_x");
    }
}


void do_dots(msi,mso,mri,mro)
t_map_scan_input *msi;
t_map_scan_output *mso;
t_map_render_input *mri;
t_map_render_output *mro;
{
  if (mri->do_stars) new_stars(msi,mri,mro,mri->star_freq);
  if (mri->do_grid) new_grid(msi,mso,mri,mro,mri->grid_big,mri->grid_small);

  qsort(mro->dots->elts, mro->dots->count, sizeof(ScanDot), dot_comp);
}

void			mro_alloc(mro)
t_map_render_output	*mro;
{
  t_status		status;

  if ((mro->dots = XEARTH_ARR_NEW(sizeof(ScanDot),
				  "mro_alloc",
				  &status)) == NULL)
    {
      XearthWarning("XEARTH_ARR_NEW");
      exit(1);
    }
}

void			mro_reset(mro)
t_map_render_output	*mro;
{
  ARR_EMPTY(mro->dots);
}

void			mro_cleanup(mro)
t_map_render_output	*mro;
{
  XearthArrDelete(mro->dots);
}

void new_stars(msi,mri,mro,freq)
     t_map_scan_input *msi;
     t_map_render_input *mri;
     t_map_render_output *mro;
     double freq;
{
  int      i;
  int      x, y;
  int      max_stars;
  ScanDot *newdot;
  t_status status;

  max_stars = msi->wdth * msi->hght * freq;

  for (i=0; i<max_stars; i++)
  {
    x = random() % msi->wdth;
    y = random() % msi->hght;

    if ((newdot = XearthArrNext(mro->dots,
			   &status)) == NULL)
      {
	XearthWarning("arr_next");
	exit(1);
      }
    newdot->x    = x;
    newdot->y    = y;
    newdot->type = DotTypeStar;

    if ((mri->big_stars) && (x+1 < msi->wdth) && ((random() % 100) < mri->big_stars))
    {
      if ((newdot = XearthArrNext(mro->dots,
			   &status)) == NULL)
      {
	XearthWarning("arr_next");
	exit(1);
      }
      newdot->x    = x+1;
      newdot->y    = y;
      newdot->type = DotTypeStar;
    }
  }
}


void new_grid(msi,mso,mri,mro,big, small)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     t_map_render_input *mri;
     t_map_render_output *mro;
     int big;
     int small;
{
  int    i, j;
  int    cnt;
  double lat, lon;
  double lat_scale, lon_scale;
  double cs_lat[2];
  double cs_lon[2];

  /* lines of longitude
   */
  lon_scale = M_PI / (2 * big);
  lat_scale = M_PI / (2 * big * small);
  for (i=(-2*big); i<(2*big); i++)
  {
    lon       = i * lon_scale;
    cs_lon[0] = cos(lon);
    cs_lon[1] = sin(lon);

    for (j=(-(big*small)+1); j<(big*small); j++)
    {
      lat       = j * lat_scale;
      cs_lat[0] = cos(lat);
      cs_lat[1] = sin(lat);

      new_grid_dot(msi,mso,mri,mro,cs_lat, cs_lon);
    }
  }

  /* lines of latitude
   */
  lat_scale = M_PI / (2 * big);
  for (i=(1-big); i<big; i++)
  {
    lat       = i * lat_scale;
    cs_lat[0] = cos(lat);
    cs_lat[1] = sin(lat);
    cnt       = 2 * ((int) ((cs_lat[0] * small) + 0.5)) * big;
    lon_scale = M_PI / cnt;

    for (j=(-cnt); j<cnt; j++)
    {
      lon       = j * lon_scale;
      cs_lon[0] = cos(lon);
      cs_lon[1] = sin(lon);

      new_grid_dot(msi,mso,mri,mro,cs_lat, cs_lon);
    }
  }
}


void new_grid_dot(msi,mso,mri,mro,cs_lat, cs_lon)
     t_map_scan_input *msi;
     t_map_scan_output *mso;
     t_map_render_input *mri;
     t_map_render_output *mro;
     double *cs_lat;
     double *cs_lon;
{
  int      x, y;
  double   pos[3];
  ScanDot *new;
  t_status status;

  pos[0] = cs_lon[1] * cs_lat[0];
  pos[1] = cs_lat[1];
  pos[2] = cs_lon[0] * cs_lat[0];

  XFORM_ROTATE(pos, mso->view_pos_info);

  if (msi->proj_type == ProjTypeOrthographic)
  {
    /* if the grid dot isn't visible, return immediately
     */
    if (pos[2] <= 0) return;
  }
  else /* (proj_type == ProjTypeMercator) */
  {
    /* apply mercator projection
     */
    pos[0] = MERCATOR_X(pos[0], pos[2]);
    pos[1] = MERCATOR_Y(pos[1]);
  }

  x = XPROJECT(pos[0]);
  y = YPROJECT(pos[1]);

  if ((x >= 0) && (x < msi->wdth) && (y >= 0) && (y < msi->hght))
  {
    if ((new = XearthArrNext(mro->dots,
			&status)) == NULL)
      {
	XearthWarning("arr_next");
	exit(1);
      }
    new->x    = x;
    new->y    = y;
    new->type = DotTypeGrid;
  }
}
