/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: arc.c,v 1.6 95/09/28 11:58:26 leon Exp $ */



#include <math.h>
#include <Kn/arc.h>
#include "TagP.h"
#ifdef LOOK3D
#include <Xm/Xm.h>
#endif /* LOOK3D */

#define TAG tag->tag



/*********************\
* 		      *
*  KlO  KnArc          *
*  BODY		      *
* 		      *
\*********************/



/* type desc */
KlType KnArcClass;


void
KnArcCopyFields(KnArc new, KnArc old)
{
    KnOCopyFields((KnO)new, (KnO)old);
    new->width = old->width;
    new->height = old->height;
    new->start = old->start;
    new->angle = old->angle;
}

/* init class fields and call super class Init proc */
void
KnArcInit(KnArc c, Widget knvas, KnO parent, KnPosition x, KnPosition y,
	   KnDimension xof, KnDimension yof, KnDimension start, KnDimension angle)
{
    KnOInit((KnO)c, knvas, parent, x, y);
    c->width = xof;
    c->height = yof;
    c->start = start*64;
    c->angle = angle*64;
}


void
KnArcClear(KnArc arc, Widget w)
{
    int extra;
    XPoint points[4];
    TagObject tag;
    
    tag = (TagObject)KnGetTag(w, (KnO)arc);
#ifdef LOOK3D    
    extra = TAG.line_width + TAG.shadowThickness;
#else /* LOOK3D */
    extra = TAG.line_width;
#endif /* LOOK3D */
    
    points[0].x = points[3].x = arc->x - extra;
    points[1].x = points[2].x = arc->x + arc->width + extra;
    points[0].y = points[1].y = arc->y - extra;
    points[2].y = points[3].y = arc->y + arc->height + extra;
    
    KlSend_clearbox(arc, points, 4, NULL);
    
    if(arc->anchors) {
	KnOClearAnchors((KnO)arc, w);
    }
}



KnO
KnArcContains(KnArc arc, Widget w, Int x, Int y)
{
    Boolean res;
    KnPosition rx = x, ry = y;

    if(arc->t) {
	TransformerInvTransform(arc->t, x, y, &rx, &ry);
    }
    res = ((rx >= arc->x) &&
	   (rx <= arc->x + arc->width) &&
	   (ry >= arc->y) &&
	   (ry <= arc->y + arc->height));
    return res ? (KnO)arc : NULL;
}



/* allocates storage space for a new arc
 * 
 * returns: the newly created Arc
 */
KnO
KnCreateArc(Widget knvas, KnO parent, KnPosition x, KnPosition y, KnDimension xof,
	    KnDimension yof, KnDimension start, KnDimension angle)
{
    KnArc arc;
    arc =  (KnArc)KlOMake(KnArcClass);
    KnArcInit(arc, knvas, parent, x, y, xof, yof, start, angle);
    return (KnO)arc;
}



void
KnArcCenter(KnArc arc, KnPosition *x, KnPosition *y)
{
    *x = arc->x + arc->width / 2;
    *y = arc->y + arc->height / 2;
}



void
KnArcBoundingBox(KnArc arc, Widget view, XRectangle *box)
{
    XPoint points[4];
    
    points[0].x = points[3].x = arc->x;
    points[1].x = points[2].x = arc->x + arc->width;
    points[0].y = points[1].y = arc->y;
    points[2].y = points[3].y = arc->y + arc->height;

    KnTransformPointList(view, arc, points, 4);
    GetBoundingBox(points, 4, box);
}


/* everything is aproximated with arcs */
void
KnArcDraw(KnArc arc, KnDrawingContext ct, Transformer tf) 
{

    Transformer t;    
    XPoint points[4];
    XRectangle box;
    TagObject tag;
    tag = (TagObject)KnGetTag(ct->w, (KnO)arc);
    
    t = tf;

    points[0].x = points[3].x = arc->x;
    points[1].x = points[2].x = arc->x + arc->width;
    points[0].y = points[1].y = arc->y;
    points[2].y = points[3].y = arc->y + arc->height;
    if(t)
	TransformerTransformPointList(t, points, 4, points);
    
    
    GetBoundingBox(points, 4, &box);
    if(KnRectInRegion(ct->clip, box.x, box.y, 
		     box.width, box.height)) {
	XDrawArc(ct->dpy, ct->win,
		 TAG.gc, points[0].x, points[0].y,
		 points[2].x - points[0].x,
		 points[3].y - points[0].y,
		 arc->start, arc->angle);
	if(DrawSelection(arc))
	    KnDrawSelectedBox(tag, ct, &box);
    }
}




void
KnArcSize(KnArc arc, KnDimension *w, KnDimension *h)
{
    XPoint points[3];
    register int tmp0, tmp1;
    
    points[0].x = arc->x;
    points[1].x = points[2].x = arc->x + arc->width;
    points[0].y = points[1].y = arc->y;
    points[2].y = arc->y + arc->height;
    if(arc->t)
	TransformerTransformPointList(arc->t, points, 3, points);
    
    tmp0 = points[1].x - points[0].x;
    tmp1 = points[1].y - points[0].y;
    *w = sqrt(tmp0*tmp0 + tmp1*tmp1);
    tmp0 = points[2].x - points[0].x;
    tmp1 = points[2].y - points[0].y;
    *h = sqrt(tmp0*tmp0 + tmp1*tmp1);
}


/*********************\
* 		      *
*  KnO  KnFArc         *
*  BODY		      *
* 		      *
\*********************/






/* type desc */
KlType KnFArcClass;


/* init class fields and call super class Init proc */
void
KnFArcInit(KnFArc c, Widget knvas, KnO parent, KnPosition x, KnPosition y,
	    KnDimension xof, KnDimension yof, KnDimension start, KnDimension angle)
{
    KnArcInit((KnArc)c, knvas, parent, x, y, xof, yof, start, angle);
}


/* allocates storage space for a new KnFArc
 * 
 * returns: the newly created KnFArc
 */
KnO
KnCreateFArc(Widget knvas, KnO parent, KnPosition x, KnPosition y,
	    KnDimension xof, KnDimension yof, KnDimension start, KnDimension angle)
{
    KnFArc arc;
    arc =  (KnFArc)KlOMake(KnFArcClass);
    KnFArcInit(arc, knvas, parent, x, y, xof, yof, start, angle);
    return (KnO)arc;
}



void
KnFArcDraw(KnFArc arc, KnDrawingContext ct, Transformer tf) 
{
    Transformer t;    
    XPoint points[4];
    XRectangle box;
    TagObject tag;
    int shw = 0;			/* shadow width */
    tag = (TagObject)KnGetTag(ct->w, (KnO)arc);
    
    t = tf;

	points[0].x = points[3].x = arc->x;
	points[1].x = points[2].x = arc->x + arc->width;
	points[0].y = points[1].y = arc->y;
	points[2].y = points[3].y = arc->y + arc->height;
    if(t)
	TransformerTransformPointList(t, points, 4, points);
	

    GetBoundingBox(points, 4, &box);
#ifdef LOOK3D
    shw = TAG.shadowThickness;
#endif
	if(KnRectInRegion(ct->clip, box.x, box.y, 
			 box.width+shw, box.height+shw)) {
#ifdef LOOK3D
	    /* yeah, this is totally empirical: It seems that only the shadowed
               part should have the specified thickness. the other (lightenned)
               will have only a thickness of 1 */
	    if(TAG.shadowThickness) {
		GC top_gc, bottom_gc;
		KnDimension top_thick = 1, bottom_thick = 1;
		if(XmSHADOW_OUT == TAG.shadowType) {
		    top_gc =  TAG.top_shadow_gc;
		    bottom_gc = TAG.bottom_shadow_gc;
		    bottom_thick = TAG.shadowThickness;
		}
		else {
		    top_gc =  TAG.bottom_shadow_gc;
		    bottom_gc = TAG.top_shadow_gc;
		    top_thick = TAG.shadowThickness;
		}
		KnSetRegion(ct->dpy, top_gc, ct->clip);
		KnSetRegion(ct->dpy, bottom_gc, ct->clip);
		/* here, we modify the current GC, because the clip region may
                   have been modified in the IconDraw function, and we must
                   work we the same clipping on the 3 GCs */
		KnSetRegion(ct->dpy, TAG.gc, ct->clip);
		XFillArc(ct->dpy, ct->win,
			 top_gc,
			 points[0].x - top_thick,
			 points[0].y - top_thick,
			 points[2].x - points[0].x,
			 points[3].y - points[0].y,
			 arc->start, arc->angle);
		XFillArc(ct->dpy, ct->win,
			 bottom_gc, 
			 points[0].x + bottom_thick,
			 points[0].y + bottom_thick,
			 points[2].x - points[0].x,
			 points[3].y - points[0].y,
			 arc->start, arc->angle);
	    }
#endif /* LOOK3D */
	    XFillArc(ct->dpy, ct->win,
		     TAG.gc, points[0].x, points[0].y,
		     points[2].x - points[0].x,
		     points[3].y - points[0].y,
			 arc->start, arc->angle);
	    if(DrawSelection(arc))
		KnDrawSelectedBox(tag, ct, &box);
	}
}


/*********************\
* 		      *
*  Module  Arc       *
*  INITIALISATIONS    *
* 		      *
\*********************/





/* declare new   selectors
 */
void
KnArcImport()
{
	
}



/* class initializations for Arc objects
 * 
 * returns: 
 */
void
KnArcClassInitialize()
{
     /* declare  types */
    KlDeclareSubType(&KnArcClass, "KnArc", KnOClass, sizeof(struct KnArcStruct));
    KlDeclareSubType(&KnFArcClass, "KnFArc", KnArcClass, sizeof(struct KnFArcStruct));

    /* declare methods */
    KlDeclareMethod(KnArcClass, KnOSelCopyFields, KnArcCopyFields);
    KlDeclareMethod(KnArcClass, KnOSelDraw, KnArcDraw);
    KlDeclareMethod(KnArcClass, KnOSelContains, KnArcContains);
    KlDeclareMethod(KnArcClass, KnOSelClear, KnArcClear);
    KlDeclareMethod(KnArcClass, KnOSelCenter, KnArcCenter);
    KlDeclareMethod(KnArcClass, KnOSelBoundingBox, KnArcBoundingBox);
    KlDeclareMethod(KnArcClass, KnOSelSize, KnArcSize);
    
    KlDeclareMethod(KnFArcClass, KnOSelDraw, KnFArcDraw);

}



KnO
KnCreateCircle(Widget knvas, KnO parent, KnPosition x, KnPosition y,
	    KnDimension xof, KnDimension yof)
{
    return KnCreateArc(knvas, parent, x, y, xof, yof, 0, 360);
}



KnO
KnCreateFCircle(Widget knvas, KnO parent, KnPosition x, KnPosition y,
	    KnDimension xof, KnDimension yof)
{
    return KnCreateFArc(knvas, parent, x, y, xof, yof, 0, 360);
}
