/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: group.c,v 1.12.2.5 96/03/15 15:09:00 leon Exp $ */


#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "misc.h"
#include "group.h"
#include "knoP.h"


/* type desc */
KlType KnGroupClass;



void
KnGroupInit(KnGroup group, Widget knvas, KnO parent,
	    KnPosition x, KnPosition y)
{
    KnOInit((KnO)group, knvas, parent, x, y);
    group->children = KlListNMake(0);
    KlIncRef(group->children);
    group->t = TransformerMake();
    group->t->flags &= ~T_IDENTITY ;
    TransformerTranslate(group->t, x, y);
    group->x = 0;
    group->y = 0;
}

void
KnGroupCopyFields(KnGroup newg, KnGroup oldg)
{
    int i;
    KlList list = oldg->children;
    newg->children = KlListNMake(list->size);
    KlIncRef(newg->children);
    for (i = 0; i < list->size; i++) {
	KlIncRef(newg->children->list[i] = list->list[i]);
    }
}

void
KnGroupFree(KnGroup group)
{
    KlDecRef(group->children);
    KnOFree((KnO)group);
}


/* allocates storage space for a new KnGroup object */
KnO
KnCreateGroup(Widget knvas, KnO parent, KnPosition x, KnPosition y)
{  
    KnGroup group;
    group =  (KnGroup)KlOMake(KnGroupClass);
    KnGroupInit(group, knvas, parent, x, y);
    return (KnO)group;
}





KnO
KnGroupAdd(KnGroup g, KnO kno)
{
    KlListAppend(g->children, (KlO)kno);
    kno->parent = (KnO)g;
    return kno;
}



void
KnGroupRemove(KnGroup g, KnO kno)
{
    Cardinal n;

    for(n = 0; n < g->children->size; n++) {
	if((KnO)g->children->list[n] == kno) {
	    KlListDeletePos(g->children, n);
	}
    }
}




void
KnGroupDraw(KnGroup me, KnDrawingContext ct, Transformer tf) 
{
    Transformer t;   
    Cardinal i;
    KnO child;
    Boolean need_free;
    
    for(i = 0; i < me->children->size; i++) {
	child = (KnO)me->children->list[i];
	if(child && KnIsMapped(child)) {
	    /* draw only mapped objects */
	    t = tf;
	    need_free = False;
	    if(t) {
		if(child->t) {
		    t = TransformerMakeCopy(tf);
		    TransformerPremultiply(t, child->t);
		    if(TransformerIdentity(t)) {
			free(t);
			t = NULL;
		    }
		    else
			need_free = True;
		}
	    }
	    else {
		t = child->t;
	    }
	    KlSend_draw(child, ct, t);
	    if(need_free) {
		free(t);
		need_free = False;
	    }
	}
    }
    
    if(DrawSelection(me)) {
	XRectangle box;
	KlSend_boundingbox((KnO)me, ct->w, &box);
	KnDrawSelectedBox((KnTag)KnGetTag(ct->w, (KnO)me), ct, &box);
    }
}

void
KnGroupClear(KnGroup me, Widget knvas)
{
    KnO child;
    int i;

    for(i = 0; i < me->children->size; i++) {
	child = (KnO)me->children->list[i];
	if(child && KnIsMapped(child)) {
	    /* force redraw of children, since the group has to be redrawn */
	    child->flags &= ~KnCLEARED_FLAG;
	    KlSend_clear(child, knvas);
	}
    }
}




KnO
KnGroupContains(KnGroup group, Widget knvas, Int x, Int y)
{
    KnO res, child;
    KnPosition rx = x, ry = y;
    int i;

    if(group->t) 
	TransformerInvTransform(group->t, x, y, &rx, &ry);
    else {
	rx -= group->x;
	ry -= group->y;
    }
    for(i = group->children->size-1; i >= 0; i--) {
	child = (KnO)group->children->list[i];
	if(child && KnIsMapped(child)) {
	    res = (KnO)KlSend_contains(child, knvas,
				       (Int)rx, (Int)ry);
	    if(res)
		return res;
	}
    }
    return NULL;
}





void
KnGroupBoundingBox(KnGroup group, Widget view, XRectangle *box)
{
    KnO child;
    int i;
    KnRegion r;
    XRectangle cbox;

    r = KnCreateRegion();
    for(i = 0; i < group->children->size; i++) {
	child = (KnO)group->children->list[i];
	if(child && KnIsMapped(child)) {
	    KlSend_boundingbox(child, view, &cbox);
	    KnUnionRectWithRegion(&cbox, r, r);
	}
    }
    KnClipBox(r, box);
    KnDestroyRegion(r);
}



/* XtDispatch this will always return True, because the event filter, is ok.
   should install the translation's event filter in the widget */
Int
KnGroupDispatch(KnGroup group, Widget w, XEvent *event, Int x, Int y)
{
    KnPosition rx = x, ry = y;
    int i;
    KnO child;
    XtTMRec tm;
    KnvasWidget cw = (KnvasWidget) w;

    if( ! KnIsSensitive(group)) return (Int)False;
    
    if(group->interactor) {
	if(KlSend_contains(group, w, (Int)x, (Int)y)) {
	    cw->knvas.target = (KnO)group;

	    tm = w->core.tm;
	    w->core.tm = group->interactor->tm;
	    
	    _XtTranslateEvent(w, event);
	    
	    group->interactor->tm = w->core.tm;
	    w->core.tm = tm;
	    
	    return (Int)(cw->knvas.dispatched);
	}
    }
    if(group->t) 
	TransformerInvTransform(group->t, x, y, &rx, &ry);
    for(i = group->children->size - 1; i >=0; i--) {
	child = (KnO)group->children->list[i];
	if(KnIsMapped(child) && KnIsSensitive(child)) {
	    if(KlSend_dispatch(child, w, event, (Int)rx, (Int)ry))
		return (Int)(cw->knvas.dispatched);
	}
    }
    return (Int)False;
}



/* declare new   selectors
 */
void
KnGroupImport()
{
	
}



/* class initializations for KnGroup objects
 * 
 * returns: 
 */
void
KnGroupClassInitialize()
{
     /* declare  type */
    KlDeclareSubType(&KnGroupClass, "KnGroup", KnOClass, sizeof(struct KnGroupStruct));

    /* declare methods */
    KlDeclareMethod(KnGroupClass, KlSelMake, KnCreateGroup);
    KlDeclareMethod(KnGroupClass, KlSelFree, KnGroupFree);
    KlDeclareMethod(KnGroupClass, KnOSelDraw, KnGroupDraw);
    KlDeclareMethod(KnGroupClass, KnOSelClear, KnGroupClear);
    KlDeclareMethod(KnGroupClass, KnOSelContains, KnGroupContains);
    KlDeclareMethod(KnGroupClass, KnOSelDispatch, KnGroupDispatch);
    KlDeclareMethod(KnGroupClass, KnOSelBoundingBox, KnGroupBoundingBox);
    KlDeclareMethod(KnGroupClass, KnOSelCopyFields, KnGroupCopyFields);
}
