/* Modul %M% Release %R% Level %L% Branch %B% Sequenz %S% Created %E% %U% */

/* Copyright 1994 University of Stuttgart
 * This Graph Widget layouts (hierarchical) directed Graphs
 * This Work is based on the X11R5 tree widget
*/

/*
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE UNIVERSITY OF STUTTGART OR
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.

 * Except as contained in this notice, the name of the University of
 * Stuttgart or the names of the authors shall not be used in
 * advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from the
 * University of Stuttgart and the authors.
*/

/*
 * $XConsortium: Tree.c,v 1.42 91/02/20 20:06:07 converse Exp $
 *
 * Copyright 1990 Massachusetts Institute of Technology
 * Copyright 1989 Prentice Hall
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation.
 * 
 * M.I.T., Prentice Hall and the authors disclaim all warranties with regard
 * to this software, including all implied warranties of merchantability and
 * fitness.  In no event shall M.I.T., Prentice Hall or the authors be liable
 * for any special, indirect or cosequential damages or any damages whatsoever
 * resulting from loss of use, data or profits, whether in an action of
 * contract, negligence or other tortious action, arising out of or in
 * connection with the use or performance of this software.
 * 
 * Authors:  Jim Fulton, MIT X Consortium,
 *           based on a version by Douglas Young, Prentice Hall
 * 
 * This widget is based on the Tree widget described on pages 397-419 of
 * Douglas Young's book "The X Window System, Programming and Applications 
 * with Xt OSF/Motif Edition."  The layout code has been rewritten to use
 * additional blank space to make the structure of the graph easier to see
 * as well as to support vertical trees.
 */

/* Author: Roland Zink
           Universitaet Stuttgart
           IPVR
           Breitwiesenstrasse 20-22
           D 70565 Stuttgart (Germany)
           EMail Roland.Zink@informatik.uni-stuttgart.de
*/

#if defined(SCCSID)
static char sccs_id[] = "%A%";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>
/*
#include <X11/Xaw/XawInit.h>
*/
#include "GraphP.h"
#include "sc-queue.h"
#include "Graph_layout.h"
#include "Graph_draw.h"

typedef struct { Widget x; Widget y; } WidgetPair;


_XFUNCPROTOBEGIN

					/* widget class method */
static void ClassInitialize(
#if NeedFunctionPrototypes
                            void
#endif
);

static void Initialize(
#if NeedFunctionPrototypes
                       Widget,Widget
#endif
);

static void ConstraintInitialize(
#if NeedFunctionPrototypes
                                 Widget,Widget
#endif
);

static void ConstraintDestroy(
#if NeedFunctionPrototypes
                              Widget
#endif
);

static Boolean ConstraintSetValues(
#if NeedFunctionPrototypes
                                   Widget,Widget,Widget,ArgList,Cardinal *
#endif
);

static void Destroy(
#if NeedFunctionPrototypes
                    Widget
#endif
);

static Boolean SetValues(
#if NeedFunctionPrototypes
                         Widget,Widget,Widget
#endif
);

static XtGeometryResult GeometryManager(
#if NeedFunctionPrototypes
       Widget,XtWidgetGeometry *,XtWidgetGeometry *
#endif
);

static void ChangeManaged(
#if NeedFunctionPrototypes
                          Widget
#endif
);

static void Redisplay(
#if NeedFunctionPrototypes
                      GraphWidget
#endif
);

static XtGeometryResult	QueryGeometry(
#if NeedFunctionPrototypes
         Widget,XtWidgetGeometry *,XtWidgetGeometry *
#endif
);
					/* utility routines */
static void layout_graph(
#if NeedFunctionPrototypes
                         GraphWidget,Boolean
#endif
);

static void SelectWidgetAction(
#if NeedFunctionPrototypes
      Widget w,XEvent *event,String *params,Cardinal *num_params
#endif
);

static void SwitchManagedAction(
#if NeedFunctionPrototypes
        Widget w,XEvent *event,String *params,Cardinal *num_params
#endif
);

/*
 * resources of the graph itself
 */
static XtResource resources[] = {
    { XtNautoReconfigure, XtCAutoReconfigure, XtRBoolean, sizeof (Boolean),
	XtOffsetOf(GraphRec, graph.auto_reconfigure), XtRImmediate,
	(XtPointer) TRUE },
    { XtNhSpace, XtCHSpace, XtRDimension, sizeof (Dimension),
	XtOffsetOf(GraphRec, graph.hpad), XtRImmediate, (XtPointer) 0 },
    { XtNvSpace, XtCVSpace, XtRDimension, sizeof (Dimension),
	XtOffsetOf(GraphRec, graph.vpad), XtRImmediate, (XtPointer) 0 },
    { XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	XtOffsetOf(GraphRec, graph.foreground), XtRString,
	XtDefaultForeground},
    { XtNlineWidth, XtCLineWidth, XtRDimension, sizeof (Dimension),
	XtOffsetOf(GraphRec, graph.line_width), XtRImmediate, (XtPointer) 0 },
    { XtNgravity, XtCGravity, XtRGravity, sizeof (XtGravity),
	XtOffsetOf(GraphRec, graph.gravity), XtRImmediate,
	(XtPointer) WestGravity }
};


/*
 * resources that are attached to all children of the graph
 */
static XtResource graphConstraintResources[] = {
    { XtNgraphGC, XtCGraphGC, XtRGC, sizeof(GC),
	XtOffsetOf(GraphConstraintsRec, graph.gc), XtRImmediate, NULL },
};

static XtActionsRec GraphActionsTable[] = {
  {"select-widget",         SelectWidgetAction},
  {"switch-managed",        SwitchManagedAction}
};

GraphClassRec graphClassRec = {
  {
					/* core_class fields  */
    (WidgetClass) &constraintClassRec,	/* superclass         */
    "Graph",             		/* class_name         */
    sizeof(GraphRec),			/* widget_size        */
    (XtProc)ClassInitialize,		/* class_init         */
    NULL,				/* class_part_init    */
    FALSE,				/* class_inited       */	
    (XtInitProc)Initialize,		/* initialize         */
    NULL,				/* initialize_hook    */	
    XtInheritRealize,			/* realize            */
    GraphActionsTable,                  /* actions            */
    XtNumber(GraphActionsTable),	/* num_actions        */	
    resources,				/* resources          */
    XtNumber(resources),		/* num_resources      */
    NULLQUARK,				/* xrm_class          */
    TRUE,				/* compress_motion    */	
    XtExposeCompressMaximal |
    XtExposeGraphicsExpose |
    XtExposeGraphicsExposeMerged,	/* compress_exposure  */	
    TRUE,				/* compress_enterleave*/	
    TRUE,				/* visible_interest   */
    Destroy,				/* destroy            */
    NULL,				/* resize             */
    (XtExposeProc)Redisplay,	      	/* expose             */
    (XtSetValuesFunc)SetValues,		/* set_values         */
    NULL,				/* set_values_hook    */	
    XtInheritSetValuesAlmost,		/* set_values_almost  */
    NULL,				/* get_values_hook    */	
    NULL,				/* accept_focus       */
    XtVersion,				/* version            */	
    NULL,				/* callback_private   */
    NULL,				/* tm_table           */
    QueryGeometry,			/* query_geometry     */	
    NULL,				/* display_accelerator*/
    NULL,				/* extension          */
  },
  {
					/* composite_class fields */
    GeometryManager,			/* geometry_manager    */
    ChangeManaged,			/* change_managed      */
    XtInheritInsertChild,		/* insert_child        */	
    XtInheritDeleteChild,		/* delete_child        */	
    NULL,				/* extension           */
  },
  { 
					/* constraint_class fields */
   graphConstraintResources,		/* subresources        */
   XtNumber(graphConstraintResources),	/* subresource_count   */
   sizeof(GraphConstraintsRec),		/* constraint_size     */
   (XtInitProc)ConstraintInitialize,    /* initialize          */
   ConstraintDestroy,			/* destroy             */
   ConstraintSetValues,			/* set_values          */
   NULL,				/* extension           */
   },
  {
					/* Graph class fields */
    0                                   /* ignore */
  }
};

WidgetClass graphWidgetClass = (WidgetClass) &graphClassRec;


/*****************************************************************************
 *                                                                           *
 *			     graph utility routines                           *
 *                                                                           *
 *****************************************************************************/

/*------------------------------------------------------------------------
            Check if widget w is in the graphWidgetClass
--------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static Boolean GraphIsGraphWidget(Widget w)
#else
static Boolean GraphIsGraphWidget(w)
Widget w;
#endif
{ return XtIsSubclass(w,graphWidgetClass); }

/*------------------------------------------------------------------------
            Check if a point is in a widget
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Boolean PointInWidget(XPoint p,RectObj x)
#else
static Boolean PointInWidget(p,x)
XPoint p;
RectObj x;
#endif
{ RectObj ro;

  if( XtIsSubclass((Widget)x,rectObjClass) )
    { ro = (RectObj)x;
      if( (p.x > ro->rectangle.x) &&
          (p.x < (short)(ro->rectangle.x + ro->rectangle.width) ) &&
          (p.y > ro->rectangle.y) &&
          (p.y < (short)(ro->rectangle.y + ro->rectangle.height) ) )
        { return True; }
    }
  return False;
}

/*------------------------------------------------------------------------
            In which subwidget is a point
            if no subwidget is found, then the widget itself is returned 
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Widget PointToSubwidget(CompositeWidget w,XPoint p)
#else
static Widget PointToSubwidget(w,p)
CompositeWidget w;
XPoint p;
#endif
{ int i; Widget subwidget;

  subwidget = (Widget)w;
  if( XtIsSubclass((Widget)w,compositeWidgetClass) )
    { for( i=0; i<w->composite.num_children; i++ )
	 { if( PointInWidget(p,(RectObj)w->composite.children[i]) )
             { subwidget = w->composite.children[i]; break; }
         }
    }
  return subwidget;
}

/*------------------------------------------------------------------------
           Translate a point to the coordinate in a subwidget
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static XPoint PointToPointInSubwidget(RectObj x,XPoint p)
#else
static XPoint PointToPointInSubwidget(x,p)
RectObj x;
XPoint p;
#endif
{ p.x -= x->rectangle.x;
  p.y -= x->rectangle.y;

  return p;
}

/*------------------------------------------------------------------------
           Set the active widget to the subwidget which contains point
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void SelectWidget(GraphWidget gw,XPoint p)
#else
static void SelectWidget(gw,p)
GraphWidget gw;
XPoint p;
#endif
{ Widget subwidget;

  subwidget = PointToSubwidget((CompositeWidget)gw,p);
  gw->graph.active_Widget = subwidget;
  if( ((Widget)gw != subwidget) && XtIsSubclass(subwidget,graphWidgetClass) )
    { p = PointToPointInSubwidget((RectObj)subwidget,p);
      SelectWidget((GraphWidget)subwidget,p);
    }
}

/*------------------------------------------------------------------------
  Find the toplevel child widget in which window the button event occurs
  store the result in the graph_widget active_Widget variable
--------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void SelectWidgetAction(Widget w, XEvent *event,
                               String *params, Cardinal *num_params)
#else
static void SelectWidgetAction(w,event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{ Widget x; GraphWidget gw; XButtonEvent *e; Window xw; XPoint p;

  if( (event->type != ButtonPress) && (event->type != ButtonRelease) )
    { return; }

  e = (XButtonEvent *)event;
  if( e->subwindow )
    { xw = e->subwindow; }
  else
    { xw = e->window; }
  x=XtWindowToWidget(e->display,xw);
  
  if( GraphIsGraphWidget(w) )
    { gw = (GraphWidget)w;
      gw->graph.active_Widget = x;
      p.x = e->x;
      p.y = e->y;
      SelectWidget(gw,p);
    }
}









/*------------------------------------------------------------------------
            Create a VirtualEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static VirtualEdge_t *CreateVirtualEdge(Edge_t edge)
#else
static VirtualEdge_t *CreateVirtualEdge(edge)
Edge_t edge;
#endif
{ VirtualEdge_t *e;

  e = XtNew(VirtualEdge_t);
  e->edge = edge;
  e->layouts.n_edges = 0;
  e->layouts.max_edges = 0;
  e->layouts.edges = NULL;
  e->reals.n_edges = 0;
  e->reals.max_edges = 0;
  e->reals.edges = NULL;

  return e;
}

/*------------------------------------------------------------------------
            appends a VirtualEdge to a VirtualEdgeList
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void AppendVirtualEdgeInEdgeList(VirtualEdgeList_t *list,
                                        VirtualEdge_t *e)
#else
static void AppendVirtualEdgeInEdgeList(list,e)
VirtualEdgeList_t *list;
VirtualEdge_t *e;
#endif
{ if( list->n_edges == list->max_edges )
    { list->max_edges += (list->max_edges/2) + 2;
      list->edges = (VirtualEdge_t **)XtRealloc((char *)list->edges,
                    (unsigned int)((list->max_edges)*sizeof(VirtualEdge_t *)));
    }
  list->edges[list->n_edges] = e;
  list->n_edges++;
}

/*------------------------------------------------------------------------
            find the position of a VirtualEdge in a VirtualEdgelist
            0..n_edges-1 -> position   n_edges -> not_found
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static int FindVirtualEdgeInEdgeList(VirtualEdgeList_t *list,VirtualEdge_t *e)
#else
static int FindVirtualEdgeInEdgeList(list,e)
VirtualEdgeList_t *list;
VirtualEdge_t *e;
#endif
{ int pos;

  for( pos=0; pos<list->n_edges; pos++ )
     { if( list->edges[pos] == e )
         { return pos; }
     }
  return list->n_edges;
}

/*------------------------------------------------------------------------
            insert a VirtualEdge in an edgelist
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void InsertVirtualEdgeInEdgeList(VirtualEdgeList_t *list,
                                        VirtualEdge_t *e)
#else
static void InsertVirtualEdgeInEdgeList(list,e)
VirtualEdgeList_t *list;
VirtualEdge_t *e;
#endif
{ int pos;

  pos = FindVirtualEdgeInEdgeList(list,e);
  if( pos==list->n_edges )
    { AppendVirtualEdgeInEdgeList(list,e); }
}


/*------------------------------------------------------------------------
            Insert a VirtualEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static VirtualEdge_t *InsertVirtualEdge(Edge_t edge,LayoutData_t *graph)
#else
static VirtualEdge_t *InsertVirtualEdge(edge,graph)
Edge_t *edge;
LayoutData_t *graph;
#endif
{ VirtualEdge_t *e;

  e = CreateVirtualEdge(edge);
  InsertVirtualEdgeInEdgeList(&graph->edges,e);

  return e;
}

/*------------------------------------------------------------------------
            Create a LayoutEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
LayoutEdge_t *CreateLayoutEdge(Edge_t edge,VirtualEdge_t *virtual)
#else
LayoutEdge_t *CreateLayoutEdge(edge,virtual)
Edge_t edge;
VirtualEdge_t *virtual;
#endif
{ LayoutEdge_t *e;

  e = XtNew(LayoutEdge_t);
  e->edge = edge;
  e->virtual = virtual;

  return e;
}

/*------------------------------------------------------------------------
            Destroy a LayoutEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
void DestroyLayoutEdge(LayoutEdge_t *e)
#else
void DestroyLayoutEdge(e)
LayoutEdge_t *e;
#endif
{ XtFree((char *)e);
}


/*------------------------------------------------------------------------
            appends a LayoutEdge to a LayoutEdgeList
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void AppendLayoutEdgeInEdgeList(LayoutEdgeList_t *list,LayoutEdge_t *e)
#else
static void AppendLayoutEdgeInEdgeList(list,e)
LayoutEdgeList_t *list;
LayoutEdge_t *e;
#endif
{ if( list->n_edges == list->max_edges )
    { list->max_edges += (list->max_edges/2) + 2;
      list->edges = (LayoutEdge_t **)XtRealloc((char *)list->edges,
                     (unsigned int)((list->max_edges)*sizeof(LayoutEdge_t *)));
    }
  list->edges[list->n_edges] = e;
  list->n_edges++;
}

/*------------------------------------------------------------------------
            find the position of a LayoutEdge in an LayoutEdgelist
            0..n_edges-1 -> position   n_edges -> not_found
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
int FindLayoutEdgeInEdgeList(LayoutEdgeList_t *list,LayoutEdge_t *e)
#else
int FindLayoutEdgeInEdgeList(list,e)
LayoutEdgeList_t *list;
LayoutEdge_t *e;
#endif
{ int pos;

  for( pos=0; pos<list->n_edges; pos++ )
     { if( list->edges[pos] == e )
         { return pos; }
     }
  return list->n_edges;
}

/*------------------------------------------------------------------------
            insert a LayoutEdge in an edgelist
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
void InsertLayoutEdgeInEdgeList(LayoutEdgeList_t *list,LayoutEdge_t *e)
#else
void InsertLayoutEdgeInEdgeList(list,e)
LayoutEdgeList_t *list;
LayoutEdge_t *e;
#endif
{ int pos;

  pos = FindLayoutEdgeInEdgeList(list,e);
  if( pos==list->n_edges )
    { AppendLayoutEdgeInEdgeList(list,e); }
}

/*------------------------------------------------------------------------
            delete an edge from an edgelist
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
void DeleteLayoutEdgeInEdgeList(LayoutEdgeList_t *list,LayoutEdge_t *e)
#else
void DeleteLayoutEdgeInEdgeList(list,e)
LayoutEdgeList_t *list;
LayoutEdge_t *e;
#endif
{ int pos;

  pos = FindLayoutEdgeInEdgeList(list,e);
  if( pos < list->n_edges )
    { list->n_edges--;
      for( ; pos<list->n_edges; pos++ )
	 { list->edges[pos] = list->edges[pos+1]; }
    }
  else
    {
#if defined(__STDC__)
      fprintf(stderr,__FILE__":DeleteLayoutEdgeInEdgeList():edge not found\n");
#else
      fprintf(stderr,":DeleteLayoutEdgeInEdgeList():edge not found\n");
#endif
    }
}

/*------------------------------------------------------------------------
        replace a layoutedge in an edgelist through another one
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
void ReplaceLayoutEdgeInEdgeList(LayoutEdgeList_t *list,
                                 LayoutEdge_t *x,LayoutEdge_t *y)
#else
void ReplaceLayoutEdgeInEdgeList(list,x,y)
LayoutEdgeList_t *list;
LayoutEdge_t *x;
LayoutEdge_t *y;
#endif
{ int pos;

  for(pos=0; pos<list->n_edges; pos++ )
    { if( list->edges[pos] == x )
        { list->edges[pos] = y; return; }
    }
#if defined(__STDC__)
  fprintf(stderr,__FILE__":ReplaceLayoutEdgeInEdgeList():edge not found\n");
#else
  fprintf(stderr,":ReplaceLayoutEdgeInEdgeList():edge not found\n");
#endif
}

/*------------------------------------------------------------------------
    Insert a LayoutEdge between two vertices corresponding to virtual edge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static LayoutEdge_t *InsertLayoutEdge(Edge_t edge,VirtualEdge_t *virtual)
#else
static LayoutEdge_t *InsertLayoutEdge(edge,virtual)
Edge_t edge;
VirtualEdge_t *virtual;
#endif
{ LayoutEdge_t *le;

  le = CreateLayoutEdge(edge,virtual);
  InsertLayoutEdgeInEdgeList(&virtual->layouts,le);
  InsertLayoutEdgeInEdgeList(&edge.from->edges,le);
  InsertLayoutEdgeInEdgeList(&edge.to->edges,le);
  return le;
}

/*------------------------------------------------------------------------
    Delete a LayoutEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteLayoutEdge(LayoutEdge_t *e)
#else
static void DeleteLayoutEdge(e)
LayoutEdge_t *e;
#endif
{ DeleteLayoutEdgeInEdgeList(&e->edge.from->edges,e);
  if( e->edge.from != e->edge.to )
    { DeleteLayoutEdgeInEdgeList(&e->edge.to->edges,e); }
  DeleteLayoutEdgeInEdgeList(&e->virtual->layouts,e);
  DestroyLayoutEdge(e);
}


/*------------------------------------------------------------------------
            find a RealEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static RealEdge_t *FindRealEdge(Edge_t e,VirtualEdge_t *ve)
#else
static RealEdge_t *FindRealEdge(e,ve)
Edge_t e;
VirtualEdge_t *ve;
#endif
{ int i;
  RealEdge_t *re;

  for( i=0; i<ve->reals.n_edges; i++ )
     { re=ve->reals.edges[i];
       if( (re->edge.from == e.from) && (re->edge.to   == e.to) )
         { return re; }
     }
  return NULL;
}

/*------------------------------------------------------------------------
            appends a RealEdge to a RealEdgeList
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void AppendRealEdgeInEdgeList(RealEdgeList_t *list,RealEdge_t *e)
#else
static void AppendRealEdgeInEdgeList(list,e)
RealEdgeList_t *list;
RealEdge_t *e;
#endif
{ if( list->n_edges == list->max_edges )
    { list->max_edges += (list->max_edges/2) + 2;
      list->edges = (RealEdge_t **)XtRealloc((char *)list->edges,
                     (unsigned int)((list->max_edges)*sizeof(RealEdge_t *)));
    }
  list->edges[list->n_edges] = e;
  list->n_edges++;
}

/*------------------------------------------------------------------------
            find the position of a RealEdge in a RealEdgelist
            0..n_edges-1 -> position   n_edges -> not_found
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static int FindRealEdgeInEdgeList(RealEdgeList_t *list,RealEdge_t *e)
#else
static int FindRealEdgeInEdgeList(list,e)
RealEdgeList_t *list;
RealEdge_t *e;
#endif
{ int pos;

  for( pos=0; pos<list->n_edges; pos++ )
     { if( list->edges[pos] == e )
         { return pos; }
     }
  return list->n_edges;
}

/*------------------------------------------------------------------------
            insert a RealEdge in an edgelist
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void InsertRealEdgeInEdgeList(RealEdgeList_t *list,RealEdge_t *e)
#else
static void InsertRealEdgeInEdgeList(list,e)
RealEdgeList_t *list;
RealEdge_t *e;
#endif
{ int pos;

  pos = FindRealEdgeInEdgeList(list,e);
  if( pos==list->n_edges )
    { AppendRealEdgeInEdgeList(list,e); }
}

/*------------------------------------------------------------------------
            Create a RealEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static RealEdge_t *CreateRealEdge(Edge_t edge,EdgeDirection direction,
                                  VirtualEdge_t *virtual)
#else
static RealEdge_t *CreateRealEdge(edge,direction,virtual)
Edge_t edge;
EdgeDirection direction;
VirtualEdge_t *virtual;
#endif
{ RealEdge_t *e;

  e = XtNew(RealEdge_t);
  e->edge = edge;
  e->direction = direction;
  e->gc = NULL;
  e->virtual = virtual;

  return e;
}

/*------------------------------------------------------------------------
    Insert a RealEdge between two vertices corresponding to virtual edge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static RealEdge_t *InsertRealEdge(Edge_t edge,EdgeDirection direction,
                                  VirtualEdge_t *virtual)
#else
static RealEdge_t *InsertRealEdge(edge,direction,virtual)
Edge_t edge;
EdgeDirection direction;
VirtualEdge_t *virtual;
#endif
{ RealEdge_t *re;

  re = CreateRealEdge(edge,direction,virtual);
  InsertRealEdgeInEdgeList(&virtual->reals,re);
  return re;
}

/*------------------------------------------------------------------------
            delete a RealEdge from an edgelist
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteRealEdgeInEdgeList(RealEdgeList_t *list,RealEdge_t *e)
#else
static void DeleteRealEdgeInEdgeList(list,e)
RealEdgeList_t *list;
RealEdge_t *e;
#endif
{ int pos;

  pos = FindRealEdgeInEdgeList(list,e);
  if( pos < list->n_edges )
    { list->n_edges--;
      for( ; pos<list->n_edges; pos++ )
	 { list->edges[pos] = list->edges[pos+1]; }
    }
  else
    {
#if defined(__STDC__)
      fprintf(stderr,__FILE__":DeleteRealEdgeInEdgeList():edge not found\n");
#else
      fprintf(stderr,":DeleteRealEdgeInEdgeList():edge not found\n");
#endif
    }
}

/*------------------------------------------------------------------------
            Destroy a RealEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DestroyRealEdge(RealEdge_t *e)
#else
static void DestroyRealEdge(e)
RealEdge_t *e;
#endif
{ XtFree((char *)e);
}

/*------------------------------------------------------------------------
    Delete a RealEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteRealEdge(RealEdge_t *e)
#else
static void DeleteRealEdge(e)
RealEdge_t *e;
#endif
{ DeleteRealEdgeInEdgeList(&e->virtual->reals,e);
  DestroyRealEdge(e);
}



/*------------------------------------------------------------------------
    Destroy a VirtualEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DestroyVirtualEdge(VirtualEdge_t *e)
#else
static void DestroyVirtualEdge(e)
VirtualEdge_t *e;
#endif
{
  if( (e->layouts.n_edges != 0) || (e->reals.n_edges != 0) )
    { 
#if defined(__STDC__)
      fprintf(stderr,__FILE__":DestroyVirtualEdge(): Edge not empty\n");
#else
      fprintf(stderr,":DestroyVirtualEdge(): Edge not empty\n");
#endif
      return;
    }
  XtFree((XtPointer)e->layouts.edges);
  XtFree((XtPointer)e->reals.edges);
}

/*------------------------------------------------------------------------
    Delete a VirtualEdge
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteVirtualEdge(VirtualEdge_t *e)
#else
static void DeleteVirtualEdge(e)
VirtualEdge_t *e;
#endif
{ int i;

  for( i=e->layouts.n_edges; i>0; i-- )
    { DeleteLayoutEdge(e->layouts.edges[i-1]); }
  for( i=e->reals.n_edges; i>0; i-- )
    { DeleteRealEdge(e->reals.edges[i-1]); }
  DestroyVirtualEdge(e);
}

/*------------------------------------------------------------------------
            Find VirtualEdge between two vertices
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static VirtualEdge_t *FindVirtualEdge(Vertice_t *from,Vertice_t *to)
#else
static VirtualEdge_t *FindVirtualEdge(from,to)
Vertice_t *from;
Vertice_t *to;
#endif
{ int i;
  LayoutEdge_t *le;
  VirtualEdge_t *ve;

  ve = NULL;
  for( i=0; i<from->edges.n_edges; i++ )
    { le = from->edges.edges[i];
      if( (le->virtual->edge.from == from) && (le->virtual->edge.to == to) )
        { ve = le->virtual; break; }
      else
        { if( (le->virtual->edge.from == to) && (le->virtual->edge.to == from))
            { ve = le->virtual; break; }
        }
    }

  return ve;
}

/*------------------------------------------------------------------------
    Delete all RealEdges of a VirtualEdge from or to vertice
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteRealEdgeInVirtualEdge(VirtualEdge_t *ve,Vertice_t *vertice)
#else
static void DeleteRealEdgeInVirtualEdge(ve,vertice)
VirtualEdge_t *ve;
Vertice_t *vertice;
#endif
{ int i;
  RealEdge_t *re;

  for( i=ve->reals.n_edges; i>0; i-- )
    { re = ve->reals.edges[i-1];
      if( (re->edge.from == vertice) || (re->edge.to == vertice) )
        { DeleteRealEdge(re); }
    }
}

/*------------------------------------------------------------------------
    Delete all Edges to parent graphs base on a real edge to vertice
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteRealEdgesUp(Widget w,Vertice_t *vertice)
#else
static void DeleteRealEdgesUp(w,vertice)
Widget w;
Vertice_t *vertice;
#endif
{ Widget parent;
  Widget grandparent;
  Vertice_t *parentvertice;
  int i;
  LayoutEdge_t *le;
  VirtualEdge_t *ve;
  GraphConstraints pgc;

  parent = XtParent(w);
  if( parent != NULL )
    { grandparent = XtParent(parent);
      if( (grandparent!=NULL) && !XtIsSubclass(grandparent,graphWidgetClass) )
        { DeleteRealEdgesUp(parent,vertice); }
      else
        { pgc = GRAPH_CONSTRAINT(parent);
          if( pgc != NULL )
            { parentvertice = &(pgc->graph.vertice);
              for( i=parentvertice->edges.n_edges; i>0; i-- )
                 { le = parentvertice->edges.edges[i-1];
                   ve = le->virtual;
                   DeleteRealEdgeInVirtualEdge(ve,vertice);
                   if( ve->reals.n_edges == 0 )
                     { DeleteVirtualEdge(ve); }
                 }
	    }
          DeleteRealEdgesUp(parent,vertice);
        }
    }
}


/*------------------------------------------------------------------------
    Delete all Edges associatet with a vertice
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void DeleteAllEdgesOfNode(Vertice_t *vertice)
#else
static void DeleteAllEdgesOfNode(vertice)
Vertice_t *vertice;
#endif
{ LayoutEdge_t *le;
  VirtualEdge_t *ve;
  int i;

  for (i = vertice->edges.n_edges; i>0; i--)
      { le = vertice->edges.edges[i-1];
        ve = le->virtual;
        DeleteVirtualEdge(ve);
      }
  DeleteRealEdgesUp(vertice->w,vertice);
}



/*------------------------------------------------------------------------
            Compare two pointers to widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Boolean widget_equalp(Widget *x,Widget *y)
#else
static Boolean widget_equalp(x,y)
Widget *x;
Widget *y;
#endif
{ return *x == *y; }

/*------------------------------------------------------------------------
            Find the common ancestor of two widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static WidgetPair FindCommonAncestor1(sc_queue_t *x,sc_queue_t *y)
#else
static WidgetPair FindCommonAncestor1(x,y)
sc_queue_t *x;
sc_queue_t *y;
#endif
{ Widget px; Widget py; sc_queue_t *ca; sc_queue_t newx; sc_queue_t newy;
  WidgetPair parents;

  px = XtParent(*(Widget *)sc_queue_peek(x));
  ca = sc_queue_search(y,(sc_queue_fkt)widget_equalp,&px);
  if( !ca )
    { x = sc_queue_push(x,&newx,&px);
      py = XtParent(*(Widget *)sc_queue_peek(y));
      ca = sc_queue_search(x,(sc_queue_fkt)widget_equalp,&py);
      if( !ca )
        { y = sc_queue_push(y,&newy,&py);
          parents = FindCommonAncestor1(x,y);
        }
      else
        { parents.y = *(Widget *)sc_queue_peek(y);
          ca = sc_queue_pop(ca);
          if( ca == NULL )
            { parents.x = parents.y; }
          else
            { parents.x = *(Widget *)sc_queue_peek(ca); }
        }
    }
  else
    { parents.x = *(Widget *)sc_queue_peek(x);
      ca = sc_queue_pop(ca);
      if( ca == NULL )
        { parents.y = parents.x; }
      else
        { parents.y = *(Widget *)sc_queue_peek(ca); }
    }
  return parents;
}

#if NeedFunctionPrototypes
static WidgetPair FindCommonAncestor(Widget x,Widget y)
#else
static WidgetPair FindCommonAncestor(x,y)
Widget x;
Widget y;
#endif
{ sc_queue_t xq; sc_queue_t yq; WidgetPair parents;

  sc_queue_push(NULL,&xq,&x);
  sc_queue_push(NULL,&yq,&y);
  parents = FindCommonAncestor1(&xq,&yq);

  return parents;
}

/*------------------------------------------------------------------------
        find an ancestor which is son of a graph widget
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static Widget findGraphSonUp(Widget w)
#else
static Widget findGraphSonUp(w)
Widget w;
#endif
{ Widget parent;

  parent = XtParent(w);
  while( !XtIsSubclass(parent,graphWidgetClass) )
    { w=parent;
      parent=XtParent(w);
    }

  return w;
}


/*------------------------------------------------------------------------
        insert an edge between to widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void insert_edge(Widget from,Widget to)
#else
static void insert_edge(from,to)
Widget from;
Widget to;
#endif
{ WidgetPair parents;
  VirtualEdge_t *ve;
  Edge_t edge;
  LayoutEdge_t *le;
  RealEdge_t *re;
  GraphConstraints fgc;
  GraphConstraints tgc;
  EdgeDirection direction;
  Boolean geometryChanged;

  geometryChanged = False;
  from = findGraphSonUp(from);
  to = findGraphSonUp(to);
  parents = FindCommonAncestor(from,to);
  fgc = GRAPH_CONSTRAINT(parents.x);
  tgc = GRAPH_CONSTRAINT(parents.y);
  ve = FindVirtualEdge(&fgc->graph.vertice,&tgc->graph.vertice);
  if( !ve )
    { edge.from = &fgc->graph.vertice;
      edge.to   = &tgc->graph.vertice;
      ve = InsertVirtualEdge(edge,&((GraphWidget)XtParent(parents.x))->graph.layout);
      le = InsertLayoutEdge(edge,ve);
      geometryChanged = True;
    }
  if( &fgc->graph.vertice == ve->edge.from )
    { direction = forward_edge; }
  else
    { direction = reverse_edge; }
  fgc = GRAPH_CONSTRAINT(from);
  tgc = GRAPH_CONSTRAINT(to);
  edge.from = &fgc->graph.vertice;
  edge.to   = &tgc->graph.vertice;
  re = FindRealEdge(edge,ve);
  if( !re )
    { InsertRealEdge(edge,direction,ve); }

  if( geometryChanged &&
      (((GraphWidget)XtParent(parents.x))->graph.auto_reconfigure) )
    { layout_graph((GraphWidget)XtParent(parents.x),False); }
  else
    { DrawVirtualEdge(&((GraphWidget)XtParent(parents.x))->graph.layout,ve); }
}

/*------------------------------------------------------------------------
        delete an edge between to widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void delete_edge(Widget from,Widget to)
#else
static void delete_edge(from,to)
Widget from;
Widget to;
#endif
{ WidgetPair parents;
  Edge_t edge;
  VirtualEdge_t *ve;
  RealEdge_t *re;
  GraphConstraints fgc;
  GraphConstraints tgc;

  parents = FindCommonAncestor(from,to);
  fgc = GRAPH_CONSTRAINT(parents.x);
  tgc = GRAPH_CONSTRAINT(parents.y);
  ve = FindVirtualEdge(&fgc->graph.vertice,&tgc->graph.vertice);
  if( ve )
    { fgc = GRAPH_CONSTRAINT(from);
      tgc = GRAPH_CONSTRAINT(to);
      edge.from = &fgc->graph.vertice;
      edge.to   = &tgc->graph.vertice;
      re = FindRealEdge(edge,ve);
      if( re )
        { DeleteRealEdge(re); }
      if( ve->reals.n_edges == 0 )
        { DeleteVirtualEdge(ve);
          if( ((GraphWidget)XtParent(parents.x))->graph.auto_reconfigure)
            { layout_graph((GraphWidget)XtParent(parents.x),False); }
        }
    }
}



/*------------------------------------------------------------------------
        set the Graphics Context of the edge between two widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void setEdgeGC(Widget from,Widget to,GC gc)
#else
static void setEdgeGC(from,to,gc)
Widget from;
Widget to;
GC gc;
#endif
{ WidgetPair parents;
  VirtualEdge_t *ve;
  Edge_t edge;
  RealEdge_t *re;
  GraphConstraints fgc;
  GraphConstraints tgc;

  parents = FindCommonAncestor(from,to);
  fgc = GRAPH_CONSTRAINT(parents.x);
  tgc = GRAPH_CONSTRAINT(parents.y);
  ve = FindVirtualEdge(&fgc->graph.vertice,&tgc->graph.vertice);
  if( ve )
    { fgc = GRAPH_CONSTRAINT(from);
      tgc = GRAPH_CONSTRAINT(to);
      edge.from = &fgc->graph.vertice;
      edge.to   = &tgc->graph.vertice;
      re = FindRealEdge(edge,ve);
      if( re )
        { re->gc = gc; }
    }
/*
  if( ((GraphWidget)XtParent(parents.x))->graph.auto_reconfigure)
    { layout_graph((GraphWidget)XtParent(parents.x),False); }
*/
  DrawVirtualEdge(&((GraphWidget)XtParent(parents.x))->graph.layout,ve);
}



/*------------------------------------------------------------------------
        set the Graphics Context of the edge between two widgets
--------------------------------------------------------------------------*/
#if NeedFunctionPrototypes
static void insertEdgeGC(Widget from,Widget to,GC gc)
#else
static void insertEdgeGC(from,to,gc)
Widget from;
Widget to;
GC gc;
#endif
{ WidgetPair parents;
  VirtualEdge_t *ve;
  Edge_t edge;
  RealEdge_t *re;
  GraphConstraints fgc;
  GraphConstraints tgc;

  parents = FindCommonAncestor(from,to);
  fgc = GRAPH_CONSTRAINT(parents.x);
  tgc = GRAPH_CONSTRAINT(parents.y);
  ve = FindVirtualEdge(&fgc->graph.vertice,&tgc->graph.vertice);
  if( !ve )
    { insert_edge(from,to);
      ve = FindVirtualEdge(&fgc->graph.vertice,&tgc->graph.vertice);
    }
  fgc = GRAPH_CONSTRAINT(from);
  tgc = GRAPH_CONSTRAINT(to);
  edge.from = &fgc->graph.vertice;
  edge.to   = &tgc->graph.vertice;
  re = FindRealEdge(edge,ve);
  if( !re )
    { insert_edge(from,to);
      re = FindRealEdge(edge,ve);
    }
  re->gc = gc;

  DrawVirtualEdge(&((GraphWidget)XtParent(parents.x))->graph.layout,ve);
/*
  if( ((GraphWidget)XtParent(parents.x))->graph.auto_reconfigure)
    { layout_graph((GraphWidget)XtParent(parents.x),False); }
*/
}











#if NeedFunctionPrototypes
static GC get_graph_gc (GraphWidget w)
#else
static GC get_graph_gc (w)
    GraphWidget w;
#endif
{
    XtGCMask valuemask = GCBackground | GCForeground;
    XGCValues values;

    values.background = w->core.background_pixel;
    values.foreground = w->graph.foreground;
    if (w->graph.line_width != 0) {
	valuemask |= GCLineWidth;
	values.line_width = w->graph.line_width;
    }

    return XtGetGC ((Widget) w, valuemask, &values);
}

#if NeedFunctionPrototypes
static void check_gravity (GraphWidget gw, XtGravity grav)
#else
static void check_gravity (gw, grav)
    GraphWidget gw;
    XtGravity grav;
#endif
{
    switch (gw->graph.gravity) {
      case WestGravity: case NorthGravity: case EastGravity: case SouthGravity:
	break;
      default:
	gw->graph.gravity = grav;
	break;
    }
}


/*****************************************************************************
 *                                                                           *
 *			     graph layout algorithm                           *
 *                                                                           *
 * Each node in the graph is "shrink-wrapped" with a minimal bounding         *
 * rectangle, laid next to its siblings (with a small about of padding in    *
 * between) and then wrapped with their parent.  Parents are centered about  *
 * their children (or vice versa if the parent is larger than the children). *
 *                                                                           *
 *****************************************************************************/


#if NeedFunctionPrototypes
static void set_graph_size (GraphWidget gw, Boolean insetvalues,
                            Dimension width, Dimension height)
#else
static void set_graph_size (gw, insetvalues, width, height)
    GraphWidget gw;
    Boolean insetvalues;
    Dimension width, height;
#endif
{
  if( (gw->core.width != width) || (gw->core.height != height))
  { if (insetvalues) {
	gw->core.width = width;
	gw->core.height = height;
    } else {
	Dimension replyWidth = 0, replyHeight = 0;
	XtGeometryResult result = XtMakeResizeRequest ((Widget) gw,
						       width, height,
						       &replyWidth,
						       &replyHeight);
	/*
	 * Accept any compromise.
	 */
	if (result == XtGeometryAlmost)
          { if( (replyWidth!=gw->core.width)||(replyHeight != gw->core.height))
	      { XtMakeResizeRequest ((Widget) gw, replyWidth, replyHeight,
		                      (Dimension *) NULL, (Dimension *) NULL);
              }
          }
    }
  }
    return;
}

/*****************************************************************************
 *                                                                           *
 * 			      graph class methods                             *
 *                                                                           *
 *****************************************************************************/

#if NeedFunctionPrototypes
static void ClassInitialize ( void )
#else
static void ClassInitialize ()
#endif
{
/*
    XawInitializeWidgetSet();
*/
    XtAddConverter (XtRString, XtRGravity, XmuCvtStringToGravity,
		    (XtConvertArgList) NULL, (Cardinal) 0);
}


#if NeedFunctionPrototypes
static void Initialize (Widget grequest, Widget gnew)
#else
static void Initialize (grequest, gnew)
Widget grequest;
Widget gnew;
#endif
{
    GraphWidget request = (GraphWidget) grequest, new = (GraphWidget) gnew;

    /*
     * Make sure the widget's width and height are 
     * greater than zero.
     */
    if (request->core.width <= 0) new->core.width = 5;
    if (request->core.height <= 0) new->core.height = 5;

    /*
     * Set the padding according to the orientation
     */
    if (request->graph.hpad == 0 && request->graph.vpad == 0) {
	if (IsHorizontal (request)) {
	    new->graph.hpad = GRAPH_HORIZONTAL_DEFAULT_SPACING;
	    new->graph.vpad = GRAPH_VERTICAL_DEFAULT_SPACING;
	} else {
	    new->graph.hpad = GRAPH_VERTICAL_DEFAULT_SPACING;
	    new->graph.vpad = GRAPH_HORIZONTAL_DEFAULT_SPACING;
	}
    }

    /*
     * Create a graphics context for the connecting lines.
     */
    new->graph.gc = get_graph_gc (new);
    new->graph.layout.w = new;
    new->graph.layout.maxwidth = 5;
    new->graph.layout.maxheight = 5;
    new->graph.layout.levels = NULL;
    new->graph.layout.n_levels = 0;
    new->graph.layout.max_levels = 0;
    new->graph.layout.dummies = NULL;
    new->graph.layout.n_dummies = 0;
    new->graph.layout.max_dummies = 0;
    new->graph.layout.edges.n_edges = 0;
    new->graph.layout.edges.max_edges = 0;
    new->graph.layout.edges.edges = NULL;
    new->graph.active_Widget = (Widget)new;
    new->graph.layout.explodep = True;
    /*
     * make sure that our gravity is one of the acceptable values
     */
    check_gravity (new, WestGravity);
} 

#if NeedFunctionPrototypes
static void ConstraintInitialize (Widget request, Widget new)
#else
static void ConstraintInitialize (request, new)
Widget request;
Widget new;
#endif
{ GraphConstraints tc = GRAPH_CONSTRAINT(new);

  /*
   * Initialize the widget to have no sub-nodes and no-parents.
   */
  tc->graph.vertice.w = new;
  tc->graph.vertice.edges.n_edges = 0;
  tc->graph.vertice.edges.max_edges = 0;
  tc->graph.vertice.edges.edges = (LayoutEdge_t **)NULL;
  tc->graph.vertice.in_current_search_path = FALSE;
  tc->graph.vertice.is_dag = FALSE;
  tc->graph.vertice.is_dummy = FALSE;
  tc->graph.vertice.level = -1;
  tc->graph.vertice.elem  = -1;
  tc->graph.vertice.x = new->core.x;
  tc->graph.vertice.y = new->core.y;
  tc->graph.vertice.width = new->core.width + 2*new->core.border_width;
  tc->graph.vertice.height = new->core.height + 2*new->core.border_width;
}


#if NeedFunctionPrototypes
static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew)
#else
static Boolean SetValues (gcurrent, grequest, gnew)
Widget gcurrent;
Widget grequest;
Widget gnew;
#endif
{
    GraphWidget current = (GraphWidget) gcurrent, new = (GraphWidget) gnew;
    Boolean redraw = FALSE;

    /*
     * If the foreground color has changed, redo the GC's
     * and indicate a redraw.
     */

    if (new->graph.foreground != current->graph.foreground ||
	new->core.background_pixel != current->core.background_pixel ||
	new->graph.line_width != current->graph.line_width) {
/*
	XtReleaseGC (gnew, new->graph.gc);
	new->graph.gc = get_graph_gc (new);
*/
	redraw = TRUE;     
    }

    /*
     * If the minimum spacing has changed, recalculate the
     * graph layout. layout_graph() does a redraw, so we don't
     * need SetValues to do another one.
     */
    if (new->graph.gravity != current->graph.gravity) {
	check_gravity (new, current->graph.gravity);
    }

    if (IsHorizontal(new) != IsHorizontal(current)) {
	if (new->graph.vpad == current->graph.vpad &&
	    new->graph.hpad == current->graph.hpad) {
	    new->graph.vpad = current->graph.hpad;
	    new->graph.hpad = current->graph.vpad;
	}
    }

    if (new->graph.vpad != current->graph.vpad ||
	new->graph.hpad != current->graph.hpad ||
	new->graph.gravity != current->graph.gravity) {
/*
	if (new->graph.auto_reconfigure)
          { layout_graph (new, TRUE); }
*/
	redraw = FALSE;
    }
    return redraw;
}


#if NeedFunctionPrototypes
static Boolean ConstraintSetValues(Widget current, Widget request,
                                   Widget new,ArgList args, Cardinal *num_args)
#else
static Boolean ConstraintSetValues(current, request, new, args, num_args)
Widget current;
Widget request;
Widget new;
ArgList args;
Cardinal *num_args;
#endif
{ return False; }


#if NeedFunctionPrototypes
static void ConstraintDestroy (Widget w) 
#else
static void ConstraintDestroy (w)
Widget w;
#endif
{ GraphConstraints tc = GRAPH_CONSTRAINT(w);

  DeleteAllEdgesOfNode(&tc->graph.vertice);

  if( tc->graph.vertice.edges.edges )
    { XtFree((XtPointer)tc->graph.vertice.edges.edges ); }
/*
    layout_graph ((GraphWidget) (w->core.parent), FALSE);
*/
}

/* ARGSUSED */
#if NeedFunctionPrototypes
static XtGeometryResult GeometryManager(Widget w,XtWidgetGeometry *request,
                                                 XtWidgetGeometry *reply)
#else
static XtGeometryResult GeometryManager(w,request,reply)
Widget w;
XtWidgetGeometry *request;
XtWidgetGeometry *reply;
#endif
{
    GraphWidget gw = (GraphWidget) w->core.parent;

    /*
     * No position changes allowed!.
     */
    if ((request->request_mode & CWX && request->x!=w->core.x)
	||(request->request_mode & CWY && request->y!=w->core.y))
       { return (XtGeometryNo); }

    /*
     * Allow all resize requests.
     */

    if (request->request_mode & CWWidth)
      { w->core.width = request->width;
        GRAPH_CONSTRAINT(w)->graph.vertice.width = request->width +
                                                   2*w->core.border_width;

      }
    if (request->request_mode & CWHeight)
      { w->core.height = request->height;
        GRAPH_CONSTRAINT(w)->graph.vertice.height = request->height +
                                                    2*w->core.border_width;
      }
    if (request->request_mode & CWBorderWidth)
      { w->core.border_width = request->border_width;
        GRAPH_CONSTRAINT(w)->graph.vertice.height = 
                      w->core.height + 2*w->core.border_width;
        GRAPH_CONSTRAINT(w)->graph.vertice.width = 
                      w->core.width + 2*w->core.border_width;
      }
    if (gw->graph.auto_reconfigure)
      { layout_graph (gw, FALSE); }
    return (XtGeometryYes);
}

#if NeedFunctionPrototypes
static void implode(GraphWidget gw)
#else
static void implode(gw)
GraphWidget gw;
#endif
{ if( gw && (gw->composite.num_children > 0) )
    { XtUnmanageChildren(&gw->composite.children[1],
                          gw->composite.num_children-1);
    }
  gw->graph.layout.explodep = FALSE;
}

/* Known Bug: Dummy Nodes should not be managed */
#if NeedFunctionPrototypes
static void explode(GraphWidget gw)
#else
static void explode(gw)
GraphWidget gw;
#endif
{ if( gw && (gw->composite.num_children > 0) )
   { XtManageChildren(&gw->composite.children[1],gw->composite.num_children-1);
   }
  gw->graph.layout.explodep = TRUE;
}

#if NeedFunctionPrototypes
static void SwitchManaged(GraphWidget gw)
#else
static void SwitchManaged(gw)
GraphWidget gw;
#endif
{ int i; Boolean doexplode; Vertice_t *child;

  doexplode = False;
  for(i=1; i<gw->composite.num_children; i++ )
    { if( !XtIsManaged(gw->composite.children[i]) )
        { child = &(GRAPH_CONSTRAINT(gw->composite.children[i])->graph.vertice);
          if( !IsDummyNode(child) )
            { doexplode = True;
              break;
            }
        }
    }
        
  if( doexplode )
    { explode(gw); }
  else
    { implode(gw); }
}

#if NeedFunctionPrototypes
static void SwitchManagedAction(Widget w, XEvent *event,
                                String *params, Cardinal *num_params)
#else
static void SwitchManagedAction(w,event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
#endif
{ GraphWidget gw;

  gw = (GraphWidget)GraphActiveWidget(w);
  if( !XtIsSubclass((Widget)gw,graphWidgetClass) )
    { gw = (GraphWidget)XtParent((Widget)gw); }
  SwitchManaged(gw);
}

#if NeedFunctionPrototypes
static void ChangeManaged(Widget gw)
#else
static void ChangeManaged(gw)
Widget gw;
#endif
{ GraphWidget w = (GraphWidget) gw;
  int i; Widget child; GraphConstraints gc;

  for(i=0; i<w->composite.num_children; i++ )
     { child = w->composite.children[i];
       gc = GRAPH_CONSTRAINT(child);
       if( XtIsManaged(child) )
         { if( child->core.x != gc->graph.vertice.x)
             { gc->graph.vertice.x = child->core.x; }
           if( child->core.y != gc->graph.vertice.y )
             { gc->graph.vertice.y = child->core.y; }
           if( (child->core.width+2*child->core.border_width) !=
               gc->graph.vertice.width)
             { gc->graph.vertice.width = child->core.width + 
                                         2*child->core.border_width;
             }
           if( (child->core.height+2*child->core.border_width) !=
                gc->graph.vertice.height)
             { gc->graph.vertice.height = child->core.height + 
                                          2*child->core.border_width;
             }
         }
     }


  layout_graph (w, FALSE);
}


#if NeedFunctionPrototypes
static void Destroy (Widget gw)
#else
static void Destroy (gw)
Widget gw;
#endif
{ int i; Level_t *l;
    GraphWidget w = (GraphWidget) gw;

    for(i=0; i<w->graph.layout.max_levels; i++ )
      { l = &w->graph.layout.levels[i];
        if( l->sequence ) { XtFree( (char *) l->sequence ); }
      }
    if( w->graph.layout.levels ) { XtFree( (char *) w->graph.layout.levels ); }
    if( w->graph.layout.dummies ) { XtFree((char *)w->graph.layout.dummies ); }
    XtReleaseGC (gw, w->graph.gc);
}


#if NeedFunctionPrototypes
static void Redisplay(GraphWidget gw)
#else
static void Redisplay(gw)
GraphWidget gw;
#endif
{ int i;

 if(gw->core.visible)
   { if( gw->graph.layout.explodep )
       { for( i=0; i<gw->graph.layout.edges.n_edges; i++)
            { DrawVirtualEdge(&gw->graph.layout,
                                   gw->graph.layout.edges.edges[i]);
            }
       }
   }
}


#if NeedFunctionPrototypes
static XtGeometryResult QueryGeometry(Widget w,XtWidgetGeometry *intended,
                                               XtWidgetGeometry *preferred)
#else
static XtGeometryResult QueryGeometry(w,intended,preferred)
Widget w;
XtWidgetGeometry *intended;
XtWidgetGeometry *preferred;
#endif
{
    register GraphWidget gw = (GraphWidget) w;

    preferred->request_mode = (CWWidth | CWHeight);
    preferred->width = gw->graph.layout.maxwidth;
    preferred->height = gw->graph.layout.maxheight;

    if (((intended->request_mode & (CWWidth | CWHeight)) ==
	 (CWWidth | CWHeight)) &&
	intended->width == preferred->width &&
	intended->height == preferred->height)
      return XtGeometryYes;
    else if (preferred->width == w->core.width &&
             preferred->height == w->core.height)
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
}


#if NeedFunctionPrototypes
static void clear_children( GraphWidget gw )
#else
static void clear_children( gw )
GraphWidget gw;
#endif
{ int i;
  Widget child;

  if( XtIsRealized ((Widget) gw) )
    { XClearArea( XtDisplay((Widget)gw),XtWindow((Widget)gw),0,0,0,0,True);

      for( i=0; i<gw->composite.num_children; i++ )
         { child = gw->composite.children[i];
           if( GraphIsGraphWidget(child) && XtIsRealized(child) )
             { clear_children((GraphWidget)child);
             }
         }
    }
}

#if NeedFunctionPrototypes
static void layout_graph (GraphWidget gw, Boolean insetvalues)
#else
static void layout_graph ( gw, insetvalues )
GraphWidget gw;
Boolean insetvalues;
#endif
{ execute_layout_alg(gw);
  set_graph_size (gw, insetvalues,
                  gw->graph.layout.maxwidth, gw->graph.layout.maxheight);
  /*
   * And redisplay.
   */
  clear_children(gw);
}



/*****************************************************************************
 *                                                                           *
 * 				Public Routines                              *
 *                                                                           *
 *****************************************************************************/

void
#if NeedFunctionPrototypes
GraphForceLayout (Widget graph)
#else
GraphForceLayout (graph)
    Widget graph;
#endif
{
    layout_graph ((GraphWidget) graph, FALSE);
}

void
#if NeedFunctionPrototypes
GraphInsertEdge (Widget from,Widget to)
#else
GraphInsertEdge(from,to)
    Widget from;
    Widget to;
#endif
{ 
  insert_edge(from,to);
}

void
#if NeedFunctionPrototypes
GraphDeleteEdge (Widget from,Widget to)
#else
GraphDeleteEdge(from,to)
    Widget from;
    Widget to;
#endif
{
  delete_edge(from,to);
}


void
#if NeedFunctionPrototypes
GraphSetEdgeGC (Widget from,Widget to,GC gc)
#else
GraphSetEdgeGC(from,to,gc)
    Widget from;
    Widget to;
    GC gc;
#endif
{ 
  setEdgeGC(from,to,gc);
}

void
#if NeedFunctionPrototypes
GraphInsertEdgeGC (Widget from,Widget to, GC gc)
#else
GraphInsertEdgeGC(from,to,gc)
     Widget from;
     Widget to;
     GC gc;
#endif
{ 
  insertEdgeGC(from,to,gc);
}


Widget
#if NeedFunctionPrototypes
GraphActiveWidget(Widget graph)
#else
GraphActiveWidget(graph)
    Widget graph;
#endif
{ GraphWidget gw; Widget active;

  gw = (GraphWidget)graph;
  active = gw->graph.active_Widget;
  while( ((Widget)gw != active) && GraphIsGraphWidget(active) )
    { gw = (GraphWidget)active;
      active = gw->graph.active_Widget;
    }
  return active;
}

void
#if NeedFunctionPrototypes
GraphExplode(Widget graph)
#else
GraphExplode(graph)
    Widget graph;
#endif
{
    explode((GraphWidget) graph);
}

void
#if NeedFunctionPrototypes
GraphImplode(Widget graph)
#else
GraphImplode(graph)
    Widget graph;
#endif
{
    implode((GraphWidget) graph);
}
