/*
 *  incm - incorporate new mail
 *      author: Yasunari Momoi <momo@bug.org>
 *      created: 2000/10/19
 */

/*
 *  Add the following to your emacs configuration file.
 *
 *  (setq mew-mailbox-type 'mbox)   ; this applies also maildir
 *  (setq mew-mbox-command "incm")
 *  (setq mew-mbox-command-arg "-d /path/to/your/mailbox_or_maildir")
 *
 *  If you want to use "Content-Length:" value when splitting mail,
 *  you MUST set option "-c".
 *
 *
 *  Copyright (C) 2000 Yasunari Momoi.  All rights reserved.
 *  Copyright notice is the same as Mew's one.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
#endif

#if HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif

static char	Version[] = "$Id: incm.c,v 1.11 2000/10/24 14:41:30 momo Exp $";
static char*	Command;

enum BOOL { FALSE, TRUE };

enum MBOXTYPE {
    T_UNKNOWN,
    T_MAILDIR,
    T_MBOX,
};

enum MBOXSTATE {
    ST_UNKNOWN,
    ST_HEADER,
    ST_BODY,
};

#define FBUFSIZ		(BUFSIZ * 32)
#ifndef PATH_MAX
# define PATH_MAX	1024
#endif

static char	FileBuf[FBUFSIZ];
static char	InboxDir[PATH_MAX];
static char	Mbox[PATH_MAX];
static int	MboxType;
static int	Backup;
static int	UseCL;

#if HAVE_MAILLOCK_H
# define USE_MAILLOCK	1
#endif

#if !HAVE_FLOCK
# if HAVE_LOCKF
#  define flock(a, b)	lockf(a, b, 0)
#  define LOCK_EX	F_LOCK
# else
#  define USE_MAILLOCK	1
# endif
#endif

#ifndef LOCK_EX
# include <sys/file.h>
#endif

#ifndef S_ISDIR
# define S_ISDIR(m)	(((m) & S_IFMT) == S_IFDIR)
#endif
#ifndef S_ISREG
# define S_ISREG(m)	(((m) & S_IFMT) == S_IFREG)
#endif

#if !HAVE_STRDUP
static char*
strdup(const char* str)
{
    char* t = malloc(strlen(str) + 1);
    if (t)
	strcpy(t, str);
    return t;
}
#endif

void
warning(const char *fmt, ...)
{
    va_list ap;
    fprintf(stderr, "warning: ");
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, "\n");
}

void
error(const char *fmt, ...)
{
    va_list ap;
    fprintf(stderr, "error: ");
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, "\n");
    perror(NULL);
    exit(EXIT_FAILURE);
}

void*
alloc(size_t size)
{
    void *tmp;
    if ((tmp = malloc(size)) == NULL)
	error("malloc(%d)", size);
    return tmp;
}

void
usage()
{
    printf("usage: %s [-h] [-b] [-d maildir] [-i inboxdir]\n", Command);
    printf("    -h            print this help\n");
    printf("    -d mail       path to mbox/maildir  (now: %s)\n", Mbox);
    printf("    -m mail       path to mbox/maildir  (now: %s)\n", Mbox);
    printf("    -i inboxdir   path to inboxdir  (now: %s)\n", InboxDir);
    printf("    -b            backup mail\n");
    printf("                    mbox: no truncate mbox file\n");
    printf("                    maildir: maildir/cur directory\n");
    printf("    -c            use Content-Length: field (for mbox)\n");
    printf("\n");
    printf("version: %s\n", Version);
    exit(EXIT_FAILURE);
}

void
init_env(int argc, char** argv)
{
    char *home = getenv("HOME");
    Command = argv[0];
    sprintf(InboxDir, ".");
    sprintf(Mbox, "%s/Maildir", home);
    MboxType = T_UNKNOWN;
    Backup = FALSE;
    UseCL = FALSE;
}

int
is_number(unsigned char* str)
{
    do {
	if (!isdigit(*str))
	    return FALSE;
    } while (*++str != '\0');
    return TRUE;
}

int
get_last_seq()
{
    struct dirent* dp;
    DIR* dirp;
    int last = 0;
    int seq;

    if ((dirp = opendir(InboxDir)) == NULL)
	error("opendir(%s)", InboxDir);
    while ((dp = readdir(dirp)) != NULL) {
	if (!is_number(dp->d_name))
	    continue;
	seq = atoi(dp->d_name);
	last = last > seq ? last : seq;
    }
    closedir(dirp);
    return last;
}

int
compare_string(char** i, char** j)
{
    return strcmp(*i, *j);
}

void
copy(char* src, char* dst)
{
    struct timeval tv[2];
    struct stat sb;
    int srcfd, dstfd;
    ssize_t rlen, wlen;

    if ((srcfd = open(src, O_RDONLY, 0)) < 0)
	error("open(%s) for read", src);
    if (fstat(srcfd, &sb))
	error("fstat(%s)", src);
    if ((dstfd = open(dst, O_EXCL | O_CREAT | O_WRONLY | O_TRUNC, 0)) < 0)
	error("open(%s) for write", dst);
    while ((rlen = read(srcfd, FileBuf, FBUFSIZ)) > 0) {
	if ((wlen = write(dstfd, FileBuf, rlen)) != rlen) {
	    close(dstfd);
	    unlink(dst);
	    error("write(%s) (read %d bytes/write %d bytes)", dst, rlen, wlen);
	}
    }
    if (rlen < 0) {
	close(dstfd);
	unlink(dst);
	error("read(%s)", src);
    }
    close(srcfd);

    tv[0].tv_sec = sb.st_atime;
    tv[0].tv_usec = 0;
    tv[1].tv_sec = sb.st_mtime;
    tv[1].tv_usec = 0;
#if HAVE_FUTIMES
    if (futimes(dstfd, tv))
	warning("futimes(%s) failed", dst);
#endif
#if HAVE_FCHMOD
    if (fchmod(dstfd, sb.st_mode))
	warning("fchmod(%s) failed", dst);
#endif
    close(dstfd);
#if !HAVE_FUTIMES
    if (utimes(dst, tv))
	warning("utimes(%s) failed", dst);
#endif
#if !HAVE_FCHMOD
    if (chmod(dst, sb.st_mode))
	warning("chmod(%s) failed", dst);
#endif
}

void
movefile(char* fromfile, char* tofile, char* backupfile, int backup)
{
    if (backup) {
	copy(fromfile, tofile);
	if (rename(fromfile, backupfile))
	    error("rename(%s, %s)", fromfile, backupfile);
    } else {
	if (rename(fromfile, tofile)) {
	    if (errno != EXDEV)
		error("rename(%s, %s)", fromfile, tofile);
	    copy(fromfile, tofile);
	    if (unlink(fromfile))
		error("unlink(%s)", fromfile);
	}
    }
}

/* maildir has {new,cur,tmp} subdirectory. */
int
maildir_names(char* maildir, char** newdir, char** curdir, char** tmpdir)
{
    int len = strlen(maildir) + 5;

    if (maildir == NULL || strlen(maildir) <= 0)
	return -1;
    if (newdir != NULL) {
	*newdir = alloc(len);
	sprintf(*newdir, "%s/new", maildir);
    }
    if (curdir != NULL) {
	*curdir = alloc(len);
	sprintf(*curdir, "%s/cur", maildir);
    }
    if (tmpdir != NULL) {
	*tmpdir = alloc(len);
	sprintf(*tmpdir, "%s/tmp", maildir);
    }
    return 0;
}

/* *WARNING* inboxfile requires PATH_MAX bytes */
int
new_inbox_file(int seq, char inboxfile[])
{
    do {
	sprintf(inboxfile, "%s/%d", InboxDir, ++seq);
	if (access(inboxfile, R_OK) && errno == ENOENT)
	    break;
    } while (TRUE);
    return seq;
}

void
process_maildir(int seq)
{
    struct stat sb;
    struct dirent* dp;
    DIR* dirp;
    char* newdir;
    char* curdir;
    char mailfile[PATH_MAX];
    char inboxfile[PATH_MAX];
    char backupfile[PATH_MAX];
    char** list;
    int listsize = BUFSIZ;
    int listend = 0;
    int i;

    list = alloc(sizeof(char*)*listsize);
    if (maildir_names(Mbox, &newdir, &curdir, NULL))
	error("maildir name is not set (%s)", Mbox);
    if ((dirp = opendir(newdir)) == NULL)
	error("opendir(%s)", newdir);
    while ((dp = readdir(dirp)) != NULL) {
	sprintf(mailfile, "%s/%s", newdir, dp->d_name);
	if (stat(mailfile, &sb))
	    continue;
	if (!(S_ISREG(sb.st_mode) && (sb.st_mode & S_IRUSR)))
	    continue;
	if (listend >= listsize) {
	    listsize *= 2;
	    if ((list = (char**)realloc(list, sizeof(char*)*listsize)) == NULL)
		error("realloc");
	}
	if ((list[listend++] = strdup(dp->d_name)) == NULL)
	    error("strdup(%s)", dp->d_name);
    }
    closedir(dirp);

    qsort(list, listend, sizeof(char*),
	  (int (*)(const void*, const void *))compare_string);

    for (i = 0; i < listend; i++) {
	seq = new_inbox_file(seq, inboxfile);
	sprintf(mailfile, "%s/%s", newdir, list[i]);
	if (Backup)
	    sprintf(backupfile, "%s/%s:2,S", curdir, list[i]);
	movefile(mailfile, inboxfile, backupfile, Backup);
	printf("%d\n", seq);
    }
}

void
lock_mbox(char* lockfile)
{
#if USE_MAILLOCK
    struct stat sb;
    time_t now;
    int fd;
    int retry = 5;

    while (TRUE) {
	if ((fd = open(lockfile, O_WRONLY | O_CREAT | O_EXCL)) < 0) {
	    if (errno != EEXIST)
		error("open(%s)", lockfile);
	    if (retry-- <= 0)
		error("can't get lock(%s)", lockfile);
	    /* if lockfile is too old, remove it. */
	    if (stat(lockfile, &sb) == 0) {
		time(&now);
		if (sb.st_ctime + 300 <= now) {
		    unlink(lockfile);
		    continue;
		}
	    }
	}
	else {
	    /* lock succeeded. */
	    write(fd, "0", 1);
	    close(fd);
	    return;
	}
	sleep(2);
    }
#endif
}

void
unlock_mbox(char* lockfile)
{
#if USE_MAILLOCK
    unlink(lockfile);
#endif
}

void
process_mbox(int seq)
{
    char inboxfile[PATH_MAX];
    char lockfile[PATH_MAX];
    int srcfd, oflag;
    FILE* srcfp = NULL;
    FILE* dstfp = NULL;
    int state = ST_UNKNOWN;
    int lend = TRUE;		/* line ended? */
    int bytes = -1;		/* UseCL (Content-Length:) */

    sprintf(lockfile, "%s.lock", Mbox);
    lock_mbox(lockfile);

    oflag = O_RDWR;
#if defined(O_EXLOCK)
    oflag |= O_EXLOCK;
#endif
    if ((srcfd = open(Mbox, oflag, 0)) < 0) {
	warning("open(%s) for rw/truncate", Mbox);  goto rerr;
    }
#if !defined(O_EXLOCK) && (HAVE_FLOCK || HAVE_LOCKF)
    if (flock(srcfd, LOCK_EX) < 0) {
	warning("flock(%s)", Mbox);  goto rerr;
    }
#endif
    if ((srcfp = fdopen(srcfd, "r")) == NULL) {
	warning("fdopen(%s) for read", Mbox);  goto rerr;
    }
    while (fgets(FileBuf, FBUFSIZ, srcfp) != NULL) {
	if (strlen(FileBuf) == FBUFSIZ-1 && FileBuf[FBUFSIZ-2] != '\n')
	    lend = FALSE;
	else
	    lend = TRUE;

	switch (state) {
	case ST_UNKNOWN:
	    if (lend && strncmp(FileBuf, "From ", 5) == 0) {
		seq = new_inbox_file(seq, inboxfile);
		if ((dstfp = fopen(inboxfile, "w")) == NULL) {
		    warning("fopen(%s) for write", inboxfile);  goto rerr;
		}
		state = ST_HEADER;
	    }
	    break;
	case ST_HEADER:
	    if (fputs(FileBuf, dstfp) == EOF) {
		warning("fputs(%s)", inboxfile);  goto werr;
	    }
	    if (UseCL) {
		if (lend && strncmp(FileBuf, "Content-Length", 14) == 0) {
		    int i;
		    for (i = 14; i < strlen(FileBuf); i++)
			if (isdigit((unsigned char)FileBuf[i]))
			    break;
		    bytes = atoi(&FileBuf[i]);
		}
	    }
	    if (lend && (FileBuf[0] == '\n' || FileBuf[0] == '\r'))
		state = ST_BODY;
	    break;
	case ST_BODY:
	    if (bytes < 0 && lend && strncmp(FileBuf, "From ", 5) == 0) {
		fclose(dstfp);
		printf("%d\n", seq);

		seq = new_inbox_file(seq, inboxfile);
		if ((dstfp = fopen(inboxfile, "w")) == NULL) {
		    warning("fopen(%s) for write", inboxfile);  goto werr;
		}
		state = ST_HEADER;
		break;
	    }
	    if (fputs(FileBuf, dstfp) == EOF)
		goto werr;
	    if (bytes >= 0) {
		bytes -= strlen(FileBuf);
		if (bytes <= 0) {
		    fclose(dstfp);  dstfp = NULL;
		    printf("%d\n", seq);
		    state = ST_UNKNOWN;
		    bytes = -1;
		    break;
		}
	    }
	    break;
	}
    }
    if (dstfp) {
	fclose(dstfp);
	printf("%d\n", seq);
    }
    if (!Backup && ftruncate(srcfd, 0)) {
	unlock_mbox(lockfile);
	error("ftruncate");
    }
    fclose(srcfp);
    unlock_mbox(lockfile);
    return;

 werr:
    if (dstfp)
	fclose(dstfp);
    unlink(inboxfile);
 rerr:
    if (srcfp)
	fclose(srcfp);
    unlock_mbox(lockfile);
    error("process_mbox(%s)", Mbox);
}

void
process()
{
    int seq = get_last_seq();

    switch (MboxType) {
    case T_MAILDIR:
	process_maildir(seq);
	break;
    case T_MBOX:
	process_mbox(seq);
	break;
    default:
	error("unknown mbox type (%s)", Mbox);
    }
}

void
sanity_check()
{
    struct stat sb;
    uid_t uid = geteuid();

    /* was directory exists? */
    if (stat(InboxDir, &sb))
	error("stat(%s)", InboxDir);
    if (!(S_ISDIR(sb.st_mode) &&
	  sb.st_uid == uid && (sb.st_mode & S_IWUSR)))
	error("can't write directory (%s)", InboxDir);

    /* mailbox type check */
    if (stat(Mbox, &sb))
	error("stat(%s)", Mbox);
    if (S_ISDIR(sb.st_mode)) {
	char* newdir;
	char* curdir;

	if (maildir_names(Mbox, &newdir, &curdir, NULL))
	    error("maildir name is not set (%s)", Mbox);
	if (stat(newdir, &sb))
	    error("stat(%s)", newdir);
	if (!(S_ISDIR(sb.st_mode) &&
	      sb.st_uid == uid && (sb.st_mode & S_IWUSR)))
	    error("can't write directory (%s)", newdir);

	if (Backup) {
	    if (stat(curdir, &sb))
		error("stat(%s)", curdir);
	    if (!(S_ISDIR(sb.st_mode) &&
		  sb.st_uid == uid && (sb.st_mode & S_IWUSR)))
		error("can't write directory (%s)", curdir);
	}
	MboxType = T_MAILDIR;
    }
    else if (S_ISREG(sb.st_mode)) {
	MboxType = T_MBOX;
    }
    else
	error("unknown mbox file type (%s)", Mbox);
}

int
main(int argc, char** argv)
{
    extern char* optarg;
    extern int optind;
    int ch;

    init_env(argc, argv);

    while ((ch = getopt(argc, argv, "hbcm:d:i:")) != -1) {
	switch (ch) {
	case 'b':
	    Backup = TRUE;
	    break;
	case 'c':
	    UseCL = TRUE;
	    break;
	case 'd':
	case 'm':
	    sprintf(Mbox, "%s", optarg);
	    break;
	case 'i':
	    sprintf(InboxDir, "%s", optarg);
	    break;
	case 'h':
	default:
	    usage();
	}
    }
    argc -= optind;
    argv += optind;

    sanity_check();
    process();
    return 0;
}
