;;; mew-sort.el --- Sorting messages for Mew

;; Author:  Takashi P.KATOH <p-katoh@shiratori.riec.tohoku.ac.jp>
;;          Kazu Yamamoto <Kazu@Mew.org>
;; Created: Feb  6, 1996
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

(defvar mew-sort-debug nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sort variety
;;;

(defvar mew-sort-switch
  '(("text" mew-sort-key-text mew-sort-string)
    ("ml"   mew-sort-key-ml   mew-sort-string)
    ("ml2"  mew-sort-key-ml2  mew-sort-number)
    ("date" mew-sort-key-date mew-sort-string)
    ("num"  mew-sort-key-num  mew-sort-number)
    ("postnum" mew-sort-key-postnum mew-sort-number)))

(defsubst mew-sort-key-text (key folder msg)
  (mew-subject-simplify key nil 'no-replace))

(defsubst mew-sort-key-ml (key folder msg)
  (let ((ret (mew-subject-simplify key nil 'no-replace)))
    (if (string-match "^[[(][^])]+[])][ \t]*" ret)
	(progn
	  (setq ret (substring ret (match-end 0) nil))
	  (mew-subject-simplify ret nil 'no-replace))
      ret)))

(defsubst mew-sort-key-ml2 (key folder msg)
  (let ((ret (mew-subject-simplify key nil 'no-replace)))
    (if (string-match "[[(][^])]+[: ]+([0-9]+)[])][ \t]+" key)
	(progn
	  (string-to-int (mew-match 0 key))
	  (mew-subject-simplify ret nil 'no-replace))
      (string-to-int key))))

(defsubst mew-sort-key-date (key folder msg)
  (if (string= key "")
      (let ((time (mew-file-get-time (mew-expand-folder folder msg))))
	(mew-time-ctz-to-sortkey time))
    (mew-time-rfc-to-sortkey key)))

(defsubst mew-sort-key-num (key folder msg)
  (string-to-int key))

(defsubst mew-sort-key-postnum (key folder msg)
  (if (string-match "[0-9]+$" key)
      (string-to-int (mew-match 0 key))
    (string-to-int key)))

(defsubst mew-sort-key (x) (cdr x))

(defsubst mew-sort-string (x y)
  (or (string= (mew-sort-key x) (mew-sort-key y))
      (string< (mew-sort-key x) (mew-sort-key y))))

(defsubst mew-sort-number (x y)
  (<= (mew-sort-key x) (mew-sort-key y)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sort body
;;;

(defun mew-sort-index  (x) (car x))

(defun mew-summary-sort-rename (src dst)
  (if mew-sort-debug
      (mew-elet
       (insert (format "move %s to %s\n" src dst)))
    (rename-file src dst)))

(defun mew-summary-sort ()
  "Sort messages and list them up again."
  (interactive)
  (mew-summary-only
   (mew-summary-not-in-queue
    (mew-summary-not-in-draft
     (mew-mark-clean)
     (let* ((buf (current-buffer))
	    (folder (mew-summary-folder-name))
	    (sort-key (or (cdr (assoc folder mew-sort-default-key-alist))
			  mew-sort-default-key))
	    (key-type (mew-input-sort-key sort-key))
	    (key (car key-type))
	    (KEY (concat (capitalize key) ":"))
	    (type (cdr key-type))
	    (ent (assoc type mew-sort-switch))
	    (func1 (nth 1 ent))
	    (func2 (nth 2 ent))
	    (tmp (mew-folder-new-message folder 'num-only))
	    (i 0)
	    (tmpbuf (generate-new-buffer mew-buffer-prefix))
	    ent idx med num value files)
       (setq mew-summary-buffer-process t)
       (message "Sorting %s ... " folder)
       (save-excursion
	 (set-buffer tmpbuf)
	 (mew-erase-buffer)
	 (call-process mew-prog-mewls nil t nil
		       "-b" mew-mail-path "-c" mew-news-path
		       "-d" key "-l" "0" "-s" folder)
	 (goto-char (point-min))
	 (while (not (eobp))
	   (if (not (looking-at "^\\([0-9]+\\)[ \t]*:[ \t]*"))
	       (forward-line)
	     (setq num (mew-match 1))
	     (setq med (match-end 0))
	     (forward-line)
	     (mew-header-goto-next)
	     (mew-header-decode-region KEY med (point))
	     (setq value (mew-buffer-substring med (1- (point))))
	     (setq files (cons num files))
	     (setq ent (cons (cons i (funcall func1 value folder num)) ent))
	     (setq i (1+ i)))))
       (mew-remove-buffer tmpbuf)
       (setq files (vconcat (nreverse files)))
       (setq ent (sort ent func2))
       (setq idx (vconcat (mapcar (function mew-sort-index) ent)))
       (if mew-sort-debug
	   (progn
	     (mew-window-configure 'message)
	     ;; message buffer
	     (mew-elet
	      (erase-buffer)
	      (insert "Sort as follows:\n")))
	 (mew-window-configure 'summary))
       ;;         sorted        sorted
       ;;   files    idx    ->  files
       ;; 0    10      1        (was 20)
       ;; 1    20      2        (was 30)
       ;; 2    30      0        (was 10)
       ;;      31(new)
       ;;
       ;;     
       ;;     src                dst
       ;; 10  0 (*a)       31 (*b)
       ;; 20  1 idx[0]     10    0
       ;; 30  2 idx[1]     20    1
       ;; 31  0 idx[2]     30    2
       ;;     (*c)
       ;; *a: initial src is 0
       ;; *b: initial files[dst] is 31 (new tmp file)
       ;; *c: break condition, src is looped!
       ;;     files[src] is 31 (new tmp file)
       ;;
       (let* ((dir (mew-expand-folder folder))
	      (default-directory dir)
	      (len (length idx));; not (length files)
	      src dst)
	 (setq i 0)
	 (while (< i len)
	   (if (= i (aref idx i))
	       ()
	     (setq dst len)
	     (setq src i)
	     (mew-summary-sort-rename (aref files src) tmp)
	     (catch 'loop
	       (while t
		 (setq dst src)
		 (setq src (aref idx dst))
		 (if (= src i) (throw 'loop nil))
		 (mew-summary-sort-rename (aref files src) (aref files dst))
		 (aset idx dst dst)))
	     (mew-summary-sort-rename tmp (aref files dst))
	     (aset idx dst dst))
	   (setq i (1+ i))))
       (message "Sorting %s ... done" folder)
       (setq mew-summary-buffer-process nil)
       (if mew-sort-debug
	   (progn
	     (mew-message-clear-end-of)
	     (set-buffer-modified-p nil)
	     (goto-char (point-min))
	     (mew-pop-to-buffer buf))
	 (mew-erase-buffer)
	 (mew-scan (mew-scan-mewls-src folder))))))))

(provide 'mew-sort)

;;; Copyright Notice:

;; Copyright (C) 1996-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-sort.el ends here
