;;; mew-summary.el --- Summary mode for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary info
;;;

(defvar mew-sinfo-list
  '("scan-id" "find-key" "cursor-line" "start-point" "reviews" "direction"
    "cache-time" "scan-form" "refile" "disp-msg"))

(mew-blinfo-defun 'mew-sinfo mew-sinfo-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Variables
;;;

(defvar mew-last-shell-command "")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Macros
;;;

(defmacro mew-summary-msg-or-part (&rest body)
  "See if the cursor is on a message or a part."
  `(cond
    ((eobp) (message "No message"))
    ((not (or (mew-summary-message-number) (mew-syntax-number)))
     (message "No message"))
    (t ,@body)))

(defmacro mew-summary-msg (&rest body)
  "See if the cursor is on a message."
  `(cond
    ((eobp) (message "No message"))
    ((not (mew-summary-message-number))
     (message "Please use this command on a message, not a part"))
    (t ,@body)))

(defmacro mew-summary-part (&rest body)
  "See if the cursor is on a part."
  `(cond
    ((eobp) (message "No part"))
    ((not (mew-syntax-number))
     (message "Please use this command on a part, not a message"))
    (t ,@body)))

(defmacro mew-summary-multi-msgs (&rest body)
  "Collect messages marked with '@' and set their corresponding
files to FILES."
  `(let* ((FLD-MSGS (mew-summary-mark-collect2 mew-mark-multi))
	  (FLD-MSG-LIST FLD-MSGS) ;; may be used in body
	  FILES ;; may be used in body
	  fld-msg)
     (cond
      ((null FLD-MSGS)
       (message "No %s marks" (char-to-string mew-mark-multi)))
      (t
       ;; a little bit complicated because of Virtual mode
       (while FLD-MSGS
	 (setq fld-msg (car FLD-MSGS))
	 (setq FLD-MSGS (cdr FLD-MSGS))
	 (setq FILES (cons (mew-expand-folder (car fld-msg) (cdr fld-msg))
			   FILES)))
       (setq FILES (nreverse FILES))
       ,@body))))

(defmacro mew-summary-prepare-draft (&rest body)
  "Common procedure to prepare a draft."
  `(progn
     (unwind-protect
	 (let ((find-file-hooks nil)
	       (inhibit-quit t))
	   ,@body
	   ;; XEmacs doesn't draw attachments unless sit for 0...
	   (mew-redraw)
	   ;; XEmacs doesn't draw toolbar, so...
	   (when (and mew-xemacs-p mew-icon-p
		      (specifier-instance default-toolbar-visible-p))
	     (set-specifier default-toolbar-visible-p nil)
	     (set-specifier default-toolbar-visible-p t)))
       (save-buffer)) ;; to make sure not to use this draft again
     (mew-touch-folder mew-draft-folder)
     (message "Draft is prepared")))

(defsubst mew-summary-prepare-three-windows ()
  "Prepare three windows: Summary mode, Message mode, and Draft mode"
  (if (get-buffer (mew-buffer-message))
      (delete-windows-on (mew-buffer-message)))
  (if (< (window-height) 25) (delete-other-windows))
  (let ((split-window-keep-point t))
    (split-window-vertically)))

(defsubst mew-summary-draft-p ()
  (mew-folder-draftp (mew-summary-folder-name 'ext)))

(defsubst mew-summary-queue-p ()
  (mew-folder-queuep (mew-summary-folder-name 'ext)))

(defsubst mew-summary-case1 ()
  (and (mew-summary-or-virtual-p) (not (mew-summary-draft-p))))

(defsubst mew-summary-case2 ()
  (and (mew-summary-p) (not (mew-summary-queue-p))))

(defsubst mew-summary-case3 ()
  (and (mew-summary-p) mew-summary-buffer-process))

(defmacro mew-summary-only (&rest body)
  "See if the mode of this buffer is Summary mode.
This macro is used to prohibit using a command in Virtual mode."
  `(cond
    ((not (mew-summary-p))
     (message "This command can be used in Summary mode only"))
    (t ,@body)))

(defmacro mew-virtual-only (&rest body)
  "See if the mode of this buffer is Virtual mode.
This macro is used to prohibit using a command in Summary mode."
  `(cond
    ((not (mew-virtual-p))
     (message "This command can be used in Virtual mode only"))
    (t ,@body)))

(defmacro mew-thread-only (&rest body)
  "See if this buffer is Thread folder.
This macro is used to prohibit using a command in Summary mode."
  `(cond
    ((not (mew-thread-p))
     (message "This command can be used in Thread folder only"))
    (t ,@body)))

(defmacro mew-summary-or-thread (&rest body)
  "See if this buffer is Thread folder.
This macro is used to prohibit using a command in Virtual mode
to the exclusion of Thread folder."
  `(cond
    ((not (mew-summary-or-thread-p))
     (message "This command can be used in Summary mode or Thread folder."))
    (t ,@body)))

(defmacro mew-summary-not-in-queue (&rest body)
  "See if this folder is not +queue."
  `(cond
    ((not (mew-summary-or-virtual-p))
     (message "This command cannot be used in this mode."))
    ((not (mew-summary-exclusive-p))
     ())
    ((not (mew-which-exec mew-prog-mewls))
     (message "%s doesn't exist" mew-prog-mewls))
    ((mew-summary-queue-p)
     (message "This command cannot be used in %s" (mew-summary-folder-name)))
    (t ,@body)))

(defmacro mew-summary-not-in-draft (&rest body)
  "See if this folder is not +draft."
  `(cond
    ((not (mew-summary-or-virtual-p))
     (message "This command cannot be used in this mode."))
    ((mew-summary-draft-p)
     (message "This command cannot be used in %s" (mew-summary-folder-name)))
    (t ,@body)))


(defsubst mew-summary-message-toobig (fld msg)
  (let ((file (mew-expand-folder fld msg)))
    (and (mew-folder-localp fld)
	 (file-readable-p file)
	 (> (mew-file-get-size file) mew-file-max-size))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Summary mode
;;;

(defun mew-summary-mode ()
  "\\<mew-summary-mode-map>
Mew Summary mode:: major mode to visualize messages in a folder.

The keys that are defined for both Summary mode and Virtual mode are:

\\[mew-summary-display]	This command lets you read through messages.
	That is, display a message, scroll it, and move-then-display
	another message or part.

	See 'mew-summary-show-direction' to set the direction that the
	cursor moves. You can select a value out of 'up, 'down,
	'next(current direction) or 'stop.  Default is
	'down. 'mew-summary-show-direction' is valid in this case only
	because the cursor stays in the other cases.

	If called with '\\[universal-argument]', this command displays
	the current message or part again. This is a convenient way to
	get back to the beginning of the current message or part.

\\[mew-summary-prev-page]	Back-scroll this message. Unnecessary header
	fields are hidden over the window. Type
	'\\[mew-summary-prev-page]' to see them when a message is
	displayed.

\\[mew-summary-analyze-again]	This command removes the cache of this
	message or part and analyzes the message, then displays this
	message or part again.

	If the size of the current message exceeds
	'mew-file-max-size', MIME analysis is skipped then the
	beginning of the raw message is displayed. In this situation,
	this command analyzes the current message without the
	limitation then displays it.

	If the length of a header exceeds 'mew-header-max-length', a
	broken message is displayed. In this situation, this command
	analyzes the current message without the limitation then
	displays it.

	If called with '\\[universal-argument]', analyze the message
	with 'mew-decode-broken' reversed.

\\[mew-summary-analyze-again-alternative] This command analyzes the message again with 
	'mew-use-alternative' reversed.

\\[mew-summary-display-asis]	Display this message in the raw
	format(i.e. without MIME analysis).  The beginning part of the
	message, whose size specified by 'mew-file-max-size', is
	displayed. If called with '\\[universal-argument]', the entire
	message is displayed in the raw format.

\\[mew-summary-scroll-up]	Make this message scroll up with one line.
\\[mew-summary-scroll-down]	Make this message scroll down with one line.

\\[mew-summary-display-down]	Move to below then display. Targets
	includes parts, messages marked with '*', and non-marked
	messages. When called with '\\[universal-argument]', parts are
	skipped.
\\[mew-summary-display-up]	Move to above then display. Targets
	includes parts, messages marked with '*', and non-marked
	messages. When called with '\\[universal-argument]', parts are
	skipped.

\\[mew-summary-jump-message]	Jump to a message according to the number
	which you input.
\\[mew-summary-jump-top]	Go to the beginning of this Summary mode.
\\[mew-summary-jump-bottom]	Go to the end of this Summary mode.

\\[mew-summary-retrieve]	Retrieve received messages to +inbox asynchronously.
	If called with '\\[universal-argument]', +queue is not flushed.
\\[mew-summary-ls]	List this folder asynchronously. 
\\[mew-summary-goto-folder]	Go to the folder which you input.

\\[mew-summary-send]	Write a message. A new draft is prepared in Draft
	mode.
\\[mew-summary-reply]	Answer to this message. A new draft is prepared
	in Draft mode.  Mew automatically decides To: and Cc:.
\\[mew-summary-reply-with-citation]	Answer to this message. A new
	draft is prepared in Draft mode.  Mew automatically decides
	To: and Cc: and cites the body.

\\[mew-summary-forward]	Forward this message to a third person. A new
	draft is prepared in Draft mode and this message is
	automatically attached.
\\[mew-summary-multi-forward]	Forward messages marked with '@' to a
	third person. A new draft is prepared in Draft mode and this
	message is automatically attached.

\\[mew-summary-reedit]	Edit this message again to retry sending. Or
	edit this rfc822 part typically included MIME-encapsulated
	error message.  In the +draft folder, it just edits the
	message. Otherwise, copy the message to the +draft folder,
	then edit.

\\[mew-summary-edit-again]	Edit an old fashioned error message in
	which the original message is encapsulated after after strings
	defined in 'mew-summary-edit-again-regex'
	(e.g. \"----- Original message follows -----\").

\\[mew-summary-review]	Put the review the '*' mark on this message.
	Use '\\[mew-summary-display-review-down]' or
	'\\[mew-summary-display-review-up]' to jump to a message
	marked with '*'.  It can overlay '@'. The cursor stays always.
	See also '\\[mew-summary-mark-refile]',
	'\\[mew-summary-mark-delete]', '\\[mew-summary-mark-regexp]',
	and '\\[mew-summary-mark-all]'.

\\[mew-summary-display-review-down]	Jump to the message marked with
	'*' below.
\\[mew-summary-display-review-up]	Jump to the message marked with '*'
	above.

\\[mew-summary-multi]	Put the multi the '@' mark on this message for
	'\\[mew-summary-multi-forward]', '\\[mew-summary-unshar]',
	'\\[mew-summary-uudecode]', '\\[mew-summary-burst-multi]'. It
	can overlay the '*' mark.  The cursor stays always.

\\[mew-summary-unshar]	Apply 'unshar' on messages marked with '@'.
\\[mew-summary-uudecode]	Apply 'uudecode' on messages marked with '@'.

\\[mew-summary-burst-multi]	De-capsulate messages embedded in the
	messages marked with '@'.
\\[mew-summary-join]	Concat Message/Partial fragments marked with '@'
	to an original message.

\\[mew-summary-undo]	Cancel the mark on this message.
\\[mew-summary-undo-all]	Cancel all marks according to what you input.

\\[mew-summary-mark-regexp]	Put the '*' mark onto Mall messages
	matched to a regular expression.

\\[mew-summary-mark-all]	Put the '*' mark onto all messages which are
	not marked.
\\[mew-summary-mark-review]	Change the '@' mark into the '*' mark.
\\[mew-summary-mark-multi]	Change the '*' mark into the '@' mark.
\\[mew-summary-mark-undo-all]   Unmark all message marked with 'o'/'D'/'X'.
\\[mew-summary-mark-swap]	Swap the '@' mark and the '*' mark.

\\[mew-summary-delete]	Put the delete mark(default is 'D') on this
	message.  This can overlay other marks. When it overlays, the
	cursor stays on the message. If it marks newly, displays the
	next message.  To know what kind of action will be taken, see
	'mew-msg-rm-policy'.

\\[mew-summary-clean-trash]	Really remove all messages in the +trash
	folder.

\\[mew-summary-save]	Save any parts. If the target is a message, you
	are asked which you want to save, the entire message or its
	body. If the target is a non-message part, the part is saved
	(with line delimiter conversion if it is a text object).

\\[mew-summary-toggle-disp-msg]	Toggle 'Summary mode only' and
	'Summary & Message mode'. If you choose 'Summary mode only',
	you can quickly put the delete marks since the next message is
	not displayed.

\\[mew-summary-execute-external] Execute an external command according
	to Content-Type:.  If called with '\\[universal-argument]',
	Content-Type: is asked.

\\[mew-summary-execute-command] Execute an inputed command.

\\[mew-summary-recenter]	Make the current line to the center of
	Summary mode.

\\[mew-summary-burst]	De-capsulate embedded messages in MIME format.

\\[mew-status-update]	Read Addrbook and update its information. If
	executed with '\\[universal-argument]', information of folders
	is also updated in addition to that of Addrbook. The default
	value is 't'.

\\[mew-summary-set-case]	Set cases.

\\[mew-summary-suspend]	Suspend Mew then switch to another buffer. All
	buffers of Mew retain, so you can resume with buffer
	operations.
\\[mew-summary-quit]	Quit Mew. All buffers of Mew are erased.
\\[mew-kill-buffer]	Kill this Summary mode.

\\[mew-summary-convert-local-cs]	Convert the character set of body to
	the one used locally.

\\[mew-summary-decode-pgp]	Decrypting/verifying old-fashioned PGP
	messages.

\\[mew-summary-x-face]	Display xface.

\\[mew-pgp-fetch-key]	Fetch the PGP public key whose key ID appears in
	the X-Mew: field.

\\[mew-pgp-select]	Select PGP version.

\\[mew-summary-addrbook-add]	Adding the value of From: in Message mode
	to Addrbook.  When executed with '\\[universal-argument], it
	will add personal information.  Otherwise, it will add an
	alias.

\\[mew-summary-kill-subprocess]	Kill a process in Summary mode.
	Sometime a process accidentally remains in Summary mode.  In
	this situation, you cannot execute '\\[mew-summary-retrieve]',
	'\\[mew-summary-ls]', nor '\\[mew-summary-exec]'.  Use this
	command to solve this problem.

\\[mew-summary-isearch-forward]	Incremental search forward in Message mode.

\\[mew-summary-isearch-backward]	Incremental search backward in
	Message mode.

\\[mew-summary-print]	Print this message or this part.

\\[mew-summary-pipe-message]	Send this message via pipe.

\\[mew-summary-send-message]	Flush messages stored in all queue folders.
	If executed with '\\[universal-argument]' in a queue folder,
	flush messages stored in the queue folder only.

The following commands are provided for Summary mode only, not for
Virtual mode.

\\[mew-summary-refile]	Put the refile mark(default is 'o') on this
	message.  If already marked with 'o', it prints where this
	message will be refiled. This can overlay other marks. When it
	overlays, the cursor stays on the message. If it marks newly,
	displays the next message. If executed with
	'\\[universal-argument]', it displays how the refile rules
	work in Message mode.

\\[mew-summary-refile-again]	Put a refile mark on this message
	according to the previous refile folder.

\\[mew-summary-exec]	Process marked messages. To cancel the '*' mark,
	use '\\[mew-summary-undo]' or '\\[mew-summary-undo-all]'.

\\[mew-summary-exec-one]	Process the current marked messages.

\\[mew-summary-exec-delete]	Process messages marked with 'D'.
\\[mew-summary-exec-refile]	Process messages marked with 'o'.

\\[mew-summary-mark-refile]	Put the refile mark onto all messages
	marked with '*'.  This is very convenient to refile all
	messages picked by '\\[mew-summary-search-mark]'.

\\[mew-summary-mark-delete]	Put the delete mark onto all messages
	marked with '*'.

\\[mew-summary-search-mark]	Pick messages according to a pick pattern
	which you input, then put the '*' mark onto them. If called
	with '\\[universal-argument]', execute
	'mew-summary-pick-with-external'. Otherwise,
	'mew-summary-pick-with-mewls' is called.  which you input, then
	put the '*' mark onto them.

\\[mew-summary-find-keyword-down]	Display a message marked with '*'
	and find a keyword and highlight it in the forward
	direction. The keyword is stored in a buffer local variable in
	Summary mode. If no key word is set to the variable, this
	command first asks you a keyword. If you want to change the
	stored keyword, execute this command with
	'\\[universal-argument]'.
\\[mew-summary-find-keyword-up]	Display a message marked with '*' and
	find a keyword and highlight it in the backward direction. The
	keyword is stored in a buffer local variable in Summary
	mode. If no key word is set to the variable, this command
	first asks you a keyword. If you want to change the stored
	keyword, execute this command with '\\[universal-argument]'.

\\[mew-summary-virtual]	Go to Virtual mode which gives a single view
	to picked messages from multiple folders. Enter a virtual
	folder name, comma-separated folders, and pick pattern.

\\[mew-summary-sort]	Sort messages and list them up again.
\\[mew-summary-pack]	Pack messages and list them up again.

Candidates of RANGE are as follows:
	all
	update
	<n1>-<n2>
	<n1>-
	-<n2>

Use 'all' to flush the summary buffer. 'update' means the range
between the last message included in Summary mode + 1 and the real last
message on the folder.

PICK <pattern> is as follows (in strong order):
- key=value
	Match if the 'key' field contains the 'value' string
	(case-insensitive).
- key==value
	Match if the 'key' field contains the 'value' string
	(case-sensitive).
- key!=value
	Match if the 'key' field doesn't contain the 'value' string
	(case-insensitive).
- key!==value
	Match if the 'key' field doesn't contain the 'value' string
	(case-sensitive).
- ( <pattern> )
	Evaluate <pattern> first.
- ! <pattern>
	Match if not <pattern>.
- <pattern1> & <pattern2>
	Match if <pattern1> AND <pattern2>.
- <pattern1> | <pattern2>
	Match if <pattern1> OR <pattern2>.

Example:
	! key1=val1 | key2=val2 & key3=val3
is equivalent to
	 key1!=val1 | (key2=val2 & key3=val3)

\"head=\" means all fields in a header.
"
  (interactive)
  (setq major-mode 'mew-summary-mode)
  (setq mode-line-buffer-identification mew-mode-line-id)
  (use-local-map mew-summary-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  ;;
  (make-local-variable 'tab-width)
  (make-local-variable 'search-invisible)
  (setq search-invisible nil)
  (cond
   (mew-gemacs-p
    (require 'font-lock) ;; xxx should be removed Emacs 21.0.104
    (jit-lock-register 'mew-summary-cook-region))
   (t
    (make-local-hook 'window-scroll-functions)
    (add-hook 'window-scroll-functions 'mew-summary-cook-window nil 'local)))
  (mew-sinfo-set-disp-msg t)
  ;;
  (mew-summary-mode-name mew-mode-name-summary)
  (mew-summary-setup-mode-line)
  (mew-summary-setup-decoration)
  (mew-highlight-cursor-line)
  (run-hooks 'mew-summary-mode-hook))

(defvar mew-mode-line-magic-number 3)

(defun mew-summary-setup-mode-line ()
  (setq mode-line-format (copy-sequence (default-value 'mode-line-format)))
  ;; (... (-3 . "%p") "-%-") -> (... Mew-stuff "-%-")
  (let* ((len (length mode-line-format)))
    (when (> len mew-mode-line-magic-number)
      (setcdr (nthcdr (- len mew-mode-line-magic-number) mode-line-format)
	      '("[" mew-summary-buffer-left-msgs " more]"
		(mew-summary-buffer-raw "*")
		"-%-"))))
  (or (assq 'mew-summary-buffer-process mode-line-process)
      (setq mode-line-process
	    (cons '(mew-summary-buffer-process
		    mew-summary-buffer-process-status)
		  mode-line-process))))

(defun mew-summary-reset-mode-line (buf)
  (save-excursion
    (set-buffer buf)
    (setq mew-summary-buffer-left-msgs  "-"))) ;; local variable

(defun mew-summary-mode-name (name)
  (let ((in (if (mew-case-default-p mew-case-input)
		"" mew-case-input))
	(out (if (mew-case-default-p mew-case-output)
		 "" mew-case-output)))
    (if (and (string= in "") (string= out ""))
	(setq mode-name name)
      (if (or (and mew-case-synchronize (string= in out))
              mew-ask-flush-case)
	  (setq mode-name (format "%s %s" name in))
	(setq mode-name (format "%s %s:%s" name in out))))
    (force-mode-line-update)))

(defun mew-summary-mode-line ()
  (unless mew-summary-buffer-process
    (let (regex left)
      (if (and (mew-thread-p) mew-use-thread-separator)
	  (progn
	    (setq regex (concat "^" (regexp-quote mew-thread-separator)))
	    (setq left (1- (mew-count-lines (point) (point-max) regex))))
	(setq left (1- (mew-count-lines (point) (point-max)))))
      (if (and (mew-decode-syntax-p)
	       (equal (mew-decode-syntax-buffer) (current-buffer)))
	  (setq left (- left (mew-count-lines (mew-decode-syntax-begin)
					      (mew-decode-syntax-end)))))
      (if (numberp left)
	  (if (= left 0)
	      (setq mew-summary-buffer-left-msgs  "-") ;; local variable
	    (setq mew-summary-buffer-left-msgs (int-to-string left))))))
  (force-mode-line-update))

(defun mew-summary-folder-name (&optional ext)
  (cond
   (ext (buffer-name))
   ((mew-summary-p)
    (buffer-name))
   ((mew-virtual-p)
    (save-excursion
      (beginning-of-line)
      (unless (mew-summary-message-number)
	(mew-summary-goto-message))
      (if (looking-at mew-regex-virtual)
	  (mew-match 1)
	nil)))
   (t nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Display stuff
;;;

(defun mew-message-set-end-of ()
  (save-restriction
    (widen)
    (save-excursion
      (mew-elet
       (goto-char (point-max))
       (unless (bolp) (insert "\n"))
       (if (and mew-xemacs-p (extent-at (point) nil nil nil 'at))
	   ;; to skip graphics
	   (insert "\n"))
       (mew-message-clear-end-of)
       (when (or mew-end-of-message-string mew-end-of-part-string)
	 (move-overlay (mew-minfo-get-overlay) (point-max) (point-max))
	 (if (mew-decode-syntax-p)
	     (if (mew-summary-end-of-message-p)
		 (mew-message-set-end-of-message)
	       (mew-message-set-end-of-part))
	   (mew-message-set-end-of-message)))))))

(defsubst mew-message-clear-end-of ()
  (unless (overlayp (mew-minfo-get-overlay))
    (mew-minfo-set-overlay (mew-overlay-make (point-max) (point-max))))
  (mew-message-set-end-of-nil))

(defun mew-summary-cache-prefetch ()
  (let ((mew-inherit-prefetching t)
	fld next)
    (save-excursion
      (mew-redraw);; need to display
      (mew-summary-goto-message)
      (cond
       ((eq (mew-sinfo-get-direction) 'up)
	(when (re-search-backward mew-regex-msg-or-part nil t)
	  (setq fld (mew-summary-folder-name))
	  (setq next (mew-summary-message-number))))
       ((eq (mew-sinfo-get-direction) 'down)
	(if (mew-decode-syntax-end)
	    (goto-char (mew-decode-syntax-end))
	  (forward-line))
	(when (re-search-forward mew-regex-msg-or-part nil t)
	  (setq fld (mew-summary-folder-name))
	  (setq next (mew-summary-message-number))))))
    ;; should get the cursor back for display
    (save-excursion
      (if (and fld next
	       (not (mew-cache-hit fld next))
	       (not (string= fld mew-draft-folder))
	       (not (mew-summary-message-toobig fld next)))
	  (mew-cache-message fld next nil 'no-err)))))

(defsubst mew-summary-display-after (direction)
  (cond 
   ((eq direction 'down)
    (mew-summary-display-down))
   ((eq direction 'up)
    (mew-summary-display-up))
   ((eq direction 'next)
    (mew-summary-display-next))
   (t ()))) ;; 'stop

(defsubst mew-summary-display-preamble ()
  ;; message buffer
  (mew-erase-buffer)
  (mew-message-clear-end-of)
  (mew-overlay-delete-buffer) ;; xxx also delete extents?
  ;; kill mark for cite
  (and (mark-marker) (set-marker (mark-marker) nil)))

(defsubst mew-summary-display-postscript (&optional no-hook)
  ;; message buffer
  (unless no-hook (run-hooks 'mew-message-hook))
  (mew-message-set-end-of)
  (set-buffer-modified-p nil))

(defun mew-summary-cursor-postscript ()
  (mew-summary-mode-line)
  (mew-summary-recenter)
  (mew-highlight-cursor-line)
  (set-buffer-modified-p nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Display
;;;

(defun mew-summary-display (&optional redisplay)
  " (1) If called interactively, this command lets you read through
messages.  That is, display a message, scroll it, and
move-then-display another message or part.

See 'mew-summary-show-direction' to set the direction that the cursor
moves. You can select a value out of 'up, 'down, 'next(current
direction) or 'stop.  Default is 'down. 'mew-summary-show-direction'
is valid in this case only because the cursor stays in the other
cases.

 (2) If called interactively with '\\[universal-argument]' (i.e.
REDISPLAY is 't'), this command displays the current message or part
again. This is a convenient way to get back to the beginning of the
current message or part.

 (3) If called internally, this function displays the current message
or part. If it is already displayed, nothing changes.

 (4) If called internally and REDISPLAY is 't', this function displays
the current message or part. Even if it is already displayed, this
function displays it again getting gack to the beginning."
  (interactive "P")
  (when (or redisplay (mew-sinfo-get-disp-msg) (interactive-p))
    (mew-summary-msg-or-part
     (let* ((fld (mew-summary-folder-name))
	    (vfld (mew-summary-folder-name 'ext))
	    (msg (mew-summary-message-number))
	    (part (mew-syntax-nums))
	    (ofld (mew-current-get-fld (mew-frame-id)))
	    (omsg (mew-current-get-msg (mew-frame-id)))
	    (opart (mew-current-get-part (mew-frame-id)))
	    (cache (mew-cache-hit fld (or msg omsg)))
	    (win (selected-window))
	    (read-through (interactive-p))
	    (sumbuf (current-buffer))
	    next prefetch)
       (unwind-protect
	   (progn
	     (mew-summary-toggle-disp-msg 'on)
	     (mew-window-configure 'message)
	     ;; messge buffer
	     (mew-current-set fld (or msg omsg) part)
	     (cond
	      ((null cache)
	       (mew-decode-syntax-delete)
	       (cond
		((string= fld mew-draft-folder)
		 (if (and (string= fld ofld) (string= msg omsg)
			  (not redisplay))
		     (if read-through
			 (if (mew-message-next-page)
			     (setq next t)))
		   (mew-decode-syntax-clear)
		   (mew-summary-display-draft fld msg)
		   (setq prefetch t)))
		((mew-summary-message-toobig fld msg)
		 (if (and (string= fld ofld) (string= msg omsg)
			  (not redisplay))
		     (if read-through
			 (if (mew-message-next-page)
			     (setq next t)))
		   (mew-decode-syntax-clear)
		   (mew-summary-display-raw fld msg mew-file-max-size)
		   (setq prefetch t)
		   (message
		    (substitute-command-keys
		     "Too large, truncated. To see the entire message, type '\\<mew-summary-mode-map>\\[mew-summary-analyze-again]'"))))
		(t
		 (mew-decode-syntax-clear)
		 (mew-summary-cache-message fld msg sumbuf)
		 (setq prefetch t))))
	      (msg
	       (cond
		((or (null ofld)
		     (not (and (string= fld ofld) (string= msg omsg)))
		     opart redisplay)
		 (mew-decode-syntax-clear)
		 (mew-decode-syntax-delete)
		 (mew-summary-display-message cache sumbuf)
		 (setq prefetch t))
		(read-through
		 (if (mew-message-next-page)
		     (setq next t)))))
	      (part
	       (cond
		((or (null opart)
		     (not (equal opart part))
		     redisplay)
		 (mew-summary-display-part cache part))
		;; If called internally, never match below
		(read-through
		 (if (mew-message-next-page)
		     (setq next t)))))))
	 (if (mew-xinfo-get-decode-err)
	     (message "MIME decoding error: %s" (mew-xinfo-get-decode-err)))
	 (mew-minfo-set-summary vfld)
	 (select-window win)
	 ;; summary buffer
	 (mew-summary-cursor-postscript)
	 (if prefetch (mew-summary-cache-prefetch))
	 (if next (mew-summary-display-after mew-summary-show-direction)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; From cache
;;;

(defun mew-summary-display-message (cache sumbuf)
  "Display a message from the CACHE."
  ;; message buffer
  (mew-elet
   (mew-summary-display-preamble)
   (mew-decode-syntax-copy cache)
   (mew-mime-message/rfc822 cache mew-decode-syntax)
   ;; Must print the syntax before mew-summary-display-postscript
   ;; to tell the end of message.
   (mew-decode-syntax-print sumbuf
			    mew-decode-syntax
			    (mew-xinfo-get-multi-form)
			    (mew-xinfo-get-icon-spec))
   (mew-summary-display-postscript)))

(defun mew-summary-display-part (cache nums)
  "Display a part from the CACHE."
  ;; message buffer
  (mew-elet
   (mew-summary-display-preamble)
   (mew-decode-syntax-copy cache)
   (mew-mime-part cache mew-decode-syntax nums)
   (mew-summary-display-postscript)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

(defun mew-summary-display-draft (fld msg)
  "Display the message in +draft specified by FLD and MSG.
The message is not in the MIME form. So, it can't be decoded as MIME.
Just display it as is. This function doesn't create a cache."
  ;; message buffer
  (mew-elet
   (mew-summary-display-preamble)
   (mew-insert-message fld msg mew-cs-autoconv nil)
   (mew-header-goto-end)
   (mew-highlight-header-region (point-min) (point))
   (mew-summary-display-postscript)))

(defun mew-summary-display-raw (fld msg &optional size)
  "DIsplay the message specified by FLD and MSG as is.
If SIZE is specified, truncates it with the size.
This function doesn't create a cache."
  ;; message buffer
  (mew-elet
   (mew-summary-display-preamble)
   (mew-insert-message fld msg mew-cs-binary size)
   (goto-char (point-min))
   (when (re-search-forward mew-eoh nil t)
     (let ((beg (point)))
       (goto-char (point-max))
       (mew-cs-decode-region beg (point) mew-cs-autoconv)))
   (goto-char (point-min))
   (condition-case nil
       (cond
	(mew-summary-display-raw-header
	 (mew-header-goto-end)
	 (setq mew-decode-syntax (mew-decode-syntax-rfc822)))
	(t
	 (mew-decode-rfc822-header) ;; xxx limit..
	 (mew-decode-syntax-arrange-warning)
	 (mew-header-goto-end)
	 (mew-header-arrange (point-min) (point))
	 (setq mew-decode-syntax (mew-decode-syntax-rfc822))))
     (error ()))
   (mew-summary-display-postscript 'no-hook)))

(defun mew-summary-cache-message (fld msg sumbuf &optional unlimit nodisplay)
  "Create a cache for the message specified with FLD and MSG.
If UNLIMIT is non-nil, decodes it without limitations.
If nodisplay is non-nil, displays the cached message."
  ;; message buffer
  (mew-elet
   (mew-summary-display-preamble)
   (let ((cache (mew-cache-message fld msg unlimit)))
     (mew-decode-syntax-copy cache)
     (unless nodisplay
       (mew-mime-message/rfc822 cache mew-decode-syntax))
     ;; Must print the syntax before mew-summary-display-postscript
     ;; to tell the end of message.
     (mew-decode-syntax-print sumbuf
			      mew-decode-syntax
			      (mew-xinfo-get-multi-form)
			      (mew-xinfo-get-icon-spec))
     (unless nodisplay
       (mew-summary-display-postscript))
     cache)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Analyze again
;;;

(defun mew-find-longest-match (nums)
  (save-excursion
    (forward-line)
    (let ((i 1) (max 0) (lim (length nums)) j part lines)
      (while (and (mew-decode-syntax-end) (< (point) (mew-decode-syntax-end)))
	(setq part (mew-syntax-nums))
	(setq j 0)
	(catch 'loop
	  (while (< j lim)
	    (unless (eq (nth j nums) (nth j part))
	      (throw 'loop nil))
	    (setq j (1+ j))))
	(when (> j max)
	  (setq max j)
	  (setq lines i))
	(setq i (1+ i))
	(forward-line))
      lines)))

(defun mew-summary-analyze-again-alternative (&optional arg)
  "This command analyzes the message again with 
'mew-use-alternative' and 'mew-use-text-body' reversed."
  (interactive "P")
  (let ((mew-use-alternative (not mew-use-alternative))
	(mew-use-text-body   (not mew-use-text-body)))
    (mew-summary-analyze-again arg)))

(defun mew-summary-analyze-again (&optional arg)
  "This command removes the cache of this message or part and
analyzes the message, then displays this message or part again.

If the size of the current message exceeds 'mew-file-max-size', MIME
analysis is skipped then the beginning of the raw message is
displayed. In this situation, this command analyzes the current
message without the limitation then displays it.

If the length of a header exceeds 'mew-header-max-length', a broken
message is displayed. In this situation, this command analyzes the
current message without the limitation then displays it.

If called with '\\[universal-argument]', analyze the message
with 'mew-decode-broken' reversed."
  (interactive "P")
  (mew-summary-msg-or-part
   (if (mew-folder-draftp (mew-summary-folder-name))
       (mew-summary-display 'redisplay)
     (let ((mew-decode-broken
	    (if arg (not mew-decode-broken) mew-decode-broken))
	   (fld (mew-summary-folder-name))
	   (vfld (mew-summary-folder-name 'ext))
	   (msg (mew-summary-message-number))
	   (part (mew-syntax-nums))
	   (win (selected-window))
	   (sumbuf (current-buffer))
	   cache lines)
       (unless msg
	 (beginning-of-line)
	 (mew-summary-goto-message)
	 (setq msg (mew-summary-message-number)))
       (mew-cache-delete2 fld msg)
       (unwind-protect
	   (progn
	     (mew-summary-toggle-disp-msg 'on)
	     (mew-window-configure 'message)
	     ;; message buffer
	     (mew-current-set fld msg part)
	     (mew-decode-syntax-clear)
	     (mew-decode-syntax-delete)
	     (if (null part)
		 (mew-summary-cache-message fld msg sumbuf 'unlimit)
	       (setq cache (mew-summary-cache-message
			    fld msg sumbuf 'unlimit 'nodisplay))
	       ;; Decode syntax is displayed.
	       ;; Let's move the cursor for end-of-*.
	       ;; Since its window is not selected, the displayed cursor
	       ;; doesn't move. Hence call (forward-line lines) twice.
	       (save-excursion
		 (set-buffer sumbuf)
		 (setq lines (mew-find-longest-match part))
		 (if lines (forward-line lines))
		 (setq part (mew-syntax-nums)))
	       (mew-summary-display-part cache part)))
	 (if (mew-xinfo-get-decode-err)
	     (message "MIME decoding error: %s" (mew-xinfo-get-decode-err)))
	 (mew-minfo-set-summary vfld)
	 (select-window win)
	 ;; summary buffer
	 (if lines (forward-line lines))
	 (mew-summary-cursor-postscript))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Display ASIS
;;;

(defun mew-summary-display-asis (&optional arg)
  "Display this message in the raw format(i.e. without MIME analysis). 
The beginning part of the message, whose size specified by
'mew-file-max-size', is displayed. If called with '\\[universal-argument]', the
entire message is displayed in the raw format."
  (interactive "P")
  (mew-summary-msg
   (let* ((fld (mew-summary-folder-name))
	  (vfld (mew-summary-folder-name 'ext))
	  (msg (mew-summary-message-number))
	  (win (selected-window))
	  size)
     (unwind-protect
	 (progn
	   (mew-summary-toggle-disp-msg 'on)
	   (mew-window-configure 'message)
	   ;; messge buffer
	   (mew-elet
	    (mew-current-set fld msg nil)
	    ;;
	    (mew-decode-syntax-clear)
	    (mew-decode-syntax-delete)
	    (when (and (null arg)
		       (mew-summary-message-toobig fld msg))
	      (message
	       (substitute-command-keys
		"Too large, truncated. To see the entire message, type 'C-u \\<mew-summary-mode-map>\\[mew-summary-display-asis]'"))
	      (setq size mew-file-max-size))
	    ;;
	    (mew-summary-display-raw fld msg size)))
       (if (mew-xinfo-get-decode-err)
	   (message "MIME decoding error: %s" (mew-xinfo-get-decode-err)))
       (mew-minfo-set-summary vfld)
       (select-window win)
       ;; summary buffer
       (mew-summary-cursor-postscript)))))

(defun mew-summary-insert-part ()
  "Insert raw message or part into Message mode."
  (interactive)
  (mew-summary-part
   (let* ((fld (mew-current-get-fld (mew-frame-id)))
	  (msg (mew-current-get-msg (mew-frame-id)))
	  (part (mew-syntax-nums))
	  (cache (mew-cache-hit fld msg 'must-hit))
	  (syntax (mew-cache-decode-syntax cache))
	  (stx (mew-syntax-get-entry syntax part))
	  (begin (mew-syntax-get-begin stx))
	  (end (mew-syntax-get-end stx))
	  (buf (current-buffer)))
     (unwind-protect
	 (progn
	   (mew-summary-toggle-disp-msg 'on)
	   (mew-window-configure 'message)
	   ;; message buffer
	   (mew-elet
	    (mew-summary-display-preamble)
	    (mew-insert-buffer-substring cache begin end))
	   (goto-char (point-min))
	   (mew-summary-display-postscript))
       (mew-pop-to-buffer buf)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Executing external commands
;;;

(defun mew-summary-execute-external (&optional arg)
  "Execute an external command according to Content-Type:. 
If called with '\\[universal-argument]', Content-Type: is asked."
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-execute-base arg nil nil)))

(defun mew-summary-execute-command (&optional arg)
  "Execute an inputed command.
If called with '\\[universal-argument]', options are asked."
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-execute-base
    nil
    (read-string "Command: ")
    (if arg (delete "" (mew-split-quoted
			(read-string "Options: ") ?\  ?\" ?\"))))))

(defun mew-summary-execute-base (ask-type com args)
  (let* ((fld     (mew-current-get-fld (mew-frame-id)))
	 (msg     (mew-current-get-msg (mew-frame-id)))
	 (part    (mew-syntax-nums))
	 (cache   (mew-cache-hit fld msg 'must-hit))
	 (syntax  (mew-cache-decode-syntax cache))
	 (stx     (mew-syntax-get-entry syntax part))
	 (ctl     (mew-syntax-get-ct stx))
	 (ct      (mew-syntax-get-value ctl 'cap))
	 begin end params program fname cdpl options async doit pt fl)
    (if (not (string= ct mew-ct-msg))
	(setq doit t)
      ;; Message/Rfc822
      (setq stx (mew-syntax-get-part stx))
      (if (mew-syntax-multipart-p stx)
	  (progn
	    (setq stx (mew-syntax-get-entry stx '(1)))
	    (setq ctl (mew-syntax-get-ct stx))
	    (setq ct (mew-syntax-get-value ctl 'cap))
	    (if (mew-ct-textp ct)
		(setq doit t)))
	;; singlepart
	(setq ctl (mew-syntax-get-ct stx))
	(setq ct (mew-syntax-get-value ctl 'cap))
	(setq doit t)))
    (if (not doit)
	(message "No body")
      (setq begin   (mew-syntax-get-begin stx))
      (setq end     (mew-syntax-get-end stx))
      (setq params  (mew-syntax-get-params ctl))
      (setq cdpl    (mew-syntax-get-cdp stx))
      (setq fname   (mew-syntax-get-filename cdpl ctl))
      (when ask-type
	(cond
	 (fname
	  (setq ct (mew-ctdb-ct (mew-ctdb-by-file fname)))
	  (setq pt "Type for %s (%s): ")
	  (setq fl fname))
	 (t
	  (setq pt "Type %s(%s): ")
	  (setq fl "")))
	(setq ct (mew-input-type pt fl ct mew-mime-content-type-list)))
      (setq program (or com (mew-ctdb-prog (mew-ctdb-by-ct ct))))
      (cond
       ((and (listp program) (nth 1 program) (symbolp (nth 1 program)))
	(if (fboundp (nth 1 program))
	    (save-excursion
	      (if (mew-ct-imagep ct)
		  (funcall (nth 1 program) cache begin end params fname)
		(funcall (nth 1 program) cache begin end params)))
	  (message (substitute-command-keys "Use '\\<mew-summary-mode-map>\\[mew-summary-execute-command]' or \\[universal-argument] \\<mew-summary-mode-map>\\[mew-summary-execute-external]'"))))
       (t
	(cond
	 ((listp program)
	  (setq options (nth 1 program))
	  (setq async (nth 2 program))
	  (setq program (nth 0 program)))
	 (t
	  (setq options args)
	  (setq async t)))
	(if (symbolp program)
	    (message "Function %s is not executable here." program)
	  (if (not (mew-which-exec program))
	      (message "Program %s is not found" program)
	    (let ((file (mew-make-temp-name fname))
		  charset rcs)
	      (save-excursion
		(set-buffer cache)
		;; NEVER use call-process-region for privacy reasons
		(if (mew-ct-linebasep ct)
		    (if (mew-ct-textp ct)
			(progn
			  (setq charset (mew-syntax-get-param ctl "charset"))
			  (if charset
			      (setq rcs (mew-charset-to-cs charset))
			    (setq rcs mew-cs-text-for-write)))
		      (setq rcs mew-cs-text-for-write))
		  (setq rcs mew-cs-binary))
		(mew-frwlet 
		 mew-cs-dummy rcs
		 (write-region begin end file nil 'no-msg))
		(if async
		    (mew-mime-start-process program options file)
		  (mew-mime-call-process program options file)))))))))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

(defun mew-summary-mouse-show (e)
  "Mouse version of 'mew-summary-display'."
  (interactive "e")
  (mouse-set-point e)
  (beginning-of-line)
  (call-interactively (function mew-summary-display)))

(defun mew-summary-toggle-disp-msg (&optional arg)
  "Toggle 'Summary mode only' and 'Summary & Message mode'. If 
you choose 'Summary mode only', you can quickly put the delete
	marks since the next message is not displayed."
  (interactive)
  (cond 
   ((eq arg 'on)
    (mew-sinfo-set-disp-msg t))
   ((eq arg 'off)
    (mew-sinfo-set-disp-msg nil)
    (mew-summary-reset-mode-line (current-buffer)))
   (t
    (mew-sinfo-set-disp-msg (not (mew-sinfo-get-disp-msg)))
    (if (mew-sinfo-get-disp-msg)
	(mew-summary-display 'redisplay)
      (mew-summary-goto-message)
      (mew-decode-syntax-delete)
      (mew-window-configure 'summary)
      (mew-current-set nil nil nil)
      (mew-summary-cook-window)
      (mew-summary-reset-mode-line (current-buffer)))))
  (run-hooks 'mew-summary-toggle-disp-msg-hook))

(defun mew-summary-recenter ()
  "Make the current line to the center of Summary mode."
  (interactive)
  (if (or mew-summary-recenter-p (interactive-p))
      (recenter (/ (- (window-height) 2) 2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; direction
;;;

(defun mew-summary-next ()
  (if (eq (mew-sinfo-get-direction) 'up)
      (mew-summary-up)
    (mew-summary-down)))

(defun mew-summary-down ()
  (forward-line)
  (cond 
   ((re-search-forward mew-regex-msg-or-part nil t)
    (mew-summary-thread-move-cursor)
    (mew-sinfo-set-direction 'down)
    t)
   (t 
    (mew-decode-syntax-delete)
    (forward-line -1)
    (mew-summary-thread-move-cursor)
    (mew-window-configure 'summary)
    (mew-current-set nil nil nil)
    (message "No more message")
    nil)))

(defun mew-summary-up ()
  (beginning-of-line)
  (cond 
   ((re-search-backward mew-regex-msg-or-part nil t)
    (mew-summary-thread-move-cursor)
    (mew-sinfo-set-direction 'up)
    t)
   (t 
    (mew-summary-thread-move-cursor)
    (mew-decode-syntax-delete)
    (mew-window-configure 'summary)
    (mew-current-set nil nil nil)
    (message "No more message")
    nil)))

;;

(defun mew-summary-display-next ()
  (if (mew-summary-next) (mew-summary-display nil)))

(defun mew-summary-display-up (&optional arg)
  "Move to above then display. Targets includes parts, messages 
marked with '*', and non-marked messages. When called with '\\[universal-argument]',
parts are skipped."
  (interactive "P")
  (beginning-of-line)
  (when arg
    (mew-summary-goto-message)
    (mew-decode-syntax-delete))
  (if (mew-summary-up) (mew-summary-display nil)))

(defun mew-summary-display-down (&optional arg)
  "Move to below then display. Targets includes parts, messages 
marked with '*', and non-marked messages. When called with '\\[universal-argument]',
parts are skipped."
  (interactive "P")
  (when arg
    (mew-summary-goto-message)
    (mew-decode-syntax-delete))
  (if (mew-summary-down) (mew-summary-display nil)))

;;

(defun mew-summary-prev-page ()
  "\\<mew-summary-mode-map>
Back-scroll this message. Unnecessary header fields are hidden
over the window. Type '\\[mew-summary-prev-page]' to see them when a message is displayed."
  (interactive)
  (mew-summary-scroll-down 'fullpage))

(defun mew-summary-scroll-up ()
  "Make this message scroll up with one line."
  (interactive)
  (mew-summary-msg-or-part
   (let ((win (selected-window))
	 (msg (mew-summary-message-number))
	 (part (mew-syntax-nums))
	 (omsg (mew-current-get-msg (mew-frame-id)))
	 (opart (mew-current-get-part (mew-frame-id))))
     ;; xxx how about folder check?
     (if (or (and msg (string= msg omsg) (null part) (null opart))
	     (and part (equal part opart))) ;; MUST be equal
	 (unwind-protect
	     (progn
	       (mew-window-configure 'message)
	       ;; message buffer
	       (mew-message-next-page 1))
	   (select-window win))
       (call-interactively (function mew-summary-display))))))

(defun mew-summary-scroll-down (&optional fullpage)
  "Make this message scroll down with one line."
  (interactive)
  (mew-summary-msg-or-part
   (let ((win (selected-window))
	 (msg (mew-summary-message-number))
	 (part (mew-syntax-nums))
	 (omsg (mew-current-get-msg (mew-frame-id)))
	 (opart (mew-current-get-part (mew-frame-id))))
     ;; xxx how about folder check?
     (if (or (and msg (string= msg omsg) (null part) (null opart))
	     (and part (equal part opart))) ;; MUST be equal
	 (unwind-protect
	     (progn
	       (mew-window-configure 'message)
	       ;; message buffer
	       (mew-message-prev-page (if fullpage nil 1)))
	   (select-window win))
       (call-interactively (function mew-summary-display))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; to Draft mode
;;;

(defun mew-summary-send (&optional to cc subject)
  "Write a message. A new draft is prepared in Draft mode."
  (interactive)
  (let ((draft (mew-folder-new-message mew-draft-folder)))
    (mew-current-set-window-config)
    (mew-window-configure 'draft)
    (mew-summary-prepare-draft
     (mew-draft-find-and-switch draft)
     (mew-delete-directory-recursively (mew-attachdir draft))
     (mew-draft-header subject nil to cc)
     (mew-draft-mode)
     (run-hooks 'mew-draft-mode-newdraft-hook))))

(defun mew-subject-simplify (str &optional action-list no-replace)
  "A function to simplify a value of Subject: according to
'mew-subject-simplify-replace-alist'."
  (let ((case-fold-search t)
       regexp replace)
    (unless action-list (setq action-list mew-subject-simplify-replace-alist))
    (while action-list
      (setq regexp  (car (car action-list))
	    replace (if no-replace nil (cdr (car action-list)))
	    action-list (cdr action-list))
      (if (string-match regexp str)
	  (setq str (replace-match (if replace (eval replace) "") nil t str))))
    str))

(defun mew-summary-reply (&optional onlytofrom)
  "Answer to this message. A new draft is prepared in Draft mode. 
Mew automatically decides To: and Cc:. Addresses on To: and Cc:
are decided as follows:

    If From: of the message to be replied is not from me:
        Reply-To: doesn't exist in the message to be replied
            Copy From: of the message to be replied to To: (1)
            Copy To: and Cc: of the message to be replied to Cc: (2)
        Reply-To: exists in the message to be replied
            Copy From: and Reply-To: of the message to be replied to To: (3)
            Copy To: and Cc: of the message to be replied to Cc: (4)
    If From: of a message to be replied is from me:
        Copy To: of the message to be replied to To: (5)
        Copy Cc: of the message to be replied to Cc: (6)

You can customize which fields are copied in the case (1)-(6) with the
following variables:

    (1) mew-noreplyto-to-list
    (2) mew-noreplyto-cc-list
    (3) mew-replyto-to-list
    (4) mew-replyto-cc-list
    (5) mew-fromme-to-list
    (6) mew-fromme-cc-list

If executed with '\\[universal-argument]', only From: of the message is copied to To:.
"
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-summary-toggle-disp-msg 'on)
    (mew-current-set-window-config)
    (let ((obuf (current-buffer))
	  (fld (mew-current-get-fld (mew-frame-id)))
	  (msg (mew-current-get-msg (mew-frame-id)))
	  cbuf draft case
	  from reply-to to cc newsgroups subject in-reply-to references
	   encrypted fromme)
      (if (string= (mew-summary-folder-name) mew-draft-folder)
	  (message "Cannot reply to draft message")
	(setq draft (mew-folder-new-message mew-draft-folder))
	(mew-summary-prepare-draft
	 (mew-summary-prepare-three-windows)
	 (mew-draft-find-and-switch draft t)
	 (mew-delete-directory-recursively (mew-attachdir draft))
	 (setq cbuf (current-buffer));; draft
	 (mew-pop-to-buffer obuf) 
	 ;; need to make a cache or a message buffer.
	 (mew-summary-display nil)
	 ;; see also mew-draft-cite
	 (set-buffer (or (save-excursion
			   (set-buffer (mew-buffer-message))
			   (if (mew-header-p) (current-buffer)))
			 ;; header exists only in cache if multipart
			 (mew-cache-hit fld msg)))
	 (when mew-case-guess-when-replied
	   (setq case (mew-draft-get-case-by-guess
		       mew-case-guess-when-replied-alist)))
	 (setq encrypted (mew-syntax-encrypted-p mew-decode-syntax))
	 (save-restriction
	   ;; if body contains ^L, header is not accessible.
	   ;; mew-header-* can't widen essentially. So widen here.
	   (widen)
	   ;; now cache buffer
	   (setq from (mew-header-parse-address mew-from:))
	   (setq reply-to (mew-header-parse-address mew-reply-to:))
	   (cond 
	    (onlytofrom (setq to from))
	    ((mew-is-my-address mew-regex-my-address-list from)
	     ;; This message was sent by me. So, maintain To: and Cc:.
	     (setq fromme t)
	     (setq to (mew-header-parse-address-list2 mew-fromme-to-list))
	     (setq cc (mew-header-parse-address-list2 mew-fromme-cc-list))
	     (unless to (setq to (or reply-to from)))) ;; don't use list
	    (t
	     (cond 
	      (reply-to
	       (setq to (mew-header-parse-address-list2 mew-replyto-to-list))
	       (setq cc (mew-header-parse-address-list2 mew-replyto-cc-list)))
	      (t
	       (setq to (mew-header-parse-address-list2 mew-noreplyto-to-list))
	       (setq cc (mew-header-parse-address-list2 mew-noreplyto-cc-list))))))
	   (setq newsgroups (or (mew-header-get-value mew-followup-to:)
				(mew-header-get-value mew-newsgroups:)))
	   (if (and newsgroups (mew-case-equal newsgroups "poster"))
	       (setq newsgroups nil))
	   (setq subject (mew-header-get-value mew-subj:))
	   (if subject
	       (setq subject (mew-subject-simplify (concat mew-reply-string subject))))
	   (let ((old-message-id  (mew-header-get-value mew-message-id:))
		 (old-in-reply-to (mew-header-get-value mew-in-reply-to:))
		 (old-references  (mew-header-get-value mew-references:))
		 (regex "<[^>]+>")
		 (start 0) tmp-ref skip)
	     (if (and old-message-id (string-match regex old-message-id))
		 (setq old-message-id (mew-match 0 old-message-id))
	       (setq old-message-id nil))
	     ;; Assuming that In-Reply-To: contains one ID.
	     (if (and old-in-reply-to (string-match regex old-in-reply-to))
		 (setq old-in-reply-to (mew-match 0 old-in-reply-to))
	       (setq old-in-reply-to nil))
	     (if (null old-message-id)
		 ();; we don't care even if old-references exist.
	       (setq in-reply-to old-message-id)
	       (if (null old-references)
		   (setq tmp-ref (if old-in-reply-to 
				     (list old-in-reply-to old-message-id)
				   (list old-message-id)))
		 (while (string-match "<[^>]+>" old-references start)
		   (setq start (match-end 0))
		   (setq tmp-ref (cons (mew-match 0 old-references) tmp-ref)))
		 ;; described in old drums but not in RFC2822
		 (if (and old-in-reply-to (not (member old-in-reply-to tmp-ref)))
		     (setq tmp-ref (cons old-in-reply-to tmp-ref)))
		 (setq tmp-ref (nreverse (cons old-message-id tmp-ref))))
	       (if (integerp mew-references-max-count)
		   (setq skip (- (length tmp-ref) mew-references-max-count)))
	       (if (and (numberp skip) (> skip 0))
		   (setq tmp-ref (nthcdr skip tmp-ref)))
	       (setq references (mew-join "\n\t" tmp-ref)))))
	 ;;
	 (mew-pop-to-buffer cbuf) ;; draft
	 (when case
	   (if mew-case-guess-addition
	       (setq case (mew-draft-add-case (mew-tinfo-get-case) case)))
	   (mew-tinfo-set-case case))
	 (mew-draft-header subject nil to cc newsgroups in-reply-to references
			   nil fromme)
	 (when (eq mew-summary-reply-position 'body)
	   (goto-char (mew-header-end))
	   (forward-line))
	 (mew-draft-mode encrypted)
	 (run-hooks 'mew-draft-mode-newdraft-hook)))))))

(defun mew-summary-reply-with-citation (&optional onlytofrom)
  "Answer to this message. A new draft is prepared in Draft mode. 
And this message is automatically cited. See also 'mew-summary-reply'."
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (let ((mew-summary-reply-position nil))
      (mew-summary-reply onlytofrom))
    ;; mew-draft-mode-hook may insert text.
    (save-excursion
      (goto-char (point-max))
      (run-hooks 'mew-before-cite-hook)
      (mew-draft-cite))
    ;; the cursor is after To:
    (cond
     ((eq mew-summary-reply-with-citation-position 'body)
      (goto-char (mew-header-end))
      (forward-line))
     ((eq mew-summary-reply-with-citation-position 'end)
      (goto-char (point-max)))))))
  
(defun mew-summary-forward ()
  "Forward this message to a third person. A new draft is prepared in 
Draft mode and this message is automatically attached."
  (interactive)
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((obuf (current-buffer))
	   (fld (mew-current-get-fld (mew-frame-id)))
	   (msg (mew-current-get-msg (mew-frame-id)))
	   (draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft))
	   file subject fwsubject cbuf)
      (save-excursion
	(mew-summary-goto-message)
	(setq file (mew-expand-folder (mew-summary-folder-name)
				      (mew-summary-message-number))))
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (setq cbuf (current-buffer));; draft
       (mew-pop-to-buffer obuf)
       ;; need to make a cache or a message buffer.
       (mew-summary-display 'redisplay)
       ;;
       (set-buffer (or (save-excursion
			 (set-buffer (mew-buffer-message))
			 (if (mew-header-p) (current-buffer)))
		       ;; header exists only in cache if multipart
		       (mew-cache-hit fld msg)))
       (setq subject (mew-header-get-value mew-subj:))
       (if subject
	   (setq fwsubject (mew-subject-simplify (concat mew-forward-string subject))))
       (mew-pop-to-buffer cbuf) ;;; draft
       ;;
       (mew-draft-header fwsubject 'nl)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft (list file))
       (setq mew-encode-syntax (mew-encode-syntax-initial-multi draftdir 1))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

(defun mew-summary-multi-forward ()
  "Forward messages marked with '@' to a third person. A new draft 
is prepared in Draft mode and this message is automatically 
attached."
  (interactive)
  (mew-summary-multi-msgs
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft)))
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (mew-draft-header nil 'nl)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft FILES)
       (setq mew-encode-syntax
	     (mew-encode-syntax-initial-multi draftdir (length FILES)))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

(defun mew-draft-multi-copy (draft files)
  (let* ((attach (mew-draft-to-attach draft))
	 (attachdir (mew-expand-folder attach)))
    (unless (file-directory-p attachdir) (mew-make-directory attachdir))
    (while files
      (if mew-use-symbolic-link-for-forwarding
	  (mew-symbolic-link (car files) (mew-folder-new-message attach))
	(copy-file (car files) (mew-folder-new-message attach)))
      (setq files (cdr files)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Packing
;;;

(defun mew-summary-pack ()
  "Pack messages and list them up again."
  (interactive)
  (mew-summary-only
   (mew-mark-clean)
   (mew-summary-not-in-queue
    (mew-summary-not-in-draft
     (let ((folder (mew-summary-folder-name))
	   lines)
       (if (and mew-ask-pack (not (y-or-n-p (format "Pack %s? " folder))))
	   ()
	 (setq lines (mew-summary-mark-collect3 mew-mark-collect))
	 (setq mew-summary-buffer-process t)
	 (let* ((dir (mew-expand-folder folder))
		(default-directory dir)
		(n 1)
		(inhibit-quit t)
		msgs msg src dst)
	   (setq msgs (mew-dir-messages "."))
	   (setq msgs (mapcar (function string-to-int) msgs))
	   (setq msgs (sort msgs (function <)));; sort is inevitable
	   (while msgs
	     (setq msg (car msgs))
	     (setq msgs (cdr msgs))
	     (setq src (int-to-string msg))
	     (cond
	      ((= msg n);; including src is a directory
	       (setq n (1+ n)))
	      ((file-directory-p src)
	       )
	      (t
	       (setq dst (int-to-string n))
	       (while (file-exists-p dst)
		 (setq n (1+ n))
		 (setq dst (int-to-string n)))
	       (rename-file src dst)
	       (setq n (1+ n))))))
	 (setq mew-summary-buffer-process nil)
	 (mew-erase-buffer);; for update
	 (mew-scan (mew-scan-mewls-src folder) lines)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folder
;;;

(defun mew-summary-goto-folder (&optional goend fld no-ls)
  "Go to the folder which you input.  If executed with
'\\[universal-argument]', the cursor always goes to the bottom of
Summary mode."
  (interactive "P")
  (let* ((case mew-case-input)
	 (folder (or fld (mew-input-folder (mew-inbox-folder case))))
	 (dir (mew-expand-folder folder))
	 new-folder)
    (cond
     ((mew-folder-newsp folder)
      (setq new-folder (mew-summary-switch-to-folder folder))
      (mew-summary-ls t (or goend new-folder)))
     ((mew-folder-imapp folder)
      (when (mew-folder-check folder)
	(setq new-folder (mew-summary-switch-to-folder folder))
	(mew-summary-ls t (or goend new-folder))))
     ((mew-folder-virtualp folder)
      (if (get-buffer folder)
	  (if (mew-virtual-thread-p folder)
	      (if (mew-thread-cache-valid-p folder)
		  (mew-summary-switch-to-folder folder)
		(message "%s is old. " folder))
	    (mew-summary-switch-to-folder folder))
        (message "No such virtual folder: %s" folder)
	(mew-folder-delete folder)))
     (t ;; mail or local news
      (if (null dir)
	  (message "Folder is wrong")
	(if (not (file-directory-p dir))
	    (message "No such folder %s" folder)
	  (if no-ls
	      (mew-summary-switch-to-folder folder) ;; Virtual mode
	    (setq new-folder (mew-summary-switch-to-folder folder))
	    (mew-summary-ls 'stay (or goend new-folder)))))))))

(defun mew-summary-switch-to-folder (folder &optional thread)
  (let ((ofolder (mew-summary-folder-name 'ext))
	new-folder)
    (cond
     ((get-buffer folder)
      (switch-to-buffer folder)
      (unless (mew-folder-virtualp folder) (mew-summary-folder-cache-load))
      (unless (string= ofolder folder) (mew-window-configure 'summary)))
     (t
      (setq new-folder t)
      (switch-to-buffer (get-buffer-create folder))
      (mew-folder-insert folder) ;; just in case
      (mew-buffers-setup folder)
      (if (mew-folder-virtualp folder)
	  (progn
	    (if thread (mew-vinfo-set-thread-p t))
	    (mew-virtual-mode))
	(mew-summary-mode)
	(if (and mew-summary-trace-directory (mew-folder-localp folder))
	    (cd (mew-expand-folder folder)))
	(mew-summary-folder-cache-load))
      (mew-window-configure 'summary)))
    new-folder))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Good old days...
;;;

(defun mew-summary-unshar ()
  "Apply 'unshar' on messages marked with '@'."
  (interactive)
  (mew-summary-multi-msgs
   (if (y-or-n-p (format "Execute %s for these messages? " mew-prog-unshar))
       (let* ((default-directory (mew-summary-input-directory-name)))
	 (message "Executing %s ... " mew-prog-unshar)
	 (apply (function call-process) mew-prog-unshar nil nil nil FILES)
	 (message "Executing %s ... done" mew-prog-unshar)))))

(defun mew-summary-uudecode ()
  "Apply 'uudecode' on messages marked with '@'."
  (interactive)
  (mew-summary-multi-msgs
   (cond
    ((not (mew-which-exec mew-prog-mime-decode))
     ())
    ((not (y-or-n-p "Uudecode these messages? "))
     ())
    (t
     (let ((dir (mew-summary-input-directory-name))
	   (fn nil)
	   (buf (generate-new-buffer mew-buffer-prefix))
	   (tmp (mew-make-temp-name))
	   (case-fold-search nil)
	   (files FILES) start)
       (save-excursion
	 (set-buffer buf)
	 (mew-frwlet
	  mew-cs-text-for-read mew-cs-text-for-write
	  (while files
	    (mew-erase-buffer)
	    (insert-file (car files))
	    (goto-char (point-min))
	    (if (re-search-forward mew-eoh nil t)
		(forward-line))
	    (setq start (point))
	    (goto-char (point-max))
	    (unless (bolp) (insert "\n"))
	    (write-region start (point-max) tmp 'append 'no-msg)
	    (setq files (cdr files))))
	 (mew-erase-buffer)
	 (cd dir)
	 ;; "+" is dummy to let mewencode use the embedded filiname
	 (call-process mew-prog-mime-decode tmp t nil "-d" "-u" "-" "+")
	 (if (file-exists-p tmp) (delete-file tmp))
	 (goto-char (point-min))
	 (if (not (looking-at "^filename: \\(.*\\)"))
	     (message "Failed to executing %s" mew-prog-mime-decode)
	   (setq fn (mew-match 1))
	   (setq fn (mew-summary-prog-exec mew-prog-compress "-df" "Z" fn))
	   (setq fn (mew-summary-prog-exec mew-prog-gzip "-df" "gz" fn))
	   (when (and (string-match "^\\(.*\\)\\.tar$" fn)
		      (y-or-n-p (format "Execute %s for %s? " mew-prog-tar fn)))
	     (message "Executing %s for %s ... " mew-prog-tar fn)
	     (call-process mew-prog-tar nil nil nil "-xf" fn)
	     (message "Executing %s for %s ... done" mew-prog-tar fn))))
       (mew-remove-buffer buf))))))

(defun mew-summary-prog-exec (prog opts suffix tarfile)
  (if (string-match (format "^\\(.*\\)\\.%s$" suffix) tarfile)
      (let ((data (match-data)))
	;; save match data here for OS/2
	(unwind-protect
	    (if (not (y-or-n-p (format "Execute %s for %s? " prog tarfile)))
		tarfile
	      (message "Executing %s for %s ... " prog tarfile)
	      (call-process prog nil nil nil opts tarfile)
	      (message "Executing %s for %s ... done" prog tarfile)
	      (set-match-data data)
	      (mew-match 1 tarfile))))
    tarfile))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; These days...
;;;

(defun mew-summary-join ()
  "Concat Message/Partial fragments marked with '@' to an original
message."
  (interactive)
  (mew-summary-only
   (mew-summary-multi-msgs
    (let ((cfld (mew-summary-folder-name))
	  (folder (mew-input-burst-folder))
	  (tfile (mew-make-temp-name))
	  (targets FLD-MSG-LIST)
	  (buf (generate-new-buffer mew-buffer-prefix))
	  fld-msg num ct med dbl vec len idx ttl TTL fid FID i ent
	  beg regex)
      (message "Joining ... ")
      (save-excursion
	(set-buffer buf)
	(mew-set-buffer-multibyte t)
	(mew-erase-buffer)
	(insert "CD: " cfld "\n")
	(while targets
	  (setq fld-msg (car targets))
	  (setq targets (cdr targets))
	  (insert (cdr fld-msg) "\n"))
	(mew-frwlet
	 mew-cs-text-for-read mew-cs-text-for-write
	 (write-region (point-min) (point-max) tfile nil 'no-msg))
	(mew-erase-buffer)
	(call-process mew-prog-mewls nil t nil
		      "-b" mew-mail-path "-c" mew-news-path
		      "-i" tfile "-d" "content-type" "-l" "0")
	(if (file-exists-p tfile) (delete-file tfile))
	(goto-char (point-min))
	(while (not (eobp))
	  (if (not (looking-at "^\\([0-9]+\\)[ \t]*:[ \t]*"))
	      (forward-line)
	    (setq num (mew-match 1))
	    (setq med (match-end 0))
	    (forward-line)
	    (mew-header-goto-next)
	    (setq ct (mew-param-decode
		      (mew-buffer-substring med (1- (point)))))
	    (setq dbl (cons (cons num ct) dbl))))
	(setq len (length dbl))
	(setq vec (make-vector len nil))
	(while dbl
	  (setq ent (car dbl))
	  (setq dbl (cdr dbl))
	  (setq num (car ent))
	  (setq ttl (mew-syntax-get-param ent "total"))
	  (setq fid (mew-syntax-get-param ent "id"))
	  (setq idx (mew-syntax-get-param ent "number"))
	  (if TTL
	      (if (and ttl (not (string= TTL ttl)))
		  (error "total mismatch"))
	    (setq TTL ttl))
	  (if FID
	      (if (or (null fid) (not (string= FID fid)))
		  (error "fragment id mismatch"))
	    (setq FID fid))
	  (unless idx (error "no number"))
	  (setq idx (1- (string-to-int idx)))
	  (if (and (>= idx 0) (< idx len))
	      (aset vec idx num)
	    (error "illegal number")))
	(unless TTL (error "no total"))
	(setq i 0)
	(while (< i len)
	  (unless (stringp (aref vec i)) (error "Not enough fragments"))
	  (setq i (1+ i)))
	;; now reassemble
	(mew-erase-buffer)
	(mew-frwlet
	 mew-cs-text-for-read mew-cs-text-for-write
	 ;; the first fragment
	 (goto-char (point-max))
	 (save-restriction
	   (narrow-to-region (point) (point))
	   (insert-file-contents (mew-expand-folder cfld (aref vec 0)))
	   ;; Removing unnecessary fields from the encapsulating
	   ;; (outer) header.
	   (goto-char (point-min))
	   (mew-header-delete-lines mew-field-delete-for-joining)
	   ;; Concatenating the two headers.
	   (goto-char (point-min))
	   (re-search-forward mew-eoh nil t)
	   (setq beg (point))
	   (while (looking-at mew-eoh) (forward-line))
	   (delete-region beg (point))
	   ;; Removing unnecessary fields from the encapsulated
	   ;; (inner) header.
	   (setq beg (point))
	   (when (re-search-forward mew-eoh nil t)
	     (setq regex (mew-make-field-regex mew-field-delete-for-joining))
	     (save-restriction
	       (narrow-to-region beg (point))
	       (goto-char (point-min))
	       (while (not (eobp))
		 (if (looking-at regex)
		     (setq beg nil) ;; logic is reversed
		   (setq beg (point)))
		 (forward-line)
		 (mew-header-goto-next)
		 (if beg (delete-region beg (point)))))))
	 ;; the second and subsequent fragments.
	 (setq i 1)
	 (while (< i len)
	   (goto-char (point-max))
	   (save-restriction
	     (narrow-to-region (point) (point))
	     (insert-file-contents (mew-expand-folder cfld (aref vec i)))
	     (goto-char (point-min))
	     (re-search-forward mew-eoh nil t)
	     (forward-line)
	     (delete-region (point-min) (point)))
	   (setq i (1+ i)))
	 (write-region (point-min) (point-max)
		       (mew-folder-new-message folder)
		       nil 'no-msg))
	(mew-touch-folder folder)
	(message "Joining ... done"))
      (mew-remove-buffer buf)
      (if (y-or-n-p (format "Go to %s? " folder))
	  (mew-summary-goto-folder t folder))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Jumping
;;;

(defun mew-summary-jump-message (&optional msg)
  "Jump to a message according to the number which you input.
If 'mew-summary-jump-message-then-display' is non-nil,
the message is then displayed."
  (interactive)
  (let ((here (point)))
    (unless msg (setq msg (read-string "Message No. : " "")))
    (cond 
     ((string= msg "") ())
     ((eq msg t) ;; xxx
      (goto-char (point-max))) ;; (forward-line -1)
     (t 
      (goto-char (point-min))
      (if (not (re-search-forward (mew-regex-jmp-msg msg) nil t))
	  (goto-char here)
	(beginning-of-line)
	(if mew-summary-jump-message-then-display
	    (mew-summary-display nil)))))))

(defun mew-summary-jump-to-draft-buffer ()
  "Jump to the newest draft (e.g. +draft/3) if exists."
  (interactive)
  (let ((bufs (buffer-list))
	(regex (mew-folder-regex mew-draft-folder))
	buf draft)
    (while bufs
      (setq buf (car bufs))
      (setq bufs (cdr bufs))
      (if (string-match regex (buffer-name buf))
	  (if (or (null draft) (string< draft (buffer-name buf)))
	      (setq draft (buffer-name buf)))))
    (if draft
	(switch-to-buffer draft)
      (message "No draft buffer exist!"))))

(defun mew-summary-jump-top ()
  "Go to the beginning of this Summary mode.
If 'mew-summary-jump-top-then-display' is non-nil, 
the top message is then displayed."
  (interactive)
  (goto-char (point-min))
  (if mew-summary-jump-top-then-display
      (mew-summary-display nil)))

(defun mew-summary-jump-bottom ()
  "Go to the end of this Summary mode.
If 'mew-summary-jump-bottom-then-display' is non-nil, 
the top message is then displayed."
  (interactive)
  (goto-char (point-max))
  (unless (bobp) (forward-line -1))
  (if mew-summary-jump-bottom-then-display
      (mew-summary-display nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Searching
;;;

(defun mew-summary-isearch-forward ()
  "Incremental search forward in Message mode."
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if (not mwin)
	(message "No message is displayed.")
      (select-window mwin)
      (unwind-protect
	  (progn
	    (widen)
	    (isearch-forward)
	    (mew-message-narrow-to-page))
	(select-window cwin)))))

(defun mew-summary-isearch-backward ()
  "Incremental search backward in Message mode."
  (interactive)
  (let ((cwin (get-buffer-window (current-buffer)))
	(mwin (get-buffer-window (mew-buffer-message))))
    (if (not mwin)
	(message "No message is displayed.")
      (select-window mwin)
      (unwind-protect
	  (progn
	    (widen)
	    (isearch-backward)
	    (mew-message-narrow-to-page))
	(select-window cwin)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Pipe and Printing
;;;

(defun mew-summary-pipe-message (prefix command)
  "Send this message via pipe."
  (interactive
   (list current-prefix-arg 
	 (read-string "Shell command on message: " mew-last-shell-command)))
  (mew-summary-display 'redisplay)
  (if (y-or-n-p "Send this message to pipe? ")
      (save-excursion
	(set-buffer (mew-buffer-message))
	(save-restriction
	  (widen)
	  (if (string= command "")
	      (setq command mew-last-shell-command))
	  (goto-char (point-min)) ; perhaps this line won't be necessary
	  (if prefix
	      (search-forward "\n\n"))
	  (shell-command-on-region (point) (point-max) command nil)
	  (setq mew-last-shell-command command)))))

(defun mew-summary-print ()
  "Print this message or this part."
  (interactive)
  (mew-summary-display 'redisplay)
  (if (y-or-n-p "Print this message? ")
      (save-excursion
	(set-buffer (mew-buffer-message))
	(save-restriction
	  (widen)
	  (funcall mew-print-function)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Misc
;;;

(defun mew-summary-get-region ()
  "Get a region according to 'mew-summary-region-include-cursor-line'
and retern (beg . end)."
  (save-excursion
    (let ((beg (region-beginning))
	  (end (region-end)))
      (goto-char beg)
      (beginning-of-line)
      (setq beg (point))
      (goto-char end)
      (cond
       ((eq mew-summary-region-include-cursor-line t)
	(forward-line))
       ((eq mew-summary-region-include-cursor-line 'end)
	(if (eq (char-after) ?\n)
	    (forward-line)
	  (beginning-of-line)))
       (t
	(if (> (current-column) 0)
	    (forward-line)
	  (beginning-of-line))))
      (cons beg (point)))))

(defun mew-summary-x-face ()
  "Display xface."
  (interactive)
  (mew-summary-msg
   (let ((filters mew-x-face-filter) file xface buf)
     (save-excursion
       (set-buffer (mew-buffer-message))
       (when (setq xface (mew-header-get-value mew-x-face:))
	 (setq buf (generate-new-buffer mew-buffer-prefix))
	 (set-buffer buf)
	 (mew-erase-buffer)
	 (insert xface)
	 (while filters
	   ;; call-process-region is OK...
	   (mew-plet
	    (call-process-region (point-min) (point-max) 
				 (car filters)
				 'delete t nil))
	   (setq filters (cdr filters)))
	 (setq file (mew-make-temp-name))
	 ;; NEVER use call-process-region for privary reasons
	 (mew-flet
	  (write-region (point-min) (point-max) file nil 'no-msg))
	 (mew-mime-start-process mew-x-face-prog mew-x-face-args file))))))

(defun mew-summary-toggle-8bit ()
  (interactive)
  (setq mew-use-8bit (not mew-use-8bit))
  (if mew-use-8bit
      (message "mew-use-8bit has been set to 't'")
    (message "mew-use-8bit has been set to 'nil'")))

(defun mew-summary-goto-msg-mode ()
  (interactive)
  (let (msgp)
    (save-excursion
      (set-buffer (window-buffer (next-window)))
      (if (mew-message-p) (setq msgp t)))
    (if msgp
	(other-window 1)
      (message "No Message mode displayed"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Flushing the queue
;;;

(defun mew-summary-send-message (&optional arg)
  "Flush messages stored in this queue folder or the default
queue folder. If executed with '\\[universal-argument]',
you can set the sending case."
  (interactive "P")
  (let ((qfld (mew-summary-folder-name 'ext)) case)
    (if (or arg mew-ask-flush-case)
 	(setq case (mew-input-case mew-case-output "Queue"))
      (setq case mew-case-output))
    (if (mew-folder-queuep qfld)
	(mew-window-configure 'summary)
      (setq qfld (mew-queue-folder case)))
    (if (and mew-ask-flush-queue
	     (not (y-or-n-p (concat "Flush " qfld  "? "))))
	(message "The queue is not flushed.")
      (mew-smtp-flush-queue qfld case))))

(provide 'mew-summary)

;;; Copyright Notice:

;; Copyright (C) 1996-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-summary.el ends here
