;;; mew.el --- Messaging in the Emacs World

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar 23, 1994
;; Revised: Dec  2, 2002

;;; Commentary:

;; The updated version is available from:
;;	ftp://ftp.Mew.org/pub/Mew/mew-current.tar.gz
;;	http://www.Mew.org/
;;
;; Minimum setup:
;;	(autoload 'mew "mew" nil t)
;;	(autoload 'mew-send "mew" nil t)
;;      ;; (setq mew-name "your name") ;; (user-full-name)
;;      ;; (setq mew-user "user name of e-mail address") ;; (user-login-name)
;;	(setq mew-mail-domain "domain of e-mail address")
;;      ;; (setq mew-pop-user "your POP account")  ;; (user-login-name)
;;	(setq mew-pop-server "your POP server")    ;; if not localhost
;;	(setq mew-smtp-server "your SMTP server")  ;; if not localhost
;;	(setq mew-icon-directory "icon directory") ;; if using XEmacs/Emacs 21
;;      ;; See also mew-config-alist for advanced use
;;
;; Optional setup (Read Mail menu for Emacs 21):
;;      (if (boundp 'read-mail-command)
;;          (setq read-mail-command 'mew))
;;
;; Optional setup (e.g. C-xm for sending a message):
;;	(autoload 'mew-user-agent-compose "mew" nil t)
;;	(if (boundp 'mail-user-agent)
;;	    (setq mail-user-agent 'mew-user-agent))
;;	(if (fboundp 'define-mail-user-agent)
;;	    (define-mail-user-agent
;;	      'mew-user-agent
;;	      'mew-user-agent-compose
;;	      'mew-draft-send-message
;;	      'mew-draft-kill
;;	      'mew-send-hook))

;;; Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Mew version
;;;

(defconst mew-version-number "2.3"
  "Version number for this version of Mew.")
(defconst mew-version (format "Mew version %s" mew-version-number)
  "Version string for this version of Mew.")
(provide 'mew)
(require 'mew-const)
(require 'mew-blvs)
(require 'mew-func)
(require 'mew-vars)
(require 'mew-vars2)
(cond
 ((memq system-type '(OS/2 emx))
  (require 'mew-os2))
 ((eq system-type 'windows-nt)
  (require 'mew-win32))
 (t
  (require 'mew-unix)))

(eval-when-compile
  (if (not (fboundp 'mew-scan-setup))
      (require 'mew-scan))
  (mew-scan-setup))

(defun mew-version-show ()
  "Show mew-version in minibuffer."
  (interactive)
  (message "%s" mew-version))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; For developers
;;;

(defvar mew-debug nil
  "'decode, 'encode, 'net, 'pgp, 'thread, 'sort, 'pack, t for all.")
(defsubst mew-debug (category)
  (or (eq mew-debug t) (eq mew-debug category)))

(defvar mew-profile nil)
(defvar mew-profile-functions-list nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Bootstrap
;;;

(defvar mew-init-p nil)

;;;###autoload
(defun mew (&optional arg)
  "Execute Mew. If 'mew-auto-get' is 't', messages stored in your
mailbox are fetched to the +inbox folder and messages in the +inbox
folder are listed up in Summary mode. If 'mew-auto-get' is 'nil', list
up messages in the inbox folder. If '\\[universal-argument]' is specified, perform this
function thinking that 'mew-auto-get' is reversed."
  (interactive "P")
  (mew-window-push)
  (if (null mew-init-p) (mew-init))
  (let ((auto (if arg (not mew-auto-get) mew-auto-get)))
    (if auto
	(mew-summary-retrieve)
      (mew-summary-goto-folder 'goend (mew-inbox-folder mew-case-input)))
    (setq mew-inbox-window (current-window-configuration))))

;;;###autoload
(defun mew-send (&optional to cc subject)
  "Execute Mew then prepare a draft. This may be used as library
function."
  (interactive)
  (mew-current-set-window-config)
  (if (null mew-init-p) (mew-init))
  (mew-summary-send to cc subject))

;;;###autoload
(defun mew-user-agent-compose (&optional to subject other-headers continue
                                             switch-function yank-action
                                             send-actions)
  "Set up mail composition draft with Mew.
This is 'mail-user-agent' entry point to Mew.

The optional arguments TO and SUBJECT specify recipients and the
initial Subject field, respectively.

OTHER-HEADERS is an alist specifying additional
header fields.  Elements look like (HEADER . VALUE) where both
HEADER and VALUE are strings.

A Draft buffer is prepared according to SWITCH-FUNCTION.

CONTINUE, YANK-ACTION and SEND-ACTIONS are ignored."
  (if (null mew-init-p) (mew-init))
  (let* ((draft (mew-folder-new-message mew-draft-folder))
	 (attachdir (mew-attachdir draft)))
    (mew-current-set-window-config)
    (mew-window-configure 'draft)
    (mew-summary-prepare-draft
     (mew-draft-find-and-switch draft switch-function)
     (mew-delete-directory-recursively attachdir)
     (mew-draft-header subject nil to nil nil nil nil other-headers)
     (mew-draft-mode)
     (run-hooks 'mew-draft-mode-newdraft-hook))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Functions for boot time
;;;

(defun mew-init ()
  (let ((buf (generate-new-buffer mew-buffer-prefix)))
    (run-hooks 'mew-env-hook)
    (load mew-rc-file 'no-err)
    (switch-to-buffer buf)
    (mew-window-configure '(1 0)) ;; Using mew-mode-name-message
    (mew-hello)
    (message "Setting up Mew world...")
    (mew-set-environment)
    (run-hooks 'mew-init-hook)
    (mew-status-update t)
    (mew-passwd-setup)
    (mew-highlight-timer-setup)
    (setq mew-init-p t)
    (message "Setting up Mew world...done")
    (mew-remove-buffer buf)))

(defun mew-set-environment (&optional no-dir)
  (let (error-message)
    (condition-case nil
	(progn
	  ;; sanity check
	  (cond
	   ((string-match "^\\(18\\|19\\)" emacs-version)
	    (setq error-message "Not support Emacs 18/19 nor Mule 1\n")
	    (error "")))
	  ;; initializing
	  (or no-dir (mew-buffers-init))
	  (or no-dir (mew-temp-dir-init))
	  (mew-mark-init)
	  (mew-config-init)
	  (mew-rotate-log-files mew-smtp-log-file))
      (error
       (set-buffer (generate-new-buffer mew-buffer-debug))
       (goto-char (point-max))
       (insert "\n\nMew errors:\n\n")
       (and error-message (insert error-message))
       (set-buffer-modified-p nil)
       (setq buffer-read-only t)
       ;; cause an error again
       (error "Mew found some errors above.")))))

(defun mew-touch-folder-check ()
  "Create dummy file and check modification of directory timestamps.
If operating system cannot update directory timestamp,
set mew-touch-folder-p to t"
  (interactive)
  (when mew-touch-folder-check-enabled-p
    (let* ((dir (expand-file-name mew-mail-path))
	   (file (expand-file-name mew-summary-touch-file dir))
	   (time1 (mew-file-get-time dir))
	   time2 curtime)
      (when (file-writable-p file)
	(setq curtime (current-time))
	(if (and (equal (nth 1 curtime) (nth 1 time1))
		 (equal (nth 0 curtime) (nth 0 time1)))
	    (sleep-for 1))
	(write-region "touched by Mew.(test)" nil file nil 'no-msg)
	(if (file-exists-p file) (delete-file file))
	(setq time2 (mew-file-get-time dir))
	(if (equal time1 time2)
	    ;; this system cannot update directory timestamp.
	    (setq mew-touch-folder-p t))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Status update
;;;

(defun mew-status-update (arg)
  "Read Addrbook and update its information. If executed with '\\[universal-argument]',
information of folders is also updated in addition to that of
Addrbook. The default value is 't'."
  (interactive "P")
  (message "Updating status...")
  (if (interactive-p) (mew-set-environment 'no-dir))
  ;; These two must be before mew-folder-setup
  (mew-config-setup)
  (mew-regex-setup)
  (if arg (mew-folder-setup (interactive-p)))
  (mew-folder-init mew-basic-folders)
  (mew-folder-init mew-inbox-folders)
  (mew-folder-init mew-queue-folders)
  (mew-folder-init mew-mdrop-folders)
  (mew-touch-folder-check)
  (mew-refile-setup)
  (mew-addrbook-setup)
  (mew-scan-setup)
  (mew-pgp-setup)
  (mew-ssh-setup)
  (mew-pop-setup)
  (mew-thread-setup)
  (mew-theme-setup)
  (mew-pop-biff-setup)
  (mew-ct-setup)
  (message "Updating status...done"))

(defvar mew-mime-content-type-list nil
  "Candidate of Content-Type: when CT: is changed in draft buffer.")

(defun mew-ct-setup ()
  (let ((cts mew-mime-content-type) ct)
    (while cts
      (setq ct (car (car cts)))
      (setq cts (cdr cts))
      (if (and (stringp ct) (not (string-match "/$" ct)))
	  (setq mew-mime-content-type-list
		(cons (capitalize ct) mew-mime-content-type-list))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folders
;;;

(defun mew-folder-init (folders)
  (let (folder dir)
    (while folders
      (setq folder (mew-canonicalize-folder (car folders)))
      (setq folders (cdr folders))
      (setq dir (mew-expand-folder folder))
      (unless (file-exists-p dir)
	(mew-make-directory dir)
	(mew-folder-insert folder)
	(message "%s was created" dir))
      (setq dir (file-chase-links dir))
      (if (file-exists-p dir)
	  (if (/= mew-folder-mode (mew-file-get-mode dir))
	      (set-file-modes dir mew-folder-mode))))))

(defvar mew-folder-list nil)
(defvar mew-folder-alist nil)

(defsubst mew-folder-make-alist (list)
  (mapcar (function mew-folder-pair) list))

(defun mew-folder-setup (&optional interactivep)
  (interactive)
  (unless interactivep
    (add-hook 'kill-emacs-hook (function mew-folder-clean-up))
    (setq mew-folder-list (mew-lisp-load mew-folder-list-file))
    (setq mew-folder-alist (mew-lisp-load mew-folder-alist-file)))
  (when (or interactivep (null mew-folder-list))
    (let ((mail-dirs (mew-dir-list mew-mail-path))
	  (news-dirs (mew-dir-list mew-news-path)))
      (setq mew-folder-list (mew-folder-make-list mail-dirs "+"))
      (setq mew-folder-list (nconc mew-folder-list
				   (mew-folder-make-list news-dirs "=")))))
  (when (or interactivep (null mew-folder-alist))
    (setq mew-folder-alist (mew-folder-make-alist mew-folder-list)))
  (when interactivep
    (mew-lisp-save mew-folder-list-file mew-folder-list)
    (mew-lisp-save mew-folder-alist-file mew-folder-alist)))

(defun mew-folder-clean-up ()
  (remove-hook 'kill-emacs-hook (function mew-folder-clean-up))
  ;; ++virtual locates before +folder.
  (let ((lst mew-folder-list)
	(alst mew-folder-alist))
    (while (and (car lst) (mew-folder-virtualp (car lst)))
      (setq lst (cdr lst)))
    (while (and (car alst) (mew-folder-virtualp (car (car alst))))
      (setq alst (cdr alst)))
    (mew-lisp-save mew-folder-list-file lst)
    (mew-lisp-save mew-folder-alist-file alst))
  (setq mew-folder-list nil)
  (setq mew-folder-alist nil))

(defun mew-folder-insert (folder)
  "Insert FOLDER to 'mew-folder-list' and 'mew-folder-alist'.
It is assumed that both variables contain ordered folders
according to 'mew-folder<'. Binary search is used for speed reasons."
  (unless (or (member folder mew-folder-list)
	      (member (file-name-as-directory folder) mew-folder-list))
    (let ((case-fold-search nil)
	  (rfld (mew-folder-regex folder))
	  (max (1- (length mew-folder-list)))
	  (min 0) mid crr1 prv1 crr2 prv2)
      (while (> (- max min) 20) ;; 20 is enough?
	(setq mid (/ (+ min max) 2))
	(if (mew-folder< (nth mid mew-folder-list) folder rfld)
	    (setq min mid)
	  (setq max mid)))
      (setq crr1 (nthcdr min mew-folder-list))
      (setq crr2 (nthcdr min mew-folder-alist))
      (while (and crr1 (mew-folder< (car crr1) folder rfld))
	(setq prv1 crr1)
	(setq crr1 (cdr crr1))
	(setq prv2 crr2)
	(setq crr2 (cdr crr2)))
      (if prv1
	  (setcdr prv1 (cons folder crr1))
	(setq mew-folder-list (cons folder crr1))) ;; xxx +foo or +foo/?
      (if prv2
	  (setcdr prv2 (cons (mew-folder-pair folder) crr2))
	;; xxx
	(setq mew-folder-alist (cons (mew-folder-pair folder) crr2))))))

(defsubst mew-folder< (fld1 fld2 rfld)
  "Folder ordering: Example is as follows:
	Mail/draft
	Mail/from/kazu
	Mail/from/
"
  (cond
   ((string-match (mew-folder-regex fld1) fld2) nil)
   ((string-match rfld fld1) t)
   (t (string< fld1 fld2))))

(defun mew-folder-delete (folder)
  ;; used to delete +virtual only. 
  ;; So, this does not take care of +foo/
  (setq mew-folder-list (delete folder mew-folder-list))
  (setq mew-folder-alist
	;; delq is right. no need to use delete.
	(delq (assoc folder mew-folder-alist) mew-folder-alist)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Listing folders
;;; 

(defsubst mew-dir-list (dir)
  (if (file-directory-p (expand-file-name dir))
      (funcall mew-dir-list-function dir)
    ()))

(defun mew-dir-list-with-link-count (dir)
  "Collecting a directory list under DIR.
Subdirectories are expressed by a list.
This function uses two techniques for speed up.

One is to gather candidates of directory by matching
'mew-regex-folder-candidate'. Its default is [^.0-9]. So, typical
messages whose name is numeric are not gathered. This makes it faster
to check wether or not each candidate is a directory in 'while' loop.

The other is to see if the link count of a directory is 2. If so, the
directory does not have subdirectories. So, it is not necessary to
trace down. This technique can be used on UNIX variants."
  (let ((default-directory (expand-file-name dir default-directory))
	file dirent dirs ent subdirs)
    (setq dirent (directory-files "." nil mew-regex-folder-candidate))
    ;; MUST sort
    (while dirent
      (setq file (car dirent))
      (setq ent (mew-file-chase-links file))
      (setq dirent (cdr dirent))
      (when (file-directory-p ent)
	(setq dirs (cons file dirs))
	(when (and (mew-file-get-links ent) ;; necessary
		   (/= (mew-file-get-links ent) 2))
	  (setq subdirs (mew-dir-list-with-link-count file))
	  (if subdirs (setq dirs (cons subdirs dirs))))))
    (nreverse dirs)))

(defun mew-dir-list-without-link-count (dir)
  "Collecting a directory list under DIR.
Subdirectories are expressed by a list.
This function uses one technique for speed up.

It is to gather candidates of directory by matching
'mew-regex-folder-candidate'. Its default is [^.0-9]. So, typical
messages whose name is numeric are not gathered. This makes it faster
to check wether or not each candidate is a directory in 'while' loop."
  (let ((default-directory (expand-file-name dir default-directory))
	file dirent dirs ent subdirs)
    (setq dirent (directory-files "." nil mew-regex-folder-candidate))
    ;; MUST sort
    (while dirent
      (setq file (car dirent))
      (setq ent (mew-file-chase-links file))
      (setq dirent (cdr dirent))
      (when (file-directory-p ent)
	(setq dirs (cons file dirs))
	(setq subdirs (mew-dir-list-without-link-count file))
	(if subdirs (setq dirs (cons subdirs dirs)))))
    (nreverse dirs)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folder lists
;;;

(defsubst mew-folder-make-list (dirs prefix)
  (nreverse (mew-folder-make-list2 dirs prefix)))

(defun mew-folder-make-list2 (dirs prefix)
  (let (fldpfx dir ret ent)
    (if (= (length prefix) 1)
	(setq fldpfx prefix)
      (setq fldpfx (file-name-as-directory prefix)))
    (while dirs
      (setq dir (car dirs))
      (setq dirs (cdr dirs))
      (setq ent (concat fldpfx dir))
      (cond
       ((consp (car dirs)) ;; not listp because nil is a list.
	(cond
	 ((or (equal mew-attach-folder ent) (equal mew-draft-folder ent))
	  (setq ret (cons ent ret))
	  (setq dirs (cdr dirs))) ;; skip subfolder
	 (t
	  (setq ret (nconc (mew-folder-make-list2 (car dirs) ent) ret))
	  (setq ret (cons (file-name-as-directory ent) ret))
	  (setq dirs (cdr dirs)))))
       (t
	(setq ret (cons ent ret)))))
    ret))

(defun mew-folder-pair (folder)
  (let* ((dir (directory-file-name (mew-folder-to-dir folder)))
	 ;; foo/bar  -> foo/bar
	 ;; foo/bar/ -> foo/bar
	 (subdir (file-name-nondirectory dir)))
	 ;; foo/bar -> bar 
	 ;; foo -> foo
    (if (or (mew-folder-virtualp folder)
	    (mew-folder-local-newsp folder)
	    (string-match mew-regex-ignore-folders folder))
	(list folder nil)
      (list folder subdir))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Current status
;;;

(defsubst mew-frame-id (&optional frame)
  (if window-system
      (let ((fram (or frame (selected-frame))))
	(concat "mew-current-info-"
		(cdr (assq 'window-id (frame-parameters fram)))))
    "mew-current-info-no-window"))

(defvar mew-current-info-list '("fld" "msg" "part" "window"))

(mew-info-defun "mew-current-" mew-current-info-list)

(defun mew-current-set (fld msg part)
  (mew-current-set-fld (mew-frame-id) fld)
  (mew-current-set-msg (mew-frame-id) msg)
  (mew-current-set-part (mew-frame-id) part))

(defun mew-current-clean-up ()
  (if window-system
      (let ((frams (frame-list)))
	(while frams
	  (mew-info-clean-up
	   (concat "mew-current-info-" (mew-frame-id (car frams))))
	  (setq frams (cdr frams))))
    (mew-info-clean-up "mew-current-info-no-window")))

(defsubst mew-current-set-window-config ()
  (mew-current-set-window (mew-frame-id) (current-window-configuration)))

(defsubst mew-current-get-window-config ()
  (let ((win (mew-current-get-window (mew-frame-id))))
    (if (not (window-configuration-p win))
	(setq win mew-inbox-window))
    (if win (set-window-configuration win))
    (mew-current-set-window (mew-frame-id) nil)
    (mew-summary-toolbar-update)
    (mew-redraw)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Window configuration stack
;;;

(defvar mew-inbox-window nil)
(defvar mew-window-stack nil)

(defun mew-window-clean-up ()
  (setq mew-window-stack nil))

(defun mew-window-push ()
  (let ((frame (selected-frame))
	(config (current-window-configuration)))
    (setq mew-window-stack (cons (cons frame config) mew-window-stack))))

(defun mew-window-pop ()
  (let* ((frame (selected-frame))
	 (assoc (assoc frame mew-window-stack)))
    (if (and assoc (window-configuration-p (cdr assoc)))
	(set-window-configuration (cdr assoc))
      (switch-to-buffer (get-buffer-create mew-window-home-buffer)))
    (setq mew-window-stack (delq assoc mew-window-stack))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Window configuration
;;;

(defun mew-window-configure (action)
  "Configure windows accroding to ACTION.
ACTION should be symbol or a list of two numbers.
Predefined symbol is 'summary, 'message, and 'draft.
They are used as a key of 'mew-window-configuration to get
a list of two numbers.

Two numbers means the ratio of the upper window and 
the lower window (i.e. the window of Message buffer).

If Message buffer does not exist, it will be created. If the height of
the lower window is not zero, switch to the buffer."
  (let* ((windows
	  (if (listp action) 
	      action
	    (car (cdr (assq action mew-window-configuration)))))
	 (msgbuf  (mew-buffer-message))
	 (obufwin (get-buffer-window (current-buffer)))
	 (msgwin  (get-buffer-window msgbuf))
	 (height nil) (winsum nil) (sum-height 0) (msg-height 0))
    (setq height (+ (if obufwin (window-height obufwin) 0)
		    (if msgwin  (window-height msgwin)  0)))
    (when (or mew-use-full-window
	      (<= height (* 2 window-min-height)))
      ;; Delete other windows and use full emacs window.
      (delete-other-windows)
      (setq height (window-height (selected-window))))
    ;;
    (if (get-buffer msgbuf)
	(delete-windows-on msgbuf)
      (save-excursion
	(set-buffer (get-buffer-create msgbuf))
	;; "truncate?" is asked in Message mode.
	;; so set the same toolbar as Sumamry mode
	(mew-summary-toolbar-update)
	(mew-message-mode)))
    ;;
    (setq winsum (apply (function +) windows))
    (if (not (zerop (nth 0 windows)))
	(setq sum-height (max window-min-height
			     (/ (* height (nth 0 windows)) winsum))))
    (if (and (eq action 'message) (= (% sum-height 2) 1)) 
	(setq sum-height (1+ sum-height)))
    (if (not (zerop (nth 1 windows)))
	(setq msg-height (max window-min-height
			     (- height sum-height))))
    (setq height (+ sum-height msg-height))
    ;;
    (unless (zerop msg-height)
      (split-window nil sum-height)
      (other-window 1)
      (switch-to-buffer msgbuf 'norecord))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Message buffer
;;;

(defun mew-buffer-message ()
  (let* ((me (selected-frame))
	 (frames (frame-list))
	 (len (length frames))
	 (i 0))
    (catch 'loop
      (while frames
      (if (equal me (car frames))
	  (throw 'loop i))
      (setq i (1+ i))
      (setq frames (cdr frames))))
    (setq i (- len i 1))
    (format "%s%d" mew-buffer-message i)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffers
;;;

(defvar mew-buffers nil)

(defun mew-buffers-init ()
  (setq mew-buffers nil))

(defun mew-buffers-setup (folder)
  (if (not (member folder mew-buffers))
      (setq mew-buffers (cons folder mew-buffers))))

(defun mew-buffers-bury ()
  (let ((buffers mew-buffers))
    (while buffers
      (if (get-buffer (car buffers))
	  (bury-buffer (car buffers)))
      (setq buffers (cdr buffers)))))

(defun mew-buffers-clean-up ()
  (while mew-buffers
    (mew-remove-buffer (car mew-buffers))
    (setq mew-buffers (cdr mew-buffers)))
  (mew-buffers-init))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Temporary directory
;;;

(defvar mew-temp-dir nil)  ;; the default is "/tmp/user_name_uniq"
(defvar mew-temp-file nil) ;; the default is "/tmp/user_name_uniq/mew"

(defun mew-temp-dir-init ()
  "Setting temporary directory for Mew.
mew-temp-file must be local and readable for the user only
for privacy/speed reasons. "
  (setq mew-temp-dir (make-temp-name mew-temp-file-initial))
  (mew-make-directory mew-temp-dir)
  (set-file-modes mew-temp-dir mew-folder-mode)
  (setq mew-temp-file (expand-file-name "mew" mew-temp-dir))
  (add-hook 'kill-emacs-hook (function mew-temp-dir-clean-up)))

(defun mew-temp-dir-clean-up ()
  "A function to remove Mew's temporary directory recursively. 
It is typically called by kill-emacs-hook."
  (remove-hook 'kill-emacs-hook (function mew-temp-dir-clean-up))
  (if (and mew-temp-dir (file-exists-p mew-temp-dir))
      (mew-delete-directory-recursively mew-temp-dir))
  (setq mew-temp-dir nil)
  (setq mew-temp-file nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Regular expressions
;;;

(defvar mew-regex-msg-show nil)
(defvar mew-regex-msg-or-part nil)
(defvar mew-regex-msg-mark nil)
(defvar mew-regex-msg-review nil)
(defvar mew-regex-msg-delete nil)
(defvar mew-regex-attach-beg nil)
(defvar mew-regex-attach-end nil)
(defvar mew-regex-my-address-list nil)
(defvar mew-regex-ignore-folders nil)

(defsubst mew-mark-regex (mark)
  (concat mew-regex-msg (regexp-quote (char-to-string mark))))

(defsubst mew-mark-list-regex (mark-list)
  (concat mew-regex-msg
	  "["
	  (mapconcat (function char-to-string) mark-list "")
	  "]"))

(defun mew-regex-setup ()
  (setq mew-eoh (format "^\\(%s\\|\\)$" (regexp-quote mew-header-separator)))
  (setq mew-regex-msg-show (mew-mark-list-regex (cons ?  mew-mark-show-list)))
  (setq mew-regex-msg-or-part (concat mew-regex-msg-show "\\|" mew-regex-part))
  (setq mew-regex-msg-mark (concat mew-regex-msg mew-regex-mark))
  (setq mew-regex-msg-review (mew-mark-regex mew-mark-review))
  (setq mew-regex-msg-delete (mew-mark-regex mew-mark-delete))
  (setq mew-regex-attach-beg
	(concat "^" mew-draft-attach-boundary-beg "$"))
  (setq mew-regex-attach-end
	(concat "^" mew-draft-attach-boundary-end "$"))
  (setq mew-regex-my-address-list (mew-get-my-address-regex-list))
  (setq mew-regex-ignore-folders
	(mapconcat
	 (function mew-folder-regex)
	 (mew-uniq-list
	  (append
	   mew-basic-folders
	   mew-inbox-folders
	   mew-queue-folders
	   mew-mdrop-folders
	   (list
	    mew-folders-default-folder
	    mew-attach-folder)))
	 "\\|")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Theme
;;;

(defun mew-theme-setup ()
  (interactive)
  (if mew-theme-file (load mew-theme-file 'no-err))
  (when mew-gemacs-p
    (put-text-property 0 (length mew-end-of-message-string)
		       'face 'mew-face-eof-message
		       mew-end-of-message-string)
    (put-text-property 0 (length mew-end-of-part-string)
		       'face 'mew-face-eof-part
		       mew-end-of-part-string)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Teer down
;;;

(defun mew-bury-buffer (&optional buf)
  (bury-buffer buf)
  (delete-windows-on buf t))

(defun mew-kill-buffer (&optional buf)
  "Erase the current buffer."
  (interactive)
  (let* ((buf (or buf (current-buffer)))
	 (folder (if (bufferp buf) (buffer-name buf) buf))
	 obuf)
    (if (get-buffer buf)
	(save-excursion
	  (set-buffer buf)
	  (when (mew-summary-or-virtual-p)
	    (mew-summary-kill-subprocess 'no-msg)
	    (setq obuf (mew-scan-buffer-name folder))
	    ;; xxx should kill pop's buffer...
	    (mew-remove-buffer obuf))
	  (if (mew-virtual-p)
	      (mew-folder-delete folder))
	  (mew-overlay-delete-buffer)))
    (mew-remove-buffer buf)))

(defun mew-buffer-clean-up (regex &optional func)
  (let ((bufs (buffer-list))
	buf bn)
    (unless func (setq func (function mew-kill-buffer)))
    (while bufs
      (setq buf (car bufs))
      (setq bufs (cdr bufs))
      (if (and (setq bn (buffer-name buf))
	       (string-match regex bn))
	  (funcall func buf)))))

(defsubst mew-quit-toolbar-update ()
  (mew-redraw) ;; due to mouse-face bug
  (if (fboundp 'redraw-frame) ;; for BOW
      (redraw-frame (selected-frame)))) ;; update toolbar

(defun mew-summary-suspend ()
  "Suspend Mew then switch to another buffer. All buffers of 
Mew remain, so you can resume with buffer operations."
  (interactive)
  (mew-buffer-clean-up
   (concat "^" (regexp-quote mew-buffer-message))
   (function mew-bury-buffer))
  (mew-buffers-bury)
  (mew-window-pop)
  (mew-quit-toolbar-update)
  (run-hooks 'mew-suspend-hook))

(defun mew-summary-quit ()
  "Quit Mew. All buffers of Mew are erased."
  (interactive)
  (when (y-or-n-p "Quit Mew? ")
    ;; killing buffers
    (mew-cache-clean-up)
    (mew-buffer-clean-up (concat "^" (regexp-quote mew-buffer-message)))
    (mew-buffer-clean-up (mew-folder-regex mew-draft-folder)) ;; +draft/*
    (mew-buffer-clean-up mew-buffer-regex) ;; other buffers
    ;;
    (mew-mark-clean-up)
    (mew-buffers-clean-up) ;; Summary mode and Virtual mode
    ;;
    (mew-temp-dir-clean-up)
    ;;
    (run-hooks 'mew-quit-hook)
    ;;
    ;; lastly, clean up variables
    ;;
    (mew-folder-clean-up)
    (mew-refile-clean-up)
    (mew-current-clean-up)
    (mew-addrbook-clean-up)
    (mew-passwd-clean-up)
    (mew-highlight-timer-clean-up)
    (mew-pop-clean-up)
    (mew-pop-biff-clean-up)
    ;;
    (mew-window-pop)
    (mew-window-clean-up)
    (mew-quit-toolbar-update)
    ;;
    (setq mew-init-p nil)
    (message nil))) ;; just clear

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Load Mew libraries
;;;

(require 'mew-addrbook)
(require 'mew-complete)
(require 'mew-minibuf)
(require 'mew-cache)
(require 'mew-encode)
(require 'mew-decode)
(require 'mew-edit)
(require 'mew-mime)
(require 'mew-mark)
(require 'mew-header)
(require 'mew-pgp)
(require 'mew-smime)
(require 'mew-bq)
(require 'mew-syntax)
(require 'mew-scan)
(require 'mew-pick)
(require 'mew-summary)
(require 'mew-virtual)
(require 'mew-thread)
(require 'mew-message)
(require 'mew-draft)
(require 'mew-attach)
(require 'mew-demo)
(require 'mew-refile)
(require 'mew-ext)
(require 'mew-fib)
(require 'mew-sort)
(require 'mew-highlight)
(require 'mew-smtp)
(require 'mew-pop)
(require 'mew-ssh)
(require 'mew-nntp)
(require 'mew-config)
(require 'mew-auth)

;;; Copyright Notice:

;; Copyright (C) 1994-2002 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew.el ends here
