(*******************************************************************
 *
 *  TTLoad.Pas                                                 1.0
 *
 *    TrueType Tables loaders
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *
 *  Difference between 1.0 and 1.1 : HUGE !!
 *
 *  - Changed the load model to get in touch with TTFile 1.1
 *  - Now loads one whole resident table in one call
 *  - defined resident and instance records/data
 *
 ******************************************************************)

Unit TTLoad;

interface

uses FreeType, TTTypes, TTTables, TTCMap, TTObjs;

 function LookUp_TrueType_Table( face : PFace;
                                 aTag : string ) : int;

 function Load_TrueType_Directory( face      : PFace;
                                   faceIndex : Int ) : TError;

 function Load_TrueType_MaxProfile( face : PFace ) : TError;
 function Load_TrueType_Header    ( face : PFace ) : TError;
 function Load_TrueType_Locations ( face : PFace ) : TError;
 function Load_TrueType_CVT       ( face : PFace ) : TError;
 function Load_TrueType_CMap      ( face : PFace ) : TError;
 function Load_TrueType_Gasp      ( face : PFace ) : TError;
 function Load_TrueType_HMTX      ( face : PFace ) : TError;
 function Load_TrueType_Programs  ( face : PFace ) : TError;
 function Load_trueType_Postscript( face : PFace ) : TError;
 function Load_TrueType_OS2       ( face : PFace ) : TError;
 function Load_TrueType_HDMX      ( face : PFace ) : TError;

 function Load_TrueType_Horizontal_Header( face : PFace ) : TError;

 function Load_TrueType_Any( face        : PFace;
                             tag         : longint;
                             offset      : longint;
                             var buffer;
                             var length  : longint ) : TError;

implementation

uses TTError, TTMemory, TTFile, TTCalc;

  (* Composite glyph decoding flags *)

(*******************************************************************
 *
 *  Function    :  LookUp_TrueType_Table
 *
 *  Description :  Looks for a TrueType table by name
 *
 *  Input  :  face   resident table to look for
 *            aTag        searched tag
 *
 *  Output :  index of table if found, -1 otherwise.
 *
 ******************************************************************)

 function LookUp_TrueType_Table( face : PFace;
                                 aTag : string ) : int;
 var
   TAG : String[4];
   i   : int;
 begin
   TAG[0] := #4;
   for i := 0 to face^.numTables-1 do
     begin

       move( face^.dirTables^[i].Tag, Tag[1], 4 );

       if Tag = ATag then
         begin
           LookUp_TrueType_Table := i;
           exit;
         end
     end;

   (* couldn't find the table *)
   LookUp_TrueType_Table := -1;
 end;


 function LookUp_Mandatory_Table( face : PFace;
                                  aTag : string ) : int;
 var
   table : int;
 begin
   table := LookUp_TrueType_Table( face, aTag );
   if table < 0 then
     error := TT_Err_Table_Missing;

   LookUp_Mandatory_Table := table;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Collection
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 *  Notes : A table directory doesn't own subttables. There is no
 *          constructor or destructor for it.
 *
 ******************************************************************)

 function Load_TrueType_Collection( face : PFace ) : TError;
 var
   n : Int;
 const
   TTC_Tag = ( ord('t') shl 24 ) +
             ( ord('t') shl 16 ) +
             ( ord('c') shl 8  ) +
             ( ord(' ')        );
 begin
   Load_TrueType_Collection := Failure;

   with face^.ttcHeader do
   begin

     if TT_Seek_File( 0 )     or
        TT_Access_Frame( 12 ) then exit;

     Tag      := Get_Tag4;
     version  := Get_Long;
     dirCount := Get_Long;

     TT_Forget_Frame;

     if Tag <> TTC_Tag then
     begin
       Tag            := 0;
       version        := 0;
       dirCount       := 0;
       tableDirectory := nil;

       error := TT_Err_File_Is_Not_Collection;
       exit;
     end;

     if Alloc( tableDirectory, dirCount * sizeof(ULong) ) or
        TT_Access_Frame( dirCount*4 ) then exit;

     for n := 0 to dirCount-1 do
       tableDirectory^[n] := Get_UShort;

     TT_Forget_Frame;
   end;

   Load_TrueType_Collection := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Directory
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 *  Notes : A table directory doesn't own subttables. There is no
 *          constructor or destructor for it.
 *
 ******************************************************************)

 function Load_TrueType_Directory( face      : PFace;
                                   faceIndex : Int ) : TError;
 var
   n        : Int;
   tableDir : TTableDir;
 begin
    Load_TrueType_Directory := Failure;

    {$IFDEF DEBUG} Write('Directory '); {$ENDIF}

    if Load_TrueType_Collection(face) then
      begin
        if error <> TT_Err_File_Is_Not_Collection then
          exit;

        (* The file isn't a collection, exit if index isn't 0 *)
        if faceIndex <> 0 then
          exit;

        error := TT_Err_Ok;

        (* Now skip to the beginning of the file *)
        if TT_Seek_File(0) then
          exit;
      end
    else
      begin
        (* file is a collection. Check the index *)
        if ( faceIndex < 0 ) or
           ( faceIndex >= face^.ttcHeader.dirCount ) then
          begin
            error := TT_Err_Bad_Argument;
            exit;
          end;

        (* select a TT Font within the ttc file *)
        if TT_Seek_File( face^.ttcHeader.tableDirectory^[faceIndex] ) then
          exit;
      end;

    if TT_Access_Frame( 12 ) then
      exit;

    tableDir.version   := GET_Long;
    tableDir.numTables := GET_UShort;

    tableDir.searchRange   := GET_UShort;
    tableDir.entrySelector := GET_UShort;
    tableDir.rangeShift    := GET_UShort;

    {$IFDEF DEBUG} Writeln('Tables number : ', tableDir.numTables ); {$ENDIF}

    TT_Forget_Frame;

    with face^ do
    begin

      numTables := tableDir.numTables;

      if Alloc( dirTables, numTables * sizeof( TTableDirEntry ) ) or
         TT_Access_Frame( 16 * numTables ) then exit;

      for n := 0 to numTables-1 do with dirTables^[n] do
      begin
        ULong(Tag) := GET_Tag4;
        Checksum   := GET_ULong;
        Offset     := GET_Long;
        Length     := Get_Long;
      end;

      TT_Forget_Frame;

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_Directory := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_MaxProfile
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 *  Notes : A maximum profile is a static table that owns no
 *          subttable. It has then no constructor nor destructor
 *
 ******************************************************************)

 function Load_TrueType_MaxProfile( face : PFace ) : TError;
 var
   table : int;
 begin

   Load_TrueType_MaxProfile := Failure;

   {$IFDEF DEBUG} Write('MaxProfile '); {$ENDIF}

   table := LookUp_Mandatory_Table( face, 'maxp');
   if table < 0 then exit;

   with face^ do
   begin

     if TT_Seek_File( dirTables^[table].Offset ) or
        TT_Access_Frame( sizeof(MaxProfile) ) then exit;

     with MaxProfile do
      begin

        ULong(Version) := GET_ULong;

        numGlyphs   := GET_UShort;
        maxPoints   := GET_UShort;
        maxContours := GET_UShort;

        maxCompositePoints   := GET_UShort;
        maxCompositeContours := GET_UShort;
        maxZones             := GET_UShort;
        maxTwilightPoints    := GET_UShort;
        maxStorage           := GET_UShort;
        maxFunctionDefs      := GET_UShort;
        maxINstructionDefs   := GET_UShort;
        maxStackElements     := GET_UShort;

        maxSizeOfInstructions := GET_UShort;
        maxComponentElements  := GET_UShort;
        maxComponentDepth     := GET_UShort;
      end;

     TT_Forget_Frame;

     numGlyphs := MaxProfile.numGlyphs;
     (* compute number of glyphs *)

     maxPoints := MaxProfile.maxCompositePoints;

     if (maxPoints < MaxProfile.maxPoints) then
       maxPoints := MaxProfile.maxPoints;
     (* compute max number of points *)

     maxContours := MaxProfile.maxCompositeContours;
     if maxContours < MaxProfile.maxContours then
       maxContours := MaxProfile.maxContours;
     (* compute max number of contours *)

     maxComponents := MaxProfile.maxComponentElements +
                      MaxProfile.maxComponentDepth;
     (* compute max number of components for glyph loading *)

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_MaxProfile := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Header
 *
 *  Description :  Load the TrueType header table in the resident
 *                 table
 *
 *  Input  :  face   current leading segment.
 *
 *  Output :  True on success. False on failure
 *
 *  Notes : A font header is a static table that owns no
 *          subttable. It has then no constructor nor destructor
 *
 ******************************************************************)

 function  Load_TrueType_Header( face : PFace ) : TError;
 var
   i : int;
 begin
   Load_TrueType_Header := Failure;

    {$IFDEF DEBUG} Write('Header '); {$ENDIF}

   i := LookUp_Mandatory_Table(face, 'head');
   if i <= 0 then exit;

   with face^ do
   begin

     if TT_Seek_File( dirTables^[i].offset ) or
        TT_Access_Frame( sizeof(TT_Header) ) then exit;

     with FontHeader do
     begin

       ULong(Table_Version) := GET_ULong;
       ULong(Font_Revision) := GET_ULong;

       Checksum_Adjust := GET_Long;
       Magic_Number    := GET_Long;

       Flags        := GET_UShort;
       Units_Per_EM := GET_UShort;

       Created [0] := GET_Long; Created [1] := GET_Long;
       Modified[0] := GET_Long; Modified[1] := GET_Long;

       xMin := GET_Short;
       yMin := GET_SHort;
       xMax := GET_SHort;
       yMax := GET_Short;

       Mac_Style       := GET_UShort;
       Lowest_Rec_PPEM := GET_UShort;

       Font_Direction      := GET_Short;
       Index_To_Loc_Format := GET_Short;
       Glyph_Data_Format   := GET_Short;

       {$IFDEF DEBUG} Writeln('Units per EM : ',Units_Per_EM ); {$ENDIF}

     end;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_Header := Success;
 end;

(*******************************************************************
 *
 *  Function    : Load_TrueType_Horizontal_Header
 *
 *  Description :
 *
 *  Input  :  face   current resident leading segment
 *
 *  Output :  True on success. False on failure
 *
 *  Notes : An horizontal header is a static table that owns no
 *          subttable. It has then no constructor nor destructor
 *
 ******************************************************************)

 function Load_TrueType_Horizontal_Header( face : PFace ) : TError;
 var
   t : int;
 begin
   Load_TrueType_Horizontal_Header := Failure;

    {$IFDEF DEBUG} Write('Horizontal Header '); {$ENDIF}

   t := LookUp_Mandatory_Table( face, 'hhea');
   if ( t < 0 ) then exit;

   with face^ do
   begin

     if TT_Seek_File( dirTables^[t].Offset ) or
        TT_Access_Frame( sizeof( TT_Horizontal_Header ) ) then
        exit;

     with HorizontalHeader do
     begin

       Long(Version) := GET_ULong;
       Ascender      := GET_Short;
       Descender     := GET_Short;
       Line_Gap      := GET_Short;

       advance_Width_Max := GET_UShort;

       min_Left_Side_Bearing  := GET_Short;
       min_Right_Side_Bearing := GET_Short;
       xMax_Extent            := GET_Short;
       caret_Slope_Rise       := GET_Short;
       caret_Slope_Run        := GET_Short;

       Reserved[0] := GET_Short;
       Reserved[1] := GET_Short;
       Reserved[2] := GET_Short;
       Reserved[3] := GET_Short;
       Reserved[4] := GET_Short;

       metric_Data_Format := GET_Short;
       number_Of_HMetrics := GET_UShort;

     end;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_Horizontal_Header := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Locations
 *
 *  Description :  Loads the location table in resident table
 *
 *  Input  :  face     Current Resident Leading Segment
 *
 *  Output :  True on success. False on failure
 *
 *  NOTES :
 *
 *    The Font Header *must* be loaded in the leading segment
 *    before calling this function.
 *
 *    This table is destroyed directly by the resident destructor.
 *
 ******************************************************************)

 function Load_TrueType_Locations( face : PFace ): TError;
 var
   t, n        : int;
   LongOffsets : int;
 begin

   Load_TrueType_Locations := Failure;

   {$IFDEF DEBUG} Write('Locations '); {$ENDIF}

   with face^ do
   begin

     LongOffsets :=  fontHeader.Index_To_Loc_Format;

     T := LookUp_Mandatory_Table( face, 'loca' );
     if t < 0 then exit;

     if TT_Seek_File( dirTables^[T].Offset ) then exit;

     if LongOffsets <> 0 then
       begin

         numLocations := dirTables^[T].Length shr 2;

         {$IFDEF DEBUG}
         Writeln('Glyph locations # ( 32 bits offsets ) : ', numLocations );
         {$ENDIF}

         if Alloc( glyphLocations, sizeof(Long)*numLocations ) or
            TT_Access_Frame( numLocations*4 ) then exit;

         for n := 0 to numLocations-1 do
           glyphLocations^[n] := GET_Long;

         TT_Forget_Frame;

       end
     else
       begin

         numLocations := dirTables^[T].Length shr 1;

         {$IFDEF DEBUG}
         Writeln('Glyph locations # ( 16 bits offsets ) : ', numLocations );
         {$ENDIF}

         if Alloc( glyphLocations, sizeof(Long)*numLocations ) or
            TT_Access_Frame( numLocations*2 ) then exit;

         for n := 0 to numLocations-1 do
           glyphLocations^[n] := Long(GET_UShort) * 2;

         TT_Forget_Frame;
       end;

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_Locations := Success;
 end;


(*******************************************************************
 *
 *  Function    :  Load_TrueType_CVT
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  :  This attribute table is destroyed by the resident
 *            destructor.
 *
 ******************************************************************)

 function Load_TrueType_CVT( face : PFace ): TError;
 var
   t, n : Int;
 begin
   Load_TrueType_CVT := Failure;

   {$IFDEF DEBUG} Write('CVT '); {$ENDIF}

   (* the CVT table is optional *)

   t := LookUp_TrueType_Table( face, 'cvt ');
   if t < 0 then
   begin
     face^.cvt     := nil;
     face^.cvtSize := 0;
     Load_TrueType_CVT := Success;
     {$IFDEF DEBUG}  writeln('none'); {$ENDIF}
     exit;
   end;

   with face^ do
   begin

     cvtSize := dirTables^[t].Length div 2;

     if Alloc( cvt, sizeof(Short)*cvtSize )  or

        TT_Seek_File( dirTables^[t].Offset ) or

        TT_Access_Frame( 2*cvtSize )         then exit;

     for n := 0 to cvtSize-1 do
       cvt^[n] := GET_Short;

     TT_Forget_Frame;
   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}
   Load_TrueType_CVT := Success;
 end;


(*******************************************************************
 *
 *  Function    :  Load_TrueType_Gasp
 *
 *  Description :
 *
 *  Input  :  face
 *
 ******************************************************************)

 function Load_TrueType_Gasp( face : PFace ) : TError;
 var
   gRanges  : PGaspRanges;
   table, i : Int;
 label
   Fail;
 begin
   Load_TrueType_Gasp := Failure;

   with face^.gasp do
   begin
     version    := 0;
     numRanges  := 0;
     gaspRanges := nil;
   end;

   table := Lookup_TrueType_Table( face, 'gasp' );
   if ( table < 0 ) then
   begin
     Load_TrueType_Gasp := Success;
     exit;
   end;

   if TT_Seek_File( face^.dirTables^[table].Offset ) or
      TT_Access_Frame( 4 ) then exit;

   with face^.gasp do
   begin
     version    := Get_UShort;
     numRanges  := Get_UShort;
     gaspRanges := nil;
   end;

   TT_Forget_Frame;

   if Alloc( gRanges, face^.gasp.numRanges * sizeof(TGaspRange) ) or
      TT_Access_Frame( face^.gasp.numRanges * 4 ) then
     goto Fail;

   face^.gasp.gaspRanges := gRanges;

   for i := 0 to face^.gasp.numRanges-1 do
     with gRanges^[i] do
     begin
       maxPPEM  := Get_UShort;
       gaspFlag := Get_UShort;
     end;

   TT_Forget_Frame;

   Load_TrueType_Gasp := Success;
   exit;

 Fail:
   Free( gRanges );
   face^.gasp.numRanges := 0;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_CMap
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  :  The Cmap table directory is destroyed by the resident
 *            destructor. The Cmap subtables must be destroyed by
 *            Free_CMap_Table.
 *
 ******************************************************************)

 function Load_TrueType_CMap( face : PFace ) : TError;
 var
   off, table_start : Longint;
   n, limit, t      : Int;

   cmap_dir : TCMapDir;
   entry    : TCMapDirEntry;
   cmap     : PCMapTable;
 label
   Fail;
 begin

   Load_TrueType_CMap := Failure;

   {$IFDEF DEBUG} Write('CMaps '); {$ENDIF}

   t := LookUp_Mandatory_Table( face,'cmap' );
   if t < 0 then exit;

   with face^ do
   begin

     table_start := dirTables^[t].offset;

     if TT_Seek_File( dirTables^[t].Offset ) or
        TT_Access_Frame( sizeof(TCMapDir) )  then exit;

     cmap_dir.tableVersionNumber := GET_UShort;
     cmap_dir.numCMaps           := GET_UShort;

     TT_Forget_Frame;

     off := TT_File_Pos;

     (* save space in face data for cmap tables *)
     numCMaps := cmap_dir.numCMaps;
     if Alloc( cMaps, numCMaps * sizeof(TCMapTable) ) then exit;

     for n := 0 to numCMaps-1 do
     begin

       if TT_Seek_File   ( off ) or
          TT_Access_Frame( 8 )   then goto Fail;

       cmap := @cMaps^[n];

       entry.platformID         := GET_UShort;
       entry.platformEncodingID := GET_UShort;
       entry.offset             := GET_Long;

       cmap^.loaded             := False;
       cmap^.platformID         := entry.platformID;
       cmap^.platformEncodingID := entry.platformEncodingID;

       TT_Forget_Frame;

       off := TT_File_Pos;

       if TT_Seek_File   ( table_start + entry.offset ) or
          TT_Access_Frame( 6 ) then goto Fail;

       cmap^.format  := Get_UShort;
       cmap^.length  := Get_UShort;
       cmap^.version := Get_UShort;

       TT_Forget_Frame;

       cmap^.offset := TT_File_Pos;

     end;  (* for n *)

   end;  (* with face^ *)

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_CMap := Success;
   exit;

 Fail:
   Free( face^.cMaps );
   Load_TrueType_CMap := Failure;
 end;


(*
 procedure Free_CMap_Table( var cmap : TCMapTable );
 begin
   if cmap.cmap0 <> nil then
     with cmap do
       case format of

         0 : begin
               Free( cmap0^.glyphIdArray );
               Free( cmap0 );
             end;

         2 : begin
               Free( cmap2^.glyphIdArray );
               Free( cmap2^.subHeaders );
               Free( cmap2 );
             end;

         4 : begin
               Free( cmap4^.glyphIdArray );
               Free( cmap4^.segments );
               Free( cmap4 );
             end;

         6 : begin
               Free( cmap6^.glyphIdArray );
               Free( cmap6 );
             end;
       end;

   cmap.format  := 0;
   cmap.length  := 0;
   cmap.version := 0;
 end;
*)

(*******************************************************************
 *
 *  Function    :  Load_TrueType_HMTX
 *
 *  Description :
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_HMTX( face : PFace ) : TError;
 var
   t, n         : int;
   nsmetrics,
   nhmetrics : integer;
 begin
   Load_trueType_HMTX := Failure;

   {$IFDEF DEBUG} Write('HMTX '); {$ENDIF}

   t := LookUp_Mandatory_Table(face,'hmtx');
   if t < 0 then exit;

   with face^ do
   begin

     nhmetrics := horizontalHeader.number_Of_HMetrics;
     nsmetrics := MaxProfile.numGlyphs - nhmetrics;

     if Alloc( LongHMetrics,
               sizeof(TLongHorMetric)*nhmetrics ) or

        Alloc( ShortMetrics,
               sizeof(Short)*nsmetrics )          or

        TT_Seek_File( dirTables^[t].Offset )      or

        TT_Access_Frame( dirTables^[t].Length )   then exit;

     for n := 0 to nhmetrics-1 do with LongHMetrics^[n] do
     begin
       advance_width := GET_Short;
       lsb           := GET_Short;
     end;

     for n := 0 to nsmetrics-1 do
       ShortMetrics^[n] := GET_Short;

     TT_Forget_Frame;
   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_HMTX := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Programs
 *
 *  Description :  Load the Font and CVT programs in the resident
 *                 table
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Programs( face : PFace ) : TError;
 var
   t : Int;
 label
   Fail_File, Fail_Memory;
 begin

   Load_TrueType_Programs := Failure;

   {$IFDEF DEBUG} Write('Font program '); {$ENDIF}

   (* The font program is optional *)

   t := Lookup_TrueType_Table( face, 'fpgm' );

   if t < 0 then

     with face^ do
     begin
       fontProgram := nil;
       fontPgmSize := 0;

       {$IFDEF DEBUG} Writeln('none in file'); {$ENDIF}
     end

   else

     with face^ do
     begin

       fontPgmSize := dirTables^[t].Length;

       if Alloc( fontProgram, fontPgmSize ) or
          TT_Read_At_File( dirTables^[t].offset,
                           fontProgram^,
                           fontPgmSize ) then exit;

       {$IFDEF DEBUG} Writeln('loaded, ',fontPgmSize,' bytes'); {$ENDIF}
     end;

   {$IFDEF DEBUG} Write('CVT program '); {$ENDIF}

   t := LookUp_trueType_Table( face, 'prep' );

   (* The CVT table is optional *)

   if t < 0 then

     with face^ do
     begin
       cvtProgram := nil;
       cvtPgmSize := 0;

       {$IFDEF DEBUG} Writeln('none in file'); {$ENDIF}
     end

   else

     with face^ do
     begin

       cvtPgmSize := dirTables^[t].Length;

       if Alloc( cvtProgram, cvtPgmSize ) or
          TT_Read_At_File( dirTables^[t].offset,
                           cvtProgram^,
                           cvtPgmSize ) then exit;

       {$IFDEF DEBUG} Writeln('loaded, ',cvtPgmSize,' bytes'); {$ENDIF}
     end;

   Load_TrueType_Programs := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_OS2
 *
 *  Description :  Load the OS2 Table
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_OS2( face : PFace ) : TError;
 var
   table : Int;
   i     : Int;
 begin
   Load_TrueType_OS2 := Failure;

   {$IFDEF DEBUG} Write('OS/2 table '); {$ENDIF}

   table := LookUp_Mandatory_Table( face, 'OS/2' );
   if table < 0 then exit;

   if TT_Seek_File( face^.dirTables^[table].offset ) or
      TT_Access_Frame( 78 ) then exit;

   with face^.os2 do
   begin
     version             := Get_UShort;
     xAvgCharWidth       := Get_Short;
     usWeightClass       := Get_UShort;
     usWidthClass        := Get_UShort;
     fsType              := Get_Short;
     ySubscriptXSize     := Get_Short;
     ySubscriptYSize     := Get_Short;
     ySubscriptXOffset   := Get_Short;
     ySubscriptYOffset   := Get_Short;
     ySuperscriptXSize   := Get_Short;
     ySuperscriptYSize   := Get_Short;
     ySuperscriptXOffset := Get_Short;
     ySuperscriptYOffset := Get_Short;
     yStrikeoutSize      := Get_Short;
     yStrikeoutPosition  := Get_Short;
     sFamilyClass        := Get_Short;

     for i := 0 to 9 do panose[i] := Get_Byte;

     ulUnicodeRange1 := Get_ULong;
     ulUnicodeRange2 := Get_ULong;
     ulUnicodeRange3 := Get_ULong;
     ulUnicodeRange4 := Get_ULong;

     for i := 0 to 3 do achVendID[i] := Get_Byte;

     fsSelection      := Get_UShort;
     usFirstCharIndex := Get_UShort;
     usLastCharIndex  := Get_UShort;
     sTypoAscender    := Get_UShort;
     sTypoDescender   := Get_UShort;
     sTypoLineGap     := Get_UShort;
     usWinAscent      := Get_UShort;
     usWinDescent     := Get_UShort;

     TT_Forget_Frame;

     if version = $0001 then
       begin
         if TT_Access_Frame(8) then exit;

         ulCodePageRange1 := Get_ULong;
         ulCodePageRange2 := Get_ULong;

         TT_Forget_Frame;
       end
     else
       begin
         ulCodePageRange1 := 0;
         ulCodePageRange2 := 0;
       end;

   end;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_TrueType_OS2 := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Postscript
 *
 *  Description :  Load the 'post' table
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_trueType_Postscript( face : PFace ) : TError;
 var
   table : Int;
   i     : Int;
 begin
   Load_TrueType_Postscript := Failure;

   {$IFDEF DEBUG} Write('post table '); {$ENDIF}

   table := LookUp_TrueType_Table( face, 'post' );
   if table < 0 then exit;

   if TT_Seek_File( face^.dirTables^[table].offset ) or
      TT_Access_Frame(32) then exit;

   with face^.postscript do
   begin
     formatType         := Get_ULong;
     italicAngle        := Get_ULong;
     underlinePosition  := Get_Short;
     underlineThickness := Get_UShort;
     isFixedPitch       := Get_ULong;
     minMemType42       := Get_ULong;
     maxMemType42       := Get_ULong;
     minMemType1        := Get_ULong;
     maxMemType1        := Get_ULong;
   end;

   TT_Forget_Frame;

   {$IFDEF DEBUG} Writeln('loaded'); {$ENDIF}

   Load_trueType_Postscript := Success;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_HDMX
 *
 *  Description :  Load the 'hdmx' tables
 *
 *  Input  :  face
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Hdmx( face : PFace ) : TError;
 var
   table, n   : Int;
   num_glyphs : Int;

   version  : UShort;
   num_rec  : Short;
   recs     : PHdmx_Records;
   rec_size : Long;
   rec      : PHdmx_Record;
 label
   Fail;
 begin
   Load_TrueType_Hdmx := Failure;

   with face^.hdmx do
   begin
     version     := 0;
     num_records := 0;
     records     := nil;
   end;

   (* This table is optional *)

   table := LookUp_TrueType_Table( face, 'hdmx' );
   if table < 0 then
   begin
     Load_TrueType_Hdmx := Success;
     exit;
   end;

   if TT_Seek_File( face^.dirTables^[table].offset ) or
      TT_Access_Frame( 8 ) then exit;

   version  := Get_UShort;
   num_rec  := Get_Short;
   rec_size := Get_Long;

   TT_Forget_Frame;

   (* right now, we only recognize format 0 *)

   if version <> 0 then
     exit;

   if Alloc( face^.hdmx.records, sizeof(THdmx_Record)*num_rec ) then
     exit;

   face^.hdmx.num_records := num_rec;
   num_glyphs := face^.NumGlyphs;

   rec_size := rec_size - num_glyphs - 2;

   for n := 0 to num_rec-1 do
   begin
     rec := @face^.hdmx.records^[n];

     (* read record *)

     if TT_Access_Frame(2) then
       goto Fail;

     rec^.ppem      := Get_Byte;
     rec^.max_width := Get_Byte;

     TT_Forget_Frame;

     if Alloc( rec^.widths, num_glyphs ) or
        TT_Read_File( rec^.widths^, num_glyphs ) then
       goto Fail;

     (* skip padding bytes *)

     if rec_size > 0 then
       if TT_Skip_File( rec_size ) then
         goto Fail;
   end;

   Load_TrueType_HDMX := Success;
   exit;

 Fail:
   for n := 0 to num_rec-1 do
    Free( face^.hdmx.records^[n].widths );

   Free( face^.hdmx.records );
   face^.hdmx.num_records := 0;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Any
 *
 *  Description :  Load any TrueType table in user memory
 *
 *  Input  :  face    the font file's face object
 *            tag     the table
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Any( face        : PFace;
                             tag         : longint;
                             offset      : longint;
                             var buffer;
                             var length  : longint ) : TError;
 var
   stream   : TT_Stream;
   found, i : integer;
 begin
   if tag <> 0 then
     begin
       found := -1;
       i     := 0;
       while i < face^.numTables do
         if Longint(face^.dirTables^[i].tag) = tag then
           begin
             found := i;
             i := face^.numTables;
           end
         else
           inc(i);

       if found < 0 then
         begin
           error := TT_Err_Table_Missing;
           Load_TrueType_Any := Failure;
           exit;
         end;

       inc( offset, face^.dirTables^[found].offset );

       (* if length = 0, the user requested the table's size *)
       if length = 0 then
         begin
           length := face^.dirTables^[found].length;
           Load_TrueType_Any := Success;
           exit;
         end;
     end
   else
     (* if length = 0 and tag = 0, the user requested the font file's size *)
     if length = 0 then
       begin
         (* return length of font file *)
         length := TT_Stream_Size( face^.stream );
         Load_TrueType_Any := Success;
         exit;
       end;

   TT_Use_Stream( face^.stream, stream );
   Load_TrueType_Any := TT_Read_At_File( offset, buffer, length );
   TT_Done_Stream( face^.stream );
 end;

end.

