/*
 *   ttf2pk.c
 *
 *   This file is part of the ttf2pk package.
 *
 *   Copyright 1997-1998 by
 *     Frederic Loyer <loyer@ensta.fr>
 *     Werner Lemberg <wl@gnu.org>
 */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>         /* for size_t */
#include <string.h>
#include <math.h>
#include <ctype.h>


#include "ttf2tfm.h"
#include "newobj.h"
#include "pklib.h"
#include "ttfenc.h"
#include "ttflib.h"
#include "errormsg.h"
#include "filesrch.h"
#include "parse.h"
#include "subfont.h"


char ident[] = "ttf2pk version 1.0";
char progname[] = "ttf2pk";             /* for error/warning messages */


Boolean have_sfd = False;


/*
 *   Checks for an equal sign surrounded by whitespace.
 */

static char *
strip_equal(char *s, char *os, char *p)
{
  while (isspace(*p))
    p++;
  if (*p != '=')
    boops(os, p - s, "Missing `='.");
  p++;
  while (isspace(*p))
    p++;
  return p;
}


#define USAGE "\
  Convert a TrueType font to TeX's PK format.\n\
\n\
-q                  suppress informational output\n\
--help              print this message and exit\n\
--version           print version number and exit\n\
"

static void
usage(void)
{
  fputs("Usage: ttf2pk [-q] <font> <dpi>\n", stdout);
  fputs(USAGE, stdout);
  exit(0);
}


#define VERSION "\
Copyright (C) 1997-1998 Frederic Loyer and Werner Lemberg.\n\
There is NO warranty.  You may redistribute this software\n\
under the terms of the GNU General Public License\n\
and the gsftopk copyright.\n\
\n\
For more information about these matters, see the files\n\
named COPYING and pklib.c.\n\
\n\
Primary authors of ttf2pk: F. Loyer and W. Lemberg.\n\
\n\
ttf2pk is partially based on gsftopk from P. Vojta\n\
and the FreeType project from\n\
David Turner, Robert Wilhelm, and Werner Lemberg\n\
"

static void
version(void)
{
  fputs(ident, stdout);
  fprintf(stdout, " (%s)\n", TeX_search_version());
  fputs(VERSION, stdout);
  exit(0);
}


static int
compare(Font *fnt, char *s, char *key)
{
  char c;
  char *p;
  char *temp, *temp1;
  int value;
  int sfd_begin, postfix_begin;


  /*
   *   We isolate the fontname.
   */

  while (isspace(*s))
    s++;

  p = s;
  while (*p && !isspace(*p))
    p++;

  c = *p;
  *p = '\0';
  temp = newstring(s);
  *p = c;

  /*
   *   We search for a subfont definition file name.
   */

  handle_sfd(temp, &sfd_begin, &postfix_begin);

  if (sfd_begin == -1)
    value = strcmp(temp, key);
  else
  {
    size_t len, len1, len2;


    /*
     *   The sfd file will be only searched if prefix and postfix match.
     */

    len = strlen(key);
    len1 = strlen(temp);
    len2 = strlen(temp + postfix_begin);

    if (len1 + len2 >= len)
      value = -1;
    else if (!strncmp(temp, key, len1) &&
             !strcmp(temp + postfix_begin, key + (len - len2)))
    {
      c = key[len - len2];
      key[len - len2] = '\0';
      temp1 = newstring(key + len1);
      key[len - len2] = c;

      if (fnt->sfdname)
        free(fnt->sfdname);
      fnt->sfdname = newstring(temp + sfd_begin);
      handle_extension(&(fnt->sfdname), ".sfd");

      /*
       *   If the sfd file can't be opened the search is continued.
       */

      value = !init_sfd(fnt, False);

      if (!value)
      {
        value = -1;

        while (get_sfd(fnt))
        {
          if (!strcmp(fnt->subfont_name, temp1))
          {
            value = 0;                              /* success */
            have_sfd = True;
            break;
          }
        }

        close_sfd();
      }

      free(temp1);
    }
    else
      value = -1;
  }

  free(temp);

  return value;
}


int 
main(int argc, char** argv)
{
  size_t i;
  long code;
  FILE *config_file;
  char *configline, *oldconfigline, *p, *q;
  Font font;
  encoding *enc;
  long inenc_array[256];
  char *fontname;
  size_t fontname_len;
  char *pk_filename, *tfm_filename, *enc_filename;
  char *real_ttfname;
  int dpi, ptsize;
  int slant, extend;
  int pid = 3, eid = 1;
  long fontindex = 0;
  Boolean hinting = True;
  Boolean quiet = False;


  TeX_search_init(argv[0], "ttf2pk", "TTF2PK");

  if (argc == 1)
    oops("Need at least two arguments.\n"
         "Try `ttf2pk --help' for more information.");

  if (argc == 2)
  {
    if (strcmp(argv[1], "--help") == 0)
      usage();
    else if (strcmp(argv[1], "--version") == 0)
      version();
  }

  if (argv[1][0] == '-')
  {
    if (argv[1][1] == 'q')
      quiet = True;
    else
      oops("Unknown option `%s'.\n"
           "Try `ttf2pk --help' for more information.", argv[1]);

    argv++;
    argc--;
  }

  if (argc != 3)
    oops("Need at most two arguments.\n"
         "Try `ttf2pk --help' for more information.");

  if ((dpi = atoi(argv[2])) <= 50)
    oops("dpi value must be larger than 50.");

  fontname = argv[1];
  fontname_len = strlen(fontname);
  enc_filename = NULL;

  ptsize = 10;
  slant = 0;
  extend = 1024;

  init_font_structure(&font);

  config_file = fopen(TeX_search_config_file("ttfonts.map"), "rt");
  
  if (config_file == NULL) 
    oops("Cannot find file ttfonts.map.");

  do
  {
    configline = get_line(config_file);
    if (!configline)
      oops("Error while reading ttfonts.map.");
    if (!*configline)
    {
      /*
       *   This is the only error message we suppress if the -q flag
       *   is set, making it possible to call ttf2pk silently.
       */
      if (!quiet)
        fprintf(stdout,
                "%s: ERROR: Cannot find font %s in ttfonts.map.\n",
                progname, fontname);
      exit(2);
    }
  } while (compare(&font, configline, fontname));

  fclose(config_file);

  /*
   *   Parse the line from the config file.  We split the config line buffer
   *   into substrings according to the given options.
   */

  i = strlen(configline);
  configline[i - 1] = '\0';     /* strip newline */

  oldconfigline = newstring(configline);

  p = configline;
  while (isspace(*p))
    p++;
  while (*p && !isspace(*p))
    p++;

  q = p;

  while (*p && isspace(*p))
    p++;
  if (!*p)
    boops(oldconfigline, q - configline, "TTF file missing.");

  font.ttfname = p;

  while (*p && !isspace(*p))
    p++;
  if (*p)
    *p++ = '\0';      

  for (; *p; p++)
  {
    if (isspace(*p))
      continue;

    if (!strncmp(p, "Slant", 5))
    {
      float s;


      p = strip_equal(configline, oldconfigline, p + 5);
      if (sscanf(p, "%f", &s) == 0)
        boops(oldconfigline, p - configline, "Bad `Slant' parameter.");
      slant = floor(s * 1024);
    }
    else if (!strncmp(p, "Encoding", 8))
    {
      if (have_sfd)
        boops(oldconfigline, p - configline,
              "No `Encoding' parameter allowed for subfonts.");

      p = strip_equal(configline, oldconfigline, p + 8);
      if (!*p)
        boops(oldconfigline, p - configline, "Bad `Encoding' parameter.");
      enc_filename = p;
    }
    else if (!strncmp(p, "Extend", 6))
    {
      float e;


      p = strip_equal(configline, oldconfigline, p + 6);
      if (sscanf(p, "%f", &e) == 0)
        boops(oldconfigline, p - configline, "Bad `Extend' parameter.");
      extend = floor(e * 1024);
    }
    else if (!strncmp(p, "Fontindex", 9))
    {
      p = strip_equal(configline, oldconfigline, p + 9);
      if (sscanf(p, "%lu", &fontindex) < 0)
        boops(oldconfigline, p - configline, "Bad `Fontindex' parameter.");
    }
    else if (!strncmp(p, "Pid", 3))
    {
      p = strip_equal(configline, oldconfigline, p + 3);
      if (sscanf(p, "%u", &pid) < 0)
        boops(oldconfigline, p - configline, "Bad `Pid' parameter.");
    }
    else if (!strncmp(p, "Eid", 3))
    {
      p = strip_equal(configline, oldconfigline, p + 3);
      if (sscanf(p, "%u", &eid) < 0)
        boops(oldconfigline, p - configline, "Bad `Eid' parameter.");
    }
    else if (!strncmp(p, "Hinting", 7))
    {
      p = strip_equal(configline, oldconfigline, p + 7);
      if (p[1] == 'N' || p[1] == 'n' || p[0] == '1')
        hinting = True;
      else if (p[1] == 'F' || p[1] == 'f' || p[0] == '0')
        hinting = False;
      else
        boops(oldconfigline, p - configline, "Bad `Hinting' parameter.");
    }
    else
      boops(oldconfigline, p - configline, "Bad parameter.");

    while (*p && !isspace(*p))
      p++;
    if (*p)
      *p = '\0';
  }

  tfm_filename = mymalloc(fontname_len + 5);
  sprintf(tfm_filename, "%s.tfm", fontname);
  TFMopen(tfm_filename);

  pk_filename = mymalloc(fontname_len + 10);
  sprintf(pk_filename, "%s.%dpk", fontname, dpi);
  PKopen(pk_filename, fontname);

  real_ttfname = TeX_search_ttf_file(font.ttfname);
  if (!real_ttfname)
    oops("Cannot find `%s'.", font.ttfname);
  TTFopen(real_ttfname, dpi, ptsize, extend, slant,
          fontindex, pid, eid, quiet);

  enc = readencoding(enc_filename, &font, True);
  if (enc)
  {
    char *name;


    for (i = 0; i <= 0xFF; i++)
    {
      name = enc->vec[i];
      code = adobename_to_code(name);
      if (code < 0 && strcmp(name, ".notdef") != 0)
        warning("Cannot map character `%s'.", name);
      inenc_array[i] = code;
    }
  }
  else
  {
    if (have_sfd)
      TTFget_subfont(&font, inenc_array);
    else
      TTFget_first_glyphs(inenc_array);
  }

  for (i = 0; i <= 0xFF; i++)
  {
    byte *bitmap;
    int w, h, hoff, voff;


    if ((code = inenc_array[i]) >= 0)
    {
      if (!quiet)
      {
        printf("Processing %3d %s %04x %s\n",
               i, (code >= 0x10000) ? "glyph" : "code",
               (unsigned int)(code & 0xFFFF), enc ? enc->vec[i] : "");
        fflush(stdout);
      }

      TTFprocess(code, &bitmap, &w, &h, &hoff, &voff, hinting, quiet);

      PKputglyph(i,
                 -hoff, -voff, w - hoff, h - voff,
                 w, h, bitmap);
    }
  }

  PKclose();
  exit(0);      /* for safety reasons */
  return 0;     /* never reached */
}


/* end */
