/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * GimpRc, the object for GIMPs user configuration file gimprc.
 * Copyright (C) 2001-2002  Sven Neumann <sven@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <glib-object.h>

#ifdef G_OS_WIN32
#include <io.h>
#endif

#include "libgimpbase/gimpbase.h"

#include "config-types.h"

#include "gimpconfig.h"
#include "gimpconfig-deserialize.h"
#include "gimpconfig-error.h"
#include "gimpconfig-params.h"
#include "gimpconfig-path.h"
#include "gimpconfig-serialize.h"
#include "gimpconfig-utils.h"
#include "gimprc.h"

#include "gimp-intl.h"


enum
{
  PROP_0,
  PROP_VERBOSE,
  PROP_SYSTEM_GIMPRC,
  PROP_USER_GIMPRC
};


static void         gimp_rc_class_init        (GimpRcClass      *klass);
static void         gimp_rc_config_iface_init (gpointer          iface,
                                               gpointer          iface_data);
static void         gimp_rc_init              (GimpRc           *rc);
static void         gimp_rc_dispose           (GObject          *object);
static void         gimp_rc_finalize          (GObject          *object);
static void         gimp_rc_set_property      (GObject          *object,
                                               guint             property_id,
                                               const GValue     *value,
                                               GParamSpec       *pspec);
static void         gimp_rc_get_property      (GObject          *object,
                                               guint             property_id,
                                               GValue           *value,
                                               GParamSpec       *pspec);
static gboolean     gimp_rc_serialize         (GimpConfig       *object,
                                               GimpConfigWriter *writer,
                                               gpointer          data);
static gboolean     gimp_rc_deserialize       (GimpConfig       *object,
                                               GScanner         *scanner,
                                               gint              nest_level,
                                               gpointer          data);
static GimpConfig * gimp_rc_duplicate         (GimpConfig       *object);
static void         gimp_rc_load              (GimpRc           *rc);
static gboolean     gimp_rc_idle_save         (GimpRc           *rc);
static void         gimp_rc_notify            (GimpRc           *rc,
                                               GParamSpec       *param,
                                               gpointer          data);


static GObjectClass *parent_class = NULL;


GType
gimp_rc_get_type (void)
{
  static GType rc_type = 0;

  if (! rc_type)
    {
      static const GTypeInfo rc_info =
      {
        sizeof (GimpRcClass),
	NULL,           /* base_init      */
        NULL,           /* base_finalize  */
        (GClassInitFunc) gimp_rc_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data     */
	sizeof (GimpRc),
	0,              /* n_preallocs    */
	(GInstanceInitFunc) gimp_rc_init
      };
      static const GInterfaceInfo rc_iface_info =
      {
        gimp_rc_config_iface_init,
        NULL,           /* iface_finalize */
        NULL            /* iface_data     */
      };

      rc_type = g_type_register_static (GIMP_TYPE_PLUGIN_CONFIG,
                                        "GimpRc", &rc_info, 0);

      g_type_add_interface_static (rc_type, GIMP_TYPE_CONFIG, &rc_iface_info);
    }

  return rc_type;
}

static void
gimp_rc_class_init (GimpRcClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);

  object_class = G_OBJECT_CLASS (klass);

  object_class->dispose      = gimp_rc_dispose;
  object_class->finalize     = gimp_rc_finalize;
  object_class->set_property = gimp_rc_set_property;
  object_class->get_property = gimp_rc_get_property;

  g_object_class_install_property (object_class, PROP_VERBOSE,
				   g_param_spec_boolean ("verbose",
							 NULL, NULL,
							 FALSE,
							 G_PARAM_READWRITE |
							 G_PARAM_CONSTRUCT));
  g_object_class_install_property (object_class, PROP_SYSTEM_GIMPRC,
				   g_param_spec_string ("system-gimprc",
                                                        NULL, NULL,
							NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT));
  g_object_class_install_property (object_class, PROP_USER_GIMPRC,
				   g_param_spec_string ("user-gimprc",
                                                        NULL, NULL,
							NULL,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT));
}

static void
gimp_rc_init (GimpRc *rc)
{
  rc->autosave     = FALSE;
  rc->save_idle_id = 0;
}

static void
gimp_rc_dispose (GObject *object)
{
  GimpRc *rc = GIMP_RC (object);

  if (rc->save_idle_id)
    gimp_rc_idle_save (rc);
}

static void
gimp_rc_finalize (GObject *object)
{
  GimpRc *rc = (GimpRc *) object;

  if (rc->system_gimprc)
    {
      g_free (rc->system_gimprc);
      rc->system_gimprc = NULL;
    }
  if (rc->user_gimprc)
    {
      g_free (rc->user_gimprc);
      rc->user_gimprc = NULL;
    }

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gimp_rc_set_property (GObject      *object,
                      guint         property_id,
                      const GValue *value,
                      GParamSpec   *pspec)
{
  GimpRc      *rc       = GIMP_RC (object);
  const gchar *filename = NULL;

  switch (property_id)
    {
    case PROP_SYSTEM_GIMPRC:
    case PROP_USER_GIMPRC:
      filename = g_value_get_string (value);
      break;
    default:
      break;
    }

  switch (property_id)
    {
    case PROP_VERBOSE:
      rc->verbose = g_value_get_boolean (value);
      break;

    case PROP_SYSTEM_GIMPRC:
      g_free (rc->system_gimprc);

      if (filename)
        rc->system_gimprc = g_strdup (filename);
      else
        rc->system_gimprc = g_build_filename (gimp_sysconf_directory (),
                                              "gimprc", NULL);
      break;
    case PROP_USER_GIMPRC:
      g_free (rc->user_gimprc);

      if (filename)
        rc->user_gimprc = g_strdup (filename);
      else
        rc->user_gimprc = gimp_personal_rc_file ("gimprc");
      break;

   default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
gimp_rc_get_property (GObject    *object,
                      guint       property_id,
                      GValue     *value,
                      GParamSpec *pspec)
{
  GimpRc *rc = GIMP_RC (object);

  switch (property_id)
    {
    case PROP_VERBOSE:
      g_value_set_boolean (value, rc->verbose);
      break;
    case PROP_SYSTEM_GIMPRC:
      g_value_set_string (value, rc->system_gimprc);
      break;
    case PROP_USER_GIMPRC:
      g_value_set_string (value, rc->user_gimprc);
      break;
   default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
gimp_rc_config_iface_init (gpointer  iface,
                           gpointer  iface_data)
{
  GimpConfigInterface *config_iface = (GimpConfigInterface *) iface;

  config_iface->serialize   = gimp_rc_serialize;
  config_iface->deserialize = gimp_rc_deserialize;
  config_iface->duplicate   = gimp_rc_duplicate;
}

static gboolean
gimp_rc_serialize (GimpConfig       *config,
                   GimpConfigWriter *writer,
                   gpointer          data)
{
  if (data && GIMP_IS_RC (data))
    {
      if (! gimp_config_serialize_properties_diff (config, GIMP_CONFIG (data),
                                                   writer))
        return FALSE;
    }
  else
    {
      if (! gimp_config_serialize_properties (config, writer))
        return FALSE;
    }

  return gimp_config_serialize_unknown_tokens (config, writer);
}

static gboolean
gimp_rc_deserialize (GimpConfig *config,
                     GScanner   *scanner,
                     gint        nest_level,
                     gpointer    data)
{
  return gimp_config_deserialize_properties (config,
                                             scanner, nest_level, TRUE);
}

static void
gimp_rc_duplicate_unknown_token (const gchar *key,
                                 const gchar *value,
                                 gpointer     user_data)
{
  gimp_config_add_unknown_token (GIMP_CONFIG (user_data), key, value);
}

static GimpConfig *
gimp_rc_duplicate (GimpConfig *config)
{
  GimpConfig *dup = g_object_new (GIMP_TYPE_RC, NULL);

  gimp_config_sync (config, dup, 0);

  gimp_config_foreach_unknown_token (config,
                                     gimp_rc_duplicate_unknown_token, dup);

  return dup;
}

static void
gimp_rc_load (GimpRc *rc)
{
  GError *error = NULL;

  g_return_if_fail (GIMP_IS_RC (rc));

  if (rc->verbose)
    g_print (_("Parsing '%s'\n"), rc->system_gimprc);

  if (! gimp_config_deserialize_file (GIMP_CONFIG (rc),
				      rc->system_gimprc, NULL, &error))
    {
      if (error->code != GIMP_CONFIG_ERROR_OPEN_ENOENT)
	g_message (error->message);

      g_clear_error (&error);
    }

  if (rc->verbose)
    g_print (_("Parsing '%s'\n"), rc->user_gimprc);

  if (! gimp_config_deserialize_file (GIMP_CONFIG (rc),
				      rc->user_gimprc, NULL, &error))
    {
      if (error->code != GIMP_CONFIG_ERROR_OPEN_ENOENT)
        {
          g_message (error->message);

          gimp_config_file_backup_on_error (rc->user_gimprc, "gimprc", NULL);
        }

      g_clear_error (&error);
    }
}

static gboolean
gimp_rc_idle_save (GimpRc *rc)
{
  gimp_rc_save (rc);

  rc->save_idle_id = 0;

  return FALSE;
}

static void
gimp_rc_notify (GimpRc     *rc,
                GParamSpec *param,
                gpointer    data)
{
  if (!rc->autosave)
    return;

  if (!rc->save_idle_id)
    rc->save_idle_id = g_idle_add ((GSourceFunc) gimp_rc_idle_save, rc);
}

/**
 * gimp_rc_new:
 * @system_gimprc: the name of the system-wide gimprc file or %NULL to
 *                 use the standard location
 * @user_gimprc:   the name of the user gimprc file or %NULL to use the
 *                 standard location
 * @verbose:       enable console messages about loading and saving
 *
 * Creates a new GimpRc object and loads the system-wide and the user
 * configuration files.
 *
 * Returns: the new #GimpRc.
 */
GimpRc *
gimp_rc_new (const gchar *system_gimprc,
             const gchar *user_gimprc,
             gboolean     verbose)
{
  GimpRc *rc = g_object_new (GIMP_TYPE_RC,
                             "verbose",       verbose,
                             "system-gimprc", system_gimprc,
                             "user-gimprc",   user_gimprc,
                             NULL);

  gimp_rc_load (rc);

  return rc;
}

void
gimp_rc_set_autosave (GimpRc   *rc,
                      gboolean  autosave)
{
  g_return_if_fail (GIMP_IS_RC (rc));

  autosave = autosave ? TRUE : FALSE;

  if (rc->autosave == autosave)
    return;

  if (autosave)
    g_signal_connect (rc, "notify",
                      G_CALLBACK (gimp_rc_notify),
                      NULL);
  else
    g_signal_handlers_disconnect_by_func (rc, gimp_rc_notify, NULL);

  rc->autosave = autosave;
}


/**
 * gimp_rc_query:
 * @rc:  a #GimpRc object.
 * @key: a string used as a key for the lookup.
 *
 * This function looks up @key in the object properties of @rc. If
 * there's a matching property, a string representation of its value
 * is returned. If no property is found, the list of unknown tokens
 * attached to the @rc object is searched.
 *
 * Return value: a newly allocated string representing the value or %NULL
 *               if the key couldn't be found.
 **/
gchar *
gimp_rc_query (GimpRc      *rc,
               const gchar *key)
{
  GObjectClass  *klass;
  GObject       *rc_object;
  GParamSpec   **property_specs;
  GParamSpec    *prop_spec;
  guint          i, n_property_specs;
  gchar         *retval = NULL;

  g_return_val_if_fail (GIMP_IS_RC (rc), NULL);
  g_return_val_if_fail (key != NULL, NULL);

  rc_object = G_OBJECT (rc);
  klass = G_OBJECT_GET_CLASS (rc);

  property_specs = g_object_class_list_properties (klass, &n_property_specs);

  if (!property_specs)
    return NULL;

  for (i = 0, prop_spec = NULL; i < n_property_specs && !prop_spec; i++)
    {
      prop_spec = property_specs[i];

      if (! (prop_spec->flags & GIMP_PARAM_SERIALIZE) ||
          strcmp (prop_spec->name, key))
        {
          prop_spec = NULL;
        }
    }

  if (prop_spec)
    {
      GString *str   = g_string_new (NULL);
      GValue   value = { 0, };

      g_value_init (&value, prop_spec->value_type);
      g_object_get_property (rc_object, prop_spec->name, &value);

      if (gimp_config_serialize_value (&value, str, FALSE))
        retval = g_string_free (str, FALSE);
      else
        g_string_free (str, TRUE);

      g_value_unset (&value);
    }
  else
    {
      retval = g_strdup (gimp_config_lookup_unknown_token (GIMP_CONFIG (rc),
                                                           key));
    }

  g_free (property_specs);

  if (!retval)
    {
      const gchar *path_tokens[] =
      {
        "gimp_dir",
        "gimp_data_dir",
        "gimp_plug_in_dir",
        "gimp_plugin_dir",
        "gimp_sysconf_dir"
      };

      for (i = 0; !retval && i < G_N_ELEMENTS (path_tokens); i++)
        if (strcmp (key, path_tokens[i]) == 0)
          retval = g_strdup_printf ("${%s}", path_tokens[i]);
    }

  if (retval)
    {
      gchar *tmp = gimp_config_path_expand (retval, FALSE, NULL);

      if (tmp)
        {
          g_free (retval);
          retval = tmp;
        }
    }

  return retval;
}

/**
 * gimp_rc_save:
 * @gimprc: a #GimpRc object.
 *
 * Saves any settings that differ from the system-wide defined
 * defaults to the users personal gimprc file.
 **/
void
gimp_rc_save (GimpRc *rc)
{
  GimpRc *global;
  gchar  *header;
  GError *error = NULL;

  const gchar *top =
    "GIMP gimprc\n"
    "\n"
    "This is your personal gimprc file.  Any variable defined in this file "
    "takes precedence over the value defined in the system-wide gimprc: ";
  const gchar *bottom =
    "\n"
    "Most values can be set within The GIMP by changing some options in "
    "the Preferences dialog.";
  const gchar *footer =
    "end of gimprc";

  g_return_if_fail (GIMP_IS_RC (rc));

  global = g_object_new (GIMP_TYPE_RC, NULL);

  gimp_config_deserialize_file (GIMP_CONFIG (global),
				rc->system_gimprc, NULL, NULL);

  header = g_strconcat (top, rc->system_gimprc, bottom, NULL);

  if (rc->verbose)
    g_print (_("Saving '%s'\n"), rc->user_gimprc);

  if (! gimp_config_serialize_to_file (GIMP_CONFIG (rc),
				       rc->user_gimprc,
				       header, footer, global,
				       &error))
    {
      g_message (error->message);
      g_error_free (error);
    }

  g_free (header);
  g_object_unref (global);
}
