package Mail::SpamAssassin::Plugin::RedisAWL;

use Mail::SpamAssassin::Plugin;
use Mail::SpamAssassin::Logger;
use strict;
use warnings;

# ABSTRACT: redis support for spamassassin AWL/TxRep
our $VERSION = '1.000'; # VERSION

use vars qw(@ISA);
@ISA = qw(Mail::SpamAssassin::Plugin);


sub new {
    my $class = shift;
    my $mailsaobject = shift;

    $class = ref($class) || $class;
    my $self = $class->SUPER::new($mailsaobject);
    bless ($self, $class);

    $mailsaobject->{conf}->{parser}->register_commands( [
        {
            setting => 'auto_whitelist_redis_server',
            type => $Mail::SpamAssassin::Conf::CONF_TYPE_STRING,
            default => '127.0.0.1:6379',
        }, {
            setting => 'auto_whitelist_redis_prefix',
            type => $Mail::SpamAssassin::Conf::CONF_TYPE_STRING,
            default => 'awl_',
        },
    ] );

    return( $self );
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Mail::SpamAssassin::Plugin::RedisAWL - redis support for spamassassin AWL/TxRep

=head1 VERSION

version 1.000

=head1 DESCRIPTION

RedisAWL implements a redis table for the spamassassin AWL/TxRep plugin.

=head1 INSTALLATION

To install the plugin from CPAN:

  $ cpanm Mail::SpamAssassin::Plugin::RedisAWL

=head1 CONFIGURATION

Load the plugin in init.pre:

  loadplugin Mail::SpamAssassin::Plugin::RedisAWL

Change the datebase type to redis in local.cf:

  auto_whitelist_factory Mail::SpamAssassin::RedisAddrList

And you may have to configure the redis server and prefix:

  auto_whitelist_redis_server 127.0.0.1:6379
  auto_whitelist_redis_prefix awl_

=head1 PARAMETERS

=head2 auto_whitelist_redis_server

The redis server to use.

Default: 127.0.0.1:6379

=head2 auto_whitelist_redis_prefix

A prefix for AWL keys.

Default: awl_

=head1 CHECK YOUR CONFIGURATION

Scan a mail an check debug output for redis/awl messages:

  $ spamassassin -D < /tmp/testmail 2>&1 | egrep -e 'auto-whitelist' -e 'redis'

Query redis for awl data:

  $ redis-cli
  redis 127.0.0.1:6379> keys awl_*
  1)...
  redis 127.0.0.1:6379> get awl_bla@blub.de|ip=1.2.3.4_score
  "-5099"
  redis 127.0.0.1:6379> get awl_bla@blub.de|ip=1.2.3.4_count
  "1"

=head1 AUTHOR

Markus Benning <ich@markusbenning.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by Markus Benning.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
