# frozen_string_literal: true

=begin
This file is generated by the templates/template.rb script and should not be
modified manually. See templates/lib/prism/node.rb.erb
if you are looking to modify the template
=end

module Prism
  # This represents a node in the tree. It is the parent class of all of the
  # various node types.
  class Node
    # A pointer to the source that this node was created from.
    attr_reader :source
    private :source

    # A Location instance that represents the location of this node in the
    # source.
    def location
      location = @location
      return location if location.is_a?(Location)
      @location = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The start offset of the node in the source. This method is effectively a
    # delegate method to the location object.
    def start_offset
      location = @location
      location.is_a?(Location) ? location.start_offset : location >> 32
    end

    # The end offset of the node in the source. This method is effectively a
    # delegate method to the location object.
    def end_offset
      location = @location
      location.is_a?(Location) ? location.end_offset : ((location >> 32) + (location & 0xFFFFFFFF))
    end

    # Returns all of the lines of the source code associated with this node.
    def source_lines
      location.source_lines
    end

    # An alias for source_lines, used to mimic the API from
    # RubyVM::AbstractSyntaxTree to make it easier to migrate.
    alias script_lines source_lines

    # Slice the location of the node from the source.
    def slice
      location.slice
    end

    # Slice the location of the node from the source, starting at the beginning
    # of the line that the location starts on, ending at the end of the line
    # that the location ends on.
    def slice_lines
      location.slice_lines
    end

    # Similar to inspect, but respects the current level of indentation given by
    # the pretty print object.
    def pretty_print(q)
      q.seplist(inspect.chomp.each_line, -> { q.breakable }) do |line|
        q.text(line.chomp)
      end
      q.current_group.break
    end

    # Convert this node into a graphviz dot graph string.
    def to_dot
      # @type self: node
      DotVisitor.new.tap { |visitor| accept(visitor) }.to_dot
    end

    # Returns a list of nodes that are descendants of this node that contain the
    # given line and column. This is useful for locating a node that is selected
    # based on the line and column of the source code.
    #
    # Important to note is that the column given to this method should be in
    # bytes, as opposed to characters or code units.
    def tunnel(line, column)
      queue = [self] #: Array[Prism::node]
      result = []

      while (node = queue.shift)
        result << node

        node.compact_child_nodes.each do |child_node|
          child_location = child_node.location

          start_line = child_location.start_line
          end_line = child_location.end_line

          if start_line == end_line
            if line == start_line && column >= child_location.start_column && column < child_location.end_column
              queue << child_node
              break
            end
          elsif (line == start_line && column >= child_location.start_column) || (line == end_line && column < child_location.end_column)
            queue << child_node
            break
          elsif line > start_line && line < end_line
            queue << child_node
            break
          end
        end
      end

      result
    end

    # Returns a list of the fields that exist for this node class. Fields
    # describe the structure of the node. This kind of reflection is useful for
    # things like recursively visiting each node _and_ field in the tree.
    def self.fields
      # This method should only be called on subclasses of Node, not Node
      # itself.
      raise NoMethodError, "undefined method `fields' for #{inspect}" if self == Node

      Reflection.fields_for(self)
    end

    # --------------------------------------------------------------------------
    # :section: Node interface
    # These methods are effectively abstract methods that must be implemented by
    # the various subclasses of Node. They are here to make it easier to work
    # with typecheckers.
    # --------------------------------------------------------------------------

    # Accepts a visitor and calls back into the specialized visit function.
    def accept(visitor)
      raise NoMethodError, "undefined method `accept' for #{inspect}"
    end

    # Returns an array of child nodes, including `nil`s in the place of optional
    # nodes that were not present.
    def child_nodes
      raise NoMethodError, "undefined method `child_nodes' for #{inspect}"
    end

    alias deconstruct child_nodes

    # Returns an array of child nodes, excluding any `nil`s in the place of
    # optional nodes that were not present.
    def compact_child_nodes
      raise NoMethodError, "undefined method `compact_child_nodes' for #{inspect}"
    end

    # Returns an array of child nodes and locations that could potentially have
    # comments attached to them.
    def comment_targets
      raise NoMethodError, "undefined method `comment_targets' for #{inspect}"
    end

    # Returns a symbol symbolizing the type of node that this represents. This
    # is particularly useful for case statements and array comparisons.
    def type
      raise NoMethodError, "undefined method `type' for #{inspect}"
    end

    # Returns a string representation of the node.
    def inspect
      raise NoMethodError, "undefined method `inspect' for #{inspect}"
    end

    # Returns the type of the node as a symbol.
    def self.type
      raise NoMethodError, "undefined method `type' for #{inspect}"
    end
  end

  # Represents the use of the `alias` keyword to alias a global variable.
  #
  #     alias $foo $bar
  #     ^^^^^^^^^^^^^^^
  class AliasGlobalVariableNode < Node
    # def initialize: (Prism::node new_name, Prism::node old_name, Location keyword_loc, Location location) -> void
    def initialize(source, new_name, old_name, keyword_loc, location)
      @source = source
      @location = location
      @new_name = new_name
      @old_name = old_name
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_alias_global_variable_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [new_name, old_name]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [new_name, old_name]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [new_name, old_name, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?new_name: Prism::node, ?old_name: Prism::node, ?keyword_loc: Location, ?location: Location) -> AliasGlobalVariableNode
    def copy(new_name: self.new_name, old_name: self.old_name, keyword_loc: self.keyword_loc, location: self.location)
      AliasGlobalVariableNode.new(source, new_name, old_name, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { new_name: Prism::node, old_name: Prism::node, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { new_name: new_name, old_name: old_name, keyword_loc: keyword_loc, location: location }
    end

    # Represents the new name of the global variable that can be used after aliasing. This can be either a global variable, a back reference, or a numbered reference.
    #
    #     alias $foo $bar
    #           ^^^^
    attr_reader :new_name

    # Represents the old name of the global variable that could be used before aliasing. This can be either a global variable, a back reference, or a numbered reference.
    #
    #     alias $foo $bar
    #                ^^^^
    attr_reader :old_name

    # The location of the `alias` keyword.
    #
    #     alias $foo $bar
    #     ^^^^^
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :alias_global_variable_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :alias_global_variable_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AliasGlobalVariableNode) &&
        (new_name === other.new_name) &&
        (old_name === other.old_name) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents the use of the `alias` keyword to alias a method.
  #
  #     alias foo bar
  #     ^^^^^^^^^^^^^
  class AliasMethodNode < Node
    # def initialize: (Prism::node new_name, Prism::node old_name, Location keyword_loc, Location location) -> void
    def initialize(source, new_name, old_name, keyword_loc, location)
      @source = source
      @location = location
      @new_name = new_name
      @old_name = old_name
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_alias_method_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [new_name, old_name]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [new_name, old_name]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [new_name, old_name, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?new_name: Prism::node, ?old_name: Prism::node, ?keyword_loc: Location, ?location: Location) -> AliasMethodNode
    def copy(new_name: self.new_name, old_name: self.old_name, keyword_loc: self.keyword_loc, location: self.location)
      AliasMethodNode.new(source, new_name, old_name, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { new_name: Prism::node, old_name: Prism::node, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { new_name: new_name, old_name: old_name, keyword_loc: keyword_loc, location: location }
    end

    # attr_reader new_name: Prism::node
    attr_reader :new_name

    # attr_reader old_name: Prism::node
    attr_reader :old_name

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :alias_method_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :alias_method_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AliasMethodNode) &&
        (new_name === other.new_name) &&
        (old_name === other.old_name) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents an alternation pattern in pattern matching.
  #
  #     foo => bar | baz
  #            ^^^^^^^^^
  class AlternationPatternNode < Node
    # def initialize: (Prism::node left, Prism::node right, Location operator_loc, Location location) -> void
    def initialize(source, left, right, operator_loc, location)
      @source = source
      @location = location
      @left = left
      @right = right
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_alternation_pattern_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [left, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [left, right]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [left, right, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?left: Prism::node, ?right: Prism::node, ?operator_loc: Location, ?location: Location) -> AlternationPatternNode
    def copy(left: self.left, right: self.right, operator_loc: self.operator_loc, location: self.location)
      AlternationPatternNode.new(source, left, right, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { left: Prism::node, right: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { left: left, right: right, operator_loc: operator_loc, location: location }
    end

    # attr_reader left: Prism::node
    attr_reader :left

    # attr_reader right: Prism::node
    attr_reader :right

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :alternation_pattern_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :alternation_pattern_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AlternationPatternNode) &&
        (left === other.left) &&
        (right === other.right) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of the `&&` operator or the `and` keyword.
  #
  #     left and right
  #     ^^^^^^^^^^^^^^
  class AndNode < Node
    # def initialize: (Prism::node left, Prism::node right, Location operator_loc, Location location) -> void
    def initialize(source, left, right, operator_loc, location)
      @source = source
      @location = location
      @left = left
      @right = right
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_and_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [left, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [left, right]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [left, right, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?left: Prism::node, ?right: Prism::node, ?operator_loc: Location, ?location: Location) -> AndNode
    def copy(left: self.left, right: self.right, operator_loc: self.operator_loc, location: self.location)
      AndNode.new(source, left, right, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { left: Prism::node, right: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { left: left, right: right, operator_loc: operator_loc, location: location }
    end

    # Represents the left side of the expression. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     left and right
    #     ^^^^
    #
    #     1 && 2
    #     ^
    attr_reader :left

    # Represents the right side of the expression. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     left && right
    #             ^^^^^
    #
    #     1 and 2
    #           ^
    attr_reader :right

    # The location of the `and` keyword or the `&&` operator.
    #
    #     left and right
    #          ^^^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :and_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :and_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AndNode) &&
        (left === other.left) &&
        (right === other.right) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a set of arguments to a method or a keyword.
  #
  #     return foo, bar, baz
  #            ^^^^^^^^^^^^^
  class ArgumentsNode < Node
    # def initialize: (Integer flags, Array[Prism::node] arguments, Location location) -> void
    def initialize(source, flags, arguments, location)
      @source = source
      @location = location
      @flags = flags
      @arguments = arguments
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_arguments_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*arguments]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*arguments]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*arguments] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?arguments: Array[Prism::node], ?location: Location) -> ArgumentsNode
    def copy(flags: self.flags, arguments: self.arguments, location: self.location)
      ArgumentsNode.new(source, flags, arguments, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, arguments: Array[Prism::node], location: Location }
    def deconstruct_keys(keys)
      { flags: flags, arguments: arguments, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader arguments: Array[Prism::node]
    attr_reader :arguments

    # def contains_keywords?: () -> bool
    def contains_keywords?
      flags.anybits?(ArgumentsNodeFlags::CONTAINS_KEYWORDS)
    end

    # def contains_keyword_splat?: () -> bool
    def contains_keyword_splat?
      flags.anybits?(ArgumentsNodeFlags::CONTAINS_KEYWORD_SPLAT)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :arguments_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :arguments_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ArgumentsNode) &&
        (flags === other.flags) &&
        (arguments.length == other.arguments.length) &&
        arguments.zip(other.arguments).all? { |left, right| left === right }
    end
  end

  # Represents an array literal. This can be a regular array using brackets or a special array using % like %w or %i.
  #
  #     [1, 2, 3]
  #     ^^^^^^^^^
  class ArrayNode < Node
    # def initialize: (Integer flags, Array[Prism::node] elements, Location? opening_loc, Location? closing_loc, Location location) -> void
    def initialize(source, flags, elements, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @flags = flags
      @elements = elements
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_array_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*elements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*elements]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*elements, *opening_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?elements: Array[Prism::node], ?opening_loc: Location?, ?closing_loc: Location?, ?location: Location) -> ArrayNode
    def copy(flags: self.flags, elements: self.elements, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      ArrayNode.new(source, flags, elements, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, elements: Array[Prism::node], opening_loc: Location?, closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, elements: elements, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # Represent the list of zero or more [non-void expressions](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression) within the array.
    attr_reader :elements

    # Represents the optional source location for the opening token.
    #
    #     [1,2,3]                 # "["
    #     %w[foo bar baz]         # "%w["
    #     %I(apple orange banana) # "%I("
    #     foo = 1, 2, 3           # nil
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # Represents the optional source location for the closing token.
    #
    #     [1,2,3]                 # "]"
    #     %w[foo bar baz]         # "]"
    #     %I(apple orange banana) # ")"
    #     foo = 1, 2, 3           # nil
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def contains_splat?: () -> bool
    def contains_splat?
      flags.anybits?(ArrayNodeFlags::CONTAINS_SPLAT)
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :array_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :array_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ArrayNode) &&
        (flags === other.flags) &&
        (elements.length == other.elements.length) &&
        elements.zip(other.elements).all? { |left, right| left === right } &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents an array pattern in pattern matching.
  #
  #     foo in 1, 2
  #     ^^^^^^^^^^^
  #
  #     foo in [1, 2]
  #     ^^^^^^^^^^^^^
  #
  #     foo in *1
  #     ^^^^^^^^^
  #
  #     foo in Bar[]
  #     ^^^^^^^^^^^^
  #
  #     foo in Bar[1, 2, 3]
  #     ^^^^^^^^^^^^^^^^^^^
  class ArrayPatternNode < Node
    # def initialize: (Prism::node? constant, Array[Prism::node] requireds, Prism::node? rest, Array[Prism::node] posts, Location? opening_loc, Location? closing_loc, Location location) -> void
    def initialize(source, constant, requireds, rest, posts, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @constant = constant
      @requireds = requireds
      @rest = rest
      @posts = posts
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_array_pattern_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [constant, *requireds, rest, *posts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << constant if constant
      compact.concat(requireds)
      compact << rest if rest
      compact.concat(posts)
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*constant, *requireds, *rest, *posts, *opening_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?constant: Prism::node?, ?requireds: Array[Prism::node], ?rest: Prism::node?, ?posts: Array[Prism::node], ?opening_loc: Location?, ?closing_loc: Location?, ?location: Location) -> ArrayPatternNode
    def copy(constant: self.constant, requireds: self.requireds, rest: self.rest, posts: self.posts, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      ArrayPatternNode.new(source, constant, requireds, rest, posts, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { constant: Prism::node?, requireds: Array[Prism::node], rest: Prism::node?, posts: Array[Prism::node], opening_loc: Location?, closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { constant: constant, requireds: requireds, rest: rest, posts: posts, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader constant: Prism::node?
    attr_reader :constant

    # attr_reader requireds: Array[Prism::node]
    attr_reader :requireds

    # attr_reader rest: Prism::node?
    attr_reader :rest

    # attr_reader posts: Array[Prism::node]
    attr_reader :posts

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :array_pattern_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :array_pattern_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ArrayPatternNode) &&
        (constant === other.constant) &&
        (requireds.length == other.requireds.length) &&
        requireds.zip(other.requireds).all? { |left, right| left === right } &&
        (rest === other.rest) &&
        (posts.length == other.posts.length) &&
        posts.zip(other.posts).all? { |left, right| left === right } &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a hash key/value pair.
  #
  #     { a => b }
  #       ^^^^^^
  class AssocNode < Node
    # def initialize: (Prism::node key, Prism::node value, Location? operator_loc, Location location) -> void
    def initialize(source, key, value, operator_loc, location)
      @source = source
      @location = location
      @key = key
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_assoc_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [key, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [key, value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [key, value, *operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?key: Prism::node, ?value: Prism::node, ?operator_loc: Location?, ?location: Location) -> AssocNode
    def copy(key: self.key, value: self.value, operator_loc: self.operator_loc, location: self.location)
      AssocNode.new(source, key, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { key: Prism::node, value: Prism::node, operator_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { key: key, value: value, operator_loc: operator_loc, location: location }
    end

    # The key of the association. This can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     { a: b }
    #       ^
    #
    #     { foo => bar }
    #       ^^^
    #
    #     { def a; end => 1 }
    #       ^^^^^^^^^^
    attr_reader :key

    # The value of the association, if present. This can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     { foo => bar }
    #              ^^^
    #
    #     { x: 1 }
    #          ^
    attr_reader :value

    # The location of the `=>` operator, if present.
    #
    #     { foo => bar }
    #           ^^
    def operator_loc
      location = @operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def operator: () -> String?
    def operator
      operator_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :assoc_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :assoc_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AssocNode) &&
        (key === other.key) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a splat in a hash literal.
  #
  #     { **foo }
  #       ^^^^^
  class AssocSplatNode < Node
    # def initialize: (Prism::node? value, Location operator_loc, Location location) -> void
    def initialize(source, value, operator_loc, location)
      @source = source
      @location = location
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_assoc_splat_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << value if value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Prism::node?, ?operator_loc: Location, ?location: Location) -> AssocSplatNode
    def copy(value: self.value, operator_loc: self.operator_loc, location: self.location)
      AssocSplatNode.new(source, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Prism::node?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { value: value, operator_loc: operator_loc, location: location }
    end

    # The value to be splatted, if present. Will be missing when keyword rest argument forwarding is used.
    #
    #     { **foo }
    #         ^^^
    attr_reader :value

    # The location of the `**` operator.
    #
    #     { **x }
    #       ^^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :assoc_splat_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :assoc_splat_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(AssocSplatNode) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents reading a reference to a field in the previous match.
  #
  #     $'
  #     ^^
  class BackReferenceReadNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_back_reference_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> BackReferenceReadNode
    def copy(name: self.name, location: self.location)
      BackReferenceReadNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # The name of the back-reference variable, including the leading `$`.
    #
    #     $& # name `:$&`
    #
    #     $+ # name `:$+`
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :back_reference_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :back_reference_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BackReferenceReadNode) &&
        (name === other.name)
    end
  end

  # Represents a begin statement.
  #
  #     begin
  #       foo
  #     end
  #     ^^^^^
  class BeginNode < Node
    # def initialize: (Location? begin_keyword_loc, StatementsNode? statements, RescueNode? rescue_clause, ElseNode? else_clause, EnsureNode? ensure_clause, Location? end_keyword_loc, Location location) -> void
    def initialize(source, begin_keyword_loc, statements, rescue_clause, else_clause, ensure_clause, end_keyword_loc, location)
      @source = source
      @location = location
      @begin_keyword_loc = begin_keyword_loc
      @statements = statements
      @rescue_clause = rescue_clause
      @else_clause = else_clause
      @ensure_clause = ensure_clause
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_begin_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements, rescue_clause, else_clause, ensure_clause]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact << rescue_clause if rescue_clause
      compact << else_clause if else_clause
      compact << ensure_clause if ensure_clause
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*begin_keyword_loc, *statements, *rescue_clause, *else_clause, *ensure_clause, *end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?begin_keyword_loc: Location?, ?statements: StatementsNode?, ?rescue_clause: RescueNode?, ?else_clause: ElseNode?, ?ensure_clause: EnsureNode?, ?end_keyword_loc: Location?, ?location: Location) -> BeginNode
    def copy(begin_keyword_loc: self.begin_keyword_loc, statements: self.statements, rescue_clause: self.rescue_clause, else_clause: self.else_clause, ensure_clause: self.ensure_clause, end_keyword_loc: self.end_keyword_loc, location: self.location)
      BeginNode.new(source, begin_keyword_loc, statements, rescue_clause, else_clause, ensure_clause, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { begin_keyword_loc: Location?, statements: StatementsNode?, rescue_clause: RescueNode?, else_clause: ElseNode?, ensure_clause: EnsureNode?, end_keyword_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { begin_keyword_loc: begin_keyword_loc, statements: statements, rescue_clause: rescue_clause, else_clause: else_clause, ensure_clause: ensure_clause, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader begin_keyword_loc: Location?
    def begin_keyword_loc
      location = @begin_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @begin_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader rescue_clause: RescueNode?
    attr_reader :rescue_clause

    # attr_reader else_clause: ElseNode?
    attr_reader :else_clause

    # attr_reader ensure_clause: EnsureNode?
    attr_reader :ensure_clause

    # attr_reader end_keyword_loc: Location?
    def end_keyword_loc
      location = @end_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def begin_keyword: () -> String?
    def begin_keyword
      begin_keyword_loc&.slice
    end

    # def end_keyword: () -> String?
    def end_keyword
      end_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :begin_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :begin_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BeginNode) &&
        (begin_keyword_loc.nil? == other.begin_keyword_loc.nil?) &&
        (statements === other.statements) &&
        (rescue_clause === other.rescue_clause) &&
        (else_clause === other.else_clause) &&
        (ensure_clause === other.ensure_clause) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents block method arguments.
  #
  #     bar(&args)
  #     ^^^^^^^^^^
  class BlockArgumentNode < Node
    # def initialize: (Prism::node? expression, Location operator_loc, Location location) -> void
    def initialize(source, expression, operator_loc, location)
      @source = source
      @location = location
      @expression = expression
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_block_argument_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [expression]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << expression if expression
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*expression, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?expression: Prism::node?, ?operator_loc: Location, ?location: Location) -> BlockArgumentNode
    def copy(expression: self.expression, operator_loc: self.operator_loc, location: self.location)
      BlockArgumentNode.new(source, expression, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { expression: Prism::node?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { expression: expression, operator_loc: operator_loc, location: location }
    end

    # attr_reader expression: Prism::node?
    attr_reader :expression

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :block_argument_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :block_argument_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BlockArgumentNode) &&
        (expression === other.expression) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a block local variable.
  #
  #     a { |; b| }
  #            ^
  class BlockLocalVariableNode < Node
    # def initialize: (Integer flags, Symbol name, Location location) -> void
    def initialize(source, flags, name, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_block_local_variable_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol, ?location: Location) -> BlockLocalVariableNode
    def copy(flags: self.flags, name: self.name, location: self.location)
      BlockLocalVariableNode.new(source, flags, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol
    attr_reader :name

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :block_local_variable_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :block_local_variable_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BlockLocalVariableNode) &&
        (flags === other.flags) &&
        (name === other.name)
    end
  end

  # Represents a block of ruby code.
  #
  #     [1, 2, 3].each { |i| puts x }
  #                    ^^^^^^^^^^^^^^
  class BlockNode < Node
    # def initialize: (Array[Symbol] locals, Prism::node? parameters, Prism::node? body, Location opening_loc, Location closing_loc, Location location) -> void
    def initialize(source, locals, parameters, body, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @locals = locals
      @parameters = parameters
      @body = body
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_block_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [parameters, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << parameters if parameters
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*parameters, *body, opening_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?parameters: Prism::node?, ?body: Prism::node?, ?opening_loc: Location, ?closing_loc: Location, ?location: Location) -> BlockNode
    def copy(locals: self.locals, parameters: self.parameters, body: self.body, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      BlockNode.new(source, locals, parameters, body, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], parameters: Prism::node?, body: Prism::node?, opening_loc: Location, closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, parameters: parameters, body: body, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader parameters: Prism::node?
    attr_reader :parameters

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :block_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :block_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BlockNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (parameters === other.parameters) &&
        (body === other.body) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a block parameter to a method, block, or lambda definition.
  #
  #     def a(&b)
  #           ^^
  #     end
  class BlockParameterNode < Node
    # def initialize: (Integer flags, Symbol? name, Location? name_loc, Location operator_loc, Location location) -> void
    def initialize(source, flags, name, name_loc, operator_loc, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_block_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*name_loc, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol?, ?name_loc: Location?, ?operator_loc: Location, ?location: Location) -> BlockParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, location: self.location)
      BlockParameterNode.new(source, flags, name, name_loc, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol?, name_loc: Location?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, operator_loc: operator_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol?
    attr_reader :name

    # attr_reader name_loc: Location?
    def name_loc
      location = @name_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :block_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :block_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BlockParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a block's parameters declaration.
  #
  #     -> (a, b = 1; local) { }
  #        ^^^^^^^^^^^^^^^^^
  #
  #     foo do |a, b = 1; local|
  #            ^^^^^^^^^^^^^^^^^
  #     end
  class BlockParametersNode < Node
    # def initialize: (ParametersNode? parameters, Array[BlockLocalVariableNode] locals, Location? opening_loc, Location? closing_loc, Location location) -> void
    def initialize(source, parameters, locals, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @parameters = parameters
      @locals = locals
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_block_parameters_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [parameters, *locals]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << parameters if parameters
      compact.concat(locals)
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*parameters, *locals, *opening_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?parameters: ParametersNode?, ?locals: Array[BlockLocalVariableNode], ?opening_loc: Location?, ?closing_loc: Location?, ?location: Location) -> BlockParametersNode
    def copy(parameters: self.parameters, locals: self.locals, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      BlockParametersNode.new(source, parameters, locals, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { parameters: ParametersNode?, locals: Array[BlockLocalVariableNode], opening_loc: Location?, closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { parameters: parameters, locals: locals, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader parameters: ParametersNode?
    attr_reader :parameters

    # attr_reader locals: Array[BlockLocalVariableNode]
    attr_reader :locals

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :block_parameters_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :block_parameters_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BlockParametersNode) &&
        (parameters === other.parameters) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents the use of the `break` keyword.
  #
  #     break foo
  #     ^^^^^^^^^
  class BreakNode < Node
    # def initialize: (ArgumentsNode? arguments, Location keyword_loc, Location location) -> void
    def initialize(source, arguments, keyword_loc, location)
      @source = source
      @location = location
      @arguments = arguments
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_break_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [arguments]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << arguments if arguments
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*arguments, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?arguments: ArgumentsNode?, ?keyword_loc: Location, ?location: Location) -> BreakNode
    def copy(arguments: self.arguments, keyword_loc: self.keyword_loc, location: self.location)
      BreakNode.new(source, arguments, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { arguments: ArgumentsNode?, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { arguments: arguments, keyword_loc: keyword_loc, location: location }
    end

    # The arguments to the break statement, if present. These can be any [non-void expressions](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     break foo
    #           ^^^
    attr_reader :arguments

    # The location of the `break` keyword.
    #
    #     break foo
    #     ^^^^^
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :break_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :break_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(BreakNode) &&
        (arguments === other.arguments) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents the use of the `&&=` operator on a call.
  #
  #     foo.bar &&= value
  #     ^^^^^^^^^^^^^^^^^
  class CallAndWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location? message_loc, Symbol read_name, Symbol write_name, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @message_loc = message_loc
      @read_name = read_name
      @write_name = write_name
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_call_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, *message_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?message_loc: Location?, ?read_name: Symbol, ?write_name: Symbol, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> CallAndWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, message_loc: self.message_loc, read_name: self.read_name, write_name: self.write_name, operator_loc: self.operator_loc, value: self.value, location: self.location)
      CallAndWriteNode.new(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, message_loc: Location?, read_name: Symbol, write_name: Symbol, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, message_loc: message_loc, read_name: read_name, write_name: write_name, operator_loc: operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader message_loc: Location?
    def message_loc
      location = @message_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @message_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader read_name: Symbol
    attr_reader :read_name

    # attr_reader write_name: Symbol
    attr_reader :write_name

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def message: () -> String?
    def message
      message_loc&.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :call_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :call_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CallAndWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (message_loc.nil? == other.message_loc.nil?) &&
        (read_name === other.read_name) &&
        (write_name === other.write_name) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents a method call, in all of the various forms that can take.
  #
  #     foo
  #     ^^^
  #
  #     foo()
  #     ^^^^^
  #
  #     +foo
  #     ^^^^
  #
  #     foo + bar
  #     ^^^^^^^^^
  #
  #     foo.bar
  #     ^^^^^^^
  #
  #     foo&.bar
  #     ^^^^^^^^
  class CallNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Symbol name, Location? message_loc, Location? opening_loc, ArgumentsNode? arguments, Location? closing_loc, Prism::node? block, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, name, message_loc, opening_loc, arguments, closing_loc, block, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @name = name
      @message_loc = message_loc
      @opening_loc = opening_loc
      @arguments = arguments
      @closing_loc = closing_loc
      @block = block
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_call_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, arguments, block]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << arguments if arguments
      compact << block if block
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, *message_loc, *opening_loc, *arguments, *closing_loc, *block] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?name: Symbol, ?message_loc: Location?, ?opening_loc: Location?, ?arguments: ArgumentsNode?, ?closing_loc: Location?, ?block: Prism::node?, ?location: Location) -> CallNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, name: self.name, message_loc: self.message_loc, opening_loc: self.opening_loc, arguments: self.arguments, closing_loc: self.closing_loc, block: self.block, location: self.location)
      CallNode.new(source, flags, receiver, call_operator_loc, name, message_loc, opening_loc, arguments, closing_loc, block, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, name: Symbol, message_loc: Location?, opening_loc: Location?, arguments: ArgumentsNode?, closing_loc: Location?, block: Prism::node?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, name: name, message_loc: message_loc, opening_loc: opening_loc, arguments: arguments, closing_loc: closing_loc, block: block, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # The object that the method is being called on. This can be either `nil` or any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     foo.bar
    #     ^^^
    #
    #     +foo
    #      ^^^
    #
    #     foo + bar
    #     ^^^
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader message_loc: Location?
    def message_loc
      location = @message_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @message_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def message: () -> String?
    def message
      message_loc&.slice
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :call_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :call_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CallNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (name === other.name) &&
        (message_loc.nil? == other.message_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (arguments === other.arguments) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (block === other.block)
    end
  end

  # Represents the use of an assignment operator on a call.
  #
  #     foo.bar += baz
  #     ^^^^^^^^^^^^^^
  class CallOperatorWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location? message_loc, Symbol read_name, Symbol write_name, Symbol binary_operator, Location binary_operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, binary_operator, binary_operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @message_loc = message_loc
      @read_name = read_name
      @write_name = write_name
      @binary_operator = binary_operator
      @binary_operator_loc = binary_operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_call_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, *message_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?message_loc: Location?, ?read_name: Symbol, ?write_name: Symbol, ?binary_operator: Symbol, ?binary_operator_loc: Location, ?value: Prism::node, ?location: Location) -> CallOperatorWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, message_loc: self.message_loc, read_name: self.read_name, write_name: self.write_name, binary_operator: self.binary_operator, binary_operator_loc: self.binary_operator_loc, value: self.value, location: self.location)
      CallOperatorWriteNode.new(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, binary_operator, binary_operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, message_loc: Location?, read_name: Symbol, write_name: Symbol, binary_operator: Symbol, binary_operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, message_loc: message_loc, read_name: read_name, write_name: write_name, binary_operator: binary_operator, binary_operator_loc: binary_operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader message_loc: Location?
    def message_loc
      location = @message_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @message_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader read_name: Symbol
    attr_reader :read_name

    # attr_reader write_name: Symbol
    attr_reader :write_name

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def message: () -> String?
    def message
      message_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :call_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :call_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CallOperatorWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (message_loc.nil? == other.message_loc.nil?) &&
        (read_name === other.read_name) &&
        (write_name === other.write_name) &&
        (binary_operator === other.binary_operator) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of the `||=` operator on a call.
  #
  #     foo.bar ||= value
  #     ^^^^^^^^^^^^^^^^^
  class CallOrWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location? message_loc, Symbol read_name, Symbol write_name, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @message_loc = message_loc
      @read_name = read_name
      @write_name = write_name
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_call_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, *message_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?message_loc: Location?, ?read_name: Symbol, ?write_name: Symbol, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> CallOrWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, message_loc: self.message_loc, read_name: self.read_name, write_name: self.write_name, operator_loc: self.operator_loc, value: self.value, location: self.location)
      CallOrWriteNode.new(source, flags, receiver, call_operator_loc, message_loc, read_name, write_name, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, message_loc: Location?, read_name: Symbol, write_name: Symbol, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, message_loc: message_loc, read_name: read_name, write_name: write_name, operator_loc: operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader message_loc: Location?
    def message_loc
      location = @message_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @message_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader read_name: Symbol
    attr_reader :read_name

    # attr_reader write_name: Symbol
    attr_reader :write_name

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def message: () -> String?
    def message
      message_loc&.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :call_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :call_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CallOrWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (message_loc.nil? == other.message_loc.nil?) &&
        (read_name === other.read_name) &&
        (write_name === other.write_name) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to a method call.
  #
  #     foo.bar, = 1
  #     ^^^^^^^
  #
  #     begin
  #     rescue => foo.bar
  #               ^^^^^^^
  #     end
  #
  #     for foo.bar in baz do end
  #         ^^^^^^^
  class CallTargetNode < Node
    # def initialize: (Integer flags, Prism::node receiver, Location call_operator_loc, Symbol name, Location message_loc, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, name, message_loc, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @name = name
      @message_loc = message_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_call_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [receiver]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [receiver, call_operator_loc, message_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node, ?call_operator_loc: Location, ?name: Symbol, ?message_loc: Location, ?location: Location) -> CallTargetNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, name: self.name, message_loc: self.message_loc, location: self.location)
      CallTargetNode.new(source, flags, receiver, call_operator_loc, name, message_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node, call_operator_loc: Location, name: Symbol, message_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, name: name, message_loc: message_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node
    attr_reader :receiver

    # attr_reader call_operator_loc: Location
    def call_operator_loc
      location = @call_operator_loc
      return location if location.is_a?(Location)
      @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader message_loc: Location
    def message_loc
      location = @message_loc
      return location if location.is_a?(Location)
      @message_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String
    def call_operator
      call_operator_loc.slice
    end

    # def message: () -> String
    def message
      message_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :call_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :call_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CallTargetNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (name === other.name) &&
        (message_loc.nil? == other.message_loc.nil?)
    end
  end

  # Represents assigning to a local variable in pattern matching.
  #
  #     foo => [bar => baz]
  #            ^^^^^^^^^^^^
  class CapturePatternNode < Node
    # def initialize: (Prism::node value, Prism::node target, Location operator_loc, Location location) -> void
    def initialize(source, value, target, operator_loc, location)
      @source = source
      @location = location
      @value = value
      @target = target
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_capture_pattern_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value, target]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value, target]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [value, target, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Prism::node, ?target: Prism::node, ?operator_loc: Location, ?location: Location) -> CapturePatternNode
    def copy(value: self.value, target: self.target, operator_loc: self.operator_loc, location: self.location)
      CapturePatternNode.new(source, value, target, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Prism::node, target: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { value: value, target: target, operator_loc: operator_loc, location: location }
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader target: Prism::node
    attr_reader :target

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :capture_pattern_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :capture_pattern_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CapturePatternNode) &&
        (value === other.value) &&
        (target === other.target) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of a case statement for pattern matching.
  #
  #     case true
  #     in false
  #     end
  #     ^^^^^^^^^
  class CaseMatchNode < Node
    # def initialize: (Prism::node? predicate, Array[Prism::node] conditions, ElseNode? consequent, Location case_keyword_loc, Location end_keyword_loc, Location location) -> void
    def initialize(source, predicate, conditions, consequent, case_keyword_loc, end_keyword_loc, location)
      @source = source
      @location = location
      @predicate = predicate
      @conditions = conditions
      @consequent = consequent
      @case_keyword_loc = case_keyword_loc
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_case_match_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, *conditions, consequent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate if predicate
      compact.concat(conditions)
      compact << consequent if consequent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*predicate, *conditions, *consequent, case_keyword_loc, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?predicate: Prism::node?, ?conditions: Array[Prism::node], ?consequent: ElseNode?, ?case_keyword_loc: Location, ?end_keyword_loc: Location, ?location: Location) -> CaseMatchNode
    def copy(predicate: self.predicate, conditions: self.conditions, consequent: self.consequent, case_keyword_loc: self.case_keyword_loc, end_keyword_loc: self.end_keyword_loc, location: self.location)
      CaseMatchNode.new(source, predicate, conditions, consequent, case_keyword_loc, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { predicate: Prism::node?, conditions: Array[Prism::node], consequent: ElseNode?, case_keyword_loc: Location, end_keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { predicate: predicate, conditions: conditions, consequent: consequent, case_keyword_loc: case_keyword_loc, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader predicate: Prism::node?
    attr_reader :predicate

    # attr_reader conditions: Array[Prism::node]
    attr_reader :conditions

    # attr_reader consequent: ElseNode?
    attr_reader :consequent

    # attr_reader case_keyword_loc: Location
    def case_keyword_loc
      location = @case_keyword_loc
      return location if location.is_a?(Location)
      @case_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def case_keyword: () -> String
    def case_keyword
      case_keyword_loc.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :case_match_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :case_match_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CaseMatchNode) &&
        (predicate === other.predicate) &&
        (conditions.length == other.conditions.length) &&
        conditions.zip(other.conditions).all? { |left, right| left === right } &&
        (consequent === other.consequent) &&
        (case_keyword_loc.nil? == other.case_keyword_loc.nil?) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents the use of a case statement.
  #
  #     case true
  #     when false
  #     end
  #     ^^^^^^^^^^
  class CaseNode < Node
    # def initialize: (Prism::node? predicate, Array[Prism::node] conditions, ElseNode? consequent, Location case_keyword_loc, Location end_keyword_loc, Location location) -> void
    def initialize(source, predicate, conditions, consequent, case_keyword_loc, end_keyword_loc, location)
      @source = source
      @location = location
      @predicate = predicate
      @conditions = conditions
      @consequent = consequent
      @case_keyword_loc = case_keyword_loc
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_case_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, *conditions, consequent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate if predicate
      compact.concat(conditions)
      compact << consequent if consequent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*predicate, *conditions, *consequent, case_keyword_loc, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?predicate: Prism::node?, ?conditions: Array[Prism::node], ?consequent: ElseNode?, ?case_keyword_loc: Location, ?end_keyword_loc: Location, ?location: Location) -> CaseNode
    def copy(predicate: self.predicate, conditions: self.conditions, consequent: self.consequent, case_keyword_loc: self.case_keyword_loc, end_keyword_loc: self.end_keyword_loc, location: self.location)
      CaseNode.new(source, predicate, conditions, consequent, case_keyword_loc, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { predicate: Prism::node?, conditions: Array[Prism::node], consequent: ElseNode?, case_keyword_loc: Location, end_keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { predicate: predicate, conditions: conditions, consequent: consequent, case_keyword_loc: case_keyword_loc, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader predicate: Prism::node?
    attr_reader :predicate

    # attr_reader conditions: Array[Prism::node]
    attr_reader :conditions

    # attr_reader consequent: ElseNode?
    attr_reader :consequent

    # attr_reader case_keyword_loc: Location
    def case_keyword_loc
      location = @case_keyword_loc
      return location if location.is_a?(Location)
      @case_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def case_keyword: () -> String
    def case_keyword
      case_keyword_loc.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :case_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :case_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(CaseNode) &&
        (predicate === other.predicate) &&
        (conditions.length == other.conditions.length) &&
        conditions.zip(other.conditions).all? { |left, right| left === right } &&
        (consequent === other.consequent) &&
        (case_keyword_loc.nil? == other.case_keyword_loc.nil?) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents a class declaration involving the `class` keyword.
  #
  #     class Foo end
  #     ^^^^^^^^^^^^^
  class ClassNode < Node
    # def initialize: (Array[Symbol] locals, Location class_keyword_loc, Prism::node constant_path, Location? inheritance_operator_loc, Prism::node? superclass, Prism::node? body, Location end_keyword_loc, Symbol name, Location location) -> void
    def initialize(source, locals, class_keyword_loc, constant_path, inheritance_operator_loc, superclass, body, end_keyword_loc, name, location)
      @source = source
      @location = location
      @locals = locals
      @class_keyword_loc = class_keyword_loc
      @constant_path = constant_path
      @inheritance_operator_loc = inheritance_operator_loc
      @superclass = superclass
      @body = body
      @end_keyword_loc = end_keyword_loc
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [constant_path, superclass, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << constant_path
      compact << superclass if superclass
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [class_keyword_loc, constant_path, *inheritance_operator_loc, *superclass, *body, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?class_keyword_loc: Location, ?constant_path: Prism::node, ?inheritance_operator_loc: Location?, ?superclass: Prism::node?, ?body: Prism::node?, ?end_keyword_loc: Location, ?name: Symbol, ?location: Location) -> ClassNode
    def copy(locals: self.locals, class_keyword_loc: self.class_keyword_loc, constant_path: self.constant_path, inheritance_operator_loc: self.inheritance_operator_loc, superclass: self.superclass, body: self.body, end_keyword_loc: self.end_keyword_loc, name: self.name, location: self.location)
      ClassNode.new(source, locals, class_keyword_loc, constant_path, inheritance_operator_loc, superclass, body, end_keyword_loc, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], class_keyword_loc: Location, constant_path: Prism::node, inheritance_operator_loc: Location?, superclass: Prism::node?, body: Prism::node?, end_keyword_loc: Location, name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, class_keyword_loc: class_keyword_loc, constant_path: constant_path, inheritance_operator_loc: inheritance_operator_loc, superclass: superclass, body: body, end_keyword_loc: end_keyword_loc, name: name, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader class_keyword_loc: Location
    def class_keyword_loc
      location = @class_keyword_loc
      return location if location.is_a?(Location)
      @class_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader constant_path: Prism::node
    attr_reader :constant_path

    # attr_reader inheritance_operator_loc: Location?
    def inheritance_operator_loc
      location = @inheritance_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @inheritance_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader superclass: Prism::node?
    attr_reader :superclass

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def class_keyword: () -> String
    def class_keyword
      class_keyword_loc.slice
    end

    # def inheritance_operator: () -> String?
    def inheritance_operator
      inheritance_operator_loc&.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (class_keyword_loc.nil? == other.class_keyword_loc.nil?) &&
        (constant_path === other.constant_path) &&
        (inheritance_operator_loc.nil? == other.inheritance_operator_loc.nil?) &&
        (superclass === other.superclass) &&
        (body === other.body) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?) &&
        (name === other.name)
    end
  end

  # Represents the use of the `&&=` operator for assignment to a class variable.
  #
  #     @@target &&= value
  #     ^^^^^^^^^^^^^^^^^^
  class ClassVariableAndWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ClassVariableAndWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ClassVariableAndWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableAndWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to a class variable using an operator that isn't `=`.
  #
  #     @@target += value
  #     ^^^^^^^^^^^^^^^^^
  class ClassVariableOperatorWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location binary_operator_loc, Prism::node value, Symbol binary_operator, Location location) -> void
    def initialize(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @binary_operator_loc = binary_operator_loc
      @value = value
      @binary_operator = binary_operator
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?binary_operator_loc: Location, ?value: Prism::node, ?binary_operator: Symbol, ?location: Location) -> ClassVariableOperatorWriteNode
    def copy(name: self.name, name_loc: self.name_loc, binary_operator_loc: self.binary_operator_loc, value: self.value, binary_operator: self.binary_operator, location: self.location)
      ClassVariableOperatorWriteNode.new(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, binary_operator_loc: Location, value: Prism::node, binary_operator: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, binary_operator_loc: binary_operator_loc, value: value, binary_operator: binary_operator, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableOperatorWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (binary_operator === other.binary_operator)
    end
  end

  # Represents the use of the `||=` operator for assignment to a class variable.
  #
  #     @@target ||= value
  #     ^^^^^^^^^^^^^^^^^^
  class ClassVariableOrWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ClassVariableOrWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ClassVariableOrWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableOrWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents referencing a class variable.
  #
  #     @@foo
  #     ^^^^^
  class ClassVariableReadNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> ClassVariableReadNode
    def copy(name: self.name, location: self.location)
      ClassVariableReadNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # The name of the class variable, which is a `@@` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     @@abc   # name `:@@abc`
    #
    #     @@_test # name `:@@_test`
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableReadNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a class variable in a context that doesn't have an explicit value.
  #
  #     @@foo, @@bar = baz
  #     ^^^^^  ^^^^^
  class ClassVariableTargetNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> ClassVariableTargetNode
    def copy(name: self.name, location: self.location)
      ClassVariableTargetNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableTargetNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a class variable.
  #
  #     @@foo = 1
  #     ^^^^^^^^^
  class ClassVariableWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Prism::node value, Location operator_loc, Location location) -> void
    def initialize(source, name, name_loc, value, operator_loc, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_class_variable_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?value: Prism::node, ?operator_loc: Location, ?location: Location) -> ClassVariableWriteNode
    def copy(name: self.name, name_loc: self.name_loc, value: self.value, operator_loc: self.operator_loc, location: self.location)
      ClassVariableWriteNode.new(source, name, name_loc, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, value: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, value: value, operator_loc: operator_loc, location: location }
    end

    # The name of the class variable, which is a `@@` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     @@abc = 123     # name `@@abc`
    #
    #     @@_test = :test # name `@@_test`
    attr_reader :name

    # The location of the variable name.
    #
    #     @@foo = :bar
    #     ^^^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the class variable. This can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     @@foo = :bar
    #             ^^^^
    #
    #     @@_xyz = 123
    #              ^^^
    attr_reader :value

    # The location of the `=` operator.
    #
    #     @@foo = :bar
    #           ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :class_variable_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :class_variable_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ClassVariableWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of the `&&=` operator for assignment to a constant.
  #
  #     Target &&= value
  #     ^^^^^^^^^^^^^^^^
  class ConstantAndWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ConstantAndWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ConstantAndWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantAndWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to a constant using an operator that isn't `=`.
  #
  #     Target += value
  #     ^^^^^^^^^^^^^^^
  class ConstantOperatorWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location binary_operator_loc, Prism::node value, Symbol binary_operator, Location location) -> void
    def initialize(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @binary_operator_loc = binary_operator_loc
      @value = value
      @binary_operator = binary_operator
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?binary_operator_loc: Location, ?value: Prism::node, ?binary_operator: Symbol, ?location: Location) -> ConstantOperatorWriteNode
    def copy(name: self.name, name_loc: self.name_loc, binary_operator_loc: self.binary_operator_loc, value: self.value, binary_operator: self.binary_operator, location: self.location)
      ConstantOperatorWriteNode.new(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, binary_operator_loc: Location, value: Prism::node, binary_operator: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, binary_operator_loc: binary_operator_loc, value: value, binary_operator: binary_operator, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantOperatorWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (binary_operator === other.binary_operator)
    end
  end

  # Represents the use of the `||=` operator for assignment to a constant.
  #
  #     Target ||= value
  #     ^^^^^^^^^^^^^^^^
  class ConstantOrWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ConstantOrWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ConstantOrWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantOrWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of the `&&=` operator for assignment to a constant path.
  #
  #     Parent::Child &&= value
  #     ^^^^^^^^^^^^^^^^^^^^^^^
  class ConstantPathAndWriteNode < Node
    # def initialize: (ConstantPathNode target, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, target, operator_loc, value, location)
      @source = source
      @location = location
      @target = target
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [target, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [target, value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [target, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?target: ConstantPathNode, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ConstantPathAndWriteNode
    def copy(target: self.target, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ConstantPathAndWriteNode.new(source, target, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { target: ConstantPathNode, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { target: target, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader target: ConstantPathNode
    attr_reader :target

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathAndWriteNode) &&
        (target === other.target) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents accessing a constant through a path of `::` operators.
  #
  #     Foo::Bar
  #     ^^^^^^^^
  class ConstantPathNode < Node
    # def initialize: (Prism::node? parent, Symbol? name, Location delimiter_loc, Location name_loc, Location location) -> void
    def initialize(source, parent, name, delimiter_loc, name_loc, location)
      @source = source
      @location = location
      @parent = parent
      @name = name
      @delimiter_loc = delimiter_loc
      @name_loc = name_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [parent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << parent if parent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*parent, delimiter_loc, name_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?parent: Prism::node?, ?name: Symbol?, ?delimiter_loc: Location, ?name_loc: Location, ?location: Location) -> ConstantPathNode
    def copy(parent: self.parent, name: self.name, delimiter_loc: self.delimiter_loc, name_loc: self.name_loc, location: self.location)
      ConstantPathNode.new(source, parent, name, delimiter_loc, name_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { parent: Prism::node?, name: Symbol?, delimiter_loc: Location, name_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { parent: parent, name: name, delimiter_loc: delimiter_loc, name_loc: name_loc, location: location }
    end

    # The left-hand node of the path, if present. It can be `nil` or any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression). It will be `nil` when the constant lookup is at the root of the module tree.
    #
    #     Foo::Bar
    #     ^^^
    #
    #     self::Test
    #     ^^^^
    #
    #     a.b::C
    #     ^^^
    attr_reader :parent

    # The name of the constant being accessed. This could be `nil` in the event of a syntax error.
    attr_reader :name

    # The location of the `::` delimiter.
    #
    #     ::Foo
    #     ^^
    #
    #     One::Two
    #        ^^
    def delimiter_loc
      location = @delimiter_loc
      return location if location.is_a?(Location)
      @delimiter_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The location of the name of the constant.
    #
    #     ::Foo
    #       ^^^
    #
    #     One::Two
    #          ^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def delimiter: () -> String
    def delimiter
      delimiter_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathNode) &&
        (parent === other.parent) &&
        (name === other.name) &&
        (delimiter_loc.nil? == other.delimiter_loc.nil?) &&
        (name_loc.nil? == other.name_loc.nil?)
    end
  end

  # Represents assigning to a constant path using an operator that isn't `=`.
  #
  #     Parent::Child += value
  #     ^^^^^^^^^^^^^^^^^^^^^^
  class ConstantPathOperatorWriteNode < Node
    # def initialize: (ConstantPathNode target, Location binary_operator_loc, Prism::node value, Symbol binary_operator, Location location) -> void
    def initialize(source, target, binary_operator_loc, value, binary_operator, location)
      @source = source
      @location = location
      @target = target
      @binary_operator_loc = binary_operator_loc
      @value = value
      @binary_operator = binary_operator
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [target, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [target, value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [target, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?target: ConstantPathNode, ?binary_operator_loc: Location, ?value: Prism::node, ?binary_operator: Symbol, ?location: Location) -> ConstantPathOperatorWriteNode
    def copy(target: self.target, binary_operator_loc: self.binary_operator_loc, value: self.value, binary_operator: self.binary_operator, location: self.location)
      ConstantPathOperatorWriteNode.new(source, target, binary_operator_loc, value, binary_operator, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { target: ConstantPathNode, binary_operator_loc: Location, value: Prism::node, binary_operator: Symbol, location: Location }
    def deconstruct_keys(keys)
      { target: target, binary_operator_loc: binary_operator_loc, value: value, binary_operator: binary_operator, location: location }
    end

    # attr_reader target: ConstantPathNode
    attr_reader :target

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathOperatorWriteNode) &&
        (target === other.target) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (binary_operator === other.binary_operator)
    end
  end

  # Represents the use of the `||=` operator for assignment to a constant path.
  #
  #     Parent::Child ||= value
  #     ^^^^^^^^^^^^^^^^^^^^^^^
  class ConstantPathOrWriteNode < Node
    # def initialize: (ConstantPathNode target, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, target, operator_loc, value, location)
      @source = source
      @location = location
      @target = target
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [target, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [target, value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [target, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?target: ConstantPathNode, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ConstantPathOrWriteNode
    def copy(target: self.target, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ConstantPathOrWriteNode.new(source, target, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { target: ConstantPathNode, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { target: target, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader target: ConstantPathNode
    attr_reader :target

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathOrWriteNode) &&
        (target === other.target) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents writing to a constant path in a context that doesn't have an explicit value.
  #
  #     Foo::Foo, Bar::Bar = baz
  #     ^^^^^^^^  ^^^^^^^^
  class ConstantPathTargetNode < Node
    # def initialize: (Prism::node? parent, Symbol? name, Location delimiter_loc, Location name_loc, Location location) -> void
    def initialize(source, parent, name, delimiter_loc, name_loc, location)
      @source = source
      @location = location
      @parent = parent
      @name = name
      @delimiter_loc = delimiter_loc
      @name_loc = name_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [parent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << parent if parent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*parent, delimiter_loc, name_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?parent: Prism::node?, ?name: Symbol?, ?delimiter_loc: Location, ?name_loc: Location, ?location: Location) -> ConstantPathTargetNode
    def copy(parent: self.parent, name: self.name, delimiter_loc: self.delimiter_loc, name_loc: self.name_loc, location: self.location)
      ConstantPathTargetNode.new(source, parent, name, delimiter_loc, name_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { parent: Prism::node?, name: Symbol?, delimiter_loc: Location, name_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { parent: parent, name: name, delimiter_loc: delimiter_loc, name_loc: name_loc, location: location }
    end

    # attr_reader parent: Prism::node?
    attr_reader :parent

    # attr_reader name: Symbol?
    attr_reader :name

    # attr_reader delimiter_loc: Location
    def delimiter_loc
      location = @delimiter_loc
      return location if location.is_a?(Location)
      @delimiter_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def delimiter: () -> String
    def delimiter
      delimiter_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathTargetNode) &&
        (parent === other.parent) &&
        (name === other.name) &&
        (delimiter_loc.nil? == other.delimiter_loc.nil?) &&
        (name_loc.nil? == other.name_loc.nil?)
    end
  end

  # Represents writing to a constant path.
  #
  #     ::Foo = 1
  #     ^^^^^^^^^
  #
  #     Foo::Bar = 1
  #     ^^^^^^^^^^^^
  #
  #     ::Foo::Bar = 1
  #     ^^^^^^^^^^^^^^
  class ConstantPathWriteNode < Node
    # def initialize: (ConstantPathNode target, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, target, operator_loc, value, location)
      @source = source
      @location = location
      @target = target
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_path_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [target, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [target, value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [target, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?target: ConstantPathNode, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> ConstantPathWriteNode
    def copy(target: self.target, operator_loc: self.operator_loc, value: self.value, location: self.location)
      ConstantPathWriteNode.new(source, target, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { target: ConstantPathNode, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { target: target, operator_loc: operator_loc, value: value, location: location }
    end

    # A node representing the constant path being written to.
    #
    #     Foo::Bar = 1
    #     ^^^^^^^^
    #
    #     ::Foo = :abc
    #     ^^^^^
    attr_reader :target

    # The location of the `=` operator.
    #
    #     ::ABC = 123
    #           ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the constant path. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     FOO::BAR = :abc
    #                ^^^^
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_path_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_path_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantPathWriteNode) &&
        (target === other.target) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents referencing a constant.
  #
  #     Foo
  #     ^^^
  class ConstantReadNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> ConstantReadNode
    def copy(name: self.name, location: self.location)
      ConstantReadNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # The name of the [constant](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#constants).
    #
    #     X              # name `:X`
    #
    #     SOME_CONSTANT  # name `:SOME_CONSTANT`
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantReadNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a constant in a context that doesn't have an explicit value.
  #
  #     Foo, Bar = baz
  #     ^^^  ^^^
  class ConstantTargetNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> ConstantTargetNode
    def copy(name: self.name, location: self.location)
      ConstantTargetNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantTargetNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a constant.
  #
  #     Foo = 1
  #     ^^^^^^^
  class ConstantWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Prism::node value, Location operator_loc, Location location) -> void
    def initialize(source, name, name_loc, value, operator_loc, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_constant_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?value: Prism::node, ?operator_loc: Location, ?location: Location) -> ConstantWriteNode
    def copy(name: self.name, name_loc: self.name_loc, value: self.value, operator_loc: self.operator_loc, location: self.location)
      ConstantWriteNode.new(source, name, name_loc, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, value: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, value: value, operator_loc: operator_loc, location: location }
    end

    # The name of the [constant](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#constants).
    #
    #     Foo = :bar # name `:Foo`
    #
    #     XYZ = 1    # name `:XYZ`
    attr_reader :name

    # The location of the constant name.
    #
    #     FOO = 1
    #     ^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the constant. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     FOO = :bar
    #           ^^^^
    #
    #     MyClass = Class.new
    #               ^^^^^^^^^
    attr_reader :value

    # The location of the `=` operator.
    #
    #     FOO = :bar
    #         ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :constant_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :constant_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ConstantWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a method definition.
  #
  #     def method
  #     end
  #     ^^^^^^^^^^
  class DefNode < Node
    # def initialize: (Symbol name, Location name_loc, Prism::node? receiver, ParametersNode? parameters, Prism::node? body, Array[Symbol] locals, Location def_keyword_loc, Location? operator_loc, Location? lparen_loc, Location? rparen_loc, Location? equal_loc, Location? end_keyword_loc, Location location) -> void
    def initialize(source, name, name_loc, receiver, parameters, body, locals, def_keyword_loc, operator_loc, lparen_loc, rparen_loc, equal_loc, end_keyword_loc, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @receiver = receiver
      @parameters = parameters
      @body = body
      @locals = locals
      @def_keyword_loc = def_keyword_loc
      @operator_loc = operator_loc
      @lparen_loc = lparen_loc
      @rparen_loc = rparen_loc
      @equal_loc = equal_loc
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_def_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, parameters, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << parameters if parameters
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, *receiver, *parameters, *body, def_keyword_loc, *operator_loc, *lparen_loc, *rparen_loc, *equal_loc, *end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?receiver: Prism::node?, ?parameters: ParametersNode?, ?body: Prism::node?, ?locals: Array[Symbol], ?def_keyword_loc: Location, ?operator_loc: Location?, ?lparen_loc: Location?, ?rparen_loc: Location?, ?equal_loc: Location?, ?end_keyword_loc: Location?, ?location: Location) -> DefNode
    def copy(name: self.name, name_loc: self.name_loc, receiver: self.receiver, parameters: self.parameters, body: self.body, locals: self.locals, def_keyword_loc: self.def_keyword_loc, operator_loc: self.operator_loc, lparen_loc: self.lparen_loc, rparen_loc: self.rparen_loc, equal_loc: self.equal_loc, end_keyword_loc: self.end_keyword_loc, location: self.location)
      DefNode.new(source, name, name_loc, receiver, parameters, body, locals, def_keyword_loc, operator_loc, lparen_loc, rparen_loc, equal_loc, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, receiver: Prism::node?, parameters: ParametersNode?, body: Prism::node?, locals: Array[Symbol], def_keyword_loc: Location, operator_loc: Location?, lparen_loc: Location?, rparen_loc: Location?, equal_loc: Location?, end_keyword_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, receiver: receiver, parameters: parameters, body: body, locals: locals, def_keyword_loc: def_keyword_loc, operator_loc: operator_loc, lparen_loc: lparen_loc, rparen_loc: rparen_loc, equal_loc: equal_loc, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader parameters: ParametersNode?
    attr_reader :parameters

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader def_keyword_loc: Location
    def def_keyword_loc
      location = @def_keyword_loc
      return location if location.is_a?(Location)
      @def_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location?
    def operator_loc
      location = @operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader equal_loc: Location?
    def equal_loc
      location = @equal_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @equal_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader end_keyword_loc: Location?
    def end_keyword_loc
      location = @end_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def def_keyword: () -> String
    def def_keyword
      def_keyword_loc.slice
    end

    # def operator: () -> String?
    def operator
      operator_loc&.slice
    end

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def equal: () -> String?
    def equal
      equal_loc&.slice
    end

    # def end_keyword: () -> String?
    def end_keyword
      end_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :def_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :def_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(DefNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (receiver === other.receiver) &&
        (parameters === other.parameters) &&
        (body === other.body) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (def_keyword_loc.nil? == other.def_keyword_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (rparen_loc.nil? == other.rparen_loc.nil?) &&
        (equal_loc.nil? == other.equal_loc.nil?) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents the use of the `defined?` keyword.
  #
  #     defined?(a)
  #     ^^^^^^^^^^^
  class DefinedNode < Node
    # def initialize: (Location? lparen_loc, Prism::node value, Location? rparen_loc, Location keyword_loc, Location location) -> void
    def initialize(source, lparen_loc, value, rparen_loc, keyword_loc, location)
      @source = source
      @location = location
      @lparen_loc = lparen_loc
      @value = value
      @rparen_loc = rparen_loc
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_defined_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*lparen_loc, value, *rparen_loc, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?lparen_loc: Location?, ?value: Prism::node, ?rparen_loc: Location?, ?keyword_loc: Location, ?location: Location) -> DefinedNode
    def copy(lparen_loc: self.lparen_loc, value: self.value, rparen_loc: self.rparen_loc, keyword_loc: self.keyword_loc, location: self.location)
      DefinedNode.new(source, lparen_loc, value, rparen_loc, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { lparen_loc: Location?, value: Prism::node, rparen_loc: Location?, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { lparen_loc: lparen_loc, value: value, rparen_loc: rparen_loc, keyword_loc: keyword_loc, location: location }
    end

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :defined_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :defined_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(DefinedNode) &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (value === other.value) &&
        (rparen_loc.nil? == other.rparen_loc.nil?) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents an `else` clause in a `case`, `if`, or `unless` statement.
  #
  #     if a then b else c end
  #                 ^^^^^^^^^^
  class ElseNode < Node
    # def initialize: (Location else_keyword_loc, StatementsNode? statements, Location? end_keyword_loc, Location location) -> void
    def initialize(source, else_keyword_loc, statements, end_keyword_loc, location)
      @source = source
      @location = location
      @else_keyword_loc = else_keyword_loc
      @statements = statements
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_else_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [else_keyword_loc, *statements, *end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?else_keyword_loc: Location, ?statements: StatementsNode?, ?end_keyword_loc: Location?, ?location: Location) -> ElseNode
    def copy(else_keyword_loc: self.else_keyword_loc, statements: self.statements, end_keyword_loc: self.end_keyword_loc, location: self.location)
      ElseNode.new(source, else_keyword_loc, statements, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { else_keyword_loc: Location, statements: StatementsNode?, end_keyword_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { else_keyword_loc: else_keyword_loc, statements: statements, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader else_keyword_loc: Location
    def else_keyword_loc
      location = @else_keyword_loc
      return location if location.is_a?(Location)
      @else_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader end_keyword_loc: Location?
    def end_keyword_loc
      location = @end_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def else_keyword: () -> String
    def else_keyword
      else_keyword_loc.slice
    end

    # def end_keyword: () -> String?
    def end_keyword
      end_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :else_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :else_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ElseNode) &&
        (else_keyword_loc.nil? == other.else_keyword_loc.nil?) &&
        (statements === other.statements) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents an interpolated set of statements.
  #
  #     "foo #{bar}"
  #          ^^^^^^
  class EmbeddedStatementsNode < Node
    # def initialize: (Location opening_loc, StatementsNode? statements, Location closing_loc, Location location) -> void
    def initialize(source, opening_loc, statements, closing_loc, location)
      @source = source
      @location = location
      @opening_loc = opening_loc
      @statements = statements
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_embedded_statements_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, *statements, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?opening_loc: Location, ?statements: StatementsNode?, ?closing_loc: Location, ?location: Location) -> EmbeddedStatementsNode
    def copy(opening_loc: self.opening_loc, statements: self.statements, closing_loc: self.closing_loc, location: self.location)
      EmbeddedStatementsNode.new(source, opening_loc, statements, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { opening_loc: Location, statements: StatementsNode?, closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { opening_loc: opening_loc, statements: statements, closing_loc: closing_loc, location: location }
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :embedded_statements_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :embedded_statements_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(EmbeddedStatementsNode) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (statements === other.statements) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents an interpolated variable.
  #
  #     "foo #@bar"
  #          ^^^^^
  class EmbeddedVariableNode < Node
    # def initialize: (Location operator_loc, Prism::node variable, Location location) -> void
    def initialize(source, operator_loc, variable, location)
      @source = source
      @location = location
      @operator_loc = operator_loc
      @variable = variable
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_embedded_variable_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [variable]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [variable]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [operator_loc, variable] #: Array[Prism::node | Location]
    end

    # def copy: (?operator_loc: Location, ?variable: Prism::node, ?location: Location) -> EmbeddedVariableNode
    def copy(operator_loc: self.operator_loc, variable: self.variable, location: self.location)
      EmbeddedVariableNode.new(source, operator_loc, variable, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { operator_loc: Location, variable: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { operator_loc: operator_loc, variable: variable, location: location }
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader variable: Prism::node
    attr_reader :variable

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :embedded_variable_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :embedded_variable_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(EmbeddedVariableNode) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (variable === other.variable)
    end
  end

  # Represents an `ensure` clause in a `begin` statement.
  #
  #     begin
  #       foo
  #     ensure
  #     ^^^^^^
  #       bar
  #     end
  class EnsureNode < Node
    # def initialize: (Location ensure_keyword_loc, StatementsNode? statements, Location end_keyword_loc, Location location) -> void
    def initialize(source, ensure_keyword_loc, statements, end_keyword_loc, location)
      @source = source
      @location = location
      @ensure_keyword_loc = ensure_keyword_loc
      @statements = statements
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_ensure_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [ensure_keyword_loc, *statements, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?ensure_keyword_loc: Location, ?statements: StatementsNode?, ?end_keyword_loc: Location, ?location: Location) -> EnsureNode
    def copy(ensure_keyword_loc: self.ensure_keyword_loc, statements: self.statements, end_keyword_loc: self.end_keyword_loc, location: self.location)
      EnsureNode.new(source, ensure_keyword_loc, statements, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { ensure_keyword_loc: Location, statements: StatementsNode?, end_keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { ensure_keyword_loc: ensure_keyword_loc, statements: statements, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader ensure_keyword_loc: Location
    def ensure_keyword_loc
      location = @ensure_keyword_loc
      return location if location.is_a?(Location)
      @ensure_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def ensure_keyword: () -> String
    def ensure_keyword
      ensure_keyword_loc.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :ensure_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :ensure_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(EnsureNode) &&
        (ensure_keyword_loc.nil? == other.ensure_keyword_loc.nil?) &&
        (statements === other.statements) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents the use of the literal `false` keyword.
  #
  #     false
  #     ^^^^^
  class FalseNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_false_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> FalseNode
    def copy(location: self.location)
      FalseNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :false_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :false_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(FalseNode)
    end
  end

  # Represents a find pattern in pattern matching.
  #
  #     foo in *bar, baz, *qux
  #            ^^^^^^^^^^^^^^^
  #
  #     foo in [*bar, baz, *qux]
  #            ^^^^^^^^^^^^^^^^^
  #
  #     foo in Foo(*bar, baz, *qux)
  #            ^^^^^^^^^^^^^^^^^^^^
  class FindPatternNode < Node
    # def initialize: (Prism::node? constant, Prism::node left, Array[Prism::node] requireds, Prism::node right, Location? opening_loc, Location? closing_loc, Location location) -> void
    def initialize(source, constant, left, requireds, right, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @constant = constant
      @left = left
      @requireds = requireds
      @right = right
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_find_pattern_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [constant, left, *requireds, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << constant if constant
      compact << left
      compact.concat(requireds)
      compact << right
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*constant, left, *requireds, right, *opening_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?constant: Prism::node?, ?left: Prism::node, ?requireds: Array[Prism::node], ?right: Prism::node, ?opening_loc: Location?, ?closing_loc: Location?, ?location: Location) -> FindPatternNode
    def copy(constant: self.constant, left: self.left, requireds: self.requireds, right: self.right, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      FindPatternNode.new(source, constant, left, requireds, right, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { constant: Prism::node?, left: Prism::node, requireds: Array[Prism::node], right: Prism::node, opening_loc: Location?, closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { constant: constant, left: left, requireds: requireds, right: right, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader constant: Prism::node?
    attr_reader :constant

    # attr_reader left: Prism::node
    attr_reader :left

    # attr_reader requireds: Array[Prism::node]
    attr_reader :requireds

    # attr_reader right: Prism::node
    attr_reader :right

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :find_pattern_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :find_pattern_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(FindPatternNode) &&
        (constant === other.constant) &&
        (left === other.left) &&
        (requireds.length == other.requireds.length) &&
        requireds.zip(other.requireds).all? { |left, right| left === right } &&
        (right === other.right) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents the use of the `..` or `...` operators to create flip flops.
  #
  #     baz if foo .. bar
  #            ^^^^^^^^^^
  class FlipFlopNode < Node
    # def initialize: (Integer flags, Prism::node? left, Prism::node? right, Location operator_loc, Location location) -> void
    def initialize(source, flags, left, right, operator_loc, location)
      @source = source
      @location = location
      @flags = flags
      @left = left
      @right = right
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_flip_flop_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [left, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << left if left
      compact << right if right
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*left, *right, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?left: Prism::node?, ?right: Prism::node?, ?operator_loc: Location, ?location: Location) -> FlipFlopNode
    def copy(flags: self.flags, left: self.left, right: self.right, operator_loc: self.operator_loc, location: self.location)
      FlipFlopNode.new(source, flags, left, right, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, left: Prism::node?, right: Prism::node?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, left: left, right: right, operator_loc: operator_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader left: Prism::node?
    attr_reader :left

    # attr_reader right: Prism::node?
    attr_reader :right

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def exclude_end?: () -> bool
    def exclude_end?
      flags.anybits?(RangeFlags::EXCLUDE_END)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :flip_flop_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :flip_flop_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(FlipFlopNode) &&
        (flags === other.flags) &&
        (left === other.left) &&
        (right === other.right) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a floating point number literal.
  #
  #     1.0
  #     ^^^
  class FloatNode < Node
    # def initialize: (Float value, Location location) -> void
    def initialize(source, value, location)
      @source = source
      @location = location
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_float_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Float, ?location: Location) -> FloatNode
    def copy(value: self.value, location: self.location)
      FloatNode.new(source, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Float, location: Location }
    def deconstruct_keys(keys)
      { value: value, location: location }
    end

    # The value of the floating point number as a Float.
    attr_reader :value

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :float_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :float_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(FloatNode) &&
        (value === other.value)
    end
  end

  # Represents the use of the `for` keyword.
  #
  #     for i in a end
  #     ^^^^^^^^^^^^^^
  class ForNode < Node
    # def initialize: (Prism::node index, Prism::node collection, StatementsNode? statements, Location for_keyword_loc, Location in_keyword_loc, Location? do_keyword_loc, Location end_keyword_loc, Location location) -> void
    def initialize(source, index, collection, statements, for_keyword_loc, in_keyword_loc, do_keyword_loc, end_keyword_loc, location)
      @source = source
      @location = location
      @index = index
      @collection = collection
      @statements = statements
      @for_keyword_loc = for_keyword_loc
      @in_keyword_loc = in_keyword_loc
      @do_keyword_loc = do_keyword_loc
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_for_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [index, collection, statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << index
      compact << collection
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [index, collection, *statements, for_keyword_loc, in_keyword_loc, *do_keyword_loc, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?index: Prism::node, ?collection: Prism::node, ?statements: StatementsNode?, ?for_keyword_loc: Location, ?in_keyword_loc: Location, ?do_keyword_loc: Location?, ?end_keyword_loc: Location, ?location: Location) -> ForNode
    def copy(index: self.index, collection: self.collection, statements: self.statements, for_keyword_loc: self.for_keyword_loc, in_keyword_loc: self.in_keyword_loc, do_keyword_loc: self.do_keyword_loc, end_keyword_loc: self.end_keyword_loc, location: self.location)
      ForNode.new(source, index, collection, statements, for_keyword_loc, in_keyword_loc, do_keyword_loc, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { index: Prism::node, collection: Prism::node, statements: StatementsNode?, for_keyword_loc: Location, in_keyword_loc: Location, do_keyword_loc: Location?, end_keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { index: index, collection: collection, statements: statements, for_keyword_loc: for_keyword_loc, in_keyword_loc: in_keyword_loc, do_keyword_loc: do_keyword_loc, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader index: Prism::node
    attr_reader :index

    # attr_reader collection: Prism::node
    attr_reader :collection

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader for_keyword_loc: Location
    def for_keyword_loc
      location = @for_keyword_loc
      return location if location.is_a?(Location)
      @for_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader in_keyword_loc: Location
    def in_keyword_loc
      location = @in_keyword_loc
      return location if location.is_a?(Location)
      @in_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader do_keyword_loc: Location?
    def do_keyword_loc
      location = @do_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @do_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def for_keyword: () -> String
    def for_keyword
      for_keyword_loc.slice
    end

    # def in_keyword: () -> String
    def in_keyword
      in_keyword_loc.slice
    end

    # def do_keyword: () -> String?
    def do_keyword
      do_keyword_loc&.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :for_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :for_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ForNode) &&
        (index === other.index) &&
        (collection === other.collection) &&
        (statements === other.statements) &&
        (for_keyword_loc.nil? == other.for_keyword_loc.nil?) &&
        (in_keyword_loc.nil? == other.in_keyword_loc.nil?) &&
        (do_keyword_loc.nil? == other.do_keyword_loc.nil?) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents forwarding all arguments to this method to another method.
  #
  #     def foo(...)
  #       bar(...)
  #           ^^^
  #     end
  class ForwardingArgumentsNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_forwarding_arguments_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> ForwardingArgumentsNode
    def copy(location: self.location)
      ForwardingArgumentsNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :forwarding_arguments_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :forwarding_arguments_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ForwardingArgumentsNode)
    end
  end

  # Represents the use of the forwarding parameter in a method, block, or lambda declaration.
  #
  #     def foo(...)
  #             ^^^
  #     end
  class ForwardingParameterNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_forwarding_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> ForwardingParameterNode
    def copy(location: self.location)
      ForwardingParameterNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :forwarding_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :forwarding_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ForwardingParameterNode)
    end
  end

  # Represents the use of the `super` keyword without parentheses or arguments.
  #
  #     super
  #     ^^^^^
  class ForwardingSuperNode < Node
    # def initialize: (BlockNode? block, Location location) -> void
    def initialize(source, block, location)
      @source = source
      @location = location
      @block = block
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_forwarding_super_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [block]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << block if block
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*block] #: Array[Prism::node | Location]
    end

    # def copy: (?block: BlockNode?, ?location: Location) -> ForwardingSuperNode
    def copy(block: self.block, location: self.location)
      ForwardingSuperNode.new(source, block, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { block: BlockNode?, location: Location }
    def deconstruct_keys(keys)
      { block: block, location: location }
    end

    # attr_reader block: BlockNode?
    attr_reader :block

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :forwarding_super_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :forwarding_super_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ForwardingSuperNode) &&
        (block === other.block)
    end
  end

  # Represents the use of the `&&=` operator for assignment to a global variable.
  #
  #     $target &&= value
  #     ^^^^^^^^^^^^^^^^^
  class GlobalVariableAndWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> GlobalVariableAndWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      GlobalVariableAndWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableAndWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to a global variable using an operator that isn't `=`.
  #
  #     $target += value
  #     ^^^^^^^^^^^^^^^^
  class GlobalVariableOperatorWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location binary_operator_loc, Prism::node value, Symbol binary_operator, Location location) -> void
    def initialize(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @binary_operator_loc = binary_operator_loc
      @value = value
      @binary_operator = binary_operator
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?binary_operator_loc: Location, ?value: Prism::node, ?binary_operator: Symbol, ?location: Location) -> GlobalVariableOperatorWriteNode
    def copy(name: self.name, name_loc: self.name_loc, binary_operator_loc: self.binary_operator_loc, value: self.value, binary_operator: self.binary_operator, location: self.location)
      GlobalVariableOperatorWriteNode.new(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, binary_operator_loc: Location, value: Prism::node, binary_operator: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, binary_operator_loc: binary_operator_loc, value: value, binary_operator: binary_operator, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableOperatorWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (binary_operator === other.binary_operator)
    end
  end

  # Represents the use of the `||=` operator for assignment to a global variable.
  #
  #     $target ||= value
  #     ^^^^^^^^^^^^^^^^^
  class GlobalVariableOrWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> GlobalVariableOrWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      GlobalVariableOrWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableOrWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents referencing a global variable.
  #
  #     $foo
  #     ^^^^
  class GlobalVariableReadNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> GlobalVariableReadNode
    def copy(name: self.name, location: self.location)
      GlobalVariableReadNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # The name of the global variable, which is a `$` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifier). Alternatively, it can be one of the special global variables designated by a symbol.
    #
    #     $foo   # name `:$foo`
    #
    #     $_Test # name `:$_Test`
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableReadNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a global variable in a context that doesn't have an explicit value.
  #
  #     $foo, $bar = baz
  #     ^^^^  ^^^^
  class GlobalVariableTargetNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> GlobalVariableTargetNode
    def copy(name: self.name, location: self.location)
      GlobalVariableTargetNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableTargetNode) &&
        (name === other.name)
    end
  end

  # Represents writing to a global variable.
  #
  #     $foo = 1
  #     ^^^^^^^^
  class GlobalVariableWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Prism::node value, Location operator_loc, Location location) -> void
    def initialize(source, name, name_loc, value, operator_loc, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_global_variable_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?value: Prism::node, ?operator_loc: Location, ?location: Location) -> GlobalVariableWriteNode
    def copy(name: self.name, name_loc: self.name_loc, value: self.value, operator_loc: self.operator_loc, location: self.location)
      GlobalVariableWriteNode.new(source, name, name_loc, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, value: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, value: value, operator_loc: operator_loc, location: location }
    end

    # The name of the global variable, which is a `$` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifier). Alternatively, it can be one of the special global variables designated by a symbol.
    #
    #     $foo = :bar  # name `:$foo`
    #
    #     $_Test = 123 # name `:$_Test`
    attr_reader :name

    # The location of the global variable's name.
    #
    #     $foo = :bar
    #     ^^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the global variable. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     $foo = :bar
    #            ^^^^
    #
    #     $-xyz = 123
    #             ^^^
    attr_reader :value

    # The location of the `=` operator.
    #
    #     $foo = :bar
    #          ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :global_variable_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :global_variable_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(GlobalVariableWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a hash literal.
  #
  #     { a => b }
  #     ^^^^^^^^^^
  class HashNode < Node
    # def initialize: (Location opening_loc, Array[AssocNode | AssocSplatNode] elements, Location closing_loc, Location location) -> void
    def initialize(source, opening_loc, elements, closing_loc, location)
      @source = source
      @location = location
      @opening_loc = opening_loc
      @elements = elements
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_hash_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*elements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*elements]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, *elements, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?opening_loc: Location, ?elements: Array[AssocNode | AssocSplatNode], ?closing_loc: Location, ?location: Location) -> HashNode
    def copy(opening_loc: self.opening_loc, elements: self.elements, closing_loc: self.closing_loc, location: self.location)
      HashNode.new(source, opening_loc, elements, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { opening_loc: Location, elements: Array[AssocNode | AssocSplatNode], closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { opening_loc: opening_loc, elements: elements, closing_loc: closing_loc, location: location }
    end

    # The location of the opening brace.
    #
    #     { a => b }
    #     ^
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The elements of the hash. These can be either `AssocNode`s or `AssocSplatNode`s.
    #
    #     { a: b }
    #       ^^^^
    #
    #     { **foo }
    #       ^^^^^
    attr_reader :elements

    # The location of the closing brace.
    #
    #     { a => b }
    #              ^
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :hash_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :hash_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(HashNode) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (elements.length == other.elements.length) &&
        elements.zip(other.elements).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a hash pattern in pattern matching.
  #
  #     foo => { a: 1, b: 2 }
  #            ^^^^^^^^^^^^^^
  #
  #     foo => { a: 1, b: 2, **c }
  #            ^^^^^^^^^^^^^^^^^^^
  class HashPatternNode < Node
    # def initialize: (Prism::node? constant, Array[AssocNode] elements, AssocSplatNode | NoKeywordsParameterNode | nil rest, Location? opening_loc, Location? closing_loc, Location location) -> void
    def initialize(source, constant, elements, rest, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @constant = constant
      @elements = elements
      @rest = rest
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_hash_pattern_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [constant, *elements, rest]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << constant if constant
      compact.concat(elements)
      compact << rest if rest
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*constant, *elements, *rest, *opening_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?constant: Prism::node?, ?elements: Array[AssocNode], ?rest: AssocSplatNode | NoKeywordsParameterNode | nil, ?opening_loc: Location?, ?closing_loc: Location?, ?location: Location) -> HashPatternNode
    def copy(constant: self.constant, elements: self.elements, rest: self.rest, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      HashPatternNode.new(source, constant, elements, rest, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { constant: Prism::node?, elements: Array[AssocNode], rest: AssocSplatNode | NoKeywordsParameterNode | nil, opening_loc: Location?, closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { constant: constant, elements: elements, rest: rest, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader constant: Prism::node?
    attr_reader :constant

    # attr_reader elements: Array[AssocNode]
    attr_reader :elements

    # attr_reader rest: AssocSplatNode | NoKeywordsParameterNode | nil
    attr_reader :rest

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :hash_pattern_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :hash_pattern_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(HashPatternNode) &&
        (constant === other.constant) &&
        (elements.length == other.elements.length) &&
        elements.zip(other.elements).all? { |left, right| left === right } &&
        (rest === other.rest) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents the use of the `if` keyword, either in the block form or the modifier form, or a ternary expression.
  #
  #     bar if foo
  #     ^^^^^^^^^^
  #
  #     if foo then bar end
  #     ^^^^^^^^^^^^^^^^^^^
  #
  #     foo ? bar : baz
  #     ^^^^^^^^^^^^^^^
  class IfNode < Node
    # def initialize: (Location? if_keyword_loc, Prism::node predicate, Location? then_keyword_loc, StatementsNode? statements, Prism::node? consequent, Location? end_keyword_loc, Location location) -> void
    def initialize(source, if_keyword_loc, predicate, then_keyword_loc, statements, consequent, end_keyword_loc, location)
      @source = source
      @location = location
      @if_keyword_loc = if_keyword_loc
      @predicate = predicate
      @then_keyword_loc = then_keyword_loc
      @statements = statements
      @consequent = consequent
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_if_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, statements, consequent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate
      compact << statements if statements
      compact << consequent if consequent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*if_keyword_loc, predicate, *then_keyword_loc, *statements, *consequent, *end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?if_keyword_loc: Location?, ?predicate: Prism::node, ?then_keyword_loc: Location?, ?statements: StatementsNode?, ?consequent: Prism::node?, ?end_keyword_loc: Location?, ?location: Location) -> IfNode
    def copy(if_keyword_loc: self.if_keyword_loc, predicate: self.predicate, then_keyword_loc: self.then_keyword_loc, statements: self.statements, consequent: self.consequent, end_keyword_loc: self.end_keyword_loc, location: self.location)
      IfNode.new(source, if_keyword_loc, predicate, then_keyword_loc, statements, consequent, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { if_keyword_loc: Location?, predicate: Prism::node, then_keyword_loc: Location?, statements: StatementsNode?, consequent: Prism::node?, end_keyword_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { if_keyword_loc: if_keyword_loc, predicate: predicate, then_keyword_loc: then_keyword_loc, statements: statements, consequent: consequent, end_keyword_loc: end_keyword_loc, location: location }
    end

    # The location of the `if` keyword if present.
    #
    #     bar if foo
    #         ^^
    #
    # The `if_keyword_loc` field will be `nil` when the `IfNode` represents a ternary expression.
    def if_keyword_loc
      location = @if_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @if_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # The node for the condition the `IfNode` is testing.
    #
    #     if foo
    #        ^^^
    #       bar
    #     end
    #
    #     bar if foo
    #            ^^^
    #
    #     foo ? bar : baz
    #     ^^^
    attr_reader :predicate

    # The location of the `then` keyword (if present) or the `?` in a ternary expression, `nil` otherwise.
    #
    #     if foo then bar end
    #            ^^^^
    #
    #     a ? b : c
    #       ^
    def then_keyword_loc
      location = @then_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @then_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # Represents the body of statements that will be executed when the predicate is evaluated as truthy. Will be `nil` when no body is provided.
    #
    #     if foo
    #       bar
    #       ^^^
    #       baz
    #       ^^^
    #     end
    attr_reader :statements

    # Represents an `ElseNode` or an `IfNode` when there is an `else` or an `elsif` in the `if` statement.
    #
    #     if foo
    #       bar
    #     elsif baz
    #     ^^^^^^^^^
    #       qux
    #       ^^^
    #     end
    #     ^^^
    #
    #     if foo then bar else baz end
    #                     ^^^^^^^^^^^^
    attr_reader :consequent

    # The location of the `end` keyword if present, `nil` otherwise.
    #
    #     if foo
    #       bar
    #     end
    #     ^^^
    def end_keyword_loc
      location = @end_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def if_keyword: () -> String?
    def if_keyword
      if_keyword_loc&.slice
    end

    # def then_keyword: () -> String?
    def then_keyword
      then_keyword_loc&.slice
    end

    # def end_keyword: () -> String?
    def end_keyword
      end_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :if_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :if_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IfNode) &&
        (if_keyword_loc.nil? == other.if_keyword_loc.nil?) &&
        (predicate === other.predicate) &&
        (then_keyword_loc.nil? == other.then_keyword_loc.nil?) &&
        (statements === other.statements) &&
        (consequent === other.consequent) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents an imaginary number literal.
  #
  #     1.0i
  #     ^^^^
  class ImaginaryNode < Node
    # def initialize: (FloatNode | IntegerNode | RationalNode numeric, Location location) -> void
    def initialize(source, numeric, location)
      @source = source
      @location = location
      @numeric = numeric
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_imaginary_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [numeric]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [numeric]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [numeric] #: Array[Prism::node | Location]
    end

    # def copy: (?numeric: FloatNode | IntegerNode | RationalNode, ?location: Location) -> ImaginaryNode
    def copy(numeric: self.numeric, location: self.location)
      ImaginaryNode.new(source, numeric, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { numeric: FloatNode | IntegerNode | RationalNode, location: Location }
    def deconstruct_keys(keys)
      { numeric: numeric, location: location }
    end

    # attr_reader numeric: FloatNode | IntegerNode | RationalNode
    attr_reader :numeric

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :imaginary_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :imaginary_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ImaginaryNode) &&
        (numeric === other.numeric)
    end
  end

  # Represents a node that is implicitly being added to the tree but doesn't correspond directly to a node in the source.
  #
  #     { foo: }
  #       ^^^^
  #
  #     { Foo: }
  #       ^^^^
  #
  #     foo in { bar: }
  #              ^^^^
  class ImplicitNode < Node
    # def initialize: (Prism::node value, Location location) -> void
    def initialize(source, value, location)
      @source = source
      @location = location
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_implicit_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [value] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Prism::node, ?location: Location) -> ImplicitNode
    def copy(value: self.value, location: self.location)
      ImplicitNode.new(source, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { value: value, location: location }
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :implicit_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :implicit_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ImplicitNode) &&
        (value === other.value)
    end
  end

  # Represents using a trailing comma to indicate an implicit rest parameter.
  #
  #     foo { |bar,| }
  #               ^
  #
  #     foo in [bar,]
  #                ^
  #
  #     for foo, in bar do end
  #            ^
  #
  #     foo, = bar
  #        ^
  class ImplicitRestNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_implicit_rest_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> ImplicitRestNode
    def copy(location: self.location)
      ImplicitRestNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :implicit_rest_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :implicit_rest_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ImplicitRestNode)
    end
  end

  # Represents the use of the `in` keyword in a case statement.
  #
  #     case a; in b then c end
  #             ^^^^^^^^^^^
  class InNode < Node
    # def initialize: (Prism::node pattern, StatementsNode? statements, Location in_loc, Location? then_loc, Location location) -> void
    def initialize(source, pattern, statements, in_loc, then_loc, location)
      @source = source
      @location = location
      @pattern = pattern
      @statements = statements
      @in_loc = in_loc
      @then_loc = then_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_in_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [pattern, statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << pattern
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [pattern, *statements, in_loc, *then_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?pattern: Prism::node, ?statements: StatementsNode?, ?in_loc: Location, ?then_loc: Location?, ?location: Location) -> InNode
    def copy(pattern: self.pattern, statements: self.statements, in_loc: self.in_loc, then_loc: self.then_loc, location: self.location)
      InNode.new(source, pattern, statements, in_loc, then_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { pattern: Prism::node, statements: StatementsNode?, in_loc: Location, then_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { pattern: pattern, statements: statements, in_loc: in_loc, then_loc: then_loc, location: location }
    end

    # attr_reader pattern: Prism::node
    attr_reader :pattern

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader in_loc: Location
    def in_loc
      location = @in_loc
      return location if location.is_a?(Location)
      @in_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader then_loc: Location?
    def then_loc
      location = @then_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @then_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def in: () -> String
    def in
      in_loc.slice
    end

    # def then: () -> String?
    def then
      then_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :in_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :in_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InNode) &&
        (pattern === other.pattern) &&
        (statements === other.statements) &&
        (in_loc.nil? == other.in_loc.nil?) &&
        (then_loc.nil? == other.then_loc.nil?)
    end
  end

  # Represents the use of the `&&=` operator on a call to the `[]` method.
  #
  #     foo.bar[baz] &&= value
  #     ^^^^^^^^^^^^^^^^^^^^^^
  class IndexAndWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location opening_loc, ArgumentsNode? arguments, Location closing_loc, Prism::node? block, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @opening_loc = opening_loc
      @arguments = arguments
      @closing_loc = closing_loc
      @block = block
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_index_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, arguments, block, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << arguments if arguments
      compact << block if block
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, opening_loc, *arguments, closing_loc, *block, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?opening_loc: Location, ?arguments: ArgumentsNode?, ?closing_loc: Location, ?block: Prism::node?, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> IndexAndWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, opening_loc: self.opening_loc, arguments: self.arguments, closing_loc: self.closing_loc, block: self.block, operator_loc: self.operator_loc, value: self.value, location: self.location)
      IndexAndWriteNode.new(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, opening_loc: Location, arguments: ArgumentsNode?, closing_loc: Location, block: Prism::node?, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, opening_loc: opening_loc, arguments: arguments, closing_loc: closing_loc, block: block, operator_loc: operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :index_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :index_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IndexAndWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (arguments === other.arguments) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (block === other.block) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of an assignment operator on a call to `[]`.
  #
  #     foo.bar[baz] += value
  #     ^^^^^^^^^^^^^^^^^^^^^
  class IndexOperatorWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location opening_loc, ArgumentsNode? arguments, Location closing_loc, Prism::node? block, Symbol binary_operator, Location binary_operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, binary_operator, binary_operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @opening_loc = opening_loc
      @arguments = arguments
      @closing_loc = closing_loc
      @block = block
      @binary_operator = binary_operator
      @binary_operator_loc = binary_operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_index_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, arguments, block, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << arguments if arguments
      compact << block if block
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, opening_loc, *arguments, closing_loc, *block, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?opening_loc: Location, ?arguments: ArgumentsNode?, ?closing_loc: Location, ?block: Prism::node?, ?binary_operator: Symbol, ?binary_operator_loc: Location, ?value: Prism::node, ?location: Location) -> IndexOperatorWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, opening_loc: self.opening_loc, arguments: self.arguments, closing_loc: self.closing_loc, block: self.block, binary_operator: self.binary_operator, binary_operator_loc: self.binary_operator_loc, value: self.value, location: self.location)
      IndexOperatorWriteNode.new(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, binary_operator, binary_operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, opening_loc: Location, arguments: ArgumentsNode?, closing_loc: Location, block: Prism::node?, binary_operator: Symbol, binary_operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, opening_loc: opening_loc, arguments: arguments, closing_loc: closing_loc, block: block, binary_operator: binary_operator, binary_operator_loc: binary_operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :index_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :index_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IndexOperatorWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (arguments === other.arguments) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (block === other.block) &&
        (binary_operator === other.binary_operator) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of the `||=` operator on a call to `[]`.
  #
  #     foo.bar[baz] ||= value
  #     ^^^^^^^^^^^^^^^^^^^^^^
  class IndexOrWriteNode < Node
    # def initialize: (Integer flags, Prism::node? receiver, Location? call_operator_loc, Location opening_loc, ArgumentsNode? arguments, Location closing_loc, Prism::node? block, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @call_operator_loc = call_operator_loc
      @opening_loc = opening_loc
      @arguments = arguments
      @closing_loc = closing_loc
      @block = block
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_index_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, arguments, block, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver if receiver
      compact << arguments if arguments
      compact << block if block
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*receiver, *call_operator_loc, opening_loc, *arguments, closing_loc, *block, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node?, ?call_operator_loc: Location?, ?opening_loc: Location, ?arguments: ArgumentsNode?, ?closing_loc: Location, ?block: Prism::node?, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> IndexOrWriteNode
    def copy(flags: self.flags, receiver: self.receiver, call_operator_loc: self.call_operator_loc, opening_loc: self.opening_loc, arguments: self.arguments, closing_loc: self.closing_loc, block: self.block, operator_loc: self.operator_loc, value: self.value, location: self.location)
      IndexOrWriteNode.new(source, flags, receiver, call_operator_loc, opening_loc, arguments, closing_loc, block, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node?, call_operator_loc: Location?, opening_loc: Location, arguments: ArgumentsNode?, closing_loc: Location, block: Prism::node?, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, call_operator_loc: call_operator_loc, opening_loc: opening_loc, arguments: arguments, closing_loc: closing_loc, block: block, operator_loc: operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node?
    attr_reader :receiver

    # attr_reader call_operator_loc: Location?
    def call_operator_loc
      location = @call_operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @call_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def call_operator: () -> String?
    def call_operator
      call_operator_loc&.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :index_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :index_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IndexOrWriteNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (call_operator_loc.nil? == other.call_operator_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (arguments === other.arguments) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (block === other.block) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to an index.
  #
  #     foo[bar], = 1
  #     ^^^^^^^^
  #
  #     begin
  #     rescue => foo[bar]
  #               ^^^^^^^^
  #     end
  #
  #     for foo[bar] in baz do end
  #         ^^^^^^^^
  class IndexTargetNode < Node
    # def initialize: (Integer flags, Prism::node receiver, Location opening_loc, ArgumentsNode? arguments, Location closing_loc, Prism::node? block, Location location) -> void
    def initialize(source, flags, receiver, opening_loc, arguments, closing_loc, block, location)
      @source = source
      @location = location
      @flags = flags
      @receiver = receiver
      @opening_loc = opening_loc
      @arguments = arguments
      @closing_loc = closing_loc
      @block = block
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_index_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [receiver, arguments, block]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << receiver
      compact << arguments if arguments
      compact << block if block
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [receiver, opening_loc, *arguments, closing_loc, *block] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?receiver: Prism::node, ?opening_loc: Location, ?arguments: ArgumentsNode?, ?closing_loc: Location, ?block: Prism::node?, ?location: Location) -> IndexTargetNode
    def copy(flags: self.flags, receiver: self.receiver, opening_loc: self.opening_loc, arguments: self.arguments, closing_loc: self.closing_loc, block: self.block, location: self.location)
      IndexTargetNode.new(source, flags, receiver, opening_loc, arguments, closing_loc, block, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, receiver: Prism::node, opening_loc: Location, arguments: ArgumentsNode?, closing_loc: Location, block: Prism::node?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, receiver: receiver, opening_loc: opening_loc, arguments: arguments, closing_loc: closing_loc, block: block, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader receiver: Prism::node
    attr_reader :receiver

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # def safe_navigation?: () -> bool
    def safe_navigation?
      flags.anybits?(CallNodeFlags::SAFE_NAVIGATION)
    end

    # def variable_call?: () -> bool
    def variable_call?
      flags.anybits?(CallNodeFlags::VARIABLE_CALL)
    end

    # def attribute_write?: () -> bool
    def attribute_write?
      flags.anybits?(CallNodeFlags::ATTRIBUTE_WRITE)
    end

    # def ignore_visibility?: () -> bool
    def ignore_visibility?
      flags.anybits?(CallNodeFlags::IGNORE_VISIBILITY)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :index_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :index_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IndexTargetNode) &&
        (flags === other.flags) &&
        (receiver === other.receiver) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (arguments === other.arguments) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (block === other.block)
    end
  end

  # Represents the use of the `&&=` operator for assignment to an instance variable.
  #
  #     @target &&= value
  #     ^^^^^^^^^^^^^^^^^
  class InstanceVariableAndWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> InstanceVariableAndWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      InstanceVariableAndWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableAndWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents assigning to an instance variable using an operator that isn't `=`.
  #
  #     @target += value
  #     ^^^^^^^^^^^^^^^^
  class InstanceVariableOperatorWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location binary_operator_loc, Prism::node value, Symbol binary_operator, Location location) -> void
    def initialize(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @binary_operator_loc = binary_operator_loc
      @value = value
      @binary_operator = binary_operator
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?binary_operator_loc: Location, ?value: Prism::node, ?binary_operator: Symbol, ?location: Location) -> InstanceVariableOperatorWriteNode
    def copy(name: self.name, name_loc: self.name_loc, binary_operator_loc: self.binary_operator_loc, value: self.value, binary_operator: self.binary_operator, location: self.location)
      InstanceVariableOperatorWriteNode.new(source, name, name_loc, binary_operator_loc, value, binary_operator, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, binary_operator_loc: Location, value: Prism::node, binary_operator: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, binary_operator_loc: binary_operator_loc, value: value, binary_operator: binary_operator, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableOperatorWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (binary_operator === other.binary_operator)
    end
  end

  # Represents the use of the `||=` operator for assignment to an instance variable.
  #
  #     @target ||= value
  #     ^^^^^^^^^^^^^^^^^
  class InstanceVariableOrWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> InstanceVariableOrWriteNode
    def copy(name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      InstanceVariableOrWriteNode.new(source, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableOrWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents referencing an instance variable.
  #
  #     @foo
  #     ^^^^
  class InstanceVariableReadNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> InstanceVariableReadNode
    def copy(name: self.name, location: self.location)
      InstanceVariableReadNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # The name of the instance variable, which is a `@` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     @x     # name `:@x`
    #
    #     @_test # name `:@_test`
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableReadNode) &&
        (name === other.name)
    end
  end

  # Represents writing to an instance variable in a context that doesn't have an explicit value.
  #
  #     @foo, @bar = baz
  #     ^^^^  ^^^^
  class InstanceVariableTargetNode < Node
    # def initialize: (Symbol name, Location location) -> void
    def initialize(source, name, location)
      @source = source
      @location = location
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?location: Location) -> InstanceVariableTargetNode
    def copy(name: self.name, location: self.location)
      InstanceVariableTargetNode.new(source, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { name: name, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableTargetNode) &&
        (name === other.name)
    end
  end

  # Represents writing to an instance variable.
  #
  #     @foo = 1
  #     ^^^^^^^^
  class InstanceVariableWriteNode < Node
    # def initialize: (Symbol name, Location name_loc, Prism::node value, Location operator_loc, Location location) -> void
    def initialize(source, name, name_loc, value, operator_loc, location)
      @source = source
      @location = location
      @name = name
      @name_loc = name_loc
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_instance_variable_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?name_loc: Location, ?value: Prism::node, ?operator_loc: Location, ?location: Location) -> InstanceVariableWriteNode
    def copy(name: self.name, name_loc: self.name_loc, value: self.value, operator_loc: self.operator_loc, location: self.location)
      InstanceVariableWriteNode.new(source, name, name_loc, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, name_loc: Location, value: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { name: name, name_loc: name_loc, value: value, operator_loc: operator_loc, location: location }
    end

    # The name of the instance variable, which is a `@` followed by an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     @x = :y       # name `:@x`
    #
    #     @_foo = "bar" # name `@_foo`
    attr_reader :name

    # The location of the variable name.
    #
    #     @_x = 1
    #     ^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the instance variable. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     @foo = :bar
    #            ^^^^
    #
    #     @_x = 1234
    #           ^^^^
    attr_reader :value

    # The location of the `=` operator.
    #
    #     @x = y
    #        ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :instance_variable_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :instance_variable_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InstanceVariableWriteNode) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents an integer number literal.
  #
  #     1
  #     ^
  class IntegerNode < Node
    # def initialize: (Integer flags, Integer value, Location location) -> void
    def initialize(source, flags, value, location)
      @source = source
      @location = location
      @flags = flags
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_integer_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?value: Integer, ?location: Location) -> IntegerNode
    def copy(flags: self.flags, value: self.value, location: self.location)
      IntegerNode.new(source, flags, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, value: Integer, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # The value of the integer literal as a number.
    attr_reader :value

    # def binary?: () -> bool
    def binary?
      flags.anybits?(IntegerBaseFlags::BINARY)
    end

    # def decimal?: () -> bool
    def decimal?
      flags.anybits?(IntegerBaseFlags::DECIMAL)
    end

    # def octal?: () -> bool
    def octal?
      flags.anybits?(IntegerBaseFlags::OCTAL)
    end

    # def hexadecimal?: () -> bool
    def hexadecimal?
      flags.anybits?(IntegerBaseFlags::HEXADECIMAL)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :integer_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :integer_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(IntegerNode) &&
        (flags === other.flags) &&
        (value === other.value)
    end
  end

  # Represents a regular expression literal that contains interpolation that is being used in the predicate of a conditional to implicitly match against the last line read by an IO object.
  #
  #     if /foo #{bar} baz/ then end
  #        ^^^^^^^^^^^^^^^^
  class InterpolatedMatchLastLineNode < Node
    # def initialize: (Integer flags, Location opening_loc, Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode] parts, Location closing_loc, Location location) -> void
    def initialize(source, flags, opening_loc, parts, closing_loc, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @parts = parts
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_interpolated_match_last_line_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*parts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*parts]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, *parts, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location, ?parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], ?closing_loc: Location, ?location: Location) -> InterpolatedMatchLastLineNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, parts: self.parts, closing_loc: self.closing_loc, location: self.location)
      InterpolatedMatchLastLineNode.new(source, flags, opening_loc, parts, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location, parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, parts: parts, closing_loc: closing_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode]
    attr_reader :parts

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def ignore_case?: () -> bool
    def ignore_case?
      flags.anybits?(RegularExpressionFlags::IGNORE_CASE)
    end

    # def extended?: () -> bool
    def extended?
      flags.anybits?(RegularExpressionFlags::EXTENDED)
    end

    # def multi_line?: () -> bool
    def multi_line?
      flags.anybits?(RegularExpressionFlags::MULTI_LINE)
    end

    # def once?: () -> bool
    def once?
      flags.anybits?(RegularExpressionFlags::ONCE)
    end

    # def euc_jp?: () -> bool
    def euc_jp?
      flags.anybits?(RegularExpressionFlags::EUC_JP)
    end

    # def ascii_8bit?: () -> bool
    def ascii_8bit?
      flags.anybits?(RegularExpressionFlags::ASCII_8BIT)
    end

    # def windows_31j?: () -> bool
    def windows_31j?
      flags.anybits?(RegularExpressionFlags::WINDOWS_31J)
    end

    # def utf_8?: () -> bool
    def utf_8?
      flags.anybits?(RegularExpressionFlags::UTF_8)
    end

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_BINARY_ENCODING)
    end

    # def forced_us_ascii_encoding?: () -> bool
    def forced_us_ascii_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_US_ASCII_ENCODING)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :interpolated_match_last_line_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :interpolated_match_last_line_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InterpolatedMatchLastLineNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (parts.length == other.parts.length) &&
        parts.zip(other.parts).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a regular expression literal that contains interpolation.
  #
  #     /foo #{bar} baz/
  #     ^^^^^^^^^^^^^^^^
  class InterpolatedRegularExpressionNode < Node
    # def initialize: (Integer flags, Location opening_loc, Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode] parts, Location closing_loc, Location location) -> void
    def initialize(source, flags, opening_loc, parts, closing_loc, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @parts = parts
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_interpolated_regular_expression_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*parts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*parts]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, *parts, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location, ?parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], ?closing_loc: Location, ?location: Location) -> InterpolatedRegularExpressionNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, parts: self.parts, closing_loc: self.closing_loc, location: self.location)
      InterpolatedRegularExpressionNode.new(source, flags, opening_loc, parts, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location, parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, parts: parts, closing_loc: closing_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode]
    attr_reader :parts

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def ignore_case?: () -> bool
    def ignore_case?
      flags.anybits?(RegularExpressionFlags::IGNORE_CASE)
    end

    # def extended?: () -> bool
    def extended?
      flags.anybits?(RegularExpressionFlags::EXTENDED)
    end

    # def multi_line?: () -> bool
    def multi_line?
      flags.anybits?(RegularExpressionFlags::MULTI_LINE)
    end

    # def once?: () -> bool
    def once?
      flags.anybits?(RegularExpressionFlags::ONCE)
    end

    # def euc_jp?: () -> bool
    def euc_jp?
      flags.anybits?(RegularExpressionFlags::EUC_JP)
    end

    # def ascii_8bit?: () -> bool
    def ascii_8bit?
      flags.anybits?(RegularExpressionFlags::ASCII_8BIT)
    end

    # def windows_31j?: () -> bool
    def windows_31j?
      flags.anybits?(RegularExpressionFlags::WINDOWS_31J)
    end

    # def utf_8?: () -> bool
    def utf_8?
      flags.anybits?(RegularExpressionFlags::UTF_8)
    end

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_BINARY_ENCODING)
    end

    # def forced_us_ascii_encoding?: () -> bool
    def forced_us_ascii_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_US_ASCII_ENCODING)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :interpolated_regular_expression_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :interpolated_regular_expression_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InterpolatedRegularExpressionNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (parts.length == other.parts.length) &&
        parts.zip(other.parts).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a string literal that contains interpolation.
  #
  #     "foo #{bar} baz"
  #     ^^^^^^^^^^^^^^^^
  class InterpolatedStringNode < Node
    # def initialize: (Integer flags, Location? opening_loc, Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode | InterpolatedStringNode] parts, Location? closing_loc, Location location) -> void
    def initialize(source, flags, opening_loc, parts, closing_loc, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @parts = parts
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_interpolated_string_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*parts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*parts]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*opening_loc, *parts, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location?, ?parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode | InterpolatedStringNode], ?closing_loc: Location?, ?location: Location) -> InterpolatedStringNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, parts: self.parts, closing_loc: self.closing_loc, location: self.location)
      InterpolatedStringNode.new(source, flags, opening_loc, parts, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location?, parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode | InterpolatedStringNode], closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, parts: parts, closing_loc: closing_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode | InterpolatedStringNode]
    attr_reader :parts

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def frozen?: () -> bool
    def frozen?
      flags.anybits?(InterpolatedStringNodeFlags::FROZEN)
    end

    # def mutable?: () -> bool
    def mutable?
      flags.anybits?(InterpolatedStringNodeFlags::MUTABLE)
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :interpolated_string_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :interpolated_string_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InterpolatedStringNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (parts.length == other.parts.length) &&
        parts.zip(other.parts).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents a symbol literal that contains interpolation.
  #
  #     :"foo #{bar} baz"
  #     ^^^^^^^^^^^^^^^^^
  class InterpolatedSymbolNode < Node
    # def initialize: (Location? opening_loc, Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode] parts, Location? closing_loc, Location location) -> void
    def initialize(source, opening_loc, parts, closing_loc, location)
      @source = source
      @location = location
      @opening_loc = opening_loc
      @parts = parts
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_interpolated_symbol_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*parts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*parts]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*opening_loc, *parts, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?opening_loc: Location?, ?parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], ?closing_loc: Location?, ?location: Location) -> InterpolatedSymbolNode
    def copy(opening_loc: self.opening_loc, parts: self.parts, closing_loc: self.closing_loc, location: self.location)
      InterpolatedSymbolNode.new(source, opening_loc, parts, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { opening_loc: Location?, parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], closing_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { opening_loc: opening_loc, parts: parts, closing_loc: closing_loc, location: location }
    end

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode]
    attr_reader :parts

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :interpolated_symbol_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :interpolated_symbol_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InterpolatedSymbolNode) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (parts.length == other.parts.length) &&
        parts.zip(other.parts).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents an xstring literal that contains interpolation.
  #
  #     `foo #{bar} baz`
  #     ^^^^^^^^^^^^^^^^
  class InterpolatedXStringNode < Node
    # def initialize: (Location opening_loc, Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode] parts, Location closing_loc, Location location) -> void
    def initialize(source, opening_loc, parts, closing_loc, location)
      @source = source
      @location = location
      @opening_loc = opening_loc
      @parts = parts
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_interpolated_x_string_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*parts]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*parts]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, *parts, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?opening_loc: Location, ?parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], ?closing_loc: Location, ?location: Location) -> InterpolatedXStringNode
    def copy(opening_loc: self.opening_loc, parts: self.parts, closing_loc: self.closing_loc, location: self.location)
      InterpolatedXStringNode.new(source, opening_loc, parts, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { opening_loc: Location, parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode], closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { opening_loc: opening_loc, parts: parts, closing_loc: closing_loc, location: location }
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader parts: Array[StringNode | EmbeddedStatementsNode | EmbeddedVariableNode]
    attr_reader :parts

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :interpolated_x_string_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :interpolated_x_string_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(InterpolatedXStringNode) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (parts.length == other.parts.length) &&
        parts.zip(other.parts).all? { |left, right| left === right } &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents an implicit set of parameters through the use of the `it` keyword within a block or lambda.
  #
  #     -> { it + it }
  #     ^^^^^^^^^^^^^^
  class ItParametersNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_it_parameters_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> ItParametersNode
    def copy(location: self.location)
      ItParametersNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :it_parameters_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :it_parameters_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ItParametersNode)
    end
  end

  # Represents a hash literal without opening and closing braces.
  #
  #     foo(a: b)
  #         ^^^^
  class KeywordHashNode < Node
    # def initialize: (Integer flags, Array[AssocNode | AssocSplatNode] elements, Location location) -> void
    def initialize(source, flags, elements, location)
      @source = source
      @location = location
      @flags = flags
      @elements = elements
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_keyword_hash_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*elements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*elements]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*elements] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?elements: Array[AssocNode | AssocSplatNode], ?location: Location) -> KeywordHashNode
    def copy(flags: self.flags, elements: self.elements, location: self.location)
      KeywordHashNode.new(source, flags, elements, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, elements: Array[AssocNode | AssocSplatNode], location: Location }
    def deconstruct_keys(keys)
      { flags: flags, elements: elements, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader elements: Array[AssocNode | AssocSplatNode]
    attr_reader :elements

    # def symbol_keys?: () -> bool
    def symbol_keys?
      flags.anybits?(KeywordHashNodeFlags::SYMBOL_KEYS)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :keyword_hash_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :keyword_hash_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(KeywordHashNode) &&
        (flags === other.flags) &&
        (elements.length == other.elements.length) &&
        elements.zip(other.elements).all? { |left, right| left === right }
    end
  end

  # Represents a keyword rest parameter to a method, block, or lambda definition.
  #
  #     def a(**b)
  #           ^^^
  #     end
  class KeywordRestParameterNode < Node
    # def initialize: (Integer flags, Symbol? name, Location? name_loc, Location operator_loc, Location location) -> void
    def initialize(source, flags, name, name_loc, operator_loc, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_keyword_rest_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*name_loc, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol?, ?name_loc: Location?, ?operator_loc: Location, ?location: Location) -> KeywordRestParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, location: self.location)
      KeywordRestParameterNode.new(source, flags, name, name_loc, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol?, name_loc: Location?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, operator_loc: operator_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol?
    attr_reader :name

    # attr_reader name_loc: Location?
    def name_loc
      location = @name_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :keyword_rest_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :keyword_rest_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(KeywordRestParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents using a lambda literal (not the lambda method call).
  #
  #     ->(value) { value * 2 }
  #     ^^^^^^^^^^^^^^^^^^^^^^^
  class LambdaNode < Node
    # def initialize: (Array[Symbol] locals, Location operator_loc, Location opening_loc, Location closing_loc, Prism::node? parameters, Prism::node? body, Location location) -> void
    def initialize(source, locals, operator_loc, opening_loc, closing_loc, parameters, body, location)
      @source = source
      @location = location
      @locals = locals
      @operator_loc = operator_loc
      @opening_loc = opening_loc
      @closing_loc = closing_loc
      @parameters = parameters
      @body = body
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_lambda_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [parameters, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << parameters if parameters
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [operator_loc, opening_loc, closing_loc, *parameters, *body] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?operator_loc: Location, ?opening_loc: Location, ?closing_loc: Location, ?parameters: Prism::node?, ?body: Prism::node?, ?location: Location) -> LambdaNode
    def copy(locals: self.locals, operator_loc: self.operator_loc, opening_loc: self.opening_loc, closing_loc: self.closing_loc, parameters: self.parameters, body: self.body, location: self.location)
      LambdaNode.new(source, locals, operator_loc, opening_loc, closing_loc, parameters, body, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], operator_loc: Location, opening_loc: Location, closing_loc: Location, parameters: Prism::node?, body: Prism::node?, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, operator_loc: operator_loc, opening_loc: opening_loc, closing_loc: closing_loc, parameters: parameters, body: body, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader parameters: Prism::node?
    attr_reader :parameters

    # attr_reader body: Prism::node?
    attr_reader :body

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :lambda_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :lambda_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LambdaNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (parameters === other.parameters) &&
        (body === other.body)
    end
  end

  # Represents the use of the `&&=` operator for assignment to a local variable.
  #
  #     target &&= value
  #     ^^^^^^^^^^^^^^^^
  class LocalVariableAndWriteNode < Node
    # def initialize: (Location name_loc, Location operator_loc, Prism::node value, Symbol name, Integer depth, Location location) -> void
    def initialize(source, name_loc, operator_loc, value, name, depth, location)
      @source = source
      @location = location
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
      @name = name
      @depth = depth
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_and_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?name: Symbol, ?depth: Integer, ?location: Location) -> LocalVariableAndWriteNode
    def copy(name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, name: self.name, depth: self.depth, location: self.location)
      LocalVariableAndWriteNode.new(source, name_loc, operator_loc, value, name, depth, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name_loc: Location, operator_loc: Location, value: Prism::node, name: Symbol, depth: Integer, location: Location }
    def deconstruct_keys(keys)
      { name_loc: name_loc, operator_loc: operator_loc, value: value, name: name, depth: depth, location: location }
    end

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader depth: Integer
    attr_reader :depth

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_and_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_and_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableAndWriteNode) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value) &&
        (name === other.name) &&
        (depth === other.depth)
    end
  end

  # Represents assigning to a local variable using an operator that isn't `=`.
  #
  #     target += value
  #     ^^^^^^^^^^^^^^^
  class LocalVariableOperatorWriteNode < Node
    # def initialize: (Location name_loc, Location binary_operator_loc, Prism::node value, Symbol name, Symbol binary_operator, Integer depth, Location location) -> void
    def initialize(source, name_loc, binary_operator_loc, value, name, binary_operator, depth, location)
      @source = source
      @location = location
      @name_loc = name_loc
      @binary_operator_loc = binary_operator_loc
      @value = value
      @name = name
      @binary_operator = binary_operator
      @depth = depth
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_operator_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, binary_operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name_loc: Location, ?binary_operator_loc: Location, ?value: Prism::node, ?name: Symbol, ?binary_operator: Symbol, ?depth: Integer, ?location: Location) -> LocalVariableOperatorWriteNode
    def copy(name_loc: self.name_loc, binary_operator_loc: self.binary_operator_loc, value: self.value, name: self.name, binary_operator: self.binary_operator, depth: self.depth, location: self.location)
      LocalVariableOperatorWriteNode.new(source, name_loc, binary_operator_loc, value, name, binary_operator, depth, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name_loc: Location, binary_operator_loc: Location, value: Prism::node, name: Symbol, binary_operator: Symbol, depth: Integer, location: Location }
    def deconstruct_keys(keys)
      { name_loc: name_loc, binary_operator_loc: binary_operator_loc, value: value, name: name, binary_operator: binary_operator, depth: depth, location: location }
    end

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader binary_operator_loc: Location
    def binary_operator_loc
      location = @binary_operator_loc
      return location if location.is_a?(Location)
      @binary_operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader binary_operator: Symbol
    attr_reader :binary_operator

    # attr_reader depth: Integer
    attr_reader :depth

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_operator_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_operator_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableOperatorWriteNode) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (binary_operator_loc.nil? == other.binary_operator_loc.nil?) &&
        (value === other.value) &&
        (name === other.name) &&
        (binary_operator === other.binary_operator) &&
        (depth === other.depth)
    end
  end

  # Represents the use of the `||=` operator for assignment to a local variable.
  #
  #     target ||= value
  #     ^^^^^^^^^^^^^^^^
  class LocalVariableOrWriteNode < Node
    # def initialize: (Location name_loc, Location operator_loc, Prism::node value, Symbol name, Integer depth, Location location) -> void
    def initialize(source, name_loc, operator_loc, value, name, depth, location)
      @source = source
      @location = location
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
      @name = name
      @depth = depth
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_or_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?name: Symbol, ?depth: Integer, ?location: Location) -> LocalVariableOrWriteNode
    def copy(name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, name: self.name, depth: self.depth, location: self.location)
      LocalVariableOrWriteNode.new(source, name_loc, operator_loc, value, name, depth, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name_loc: Location, operator_loc: Location, value: Prism::node, name: Symbol, depth: Integer, location: Location }
    def deconstruct_keys(keys)
      { name_loc: name_loc, operator_loc: operator_loc, value: value, name: name, depth: depth, location: location }
    end

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader depth: Integer
    attr_reader :depth

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_or_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_or_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableOrWriteNode) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value) &&
        (name === other.name) &&
        (depth === other.depth)
    end
  end

  # Represents reading a local variable. Note that this requires that a local variable of the same name has already been written to in the same scope, otherwise it is parsed as a method call.
  #
  #     foo
  #     ^^^
  class LocalVariableReadNode < Node
    # def initialize: (Symbol name, Integer depth, Location location) -> void
    def initialize(source, name, depth, location)
      @source = source
      @location = location
      @name = name
      @depth = depth
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?depth: Integer, ?location: Location) -> LocalVariableReadNode
    def copy(name: self.name, depth: self.depth, location: self.location)
      LocalVariableReadNode.new(source, name, depth, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, depth: Integer, location: Location }
    def deconstruct_keys(keys)
      { name: name, depth: depth, location: location }
    end

    # The name of the local variable, which is an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     x      # name `:x`
    #
    #     _Test  # name `:_Test`
    #
    # Note that this can also be an underscore followed by a number for the default block parameters.
    #
    #     _1     # name `:_1`
    #
    # Finally, for the default `it` block parameter, the name is `0it`. This is to distinguish it from an `it` local variable that is explicitly declared.
    #
    #     it     # name `:0it`
    attr_reader :name

    # The number of visible scopes that should be searched to find the origin of this local variable.
    #
    #     foo = 1; foo # depth 0
    #
    #     bar = 2; tap { bar } # depth 1
    #
    # The specific rules for calculating the depth may differ from individual Ruby implementations, as they are not specified by the language. For more information, see [the Prism documentation](https://github.com/ruby/prism/blob/main/docs/local_variable_depth.md).
    attr_reader :depth

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableReadNode) &&
        (name === other.name) &&
        (depth === other.depth)
    end
  end

  # Represents writing to a local variable in a context that doesn't have an explicit value.
  #
  #     foo, bar = baz
  #     ^^^  ^^^
  class LocalVariableTargetNode < Node
    # def initialize: (Symbol name, Integer depth, Location location) -> void
    def initialize(source, name, depth, location)
      @source = source
      @location = location
      @name = name
      @depth = depth
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?depth: Integer, ?location: Location) -> LocalVariableTargetNode
    def copy(name: self.name, depth: self.depth, location: self.location)
      LocalVariableTargetNode.new(source, name, depth, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, depth: Integer, location: Location }
    def deconstruct_keys(keys)
      { name: name, depth: depth, location: location }
    end

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader depth: Integer
    attr_reader :depth

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableTargetNode) &&
        (name === other.name) &&
        (depth === other.depth)
    end
  end

  # Represents writing to a local variable.
  #
  #     foo = 1
  #     ^^^^^^^
  class LocalVariableWriteNode < Node
    # def initialize: (Symbol name, Integer depth, Location name_loc, Prism::node value, Location operator_loc, Location location) -> void
    def initialize(source, name, depth, name_loc, value, operator_loc, location)
      @source = source
      @location = location
      @name = name
      @depth = depth
      @name_loc = name_loc
      @value = value
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_local_variable_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?name: Symbol, ?depth: Integer, ?name_loc: Location, ?value: Prism::node, ?operator_loc: Location, ?location: Location) -> LocalVariableWriteNode
    def copy(name: self.name, depth: self.depth, name_loc: self.name_loc, value: self.value, operator_loc: self.operator_loc, location: self.location)
      LocalVariableWriteNode.new(source, name, depth, name_loc, value, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { name: Symbol, depth: Integer, name_loc: Location, value: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { name: name, depth: depth, name_loc: name_loc, value: value, operator_loc: operator_loc, location: location }
    end

    # The name of the local variable, which is an [identifier](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#identifiers).
    #
    #     foo = :bar # name `:foo`
    #
    #     abc = 123  # name `:abc`
    attr_reader :name

    # The number of semantic scopes we have to traverse to find the declaration of this variable.
    #
    #     foo = 1         # depth 0
    #
    #     tap { foo = 1 } # depth 1
    #
    # The specific rules for calculating the depth may differ from individual Ruby implementations, as they are not specified by the language. For more information, see [the Prism documentation](https://github.com/ruby/prism/blob/main/docs/local_variable_depth.md).
    attr_reader :depth

    # The location of the variable name.
    #
    #     foo = :bar
    #     ^^^
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The value to write to the local variable. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     foo = :bar
    #           ^^^^
    #
    #     abc = 1234
    #           ^^^^
    #
    # Note that since the name of a local variable is known before the value is parsed, it is valid for a local variable to appear within the value of its own write.
    #
    #     foo = foo
    attr_reader :value

    # The location of the `=` operator.
    #
    #     x = :y
    #       ^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :local_variable_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :local_variable_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(LocalVariableWriteNode) &&
        (name === other.name) &&
        (depth === other.depth) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a regular expression literal used in the predicate of a conditional to implicitly match against the last line read by an IO object.
  #
  #     if /foo/i then end
  #        ^^^^^^
  class MatchLastLineNode < Node
    # def initialize: (Integer flags, Location opening_loc, Location content_loc, Location closing_loc, String unescaped, Location location) -> void
    def initialize(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @content_loc = content_loc
      @closing_loc = closing_loc
      @unescaped = unescaped
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_match_last_line_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, content_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location, ?content_loc: Location, ?closing_loc: Location, ?unescaped: String, ?location: Location) -> MatchLastLineNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, content_loc: self.content_loc, closing_loc: self.closing_loc, unescaped: self.unescaped, location: self.location)
      MatchLastLineNode.new(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location, content_loc: Location, closing_loc: Location, unescaped: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, content_loc: content_loc, closing_loc: closing_loc, unescaped: unescaped, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader content_loc: Location
    def content_loc
      location = @content_loc
      return location if location.is_a?(Location)
      @content_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader unescaped: String
    attr_reader :unescaped

    # def ignore_case?: () -> bool
    def ignore_case?
      flags.anybits?(RegularExpressionFlags::IGNORE_CASE)
    end

    # def extended?: () -> bool
    def extended?
      flags.anybits?(RegularExpressionFlags::EXTENDED)
    end

    # def multi_line?: () -> bool
    def multi_line?
      flags.anybits?(RegularExpressionFlags::MULTI_LINE)
    end

    # def once?: () -> bool
    def once?
      flags.anybits?(RegularExpressionFlags::ONCE)
    end

    # def euc_jp?: () -> bool
    def euc_jp?
      flags.anybits?(RegularExpressionFlags::EUC_JP)
    end

    # def ascii_8bit?: () -> bool
    def ascii_8bit?
      flags.anybits?(RegularExpressionFlags::ASCII_8BIT)
    end

    # def windows_31j?: () -> bool
    def windows_31j?
      flags.anybits?(RegularExpressionFlags::WINDOWS_31J)
    end

    # def utf_8?: () -> bool
    def utf_8?
      flags.anybits?(RegularExpressionFlags::UTF_8)
    end

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_BINARY_ENCODING)
    end

    # def forced_us_ascii_encoding?: () -> bool
    def forced_us_ascii_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_US_ASCII_ENCODING)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def content: () -> String
    def content
      content_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :match_last_line_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :match_last_line_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MatchLastLineNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (content_loc.nil? == other.content_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (unescaped === other.unescaped)
    end
  end

  # Represents the use of the modifier `in` operator.
  #
  #     foo in bar
  #     ^^^^^^^^^^
  class MatchPredicateNode < Node
    # def initialize: (Prism::node value, Prism::node pattern, Location operator_loc, Location location) -> void
    def initialize(source, value, pattern, operator_loc, location)
      @source = source
      @location = location
      @value = value
      @pattern = pattern
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_match_predicate_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value, pattern]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value, pattern]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [value, pattern, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Prism::node, ?pattern: Prism::node, ?operator_loc: Location, ?location: Location) -> MatchPredicateNode
    def copy(value: self.value, pattern: self.pattern, operator_loc: self.operator_loc, location: self.location)
      MatchPredicateNode.new(source, value, pattern, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Prism::node, pattern: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { value: value, pattern: pattern, operator_loc: operator_loc, location: location }
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader pattern: Prism::node
    attr_reader :pattern

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :match_predicate_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :match_predicate_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MatchPredicateNode) &&
        (value === other.value) &&
        (pattern === other.pattern) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of the `=>` operator.
  #
  #     foo => bar
  #     ^^^^^^^^^^
  class MatchRequiredNode < Node
    # def initialize: (Prism::node value, Prism::node pattern, Location operator_loc, Location location) -> void
    def initialize(source, value, pattern, operator_loc, location)
      @source = source
      @location = location
      @value = value
      @pattern = pattern
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_match_required_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value, pattern]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value, pattern]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [value, pattern, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?value: Prism::node, ?pattern: Prism::node, ?operator_loc: Location, ?location: Location) -> MatchRequiredNode
    def copy(value: self.value, pattern: self.pattern, operator_loc: self.operator_loc, location: self.location)
      MatchRequiredNode.new(source, value, pattern, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { value: Prism::node, pattern: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { value: value, pattern: pattern, operator_loc: operator_loc, location: location }
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # attr_reader pattern: Prism::node
    attr_reader :pattern

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :match_required_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :match_required_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MatchRequiredNode) &&
        (value === other.value) &&
        (pattern === other.pattern) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents writing local variables using a regular expression match with named capture groups.
  #
  #     /(?<foo>bar)/ =~ baz
  #     ^^^^^^^^^^^^^^^^^^^^
  class MatchWriteNode < Node
    # def initialize: (CallNode call, Array[LocalVariableTargetNode] targets, Location location) -> void
    def initialize(source, call, targets, location)
      @source = source
      @location = location
      @call = call
      @targets = targets
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_match_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [call, *targets]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [call, *targets]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [call, *targets] #: Array[Prism::node | Location]
    end

    # def copy: (?call: CallNode, ?targets: Array[LocalVariableTargetNode], ?location: Location) -> MatchWriteNode
    def copy(call: self.call, targets: self.targets, location: self.location)
      MatchWriteNode.new(source, call, targets, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { call: CallNode, targets: Array[LocalVariableTargetNode], location: Location }
    def deconstruct_keys(keys)
      { call: call, targets: targets, location: location }
    end

    # attr_reader call: CallNode
    attr_reader :call

    # attr_reader targets: Array[LocalVariableTargetNode]
    attr_reader :targets

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :match_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :match_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MatchWriteNode) &&
        (call === other.call) &&
        (targets.length == other.targets.length) &&
        targets.zip(other.targets).all? { |left, right| left === right }
    end
  end

  # Represents a node that is missing from the source and results in a syntax error.
  class MissingNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_missing_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> MissingNode
    def copy(location: self.location)
      MissingNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :missing_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :missing_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MissingNode)
    end
  end

  # Represents a module declaration involving the `module` keyword.
  #
  #     module Foo end
  #     ^^^^^^^^^^^^^^
  class ModuleNode < Node
    # def initialize: (Array[Symbol] locals, Location module_keyword_loc, Prism::node constant_path, Prism::node? body, Location end_keyword_loc, Symbol name, Location location) -> void
    def initialize(source, locals, module_keyword_loc, constant_path, body, end_keyword_loc, name, location)
      @source = source
      @location = location
      @locals = locals
      @module_keyword_loc = module_keyword_loc
      @constant_path = constant_path
      @body = body
      @end_keyword_loc = end_keyword_loc
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_module_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [constant_path, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << constant_path
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [module_keyword_loc, constant_path, *body, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?module_keyword_loc: Location, ?constant_path: Prism::node, ?body: Prism::node?, ?end_keyword_loc: Location, ?name: Symbol, ?location: Location) -> ModuleNode
    def copy(locals: self.locals, module_keyword_loc: self.module_keyword_loc, constant_path: self.constant_path, body: self.body, end_keyword_loc: self.end_keyword_loc, name: self.name, location: self.location)
      ModuleNode.new(source, locals, module_keyword_loc, constant_path, body, end_keyword_loc, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], module_keyword_loc: Location, constant_path: Prism::node, body: Prism::node?, end_keyword_loc: Location, name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, module_keyword_loc: module_keyword_loc, constant_path: constant_path, body: body, end_keyword_loc: end_keyword_loc, name: name, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader module_keyword_loc: Location
    def module_keyword_loc
      location = @module_keyword_loc
      return location if location.is_a?(Location)
      @module_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader constant_path: Prism::node
    attr_reader :constant_path

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader name: Symbol
    attr_reader :name

    # def module_keyword: () -> String
    def module_keyword
      module_keyword_loc.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :module_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :module_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ModuleNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (module_keyword_loc.nil? == other.module_keyword_loc.nil?) &&
        (constant_path === other.constant_path) &&
        (body === other.body) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?) &&
        (name === other.name)
    end
  end

  # Represents a multi-target expression.
  #
  #     a, (b, c) = 1, 2, 3
  #        ^^^^^^
  class MultiTargetNode < Node
    # def initialize: (Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode | NumberedReferenceReadNode] lefts, Prism::node? rest, Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode] rights, Location? lparen_loc, Location? rparen_loc, Location location) -> void
    def initialize(source, lefts, rest, rights, lparen_loc, rparen_loc, location)
      @source = source
      @location = location
      @lefts = lefts
      @rest = rest
      @rights = rights
      @lparen_loc = lparen_loc
      @rparen_loc = rparen_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_multi_target_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*lefts, rest, *rights]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact.concat(lefts)
      compact << rest if rest
      compact.concat(rights)
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*lefts, *rest, *rights, *lparen_loc, *rparen_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode | NumberedReferenceReadNode], ?rest: Prism::node?, ?rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode], ?lparen_loc: Location?, ?rparen_loc: Location?, ?location: Location) -> MultiTargetNode
    def copy(lefts: self.lefts, rest: self.rest, rights: self.rights, lparen_loc: self.lparen_loc, rparen_loc: self.rparen_loc, location: self.location)
      MultiTargetNode.new(source, lefts, rest, rights, lparen_loc, rparen_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode | NumberedReferenceReadNode], rest: Prism::node?, rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode], lparen_loc: Location?, rparen_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { lefts: lefts, rest: rest, rights: rights, lparen_loc: lparen_loc, rparen_loc: rparen_loc, location: location }
    end

    # attr_reader lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode | NumberedReferenceReadNode]
    attr_reader :lefts

    # attr_reader rest: Prism::node?
    attr_reader :rest

    # attr_reader rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode | RequiredParameterNode | BackReferenceReadNode]
    attr_reader :rights

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :multi_target_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :multi_target_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MultiTargetNode) &&
        (lefts.length == other.lefts.length) &&
        lefts.zip(other.lefts).all? { |left, right| left === right } &&
        (rest === other.rest) &&
        (rights.length == other.rights.length) &&
        rights.zip(other.rights).all? { |left, right| left === right } &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (rparen_loc.nil? == other.rparen_loc.nil?)
    end
  end

  # Represents a write to a multi-target expression.
  #
  #     a, b, c = 1, 2, 3
  #     ^^^^^^^^^^^^^^^^^
  class MultiWriteNode < Node
    # def initialize: (Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode] lefts, Prism::node? rest, Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode] rights, Location? lparen_loc, Location? rparen_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, lefts, rest, rights, lparen_loc, rparen_loc, operator_loc, value, location)
      @source = source
      @location = location
      @lefts = lefts
      @rest = rest
      @rights = rights
      @lparen_loc = lparen_loc
      @rparen_loc = rparen_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_multi_write_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*lefts, rest, *rights, value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact.concat(lefts)
      compact << rest if rest
      compact.concat(rights)
      compact << value
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*lefts, *rest, *rights, *lparen_loc, *rparen_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode], ?rest: Prism::node?, ?rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode], ?lparen_loc: Location?, ?rparen_loc: Location?, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> MultiWriteNode
    def copy(lefts: self.lefts, rest: self.rest, rights: self.rights, lparen_loc: self.lparen_loc, rparen_loc: self.rparen_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      MultiWriteNode.new(source, lefts, rest, rights, lparen_loc, rparen_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode], rest: Prism::node?, rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode], lparen_loc: Location?, rparen_loc: Location?, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { lefts: lefts, rest: rest, rights: rights, lparen_loc: lparen_loc, rparen_loc: rparen_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # attr_reader lefts: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode]
    attr_reader :lefts

    # attr_reader rest: Prism::node?
    attr_reader :rest

    # attr_reader rights: Array[LocalVariableTargetNode | InstanceVariableTargetNode | ClassVariableTargetNode | GlobalVariableTargetNode | ConstantTargetNode | ConstantPathTargetNode | CallTargetNode | IndexTargetNode | MultiTargetNode]
    attr_reader :rights

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :multi_write_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :multi_write_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(MultiWriteNode) &&
        (lefts.length == other.lefts.length) &&
        lefts.zip(other.lefts).all? { |left, right| left === right } &&
        (rest === other.rest) &&
        (rights.length == other.rights.length) &&
        rights.zip(other.rights).all? { |left, right| left === right } &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (rparen_loc.nil? == other.rparen_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of the `next` keyword.
  #
  #     next 1
  #     ^^^^^^
  class NextNode < Node
    # def initialize: (ArgumentsNode? arguments, Location keyword_loc, Location location) -> void
    def initialize(source, arguments, keyword_loc, location)
      @source = source
      @location = location
      @arguments = arguments
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_next_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [arguments]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << arguments if arguments
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*arguments, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?arguments: ArgumentsNode?, ?keyword_loc: Location, ?location: Location) -> NextNode
    def copy(arguments: self.arguments, keyword_loc: self.keyword_loc, location: self.location)
      NextNode.new(source, arguments, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { arguments: ArgumentsNode?, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { arguments: arguments, keyword_loc: keyword_loc, location: location }
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :next_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :next_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(NextNode) &&
        (arguments === other.arguments) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents the use of the `nil` keyword.
  #
  #     nil
  #     ^^^
  class NilNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_nil_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> NilNode
    def copy(location: self.location)
      NilNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :nil_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :nil_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(NilNode)
    end
  end

  # Represents the use of `**nil` inside method arguments.
  #
  #     def a(**nil)
  #           ^^^^^
  #     end
  class NoKeywordsParameterNode < Node
    # def initialize: (Location operator_loc, Location keyword_loc, Location location) -> void
    def initialize(source, operator_loc, keyword_loc, location)
      @source = source
      @location = location
      @operator_loc = operator_loc
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_no_keywords_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [operator_loc, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?operator_loc: Location, ?keyword_loc: Location, ?location: Location) -> NoKeywordsParameterNode
    def copy(operator_loc: self.operator_loc, keyword_loc: self.keyword_loc, location: self.location)
      NoKeywordsParameterNode.new(source, operator_loc, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { operator_loc: Location, keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { operator_loc: operator_loc, keyword_loc: keyword_loc, location: location }
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :no_keywords_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :no_keywords_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(NoKeywordsParameterNode) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents an implicit set of parameters through the use of numbered parameters within a block or lambda.
  #
  #     -> { _1 + _2 }
  #     ^^^^^^^^^^^^^^
  class NumberedParametersNode < Node
    # def initialize: (Integer maximum, Location location) -> void
    def initialize(source, maximum, location)
      @source = source
      @location = location
      @maximum = maximum
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_numbered_parameters_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?maximum: Integer, ?location: Location) -> NumberedParametersNode
    def copy(maximum: self.maximum, location: self.location)
      NumberedParametersNode.new(source, maximum, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { maximum: Integer, location: Location }
    def deconstruct_keys(keys)
      { maximum: maximum, location: location }
    end

    # attr_reader maximum: Integer
    attr_reader :maximum

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :numbered_parameters_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :numbered_parameters_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(NumberedParametersNode) &&
        (maximum === other.maximum)
    end
  end

  # Represents reading a numbered reference to a capture in the previous match.
  #
  #     $1
  #     ^^
  class NumberedReferenceReadNode < Node
    # def initialize: (Integer number, Location location) -> void
    def initialize(source, number, location)
      @source = source
      @location = location
      @number = number
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_numbered_reference_read_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?number: Integer, ?location: Location) -> NumberedReferenceReadNode
    def copy(number: self.number, location: self.location)
      NumberedReferenceReadNode.new(source, number, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { number: Integer, location: Location }
    def deconstruct_keys(keys)
      { number: number, location: location }
    end

    # The (1-indexed, from the left) number of the capture group. Numbered references that are too large result in this value being `0`.
    #
    #     $1          # number `1`
    #
    #     $5432       # number `5432`
    #
    #     $4294967296 # number `0`
    attr_reader :number

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :numbered_reference_read_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :numbered_reference_read_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(NumberedReferenceReadNode) &&
        (number === other.number)
    end
  end

  # Represents an optional keyword parameter to a method, block, or lambda definition.
  #
  #     def a(b: 1)
  #           ^^^^
  #     end
  class OptionalKeywordParameterNode < Node
    # def initialize: (Integer flags, Symbol name, Location name_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, name, name_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_optional_keyword_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol, ?name_loc: Location, ?value: Prism::node, ?location: Location) -> OptionalKeywordParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, value: self.value, location: self.location)
      OptionalKeywordParameterNode.new(source, flags, name, name_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol, name_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :optional_keyword_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :optional_keyword_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(OptionalKeywordParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents an optional parameter to a method, block, or lambda definition.
  #
  #     def a(b = 1)
  #           ^^^^^
  #     end
  class OptionalParameterNode < Node
    # def initialize: (Integer flags, Symbol name, Location name_loc, Location operator_loc, Prism::node value, Location location) -> void
    def initialize(source, flags, name, name_loc, operator_loc, value, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
      @value = value
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_optional_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [value]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [value]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc, operator_loc, value] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol, ?name_loc: Location, ?operator_loc: Location, ?value: Prism::node, ?location: Location) -> OptionalParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, value: self.value, location: self.location)
      OptionalParameterNode.new(source, flags, name, name_loc, operator_loc, value, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol, name_loc: Location, operator_loc: Location, value: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, operator_loc: operator_loc, value: value, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader value: Prism::node
    attr_reader :value

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :optional_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :optional_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(OptionalParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (value === other.value)
    end
  end

  # Represents the use of the `||` operator or the `or` keyword.
  #
  #     left or right
  #     ^^^^^^^^^^^^^
  class OrNode < Node
    # def initialize: (Prism::node left, Prism::node right, Location operator_loc, Location location) -> void
    def initialize(source, left, right, operator_loc, location)
      @source = source
      @location = location
      @left = left
      @right = right
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_or_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [left, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [left, right]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [left, right, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?left: Prism::node, ?right: Prism::node, ?operator_loc: Location, ?location: Location) -> OrNode
    def copy(left: self.left, right: self.right, operator_loc: self.operator_loc, location: self.location)
      OrNode.new(source, left, right, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { left: Prism::node, right: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { left: left, right: right, operator_loc: operator_loc, location: location }
    end

    # Represents the left side of the expression. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     left or right
    #     ^^^^
    #
    #     1 || 2
    #     ^
    attr_reader :left

    # Represents the right side of the expression. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     left || right
    #             ^^^^^
    #
    #     1 or 2
    #          ^
    attr_reader :right

    # The location of the `or` keyword or the `||` operator.
    #
    #     left or right
    #          ^^
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :or_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :or_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(OrNode) &&
        (left === other.left) &&
        (right === other.right) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the list of parameters on a method, block, or lambda definition.
  #
  #     def a(b, c, d)
  #           ^^^^^^^
  #     end
  class ParametersNode < Node
    # def initialize: (Array[RequiredParameterNode | MultiTargetNode] requireds, Array[OptionalParameterNode] optionals, RestParameterNode | ImplicitRestNode | nil rest, Array[RequiredParameterNode | MultiTargetNode | KeywordRestParameterNode | NoKeywordsParameterNode | ForwardingParameterNode] posts, Array[RequiredKeywordParameterNode | OptionalKeywordParameterNode] keywords, KeywordRestParameterNode | ForwardingParameterNode | NoKeywordsParameterNode | nil keyword_rest, BlockParameterNode? block, Location location) -> void
    def initialize(source, requireds, optionals, rest, posts, keywords, keyword_rest, block, location)
      @source = source
      @location = location
      @requireds = requireds
      @optionals = optionals
      @rest = rest
      @posts = posts
      @keywords = keywords
      @keyword_rest = keyword_rest
      @block = block
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_parameters_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*requireds, *optionals, rest, *posts, *keywords, keyword_rest, block]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact.concat(requireds)
      compact.concat(optionals)
      compact << rest if rest
      compact.concat(posts)
      compact.concat(keywords)
      compact << keyword_rest if keyword_rest
      compact << block if block
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*requireds, *optionals, *rest, *posts, *keywords, *keyword_rest, *block] #: Array[Prism::node | Location]
    end

    # def copy: (?requireds: Array[RequiredParameterNode | MultiTargetNode], ?optionals: Array[OptionalParameterNode], ?rest: RestParameterNode | ImplicitRestNode | nil, ?posts: Array[RequiredParameterNode | MultiTargetNode | KeywordRestParameterNode | NoKeywordsParameterNode | ForwardingParameterNode], ?keywords: Array[RequiredKeywordParameterNode | OptionalKeywordParameterNode], ?keyword_rest: KeywordRestParameterNode | ForwardingParameterNode | NoKeywordsParameterNode | nil, ?block: BlockParameterNode?, ?location: Location) -> ParametersNode
    def copy(requireds: self.requireds, optionals: self.optionals, rest: self.rest, posts: self.posts, keywords: self.keywords, keyword_rest: self.keyword_rest, block: self.block, location: self.location)
      ParametersNode.new(source, requireds, optionals, rest, posts, keywords, keyword_rest, block, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { requireds: Array[RequiredParameterNode | MultiTargetNode], optionals: Array[OptionalParameterNode], rest: RestParameterNode | ImplicitRestNode | nil, posts: Array[RequiredParameterNode | MultiTargetNode | KeywordRestParameterNode | NoKeywordsParameterNode | ForwardingParameterNode], keywords: Array[RequiredKeywordParameterNode | OptionalKeywordParameterNode], keyword_rest: KeywordRestParameterNode | ForwardingParameterNode | NoKeywordsParameterNode | nil, block: BlockParameterNode?, location: Location }
    def deconstruct_keys(keys)
      { requireds: requireds, optionals: optionals, rest: rest, posts: posts, keywords: keywords, keyword_rest: keyword_rest, block: block, location: location }
    end

    # attr_reader requireds: Array[RequiredParameterNode | MultiTargetNode]
    attr_reader :requireds

    # attr_reader optionals: Array[OptionalParameterNode]
    attr_reader :optionals

    # attr_reader rest: RestParameterNode | ImplicitRestNode | nil
    attr_reader :rest

    # attr_reader posts: Array[RequiredParameterNode | MultiTargetNode | KeywordRestParameterNode | NoKeywordsParameterNode | ForwardingParameterNode]
    attr_reader :posts

    # attr_reader keywords: Array[RequiredKeywordParameterNode | OptionalKeywordParameterNode]
    attr_reader :keywords

    # attr_reader keyword_rest: KeywordRestParameterNode | ForwardingParameterNode | NoKeywordsParameterNode | nil
    attr_reader :keyword_rest

    # attr_reader block: BlockParameterNode?
    attr_reader :block

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :parameters_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :parameters_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ParametersNode) &&
        (requireds.length == other.requireds.length) &&
        requireds.zip(other.requireds).all? { |left, right| left === right } &&
        (optionals.length == other.optionals.length) &&
        optionals.zip(other.optionals).all? { |left, right| left === right } &&
        (rest === other.rest) &&
        (posts.length == other.posts.length) &&
        posts.zip(other.posts).all? { |left, right| left === right } &&
        (keywords.length == other.keywords.length) &&
        keywords.zip(other.keywords).all? { |left, right| left === right } &&
        (keyword_rest === other.keyword_rest) &&
        (block === other.block)
    end
  end

  # Represents a parenthesized expression
  #
  #     (10 + 34)
  #     ^^^^^^^^^
  class ParenthesesNode < Node
    # def initialize: (Prism::node? body, Location opening_loc, Location closing_loc, Location location) -> void
    def initialize(source, body, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @body = body
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_parentheses_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*body, opening_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?body: Prism::node?, ?opening_loc: Location, ?closing_loc: Location, ?location: Location) -> ParenthesesNode
    def copy(body: self.body, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      ParenthesesNode.new(source, body, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { body: Prism::node?, opening_loc: Location, closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { body: body, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :parentheses_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :parentheses_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ParenthesesNode) &&
        (body === other.body) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents the use of the `^` operator for pinning an expression in a pattern matching expression.
  #
  #     foo in ^(bar)
  #            ^^^^^^
  class PinnedExpressionNode < Node
    # def initialize: (Prism::node expression, Location operator_loc, Location lparen_loc, Location rparen_loc, Location location) -> void
    def initialize(source, expression, operator_loc, lparen_loc, rparen_loc, location)
      @source = source
      @location = location
      @expression = expression
      @operator_loc = operator_loc
      @lparen_loc = lparen_loc
      @rparen_loc = rparen_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_pinned_expression_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [expression]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [expression]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [expression, operator_loc, lparen_loc, rparen_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?expression: Prism::node, ?operator_loc: Location, ?lparen_loc: Location, ?rparen_loc: Location, ?location: Location) -> PinnedExpressionNode
    def copy(expression: self.expression, operator_loc: self.operator_loc, lparen_loc: self.lparen_loc, rparen_loc: self.rparen_loc, location: self.location)
      PinnedExpressionNode.new(source, expression, operator_loc, lparen_loc, rparen_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { expression: Prism::node, operator_loc: Location, lparen_loc: Location, rparen_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { expression: expression, operator_loc: operator_loc, lparen_loc: lparen_loc, rparen_loc: rparen_loc, location: location }
    end

    # attr_reader expression: Prism::node
    attr_reader :expression

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader lparen_loc: Location
    def lparen_loc
      location = @lparen_loc
      return location if location.is_a?(Location)
      @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader rparen_loc: Location
    def rparen_loc
      location = @rparen_loc
      return location if location.is_a?(Location)
      @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def lparen: () -> String
    def lparen
      lparen_loc.slice
    end

    # def rparen: () -> String
    def rparen
      rparen_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :pinned_expression_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :pinned_expression_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(PinnedExpressionNode) &&
        (expression === other.expression) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (rparen_loc.nil? == other.rparen_loc.nil?)
    end
  end

  # Represents the use of the `^` operator for pinning a variable in a pattern matching expression.
  #
  #     foo in ^bar
  #            ^^^^
  class PinnedVariableNode < Node
    # def initialize: (Prism::node variable, Location operator_loc, Location location) -> void
    def initialize(source, variable, operator_loc, location)
      @source = source
      @location = location
      @variable = variable
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_pinned_variable_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [variable]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [variable]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [variable, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?variable: Prism::node, ?operator_loc: Location, ?location: Location) -> PinnedVariableNode
    def copy(variable: self.variable, operator_loc: self.operator_loc, location: self.location)
      PinnedVariableNode.new(source, variable, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { variable: Prism::node, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { variable: variable, operator_loc: operator_loc, location: location }
    end

    # attr_reader variable: Prism::node
    attr_reader :variable

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :pinned_variable_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :pinned_variable_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(PinnedVariableNode) &&
        (variable === other.variable) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of the `END` keyword.
  #
  #     END { foo }
  #     ^^^^^^^^^^^
  class PostExecutionNode < Node
    # def initialize: (StatementsNode? statements, Location keyword_loc, Location opening_loc, Location closing_loc, Location location) -> void
    def initialize(source, statements, keyword_loc, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @statements = statements
      @keyword_loc = keyword_loc
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_post_execution_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*statements, keyword_loc, opening_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?statements: StatementsNode?, ?keyword_loc: Location, ?opening_loc: Location, ?closing_loc: Location, ?location: Location) -> PostExecutionNode
    def copy(statements: self.statements, keyword_loc: self.keyword_loc, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      PostExecutionNode.new(source, statements, keyword_loc, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { statements: StatementsNode?, keyword_loc: Location, opening_loc: Location, closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { statements: statements, keyword_loc: keyword_loc, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :post_execution_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :post_execution_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(PostExecutionNode) &&
        (statements === other.statements) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # Represents the use of the `BEGIN` keyword.
  #
  #     BEGIN { foo }
  #     ^^^^^^^^^^^^^
  class PreExecutionNode < Node
    # def initialize: (StatementsNode? statements, Location keyword_loc, Location opening_loc, Location closing_loc, Location location) -> void
    def initialize(source, statements, keyword_loc, opening_loc, closing_loc, location)
      @source = source
      @location = location
      @statements = statements
      @keyword_loc = keyword_loc
      @opening_loc = opening_loc
      @closing_loc = closing_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_pre_execution_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*statements, keyword_loc, opening_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?statements: StatementsNode?, ?keyword_loc: Location, ?opening_loc: Location, ?closing_loc: Location, ?location: Location) -> PreExecutionNode
    def copy(statements: self.statements, keyword_loc: self.keyword_loc, opening_loc: self.opening_loc, closing_loc: self.closing_loc, location: self.location)
      PreExecutionNode.new(source, statements, keyword_loc, opening_loc, closing_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { statements: StatementsNode?, keyword_loc: Location, opening_loc: Location, closing_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { statements: statements, keyword_loc: keyword_loc, opening_loc: opening_loc, closing_loc: closing_loc, location: location }
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :pre_execution_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :pre_execution_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(PreExecutionNode) &&
        (statements === other.statements) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?)
    end
  end

  # The top level node of any parse tree.
  class ProgramNode < Node
    # def initialize: (Array[Symbol] locals, StatementsNode statements, Location location) -> void
    def initialize(source, locals, statements, location)
      @source = source
      @location = location
      @locals = locals
      @statements = statements
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_program_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [statements]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [statements] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?statements: StatementsNode, ?location: Location) -> ProgramNode
    def copy(locals: self.locals, statements: self.statements, location: self.location)
      ProgramNode.new(source, locals, statements, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], statements: StatementsNode, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, statements: statements, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader statements: StatementsNode
    attr_reader :statements

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :program_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :program_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ProgramNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (statements === other.statements)
    end
  end

  # Represents the use of the `..` or `...` operators.
  #
  #     1..2
  #     ^^^^
  #
  #     c if a =~ /left/ ... b =~ /right/
  #          ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  class RangeNode < Node
    # def initialize: (Integer flags, Prism::node? left, Prism::node? right, Location operator_loc, Location location) -> void
    def initialize(source, flags, left, right, operator_loc, location)
      @source = source
      @location = location
      @flags = flags
      @left = left
      @right = right
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_range_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [left, right]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << left if left
      compact << right if right
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*left, *right, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?left: Prism::node?, ?right: Prism::node?, ?operator_loc: Location, ?location: Location) -> RangeNode
    def copy(flags: self.flags, left: self.left, right: self.right, operator_loc: self.operator_loc, location: self.location)
      RangeNode.new(source, flags, left, right, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, left: Prism::node?, right: Prism::node?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, left: left, right: right, operator_loc: operator_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # The left-hand side of the range, if present. It can be either `nil` or any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     1...
    #     ^
    #
    #     hello...goodbye
    #     ^^^^^
    attr_reader :left

    # The right-hand side of the range, if present. It can be either `nil` or any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     ..5
    #       ^
    #
    #     1...foo
    #         ^^^
    # If neither right-hand or left-hand side was included, this will be a MissingNode.
    attr_reader :right

    # The location of the `..` or `...` operator.
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def exclude_end?: () -> bool
    def exclude_end?
      flags.anybits?(RangeFlags::EXCLUDE_END)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :range_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :range_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RangeNode) &&
        (flags === other.flags) &&
        (left === other.left) &&
        (right === other.right) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents a rational number literal.
  #
  #     1.0r
  #     ^^^^
  class RationalNode < Node
    # def initialize: (Prism::node numeric, Location location) -> void
    def initialize(source, numeric, location)
      @source = source
      @location = location
      @numeric = numeric
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_rational_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [numeric]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [numeric]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [numeric] #: Array[Prism::node | Location]
    end

    # def copy: (?numeric: Prism::node, ?location: Location) -> RationalNode
    def copy(numeric: self.numeric, location: self.location)
      RationalNode.new(source, numeric, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { numeric: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { numeric: numeric, location: location }
    end

    # attr_reader numeric: Prism::node
    attr_reader :numeric

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :rational_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :rational_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RationalNode) &&
        (numeric === other.numeric)
    end
  end

  # Represents the use of the `redo` keyword.
  #
  #     redo
  #     ^^^^
  class RedoNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_redo_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> RedoNode
    def copy(location: self.location)
      RedoNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :redo_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :redo_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RedoNode)
    end
  end

  # Represents a regular expression literal with no interpolation.
  #
  #     /foo/i
  #     ^^^^^^
  class RegularExpressionNode < Node
    # def initialize: (Integer flags, Location opening_loc, Location content_loc, Location closing_loc, String unescaped, Location location) -> void
    def initialize(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @content_loc = content_loc
      @closing_loc = closing_loc
      @unescaped = unescaped
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_regular_expression_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, content_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location, ?content_loc: Location, ?closing_loc: Location, ?unescaped: String, ?location: Location) -> RegularExpressionNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, content_loc: self.content_loc, closing_loc: self.closing_loc, unescaped: self.unescaped, location: self.location)
      RegularExpressionNode.new(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location, content_loc: Location, closing_loc: Location, unescaped: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, content_loc: content_loc, closing_loc: closing_loc, unescaped: unescaped, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader content_loc: Location
    def content_loc
      location = @content_loc
      return location if location.is_a?(Location)
      @content_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader unescaped: String
    attr_reader :unescaped

    # def ignore_case?: () -> bool
    def ignore_case?
      flags.anybits?(RegularExpressionFlags::IGNORE_CASE)
    end

    # def extended?: () -> bool
    def extended?
      flags.anybits?(RegularExpressionFlags::EXTENDED)
    end

    # def multi_line?: () -> bool
    def multi_line?
      flags.anybits?(RegularExpressionFlags::MULTI_LINE)
    end

    # def once?: () -> bool
    def once?
      flags.anybits?(RegularExpressionFlags::ONCE)
    end

    # def euc_jp?: () -> bool
    def euc_jp?
      flags.anybits?(RegularExpressionFlags::EUC_JP)
    end

    # def ascii_8bit?: () -> bool
    def ascii_8bit?
      flags.anybits?(RegularExpressionFlags::ASCII_8BIT)
    end

    # def windows_31j?: () -> bool
    def windows_31j?
      flags.anybits?(RegularExpressionFlags::WINDOWS_31J)
    end

    # def utf_8?: () -> bool
    def utf_8?
      flags.anybits?(RegularExpressionFlags::UTF_8)
    end

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_BINARY_ENCODING)
    end

    # def forced_us_ascii_encoding?: () -> bool
    def forced_us_ascii_encoding?
      flags.anybits?(RegularExpressionFlags::FORCED_US_ASCII_ENCODING)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def content: () -> String
    def content
      content_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :regular_expression_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :regular_expression_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RegularExpressionNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (content_loc.nil? == other.content_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (unescaped === other.unescaped)
    end
  end

  # Represents a required keyword parameter to a method, block, or lambda definition.
  #
  #     def a(b: )
  #           ^^
  #     end
  class RequiredKeywordParameterNode < Node
    # def initialize: (Integer flags, Symbol name, Location name_loc, Location location) -> void
    def initialize(source, flags, name, name_loc, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_required_keyword_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [name_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol, ?name_loc: Location, ?location: Location) -> RequiredKeywordParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, location: self.location)
      RequiredKeywordParameterNode.new(source, flags, name, name_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol, name_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol
    attr_reader :name

    # attr_reader name_loc: Location
    def name_loc
      location = @name_loc
      return location if location.is_a?(Location)
      @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :required_keyword_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :required_keyword_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RequiredKeywordParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?)
    end
  end

  # Represents a required parameter to a method, block, or lambda definition.
  #
  #     def a(b)
  #           ^
  #     end
  class RequiredParameterNode < Node
    # def initialize: (Integer flags, Symbol name, Location location) -> void
    def initialize(source, flags, name, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_required_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol, ?location: Location) -> RequiredParameterNode
    def copy(flags: self.flags, name: self.name, location: self.location)
      RequiredParameterNode.new(source, flags, name, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol
    attr_reader :name

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :required_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :required_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RequiredParameterNode) &&
        (flags === other.flags) &&
        (name === other.name)
    end
  end

  # Represents an expression modified with a rescue.
  #
  #     foo rescue nil
  #     ^^^^^^^^^^^^^^
  class RescueModifierNode < Node
    # def initialize: (Prism::node expression, Location keyword_loc, Prism::node rescue_expression, Location location) -> void
    def initialize(source, expression, keyword_loc, rescue_expression, location)
      @source = source
      @location = location
      @expression = expression
      @keyword_loc = keyword_loc
      @rescue_expression = rescue_expression
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_rescue_modifier_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [expression, rescue_expression]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [expression, rescue_expression]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [expression, keyword_loc, rescue_expression] #: Array[Prism::node | Location]
    end

    # def copy: (?expression: Prism::node, ?keyword_loc: Location, ?rescue_expression: Prism::node, ?location: Location) -> RescueModifierNode
    def copy(expression: self.expression, keyword_loc: self.keyword_loc, rescue_expression: self.rescue_expression, location: self.location)
      RescueModifierNode.new(source, expression, keyword_loc, rescue_expression, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { expression: Prism::node, keyword_loc: Location, rescue_expression: Prism::node, location: Location }
    def deconstruct_keys(keys)
      { expression: expression, keyword_loc: keyword_loc, rescue_expression: rescue_expression, location: location }
    end

    # attr_reader expression: Prism::node
    attr_reader :expression

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader rescue_expression: Prism::node
    attr_reader :rescue_expression

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :rescue_modifier_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :rescue_modifier_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RescueModifierNode) &&
        (expression === other.expression) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (rescue_expression === other.rescue_expression)
    end
  end

  # Represents a rescue statement.
  #
  #     begin
  #     rescue Foo, *splat, Bar => ex
  #       foo
  #     ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  #     end
  #
  # `Foo, *splat, Bar` are in the `exceptions` field. `ex` is in the `exception` field.
  class RescueNode < Node
    # def initialize: (Location keyword_loc, Array[Prism::node] exceptions, Location? operator_loc, Prism::node? reference, StatementsNode? statements, RescueNode? consequent, Location location) -> void
    def initialize(source, keyword_loc, exceptions, operator_loc, reference, statements, consequent, location)
      @source = source
      @location = location
      @keyword_loc = keyword_loc
      @exceptions = exceptions
      @operator_loc = operator_loc
      @reference = reference
      @statements = statements
      @consequent = consequent
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_rescue_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*exceptions, reference, statements, consequent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact.concat(exceptions)
      compact << reference if reference
      compact << statements if statements
      compact << consequent if consequent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *exceptions, *operator_loc, *reference, *statements, *consequent] #: Array[Prism::node | Location]
    end

    # def copy: (?keyword_loc: Location, ?exceptions: Array[Prism::node], ?operator_loc: Location?, ?reference: Prism::node?, ?statements: StatementsNode?, ?consequent: RescueNode?, ?location: Location) -> RescueNode
    def copy(keyword_loc: self.keyword_loc, exceptions: self.exceptions, operator_loc: self.operator_loc, reference: self.reference, statements: self.statements, consequent: self.consequent, location: self.location)
      RescueNode.new(source, keyword_loc, exceptions, operator_loc, reference, statements, consequent, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { keyword_loc: Location, exceptions: Array[Prism::node], operator_loc: Location?, reference: Prism::node?, statements: StatementsNode?, consequent: RescueNode?, location: Location }
    def deconstruct_keys(keys)
      { keyword_loc: keyword_loc, exceptions: exceptions, operator_loc: operator_loc, reference: reference, statements: statements, consequent: consequent, location: location }
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader exceptions: Array[Prism::node]
    attr_reader :exceptions

    # attr_reader operator_loc: Location?
    def operator_loc
      location = @operator_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader reference: Prism::node?
    attr_reader :reference

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # attr_reader consequent: RescueNode?
    attr_reader :consequent

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def operator: () -> String?
    def operator
      operator_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :rescue_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :rescue_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RescueNode) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (exceptions.length == other.exceptions.length) &&
        exceptions.zip(other.exceptions).all? { |left, right| left === right } &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (reference === other.reference) &&
        (statements === other.statements) &&
        (consequent === other.consequent)
    end
  end

  # Represents a rest parameter to a method, block, or lambda definition.
  #
  #     def a(*b)
  #           ^^
  #     end
  class RestParameterNode < Node
    # def initialize: (Integer flags, Symbol? name, Location? name_loc, Location operator_loc, Location location) -> void
    def initialize(source, flags, name, name_loc, operator_loc, location)
      @source = source
      @location = location
      @flags = flags
      @name = name
      @name_loc = name_loc
      @operator_loc = operator_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_rest_parameter_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*name_loc, operator_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?name: Symbol?, ?name_loc: Location?, ?operator_loc: Location, ?location: Location) -> RestParameterNode
    def copy(flags: self.flags, name: self.name, name_loc: self.name_loc, operator_loc: self.operator_loc, location: self.location)
      RestParameterNode.new(source, flags, name, name_loc, operator_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, name: Symbol?, name_loc: Location?, operator_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, name: name, name_loc: name_loc, operator_loc: operator_loc, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader name: Symbol?
    attr_reader :name

    # attr_reader name_loc: Location?
    def name_loc
      location = @name_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @name_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def repeated_parameter?: () -> bool
    def repeated_parameter?
      flags.anybits?(ParameterFlags::REPEATED_PARAMETER)
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :rest_parameter_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :rest_parameter_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RestParameterNode) &&
        (flags === other.flags) &&
        (name === other.name) &&
        (name_loc.nil? == other.name_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?)
    end
  end

  # Represents the use of the `retry` keyword.
  #
  #     retry
  #     ^^^^^
  class RetryNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_retry_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> RetryNode
    def copy(location: self.location)
      RetryNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :retry_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :retry_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(RetryNode)
    end
  end

  # Represents the use of the `return` keyword.
  #
  #     return 1
  #     ^^^^^^^^
  class ReturnNode < Node
    # def initialize: (Integer flags, Location keyword_loc, ArgumentsNode? arguments, Location location) -> void
    def initialize(source, flags, keyword_loc, arguments, location)
      @source = source
      @location = location
      @flags = flags
      @keyword_loc = keyword_loc
      @arguments = arguments
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_return_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [arguments]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << arguments if arguments
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *arguments] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?keyword_loc: Location, ?arguments: ArgumentsNode?, ?location: Location) -> ReturnNode
    def copy(flags: self.flags, keyword_loc: self.keyword_loc, arguments: self.arguments, location: self.location)
      ReturnNode.new(source, flags, keyword_loc, arguments, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, keyword_loc: Location, arguments: ArgumentsNode?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, keyword_loc: keyword_loc, arguments: arguments, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # def redundant?: () -> bool
    def redundant?
      flags.anybits?(ReturnNodeFlags::REDUNDANT)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :return_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :return_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ReturnNode) &&
        (flags === other.flags) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (arguments === other.arguments)
    end
  end

  # Represents the `self` keyword.
  #
  #     self
  #     ^^^^
  class SelfNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_self_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> SelfNode
    def copy(location: self.location)
      SelfNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :self_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :self_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SelfNode)
    end
  end

  # This node wraps a constant write to indicate that when the value is written, it should have its shareability state modified.
  #
  #     # shareable_constant_value: literal
  #     C = { a: 1 }
  #     ^^^^^^^^^^^^
  class ShareableConstantNode < Node
    # def initialize: (Integer flags, ConstantWriteNode | ConstantAndWriteNode | ConstantOrWriteNode | ConstantOperatorWriteNode | ConstantPathWriteNode | ConstantPathAndWriteNode | ConstantPathOrWriteNode | ConstantPathOperatorWriteNode write, Location location) -> void
    def initialize(source, flags, write, location)
      @source = source
      @location = location
      @flags = flags
      @write = write
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_shareable_constant_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [write]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [write]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [write] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?write: ConstantWriteNode | ConstantAndWriteNode | ConstantOrWriteNode | ConstantOperatorWriteNode | ConstantPathWriteNode | ConstantPathAndWriteNode | ConstantPathOrWriteNode | ConstantPathOperatorWriteNode, ?location: Location) -> ShareableConstantNode
    def copy(flags: self.flags, write: self.write, location: self.location)
      ShareableConstantNode.new(source, flags, write, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, write: ConstantWriteNode | ConstantAndWriteNode | ConstantOrWriteNode | ConstantOperatorWriteNode | ConstantPathWriteNode | ConstantPathAndWriteNode | ConstantPathOrWriteNode | ConstantPathOperatorWriteNode, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, write: write, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # The constant write that should be modified with the shareability state.
    attr_reader :write

    # def literal?: () -> bool
    def literal?
      flags.anybits?(ShareableConstantNodeFlags::LITERAL)
    end

    # def experimental_everything?: () -> bool
    def experimental_everything?
      flags.anybits?(ShareableConstantNodeFlags::EXPERIMENTAL_EVERYTHING)
    end

    # def experimental_copy?: () -> bool
    def experimental_copy?
      flags.anybits?(ShareableConstantNodeFlags::EXPERIMENTAL_COPY)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :shareable_constant_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :shareable_constant_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(ShareableConstantNode) &&
        (flags === other.flags) &&
        (write === other.write)
    end
  end

  # Represents a singleton class declaration involving the `class` keyword.
  #
  #     class << self end
  #     ^^^^^^^^^^^^^^^^^
  class SingletonClassNode < Node
    # def initialize: (Array[Symbol] locals, Location class_keyword_loc, Location operator_loc, Prism::node expression, Prism::node? body, Location end_keyword_loc, Location location) -> void
    def initialize(source, locals, class_keyword_loc, operator_loc, expression, body, end_keyword_loc, location)
      @source = source
      @location = location
      @locals = locals
      @class_keyword_loc = class_keyword_loc
      @operator_loc = operator_loc
      @expression = expression
      @body = body
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_singleton_class_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [expression, body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << expression
      compact << body if body
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [class_keyword_loc, operator_loc, expression, *body, end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?locals: Array[Symbol], ?class_keyword_loc: Location, ?operator_loc: Location, ?expression: Prism::node, ?body: Prism::node?, ?end_keyword_loc: Location, ?location: Location) -> SingletonClassNode
    def copy(locals: self.locals, class_keyword_loc: self.class_keyword_loc, operator_loc: self.operator_loc, expression: self.expression, body: self.body, end_keyword_loc: self.end_keyword_loc, location: self.location)
      SingletonClassNode.new(source, locals, class_keyword_loc, operator_loc, expression, body, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { locals: Array[Symbol], class_keyword_loc: Location, operator_loc: Location, expression: Prism::node, body: Prism::node?, end_keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { locals: locals, class_keyword_loc: class_keyword_loc, operator_loc: operator_loc, expression: expression, body: body, end_keyword_loc: end_keyword_loc, location: location }
    end

    # attr_reader locals: Array[Symbol]
    attr_reader :locals

    # attr_reader class_keyword_loc: Location
    def class_keyword_loc
      location = @class_keyword_loc
      return location if location.is_a?(Location)
      @class_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader expression: Prism::node
    attr_reader :expression

    # attr_reader body: Prism::node?
    attr_reader :body

    # attr_reader end_keyword_loc: Location
    def end_keyword_loc
      location = @end_keyword_loc
      return location if location.is_a?(Location)
      @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def class_keyword: () -> String
    def class_keyword
      class_keyword_loc.slice
    end

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def end_keyword: () -> String
    def end_keyword
      end_keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :singleton_class_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :singleton_class_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SingletonClassNode) &&
        (locals.length == other.locals.length) &&
        locals.zip(other.locals).all? { |left, right| left === right } &&
        (class_keyword_loc.nil? == other.class_keyword_loc.nil?) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (expression === other.expression) &&
        (body === other.body) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents the use of the `__ENCODING__` keyword.
  #
  #     __ENCODING__
  #     ^^^^^^^^^^^^
  class SourceEncodingNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_source_encoding_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> SourceEncodingNode
    def copy(location: self.location)
      SourceEncodingNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :source_encoding_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :source_encoding_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SourceEncodingNode)
    end
  end

  # Represents the use of the `__FILE__` keyword.
  #
  #     __FILE__
  #     ^^^^^^^^
  class SourceFileNode < Node
    # def initialize: (Integer flags, String filepath, Location location) -> void
    def initialize(source, flags, filepath, location)
      @source = source
      @location = location
      @flags = flags
      @filepath = filepath
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_source_file_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?filepath: String, ?location: Location) -> SourceFileNode
    def copy(flags: self.flags, filepath: self.filepath, location: self.location)
      SourceFileNode.new(source, flags, filepath, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, filepath: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, filepath: filepath, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # Represents the file path being parsed. This corresponds directly to the `filepath` option given to the various `Prism::parse*` APIs.
    attr_reader :filepath

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(StringFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(StringFlags::FORCED_BINARY_ENCODING)
    end

    # def frozen?: () -> bool
    def frozen?
      flags.anybits?(StringFlags::FROZEN)
    end

    # def mutable?: () -> bool
    def mutable?
      flags.anybits?(StringFlags::MUTABLE)
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :source_file_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :source_file_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SourceFileNode) &&
        (flags === other.flags) &&
        (filepath === other.filepath)
    end
  end

  # Represents the use of the `__LINE__` keyword.
  #
  #     __LINE__
  #     ^^^^^^^^
  class SourceLineNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_source_line_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> SourceLineNode
    def copy(location: self.location)
      SourceLineNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :source_line_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :source_line_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SourceLineNode)
    end
  end

  # Represents the use of the splat operator.
  #
  #     [*a]
  #      ^^
  class SplatNode < Node
    # def initialize: (Location operator_loc, Prism::node? expression, Location location) -> void
    def initialize(source, operator_loc, expression, location)
      @source = source
      @location = location
      @operator_loc = operator_loc
      @expression = expression
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_splat_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [expression]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << expression if expression
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [operator_loc, *expression] #: Array[Prism::node | Location]
    end

    # def copy: (?operator_loc: Location, ?expression: Prism::node?, ?location: Location) -> SplatNode
    def copy(operator_loc: self.operator_loc, expression: self.expression, location: self.location)
      SplatNode.new(source, operator_loc, expression, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { operator_loc: Location, expression: Prism::node?, location: Location }
    def deconstruct_keys(keys)
      { operator_loc: operator_loc, expression: expression, location: location }
    end

    # attr_reader operator_loc: Location
    def operator_loc
      location = @operator_loc
      return location if location.is_a?(Location)
      @operator_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader expression: Prism::node?
    attr_reader :expression

    # def operator: () -> String
    def operator
      operator_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :splat_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :splat_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SplatNode) &&
        (operator_loc.nil? == other.operator_loc.nil?) &&
        (expression === other.expression)
    end
  end

  # Represents a set of statements contained within some scope.
  #
  #     foo; bar; baz
  #     ^^^^^^^^^^^^^
  class StatementsNode < Node
    # def initialize: (Array[Prism::node] body, Location location) -> void
    def initialize(source, body, location)
      @source = source
      @location = location
      @body = body
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_statements_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*body]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*body]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*body] #: Array[Prism::node | Location]
    end

    # def copy: (?body: Array[Prism::node], ?location: Location) -> StatementsNode
    def copy(body: self.body, location: self.location)
      StatementsNode.new(source, body, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { body: Array[Prism::node], location: Location }
    def deconstruct_keys(keys)
      { body: body, location: location }
    end

    # attr_reader body: Array[Prism::node]
    attr_reader :body

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :statements_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :statements_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(StatementsNode) &&
        (body.length == other.body.length) &&
        body.zip(other.body).all? { |left, right| left === right }
    end
  end

  # Represents a string literal, a string contained within a `%w` list, or plain string content within an interpolated string.
  #
  #     "foo"
  #     ^^^^^
  #
  #     %w[foo]
  #        ^^^
  #
  #     "foo #{bar} baz"
  #      ^^^^      ^^^^
  class StringNode < Node
    # def initialize: (Integer flags, Location? opening_loc, Location content_loc, Location? closing_loc, String unescaped, Location location) -> void
    def initialize(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @content_loc = content_loc
      @closing_loc = closing_loc
      @unescaped = unescaped
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_string_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*opening_loc, content_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location?, ?content_loc: Location, ?closing_loc: Location?, ?unescaped: String, ?location: Location) -> StringNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, content_loc: self.content_loc, closing_loc: self.closing_loc, unescaped: self.unescaped, location: self.location)
      StringNode.new(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location?, content_loc: Location, closing_loc: Location?, unescaped: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, content_loc: content_loc, closing_loc: closing_loc, unescaped: unescaped, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader content_loc: Location
    def content_loc
      location = @content_loc
      return location if location.is_a?(Location)
      @content_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader unescaped: String
    attr_reader :unescaped

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(StringFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(StringFlags::FORCED_BINARY_ENCODING)
    end

    # def frozen?: () -> bool
    def frozen?
      flags.anybits?(StringFlags::FROZEN)
    end

    # def mutable?: () -> bool
    def mutable?
      flags.anybits?(StringFlags::MUTABLE)
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def content: () -> String
    def content
      content_loc.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :string_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :string_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(StringNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (content_loc.nil? == other.content_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (unescaped === other.unescaped)
    end
  end

  # Represents the use of the `super` keyword with parentheses or arguments.
  #
  #     super()
  #     ^^^^^^^
  #
  #     super foo, bar
  #     ^^^^^^^^^^^^^^
  class SuperNode < Node
    # def initialize: (Location keyword_loc, Location? lparen_loc, ArgumentsNode? arguments, Location? rparen_loc, Prism::node? block, Location location) -> void
    def initialize(source, keyword_loc, lparen_loc, arguments, rparen_loc, block, location)
      @source = source
      @location = location
      @keyword_loc = keyword_loc
      @lparen_loc = lparen_loc
      @arguments = arguments
      @rparen_loc = rparen_loc
      @block = block
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_super_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [arguments, block]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << arguments if arguments
      compact << block if block
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *lparen_loc, *arguments, *rparen_loc, *block] #: Array[Prism::node | Location]
    end

    # def copy: (?keyword_loc: Location, ?lparen_loc: Location?, ?arguments: ArgumentsNode?, ?rparen_loc: Location?, ?block: Prism::node?, ?location: Location) -> SuperNode
    def copy(keyword_loc: self.keyword_loc, lparen_loc: self.lparen_loc, arguments: self.arguments, rparen_loc: self.rparen_loc, block: self.block, location: self.location)
      SuperNode.new(source, keyword_loc, lparen_loc, arguments, rparen_loc, block, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { keyword_loc: Location, lparen_loc: Location?, arguments: ArgumentsNode?, rparen_loc: Location?, block: Prism::node?, location: Location }
    def deconstruct_keys(keys)
      { keyword_loc: keyword_loc, lparen_loc: lparen_loc, arguments: arguments, rparen_loc: rparen_loc, block: block, location: location }
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader block: Prism::node?
    attr_reader :block

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :super_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :super_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SuperNode) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (arguments === other.arguments) &&
        (rparen_loc.nil? == other.rparen_loc.nil?) &&
        (block === other.block)
    end
  end

  # Represents a symbol literal or a symbol contained within a `%i` list.
  #
  #     :foo
  #     ^^^^
  #
  #     %i[foo]
  #        ^^^
  class SymbolNode < Node
    # def initialize: (Integer flags, Location? opening_loc, Location? value_loc, Location? closing_loc, String unescaped, Location location) -> void
    def initialize(source, flags, opening_loc, value_loc, closing_loc, unescaped, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @value_loc = value_loc
      @closing_loc = closing_loc
      @unescaped = unescaped
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_symbol_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*opening_loc, *value_loc, *closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location?, ?value_loc: Location?, ?closing_loc: Location?, ?unescaped: String, ?location: Location) -> SymbolNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, value_loc: self.value_loc, closing_loc: self.closing_loc, unescaped: self.unescaped, location: self.location)
      SymbolNode.new(source, flags, opening_loc, value_loc, closing_loc, unescaped, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location?, value_loc: Location?, closing_loc: Location?, unescaped: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, value_loc: value_loc, closing_loc: closing_loc, unescaped: unescaped, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location?
    def opening_loc
      location = @opening_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader value_loc: Location?
    def value_loc
      location = @value_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @value_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader unescaped: String
    attr_reader :unescaped

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(SymbolFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(SymbolFlags::FORCED_BINARY_ENCODING)
    end

    # def forced_us_ascii_encoding?: () -> bool
    def forced_us_ascii_encoding?
      flags.anybits?(SymbolFlags::FORCED_US_ASCII_ENCODING)
    end

    # def opening: () -> String?
    def opening
      opening_loc&.slice
    end

    # def value: () -> String?
    def value
      value_loc&.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :symbol_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :symbol_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(SymbolNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (value_loc.nil? == other.value_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (unescaped === other.unescaped)
    end
  end

  # Represents the use of the literal `true` keyword.
  #
  #     true
  #     ^^^^
  class TrueNode < Node
    # def initialize: (Location location) -> void
    def initialize(source, location)
      @source = source
      @location = location
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_true_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [] #: Array[Prism::node | Location]
    end

    # def copy: (?location: Location) -> TrueNode
    def copy(location: self.location)
      TrueNode.new(source, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { location: Location }
    def deconstruct_keys(keys)
      { location: location }
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :true_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :true_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(TrueNode)
    end
  end

  # Represents the use of the `undef` keyword.
  #
  #     undef :foo, :bar, :baz
  #     ^^^^^^^^^^^^^^^^^^^^^^
  class UndefNode < Node
    # def initialize: (Array[SymbolNode | InterpolatedSymbolNode] names, Location keyword_loc, Location location) -> void
    def initialize(source, names, keyword_loc, location)
      @source = source
      @location = location
      @names = names
      @keyword_loc = keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_undef_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*names]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      [*names]
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [*names, keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?names: Array[SymbolNode | InterpolatedSymbolNode], ?keyword_loc: Location, ?location: Location) -> UndefNode
    def copy(names: self.names, keyword_loc: self.keyword_loc, location: self.location)
      UndefNode.new(source, names, keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { names: Array[SymbolNode | InterpolatedSymbolNode], keyword_loc: Location, location: Location }
    def deconstruct_keys(keys)
      { names: names, keyword_loc: keyword_loc, location: location }
    end

    # attr_reader names: Array[SymbolNode | InterpolatedSymbolNode]
    attr_reader :names

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :undef_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :undef_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(UndefNode) &&
        (names.length == other.names.length) &&
        names.zip(other.names).all? { |left, right| left === right } &&
        (keyword_loc.nil? == other.keyword_loc.nil?)
    end
  end

  # Represents the use of the `unless` keyword, either in the block form or the modifier form.
  #
  #     bar unless foo
  #     ^^^^^^^^^^^^^^
  #
  #     unless foo then bar end
  #     ^^^^^^^^^^^^^^^^^^^^^^^
  class UnlessNode < Node
    # def initialize: (Location keyword_loc, Prism::node predicate, Location? then_keyword_loc, StatementsNode? statements, ElseNode? consequent, Location? end_keyword_loc, Location location) -> void
    def initialize(source, keyword_loc, predicate, then_keyword_loc, statements, consequent, end_keyword_loc, location)
      @source = source
      @location = location
      @keyword_loc = keyword_loc
      @predicate = predicate
      @then_keyword_loc = then_keyword_loc
      @statements = statements
      @consequent = consequent
      @end_keyword_loc = end_keyword_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_unless_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, statements, consequent]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate
      compact << statements if statements
      compact << consequent if consequent
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, predicate, *then_keyword_loc, *statements, *consequent, *end_keyword_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?keyword_loc: Location, ?predicate: Prism::node, ?then_keyword_loc: Location?, ?statements: StatementsNode?, ?consequent: ElseNode?, ?end_keyword_loc: Location?, ?location: Location) -> UnlessNode
    def copy(keyword_loc: self.keyword_loc, predicate: self.predicate, then_keyword_loc: self.then_keyword_loc, statements: self.statements, consequent: self.consequent, end_keyword_loc: self.end_keyword_loc, location: self.location)
      UnlessNode.new(source, keyword_loc, predicate, then_keyword_loc, statements, consequent, end_keyword_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { keyword_loc: Location, predicate: Prism::node, then_keyword_loc: Location?, statements: StatementsNode?, consequent: ElseNode?, end_keyword_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { keyword_loc: keyword_loc, predicate: predicate, then_keyword_loc: then_keyword_loc, statements: statements, consequent: consequent, end_keyword_loc: end_keyword_loc, location: location }
    end

    # The location of the `unless` keyword.
    #
    #     unless cond then bar end
    #     ^^^^^^
    #
    #     bar unless cond
    #         ^^^^^^
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # The condition to be evaluated for the unless expression. It can be any [non-void expression](https://github.com/ruby/prism/blob/main/docs/parsing_rules.md#non-void-expression).
    #
    #     unless cond then bar end
    #            ^^^^
    #
    #     bar unless cond
    #                ^^^^
    attr_reader :predicate

    # The location of the `then` keyword, if present.
    # unless cond then bar end ^^^^
    def then_keyword_loc
      location = @then_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @then_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # The body of statements that will executed if the unless condition is
    # falsey. Will be `nil` if no body is provided.
    #
    #     unless cond then bar end
    #                      ^^^
    attr_reader :statements

    # The else clause of the unless expression, if present.
    #
    #     unless cond then bar else baz end
    #                          ^^^^^^^^
    attr_reader :consequent

    # The location of the `end` keyword, if present.
    #
    #     unless cond then bar end
    #                          ^^^
    def end_keyword_loc
      location = @end_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @end_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def then_keyword: () -> String?
    def then_keyword
      then_keyword_loc&.slice
    end

    # def end_keyword: () -> String?
    def end_keyword
      end_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :unless_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :unless_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(UnlessNode) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (predicate === other.predicate) &&
        (then_keyword_loc.nil? == other.then_keyword_loc.nil?) &&
        (statements === other.statements) &&
        (consequent === other.consequent) &&
        (end_keyword_loc.nil? == other.end_keyword_loc.nil?)
    end
  end

  # Represents the use of the `until` keyword, either in the block form or the modifier form.
  #
  #     bar until foo
  #     ^^^^^^^^^^^^^
  #
  #     until foo do bar end
  #     ^^^^^^^^^^^^^^^^^^^^
  class UntilNode < Node
    # def initialize: (Integer flags, Location keyword_loc, Location? closing_loc, Prism::node predicate, StatementsNode? statements, Location location) -> void
    def initialize(source, flags, keyword_loc, closing_loc, predicate, statements, location)
      @source = source
      @location = location
      @flags = flags
      @keyword_loc = keyword_loc
      @closing_loc = closing_loc
      @predicate = predicate
      @statements = statements
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_until_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *closing_loc, predicate, *statements] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?keyword_loc: Location, ?closing_loc: Location?, ?predicate: Prism::node, ?statements: StatementsNode?, ?location: Location) -> UntilNode
    def copy(flags: self.flags, keyword_loc: self.keyword_loc, closing_loc: self.closing_loc, predicate: self.predicate, statements: self.statements, location: self.location)
      UntilNode.new(source, flags, keyword_loc, closing_loc, predicate, statements, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, keyword_loc: Location, closing_loc: Location?, predicate: Prism::node, statements: StatementsNode?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, keyword_loc: keyword_loc, closing_loc: closing_loc, predicate: predicate, statements: statements, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader predicate: Prism::node
    attr_reader :predicate

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # def begin_modifier?: () -> bool
    def begin_modifier?
      flags.anybits?(LoopFlags::BEGIN_MODIFIER)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :until_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :until_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(UntilNode) &&
        (flags === other.flags) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (predicate === other.predicate) &&
        (statements === other.statements)
    end
  end

  # Represents the use of the `when` keyword within a case statement.
  #
  #     case true
  #     when true
  #     ^^^^^^^^^
  #     end
  class WhenNode < Node
    # def initialize: (Location keyword_loc, Array[Prism::node] conditions, Location? then_keyword_loc, StatementsNode? statements, Location location) -> void
    def initialize(source, keyword_loc, conditions, then_keyword_loc, statements, location)
      @source = source
      @location = location
      @keyword_loc = keyword_loc
      @conditions = conditions
      @then_keyword_loc = then_keyword_loc
      @statements = statements
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_when_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [*conditions, statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact.concat(conditions)
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *conditions, *then_keyword_loc, *statements] #: Array[Prism::node | Location]
    end

    # def copy: (?keyword_loc: Location, ?conditions: Array[Prism::node], ?then_keyword_loc: Location?, ?statements: StatementsNode?, ?location: Location) -> WhenNode
    def copy(keyword_loc: self.keyword_loc, conditions: self.conditions, then_keyword_loc: self.then_keyword_loc, statements: self.statements, location: self.location)
      WhenNode.new(source, keyword_loc, conditions, then_keyword_loc, statements, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { keyword_loc: Location, conditions: Array[Prism::node], then_keyword_loc: Location?, statements: StatementsNode?, location: Location }
    def deconstruct_keys(keys)
      { keyword_loc: keyword_loc, conditions: conditions, then_keyword_loc: then_keyword_loc, statements: statements, location: location }
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader conditions: Array[Prism::node]
    attr_reader :conditions

    # attr_reader then_keyword_loc: Location?
    def then_keyword_loc
      location = @then_keyword_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @then_keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def then_keyword: () -> String?
    def then_keyword
      then_keyword_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :when_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :when_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(WhenNode) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (conditions.length == other.conditions.length) &&
        conditions.zip(other.conditions).all? { |left, right| left === right } &&
        (then_keyword_loc.nil? == other.then_keyword_loc.nil?) &&
        (statements === other.statements)
    end
  end

  # Represents the use of the `while` keyword, either in the block form or the modifier form.
  #
  #     bar while foo
  #     ^^^^^^^^^^^^^
  #
  #     while foo do bar end
  #     ^^^^^^^^^^^^^^^^^^^^
  class WhileNode < Node
    # def initialize: (Integer flags, Location keyword_loc, Location? closing_loc, Prism::node predicate, StatementsNode? statements, Location location) -> void
    def initialize(source, flags, keyword_loc, closing_loc, predicate, statements, location)
      @source = source
      @location = location
      @flags = flags
      @keyword_loc = keyword_loc
      @closing_loc = closing_loc
      @predicate = predicate
      @statements = statements
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_while_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [predicate, statements]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << predicate
      compact << statements if statements
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *closing_loc, predicate, *statements] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?keyword_loc: Location, ?closing_loc: Location?, ?predicate: Prism::node, ?statements: StatementsNode?, ?location: Location) -> WhileNode
    def copy(flags: self.flags, keyword_loc: self.keyword_loc, closing_loc: self.closing_loc, predicate: self.predicate, statements: self.statements, location: self.location)
      WhileNode.new(source, flags, keyword_loc, closing_loc, predicate, statements, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, keyword_loc: Location, closing_loc: Location?, predicate: Prism::node, statements: StatementsNode?, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, keyword_loc: keyword_loc, closing_loc: closing_loc, predicate: predicate, statements: statements, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location?
    def closing_loc
      location = @closing_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader predicate: Prism::node
    attr_reader :predicate

    # attr_reader statements: StatementsNode?
    attr_reader :statements

    # def begin_modifier?: () -> bool
    def begin_modifier?
      flags.anybits?(LoopFlags::BEGIN_MODIFIER)
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def closing: () -> String?
    def closing
      closing_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :while_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :while_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(WhileNode) &&
        (flags === other.flags) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (predicate === other.predicate) &&
        (statements === other.statements)
    end
  end

  # Represents an xstring literal with no interpolation.
  #
  #     `foo`
  #     ^^^^^
  class XStringNode < Node
    # def initialize: (Integer flags, Location opening_loc, Location content_loc, Location closing_loc, String unescaped, Location location) -> void
    def initialize(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
      @source = source
      @location = location
      @flags = flags
      @opening_loc = opening_loc
      @content_loc = content_loc
      @closing_loc = closing_loc
      @unescaped = unescaped
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_x_string_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      []
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      []
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [opening_loc, content_loc, closing_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?flags: Integer, ?opening_loc: Location, ?content_loc: Location, ?closing_loc: Location, ?unescaped: String, ?location: Location) -> XStringNode
    def copy(flags: self.flags, opening_loc: self.opening_loc, content_loc: self.content_loc, closing_loc: self.closing_loc, unescaped: self.unescaped, location: self.location)
      XStringNode.new(source, flags, opening_loc, content_loc, closing_loc, unescaped, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { flags: Integer, opening_loc: Location, content_loc: Location, closing_loc: Location, unescaped: String, location: Location }
    def deconstruct_keys(keys)
      { flags: flags, opening_loc: opening_loc, content_loc: content_loc, closing_loc: closing_loc, unescaped: unescaped, location: location }
    end

    # protected attr_reader flags: Integer
    attr_reader :flags
    protected :flags

    # attr_reader opening_loc: Location
    def opening_loc
      location = @opening_loc
      return location if location.is_a?(Location)
      @opening_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader content_loc: Location
    def content_loc
      location = @content_loc
      return location if location.is_a?(Location)
      @content_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader closing_loc: Location
    def closing_loc
      location = @closing_loc
      return location if location.is_a?(Location)
      @closing_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader unescaped: String
    attr_reader :unescaped

    # def forced_utf8_encoding?: () -> bool
    def forced_utf8_encoding?
      flags.anybits?(EncodingFlags::FORCED_UTF8_ENCODING)
    end

    # def forced_binary_encoding?: () -> bool
    def forced_binary_encoding?
      flags.anybits?(EncodingFlags::FORCED_BINARY_ENCODING)
    end

    # def opening: () -> String
    def opening
      opening_loc.slice
    end

    # def content: () -> String
    def content
      content_loc.slice
    end

    # def closing: () -> String
    def closing
      closing_loc.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :x_string_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :x_string_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(XStringNode) &&
        (flags === other.flags) &&
        (opening_loc.nil? == other.opening_loc.nil?) &&
        (content_loc.nil? == other.content_loc.nil?) &&
        (closing_loc.nil? == other.closing_loc.nil?) &&
        (unescaped === other.unescaped)
    end
  end

  # Represents the use of the `yield` keyword.
  #
  #     yield 1
  #     ^^^^^^^
  class YieldNode < Node
    # def initialize: (Location keyword_loc, Location? lparen_loc, ArgumentsNode? arguments, Location? rparen_loc, Location location) -> void
    def initialize(source, keyword_loc, lparen_loc, arguments, rparen_loc, location)
      @source = source
      @location = location
      @keyword_loc = keyword_loc
      @lparen_loc = lparen_loc
      @arguments = arguments
      @rparen_loc = rparen_loc
    end

    # def accept: (Visitor visitor) -> void
    def accept(visitor)
      visitor.visit_yield_node(self)
    end

    # def child_nodes: () -> Array[nil | Node]
    def child_nodes
      [arguments]
    end

    # def compact_child_nodes: () -> Array[Node]
    def compact_child_nodes
      compact = [] #: Array[Prism::node]
      compact << arguments if arguments
      compact
    end

    # def comment_targets: () -> Array[Node | Location]
    def comment_targets
      [keyword_loc, *lparen_loc, *arguments, *rparen_loc] #: Array[Prism::node | Location]
    end

    # def copy: (?keyword_loc: Location, ?lparen_loc: Location?, ?arguments: ArgumentsNode?, ?rparen_loc: Location?, ?location: Location) -> YieldNode
    def copy(keyword_loc: self.keyword_loc, lparen_loc: self.lparen_loc, arguments: self.arguments, rparen_loc: self.rparen_loc, location: self.location)
      YieldNode.new(source, keyword_loc, lparen_loc, arguments, rparen_loc, location)
    end

    # def deconstruct: () -> Array[nil | Node]
    alias deconstruct child_nodes

    # def deconstruct_keys: (Array[Symbol] keys) -> { keyword_loc: Location, lparen_loc: Location?, arguments: ArgumentsNode?, rparen_loc: Location?, location: Location }
    def deconstruct_keys(keys)
      { keyword_loc: keyword_loc, lparen_loc: lparen_loc, arguments: arguments, rparen_loc: rparen_loc, location: location }
    end

    # attr_reader keyword_loc: Location
    def keyword_loc
      location = @keyword_loc
      return location if location.is_a?(Location)
      @keyword_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
    end

    # attr_reader lparen_loc: Location?
    def lparen_loc
      location = @lparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @lparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # attr_reader arguments: ArgumentsNode?
    attr_reader :arguments

    # attr_reader rparen_loc: Location?
    def rparen_loc
      location = @rparen_loc
      case location
      when nil
        nil
      when Location
        location
      else
        @rparen_loc = Location.new(source, location >> 32, location & 0xFFFFFFFF)
      end
    end

    # def keyword: () -> String
    def keyword
      keyword_loc.slice
    end

    # def lparen: () -> String?
    def lparen
      lparen_loc&.slice
    end

    # def rparen: () -> String?
    def rparen
      rparen_loc&.slice
    end

    # def inspect -> String
    def inspect
      InspectVisitor.compose(self)
    end

    # Sometimes you want to check an instance of a node against a list of
    # classes to see what kind of behavior to perform. Usually this is done by
    # calling `[cls1, cls2].include?(node.class)` or putting the node into a
    # case statement and doing `case node; when cls1; when cls2; end`. Both of
    # these approaches are relatively slow because of the constant lookups,
    # method calls, and/or array allocations.
    #
    # Instead, you can call #type, which will return to you a symbol that you
    # can use for comparison. This is faster than the other approaches because
    # it uses a single integer comparison, but also because if you're on CRuby
    # you can take advantage of the fact that case statements with all symbol
    # keys will use a jump table.
    #
    # def type: () -> Symbol
    def type
      :yield_node
    end

    # Similar to #type, this method returns a symbol that you can use for
    # splitting on the type of the node without having to do a long === chain.
    # Note that like #type, it will still be slower than using == for a single
    # class, but should be faster in a case statement or an array comparison.
    #
    # def self.type: () -> Symbol
    def self.type
      :yield_node
    end

    # Implements case-equality for the node. This is effectively == but without
    # comparing the value of locations. Locations are checked only for presence.
    def ===(other)
      other.is_a?(YieldNode) &&
        (keyword_loc.nil? == other.keyword_loc.nil?) &&
        (lparen_loc.nil? == other.lparen_loc.nil?) &&
        (arguments === other.arguments) &&
        (rparen_loc.nil? == other.rparen_loc.nil?)
    end
  end

  # Flags for arguments nodes.
  module ArgumentsNodeFlags
    # if arguments contain keywords
    CONTAINS_KEYWORDS = 1 << 0

    # if arguments contain keyword splat
    CONTAINS_KEYWORD_SPLAT = 1 << 1
  end

  # Flags for array nodes.
  module ArrayNodeFlags
    # if array contains splat nodes
    CONTAINS_SPLAT = 1 << 0
  end

  # Flags for call nodes.
  module CallNodeFlags
    # &. operator
    SAFE_NAVIGATION = 1 << 0

    # a call that could have been a local variable
    VARIABLE_CALL = 1 << 1

    # a call that is an attribute write, so the value being written should be returned
    ATTRIBUTE_WRITE = 1 << 2

    # a call that ignores method visibility
    IGNORE_VISIBILITY = 1 << 3
  end

  # Flags for nodes that have unescaped content.
  module EncodingFlags
    # internal bytes forced the encoding to UTF-8
    FORCED_UTF8_ENCODING = 1 << 0

    # internal bytes forced the encoding to binary
    FORCED_BINARY_ENCODING = 1 << 1
  end

  # Flags for integer nodes that correspond to the base of the integer.
  module IntegerBaseFlags
    # 0b prefix
    BINARY = 1 << 0

    # 0d or no prefix
    DECIMAL = 1 << 1

    # 0o or 0 prefix
    OCTAL = 1 << 2

    # 0x prefix
    HEXADECIMAL = 1 << 3
  end

  # Flags for interpolated string nodes that indicated mutability if they are also marked as literals.
  module InterpolatedStringNodeFlags
    # frozen by virtue of a `frozen_string_literal: true` comment or `--enable-frozen-string-literal`; only for adjacent string literals like `'a' 'b'`
    FROZEN = 1 << 0

    # mutable by virtue of a `frozen_string_literal: false` comment or `--disable-frozen-string-literal`; only for adjacent string literals like `'a' 'b'`
    MUTABLE = 1 << 1
  end

  # Flags for keyword hash nodes.
  module KeywordHashNodeFlags
    # a keyword hash which only has `AssocNode` elements all with symbol keys, which means the elements can be treated as keyword arguments
    SYMBOL_KEYS = 1 << 0
  end

  # Flags for while and until loop nodes.
  module LoopFlags
    # a loop after a begin statement, so the body is executed first before the condition
    BEGIN_MODIFIER = 1 << 0
  end

  # Flags for parameter nodes.
  module ParameterFlags
    # a parameter name that has been repeated in the method signature
    REPEATED_PARAMETER = 1 << 0
  end

  # Flags for range and flip-flop nodes.
  module RangeFlags
    # ... operator
    EXCLUDE_END = 1 << 0
  end

  # Flags for regular expression and match last line nodes.
  module RegularExpressionFlags
    # i - ignores the case of characters when matching
    IGNORE_CASE = 1 << 0

    # x - ignores whitespace and allows comments in regular expressions
    EXTENDED = 1 << 1

    # m - allows $ to match the end of lines within strings
    MULTI_LINE = 1 << 2

    # o - only interpolates values into the regular expression once
    ONCE = 1 << 3

    # e - forces the EUC-JP encoding
    EUC_JP = 1 << 4

    # n - forces the ASCII-8BIT encoding
    ASCII_8BIT = 1 << 5

    # s - forces the Windows-31J encoding
    WINDOWS_31J = 1 << 6

    # u - forces the UTF-8 encoding
    UTF_8 = 1 << 7

    # internal bytes forced the encoding to UTF-8
    FORCED_UTF8_ENCODING = 1 << 8

    # internal bytes forced the encoding to binary
    FORCED_BINARY_ENCODING = 1 << 9

    # internal bytes forced the encoding to US-ASCII
    FORCED_US_ASCII_ENCODING = 1 << 10
  end

  # Flags for return nodes.
  module ReturnNodeFlags
    # a return statement that is redundant because it is the last statement in a method
    REDUNDANT = 1 << 0
  end

  # Flags for shareable constant nodes.
  module ShareableConstantNodeFlags
    # constant writes that should be modified with shareable constant value literal
    LITERAL = 1 << 0

    # constant writes that should be modified with shareable constant value experimental everything
    EXPERIMENTAL_EVERYTHING = 1 << 1

    # constant writes that should be modified with shareable constant value experimental copy
    EXPERIMENTAL_COPY = 1 << 2
  end

  # Flags for string nodes.
  module StringFlags
    # internal bytes forced the encoding to UTF-8
    FORCED_UTF8_ENCODING = 1 << 0

    # internal bytes forced the encoding to binary
    FORCED_BINARY_ENCODING = 1 << 1

    # frozen by virtue of a `frozen_string_literal: true` comment or `--enable-frozen-string-literal`
    FROZEN = 1 << 2

    # mutable by virtue of a `frozen_string_literal: false` comment or `--disable-frozen-string-literal`
    MUTABLE = 1 << 3
  end

  # Flags for symbol nodes.
  module SymbolFlags
    # internal bytes forced the encoding to UTF-8
    FORCED_UTF8_ENCODING = 1 << 0

    # internal bytes forced the encoding to binary
    FORCED_BINARY_ENCODING = 1 << 1

    # internal bytes forced the encoding to US-ASCII
    FORCED_US_ASCII_ENCODING = 1 << 2
  end
end
