= tsort.rb

#tsort.rb provides a module for topological sorting and
#strongly connected components.
tsort.rb ϥȥݥ륽ȤȶϢʬ˴ؤ⥸塼󶡤ޤ

== Example

  require 'tsort'

  class Hash
    include TSort
    alias tsort_each_node each_key
    def tsort_each_child(node, &block)
      fetch(node).each(&block)
    end
  end

  {1=>[2, 3], 2=>[3], 3=>[], 4=>[]}.tsort
  #=> [3, 2, 1, 4]

  {1=>[2], 2=>[3, 4], 3=>[2], 4=>[]}.strongly_connected_components
  #=> [[4], [2, 3], [1]]

== TSort module
#TSort implements topological sorting using Tarjan's algorithm for
#strongly connected components.

TSort ϶Ϣʬ˴ؤ Tarjan Υ르ꥺѤ
ȥݥ륽ȤμǤ

#TSort is designed to be able to use with any object which can be interpreted
#as a directed graph.
#TSort requires two methods to interpret a object as a graph:
#tsort_each_node and tsort_each_child.

TSort ǤդΥ֥ȤͭդȤƲǤ褦߷פƤޤ
TSort ֥Ȥ򥰥դȤƲ᤹ˤ2ĤΥ᥽åɤ׵ᤷޤ
ʤ(({tsort_each_node}))  (({tsort_each_child})) Ǥ

# * tsort_each_node is used to iterate for all nodes over a graph.
# * tsort_each_child is used to iterate for child nodes of a given node.

 * (({tsort_each_node})) ϥվΤ٤Ƥĺ󤹤ΤѤޤ
 * (({tsort_each_child})) Ϳ줿ĺλҤ󤹤ΤѤޤ

#The equality of nodes are defined by eql? and hash since
#TSort uses Hash internally.

ĺƱΤ eql?  hash ˤäޤ
 TSort ǥϥåѤƤ뤫Ǥ

=== methods
--- tsort 
#    returns a topologically sorted array of nodes.
#    The array is sorted from children to parents:
#    I.e. the first element has no child and the last node has no parent.

    ĺȥݥ륽Ȥ֤ޤ
    ϻҤƤ˸äƥȤƤޤ
    ʤǽǤϻҤǸǤϿƤޤ

#    If there is a cycle, (({TSort::Cyclic})) is raised.

    ϩ¸ߤȤ㳰 (({TSort::Cyclic})) 򵯤ޤ

--- tsort_each {|node| ...}
#    is the iterator version of the (({tsort})) method.
#    (({((|obj|)).tsort_each})) is similar to (({((|obj|)).tsort.each})) but
#    modification of ((|obj|)) during the iteration may cause unexpected result.

    (({tsort})) ᥽åɤΥƥ졼ǤǤ
    (({((|obj|)).tsort_each}))  (({((|obj|)).tsort.each})) ȻƤޤ
    ֥åɾ ((|obj|)) ѹ줿ͽʤ̤ˤʤ
    Ȥޤ

#    (({tsort_each})) returns (({nil})).
#    If there is a cycle, (({TSort::Cyclic})) is raised.

    (({tsort_each}))  (({nil})) ֤ޤ
    ϩ¸ߤȤ㳰 (({TSort::Cyclic})) 򵯤ޤ

--- strongly_connected_components
#    returns strongly connected components as an array of array of nodes.
#    The array is sorted from children to parents.
#    Each elements of the array represents a strongly connected component.

    ϢʬνޤȤ֤ޤ
    ϻҤƤ˸äƥȤƤޤ
    Ǥ϶ϢʬɽǤ

--- each_strongly_connected_component {|nodes| ...}
#    is the iterator version of the (({strongly_connected_components})) method.
#    (({((|obj|)).each_strongly_connected_component})) is similar to
#    (({((|obj|)).strongly_connected_components.each})) but
#    modification of ((|obj|)) during the iteration may cause unexpected result.

    (({strongly_connected_components})) ᥽åɤΥƥ졼ǤǤ
    (({((|obj|)).each_strongly_connected_component})) 
    (({((|obj|)).strongly_connected_components.each})) ˻Ƥޤ
    ֥åɾ ((|obj|)) ѹ줿ͽʤ̤ˤʤ
    Ȥޤ

#    (({each_strongly_connected_component})) returns (({nil})).

    (({each_strongly_connected_component}))  (({nil})) ֤ޤ

--- each_strongly_connected_component_from(node) {|nodes| ...}
#    iterates over strongly connected component in the subgraph reachable from 
#    ((|node|)).

    ((|node|)) ãǽʶϢʬˤĤƤΥƥ졼Ǥ

#    Return value is unspecified.
    ֤ͤϵꤵƤޤ

#    (({each_strongly_connected_component_from})) doesn't call
#    (({tsort_each_node})).
    (({each_strongly_connected_component_from})) 
    (({tsort_each_node})) ƤӤޤ

--- tsort_each_node {|node| ...}
#    should be implemented by a extended class.
    TSort ǳĥ륯饹ƤʤФʤʤ᥽åɤǤ

    (({tsort_each_node})) is used to iterate for all nodes over a graph.

--- tsort_each_child(node) {|child| ...}
#    should be implemented by a extended class.
    TSort ǳĥ륯饹ƤʤФʤʤ᥽åɤǤ

    (({tsort_each_child})) is used to iterate for child nodes of ((|node|)).

#== More Realistic Example
#Very simple `make' like tool can be implemented as follows:

== 긽Ū

ñ `make' ˻ġϰʲΤ褦˼Ǥޤ

  require 'tsort'

  class Make
    def initialize
      @dep = {}
      @dep.default = []
    end

    def rule(outputs, inputs=[], &block)
      triple = [outputs, inputs, block]
      outputs.each {|f| @dep[f] = [triple]}
      @dep[triple] = inputs
    end

    def build(target)
      each_strongly_connected_component_from(target) {|ns|
        if ns.length != 1
          fs = ns.delete_if {|n| Array === n}
          raise TSort::Cyclic.new("cyclic dependencies: #{fs.join ', '}")
        end
        n = ns.first
        if Array === n
          outputs, inputs, block = n
          inputs_time = inputs.map {|f| File.mtime f}.max
          begin
            outputs_time = outputs.map {|f| File.mtime f}.min
          rescue Errno::ENOENT
            outputs_time = nil
          end
          if outputs_time == nil ||
             inputs_time != nil && outputs_time <= inputs_time
            sleep 1 if inputs_time != nil && inputs_time.to_i == Time.now.to_i
            block.call
          end
        end
      }
    end

    def tsort_each_child(node, &block)
      @dep[node].each(&block)
    end
    include TSort
  end

  def command(arg)
    print arg, "\n"
    system arg
  end

  m = Make.new
  m.rule(%w[t1]) { command 'date > t1' }
  m.rule(%w[t2]) { command 'date > t2' }
  m.rule(%w[t3]) { command 'date > t3' }
  m.rule(%w[t4], %w[t1 t3]) { command 'cat t1 t3 > t4' }
  m.rule(%w[t5], %w[t4 t2]) { command 'cat t4 t2 > t5' }
  m.build('t5')

== Bugs

#* (('tsort.rb')) is wrong name because this library uses
#  Tarjan's algorithm for strongly connected components.
#  Although (('strongly_connected_components.rb')) is correct but too long,

* (('tsort.rb')) Ȥ̾ΤǤϤޤ󡣤ʤʤ
  Υ饤֥ Tarjan ζϢʬ˴ؤ륢르ꥺȤäƤ뤫Ǥ
  ȤϤ (('strongly_connected_components.rb')) ȤΤ̾Ĺ᤮ޤ

== References
R. E. Tarjan, 
Depth First Search and Linear Graph Algorithms,
SIAM Journal on Computing, Vol. 1, No. 2, pp. 146-160, June 1972.

#@Article{Tarjan:1972:DFS,
#  author =       "R. E. Tarjan",
#  key =          "Tarjan",
#  title =        "Depth First Search and Linear Graph Algorithms",
#  journal =      j-SIAM-J-COMPUT,
#  volume =       "1",
#  number =       "2",
#  pages =        "146--160",
#  month =        jun,
#  year =         "1972",
#  CODEN =        "SMJCAT",
#  ISSN =         "0097-5397 (print), 1095-7111 (electronic)",
#  bibdate =      "Thu Jan 23 09:56:44 1997",
#  bibsource =    "Parallel/Multi.bib, Misc/Reverse.eng.bib",
#}
