/*
 * Driver for PowerMac AWACS/Burgundy/DACA/Tumbler
 * Copyright (c) 2000 by Takashi Iwai <tiwai@suse.de>
 *   based on dmasound.c.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE

#include "../../include/driver.h"
#include "../../include/info.h"
#include "../../include/control.h"
#include "../../include/mixer.h"
#include "../../include/pcm.h"
#include "../../include/initval.h"
#include <linux/init.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
#include <asm/adb.h>
#include <asm/cuda.h>
#include <asm/pmu.h>
#else /* 2.4.0 kernel */
#include <linux/adb.h>
#ifdef CONFIG_ADB_CUDA
#include <linux/cuda.h>
#endif
#ifdef CONFIG_ADB_PMU
#include <linux/pmu.h>
#endif
#endif
#include <linux/nvram.h>
#include <linux/vt_kern.h>
#include <linux/kmod.h>
#include <asm/dbdma.h>
#include <asm/prom.h>
#include <asm/machdep.h>
#ifdef CONFIG_PPC_HAS_FEATURE_CALLS
#include <asm/pmac_feature.h>
#else
#include <asm/feature.h>
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
#define pmu_suspend()	/**/
#define pmu_resume()	/**/
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0) || defined(CONFIG_ADB_CUDA)
#define AMP_AVAIL
#endif

/* enable beep hack */
#define ENABLE_BEEP

#define CARD_NAME "PMac AWACS"
#define DRIVER_NAME CARD_NAME

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: PowerMac AWACS\n\
Card: PowerMac AWACS\n\
");
MODULE_LICENSE("GPL");

static int snd_index = SND_DEFAULT_IDX1;		/* Index 0-MAX */
static char *snd_id = SND_DEFAULT_STR1;		/* ID for this card */
static int snd_enable = 1;
static int snd_dac_frame_size = 128;
static int snd_adc_frame_size = 128;
static int snd_auto_mute = 1;
static int snd_auto_mute_mask = 0;
#ifdef ENABLE_BEEP
static int snd_enable_beep = 1;
#endif

MODULE_PARM(snd_index, "i");
MODULE_PARM_DESC(snd_index, "Index value for " CARD_NAME " soundcard.");
MODULE_PARM(snd_id, "s");
MODULE_PARM_DESC(snd_id, "ID string for " CARD_NAME " soundcard.");
MODULE_PARM(snd_enable, "i");
MODULE_PARM_DESC(snd_enable, "Enable this soundcard. [BOOL]");
MODULE_PARM(snd_dac_frame_size, "i");
MODULE_PARM_DESC(snd_dac_frame_size, "DAC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_adc_frame_size, "i");
MODULE_PARM_DESC(snd_adc_frame_size, "ADC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_auto_mute, "i");
MODULE_PARM_DESC(snd_auto_mute, "Switch headphone and speaker automatically. [BOOL]");
MODULE_PARM(snd_auto_mute_mask, "i");
MODULE_PARM_DESC(snd_auto_mute_mask, "Bit mask for control port to check the headphone status.");
#ifdef ENABLE_BEEP
MODULE_PARM(snd_enable_beep, "i");
MODULE_PARM_DESC(snd_enable_beep, "Enable beep using PCM. [BOOL]");
#endif


/*
 * card entry
 */

static snd_card_t *snd_pmac_card = NULL;

/*
 */

/*******************************/
/* AWACs Audio Register Layout */
/*******************************/

struct awacs_regs {
    unsigned	control;	/* Audio control register */
    unsigned	pad0[3];
    unsigned	codec_ctrl;	/* Codec control register */
    unsigned	pad1[3];
    unsigned	codec_stat;	/* Codec status register */
    unsigned	pad2[3];
    unsigned	clip_count;	/* Clipping count register */
    unsigned	pad3[3];
    unsigned	byteswap;	/* Data is little-endian if 1 */
};

/*******************/
/* Audio Bit Masks */
/*******************/

/* Audio Control Reg Bit Masks */
/* ----- ------- --- --- ----- */
#define MASK_ISFSEL	(0xf)		/* Input SubFrame Select */
#define MASK_OSFSEL	(0xf << 4)	/* Output SubFrame Select */
#define MASK_RATE	(0x7 << 8)	/* Sound Rate */
#define MASK_CNTLERR	(0x1 << 11)	/* Error */
#define MASK_PORTCHG	(0x1 << 12)	/* Port Change */
#define MASK_IEE	(0x1 << 13)	/* Enable Interrupt on Error */
#define MASK_IEPC	(0x1 << 14)	/* Enable Interrupt on Port Change */
#define MASK_SSFSEL	(0x3 << 15)	/* Status SubFrame Select */

/* Audio Codec Control Reg Bit Masks */
/* ----- ----- ------- --- --- ----- */
#define MASK_NEWECMD	(0x1 << 24)	/* Lock: don't write to reg when 1 */
#define MASK_EMODESEL	(0x3 << 22)	/* Send info out on which frame? */
#define MASK_EXMODEADDR	(0x3ff << 12)	/* Extended Mode Address -- 10 bits */
#define MASK_EXMODEDATA	(0xfff)		/* Extended Mode Data -- 12 bits */

/* Audio Codec Control Address Values / Masks */
/* ----- ----- ------- ------- ------ - ----- */
#define MASK_ADDR0	(0x0 << 12)	/* Expanded Data Mode Address 0 */
#define MASK_ADDR_MUX	MASK_ADDR0	/* Mux Control */
#define MASK_ADDR_GAIN	MASK_ADDR0

#define MASK_ADDR1	(0x1 << 12)	/* Expanded Data Mode Address 1 */
#define MASK_ADDR_MUTE	MASK_ADDR1
#define MASK_ADDR_RATE	MASK_ADDR1

#define MASK_ADDR2	(0x2 << 12)	/* Expanded Data Mode Address 2 */
#define MASK_ADDR_VOLA	MASK_ADDR2	/* Volume Control A -- Headphones */
#define MASK_ADDR_VOLHD MASK_ADDR2

#define MASK_ADDR4	(0x4 << 12)	/* Expanded Data Mode Address 4 */
#define MASK_ADDR_VOLC	MASK_ADDR4	/* Volume Control C -- Speaker */
#define MASK_ADDR_VOLSPK MASK_ADDR4

/* additional registers of screamer */
#define MASK_ADDR5	(0x5 << 12)	/* Expanded Data Mode Address 5 */
#define MASK_ADDR6	(0x6 << 12)	/* Expanded Data Mode Address 6 */
#define MASK_ADDR7	(0x7 << 12)	/* Expanded Data Mode Address 7 */

/* Address 0 Bit Masks & Macros */
/* ------- - --- ----- - ------ */
#define MASK_GAINRIGHT	(0xf)		/* Gain Right Mask */
#define MASK_GAINLEFT	(0xf << 4)	/* Gain Left Mask */
#define MASK_GAINLINE	(0x1 << 8)	/* Disable Mic preamp */
#define MASK_GAINMIC	(0x0 << 8)	/* Enable Mic preamp */

#define MASK_MUX_CD	(0x1 << 9)	/* Select CD in MUX */
#define MASK_MUX_MIC	(0x1 << 10)	/* Select Mic in MUX */
#define MASK_MUX_AUDIN	(0x1 << 11)	/* Select Audio In in MUX */
#define MASK_MUX_LINE	MASK_MUX_AUDIN

#define GAINRIGHT(x)	((x) & MASK_GAINRIGHT)
#define GAINLEFT(x)	(((x) << 4) & MASK_GAINLEFT)

/* Address 1 Bit Masks */
/* ------- - --- ----- */
#define MASK_ADDR1RES1	(0x3)		/* Reserved */
#define MASK_RECALIBRATE (0x1 << 2)	/* Recalibrate */
#define MASK_SAMPLERATE	(0x7 << 3)	/* Sample Rate: */
#define MASK_LOOPTHRU	(0x1 << 6)	/* Loopthrough Enable */
#define MASK_CMUTE	(0x1 << 7)	/* Output C (Speaker) Mute when 1 */
#define MASK_SPKMUTE	MASK_CMUTE
#define MASK_ADDR1RES2	(0x1 << 8)	/* Reserved */
#define MASK_AMUTE	(0x1 << 9)	/* Output A (Headphone) Mute when 1 */
#define MASK_HDMUTE	MASK_AMUTE
#define MASK_PAROUT0	(0x1 << 10)	/* Parallel Output 0 */
#define MASK_PAROUT1	(0x2 << 10)	/* Parallel Output 1 */
#define MASK_PAROUT	(MASK_PAROUT0|MASK_PAROUT1)

#define MASK_MIC_BOOST  (0x4)           /* screamer mic boost */

#define SAMPLERATE_48000	(0x0 << 3)	/* 48 or 44.1 kHz */
#define SAMPLERATE_32000	(0x1 << 3)	/* 32 or 29.4 kHz */
#define SAMPLERATE_24000	(0x2 << 3)	/* 24 or 22.05 kHz */
#define SAMPLERATE_19200	(0x3 << 3)	/* 19.2 or 17.64 kHz */
#define SAMPLERATE_16000	(0x4 << 3)	/* 16 or 14.7 kHz */
#define SAMPLERATE_12000	(0x5 << 3)	/* 12 or 11.025 kHz */
#define SAMPLERATE_9600		(0x6 << 3)	/* 9.6 or 8.82 kHz */
#define SAMPLERATE_8000		(0x7 << 3)	/* 8 or 7.35 kHz */

/* Address 2 & 4 Bit Masks & Macros */
/* ------- - - - --- ----- - ------ */
#define MASK_OUTVOLRIGHT (0xf)		/* Output Right Volume */
#define MASK_ADDR2RES1	(0x2 << 4)	/* Reserved */
#define MASK_ADDR4RES1	MASK_ADDR2RES1
#define MASK_OUTVOLLEFT	(0xf << 6)	/* Output Left Volume */
#define MASK_ADDR2RES2	(0x2 << 10)	/* Reserved */
#define MASK_ADDR4RES2	MASK_ADDR2RES2

#define VOLRIGHT(x)	(((~(x)) & MASK_OUTVOLRIGHT))
#define VOLLEFT(x)	(((~(x)) << 6) & MASK_OUTVOLLEFT)

/* Audio Codec Status Reg Bit Masks */
/* ----- ----- ------ --- --- ----- */
#define MASK_EXTEND	(0x1 << 23)	/* Extend */
#define MASK_VALID	(0x1 << 22)	/* Valid Data? */
#define MASK_OFLEFT	(0x1 << 21)	/* Overflow Left */
#define MASK_OFRIGHT	(0x1 << 20)	/* Overflow Right */
#define MASK_ERRCODE	(0xf << 16)	/* Error Code */
#define MASK_REVISION	(0xf << 12)	/* Revision Number */
#define MASK_MFGID	(0xf << 8)	/* Mfg. ID */
#define MASK_CODSTATRES	(0xf << 4)	/* bits 4 - 7 reserved */
#define MASK_INPPORT	(0xf)		/* Input Port */
#define MASK_HDPCONN	8		/* headphone plugged in */
#define MASK_MICCONN	2		/* microphone plugged in */

/* Clipping Count Reg Bit Masks */
/* -------- ----- --- --- ----- */
#define MASK_CLIPLEFT	(0xff << 7)	/* Clipping Count, Left Channel */
#define MASK_CLIPRIGHT	(0xff)		/* Clipping Count, Right Channel */

/* DBDMA ChannelStatus Bit Masks */
/* ----- ------------- --- ----- */
#define MASK_CSERR	(0x1 << 7)	/* Error */
#define MASK_EOI	(0x1 << 6)	/* End of Input -- only for Input Channel */
#define MASK_CSUNUSED	(0x1f << 1)	/* bits 1-5 not used */
#define MASK_WAIT	(0x1)		/* Wait */

/* Various Rates */
/* ------- ----- */
#define RATE_48000	(0x0 << 8)	/* 48 kHz */
#define RATE_44100	(0x0 << 8)	/* 44.1 kHz */
#define RATE_32000	(0x1 << 8)	/* 32 kHz */
#define RATE_29400	(0x1 << 8)	/* 29.4 kHz */
#define RATE_24000	(0x2 << 8)	/* 24 kHz */
#define RATE_22050	(0x2 << 8)	/* 22.05 kHz */
#define RATE_19200	(0x3 << 8)	/* 19.2 kHz */
#define RATE_17640	(0x3 << 8)	/* 17.64 kHz */
#define RATE_16000	(0x4 << 8)	/* 16 kHz */
#define RATE_14700	(0x4 << 8)	/* 14.7 kHz */
#define RATE_12000	(0x5 << 8)	/* 12 kHz */
#define RATE_11025	(0x5 << 8)	/* 11.025 kHz */
#define RATE_9600	(0x6 << 8)	/* 9.6 kHz */
#define RATE_8820	(0x6 << 8)	/* 8.82 kHz */
#define RATE_8000	(0x7 << 8)	/* 8 kHz */
#define RATE_7350	(0x7 << 8)	/* 7.35 kHz */

#define RATE_LOW	1	/* HIGH = 48kHz, etc;  LOW = 44.1kHz, etc. */


/* Burgundy values */

#define MASK_ADDR_BURGUNDY_INPSEL21 (0x11 << 12)
#define MASK_ADDR_BURGUNDY_INPSEL3 (0x12 << 12)

#define MASK_ADDR_BURGUNDY_GAINCH1 (0x13 << 12)
#define MASK_ADDR_BURGUNDY_GAINCH2 (0x14 << 12)
#define MASK_ADDR_BURGUNDY_GAINCH3 (0x15 << 12)
#define MASK_ADDR_BURGUNDY_GAINCH4 (0x16 << 12)

#define MASK_ADDR_BURGUNDY_VOLCH1 (0x20 << 12)
#define MASK_ADDR_BURGUNDY_VOLCH2 (0x21 << 12)
#define MASK_ADDR_BURGUNDY_VOLCH3 (0x22 << 12)
#define MASK_ADDR_BURGUNDY_VOLCH4 (0x23 << 12)

#define MASK_ADDR_BURGUNDY_OUTPUTSELECTS (0x2B << 12)
#define MASK_ADDR_BURGUNDY_OUTPUTENABLES (0x2F << 12)

#define MASK_ADDR_BURGUNDY_MASTER_VOLUME (0x30 << 12)

#define MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES (0x60 << 12)

#define MASK_ADDR_BURGUNDY_ATTENSPEAKER (0x62 << 12)
#define MASK_ADDR_BURGUNDY_ATTENLINEOUT (0x63 << 12)
#define MASK_ADDR_BURGUNDY_ATTENHP (0x64 << 12)

#define MASK_ADDR_BURGUNDY_VOLCD (MASK_ADDR_BURGUNDY_VOLCH1)
#define MASK_ADDR_BURGUNDY_VOLLINE (MASK_ADDR_BURGUNDY_VOLCH2)
#define MASK_ADDR_BURGUNDY_VOLMIC (MASK_ADDR_BURGUNDY_VOLCH3)
#define MASK_ADDR_BURGUNDY_VOLMODEM (MASK_ADDR_BURGUNDY_VOLCH4)

#define MASK_ADDR_BURGUNDY_GAINCD (MASK_ADDR_BURGUNDY_GAINCH1)
#define MASK_ADDR_BURGUNDY_GAINLINE (MASK_ADDR_BURGUNDY_GAINCH2)
#define MASK_ADDR_BURGUNDY_GAINMIC (MASK_ADDR_BURGUNDY_GAINCH3)
#define MASK_ADDR_BURGUNDY_GAINMODEM (MASK_ADDR_BURGUNDY_VOLCH4)


/* These are all default values for the burgundy */
#define DEF_BURGUNDY_INPSEL21 (0xAA)
#define DEF_BURGUNDY_INPSEL3 (0x0A)

#define DEF_BURGUNDY_GAINCD (0x33)
#define DEF_BURGUNDY_GAINLINE (0x44)
#define DEF_BURGUNDY_GAINMIC (0x44)
#define DEF_BURGUNDY_GAINMODEM (0x06)

/* Remember: lowest volume here is 0x9b */
#define DEF_BURGUNDY_VOLCD (0xCCCCCCCC)
#define DEF_BURGUNDY_VOLLINE (0x00000000)
#define DEF_BURGUNDY_VOLMIC (0x00000000)
#define DEF_BURGUNDY_VOLMODEM (0xCCCCCCCC)

#define DEF_BURGUNDY_OUTPUTSELECTS (0x010f010f)
#define DEF_BURGUNDY_OUTPUTENABLES (0x0A)

#define DEF_BURGUNDY_MASTER_VOLUME (0xFFFFFFFF)

#define DEF_BURGUNDY_MORE_OUTPUTENABLES (0x7E)

#define DEF_BURGUNDY_ATTENSPEAKER (0x44)
#define DEF_BURGUNDY_ATTENLINEOUT (0xCC)
#define DEF_BURGUNDY_ATTENHP (0xCC)

/* known register values for OUTPUTENABLES */
#define BURGUNDY_OUTPUT_LEFT	0x02
#define BURGUNDY_OUTPUT_RIGHT	0x04
#define BURGUNDY_OUTPUT_INTERN	0x80


/*
 *
 */

/* maximum number of fragments */
#define PMAC_MAX_FRAGS		32

/* frequency table */
static int awacs_freqs[8] = {
	44100, 29400, 22050, 17640, 14700, 11025, 8820, 7350
};
static int tumbler_freqs[2] = {
	48000, 44100
};

typedef struct snd_pmac_stream {
	int running;

	int stream;	/* PLAYBACK/CAPTURE */

	int dma_size, frag_size; /* in bytes */
	int buffer_size; /* in kb */
	int nfrags, cur_frag;

	snd_dma_t *dmaptr;
	volatile struct dbdma_regs *dma;
	volatile struct dbdma_cmd *space;
	volatile struct dbdma_cmd *cmds;

	snd_irq_t *irqptr;

	snd_pcm_subchn_t *substream;

	snd_pcm_hardware_t *hwinfo;

	spinlock_t lock;

} pmac_stream_t;

typedef struct {
	unsigned int voices;
	unsigned int caps;
	snd_kmixer_element_t *me_io;
	snd_mixer_volume1_control_t *volume;
	snd_kmixer_element_t *me_volume;
	int vol_min, vol_max;
	snd_mixer_sw1_control_t *mute; /* sw2 too */
	snd_kmixer_element_t *me_mute;
	snd_mixer_sw2_control_t *capture;
	snd_kmixer_element_t *me_capture;
	snd_kmixer_element_t *in, *out;
} pmac_mixer_group_t;

enum {
	PMAC_AWACS, PMAC_SCREAMER, PMAC_BURGUNDY, PMAC_DACA, PMAC_TUMBLER
}; /* model */

typedef struct snd_pmac {
	snd_card_t *card;

	struct device_node *node;
	unsigned int revision;
	unsigned int subframe;
	unsigned int device_id;
	unsigned int has_iic : 1;

	int model;
	unsigned int is_pbook_3400 : 1;
	unsigned int is_pbook_G3 : 1;

	unsigned int can_byte_swap : 1;
	unsigned int can_capture : 1;
	unsigned int can_duplex : 1;

#ifdef AMP_AVAIL
	unsigned int amp_only;
	int amp_vol[2];
#endif

	int initialized : 1;

	int num_freqs;
	int *freq_table;
	unsigned int freqs_ok; /* bit flags */
	int control; /* control byte */

	int active;
	int rate_index;
	int format;

	spinlock_t reg_lock;
	volatile struct awacs_regs *awacs;
	int awacs_reg[8]; /* register cache */

	unsigned char *latch_base;
	unsigned char *macio_base;

	pmac_stream_t playback;
	pmac_stream_t capture;

	volatile struct dbdma_cmd *extra_space;
	volatile struct dbdma_cmd *extra_cmd;

#ifdef ENABLE_BEEP
	int beep_on;
	int beep_volume;
	int beep_volume_play;
	int beep_hz;
	int beep_nsamples;
	short *beep_buf;
	struct timer_list beep_timer;
	void (*orig_mksound)(unsigned int, unsigned int);
#endif

	snd_irq_t *irqptr; /* control interrupt */

	snd_pcm_t *pcm;

	int auto_mute;
	int auto_mute_state;
	unsigned int hp_stat_mask;

	unsigned long i2c_base;
	int i2c_addr;
	int i2c_steps;

	int awacs_volume_cache[5][2];

	int mixer_volume[2]; /* for tumbler/daca */
	int bass_volume, treble_volume;
	unsigned int daca_amp : 1; /* DACA amp */

	snd_kmixer_element_t *me_tone;
	pmac_mixer_group_t mix_master;
	pmac_mixer_group_t mix_lineout;
	pmac_mixer_group_t mix_speaker;
	pmac_mixer_group_t mix_beep;

	pmac_mixer_group_t mix_line;
	pmac_mixer_group_t mix_cd;
	pmac_mixer_group_t mix_mic;
	pmac_mixer_group_t mix_mic_gain;
	pmac_mixer_group_t mix_capture;

	/* tumbler stuffs */
	snd_irq_t *headphone_irqptr;
	void *amp_mute;
	void *headphone_mute;
	void *headphone_status;

} pmac_t;


/*
 * prototypes
 */

static void snd_pmac_awacs_intr(int irq, void *devid, struct pt_regs *regs);
static void snd_pmac_awacs_tx_intr(int irq, void *devid, struct pt_regs *regs);
static void snd_pmac_awacs_rx_intr(int irq, void *devid, struct pt_regs *regs);

static int snd_pmac_awacs_init(pmac_t *chip);
static int snd_pmac_burgundy_init(pmac_t *chip);
static int snd_pmac_tumbler_init(pmac_t *chip);
static void snd_pmac_awacs_enable_amp(pmac_t *chip, int *spkr_vol);
static unsigned int snd_pmac_fixed_rate(pmac_t *chip, unsigned int rate);

static int snd_pmac_i2c_init(pmac_t *chip);
static void snd_pmac_i2c_cleanup(pmac_t *chip);
 
static void snd_pmac_auto_mute(pmac_t *chip, int intr);

/*
 * allocate DBDMA command arrays
 */
static int snd_pmac_dbdma_alloc(volatile struct dbdma_cmd **space,
				volatile struct dbdma_cmd **cmds,
				int size)
{
	*space = snd_kcalloc(sizeof(struct dbdma_cmd) * (size + 1), GFP_KERNEL);
	if (*space == NULL)
		return -ENOMEM;
	*cmds = (volatile struct dbdma_cmd *)DBDMA_ALIGN(*space);
	return 0;
}

/*
 * write
 */
static void
snd_pmac_awacs_write(pmac_t *chip, int val)
{
	long timeout = 5000000;

	if (chip->model > PMAC_SCREAMER)
		return;

	while (in_le32(&chip->awacs->codec_ctrl) & MASK_NEWECMD) {
		if (! --timeout) {
			snd_printd("snd_pmac_awacs_write timeout\n");
			break;
		}
	}
	out_le32(&chip->awacs->codec_ctrl, val | (chip->subframe << 22));
}

inline static void
snd_pmac_awacs_write_reg(pmac_t *chip, int reg, int val)
{
	snd_pmac_awacs_write(chip, val | (reg << 12));
	chip->awacs_reg[reg] = val;
}

inline static void
snd_pmac_awacs_write_noreg(pmac_t *chip, int reg, int val)
{
	snd_pmac_awacs_write(chip, val | (reg << 12));
}

/*
 * detect sound chip
 */
static int __init snd_pmac_detect(pmac_t *chip)
{
	struct device_node *sound;
	unsigned int *prop, l;

	if (_machine != _MACH_Pmac)
		return -ENODEV;

	chip->subframe = 0;
	chip->revision = 0;
	chip->freqs_ok = ~0; /* all ok */
	chip->model = PMAC_AWACS;
	chip->can_byte_swap = 1;
	chip->can_capture = 1;
	chip->can_duplex = 1;
	chip->num_freqs = 8;
	chip->freq_table = awacs_freqs;
	chip->control = 0x11 | MASK_IEE | MASK_IEPC;

	/* it seems the Pismo & iBook can't byte-swap in hardware. */
	if (machine_is_compatible("PowerBook3,1") ||
	    machine_is_compatible("PowerBook2,1"))
		chip->can_byte_swap = 0 ;
	if (machine_is_compatible("PowerBook2,1"))
		chip->can_duplex = 0;

	/* check machine type */
	if (machine_is_compatible("AAPL,3400/2400")
	    || machine_is_compatible("AAPL,3500"))
		chip->is_pbook_3400 = 1;
	else if (machine_is_compatible("PowerBook1,1")
		 || machine_is_compatible("AAPL,PowerBook1998"))
		chip->is_pbook_G3 = 1;
	chip->node = find_devices("awacs");
	if (chip->node)
		return 0; /* ok */

	/*
	 * powermac G3 models have a node called "davbus"
	 * with a child called "sound".
	 */
	chip->node = find_devices("davbus");
	/*
	 * if we didn't find a davbus device, try 'i2s-a' since
	 * this seems to be what iBooks have
	 */
	if (! chip->node)
		chip->node = find_devices("i2s-a");
	if (! chip->node)
		return -ENODEV;
	sound = find_devices("sound");
	while (sound && sound->parent != chip->node)
		sound = sound->next;
	if (! sound)
		return -ENODEV;
	prop = (unsigned int *) get_property(sound, "sub-frame", 0);
	if (prop && *prop < 16)
		chip->subframe = *prop;
	if (device_is_compatible(sound, "screamer")) {
		chip->model = PMAC_SCREAMER;
		chip->can_byte_swap = 0;  /* XXX correct? */
	}
	if (device_is_compatible(sound, "burgundy")) {
		chip->model = PMAC_BURGUNDY;
		chip->control = 0x11 | MASK_IEPC; /* no error intr */
	}
	if (device_is_compatible(sound, "daca")) {
		chip->model = PMAC_DACA;
		chip->can_capture = 0;
		chip->can_duplex = 0; /* no capture */
		chip->can_byte_swap = 0; /* no le support */
		chip->control = 0x11 | MASK_IEPC; /* no error intr */
	}
	if (device_is_compatible(sound, "tumbler")) {
		chip->model = PMAC_TUMBLER;
		chip->can_capture = 0;
		chip->can_duplex = 0;
		chip->can_byte_swap = 0;
		chip->num_freqs = 2;
		chip->freq_table = tumbler_freqs;
		chip->control = 0x11 | MASK_IEPC; /* no error intr */
	}
	prop = (unsigned int *)get_property(sound, "device-id", 0);
	if (prop)
		chip->device_id = *prop;
	chip->has_iic = (find_devices("perch") != NULL);

	/* look for a property saying what sample rates
	   are available */
	prop = (unsigned int *) get_property(sound, "sample-rates", &l);
	if (! prop)
		prop = (unsigned int *) get_property(sound, "output-frame-rates", &l);
	if (prop) {
		int i;
		chip->freqs_ok = 0;
		for (l /= sizeof(int); l > 0; --l) {
			unsigned int r = *prop++;
			/* Apple 'Fixed' format */
			if (r >= 0x10000)
				r >>= 16;
			for (i = 0; i < chip->num_freqs; ++i) {
				if (r == chip->freq_table[i]) {
					chip->freqs_ok |= (1 << i);
					break;
				}
			}
		}
	} else {
		/* assume only 44.1khz */
		chip->freqs_ok = 1;
	}
	return 0;
}

/*
 * allocate resources
 */
static int __init snd_pmac_resources(pmac_t *chip)
{
	struct device_node *np = chip->node;
	int err;

	if (np->n_addrs < 3 || np->n_intrs < 3)
		return -ENODEV;

	/* all OF versions I've seen use this value */
	chip->awacs = (volatile struct awacs_regs *) ioremap(np->addrs[0].address, 0x1000);
	chip->playback.dma = (volatile struct dbdma_regs *) ioremap(np->addrs[1].address, 0x100);
	chip->capture.dma = (volatile struct dbdma_regs *) ioremap(np->addrs[2].address, 0x100);
	if ((err = snd_register_interrupt(chip->card, "AWACS",
					  np->intrs[0].line,
					  SND_IRQ_TYPE_PCI,
					  snd_pmac_awacs_intr,
					  chip, NULL, &chip->irqptr)) < 0)
		return err;
	if ((err = snd_register_interrupt(chip->card, "AWACS Output",
					  np->intrs[1].line,
					  SND_IRQ_TYPE_PCI,
					  snd_pmac_awacs_tx_intr,
					  chip, NULL, &chip->playback.irqptr)) < 0)
		return err;
	if ((err = snd_register_interrupt(chip->card, "AWACS Input",
					  np->intrs[2].line,
					  SND_IRQ_TYPE_PCI,
					  snd_pmac_awacs_rx_intr,
					  chip, NULL, &chip->capture.irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(chip->card, "AWACS - DAC", 0,
					    SND_DMA_TYPE_PCI,
					    chip->playback.buffer_size,
					    NULL, &chip->playback.dmaptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(chip->card, "AWACS - ADC", 1,
					    SND_DMA_TYPE_PCI,
					    chip->capture.buffer_size,
					    NULL, &chip->capture.dmaptr)) < 0)
		return err;


	return 0;
}

inline static void
snd_pmac_wait_ack(pmac_stream_t *rec)
{
	int timeout = 50000;
	while ((in_le32(&rec->dma->status) & RUN) && timeout-- > 0)
		udelay(1);
}


#ifdef ENABLE_BEEP
/*
 * beep stuff
 */

/*
 * Stuff for outputting a beep.  The values range from -327 to +327
 * so we can multiply by an amplitude in the range 0..100 to get a
 * signed short value to put in the output buffer.
 */
static short beep_wform[256] = {
	0,	40,	79,	117,	153,	187,	218,	245,
	269,	288,	304,	316,	323,	327,	327,	324,
	318,	310,	299,	288,	275,	262,	249,	236,
	224,	213,	204,	196,	190,	186,	183,	182,
	182,	183,	186,	189,	192,	196,	200,	203,
	206,	208,	209,	209,	209,	207,	204,	201,
	197,	193,	188,	183,	179,	174,	170,	166,
	163,	161,	160,	159,	159,	160,	161,	162,
	164,	166,	168,	169,	171,	171,	171,	170,
	169,	167,	163,	159,	155,	150,	144,	139,
	133,	128,	122,	117,	113,	110,	107,	105,
	103,	103,	103,	103,	104,	104,	105,	105,
	105,	103,	101,	97,	92,	86,	78,	68,
	58,	45,	32,	18,	3,	-11,	-26,	-41,
	-55,	-68,	-79,	-88,	-95,	-100,	-102,	-102,
	-99,	-93,	-85,	-75,	-62,	-48,	-33,	-16,
	0,	16,	33,	48,	62,	75,	85,	93,
	99,	102,	102,	100,	95,	88,	79,	68,
	55,	41,	26,	11,	-3,	-18,	-32,	-45,
	-58,	-68,	-78,	-86,	-92,	-97,	-101,	-103,
	-105,	-105,	-105,	-104,	-104,	-103,	-103,	-103,
	-103,	-105,	-107,	-110,	-113,	-117,	-122,	-128,
	-133,	-139,	-144,	-150,	-155,	-159,	-163,	-167,
	-169,	-170,	-171,	-171,	-171,	-169,	-168,	-166,
	-164,	-162,	-161,	-160,	-159,	-159,	-160,	-161,
	-163,	-166,	-170,	-174,	-179,	-183,	-188,	-193,
	-197,	-201,	-204,	-207,	-209,	-209,	-209,	-208,
	-206,	-203,	-200,	-196,	-192,	-189,	-186,	-183,
	-182,	-182,	-183,	-186,	-190,	-196,	-204,	-213,
	-224,	-236,	-249,	-262,	-275,	-288,	-299,	-310,
	-318,	-324,	-327,	-327,	-323,	-316,	-304,	-288,
	-269,	-245,	-218,	-187,	-153,	-117,	-79,	-40,
};

#define BEEP_SRATE	22050	/* 22050 Hz sample rate */
#define BEEP_BUFLEN	512
#define BEEP_VOLUME	15	/* 0 - 100 */

static void snd_pmac_beep_stop(unsigned long data)
{
	//pmac_t *chip = snd_magic_cast(pmac_t, (void*)data,);
	pmac_t *chip = snd_magic_cast(pmac_t, snd_pmac_card->private_data,);
	pmac_stream_t *rec = &chip->playback;

	spin_lock(&chip->reg_lock);
	if (chip->beep_on) {
		st_le16(&chip->extra_cmd->command, DBDMA_STOP);
		spin_lock(&rec->lock);
		out_le32(&rec->dma->control, (RUN|PAUSE|FLUSH|WAKE) << 16);
		spin_unlock(&rec->lock);
		snd_pmac_wait_ack(rec);
		out_le32(&chip->awacs->control, chip->control | (chip->rate_index << 8));
		out_le32(&chip->awacs->byteswap,
			 chip->format == SND_PCM_SFMT_S16_LE);
		chip->beep_on = 0;
	}
	spin_unlock(&chip->reg_lock);
}

static void snd_pmac_mksound(unsigned int hz, unsigned int ticks)
{
	pmac_t *chip;
	pmac_stream_t *rec;
	unsigned long flags;
	int beep_speed = 0;
	int srate;
	int period, ncycles, nsamples;
	int i, j, f;
	short *p;

	/* hmm, this callback has no private data.. ;-< */
	snd_debug_check(snd_pmac_card == NULL,);
	chip = snd_magic_cast(pmac_t, snd_pmac_card->private_data,);
	rec = &chip->playback;

	beep_speed = snd_pmac_fixed_rate(chip, BEEP_SRATE);
	srate = chip->freq_table[beep_speed];

	if (hz <= srate / BEEP_BUFLEN || hz > srate / 2) {
		/* this is a hack for broken X server code */
		hz = 750;
		ticks = 12;
	}

	spin_lock_irqsave(&chip->reg_lock, flags);
	if (chip->playback.running || chip->capture.running || chip->beep_on) {
		spin_unlock_irqrestore(&chip->reg_lock, flags);
		return;
	}
	chip->beep_on = 1;
	spin_unlock_irqrestore(&chip->reg_lock, flags);

	if (hz == chip->beep_hz && chip->beep_volume == chip->beep_volume_play) {
		nsamples = chip->beep_nsamples;
	} else {
		period = srate * 256 / hz;	/* fixed point */
		ncycles = BEEP_BUFLEN * 256 / period;
		nsamples = (period * ncycles) >> 8;
		f = ncycles * 65536 / nsamples;
		j = 0;
		p = chip->beep_buf;
		for (i = 0; i < nsamples; ++i, p += 2) {
			p[0] = p[1] = beep_wform[j >> 8] * chip->beep_volume;
			j = (j + f) & 0xffff;
		}
		chip->beep_hz = hz;
		chip->beep_volume_play = chip->beep_volume;
		chip->beep_nsamples = nsamples;
	}

	spin_lock_irqsave(&chip->reg_lock, flags);
	if (chip->beep_on) {
		if (ticks <= 0)
			ticks = 1;
		chip->beep_timer.expires = jiffies + ticks;
		add_timer(&chip->beep_timer);
		spin_lock(&rec->lock);
		out_le32(&rec->dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
		spin_unlock(&rec->lock);
		snd_pmac_wait_ack(rec);
		st_le16(&chip->extra_cmd->req_count, nsamples * 4);
		st_le16(&chip->extra_cmd->xfer_status, 0);
		st_le32(&chip->extra_cmd->cmd_dep, virt_to_bus(chip->extra_cmd));
		st_le32(&chip->extra_cmd->phy_addr, virt_to_bus(chip->beep_buf));
		st_le16(&chip->extra_cmd->command, OUTPUT_MORE + BR_ALWAYS);
		out_le32(&chip->awacs->control, chip->control | (beep_speed << 8));
		out_le32(&chip->awacs->byteswap, 0);
		spin_lock(&rec->lock);
		out_le32(&rec->dma->cmdptr, virt_to_bus(chip->extra_cmd));
		out_le32(&rec->dma->control, RUN | (RUN << 16));
		spin_unlock(&rec->lock);
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}

#endif


#ifdef CONFIG_PMAC_PBOOK
static void
snd_pmac_screamer_recalibrate(pmac_t *chip)
{
	/* Sorry for the horrible delays... I hope to get that improved
	 * by making the whole PM process asynchronous in a future version
	 */
	mdelay(750);
	snd_pmac_awacs_write_noreg(chip, 1,
				   chip->awacs_reg[1] | MASK_RECALIBRATE | MASK_CMUTE | MASK_AMUTE);
	mdelay(1000);
	snd_pmac_awacs_write_noreg(chip, 1, chip->awacs_reg[1]);
}
#endif

/*
 * initialize pmac - common part
 */
static int __init snd_pmac_init(pmac_t *chip)
{
	chip->auto_mute = snd_auto_mute;
	if (snd_auto_mute_mask)
		chip->hp_stat_mask = snd_auto_mute_mask;

#ifdef CONFIG_PMAC_PBOOK
#ifdef CONFIG_PPC_HAS_FEATURE_CALLS
	ppc_md.feature_call(PMAC_FTR_SOUND_CHIP_ENABLE, chip->node, 0, 1);
#else
	if (chip->is_pbook_G3) {
		pmu_suspend();
		feature_set(chip->node, FEATURE_Sound_CLK_enable);
		feature_set(chip->node, FEATURE_Sound_power);
		/* Shorter delay will not work */
		mdelay(1000);
		pmu_resume();
	}
#endif /* CONFIG_PPC_HAS_FEATURE_CALLS */
	/* Recalibrate chip */
	if (chip->model == PMAC_SCREAMER)
		snd_pmac_screamer_recalibrate(chip);
#endif /* CONFIG_PMAC_PBOOK */

	out_le32(&chip->awacs->control, 0x11);

	switch (chip->model) {
	case PMAC_AWACS:
	case PMAC_SCREAMER:
		snd_pmac_awacs_init(chip);
		break;

	case PMAC_BURGUNDY:
		snd_pmac_burgundy_init(chip);
		break;

	case PMAC_DACA:
		snd_pmac_i2c_init(chip);
		break;

	case PMAC_TUMBLER:
		snd_pmac_tumbler_init(chip);
		snd_pmac_i2c_init(chip);
		break;
	}

	/* Powerbooks have odd ways of enabling inputs such as
	   an expansion-bay CD or sound from an internal modem
	   or a PC-card modem. */
	if (chip->is_pbook_3400) {
		/* Enable CD and PC-card sound inputs. */
		/* This is done by reading from address
		 * f301a000, + 0x10 to enable the expansion-bay
		 * CD sound input, + 0x80 to enable the PC-card
		 * sound input.  The 0x100 enables the SCSI bus
		 * terminator power.
		 */
		chip->latch_base = (unsigned char *) ioremap (0xf301a000, 0x1000);
		in_8(chip->latch_base + 0x190);
	} else if (chip->is_pbook_G3) {
		struct device_node* mio;
		for (mio = chip->node->parent; mio; mio = mio->parent) {
			if (strcmp(mio->name, "mac-io") == 0
			    && mio->n_addrs > 0) {
				chip->macio_base = (unsigned char *) ioremap
					(mio->addrs[0].address, 0x40);
				break;
			}
		}
		/* Enable CD sound input. */
		/* The relevant bits for writing to this byte are 0x8f.
		 * I haven't found out what the 0x80 bit does.
		 * For the 0xf bits, writing 3 or 7 enables the CD
		 * input, any other value disables it.  Values
		 * 1, 3, 5, 7 enable the microphone.  Values 0, 2,
		 * 4, 6, 8 - f enable the input from the modem.
		 */
		if (chip->macio_base)
			out_8(chip->macio_base + 0x37, 3);
	}

	/* Reset dbdma channels */
	out_le32(&chip->playback.dma->control, (RUN|PAUSE|FLUSH|WAKE|DEAD) << 16);
	snd_pmac_wait_ack(&chip->playback);
	out_le32(&chip->capture.dma->control, (RUN|PAUSE|FLUSH|WAKE|DEAD) << 16);
	snd_pmac_wait_ack(&chip->capture);

#ifdef ENABLE_BEEP
	if (snd_enable_beep) {
		/* Initialize beep stuff */
		chip->beep_buf = (short *) snd_kmalloc(BEEP_BUFLEN * 4, GFP_KERNEL);
		if (! chip->beep_buf) {
			snd_printk("awacs: no memory for beep buffer\n");
		} else {
			chip->beep_timer.function = snd_pmac_beep_stop;
			chip->beep_timer.data = (unsigned long) chip;
			chip->orig_mksound = kd_mksound;
			chip->beep_volume = 15;
			kd_mksound = snd_pmac_mksound;
		}
	}
#endif
	// printk("init: pmac: inpport = 0x%x\n", in_le32(&chip->awacs->codec_stat));
	out_le32(&chip->awacs->control, chip->control);
	if (chip->auto_mute)
		snd_pmac_auto_mute(chip, 0);
	return 0;
}


/*
 * initialize AWACS / Screamer
 */
static int __init snd_pmac_awacs_init(pmac_t *chip)
{
	int vol;

	snd_pmac_awacs_write_reg(chip, 0, MASK_MUX_CD);
	/* FIXME: Only machines with external SRS module need MASK_PAROUT */
	chip->awacs_reg[1] = MASK_LOOPTHRU;
	if (chip->has_iic || chip->device_id == 0x5 ||
	    /*chip->_device_id == 0x8 || */
	    chip->device_id == 0xb)
		chip->awacs_reg[1] |= MASK_PAROUT;
	if (chip->hp_stat_mask == 0) {
		if (chip->model == PMAC_AWACS)
			chip->hp_stat_mask = 0x04;
		else { /* screamer */
			switch (chip->device_id) {
			case 0x08:
				/* 1 = side jack, 2 = front jack */
				chip->hp_stat_mask = 0x03;
				break;
			case 0x00:
			case 0x05:
				chip->hp_stat_mask = 0x04;
				break;
			default:
				chip->hp_stat_mask = 0x08;
				break;
			}
		}
	}
	snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
	/* get default volume from nvram */
	vol = (~nvram_read_byte(0x1308) & 7) << 1;
	snd_pmac_awacs_write_reg(chip, 2, vol + (vol << 6));
	snd_pmac_awacs_write_reg(chip, 4, vol + (vol << 6));
	if (chip->model == PMAC_SCREAMER) {
		snd_pmac_awacs_write_reg(chip, 5, 0);
		snd_pmac_awacs_write_reg(chip, 6, 0);
		snd_pmac_awacs_write_reg(chip, 7, 0);
	}

	if (chip->revision == 0) {
		chip->revision =
			(in_le32(&chip->awacs->codec_stat) >> 12) & 0xf;
		if (chip->revision == 3) {
			mdelay(100);
			snd_pmac_awacs_write(chip, 0x6000);
			mdelay(2);
			snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
#ifdef AMP_AVAIL
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
			if (adb_hardware == ADB_VIACUDA)
				chip->amp_only = 1;
#elif defined(CONFIG_ADB_CUDA)
			if (sys_ctrler == SYS_CTRLER_CUDA)
				chip->amp_only = 1;
#endif
			if (chip->amp_only) {
				/*chip->amp_vol[0] = chip->amp_vol[1] = 31;*/
				snd_pmac_awacs_enable_amp(chip, chip->amp_vol);
			}
#endif /* AMP_AVAIL */
		}
	}
	return 0;
}


/*
 * pcm stuff
 */

/*
 * look up frequency table
 */

static unsigned int snd_pmac_fixed_rate(pmac_t *chip, unsigned int rate)
{
	int i, ok, found;

	ok = chip->freqs_ok;
	if (rate > chip->freq_table[0])
		return 0;
	found = 0;
	for (i = 0; i < chip->num_freqs; i++, ok >>= 1) {
		if (! (ok & 1)) continue;
		found = i;
		if (rate >= chip->freq_table[i])
			break;
	}
	return found;
}

/*
 * check whether another stream is active
 */
static inline int snd_pmac_stream_busy(pmac_t *chip, pmac_stream_t *rec)
{
	int check_flag = (rec->stream == SND_PCM_CHANNEL_PLAYBACK) ?
		SND_PCM_CHANNEL_CAPTURE : SND_PCM_CHANNEL_PLAYBACK;
	return test_bit(check_flag, &chip->active);
}

/*
 * prepare playback/capture stream
 */
static int snd_pmac_pcm_prepare(pmac_t *chip, pmac_stream_t *rec, snd_pcm_subchn_t *subchn)
{
	int i;
	int rate_index;
	volatile struct dbdma_cmd *cp;
	unsigned long flags;
	snd_pcm_runtime_t *runtime = subchn->runtime;
	long offset;

	snd_debug_check(runtime->format.format != SND_PCM_SFMT_S16_BE &&
			runtime->format.format != SND_PCM_SFMT_S16_LE, -EINVAL);
	snd_debug_check(runtime->format.voices != 2, -EINVAL);

	/*printk("awacs: prepare: format = %d, rate = %d, voices = %d\n", runtime->format.format, runtime->format.rate, runtime->format.voices);*/

	rec->dma_size = snd_pcm_lib_transfer_size(subchn);
	rec->frag_size = snd_pcm_lib_transfer_fragment(subchn);
	rec->nfrags = rec->dma_size / rec->frag_size;
	if (rec->nfrags > PMAC_MAX_FRAGS) {
		rec->nfrags = PMAC_MAX_FRAGS;
		rec->dma_size = rec->frag_size * rec->nfrags;
		snd_pcm_lib_set_buffer_size(subchn, rec->dma_size);
	}
	rec->cur_frag = 0;

	/*printk("  dma_size = %d, frag_size = %d, nfrags = %d\n", rec->dma_size, rec->frag_size, rec->nfrags);*/
	rate_index = snd_pmac_fixed_rate(chip, runtime->format.rate);
	runtime->format.rate = chip->freq_table[rate_index];

	/*printk("  rate_index = %d, rate = %d, format = %d\n", rate_index, runtime->format.rate, runtime->format.format);*/
	spin_lock_irqsave(&chip->reg_lock, flags);
	if (snd_pmac_stream_busy(chip, rec)) {
		if (chip->rate_index != rate_index ||
		    chip->format != runtime->format.format) {
			spin_unlock_irqrestore(&chip->reg_lock, flags);
			snd_printd("  busy! rate = %d/%d, format = %d/%d\n",
				   chip->rate_index, rate_index, chip->format, runtime->format.format);
			return -EBUSY;
		}
	} else {
		set_bit(rec->stream, &chip->active);
		chip->rate_index = rate_index;
		chip->format = runtime->format.format;
		/* set up frequency and format */
		/* disable error interrupt on burgundy for now */
		out_le32(&chip->awacs->control, chip->control | (chip->rate_index << 8));
		chip->awacs_reg[1] &= ~MASK_SAMPLERATE;
		chip->awacs_reg[1] |= chip->rate_index << 3;
		snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
		out_le32(&chip->awacs->byteswap,
			 chip->format == SND_PCM_SFMT_S16_LE);
		/*printk("  set rate = %d, format = %d, swap = %d\n",
		       chip->rate_index, chip->format, chip->format == SND_PCM_SFMT_S16_LE);*/
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);

	/* We really want to execute a DMA stop command, after the AWACS
	 * is initialized.
	 * For reasons I don't understand, it stops the hissing noise
	 * common to many PowerBook G3 systems (like mine :-).
	 */
	spin_lock_irqsave(&rec->lock, flags);
	out_le32(&rec->dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
	snd_pmac_wait_ack(rec);
	if (rec->stream == SND_PCM_CHANNEL_PLAYBACK) {
		st_le16(&chip->extra_cmd->command, DBDMA_STOP);
		out_le32(&rec->dma->cmdptr, virt_to_bus(chip->extra_cmd));
		out_le32(&rec->dma->control, RUN | (RUN << 16));
	}
	spin_unlock_irqrestore(&rec->lock, flags);

	spin_lock_irqsave(&rec->lock, flags);
	offset = virt_to_bus(runtime->dma_area->buf);
	for (i = 0, cp = rec->cmds; i < rec->nfrags; i++, cp++) {
		st_le32(&cp->phy_addr, offset);
		st_le16(&cp->req_count, rec->frag_size);
		/*st_le16(&cp->res_count, 0);*/
		st_le16(&cp->xfer_status, 0);
		offset += rec->frag_size;
	}
	/* make loop */
	st_le16(&cp->command, DBDMA_NOP + BR_ALWAYS);
	st_le32(&cp->cmd_dep, virt_to_bus(rec->cmds));

	out_le32(&rec->dma->control, (RUN|PAUSE|FLUSH|WAKE) << 16);
	snd_pmac_wait_ack(rec);
	out_le32(&rec->dma->cmdptr, virt_to_bus(rec->cmds));
	spin_unlock_irqrestore(&rec->lock, flags);

	/*printk("  ok!!\n");*/
	return 0;
}

/*
 * ioctl - change rate
 */
static int snd_pmac_pcm_ioctl(pmac_t *chip, pmac_stream_t *rec,
			      snd_pcm_subchn_t *subchn,
			      unsigned int cmd, unsigned long *arg)
{
	unsigned long flags;
	snd_pcm_runtime_t *runtime = subchn->runtime;

	if (cmd == SND_PCM_IOCTL1_PARAMS) {
		spin_lock_irqsave(&chip->reg_lock, flags);
		if (snd_pmac_stream_busy(chip, rec)) {
			runtime->format.rate = chip->freq_table[chip->rate_index];
			runtime->format.format = chip->format;
		} else {
			int rate_index = snd_pmac_fixed_rate(chip, runtime->format.rate);
			runtime->format.rate = chip->freq_table[rate_index];
		}
		spin_unlock_irqrestore(&chip->reg_lock, flags);
	}
	return 0;
}

/*
 * PCM trigger/stop
 */
static int snd_pmac_pcm_trigger(pmac_t *chip, pmac_stream_t *rec,
				snd_pcm_subchn_t *subchn, int cmd)
{
	unsigned long flags;
	volatile struct dbdma_cmd *cp;
	int i, command;

	switch (cmd) {
	case SND_PCM_TRIGGER_GO:
		/*printk("triggered!!\n");*/
		spin_lock_irqsave(&rec->lock, flags);
		if (rec->running) {
			spin_unlock_irqrestore(&rec->lock, flags);
			return -EBUSY;
		}
#ifdef ENABLE_BEEP
		if (chip->beep_buf) {
			spin_lock(&chip->reg_lock);
			if (chip->beep_on) {
				chip->beep_on = 0;
				del_timer(&chip->beep_timer);
				st_le16(&chip->extra_cmd->command, DBDMA_STOP);
				out_le32(&rec->dma->control, (RUN|PAUSE|FLUSH|WAKE) << 16);
				snd_pmac_wait_ack(rec);
			}
			spin_unlock(&chip->reg_lock);
		}
#endif
		command = (subchn->channel == SND_PCM_CHANNEL_PLAYBACK ?
			   OUTPUT_MORE : INPUT_MORE) + INTR_ALWAYS;
		for (i = 0, cp = rec->cmds; i < rec->nfrags; i++, cp++)
			out_le16(&cp->command, command);
		out_le32(&rec->dma->cmdptr, virt_to_bus(rec->cmds));
		(void)in_le32(&rec->dma->status);
		out_le32(&rec->dma->control, ((RUN|WAKE) << 16) + (RUN|WAKE));
		rec->running = 1;
		spin_unlock_irqrestore(&rec->lock, flags);
		break;

	case SND_PCM_TRIGGER_STOP:
		spin_lock_irqsave(&rec->lock, flags);
		rec->running = 0;
		/*printk("stopped!!\n");*/
		out_le32(&rec->dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
		snd_pmac_wait_ack(rec);
		for (i = 0, cp = rec->cmds; i < rec->nfrags; i++, cp++)
			out_le16(&cp->command, DBDMA_STOP);
		spin_unlock_irqrestore(&rec->lock, flags);
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/*
 * return the current pointer
 */
inline
static unsigned int snd_pmac_pcm_pointer(pmac_t *chip, pmac_stream_t *rec)
{
	int count;

#if 0
	if (! rec->cmds[rec->cur_frag].xfer_status) {
		count = in_le16(&rec->cmds[rec->cur_frag].res_count);
		count = rec->frag_size - count;
	} else
#endif
		count = 0;
	count += rec->cur_frag * rec->frag_size;
	/*printk("pointer=%d\n", count);*/
	return count;
}

/*
 * playback
 */

static int snd_pmac_playback_prepare(void *private_data,
				     snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_prepare(chip, &chip->playback, subchn);
}

static int snd_pmac_playback_ioctl(void *private_data,
				   snd_pcm_subchn_t *subchn,
				   unsigned int cmd,
				   unsigned long *arg)
{
	int result;
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	return snd_pmac_pcm_ioctl(chip, &chip->playback, subchn, cmd, arg);
}

static int snd_pmac_playback_trigger(void *private_data,
				     snd_pcm_subchn_t *subchn,
				     int cmd)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_trigger(chip, &chip->playback, subchn, cmd);
}

static unsigned int snd_pmac_playback_pointer(void *private_data,
					      snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_pointer(chip, &chip->playback);
}


/*
 * capture
 */

static int snd_pmac_capture_prepare(void *private_data,
				    snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_prepare(chip, &chip->capture, subchn);
}

static int snd_pmac_capture_ioctl(void *private_data,
				  snd_pcm_subchn_t *subchn,
				  unsigned int cmd,
				  unsigned long *arg)
{
	int result;
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);

	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	return snd_pmac_pcm_ioctl(chip, &chip->capture, subchn, cmd, arg);
}

static int snd_pmac_capture_trigger(void *private_data,
				    snd_pcm_subchn_t *subchn,
				    int cmd)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_trigger(chip, &chip->capture, subchn, cmd);
}

static unsigned int snd_pmac_capture_pointer(void *private_data,
					     snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	return snd_pmac_pcm_pointer(chip, &chip->capture);
}


/*
 * update playback/capture pointer
 */
static void snd_pmac_pcm_update(pmac_t *pmac, pmac_stream_t *rec)
{
	volatile struct dbdma_cmd *cp;
	int c;
	int stat;

	spin_lock(&rec->lock);
	if (rec->running) {
		cp = &rec->cmds[rec->cur_frag];
		for (c = 0; c < rec->nfrags; c++) {
			stat = ld_le16(&cp->xfer_status);
			if (! (stat & ACTIVE))
				break;
			/*printk("update frag %d\n", rec->cur_frag);*/
			st_le16(&cp->xfer_status, 0);
			st_le16(&cp->req_count, rec->frag_size);
			/*st_le16(&cp->res_count, 0);*/
			rec->cur_frag++;
			if (rec->cur_frag >= rec->nfrags) {
				rec->cur_frag = 0;
				cp = rec->cmds;
			} else
				cp++;
			spin_unlock(&rec->lock);
			snd_pcm_transfer_done(rec->substream);
			spin_lock(&rec->lock);
		}
	}
	spin_unlock(&rec->lock);
}


/*
 * interrupt handlers
 */
static void
snd_pmac_awacs_tx_intr(int irq, void *devid, struct pt_regs *regs)
{
	pmac_t *chip = snd_magic_cast(pmac_t, devid, );
	snd_pmac_pcm_update(chip, &chip->playback);
}


static void
snd_pmac_awacs_rx_intr(int irq, void *devid, struct pt_regs *regs)
{
	pmac_t *chip = snd_magic_cast(pmac_t, devid, );
	snd_pmac_pcm_update(chip, &chip->capture);
}

static void
snd_pmac_awacs_intr(int irq, void *devid, struct pt_regs *regs)
{
	pmac_t *chip = snd_magic_cast(pmac_t, devid, );
	int ctrl = in_le32(&chip->awacs->control);

	if (ctrl & MASK_PORTCHG) {
		if (chip->auto_mute)
			snd_pmac_auto_mute(chip, 1);
	}
	if (ctrl & MASK_CNTLERR) {
		int err = (in_le32(&chip->awacs->codec_stat) & MASK_ERRCODE) >> 16;
		if (err && chip->model <= PMAC_SCREAMER)
			snd_printk("AWACS: error %x\n", err);
	}
	/* Writing 1s to the CNTLERR and PORTCHG bits clears them... */
	out_le32(&chip->awacs->control, ctrl);
}

/*
 * hw info: data will be copied to instances, so safe to set __initdata here
 */

static snd_pcm_hardware_t snd_pmac_playback __initdata =
{
	chninfo:	(SND_PCM_CHNINFO_BLOCK
			 /*! SND_PCM_CHNINFO_STREAM*/ /* XXX doesn't work */
 			 | SND_PCM_CHNINFO_INTERLEAVE
			 | SND_PCM_CHNINFO_BLOCK_TRANSFER
			 /*| SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_MMAP_VALID*/ /* XXX no mmap! */
			 ),
	formats:	SND_PCM_FMT_S16_BE | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_CONTINUOUS | SND_PCM_RATE_8000_44100,
	min_rate:	7350,
	max_rate:	44100,
	min_voices:	2,
	max_voices:	2,
	min_fragment_size:	256,
	max_fragment_size:	16384,
	fragment_align:	31,
	ioctl:		snd_pmac_playback_ioctl,
	prepare:	snd_pmac_playback_prepare,
	trigger:	snd_pmac_playback_trigger,
	pointer:	snd_pmac_playback_pointer,
};

static snd_pcm_hardware_t snd_pmac_capture __initdata =
{
	chninfo:	(SND_PCM_CHNINFO_BLOCK
			 /*| SND_PCM_CHNINFO_STREAM*/
			 | SND_PCM_CHNINFO_INTERLEAVE
			 | SND_PCM_CHNINFO_BLOCK_TRANSFER
			 /*| SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_MMAP_VALID*/
			 ),
	formats:	SND_PCM_FMT_S16_BE | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_CONTINUOUS | SND_PCM_RATE_8000_44100,
	min_rate:	7350,
	max_rate:	44100,
	min_voices:	2,
	max_voices:	2,
	min_fragment_size:	256,
	max_fragment_size:	16384,
	fragment_align:	31,
	ioctl:		snd_pmac_capture_ioctl,
	prepare:	snd_pmac_capture_prepare,
	trigger:	snd_pmac_capture_trigger,
	pointer:	snd_pmac_capture_pointer,
};

static int snd_pmac_playback_open(void *private_data,
				  snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, chip->playback.dmaptr,
				     "PMac AWACS - DAC")) < 0)
		return err;
	subchn->runtime->hw = chip->playback.hwinfo;
	chip->playback.substream = subchn;

	if (chip->can_duplex)
		snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_pmac_capture_open(void *private_data,
				 snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, chip->capture.dmaptr,
				     "PMac AWACS - ADC")) < 0)
		return err;
	subchn->runtime->hw = chip->capture.hwinfo;
	chip->capture.substream = subchn;

	if (chip->can_duplex)
		snd_pcm_set_sync(subchn);

	return 0;
}

static int snd_pmac_playback_close(void *private_data,
				   snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);

	out_le32(&chip->playback.dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
	snd_pmac_wait_ack(&chip->playback);
	snd_pcm_dma_free(subchn);
	clear_bit(chip->playback.stream, &chip->active);
	return 0;
}

static int snd_pmac_capture_close(void *private_data,
				  snd_pcm_subchn_t *subchn)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, -ENXIO);

	out_le32(&chip->capture.dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
	snd_pmac_wait_ack(&chip->capture);
	snd_pcm_dma_free(subchn);
	clear_bit(chip->capture.stream, &chip->active);
	return 0;
}

struct pmac_freq_check {
	int rate;
	unsigned int flag;
};

static int __init snd_pmac_pcm_new(pmac_t *chip)
{
	snd_pcm_t *pcm;
	int err, i, rates, ok_rest;
	int num_captures;
	static struct pmac_freq_check fixed_freqs[] = {
		{ 11025, SND_PCM_RATE_11025 },
		{ 22050, SND_PCM_RATE_22050 },
		{ 44100, SND_PCM_RATE_44100 },
		{ 48000, SND_PCM_RATE_48000 },
	};

	num_captures = chip->can_capture ? 1 : 0;
	err = snd_pcm_new(chip->card, "PMac AWACS", 0, 1, num_captures, &pcm);
	if (err < 0)
		return err;

	chip->playback.hwinfo = snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (chip->playback.hwinfo == NULL)
		return -ENOMEM;
	chip->capture.hwinfo = snd_kmalloc(sizeof(snd_pcm_hardware_t), GFP_KERNEL);
	if (chip->capture.hwinfo == NULL)
		return -ENOMEM;
	*chip->playback.hwinfo = snd_pmac_playback;
	*chip->capture.hwinfo = snd_pmac_capture;
	rates = 0;
	ok_rest = chip->freqs_ok;
	for (i = 0; i < sizeof(fixed_freqs)/sizeof(fixed_freqs[0]); i++) {
		int j;
		for (j = 0; j < chip->num_freqs; j++) {
			if ((chip->freqs_ok & (1 << j)) &&
			    chip->freq_table[j] == fixed_freqs[i].rate) {
				rates |= fixed_freqs[i].flag;
				ok_rest &= ~(1 << j);
			}
		}
	}
	if (ok_rest)
		rates |= SND_PCM_RATE_KNOT;
	chip->playback.hwinfo->rates = rates;
	chip->capture.hwinfo->rates = rates;
	for (i = 0; i < chip->num_freqs; i++) {
		if (chip->freqs_ok & (1 << i)) {
			chip->playback.hwinfo->max_rate = 
				chip->capture.hwinfo->max_rate = chip->freq_table[i];
			break;
		}
	}
	for (i = chip->num_freqs - 1; i >= 0; i--) {
		if (chip->freqs_ok & (1 << i)) {
			chip->playback.hwinfo->min_rate =
				chip->capture.hwinfo->min_rate = chip->freq_table[i];
			break;
		}
	}
	if (! chip->can_byte_swap) /* no LE support */
		chip->playback.hwinfo->formats =
			chip->capture.hwinfo->formats = SND_PCM_FMT_S16_BE;

	/*printk("pmac: rates = 0x%x, min = %d, max = %d\n", rates, chip->playback.hwinfo->min_rate, chip->playback.hwinfo->max_rate);*/
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = chip;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_pmac_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_pmac_playback_close;

	if (chip->can_capture) {
		pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = chip;
		pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_pmac_capture_open;
		pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_pmac_capture_close;
	}

	pcm->private_data = chip;
	if (! chip->can_capture)
		pcm->info_flags = SND_PCM_INFO_PLAYBACK;
	else {
		pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE;
		if (chip->can_duplex)
			pcm->info_flags |= SND_PCM_INFO_DUPLEX | SND_PCM_INFO_DUPLEX_RATE;
	}
	strcpy(pcm->name, "PMac AWACS DAC/ADC");
	chip->pcm = pcm;

	return 0;
}


/*
 * Burgundy lowlevel functions
 */

/* Waits for busy flag to clear */
inline static void
snd_pmac_burgundy_busy_wait(pmac_t *chip)
{
	while (in_le32(&chip->awacs->codec_ctrl) & MASK_NEWECMD)
		;
}

inline static void
snd_pmac_burgundy_extend_wait(pmac_t *chip)
{
	while (!(in_le32(&chip->awacs->codec_stat) & MASK_EXTEND))
		;
	while (in_le32(&chip->awacs->codec_stat) & MASK_EXTEND)
		;
}

static void
snd_pmac_burgundy_wcw(pmac_t *chip, unsigned addr, unsigned val)
{
	out_le32(&chip->awacs->codec_ctrl, addr + 0x200c00 + (val & 0xff));
	snd_pmac_burgundy_busy_wait(chip);
	out_le32(&chip->awacs->codec_ctrl, addr + 0x200d00 +((val>>8) & 0xff));
	snd_pmac_burgundy_busy_wait(chip);
	out_le32(&chip->awacs->codec_ctrl, addr + 0x200e00 +((val>>16) & 0xff));
	snd_pmac_burgundy_busy_wait(chip);
	out_le32(&chip->awacs->codec_ctrl, addr + 0x200f00 +((val>>24) & 0xff));
	snd_pmac_burgundy_busy_wait(chip);
}

static unsigned
snd_pmac_burgundy_rcw(pmac_t *chip, unsigned addr)
{
	unsigned val = 0;
	unsigned long flags;

	/* should have timeouts here */
	spin_lock_irqsave(&chip->reg_lock, flags);

	out_le32(&chip->awacs->codec_ctrl, addr + 0x100000);
	snd_pmac_burgundy_busy_wait(chip);
	snd_pmac_burgundy_extend_wait(chip);
	val += (in_le32(&chip->awacs->codec_stat) >> 4) & 0xff;

	out_le32(&chip->awacs->codec_ctrl, addr + 0x100100);
	snd_pmac_burgundy_busy_wait(chip);
	snd_pmac_burgundy_extend_wait(chip);
	val += ((in_le32(&chip->awacs->codec_stat)>>4) & 0xff) <<8;

	out_le32(&chip->awacs->codec_ctrl, addr + 0x100200);
	snd_pmac_burgundy_busy_wait(chip);
	snd_pmac_burgundy_extend_wait(chip);
	val += ((in_le32(&chip->awacs->codec_stat)>>4) & 0xff) <<16;

	out_le32(&chip->awacs->codec_ctrl, addr + 0x100300);
	snd_pmac_burgundy_busy_wait(chip);
	snd_pmac_burgundy_extend_wait(chip);
	val += ((in_le32(&chip->awacs->codec_stat)>>4) & 0xff) <<24;

	spin_unlock_irqrestore(&chip->reg_lock, flags);

	return val;
}

static void
snd_pmac_burgundy_wcb(pmac_t *chip, unsigned addr, unsigned val)
{
	out_le32(&chip->awacs->codec_ctrl, addr + 0x300000 + (val & 0xff));
	snd_pmac_burgundy_busy_wait(chip);
}

static unsigned
snd_pmac_burgundy_rcb(pmac_t *chip, unsigned addr)
{
	unsigned val = 0;
	unsigned long flags;

	/* should have timeouts here */
	spin_lock_irqsave(&chip->reg_lock, flags);

	out_le32(&chip->awacs->codec_ctrl, addr + 0x100000);
	snd_pmac_burgundy_busy_wait(chip);
	snd_pmac_burgundy_extend_wait(chip);
	val += (in_le32(&chip->awacs->codec_stat) >> 4) & 0xff;

	spin_unlock_irqrestore(&chip->reg_lock, flags);

	return val;
}

static int __init
snd_pmac_burgundy_check(pmac_t *chip)
{
	/* Checks to see the chip is alive and kicking */
	int error = in_le32(&chip->awacs->codec_ctrl) & MASK_ERRCODE;

	return error == 0xf0000;
}

/*
 * initialize burgundy
 */
static int __init
snd_pmac_burgundy_init(pmac_t *chip)
{
	if (snd_pmac_burgundy_check(chip)) {
		printk(KERN_WARNING "AWACS: disabled by MacOS :-(\n");
		return 1;
	}

	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_OUTPUTENABLES,
			   DEF_BURGUNDY_OUTPUTENABLES);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES,
			   DEF_BURGUNDY_MORE_OUTPUTENABLES);
	snd_pmac_burgundy_wcw(chip, MASK_ADDR_BURGUNDY_OUTPUTSELECTS,
			   DEF_BURGUNDY_OUTPUTSELECTS);

	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_INPSEL21,
			   DEF_BURGUNDY_INPSEL21);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_INPSEL3,
			   DEF_BURGUNDY_INPSEL3);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_GAINCD,
			   DEF_BURGUNDY_GAINCD);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_GAINLINE,
			   DEF_BURGUNDY_GAINLINE);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_GAINMIC,
			   DEF_BURGUNDY_GAINMIC);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_GAINMODEM,
			   DEF_BURGUNDY_GAINMODEM);

	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_ATTENSPEAKER,
			   DEF_BURGUNDY_ATTENSPEAKER);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_ATTENLINEOUT,
			   DEF_BURGUNDY_ATTENLINEOUT);
	snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_ATTENHP,
			   DEF_BURGUNDY_ATTENHP);

	snd_pmac_burgundy_wcw(chip, MASK_ADDR_BURGUNDY_MASTER_VOLUME,
			   DEF_BURGUNDY_MASTER_VOLUME);
	snd_pmac_burgundy_wcw(chip, MASK_ADDR_BURGUNDY_VOLCD,
			   DEF_BURGUNDY_VOLCD);
	snd_pmac_burgundy_wcw(chip, MASK_ADDR_BURGUNDY_VOLLINE,
			   DEF_BURGUNDY_VOLLINE);
	snd_pmac_burgundy_wcw(chip, MASK_ADDR_BURGUNDY_VOLMIC,
			   DEF_BURGUNDY_VOLMIC);
	if (chip->hp_stat_mask == 0)
		chip->hp_stat_mask = 0x04;
	return 0;
}

/*
 * Burgundy volume: 0 - 100, stereo
 */
static void
snd_pmac_burgundy_write_volume(pmac_t *chip, unsigned address, int *volume, int shift)
{
	int hardvolume, lvolume, rvolume;

	lvolume = volume[0] ? volume[0] + 155 : 0;
	rvolume = volume[1] ? volume[1] + 155 : 0;

	hardvolume = lvolume + (rvolume << shift);
	if (shift == 8)
		hardvolume |= hardvolume << 16;

	snd_pmac_burgundy_wcw(chip, address, hardvolume);
}

static void
snd_pmac_burgundy_read_volume(pmac_t *chip, unsigned address, int *volume, int shift)
{
	int wvolume;

	wvolume = snd_pmac_burgundy_rcw(chip, address);

	volume[0] = wvolume & 0xff;
	if (volume[0] >= 155)
		volume[0] -= 155;
	else
		volume[0] = 0;
	volume[1] = (wvolume >> shift) & 0xff;
	if (volume[1] >= 155)
		volume[1] -= 155;
	else
		volume[1] = 0;
}


/*
 * Tumbler lowlevel routines
 */

static void
tumbler_headphone_intr(int irq, void *devid, struct pt_regs *regs)
{
	pmac_t *chip = snd_magic_cast(pmac_t, devid, );
	if (! chip->headphone_status ||
	    ! chip->amp_mute ||
	    ! chip->headphone_mute)
		return;
	if (readb(chip->headphone_status) & 2) {
		writeb(4+1, chip->amp_mute);
		writeb(4+0, chip->headphone_mute);
	} else {
		writeb(4+0, chip->amp_mute);
		writeb(4+1, chip->headphone_mute);
	}
}


struct device_node *find_audio_device(char *name)
{
	struct device_node *np;
 
	np = find_devices("gpio");
	if (np == NULL) {
		snd_printk("pmac: no gpio!\n");
		return NULL;
	}
	for (np = np->child; np; np = np->sibling) {
		char *prop = get_property(np, "audio-gpio", NULL);
		if (prop && strcmp(prop, name) == 0)
			return np;
	}  
	return NULL;
}


static void *snd_pmac_get_property(char *dev, char *prop)
{
	struct device_node *np;
	void *base;

	np = find_audio_device(dev);
	if (np == NULL) {
		snd_printk("pmac: cannot find %s\n", dev);
		return NULL;
	}
	base = (void*)get_property(np, prop, NULL);
	if (base == NULL) {
		snd_printk("pmac: cannot find %s\n", prop);
		return NULL;
	}
	return *(void**)base;
}

/* reset TAS */
static void tumbler_reset(pmac_t *chip)
{
	void *base;

	if ((base = snd_pmac_get_property("audio-hw-reset", "AAPL,address")) != NULL) {
		// int status;
		base = (void*)ioremap((unsigned long)base, 1);

		// status = readb(base) & 1;
		writeb(5, base);
		mdelay(100);
		writeb(4, base);
		mdelay(1);
		writeb(5, base);
		mdelay(1);
  
		iounmap(base);
	}
}

static int __init
snd_pmac_tumbler_init(pmac_t *chip)
{
	struct device_node *device;
	void *base;

	tumbler_reset(chip);
	if ((base = snd_pmac_get_property("amp-mute", "AAPL,address")) != NULL) {
		chip->amp_mute = (void*)ioremap((unsigned long)base, 1);
	}
	if ((base = snd_pmac_get_property("headphone-mute", "AAPL,address")) != NULL) {
		chip->headphone_mute = (void*)ioremap((unsigned long)base, 1);
	}
	device = find_audio_device("headphone-detect");
	if (device && device->n_intrs > 0) {
		snd_register_interrupt(chip->card, "PMac Headphone",
				       device->intrs[0].line,
				       SND_IRQ_TYPE_PCI,
				       tumbler_headphone_intr,
				       chip, NULL, &chip->headphone_irqptr);
	}
	if (chip->headphone_irqptr &&
	    (base = snd_pmac_get_property("headphone-detect", "AAPL,address")) != NULL) {
		chip->headphone_status = (void*)ioremap((unsigned long)base, 1);
		/* Activate headphone status interrupts */
		writeb(readb(chip->headphone_status) | (1<<7), chip->headphone_status);
		tumbler_headphone_intr(0, chip, 0); // unmute
	}

	return 0;
}


/*
 *
 */
#ifdef AMP_AVAIL
/* Turn on sound output, needed on G3 desktop powermacs */
/* vol = 0 - 31, stereo */
static void
snd_pmac_awacs_enable_amp(pmac_t *chip, int *spkr_vol)
{
	struct adb_request req;

	if (! chip->amp_only)
		return;

	/* turn on headphones */
	cuda_request(&req, NULL, 5, CUDA_PACKET, CUDA_GET_SET_IIC,
		     0x8a, 4, 0);
	while (!req.complete) cuda_poll();
	cuda_request(&req, NULL, 5, CUDA_PACKET, CUDA_GET_SET_IIC,
		     0x8a, 6, 0);
	while (!req.complete) cuda_poll();

	/* turn on speaker */
	cuda_request(&req, NULL, 5, CUDA_PACKET, CUDA_GET_SET_IIC,
		     0x8a, 3, spkr_vol[0] & 0xff);
	while (!req.complete) cuda_poll();
	cuda_request(&req, NULL, 5, CUDA_PACKET, CUDA_GET_SET_IIC,
		     0x8a, 5, spkr_vol[1] & 0xff);
	while (!req.complete) cuda_poll();

	cuda_request(&req, NULL, 5, CUDA_PACKET,
		     CUDA_GET_SET_IIC, 0x8a, 1, 0x29);
	while (!req.complete) cuda_poll();
}
#endif /* AMP_AVAIL */

/*
 * read AWACS volume
 * volume range = 0 - 15, mono
 */
static void snd_pmac_awacs_read_volume(pmac_t *chip, int *volume, int n, int lshift)
{
	unsigned long flags;
	spin_lock_irqsave(&chip->reg_lock, flags);
#if 0
	volume[0] = 0x0f - ((chip->awacs_reg[n] >> lshift) & 0xf);
	volume[1] = 0x0f - (chip->awacs_reg[n] & 0xf);
#else
	volume[0] = chip->awacs_volume_cache[n][0];
	volume[1] = chip->awacs_volume_cache[n][1];
#endif
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}

/*
 * write AWACS volume
 * volume range = 0 - 15, mono
 */
static void snd_pmac_awacs_write_volume(pmac_t *chip, int *volume, int n, int lshift)
{
	int rn;
	unsigned long flags;
	spin_lock_irqsave(&chip->reg_lock, flags);
	rn = chip->awacs_reg[n] & ~(0xf | (0xf << lshift));
#if 0
	rn |= (0x0f - (volume[0] & 0xf)) << lshift;
	rn |= 0x0f - (volume[1] & 0xf);
#else
	chip->awacs_volume_cache[n][0] = volume[0];
	chip->awacs_volume_cache[n][1] = volume[1];
	rn |= (0x0f - (((volume[0] * chip->awacs_volume_cache[3][0])/100) & 0x0f)) << lshift;
	rn |= 0x0f - (((volume[1] * chip->awacs_volume_cache[3][1])/100) & 0x0f);
#endif
	snd_pmac_awacs_write_reg(chip, n, rn);
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}


/*
 * mixer stuff - oh my... so complicated..
 */

/*
 * AWACS volume callback
 */
static int snd_pmac_awacs_volume_callback(pmac_t *chip, int w_flag, int *voices,
					  int n, int lshift)
{
	if (w_flag) {
		int nvoices[2];
		snd_pmac_awacs_write_volume(chip, voices, n, lshift);
		snd_pmac_awacs_read_volume(chip, nvoices, n, lshift);
		return voices[0] != nvoices[0] || voices[1] != nvoices[1];
	} else {
		snd_pmac_awacs_read_volume(chip, voices, n, lshift);
		return 0;
	}
}


#define MAKE_AWACS_VOLUME(name,reg,shift) \
static int snd_pmac_awacs_volume_##name\
(snd_kmixer_element_t *element, int w_flag, int *voices)\
{\
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);\
	return snd_pmac_awacs_volume_callback(chip, w_flag, voices, reg, shift);\
}

MAKE_AWACS_VOLUME(lineout, 2, 6);
MAKE_AWACS_VOLUME(speaker, 4, 6);
MAKE_AWACS_VOLUME(capture, 0, 4);

/*
 * pseudo master volume for AWACS
 */
static int snd_pmac_awacs_volume_master(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	if (w_flag) {
		int change;
		change = chip->awacs_volume_cache[3][0] != voices[0] ||
			chip->awacs_volume_cache[3][1] != voices[1];
		chip->awacs_volume_cache[3][0] = voices[0];
		chip->awacs_volume_cache[3][1] = voices[1];
		/* update both lineout & speaker */
		if (change) {
			snd_pmac_awacs_write_volume(chip, chip->awacs_volume_cache[2], 2, 6);
			snd_pmac_awacs_write_volume(chip, chip->awacs_volume_cache[4], 4, 6);
		}
		return change;
	} else {
		voices[0] = chip->awacs_volume_cache[3][0];
		voices[1] = chip->awacs_volume_cache[3][1];
		return 0;
	}
}


/*
 * mute master/ogain for AWACS
 */
static int snd_pmac_awacs_mute_callback(pmac_t *chip, int w_flag, int *value, int mask)
{
	int oval = 0;
	int change = 0;
	unsigned long flags;
	
	spin_lock_irqsave(&chip->reg_lock, flags);
	oval = (chip->awacs_reg[1] & mask) ? 0 : 1;
	if (w_flag) {
		if (*value & 1)
			chip->awacs_reg[1] &= ~mask;
		else
			chip->awacs_reg[1] |= mask;
		snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
		change = *value != oval;
	} else {
		*value = oval;
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return change;
}

static int snd_pmac_awacs_mute_lineout(snd_kmixer_element_t *element, int w_flag, unsigned int *value)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	return snd_pmac_awacs_mute_callback(chip, w_flag, value, MASK_AMUTE);
}

static int snd_pmac_awacs_mute_speaker(snd_kmixer_element_t *element, int w_flag, unsigned int *value)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	return snd_pmac_awacs_mute_callback(chip, w_flag, value, MASK_CMUTE);
}

/* mute or auto-mute */
static void snd_pmac_auto_mute(pmac_t *chip, int intr)
{
	int reg, oreg;

	if (intr && ! chip->initialized)
		return;

	switch (chip->model) {
	case PMAC_AWACS:
	case PMAC_SCREAMER:
		reg = chip->awacs_reg[1] | (MASK_AMUTE|MASK_CMUTE);
		if (chip->auto_mute_state) {
			if (in_le32(&chip->awacs->codec_stat) & chip->hp_stat_mask)
				reg &= ~MASK_AMUTE;
			else
				reg &= ~MASK_CMUTE;
		}
		if (reg != chip->awacs_reg[1])
			snd_pmac_awacs_write_reg(chip, 1, reg);
		break;
	case PMAC_BURGUNDY:
		reg = oreg = snd_pmac_burgundy_rcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES);
		reg &= ~(BURGUNDY_OUTPUT_LEFT | BURGUNDY_OUTPUT_RIGHT | BURGUNDY_OUTPUT_INTERN);
		if (chip->auto_mute_state) {
			if (in_le32(&chip->awacs->codec_stat) & chip->hp_stat_mask)
				reg |= BURGUNDY_OUTPUT_LEFT | BURGUNDY_OUTPUT_RIGHT;
			else
				reg |= BURGUNDY_OUTPUT_INTERN;
		}
		if (reg != oreg)
			snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES, reg);
		break;
	}
}

static int snd_pmac_mixer_mute_all(snd_kmixer_element_t *element, int w_flag, unsigned int *value)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	if (w_flag) {
		int state = (*value & 1);
		if (state != chip->auto_mute_state) {
			chip->auto_mute_state = state;
			snd_pmac_auto_mute(chip, 0);
			return 1;
		}
	} else {
		*value = chip->auto_mute_state;
	}
	return 0;
}

#ifdef AMP_AVAIL
/*
 * Master volume for awacs revision 3
 */
static int snd_pmac_awacs_volume_amp(snd_kmixer_element_t *element, int w_flag, int *value)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int change = 0;

	if (w_flag) {
		snd_pmac_awacs_enable_amp(chip, value);
		change = value[0] != chip->amp_vol[0] ||
			value[1] != chip->amp_vol[1];
		chip->amp_vol[0] = value[0];
		chip->amp_vol[1] = value[1];
	} else {
		value[0] = chip->amp_vol[0];
		value[1] = chip->amp_vol[1];
	}
	return change;
}
#endif

/*
 * Burgundy master: volume only, 0-100
 */
static int snd_pmac_burgundy_volume_master(snd_kmixer_element_t *element,
					   int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);

	if (w_flag) {
		int nvoices[2];
		snd_pmac_burgundy_write_volume(chip, MASK_ADDR_BURGUNDY_MASTER_VOLUME, voices, 8);
		snd_pmac_burgundy_read_volume(chip, MASK_ADDR_BURGUNDY_MASTER_VOLUME, nvoices, 8);
		return (voices[0] != nvoices[0] || voices[1] != nvoices[1]);
	} else {
		snd_pmac_burgundy_read_volume(chip, MASK_ADDR_BURGUNDY_MASTER_VOLUME, voices, 8);
		return 0;
	}
}

/*
 * Burgundy line-out: volume/mute
 */

/* mute stereo */
static int snd_pmac_burgundy_mute_lineout(snd_kmixer_element_t *element,
					  int w_flag, unsigned int *bitmap)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int oval, val;
	int change = 0;
	unsigned long flags;
	
	spin_lock_irqsave(&chip->reg_lock, flags);
	oval = snd_pmac_burgundy_rcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES);
	val = oval;
	if (w_flag) {
		if (snd_mixer_get_bit(bitmap, 0))
			val |= BURGUNDY_OUTPUT_LEFT;
		else
			val &= ~BURGUNDY_OUTPUT_LEFT;
		if (snd_mixer_get_bit(bitmap, 1))
			val |= BURGUNDY_OUTPUT_RIGHT;
		else
			val &= ~BURGUNDY_OUTPUT_RIGHT;
		if (val != oval) {
			snd_pmac_burgundy_wcb(chip,  MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES, val);
			change = 1;
		}
	} else {
		snd_mixer_set_bit(bitmap, 0, (oval & BURGUNDY_OUTPUT_LEFT) ? 1 : 0);
		snd_mixer_set_bit(bitmap, 1, (oval & BURGUNDY_OUTPUT_RIGHT) ? 1 : 0);
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return change;
}

/* volume: stereo, 0-15 */
static int snd_pmac_burgundy_volume_lineout(snd_kmixer_element_t *element,
					    int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int val, oval;
	int change = 0;

	oval = ~snd_pmac_burgundy_rcb(chip, MASK_ADDR_BURGUNDY_ATTENSPEAKER) & 0xff;
	if (w_flag) {
		val = ~((voices[1] << 4) | voices[0]);
		snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_ATTENSPEAKER, val);
		change = val != oval;
	} else {
		voices[0] = oval & 0xf;
		voices[1] = (oval >> 4) & 0xf;
	}
	return change;
}

/* speaker (internal speaker): mono */
/* mute mono */
static int snd_pmac_burgundy_mute_speaker(snd_kmixer_element_t *element,
					  int w_flag, unsigned int *bitmap)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int oval, val;
	int change = 0;
	unsigned long flags;
	
	spin_lock_irqsave(&chip->reg_lock, flags);
	oval = snd_pmac_burgundy_rcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES) & 0xff;
	val = oval;
	if (w_flag) {
		if (*bitmap & 1)
			val |= BURGUNDY_OUTPUT_INTERN;
		else
			val &= ~BURGUNDY_OUTPUT_INTERN;
		val &= 0xff;
		if (val != oval) {
			snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_MORE_OUTPUTENABLES, val);
			change = 1;
		}
	} else {
		*bitmap = (oval & BURGUNDY_OUTPUT_INTERN) ? 1 : 0;
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return change;
}

/* volume: mono, 0-15 */
static int snd_pmac_burgundy_volume_speaker(snd_kmixer_element_t *element,
					    int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int val, oval;
	int change = 0;

	oval = ~snd_pmac_burgundy_rcb(chip, MASK_ADDR_BURGUNDY_ATTENHP) & 0xff;
	if (w_flag) {
		val = ~((voices[0] << 4) | voices[0]);
		snd_pmac_burgundy_wcb(chip, MASK_ADDR_BURGUNDY_ATTENHP, val);
		change = val != oval;
	} else {
		voices[0] = oval & 0xf;
	}
	return change;
}

/*
 * Burgundy line, cd, mic inputs: volume/capture
 */

/* volume stereo */
static int snd_pmac_burgundy_volume_input(pmac_t *chip, int w_flag, int *voices, int addr)
{
	if (w_flag) {
		int nvoices[2];
		snd_pmac_burgundy_write_volume(chip, addr, voices, 16);
		snd_pmac_burgundy_read_volume(chip, addr, nvoices, 16);
		return (voices[0] != nvoices[0] || voices[1] != nvoices[1]);
	} else {
		snd_pmac_burgundy_read_volume(chip, addr, voices, 16);
		return 0;
	}
}

#define MAKE_BURGUNDY_VOLUME_INPUT(name,addr) \
static int snd_pmac_burgundy_volume_##name(snd_kmixer_element_t *element, int w_flag, int *voices)\
{\
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);\
	return snd_pmac_burgundy_volume_input(chip, w_flag, voices, addr);\
}

MAKE_BURGUNDY_VOLUME_INPUT(line, MASK_ADDR_BURGUNDY_VOLLINE);
MAKE_BURGUNDY_VOLUME_INPUT(cd, MASK_ADDR_BURGUNDY_VOLCD);
MAKE_BURGUNDY_VOLUME_INPUT(mic, MASK_ADDR_BURGUNDY_VOLMIC);

/*
 * AWACS line, cd, mic switches
 */
/* capture mono */
static int snd_pmac_awacs_capture_input(pmac_t *chip, int w_flag, unsigned int *value, int reg, int mask)
{
	unsigned long flags;
	int change = 0;
	spin_lock_irqsave(&chip->reg_lock, flags);
	if (w_flag) {
		int val = chip->awacs_reg[reg], oval;
		oval = val & mask;
		if (*value)
			val |= mask;
		else
			val &= ~mask;
		snd_pmac_awacs_write_reg(chip, reg, val);
		change = oval == (val & mask);
	} else {
		*value = (chip->awacs_reg[reg] & mask) ? 1 : 0;
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return change;
}

#define MAKE_AWACS_INPUT(name,reg,mask) \
static int snd_pmac_awacs_capture_##name(snd_kmixer_element_t *element, int w_flag, int *voices)\
{\
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);\
	return snd_pmac_awacs_capture_input(chip, w_flag, voices, reg, mask);\
}

MAKE_AWACS_INPUT(line, 0, MASK_MUX_AUDIN);
MAKE_AWACS_INPUT(cd, 0, MASK_MUX_CD);
MAKE_AWACS_INPUT(mic, 0, MASK_MUX_MIC);
MAKE_AWACS_INPUT(micgain, 0, MASK_GAINLINE);
MAKE_AWACS_INPUT(master, 1, MASK_LOOPTHRU);


/*
 * beep volume/mute
 */
static int snd_pmac_mixer_volume_beep(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	if (w_flag) {
		int change = chip->beep_volume != voices[0];
		chip->beep_volume = voices[0];
		return change;
	} else {
		voices[0] = chip->beep_volume;
		return 0;
	}
}

/*
 * general mixer group callback
 */
static int snd_pmac_mixer_group(snd_kmixer_group_t *group,
				snd_kmixer_file_t *file, int w_flag,
				snd_mixer_group_t *ugroup)
{
	pmac_mixer_group_t *grp = (pmac_mixer_group_t*)group->private_data;
	int voices[2];
	int val;
	unsigned int bitmap;
	int change = 0;

	if (!w_flag) {
		ugroup->channels = grp->voices > 1 ? SND_MIXER_CHN_MASK_STEREO : SND_MIXER_CHN_MASK_MONO;
		ugroup->caps = grp->caps;
		if (grp->caps & SND_MIXER_GRPCAP_VOLUME) {
			grp->volume(grp->me_volume, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			if (grp->caps & SND_MIXER_GRPCAP_JOINTLY_VOLUME)
				voices[1] = voices[0];
			ugroup->volume.names.front_right = voices[0];
			ugroup->min = grp->vol_min;
			ugroup->max = grp->vol_max;
		}
		if (grp->caps & SND_MIXER_GRPCAP_MUTE) {
			grp->mute(grp->me_mute, 0, &bitmap);
			if (grp->caps & SND_MIXER_GRPCAP_JOINTLY_MUTE)
				ugroup->mute = bitmap ? 0 : SND_MIXER_CHN_MASK_STEREO;
			else {
				ugroup->mute = 0;
				if (! snd_mixer_get_bit(&bitmap, 0))
					ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
				if (! snd_mixer_get_bit(&bitmap, 1))
					ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
			}
		}
		if (grp->caps & SND_MIXER_GRPCAP_CAPTURE) {
			grp->capture(grp->me_capture, 0, &val);
			ugroup->capture = val ? SND_MIXER_CHN_MASK_STEREO : 0;
		}
	} else {
		if (grp->caps & SND_MIXER_GRPCAP_VOLUME) {
			voices[0] = ugroup->volume.names.front_left;
			if (grp->caps & SND_MIXER_GRPCAP_JOINTLY_VOLUME)
				voices[1] = voices[0];
			else
				voices[1] = ugroup->volume.names.front_right;
			if (grp->volume(grp->me_volume, 1, voices) > 0) {
				snd_mixer_element_value_change(file, grp->me_volume, 0);
				change = 1;
			}
		}
		if (grp->caps & SND_MIXER_GRPCAP_MUTE) {
			bitmap = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_LEFT))
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_RIGHT))
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (grp->mute(grp->me_mute, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, grp->me_mute, 0);
				change = 1;
			}
		}
		if (grp->caps & SND_MIXER_GRPCAP_CAPTURE) {
			val = (ugroup->capture & SND_MIXER_CHN_MASK_STEREO) ? 1 : 0;
			if (grp->capture(grp->me_capture, 1, &val) > 0) {
				snd_mixer_element_value_change(file, grp->me_capture, 0);
				change = 1;
			}
		}
	}
	return change;
}

/*
 * build a group and elements
 */
static snd_kmixer_element_t __init *
snd_pmac_mixer_build_group(pmac_t *chip, snd_kmixer_t *mixer,
			   pmac_mixer_group_t *grp,
			   char *name, int oss_dev, int dir_input,
			   unsigned int voices, unsigned int caps,
			   snd_mixer_volume1_control_t *fn_volume,
			   struct snd_mixer_element_volume1_range *db_range,
			   snd_mixer_sw1_control_t *fn_mute,
			   snd_mixer_sw2_control_t *fn_capture)
{
	snd_kmixer_group_t *group;
	snd_kmixer_element_t *route;
	char e_name[128];

	if ((group = snd_mixer_lib_group_ctrl
	     (mixer, name, 0, oss_dev, snd_pmac_mixer_group, grp)) == NULL)
		return NULL;

	grp->voices = voices;
	grp->caps = caps;

	if (voices > 1)
		grp->me_io = snd_mixer_lib_io_stereo(mixer, e_name, 0,
						     dir_input ? SND_MIXER_ETYPE_INPUT : SND_MIXER_ETYPE_OUTPUT, 0);
	else
		grp->me_io = snd_mixer_lib_io_mono(mixer, e_name, 0,
						     dir_input ? SND_MIXER_ETYPE_INPUT : SND_MIXER_ETYPE_OUTPUT, 0);
	if (snd_mixer_group_element_add(mixer, group, grp->me_io) < 0)
		return NULL;
	route = grp->me_io;

	if (fn_volume) {
		grp->vol_min = db_range->min;
		grp->vol_max = db_range->max;
		sprintf(e_name, "%s Volume", name);
		grp->volume = fn_volume;
		if ((grp->me_volume = snd_mixer_lib_volume1(mixer, e_name, 0, voices,
							    db_range, fn_volume, chip)) == NULL)
			return NULL;
		grp->caps |= SND_MIXER_GRPCAP_VOLUME;
		if (snd_mixer_group_element_add(mixer, group, grp->me_volume) < 0)
			return NULL;
		if (dir_input)
			snd_mixer_element_route_add(mixer, route, grp->me_volume);
		else
			snd_mixer_element_route_add(mixer, grp->me_volume, route);
		route = grp->me_volume;
	}

	if (fn_mute) {
		sprintf(e_name, "%s Volume", name);
		grp->mute = fn_mute;
		if (voices == 1 || (caps & SND_MIXER_GRPCAP_JOINTLY_MUTE))
			grp->me_mute = snd_mixer_lib_sw2(mixer, e_name, 0,
							 (snd_mixer_sw2_control_t*)fn_mute, chip);
		else
			grp->me_mute = snd_mixer_lib_sw1(mixer, e_name, 0, voices, fn_mute, chip);
		if (grp->me_mute == NULL)
			return NULL;
		if (snd_mixer_group_element_add(mixer, group, grp->me_mute) < 0)
			return NULL;
		grp->caps |= SND_MIXER_GRPCAP_MUTE;
		if (dir_input)
			snd_mixer_element_route_add(mixer, route, grp->me_mute);
		else
			snd_mixer_element_route_add(mixer, grp->me_mute, route);
		route = grp->me_mute;
	}

	if (fn_capture) {
		sprintf(e_name, "%s Switch", name);
		grp->capture = fn_capture;
		grp->me_capture = snd_mixer_lib_sw2(mixer, e_name, 0, fn_capture, chip);
		if (snd_mixer_group_element_add(mixer, group, grp->me_capture) < 0)
			return NULL;
		grp->caps |= SND_MIXER_GRPCAP_CAPTURE;
		if (voices > 1)
			grp->caps |= SND_MIXER_GRPCAP_JOINTLY_CAPTURE;
		if (dir_input)
			snd_mixer_element_route_add(mixer, route, grp->me_capture);
		route = grp->me_capture;
	}

	if (dir_input) {
		grp->in = grp->me_io; grp->out = route;
	} else {
		grp->in = route; grp->out = grp->me_io;
	}

	return route;
}

/*
 * dB ranges
 */
static struct snd_mixer_element_volume1_range db_range[2] = {
	{0, 100, -6200, 0},
	{0, 100, -6200, 0},
};
static struct snd_mixer_element_volume1_range db_range_s[2] = {
	{0, 15, -6200, 0},
	{0, 15, -6200, 0},
};
static struct snd_mixer_element_volume1_range db_range_amp[2] = {
	{0, 31, -6200, 0},
	{0, 31, -6200, 0},
};
static struct snd_mixer_element_volume1_range tumbler_db_range[2] = {
	{0, 500, -6200, 0},
	{0, 500, -6200, 0},
};
static struct snd_mixer_element_volume1_range daca_db_range[2] = {
	{0, 0x38, -7500, 1800},
	{0, 0x38, -7500, 1800},
};

/*
 * build burgundy mixers
 */
static int __init
snd_pmac_burgundy_mixer_new(pmac_t *chip, snd_kmixer_t *mixer)
{
	int val = 0;
	snd_kmixer_element_t *me_in_accu;
	snd_kmixer_element_t *me_out_accu;
	snd_kmixer_element_t *me_capend;

	/* Outputs */
	me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0);
	if (! me_out_accu)
		return -ENOMEM;

	/* master volume  */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_master,
					SND_MIXER_OUT_MASTER, SND_MIXER_OSS_VOLUME, 0,
					2,  chip->auto_mute ? SND_MIXER_GRPCAP_JOINTLY_MUTE : 0,
					snd_pmac_burgundy_volume_master,
					db_range,
					chip->auto_mute ? snd_pmac_mixer_mute_all : NULL,
					NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, me_out_accu, chip->mix_master.in);
	/* headphone (internal speaker) */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_speaker,
					"Speaker", SND_MIXER_OSS_SPEAKER, 0,
					1, 0,
					snd_pmac_burgundy_volume_speaker,
					db_range_s,
					chip->auto_mute ? NULL : snd_pmac_burgundy_mute_speaker, NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_master.out, chip->mix_speaker.in);
	/* line-out */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_lineout,
					"Line Out", SND_MIXER_OSS_LINE1, 0,
					2, 0,
					snd_pmac_burgundy_volume_lineout,
					db_range_s,
					chip->auto_mute ? NULL : snd_pmac_burgundy_mute_lineout, NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_master.out, chip->mix_lineout.in);
	/* beep volume */
	if (chip->beep_buf) {
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_beep,
						"Beep", SND_MIXER_OSS_ALTPCM, 0,
						1, 0,
						snd_pmac_mixer_volume_beep,
						db_range,
						NULL, NULL))
			return -ENOMEM;
		snd_mixer_element_route_add(mixer, chip->mix_beep.out, me_out_accu);
	}

	/* Inputs */
	me_in_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_INPUT_ACCU, 0, 0);
	if (! me_in_accu)
		return -ENOMEM;

	/* line-in */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_line,
					SND_MIXER_IN_LINE, SND_MIXER_OSS_LINE, 1,
					2, 0,
					snd_pmac_burgundy_volume_line,
					db_range,
					NULL, NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_line.out, me_in_accu);
	/* CD */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_cd,
				 SND_MIXER_IN_CD, SND_MIXER_OSS_CD, 1,
				 2, 0,
				 snd_pmac_burgundy_volume_cd,
				 db_range,
				 NULL, NULL) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_cd.out, me_in_accu);
	/* Mic */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_mic,
				 SND_MIXER_IN_MIC, SND_MIXER_OSS_MIC, 1,
				 1, 0,
				 snd_pmac_burgundy_volume_mic,
				 db_range,
				 NULL, NULL) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_mic.out, me_in_accu);
	/* Capture endpoint */
	me_capend = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE1, 1, &val);
	if (! me_capend) return -ENOMEM;
	snd_mixer_element_route_add(mixer, me_in_accu, me_capend);

	return 0;
}

static int __init
snd_pmac_awacs_mixer_new(pmac_t *chip, snd_kmixer_t *mixer)
{
	int val = 0;
	snd_kmixer_element_t *me_in_accu;
	snd_kmixer_element_t *me_out_accu;
	snd_kmixer_element_t *me_capend;

	/* Outputs */
	me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0);
	if (! me_out_accu)
		return -ENOMEM;

	/* pseudo master */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_master,
					SND_MIXER_OUT_MASTER, SND_MIXER_OSS_VOLUME, 0,
					2, SND_MIXER_GRPCAP_JOINTLY_CAPTURE |
					(chip->auto_mute ? SND_MIXER_GRPCAP_JOINTLY_MUTE : 0),
					snd_pmac_awacs_volume_master,
					db_range,
					chip->auto_mute ? snd_pmac_mixer_mute_all : NULL,
					snd_pmac_awacs_capture_master))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, me_out_accu, chip->mix_master.in);
	/* speaker */
#ifdef AMP_AVAIL
	if (chip->amp_only) {
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_speaker,
						"Speaker", SND_MIXER_OSS_SPEAKER, 0,
						2, 0,
						snd_pmac_awacs_volume_amp,
						db_range_amp,
						NULL, NULL))
			return -ENOMEM;
	} else {
#endif /* AMP_AVAIL */
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_speaker,
						"Speaker", SND_MIXER_OSS_SPEAKER, 0,
						2, chip->auto_mute ? 0 : SND_MIXER_GRPCAP_JOINTLY_MUTE,
						snd_pmac_awacs_volume_speaker,
						db_range_s,
						chip->auto_mute ? NULL : snd_pmac_awacs_mute_speaker, NULL))
			return -ENOMEM;
#ifdef AMP_AVAIL
	}
#endif /* AMP_AVAIL */
	snd_mixer_element_route_add(mixer, me_out_accu, chip->mix_speaker.in);
	/* beep volume */
	if (chip->beep_buf) {
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_beep,
						"Beep", SND_MIXER_OSS_ALTPCM, 0,
						1, 0,
						snd_pmac_mixer_volume_beep,
						db_range,
						NULL, NULL))
			return -ENOMEM;
		snd_mixer_element_route_add(mixer, chip->mix_beep.out, me_out_accu);
	}

	/* line-out */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_lineout,
					"Line Out", SND_MIXER_OSS_LINE1, 0,
					2, chip->auto_mute ? 0 : SND_MIXER_GRPCAP_JOINTLY_MUTE,
					snd_pmac_awacs_volume_lineout,
					db_range_s,
					chip->auto_mute ? NULL : snd_pmac_awacs_mute_lineout,
					NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_master.out, chip->mix_lineout.in);

	/* Inputs */
	me_in_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_INPUT_ACCU, 0, 0);
	if (! me_in_accu)
		return -ENOMEM;

	/* line-in */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_line,
				 SND_MIXER_IN_LINE, SND_MIXER_OSS_LINE, 1,
				 2, 0,
				 NULL, NULL,
				 NULL, snd_pmac_awacs_capture_line) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_line.out, me_in_accu);
	/* CD */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_cd,
				 SND_MIXER_IN_CD, SND_MIXER_OSS_CD, 1,
				 2, 0,
				 NULL, NULL,
				 NULL, snd_pmac_awacs_capture_cd) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_cd.out, me_in_accu);
	/* Mic gain */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_mic_gain,
				 SND_MIXER_GRP_MIC_GAIN, SND_MIXER_OSS_IGAIN, 1,
				 1, 0,
				 NULL, NULL,
				 NULL, snd_pmac_awacs_capture_micgain) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_mic_gain.out, me_in_accu);
	/* Mic */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_mic,
				 SND_MIXER_IN_MIC, SND_MIXER_OSS_MIC, 1,
				 1, 0,
				 NULL, NULL,
				 NULL, snd_pmac_awacs_capture_mic) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_mic.out, chip->mix_mic_gain.in);
	/* capture level */
	if (snd_pmac_mixer_build_group(chip, mixer, &chip->mix_capture,
				 SND_MIXER_GRP_IGAIN, SND_MIXER_OSS_RECLEV, 1,
				 2, 0,
				 snd_pmac_awacs_volume_capture,
				 db_range_s,
				 NULL, NULL) < 0)
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, me_in_accu, chip->mix_capture.in);
	/* Capture endpoint */
	me_capend = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE1, 1, &val);
	if (! me_capend) return -ENOMEM;
	snd_mixer_element_route_add(mixer, chip->mix_capture.out, me_capend);

	return 0;
}


/*
 * Keywest i2c code
 *
 * based on i2c-keywest.c from lm_sensors.
 *    Copyright (c) 2000 Philip Edelbrock <phil@stimpy.netroedge.com>
 *
 */

/* The Tumbler audio equalizer can be really slow sometimes */
#define KW_POLL_SANITY 10000

/* address indices */
#define KW_ADDR_MODE	0
#define KW_ADDR_CONTROL	1
#define KW_ADDR_STATUS	2
#define KW_ADDR_ISR	3
#define KW_ADDR_IER	4
#define KW_ADDR_ADDR	5
#define KW_ADDR_SUBADDR	6
#define KW_ADDR_DATA	7

#define KW_I2C_ADDR(chip,type)	((chip)->i2c_base + (type) * (chip)->i2c_steps)

/* keywest needs a small delay to defuddle itself after changing a setting */
inline static void keywest_writeb_wait(pmac_t *chip, int addr, int value)
{
	writeb(value, KW_I2C_ADDR(chip, addr));
	udelay(10);
}

inline static void keywest_writeb(pmac_t *chip, int addr, int value)
{
	writeb(value, KW_I2C_ADDR(chip, addr));
}

inline unsigned char keywest_readb(pmac_t *chip, int addr)
{
	return readb(KW_I2C_ADDR(chip, addr));
}

static int keywest_poll_interrupt(pmac_t *chip)
{
	int i, res;
	for (i = 0; i < KW_POLL_SANITY; i++) {
		udelay(100);
		res = keywest_readb(chip, KW_ADDR_ISR);
		if (res > 0)
			return res;
	}

	// snd_printd("pmac: Sanity check failed!  Expected interrupt never happened.\n");
	return -ENODEV;
}


static void keywest_reset(pmac_t *chip)
{
	int interrupt_state;

	/* Clear all past interrupts */
	interrupt_state = keywest_readb(chip, KW_ADDR_ISR) & 0x0f;
	if (interrupt_state > 0)
		keywest_writeb(chip, KW_ADDR_ISR, interrupt_state);
}

static int keywest_start(pmac_t *chip, unsigned char cmd, int is_read)
{
	int interrupt_state;
	keywest_reset(chip);

	/* Set up address and r/w bit */
	keywest_writeb_wait(chip, KW_ADDR_ADDR, (chip->i2c_addr << 1) | (is_read ? 1 : 0));

	/* Set up 'sub address' which I'm guessing is the command field? */
	keywest_writeb_wait(chip, KW_ADDR_SUBADDR, cmd);
	
	/* Start sending address */
	keywest_writeb_wait(chip, KW_ADDR_CONTROL, keywest_readb(chip, KW_ADDR_CONTROL) | 2);
	interrupt_state = keywest_poll_interrupt(chip);
	if (interrupt_state < 0)
		return interrupt_state;

	if ((keywest_readb(chip, KW_ADDR_STATUS) & 0x02) == 0) {
		snd_printd("pmac: Ack Status on addr expected but got: 0x%02x on addr: 0x%02x\n", ack, chip->i2c_addr);
		return -EINVAL;
	} 
	return interrupt_state;
}

static int snd_pmac_keywest_write(pmac_t *chip, unsigned char cmd, int len, unsigned char *data)
{
	int interrupt_state;
	int error_state = 0;
	int i;

	snd_debug_check(len < 1 || len > 32, -EINVAL);

	if ((interrupt_state = keywest_start(chip, cmd, 0)) < 0)
		return -EINVAL;

	for(i = 0; i < len; i++) {
		keywest_writeb_wait(chip, KW_ADDR_DATA, data[i]);

		/* Clear interrupt and go */
		keywest_writeb_wait(chip, KW_ADDR_ISR, interrupt_state);

		interrupt_state = keywest_poll_interrupt(chip);
		if (interrupt_state < 0) {
			error_state = -EINVAL;
			interrupt_state = 0;
		}
		if ((keywest_readb(chip, KW_ADDR_STATUS) & 0x02) == 0) {
			snd_printd("pmac: Ack Expected by not received(block)!\n");
			error_state = -EINVAL;
		}
	}

	/* Send stop */
	keywest_writeb_wait(chip, KW_ADDR_CONTROL,
			    keywest_readb(chip, KW_ADDR_CONTROL) | 4);

	keywest_writeb_wait(chip, KW_ADDR_CONTROL, interrupt_state);
		
	interrupt_state = keywest_poll_interrupt(chip);
	if (interrupt_state < 0) {
		interrupt_state = 0;
		error_state = -EINVAL;
	}
	keywest_writeb_wait(chip, KW_ADDR_ISR, interrupt_state);

	return error_state;
}

inline static int snd_pmac_keywest_write_byte(pmac_t *chip, unsigned char cmd, unsigned char data)
{
	return snd_pmac_keywest_write(chip, cmd, 1, &data);
}

static void snd_pmac_keywest_cleanup(pmac_t *chip)
{
	if (chip->i2c_base) {
		iounmap((void*)chip->i2c_base);
		chip->i2c_base = 0;
	}
}

static int snd_pmac_keywest_find(pmac_t *chip, int addr, int (*init_client)(pmac_t *))
{
	struct device_node *i2c_device;
	void **temp;
	void *base = NULL;
	u32 steps = 0;

	i2c_device = find_compatible_devices("i2c", "keywest");
	
	if (i2c_device == 0) {
		snd_printk("No Keywest i2c devices found.\n");
		return -ENODEV;
	}
	
	for (; i2c_device; i2c_device = i2c_device->next) {
		snd_printd("Keywest device found: %s\n", i2c_device->full_name);
		temp = (void **) get_property(i2c_device, "AAPL,address", NULL);
		if (temp != NULL) {
			base = *temp;
		} else {
			snd_printd("pmac: no 'address' prop!\n");
			continue;
		}

		temp = (void **) get_property(i2c_device, "AAPL,address-step", NULL);
		if (temp != NULL) {
			steps = *(uint *)temp;
		} else {
			snd_printd("pmac: no 'address-step' prop!\n");
			continue;
		}
		
		chip->i2c_base = (unsigned long)ioremap((unsigned long)base, steps * 8);
		chip->i2c_steps = steps;
		chip->i2c_addr = addr;

		/* Select standard sub mode 
		 *  
		 * ie for <Address><Ack><Command><Ack><data><Ack>... style transactions
		 */
		keywest_writeb_wait(chip, KW_ADDR_MODE, 0x08);

		/* Enable interrupts */
		keywest_writeb_wait(chip, KW_ADDR_IER, 1 + 2 + 4 + 8);

		keywest_reset(chip);

		if (init_client(chip) < 0) {
			snd_pmac_keywest_cleanup(chip);
			continue;
		}

		return 0; /* ok */
	}
	
	return -ENODEV;
}


/*
 * tumbler (TAS3001c) interface
 */

#define TAS_REG_MCS	0x01
#define TAS_REG_VOL	0x04
#define TAS_VOL_MAX ((1<<20) - 1)

#define TAS_REG_TREBLE	0x05
#define TAS_VOL_MAX_TREBLE	0x96	/* 1 = max, 0x96 = min */
#define TAS_REG_BASS	0x06
#define TAS_VOL_MAX_BASS	0x86	/* 1 = max, 0x86 = min */

static int tas_init_client(pmac_t *chip)
{
	/* normal operation, SCLK=64fps, i2s output, i2s input, 16bit width */
	return snd_pmac_keywest_write_byte(chip, TAS_REG_MCS,
					   (1<<6)+(2<<4)+(2<<2)+0);
}

static int tumbler_set_volume(pmac_t *chip, unsigned int left_vol, unsigned int right_vol)
{
	unsigned char block[6];
  
	if (! chip->i2c_base) {
		//printk("Try to set volume with no client !!!\n");  
		return -ENODEV;
	}
  
	left_vol<<=4;
	right_vol<<=4;

	if (left_vol > TAS_VOL_MAX)
		left_vol = TAS_VOL_MAX;
	if (right_vol > TAS_VOL_MAX)
		right_vol = TAS_VOL_MAX;
  
	block[0] = (left_vol >> 16) & 0xff;
	block[1] = (left_vol >> 8)  & 0xff;
	block[2] = (left_vol >> 0)  & 0xff;

	block[3] = (right_vol >> 16) & 0xff;
	block[4] = (right_vol >> 8)  & 0xff;
	block[5] = (right_vol >> 0)  & 0xff;
  
	if (snd_pmac_keywest_write(chip, TAS_REG_VOL, 6, block) < 0) {
		printk("tumbler: failed to set volume \n");  
		return -EINVAL; 
	}
	return 0;
}

#ifdef TUMBLER_TONE_CONTROL_SUPPORT
static int tumbler_set_bass(pmac_t *chip, int val)
{
	unsigned char data;

	if (! chip->i2c_base) {
		//printk("Try to set volume with no client !!!\n");  
		return -ENODEV;
	}
  
	val = TAS_VOL_MAX_BASS - val + 1;
	if (val < 1)
		data = 1;
	else if (val > TAS_VOL_MAX_BASS)
		data = TAS_VOL_MAX_BASS;
	else
		data = val;
	if (snd_pmac_keywest_write(chip TAS_REG_BASS, 1, &data) < 0) {
		printk("tumbler: failed to set bass volume\n");  
		return -EINVAL; 
	}
	return 0;
}

static int tumbler_set_treble(pmac_t *chip, int val)
{
	unsigned char data;

	if (! chip->i2c_base) {
		//printk("Try to set volume with no client !!!\n");  
		return -ENODEV;
	}
  
	val = TAS_VOL_MAX_TREBLE - val + 1;
	if (val < 1)
		data = 1;
	else if (val > TAS_VOL_MAX_BASS)
		data = TAS_VOL_MAX_BASS;
	else
		data = val;
	if (snd_pmac_keywest_write(chip, TAS_REG_TREBLE, 1, &data) < 0) {
		printk("tumbler: failed to set treble volume\n");  
		return -EINVAL; 
	}
	return 0;
}
#endif


/*
 * DACA interface
 */

#define DACA_REG_SR	0x01
#define DACA_REG_AVOL	0x02
#define DACA_REG_GCFG	0x03

#define DACA_VOL_MAX	0x38

/* amplifier switch */
static int daca_amp_get_switch(snd_kmixer_t *mixer, snd_kswitch_t * kswitch, snd_switch_t * uswitch)
{
	pmac_t *chip = snd_magic_cast(pmac_t, kswitch->private_data, -ENXIO);
	uswitch->type = SND_SW_TYPE_BOOLEAN;
	uswitch->value.enable = chip->daca_amp;
	return 0;
}

static int daca_amp_set_switch(snd_kmixer_t *mixer, snd_kswitch_t * kswitch, snd_switch_t * uswitch)
{
	pmac_t *chip = snd_magic_cast(pmac_t, kswitch->private_data, -ENXIO);
	int change;

	if (uswitch->type != SND_SW_TYPE_BOOLEAN)
		return -EINVAL;
	change = (chip->daca_amp != uswitch->value.enable);
	if (change) {
		chip->daca_amp = uswitch->value.enable;
		snd_pmac_keywest_write_byte(chip, DACA_REG_GCFG,
					    chip->daca_amp ? 0x05 : 0x04);
	}
	return change;
}

static snd_kswitch_t daca_amp_switch = {
	name:           "Power Amplifier",
	get:            (snd_get_switch_t *)daca_amp_get_switch,
	set:            (snd_set_switch_t *)daca_amp_set_switch,
};

static int daca_init_client(pmac_t *chip)
{
	unsigned short data = 0x00;

	/* 1 bit delay */
	if (snd_pmac_keywest_write_byte(chip, DACA_REG_SR, 0x08) < 0)
		return -EINVAL;
	/* power amp inverted, DAC selected */ 
	chip->daca_amp = 1; /* default on */
	if (snd_pmac_keywest_write_byte(chip, DACA_REG_GCFG, 0x05) < 0)
		return -EINVAL;
	/* mute */
	return snd_pmac_keywest_write(chip, DACA_REG_AVOL, 2, (unsigned char*)&data);
}

static int daca_set_volume(pmac_t *chip, unsigned int left_vol, unsigned int right_vol)
{
	unsigned char data[2];
  
	if (! chip->i2c_base) {
		// printk("Try to set volume with no client !!!\n");  
		return -ENODEV;
	}
  
	if (left_vol > DACA_VOL_MAX)
		left_vol = DACA_VOL_MAX;
	if (right_vol > DACA_VOL_MAX)
		right_vol = DACA_VOL_MAX;
  
	data[0] = right_vol;
	data[1] = left_vol;
	// data[1] |= chip->deemphasis ? 0x40 : 0;
  
	if (snd_pmac_keywest_write(chip, DACA_REG_AVOL, 2, data) < 0) {
		printk("daca: failed to set volume \n");  
		return -EINVAL; 
	}
	return 0;
}


/*
 *
 */
static int snd_pmac_i2c_init(pmac_t *chip)
{
	if (chip->i2c_base)
		return 0;

	switch (chip->model) {
	case PMAC_TUMBLER:
		snd_pmac_keywest_find(chip, 0x34, tas_init_client);
		break;
	case PMAC_DACA:
		snd_pmac_keywest_find(chip, 0x4d, daca_init_client);
		break;
	default:
		break;
	}
	return 0;
}

static void snd_pmac_i2c_cleanup(pmac_t *chip)
{
	snd_pmac_keywest_cleanup(chip);
}


/*
 * build tumbler mixers
 */
static int tumbler_volume_master(snd_kmixer_element_t *element,
				 int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);

	if (w_flag) {
		int changed;
		tumbler_set_volume(chip, voices[0], voices[1]);
		changed = (voices[0] != chip->mixer_volume[0] ||
			voices[1] != chip->mixer_volume[1]);
		chip->mixer_volume[0] = voices[0];
		chip->mixer_volume[1] = voices[1];
		return changed;
	} else {
		voices[0] = chip->mixer_volume[0];
		voices[1] = chip->mixer_volume[1];
		return 0;
	}
}

#ifdef TUMBLER_TONE_CONTROL_SUPPORT
static int tumbler_tone_control(snd_kmixer_element_t *element, int w_flag,
				struct snd_mixer_element_tone_control1 *tc1)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);
	int change = 0;

	if (w_flag) {
		if (tc1->tc & SND_MIXER_TC1_BASS) {
			change |= chip->bass_volume != tc1->bass;
			tumbler_set_bass(chip, tc1->bass);
			chip->bass_volume = tc1->bass;
		}
		if (tc1->tc & SND_MIXER_TC1_TREBLE) {
			change |= chip->treble_volume != tc1->treble;
			tumbler_set_treble(chip, tc1->treble);
			chip->treble_volume = tc1->treble;
		}
	} else {
		tc1->tc = SND_MIXER_TC1_BASS | SND_MIXER_TC1_TREBLE;
		tc1->bass = chip->bass_volume;
		tc1->treble = chip->treble_volume;
	}
	return change;
}

static int tumbler_group_bass(snd_kmixer_group_t * group,
			      snd_kmixer_file_t * file, int w_flag,
			      snd_mixer_group_t * ugroup)
{
	pmac_t *chip = snd_magic_cast(pmac_t, group->private_data, -ENXIO);
	struct snd_mixer_element_tone_control1 tc;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = SND_MIXER_GRPCAP_VOLUME;
		ugroup->channels = SND_MIXER_CHN_MASK_MONO;
		tumbler_tone_control(chip->me_tone, 0, &tc);
		ugroup->volume.names.front_left = tc.bass;
		ugroup->min = 0;
		ugroup->max = TAS_VOL_MAX_BASS - 1;
	} else {
		tc.tc = SND_MIXER_TC1_BASS;
		tc.bass = ugroup->volume.names.front_left;
		if (tumbler_tone_control(chip->me_tone, 1, &tc) > 0) {
			snd_mixer_element_value_change(file, chip->me_tone, 0);
			change = 1;
		}
	}
	return change;
}

static int tumbler_group_treble(snd_kmixer_group_t * group,
				snd_kmixer_file_t * file, int w_flag,
				snd_mixer_group_t * ugroup)
{
	pmac_t *chip = snd_magic_cast(pmac_t, group->private_data, -ENXIO);
	struct snd_mixer_element_tone_control1 tc;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = SND_MIXER_GRPCAP_VOLUME;
		ugroup->channels = SND_MIXER_CHN_MASK_MONO;
		tumbler_tone_control(chip->me_tone, 0, &tc);
		ugroup->volume.names.front_left = tc.treble;
		ugroup->min = 0;
		ugroup->max = TAS_VOL_MAX_TREBLE - 1;
	} else {
		tc.tc = SND_MIXER_TC1_TREBLE;
		tc.treble = ugroup->volume.names.front_left;
		if (tumbler_tone_control(chip->me_tone, 1, &tc) > 0) {
			snd_mixer_element_value_change(file, chip->me_tone, 0);
			change = 1;
		}
	}
	return change;
}
#endif

static int __init
snd_pmac_tumbler_mixer_new(pmac_t *chip, snd_kmixer_t *mixer)
{
	snd_kmixer_element_t *me_out_accu;
#ifdef TUMBLER_TONE_CONTROL_SUPPORT
	snd_kmixer_group_t *group, *group2;
	static struct snd_mixer_element_tone_control1_info tone_control =
	{
		SND_MIXER_TC1_BASS | SND_MIXER_TC1_TREBLE,
		0, TAS_VOL_MAX_BASS - 1, -1800, 1800,
		0, TAS_VOL_MAX_TREBLE - 1, -1800, 1800
	};
#endif

	/* Outputs */
	me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0);
	if (! me_out_accu)
		return -ENOMEM;

	/* master volume  */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_master,
					SND_MIXER_OUT_MASTER, SND_MIXER_OSS_VOLUME, 0,
					2, 0,
					tumbler_volume_master,
					tumbler_db_range,
					NULL, NULL))
		return -ENOMEM;
#ifdef TUMBLER_TONE_CONTROL_SUPPORT
	/* Tone control */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_BASS, 0, SND_MIXER_OSS_BASS, tumbler_group_bass, chip)) == NULL)
		return -ENOMEM;
	if ((group2 = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_TREBLE, 0, SND_MIXER_OSS_TREBLE, tumbler_group_treble, chip)) == NULL)
		return -ENOMEM;
	if ((chip->me_tone = snd_mixer_lib_tone_control1(mixer, SND_MIXER_ELEMENT_TONE_CONTROL, 0, &tone_control, tumbler_tone_control, chip)) == NULL)
		return -ENOMEM;
	if (snd_mixer_group_element_add(mixer, group, chip->me_tone) < 0)
		return -ENOMEM;
	if (snd_mixer_group_element_add(mixer, group2, chip->me_tone) < 0)
		return -ENOMEM;
	if (snd_mixer_element_route_add(mixer, chip->me_tone, chip->mix_master.in) < 0 ||
	    snd_mixer_element_route_add(mixer, me_out_accu, chip->me_tone) < 0)
		return -ENOMEM;
#else
	if (snd_mixer_element_route_add(mixer, chip->mix_master.in, me_out_accu) < 0)
		return -ENOMEM;
#endif
	/* beep volume */
	if (chip->beep_buf) {
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_beep,
						"Beep", SND_MIXER_OSS_ALTPCM, 0,
						1, 0,
						snd_pmac_mixer_volume_beep,
						db_range,
						NULL, NULL))
			return -ENOMEM;
		snd_mixer_element_route_add(mixer, chip->mix_beep.out, me_out_accu);
	}

	return 0;
}


/*
 * DACA mixer
 */
static int snd_pmac_daca_volume_master(snd_kmixer_element_t *element,
				       int w_flag, int *voices)
{
	pmac_t *chip = snd_magic_cast(pmac_t, element->private_data, -ENXIO);

	if (w_flag) {
		int changed;
		daca_set_volume(chip, voices[0], voices[1]);
		changed = (voices[0] != chip->mixer_volume[0] ||
			voices[1] != chip->mixer_volume[1]);
		chip->mixer_volume[0] = voices[0];
		chip->mixer_volume[1] = voices[1];
		return changed;
	} else {
		voices[0] = chip->mixer_volume[0];
		voices[1] = chip->mixer_volume[1];
		return 0;
	}
}

static int __init
snd_pmac_daca_mixer_new(pmac_t *chip, snd_kmixer_t *mixer)
{
	snd_kmixer_element_t *me_out_accu;

	/* Outputs */
	me_out_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0);
	if (! me_out_accu)
		return -ENOMEM;

	/* master volume  */
	if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_master,
					SND_MIXER_OUT_MASTER, SND_MIXER_OSS_VOLUME, 0,
					2, 0,
					snd_pmac_daca_volume_master,
					daca_db_range,
					NULL, NULL))
		return -ENOMEM;
	snd_mixer_element_route_add(mixer, me_out_accu, chip->mix_master.in);
	/* beep volume */
	if (chip->beep_buf) {
		if (!snd_pmac_mixer_build_group(chip, mixer, &chip->mix_beep,
						"Beep", SND_MIXER_OSS_ALTPCM, 0,
						1, 0,
						snd_pmac_mixer_volume_beep,
						db_range,
						NULL, NULL))
			return -ENOMEM;
		snd_mixer_element_route_add(mixer, chip->mix_beep.out, me_out_accu);
	}

	snd_mixer_switch_new(mixer, &daca_amp_switch, chip);

	return 0;
}


/*
 *
 */

static int __init snd_pmac_mixer_new(pmac_t *chip)
{
	snd_kmixer_t *mixer;
	int err;

	if ((err = snd_mixer_new(chip->card, "PMac", 0, &mixer)) < 0)
		return err;
	strcpy(mixer->name, chip->pcm->name);

	switch (chip->model) {
	case PMAC_BURGUNDY:
		return snd_pmac_burgundy_mixer_new(chip, mixer);
	case PMAC_TUMBLER:
		return snd_pmac_tumbler_mixer_new(chip, mixer);
	case PMAC_DACA:
		return snd_pmac_daca_mixer_new(chip, mixer);
	case PMAC_SCREAMER:
	case PMAC_AWACS:
	default:
		return snd_pmac_awacs_mixer_new(chip, mixer);
	}
}


/*
 * sleep notify for powerbook
 */

#ifdef CONFIG_PMAC_PBOOK
/*
 * Save state when going to sleep, restore it afterwards.
 */
static int snd_pmac_sleep_notify(struct pmu_sleep_notifier *self, int when)
{
	unsigned long flags;
	pmac_t *chip;

	/* hmm, this callback has no private data.. ;-< */
	snd_debug_check(snd_pmac_card == NULL, 0);
	chip = snd_magic_cast(pmac_t, snd_pmac_card->private_data, 0);

	switch (when) {
	case PBOOK_SLEEP_NOW:
		/* XXX we should stop any dma in progress when going to sleep
		   and restart it when we wake. */
		spin_lock_irqsave(&chip->playback.lock, flags);
		if (chip->playback.running || chip->beep_on) {
			out_le32(&chip->playback.dma->control, (RUN|WAKE|FLUSH|PAUSE) << 16);
			snd_pmac_wait_ack(&chip->playback);
			spin_lock(&chip->reg_lock);
			if (chip->beep_on) {
				del_timer(&chip->beep_timer);
				chip->beep_on = 0;
			}
			spin_unlock(&chip->reg_lock);
		}
		spin_unlock_irqrestore(&chip->playback.lock, flags);
		disable_irq(chip->irqptr->irq);
		disable_irq(chip->playback.irqptr->irq);
#ifdef CONFIG_PPC_HAS_FEATURE_CALLS
		ppc_md.feature_call(PMAC_FTR_SOUND_CHIP_ENABLE, chip->node, 0, 0);
#else
		if (chip->is_pbook_G3) {
			feature_clear(chip->node, FEATURE_Sound_CLK_enable);
			feature_clear(chip->node, FEATURE_Sound_power);
		}
#endif
		/* According to Darwin, we do that after turning off the sound
		 * chip clock. All this will have to be cleaned up once we properly
		 * parse the OF sound-objects
		 */
		if (machine_is_compatible("PowerBook3,1") ||
		    machine_is_compatible("PowerBook3,2")) {
			snd_pmac_awacs_write_noreg(chip, 1, chip->awacs_reg[1] | MASK_PAROUT);
			mdelay(200);
		}
		break;
	case PBOOK_WAKE:
		/* There is still a problem on wake. Sound seems to work fine
		   if I launch mpg123 and resumes fine if mpg123 was playing,
		   but the console beep is dead until I do something with the
		   mixer. Probably yet another timing issue */
#ifdef CONFIG_PPC_HAS_FEATURE_CALLS
		ppc_md.feature_call(PMAC_FTR_SOUND_CHIP_ENABLE, chip->node, 0, 1);
#else
		if (!feature_test(chip->node, FEATURE_Sound_CLK_enable)
		    || !feature_test(chip->node, FEATURE_Sound_power)) {
			/* these aren't present on the 3400 AFAIK -- paulus */
			feature_set(chip->node, FEATURE_Sound_CLK_enable);
			feature_set(chip->node, FEATURE_Sound_power);
		}
#endif
		if (machine_is_compatible("PowerBook3,1") ||
		    machine_is_compatible("PowerBook3,2")) {
			mdelay(80);
			snd_pmac_awacs_write_noreg(chip, 1, chip->awacs_reg[1] & ~MASK_PAROUT);
			mdelay(200);
		} else
			mdelay(1000);

		/* hardware dependent resume */
		switch (chip->model) {
		case PMAC_AWACS:
		case PMAC_SCREAMER:
			snd_pmac_awacs_write_reg(chip, 0, chip->awacs_reg[0]);
			snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
			snd_pmac_awacs_write_reg(chip, 2, chip->awacs_reg[2]);
			snd_pmac_awacs_write_reg(chip, 4, chip->awacs_reg[4]);
			if (chip->model == PMAC_SCREAMER) {
				snd_pmac_awacs_write_reg(chip, 5, chip->awacs_reg[5]);
				snd_pmac_awacs_write_reg(chip, 6, chip->awacs_reg[6]);
				snd_pmac_awacs_write_reg(chip, 7, chip->awacs_reg[7]);
			}
			/* Recalibrate chip */
			if (chip->model == PMAC_SCREAMER)
				snd_pmac_screamer_recalibrate(chip);
			break;
		case PMAC_BURGUNDY:
			/* XXX: do we have suspend/resume on machines with burgundy? */
			break;
		case PMAC_DACA:
			snd_pmac_keywest_write_byte(chip, DACA_REG_SR, 0x08);
			snd_pmac_keywest_write_byte(chip, DACA_REG_GCFG,
						    chip->daca_amp ? 0x05 : 0x04);
			daca_set_volume(chip, chip->mixer_volume[0], chip->mixer_volume[1]);
			break;
		case PMAC_TUMBLER:
			tumbler_reset(chip);
			snd_pmac_keywest_write_byte(chip, TAS_REG_MCS,
						    (1<<6)+(2<<4)+(2<<2)+0);
			tumbler_set_volume(chip, chip->mixer_volume[0], chip->mixer_volume[1]);
			tumbler_headphone_intr(0, chip, 0); /* update mute */
			break;
		}

		out_le32(&chip->awacs->control, chip->control | (chip->rate_index << 8));
		out_le32(&chip->awacs->byteswap, chip->format == SND_PCM_SFMT_S16_LE);

		enable_irq(chip->irqptr->irq);
		enable_irq(chip->playback.irqptr->irq);
		if (chip->revision == 3) {
			mdelay(100);
			snd_pmac_awacs_write(chip, 0x6000);
			mdelay(2);
			snd_pmac_awacs_write_reg(chip, 1, chip->awacs_reg[1]);
		}
		/* enable CD sound input */
		if (chip->macio_base && chip->is_pbook_G3) {
			out_8(chip->macio_base + 0x37, 3);
		} else if (chip->is_pbook_3400) {
#ifndef CONFIG_PPC_HAS_FEATURE_CALLS
			feature_set(chip->node, FEATURE_IOBUS_enable);
			udelay(10);
#endif
			in_8(chip->latch_base + 0x190);
		}
		/* Resume pending sounds. */
		spin_lock_irqsave(&chip->playback.lock, flags);
		if (chip->playback.running)
			out_le32(&chip->playback.dma->control, ((RUN|WAKE) << 16) + (RUN|WAKE));
		spin_unlock_irqrestore(&chip->playback.lock, flags);
	}
	return PBOOK_SLEEP_OK;
}

static struct pmu_sleep_notifier snd_pmac_sleep_notifier = {
	snd_pmac_sleep_notify, SLEEP_LEVEL_SOUND,
};

#endif /* CONFIG_PMAC_PBOOK */

/*
 */

static int __init snd_pmac_create(snd_card_t *card,
				  unsigned long dma_playback_size,
				  unsigned long dma_capture_size,
				  pmac_t **chip_return)
{
	pmac_t *chip;
	int err;

	*chip_return = NULL;
	chip = snd_magic_kcalloc(pmac_t, 0, GFP_KERNEL);
	if (chip == NULL)
		return -ENOMEM;
	card->private_data = chip;

	chip->card = card;
	spin_lock_init(&chip->reg_lock);

	spin_lock_init(&chip->playback.lock);
	chip->playback.buffer_size = dma_playback_size;
	chip->playback.stream = SND_PCM_CHANNEL_PLAYBACK;
	if (snd_pmac_dbdma_alloc(&chip->playback.space, &chip->playback.cmds, PMAC_MAX_FRAGS + 1) < 0)
		return -ENOMEM;

	spin_lock_init(&chip->capture.lock);
	chip->capture.buffer_size = dma_capture_size;
	chip->capture.stream = SND_PCM_CHANNEL_CAPTURE;
	if (snd_pmac_dbdma_alloc(&chip->capture.space, &chip->capture.cmds, PMAC_MAX_FRAGS + 1) < 0)
		return -ENOMEM;

	if (snd_pmac_dbdma_alloc(&chip->extra_space, &chip->extra_cmd, 2) < 0)
		return -ENOMEM;

	if ((err = snd_pmac_detect(chip)) < 0)
		return err;
	if ((err = snd_pmac_resources(chip)) < 0)
		return err;
	if ((err = snd_pmac_init(chip)) < 0)
		return err;

	*chip_return = chip;

	return 0;
}

static void snd_pmac_free(void *private_data)
{
	pmac_t *chip = snd_magic_cast(pmac_t, private_data, );

	/* stop sounds */
	if (chip->initialized) {
		if (chip->playback.dma)
			out_le32(&chip->playback.dma->control, RUN<<16);
		if (chip->capture.dma)
			out_le32(&chip->capture.dma->control, RUN<<16);
		/* disable interrupts from awacs interface */
		out_le32(&chip->awacs->control, in_le32(&chip->awacs->control) & 0xfff);
	}

#ifdef CONFIG_PMAC_PBOOK
#ifdef CONFIG_PPC_HAS_FEATURE_CALLS
	/* Switch off the sound clock */
	ppc_md.feature_call(PMAC_FTR_SOUND_CHIP_ENABLE, chip->node, 0, 0);
#else
	if (chip->is_pbook_G3) {
		feature_clear(chip->node, FEATURE_Sound_power);
		feature_clear(chip->node, FEATURE_Sound_CLK_enable);
	}
#endif /* CONFIG_PPC_HAS_FEATURE_CALLS */
	pmu_unregister_sleep_notifier(&snd_pmac_sleep_notifier);
#endif /* CONFIG_PMAC_PBOOK */

#ifdef ENABLE_BEEP
	if (chip->beep_buf) {
		unsigned long flags;
		spin_lock_irqsave(&chip->reg_lock, flags);
		if (chip->beep_on)
			del_timer(&chip->beep_timer);
		spin_unlock_irqrestore(&chip->reg_lock, flags);
		kd_mksound = chip->orig_mksound;
		snd_kfree(chip->beep_buf);
	}
#endif

	/* clean up i2c clients */
	switch (chip->model) {
	case PMAC_TUMBLER:
	case PMAC_DACA:
		snd_pmac_i2c_cleanup(chip);
		break;
	}

	/* release resources */
	if (chip->extra_space)
		snd_kfree((void*)chip->extra_space);
	if (chip->playback.space)
		snd_kfree((void*)chip->playback.space);
	if (chip->capture.space)
		snd_kfree((void*)chip->capture.space);
	if (chip->macio_base)
		iounmap(chip->macio_base);
	if (chip->latch_base)
		iounmap(chip->latch_base);
	if (chip->awacs)
		iounmap((void*)chip->awacs);
	if (chip->playback.dma)
		iounmap((void*)chip->playback.dma);
	if (chip->capture.dma)
		iounmap((void*)chip->capture.dma);
	if (chip->playback.hwinfo)
		snd_kfree(chip->playback.hwinfo);
	if (chip->capture.hwinfo)
		snd_kfree(chip->capture.hwinfo);

	/* for tumbler */
	if (chip->amp_mute)
		iounmap(chip->amp_mute);
	if (chip->headphone_mute)
		iounmap(chip->headphone_mute);
	if (chip->headphone_status)
		iounmap(chip->headphone_status);

	snd_magic_kfree(chip);
}


inline static int snd_dma_size(int size, int min, int max)
{
	if (size < min) {
		snd_printk("too small dma size %d, using %d\n", size, min);
		return min;
	} else if (size > max) {
		snd_printk("too large dma size %d, using %d\n", size, max);
		return max;
	} else
		return size;
}

static void snd_pmac_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_pmac_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_pmac_probe(void)
{
	snd_card_t *card;
	pmac_t *chip;
	char *chip_name, *name_ext = "";
	int err;

	card = snd_card_new(snd_index, snd_id,
			    snd_pmac_use_inc, snd_pmac_use_dec);
	if (card == NULL)
		return -ENOMEM;

	card->type = SND_CARD_TYPE_AWACS;
	card->private_free = snd_pmac_free;

	if ((err = snd_pmac_create(card,
				   snd_dma_size(snd_dac_frame_size, 4, 128),
				   snd_dma_size(snd_adc_frame_size, 4, 128),
				   &chip)) < 0)
		goto __error;

	if ((err = snd_pmac_pcm_new(chip)) < 0)
		goto __error;
	if ((err = snd_pmac_mixer_new(chip)) < 0)
		goto __error;

	strcpy(card->abbreviation, "PMac");
	strcpy(card->shortname, "PowerMac");
	switch (chip->model) {
	case PMAC_BURGUNDY:
		chip_name = "Burgundy";
		break;
	case PMAC_DACA:
		chip_name = "DACA";
		break;
	case PMAC_TUMBLER:
		chip_name = "Tumbler";
		break;
	case PMAC_SCREAMER:
		chip_name = "Screamer";
		break;
	default:
		chip_name = "AWACS";
		break;
	}
	if (chip->is_pbook_3400)
		name_ext = ":PB3400";
	else if (chip->is_pbook_G3)
		name_ext = ":PBG3";
	sprintf(card->longname, "%s [%s%s] Rev:%d Dev:%d SF:%d",
		card->shortname, chip_name, name_ext, chip->revision, chip->device_id, chip->subframe);

	if ((err = snd_card_register(card)) < 0)
		goto __error;

	chip->initialized = 1;
	snd_pmac_card = card;

#ifdef CONFIG_PMAC_PBOOK
	pmu_register_sleep_notifier(&snd_pmac_sleep_notifier);
#endif /* CONFIG_PMAC_PBOOK */

	return 0;

__error:
	snd_card_free(card);
	return err;
}


/*
 * MODULE sutff
 */

static int __init alsa_card_pmac_init(void)
{
	return snd_pmac_probe();
}

static void __exit alsa_card_pmac_exit(void)
{
	if (snd_pmac_card)
		snd_card_free(snd_pmac_card);
}

module_init(alsa_card_pmac_init)
module_exit(alsa_card_pmac_exit)
