/*
 *  Copyright (C) 2001 Luca Deri <deri@ntop.org>
 *
 * 	               http://www.ntop.org/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"

/*
  For more info see:

   http://www.cisco.com/warp/public/cc/pd/iosw/ioft/neflct/tech/napps_wp.htm

   ftp://ftp.net.ohio-state.edu/users/maf/cisco/
*/

struct flow_ver5_hdr {
  u_int16_t version;         /* Current version=5*/
  u_int16_t count;           /* The number of records in PDU. */
  u_int32_t sysUptime;       /* Current time in msecs since router booted */
  u_int32_t unix_secs;       /* Current seconds since 0000 UTC 1970 */
  u_int32_t unix_nsecs;      /* Residual nanoseconds since 0000 UTC 1970 */
  u_int32_t flow_sequence;   /* Sequence number of total flows seen */
  u_int8_t  engine_type;     /* Type of flow switching engine (RP,VIP,etc.)*/
  u_int8_t  engine_id;       /* Slot number of the flow switching engine */
};

 struct flow_ver5_rec {
   u_int32_t srcaddr;    /* Source IP Address */
   u_int32_t dstaddr;    /* Destination IP Address */
   u_int32_t nexthop;    /* Next hop router's IP Address */
   u_int16_t input;      /* Input interface index */
   u_int16_t output;     /* Output interface index */
   u_int32_t dPkts;      /* Packets sent in Duration (milliseconds between 1st
			   & last packet in this flow)*/
   u_int32_t dOctets;    /* Octets sent in Duration (milliseconds between 1st
			   & last packet in  this flow)*/
   u_int32_t First;      /* SysUptime at start of flow */
   u_int32_t Last;       /* and of last packet of the flow */
   u_int16_t srcport;    /* TCP/UDP source port number (.e.g, FTP, Telnet, etc.,or equivalent) */
   u_int16_t dstport;    /* TCP/UDP destination port number (.e.g, FTP, Telnet, etc.,or equivalent) */
   u_int8_t pad1;        /* pad to word boundary */
   u_int8_t tcp_flags;   /* Cumulative OR of tcp flags */
   u_int8_t prot;        /* IP protocol, e.g., 6=TCP, 17=UDP, etc... */
   u_int8_t tos;         /* IP Type-of-Service */
   u_int16_t dst_as;     /* dst peer/origin Autonomous System */
   u_int16_t src_as;     /* source peer/origin Autonomous System */
   u_int8_t dst_mask;    /* destination route's mask bits */
   u_int8_t src_mask;    /* source route's mask bits */
   u_int16_t pad2;       /* pad to word boundary */
};

typedef struct single_flow_ver5_rec {
  struct flow_ver5_hdr flowHeader;
  struct flow_ver5_rec flowRecord;
} SingleNetFlow5Record;

typedef struct double_flow_ver5_rec {
  struct flow_ver5_hdr flowHeader;
  struct flow_ver5_rec firstRecord;
  struct flow_ver5_rec secondRecord;
} DoubleNetFlow5Record;


/* Global */
static int netflowSocket = -1;
static struct sockaddr_in dest;
static u_int32_t destAddr;
static u_int32_t globalFlowSequence=0;

/* *********************************** */

static void initNetFlowExporter(char* dstHost, int dstPort) {
  struct hostent *hostAddr = gethostbyname(dstHost);

  if(hostAddr == NULL) {
    traceEvent(TRACE_INFO, "Unable to resolve address '%s'\n", dstHost);
    exit(-1);
  }

  memcpy(&dest.sin_addr.s_addr, hostAddr->h_addr_list[0], hostAddr->h_length);
  dest.sin_family      = AF_INET;
  dest.sin_port        = (int)htons((unsigned short int)dstPort);

  destAddr = ntohl(dest.sin_addr.s_addr);

  netflowSocket = socket (AF_INET, SOCK_DGRAM, 0);

  if(netflowSocket <= 0) {
    traceEvent(TRACE_INFO, "Unable to open NetFlow socket");
    exit(-1);
  }
}

/* **************************************** */

int handleNetFlowSupport(char* addr /* host:port */) {
  char *hostName, *strtokState;
  int  portNumber;

  if((addr == NULL) || (addr[0] == '\0'))
    return(-1);

  hostName = strtok_r(addr, ":", &strtokState);
  portNumber = atoi(strtok_r(NULL, ":", &strtokState));

  if((hostName == NULL) || (portNumber == 0)) {
    traceEvent(TRACE_WARNING,
	       "WARNING: invalid value specified for '-b' parameter. \n"
	       "         It should be host:port.");
    return(-1);
  } else {
    initNetFlowExporter(hostName, portNumber);
    enableNetFlowSupport = 1;
    traceEvent(TRACE_INFO, "Cisco NetFlow exporter towards %s:%d up and running",
	       hostName, portNumber);
    return(0);
  }
}

/* *********************************** */

void termNetFlowExporter() {
  if(enableNetFlowSupport) {
    traceEvent(TRACE_INFO, "Cisco NetFlow exporter terminated.");
#ifndef WIN32
    close(netflowSocket);
#else
    closesocket(netflowSocket);
#endif
    enableNetFlowSupport = 0;
  }
}

/* *********************************** */

static void initFlowHeader(struct flow_ver5_hdr *flowHeader, int numCount) {
  flowHeader->version        = htons(5);
  flowHeader->count          = htons(numCount);
  flowHeader->sysUptime      = htonl((actTime-initialSniffTime)*1000);
  flowHeader->unix_secs      = htonl(actTime);
  flowHeader->unix_nsecs     = htonl(0);
  flowHeader->flow_sequence  = htonl(globalFlowSequence);
  flowHeader->engine_type    = 0;
  flowHeader->engine_id      = 0;

  globalFlowSequence += numCount;

}

/* *********************************** */

/* 
   First/Last calculation time fixed by
   Wies-Software <wies@wiessoft.de>
*/

void sendTCPSessionFlow(IPSession *theSession) {
  DoubleNetFlow5Record theRecord;
  int rc;

  if((device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->initiatorIdx)] == NULL)
     || (device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->remotePeerIdx)] == NULL))
    return; /* Sanity check */

  memset(&theRecord, 0, sizeof(theRecord));

  initFlowHeader(&theRecord.flowHeader, 2);

  theRecord.firstRecord.srcaddr  = htonl(device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->initiatorIdx)]->hostIpAddress.s_addr);
  theRecord.firstRecord.dstaddr  = htonl(device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->remotePeerIdx)]->hostIpAddress.s_addr);
  theRecord.firstRecord.input    = htonl(1);
  theRecord.firstRecord.output   = htonl(1);
  theRecord.firstRecord.dPkts    = htonl(theSession->pktSent);
  theRecord.firstRecord.dOctets  = htonl(theSession->bytesSent);
  theRecord.firstRecord.First    = htonl((theSession->firstSeen-initialSniffTime)*1000);
  theRecord.firstRecord.Last     = htonl((theSession->lastSeen-initialSniffTime)*1000);
  theRecord.firstRecord.srcport  = htons(theSession->sport);
  theRecord.firstRecord.dstport  = htons(theSession->dport);
  theRecord.firstRecord.prot     = 6 /* TCP */;

  theRecord.secondRecord.srcaddr = theRecord.firstRecord.dstaddr;
  theRecord.secondRecord.dstaddr = theRecord.firstRecord.srcaddr;
  theRecord.secondRecord.input   = htonl(1);
  theRecord.secondRecord.output  = htonl(1);
  theRecord.secondRecord.dPkts   = htonl(theSession->pktRcvd);
  theRecord.secondRecord.dOctets = htonl(theSession->bytesReceived);
  theRecord.secondRecord.First   = htonl((theSession->firstSeen-initialSniffTime)*1000);
  theRecord.secondRecord.Last    = htonl((theSession->lastSeen-initialSniffTime)*1000);
  theRecord.secondRecord.srcport = htons(theSession->dport);
  theRecord.secondRecord.dstport = htons(theSession->sport);
  theRecord.secondRecord.prot    = 6 /* TCP */;

  rc = sendto(netflowSocket, (void*)&theRecord, sizeof(theRecord), 0, (struct sockaddr *)&dest, sizeof(dest));

#ifdef DEBUG
    traceEvent(TRACE_INFO, "Exported TCP flow... (rc=%d)", rc);
#endif
}

/* *********************************** */

void sendICMPflow(HostTraffic *srcHost,
		  HostTraffic *dstHost,
		  u_int length) {
  SingleNetFlow5Record theRecord;
  int rc;

  if(dstHost->hostIpAddress.s_addr == destAddr) {
    /*
      Flows generated by NetFlow collector host
      MUST be ignored (avoid race conditions)
    */
    return;
  }

  memset(&theRecord, 0, sizeof(theRecord));

  initFlowHeader(&theRecord.flowHeader, 1);

  theRecord.flowRecord.srcaddr = htonl(srcHost->hostIpAddress.s_addr);
  theRecord.flowRecord.dstaddr = htonl(dstHost->hostIpAddress.s_addr);
  theRecord.flowRecord.input   = htonl(1);
  theRecord.flowRecord.output  = htonl(1);
  theRecord.flowRecord.dPkts   = htonl(1);
  theRecord.flowRecord.dOctets = htonl(length);
  theRecord.flowRecord.First   = htonl((actTime-initialSniffTime)*1000);
  theRecord.flowRecord.Last    = htonl((actTime-initialSniffTime)*1000);
  theRecord.flowRecord.prot    = 1 /* ICMP */;

  rc = sendto(netflowSocket, (void*)&theRecord, sizeof(theRecord), 0,
	      (struct sockaddr *)&dest, sizeof(dest));

#ifdef DEBUG
    traceEvent(TRACE_INFO, "Exported ICMP flow... (rc=%d)", rc);
#endif
}

/* *********************************** */

void sendUDPflow(HostTraffic *srcHost,
		 HostTraffic *dstHost,
		 u_int sport, u_int dport,
		 u_int length) {
  SingleNetFlow5Record theRecord;
  int rc;

  if((dstHost->hostIpAddress.s_addr == destAddr)
     && (dest.sin_port == dport)) {
    /*
      Flows generated by UDP packets due to NetFlow
      MUST be ignored
    */
    return;
  }

  memset(&theRecord, 0, sizeof(theRecord));

  initFlowHeader(&theRecord.flowHeader, 1);

  theRecord.flowRecord.srcaddr   = htonl(srcHost->hostIpAddress.s_addr);
  theRecord.flowRecord.dstaddr   = htonl(dstHost->hostIpAddress.s_addr);
  theRecord.flowRecord.input     = htonl(1);
  theRecord.flowRecord.output    = htonl(1);
  theRecord.flowRecord.dPkts     = htonl(1);
  theRecord.flowRecord.dOctets   = htonl(length);
  theRecord.flowRecord.First     = htonl((actTime-initialSniffTime)*1000);
  theRecord.flowRecord.Last      = htonl((actTime-initialSniffTime)*1000);
  theRecord.flowRecord.srcport   = htons(sport);
  theRecord.flowRecord.dstport   = htons(dport);
  theRecord.flowRecord.prot      = 17 /* UDP */;

  rc = sendto(netflowSocket, (void*)&theRecord, sizeof(theRecord), 0,
	      (struct sockaddr *)&dest, sizeof(dest));

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Exported UDP flow... (rc=%d)", rc);
#endif
}

