#include <mac/All.h>
#include <mac/MACLib.h>
#include <mac/APEInfo.h>
#include <mac/APETag.h>
#include <mac/CharacterHelper.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <wchar.h>

#include "common.h"

#define COMMON_TAG_COUNT 9

static gchar *common_tags[] = 
{
    _("ALBUM"),
    _("ARTIST"),
    _("COMMENT"),
    _("COMPOSER"),
    _("DATE"),
    _("GENRE"),
    _("PERFORMER"),
    _("TITLE"),
    _("TRACKNUMBER")
};

typedef struct
{
    GtkWidget *window;
    GtkWidget *location;
    GtkWidget *techinfo;
    GtkWidget *metadata;

    IAPEDecompress *pAPEDecompress;
    gchar *current_filename;

} MAC_META_INFO;

typedef struct
{
    GtkWidget *tag_window;
    GtkWidget *field_name_widget;
    GtkWidget *field_value_widget;

    gboolean is_edit;
} MAC_META_TAG;
    
static MAC_META_INFO mac_info;
static MAC_META_TAG mac_tag;

#ifdef __cplusplus
extern "C"{
#endif

    /* callbacks of the whole APE info window */
    static void cb_info_box_close(GtkWidget *widget, gpointer data);
    static void cb_info_box_destroy(GtkWidget *widget, gpointer data);
    static gint cb_info_box_delete(GtkWidget *widget,
				   GdkEvent  *event,
				   gpointer   data );

    /* callbacks of the meta data related */
    static void cb_metadata_add(GtkWidget *widget, gpointer data);
    static void cb_metadata_edit(GtkWidget *widget, gpointer data);
    static void cb_metadata_remove(GtkWidget *widget, gpointer data);
    static void cb_metadata_update_file(GtkWidget *widget, gpointer data);

    /* callbacks of tag editing related */
    static void cb_tag_window_ok(GtkWidget *widget, gpointer data);
    static void cb_tag_window_cancel(GtkWidget *widget, gpointer data);
    static void cb_tag_window_destroy(GtkWidget *widget, gpointer data);
    static gint cb_tag_window_delete(GtkWidget *widget,
				     GdkEvent  *event,
				     gpointer   data );

    /* internal util functions */
    static void insert_metadata(GtkCList* clist, CAPETag *pAPETag);
    static void insert_techinfo(GtkCList* clist, IAPEDecompress *pAPEDecompress);

    static void mac_tag_window_create();
    static void mac_info_box_create();

/**************************
 *   tag window related   *
 **************************/

void cb_tag_window_destroy(GtkWidget *widget, gpointer data)
{

    /* do some clean stuff of tag window */

    mac_tag.tag_window = NULL;
    mac_tag.field_name_widget = NULL;
    mac_tag.field_value_widget = NULL;
    mac_tag.is_edit = FALSE;
}

gint cb_tag_window_delete(GtkWidget *widget,
			GdkEvent  *event,
			gpointer   data )
{
    /* Don't destroy the window */

    gtk_widget_hide_all(mac_tag.tag_window);
    return TRUE;
}

void cb_tag_window_ok(GtkWidget *widget, gpointer data)
{
    gchar *field_name, *field_value;
    GtkEntry *name_widget;
    GtkText *value_widget;
    
    name_widget = GTK_ENTRY(GTK_COMBO(mac_tag.field_name_widget)->entry);
    value_widget = GTK_TEXT(mac_tag.field_value_widget);

    field_name = gtk_entry_get_text(name_widget);
    field_value = gtk_editable_get_chars(GTK_EDITABLE(value_widget), 0, -1);

    if (mac_tag.is_edit)
    {
	/* Editing a tag */
	GList *sels;
	gint row;

	sels = GTK_CLIST(mac_info.metadata)->selection;
	if (sels)
	{
	    row = GPOINTER_TO_INT(sels->data);	
	    gtk_clist_set_text(GTK_CLIST(mac_info.metadata), 
			       row, 1,
			       field_value);
	}
    }
    else
    {
	/* Adding a tag */
	gchar *data[2];

	if (strlen(field_name) > 0 
	    && strlen(field_value) > 0)
	{
	    data[0] = field_name;
	    data[1] = field_value;
	    gtk_clist_append(GTK_CLIST(mac_info.metadata), data);
	}
    }
    gtk_widget_hide_all(mac_tag.tag_window);
}

void cb_tag_window_cancel(GtkWidget *widget, gpointer data)
{
    gtk_widget_hide_all(mac_tag.tag_window);
}

/**************************
 *    info box related    *
 **************************/

void cb_info_box_destroy(GtkWidget *widget, gpointer data)
{

    /* do some clean stuff of info box window */

    mac_info.window = NULL;
    mac_info.location = NULL;
    mac_info.techinfo = NULL;
    mac_info.metadata = NULL;

    g_free(mac_info.current_filename);
    mac_info.current_filename = NULL;
    if (mac_info.pAPEDecompress)
    {
	delete mac_info.pAPEDecompress;
    }
    mac_info.pAPEDecompress = NULL;
}

gint cb_info_box_delete(GtkWidget *widget,
			GdkEvent  *event,
			gpointer   data )
{

    gtk_widget_hide_all(mac_info.window);

    return TRUE;
}

void cb_info_box_close(GtkWidget *widget, gpointer data)
{
    gtk_widget_hide_all(mac_info.window);
}

/**************************
 *   meta data related    *
 **************************/

void cb_metadata_update_file(GtkWidget *widget, gpointer data)
{

    gint res1, res2, res;
    gint index;
    gchar *field_name, *field_value;

    /* remove all tags, then add one by one */

    CAPETag *apeTag = (CAPETag *)mac_info.pAPEDecompress->GetInfo(APE_INFO_TAG);
    apeTag->ClearFields();
    
    index = 0;
    while ((res1 = gtk_clist_get_text(GTK_CLIST(mac_info.metadata), index, 0, &field_name)))
    {
	res2 = gtk_clist_get_text(GTK_CLIST(mac_info.metadata), index, 1, &field_value);
	if (res1 && res2)
	{
	    wchar_t *utf16_name;
	    utf16_name = GetUTF16FromANSI(field_name);
	    res = apeTag->SetFieldString(utf16_name, field_value, FALSE);
	    g_free(utf16_name);
	}    
	index ++;
    }

    res = apeTag->Save();
    if (res != ERROR_SUCCESS)
    {
	printf("error update files, error code = %d\n", res);
    }
}

void cb_metadata_add(GtkWidget *widget, gpointer data)
{

    GtkEntry *entry;
    GtkText *field_value;
    gint length;

    if (!mac_tag.tag_window)
    {
	mac_tag_window_create();
    }

    entry = GTK_ENTRY(GTK_COMBO(mac_tag.field_name_widget)->entry);
    field_value = GTK_TEXT(mac_tag.field_value_widget);

    gtk_widget_set_sensitive(mac_tag.field_name_widget, TRUE);
    gtk_entry_set_text(entry, "");

    length = (guint)gtk_text_get_length(field_value);
    if (length > 0)
    {
	gtk_text_freeze(field_value);
	gtk_text_set_point(field_value, 0);
	gtk_text_forward_delete(field_value, length);
	gtk_text_thaw(field_value);
    }

    mac_tag.is_edit = FALSE;

    gtk_widget_show_all(mac_tag.tag_window);

}

void cb_metadata_edit(GtkWidget *widget, gpointer data)
{

    GtkEntry *entry; 
    GtkText *field_value;

    gint row;
    gchar *text;
    guint length;
    GList *sels;
    
    sels = GTK_CLIST(mac_info.metadata)->selection;
    if (!sels)
    {
	return ;
    }
    row = GPOINTER_TO_INT(sels->data);

    if (!mac_tag.tag_window)
    {
	mac_tag_window_create();
    }

    entry = GTK_ENTRY(GTK_COMBO(mac_tag.field_name_widget)->entry);
    field_value = GTK_TEXT(mac_tag.field_value_widget);

    gtk_clist_get_text(GTK_CLIST(mac_info.metadata), row, 0, &text);

    gtk_entry_set_text(entry, text);
    gtk_widget_set_sensitive(mac_tag.field_name_widget, FALSE);

    gtk_clist_get_text(GTK_CLIST(mac_info.metadata), row, 1, &text);

    length = (guint)gtk_text_get_length(field_value);
    if (length > 0)
    {
	gtk_text_freeze(field_value);
	gtk_text_set_point(field_value, 0);
	gtk_text_forward_delete(field_value, length);
	gtk_text_thaw(field_value);
    }
    gtk_text_insert(field_value, NULL, NULL,
		    NULL, text, -1);

    mac_tag.is_edit = TRUE;

    gtk_widget_show_all(mac_tag.tag_window);

}

void cb_metadata_remove(GtkWidget *widget, gpointer data)
{
    gint row;
    GList *sels;

    sels = GTK_CLIST(mac_info.metadata)->selection;
    if (!sels)
    {
	return ;
    }
    row = GPOINTER_TO_INT(sels->data);

    gtk_clist_remove(GTK_CLIST(mac_info.metadata), row);
}

/***************************
 * internal util functions *
 ***************************/

void insert_metadata(GtkCList* clist, CAPETag *pAPETag)
{
    if (!clist || !pAPETag)
    {
	return;
    }

    gtk_clist_freeze(clist);

    gtk_clist_clear(clist);
    
    BOOL bHasID3Tag = pAPETag->GetHasID3Tag();
    BOOL bHasAPETag = pAPETag->GetHasAPETag();

    if (bHasID3Tag || bHasAPETag)
    {
	CAPETagField * pTagField;
	int index = 0;
	while ((pTagField = pAPETag->GetTagField(index)) != NULL)
	{
	    index ++;

	    const wchar_t *field_name;
	    char field_value[255];

	    gchar *name;

	    field_name = pTagField->GetFieldName();
	    name = (gchar *)GetANSIFromUTF16(field_name);

	    memset(field_value, 0, 255);
	    int size = 255;
	    pAPETag->GetFieldString(field_name, (char *)field_value, &size, FALSE);

	    gchar *data[2];
	    data[0] = g_strdup(name);
	    data[1] = g_strdup(field_value);
	    
	    gtk_clist_append(clist, data);

	    g_free(data[0]);
	    g_free(data[1]);

	    g_free(name);
	}
    }
    gtk_clist_thaw(clist);
}

void insert_techinfo(GtkCList* clist, IAPEDecompress *pAPEDecompress)
{
    if (!clist || !pAPEDecompress)
    {
	return;
    }

    gchar *data[2];

    gtk_clist_freeze(clist);

    gtk_clist_clear(clist);

    /* APE Version */
    data[0] = _("Version");
    data[1] = g_strdup_printf("%.2f", (float)pAPEDecompress->GetInfo(APE_INFO_FILE_VERSION) / float(1000));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Compression Level */
    data[0] = _("Compression Level");
    switch (pAPEDecompress->GetInfo(APE_INFO_COMPRESSION_LEVEL))
    {
    case COMPRESSION_LEVEL_FAST:
	data[1] = _("Fast");
	break;
    case COMPRESSION_LEVEL_NORMAL:
	data[1] = _("Normal");
	break;
    case COMPRESSION_LEVEL_HIGH:
	data[1] = _("High");
	break;
    case COMPRESSION_LEVEL_EXTRA_HIGH:
	data[1] = _("Extra High");
	break;
    case COMPRESSION_LEVEL_INSANE:
	data[1] = _("Insane");
	break;
    }
    gtk_clist_append(clist, data);

    /* Format Flags */
    data[0] = _("Flags");
    data[1] = g_strdup_printf("%d", pAPEDecompress->GetInfo(APE_INFO_FORMAT_FLAGS));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Sample Rate */
    data[0] = _("Sample Rate");
    data[1] = g_strdup_printf("%d", pAPEDecompress->GetInfo(APE_INFO_SAMPLE_RATE));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Bits Per Sample */
    data[0] = _("Bits Per Sample");
    data[1] = g_strdup_printf("%d", pAPEDecompress->GetInfo(APE_INFO_BITS_PER_SAMPLE));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Channels */
    data[0] = _("Channels");
    data[1] = g_strdup_printf("%d", pAPEDecompress->GetInfo(APE_INFO_CHANNELS));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Actrual File if the file is APL */
    char *ext = strrchr(mac_info.current_filename, '.');
    if(ext)
    {
	if (!strcasecmp(ext, ".apl"))
	{
	    data[0] = _("Actrual File");
	    CIO *pIO = (CIO *)pAPEDecompress->GetInfo(APE_INFO_IO_SOURCE);
	    if (pIO)
	    {
		wchar_t nameBuf[255];
		memset(nameBuf, 0, 255 * sizeof(wchar_t));
		pIO->GetName(nameBuf);
		char *name = GetANSIFromUTF16(nameBuf);
		char *basename = g_basename(name);
		data[1] = basename;
		gtk_clist_append(clist, data);
		g_free(name);
	    }
	}
    }
    
    /* Total File Size */
    data[0] = _("File Size");
    data[1] = g_strdup_printf("%d bytes", pAPEDecompress->GetInfo(APE_INFO_APE_TOTAL_BYTES));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Length (Time) */
    data[0] = _("Length");
    unsigned int length = pAPEDecompress->GetInfo(APE_DECOMPRESS_LENGTH_MS);
    int min, sec, ms;
    ms = length % 1000;
    sec = length / 1000;
    min = sec / 60;
    sec = sec % 60;
    data[1] = g_strdup_printf("%d:%d.%d ", min, sec, ms);
    gtk_clist_append(clist, data);
    g_free(data[1]);

    /* Average Bitrate */
    data[0] = _("Bitrate");
    data[1] = g_strdup_printf("%d", pAPEDecompress->GetInfo(APE_DECOMPRESS_AVERAGE_BITRATE));
    gtk_clist_append(clist, data);
    g_free(data[1]);

    gtk_clist_thaw(clist);
}


void mac_tag_window_create()
{
    GtkWidget *tag_window;
    GtkWidget *vbox, *hbox;
    GtkWidget *field_name_label;
    GtkWidget *field_value_label;
    GtkWidget *field_name;
    GtkWidget *field_value;
    GtkWidget *button_ok, *button_cancel;

    GList *field_list;
    gint i = 0;

    tag_window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_signal_connect(GTK_OBJECT(tag_window), "destroy",
		       GTK_SIGNAL_FUNC(cb_tag_window_destroy), NULL);

    gtk_signal_connect(GTK_OBJECT (tag_window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_tag_window_delete), NULL);
    gtk_container_set_border_width(GTK_CONTAINER(tag_window), 10);
    gtk_window_set_modal(GTK_WINDOW(tag_window), TRUE);

    vbox = gtk_vbox_new(FALSE, 5);

    gtk_container_add(GTK_CONTAINER(tag_window), vbox);

    field_name_label = gtk_label_new(_("Field Name:"));
    gtk_misc_set_alignment(GTK_MISC(field_name_label), 0, 0);
    field_value_label = gtk_label_new(_("Field Value:"));
    gtk_misc_set_alignment(GTK_MISC(field_value_label), 0, 0);

    field_name = gtk_combo_new();
    field_list = NULL;
    for (i = 0; i < COMMON_TAG_COUNT; i ++)
    {
	field_list = g_list_append(field_list, (gpointer)common_tags[i]);
    }
    gtk_combo_set_popdown_strings(GTK_COMBO(field_name), field_list);

    field_value = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(field_value), TRUE);

    hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);

    button_ok = gtk_button_new_with_label("OK");
    gtk_signal_connect(GTK_OBJECT(button_ok), "clicked",
		       GTK_SIGNAL_FUNC(cb_tag_window_ok),
		       NULL);
    button_cancel = gtk_button_new_with_label("Cancel");
    gtk_signal_connect(GTK_OBJECT(button_cancel), "clicked",
		       GTK_SIGNAL_FUNC(cb_tag_window_cancel),
		       NULL);

    gtk_box_pack_start(GTK_BOX(hbox), button_ok,
		       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), button_cancel,
		       TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(vbox), field_name_label,
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), field_name,
		       FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), field_value_label,
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), field_value,
		       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox,
		       FALSE, TRUE, 0);

    mac_tag.tag_window = tag_window;
    mac_tag.field_name_widget = field_name;
    mac_tag.field_value_widget = field_value;

}

void mac_info_box_create()
{

    GtkWidget *window = NULL;
    GtkWidget *location = NULL;
    GtkWidget *metadata = NULL;
    GtkWidget *techinfo = NULL;

    GtkWidget *location_frame, *metadata_frame, *techinfo_frame;
    GtkWidget *vbox, *hbox;
    GtkWidget *bottom_hbox;
    GtkWidget *close_button, *update_button;

    GtkWidget *meta_vbox;

    GtkWidget *buttons_hbox;
    GtkWidget *meta_add, *meta_remove, *meta_edit;
	
    window = gtk_window_new(GTK_WINDOW_DIALOG);
//	gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
    gtk_signal_connect(GTK_OBJECT(window), "destroy",
		       GTK_SIGNAL_FUNC(cb_info_box_destroy), NULL);

    gtk_signal_connect(GTK_OBJECT (window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_info_box_delete), NULL);

    gtk_container_set_border_width(GTK_CONTAINER(window), 10);

    location_frame = gtk_frame_new(_("Location"));
    metadata_frame = gtk_frame_new(_("Metadata"));
    techinfo_frame = gtk_frame_new(_("Technical info"));

    vbox = gtk_vbox_new(FALSE, 5);
    hbox = gtk_hbox_new(FALSE, 5);

    bottom_hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bottom_hbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bottom_hbox), 5);

    gtk_container_add(GTK_CONTAINER(window), vbox);

    gtk_box_pack_start(GTK_BOX(vbox), location_frame, 
		       FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox,
		       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), bottom_hbox,
		       FALSE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(hbox), metadata_frame,
		       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), techinfo_frame,
		       FALSE, FALSE, 0);

    update_button = gtk_button_new_with_label(_("Update File"));
    gtk_box_pack_start(GTK_BOX(bottom_hbox), update_button,
		       TRUE, TRUE, 0);

    gtk_signal_connect_object(GTK_OBJECT(update_button), "clicked",
			      GTK_SIGNAL_FUNC(cb_metadata_update_file),
			      NULL);

    close_button = gtk_button_new_with_label(_("Close"));
    gtk_signal_connect(GTK_OBJECT(close_button), "clicked",
		       GTK_SIGNAL_FUNC(cb_info_box_close), 
		       NULL);
    gtk_box_pack_end(GTK_BOX(bottom_hbox), close_button,
		     TRUE, TRUE, 0);


    location = gtk_entry_new();
    gtk_container_add(GTK_CONTAINER(location_frame), location);
    gtk_entry_set_editable(GTK_ENTRY(location), FALSE);

    meta_vbox = gtk_vbox_new(FALSE, 0);

    buttons_hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttons_hbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttons_hbox), 5);

    meta_add = gtk_button_new_with_label(_("Add"));
    gtk_signal_connect(GTK_OBJECT(meta_add), "clicked",
			      GTK_SIGNAL_FUNC(cb_metadata_add),
			      NULL);
    meta_remove = gtk_button_new_with_label(_("Remove"));
    gtk_signal_connect(GTK_OBJECT(meta_remove), "clicked",
			      GTK_SIGNAL_FUNC(cb_metadata_remove),
			      NULL);
    meta_edit = gtk_button_new_with_label(_("Edit"));
    gtk_signal_connect(GTK_OBJECT(meta_edit), "clicked",
			      GTK_SIGNAL_FUNC(cb_metadata_edit),
			      NULL);

    gtk_box_pack_start(GTK_BOX(buttons_hbox), meta_add, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttons_hbox), meta_edit, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttons_hbox), meta_remove, TRUE, TRUE, 0);


    gtk_box_pack_start(GTK_BOX(meta_vbox), buttons_hbox, FALSE, TRUE, 0);

    gchar *column[2] = {_("Name"), _("Value")};
    metadata = gtk_clist_new_with_titles(2, column);
    gtk_clist_column_titles_passive(GTK_CLIST(metadata));
    gtk_clist_set_column_width(GTK_CLIST(metadata), 0, 70);
    gtk_clist_set_column_width(GTK_CLIST(metadata), 1, 160);

    gtk_box_pack_start(GTK_BOX(meta_vbox), metadata, TRUE, TRUE, 0);

    gtk_container_add(GTK_CONTAINER(metadata_frame), meta_vbox);


    techinfo = gtk_clist_new_with_titles(2, column);
    gtk_clist_column_titles_passive(GTK_CLIST(techinfo));
    gtk_clist_set_column_width(GTK_CLIST(techinfo), 0, 130);
    gtk_clist_set_column_width(GTK_CLIST(techinfo), 1, 100);

    gtk_container_add(GTK_CONTAINER(techinfo_frame), techinfo);

    mac_info.window = window;
    mac_info.location = location;
    mac_info.techinfo = techinfo;
    mac_info.metadata = metadata;
}

void mac_file_info_box(char *filename)
{
    char *title;

    int nRetVal;
    wchar_t *pUTF16Name;

    if (!mac_info.window)
    {
	mac_info_box_create();
    }
    
    if (mac_info.current_filename)
    {
	g_free(mac_info.current_filename);
    }
    mac_info.current_filename = g_strdup(filename);
    
    title = g_strdup_printf(_("File Info - %s"), g_basename(filename));
    gtk_window_set_title(GTK_WINDOW(mac_info.window), title);
    g_free(title);
    
    gtk_entry_set_text(GTK_ENTRY(mac_info.location), filename);

    if (mac_info.pAPEDecompress)
    {
	delete mac_info.pAPEDecompress;
    }

    pUTF16Name = GetUTF16FromANSI(filename);
    mac_info.pAPEDecompress = CreateIAPEDecompress(pUTF16Name, &nRetVal);

    if (mac_info.pAPEDecompress == NULL || nRetVal != ERROR_SUCCESS)
    {
	printf("Oops, something is wrong with the file, error_code = %d\n", nRetVal);
	return;
    }

    CAPETag *apeTag = (CAPETag *)mac_info.pAPEDecompress->GetInfo(APE_INFO_TAG);
    insert_metadata(GTK_CLIST(mac_info.metadata), apeTag);
    insert_techinfo(GTK_CLIST(mac_info.techinfo), mac_info.pAPEDecompress);

    gtk_widget_show_all(mac_info.window);
}

int main(int argc, char *argv[])
{
    if (argc < 2)
    {
	printf("usage: fileinfo filename\n");
	return 0;
    }
    gtk_set_locale();
    gtk_init(&argc, &argv);

    mac_file_info_box(argv[1]);

    gtk_signal_connect(GTK_OBJECT (mac_info.window), "delete_event",
                       (GtkSignalFunc) gtk_main_quit, NULL);

    gtk_main();
    
    return 0;
}

#ifdef __cplusplus
}
#endif
