/* Terminal settings for SGI's xwsh */
/* By Oliver Kiddle (opk@thoth.u-net.com) */

/* This is based on the IRIX 5.3 xwsh and also tested with IRIX 4 xwsh. */
/* IRIX 6 xwsh may have other capabilites */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "setterm.h"

static void set_title(const char *arg)
{
  /* Note that the xterm codes also work */
  printf("\2201.y%s\234",arg);
}

static void set_icon(const char *arg)
{
  printf("\2203.y%s\234",arg);
}

/* Colors can be any X color or of the form #rrggbb */
static void set_half(const char *arg)
{ /* Half intensity, this can't be set with an xwsh option */
  printf("\22010.y%s\234",arg);
}

static void set_bg(const char *arg)
{
  printf("\2205.y%s\234",arg);
}

static void set_bold(const char *arg)
{
  printf("\22011.y%s\234",arg);
}

static void set_cursorbg(const char *arg)
{
  printf("\2209.y%s\234",arg);
}

static void set_cursorfg(const char *arg)
{
  printf("\2208.y%s\234",arg);
}

static void set_fg(const char *arg)
{
  printf("\2204.y%s\234",arg);
}

static void set_selbg(const char *arg)
{
  printf("\2207.y%s\234",arg);
}

static void set_selfg(const char *arg)
{
  printf("\2206.y%s\234",arg);
}

static void set_size(const char *arg)
{
  char *arg2,*end;
  int rows,cols;

  if ((arg2=strchr(arg,'x'))!=(char*)0)
  {
    *arg2++='\0';
    rows=strtol(arg,&end,10);
    if (*end || rows<1) fprintf(stderr, "setterm: rows must be a positive number.\n");
    cols=strtol(arg2,&end,10);
    if (*end || cols<1) fprintf(stderr, "setterm: columns must be a positive number.\n");
    printf("\033[203;%d;%d/y",rows,cols);
  }
  else
  {
    fprintf(stderr,"setterm: %s must be in the format ROWSxCOLS\n",arg);
    exit(2);
  }
}  

static void set_psize(const char *arg)
{
  char *arg2,*end;
  int width,height;

  if ((arg2=strchr(arg,'x'))!=(char*)0)
  {
    *arg2++='\0';
    width=strtol(arg,&end,10);
    if (*end || width<0) fprintf(stderr, "setterm: width must be a positive number.\n");
    height=strtol(arg2,&end,10);
    if (*end || height<0) fprintf(stderr, "setterm: height must be a positive number.\n");
    printf("\033[204;%d;%d/y",width,height);
  }
  else
  {
    fprintf(stderr,"setterm: %s must be in the format WIDTHxHEIGHT\n",arg);
    exit(2);
  }
}  

static void set_position(const char *arg)
{
  char *arg2;

  if ((arg2=strchr(arg,','))!=(char*)0)
  {
    *arg2=';';
    printf("\033[205;%s/y",arg);
  }
  else
  {
    fprintf(stderr,"setterm: %s must be in the format X,Y\n",arg);
    exit(2);
  }
}

static void get_cursory(const char *arg)
{
  char posn[10];
  int i = 0;
  
  fprintf(tty,"\033[6n"); fflush(tty);
  cbreak();
  for (i=0; i<(sizeof(posn)-1) && ((posn[i] = getchar()) != ';'); i++);
  posn[i] = '\0';
  if (i>2) printf("%s\n",posn+2); fflush(stdout);
  cooked();
}

static void get_cursorx(const char *arg)
{
  char posn[14];
  int i = 0;
  
  fprintf(tty,"\033[6n"); fflush(tty);
  cbreak();
  for (i=0; i<(sizeof(posn)-1) && ((posn[i] = getchar()) != 'R'); i++);
  posn[i] = '\0';
  if (i>2) printf("%s\n",posn+5); fflush(stdout);
  cooked();
}

static struct get_color_codes
{
  const char *name;
  char number;
} color_codes[] =
{
  { "fg", '0' },
  { "bg", '1' },
  { "selfg", '2' },
  { "selbg", '3' },
  { "cursorfg", '4' },
  { "cursorbg", '5' },
  { "half", '6' },
  { "bold", '7' },
  { (const char*)0, '\0' }
};

static void get_color(const char *arg)
{
  char color[8];
  int i;
  
  for (i=0;color_codes[i].name;i++)
  {
    if (! strcmp(arg,color_codes[i].name))
    {
      fprintf(tty, "\033[10%cn", color_codes[i].number); fflush(tty);
      break;
    }
  }
  
  if (color_codes[i].name)
  {
    cbreak();
    while (getchar() != 'y');
    for (i=0; i<(sizeof(color)-1) && ((color[i] = getchar()) != '\033'); i++);
    color[i] = '\0';
    if (i>2) printf("%s\n",color); fflush(stdout);
    cooked();
  }
  else
  {
    fprintf(stderr, "setterm: invalid argument %s to option -getcolor\n", arg);
    exit(2);
  }
}

static Name names[]=
{
  "iris-ansi",
  "iris-ansi-net",
/* I've never come across the following but they are in the terminfo database */
  "iris-ansi-24",
  "iris-ansi-24-net",
  "iris-ansi-66",
  "iris-ansi-66-net",
  "iris40",
/* iris-tp (used in single-user mode) has no extra capabilites so is not here */
  (const char*)0
};

static Option opts[]=
{
  { "-title", "Title", (const char*)0, set_title },
  { "-icon", "Title", (const char*)0, set_icon },
  { "-lock", "on", "\033[=6h", (void(*)(const char*))0 },
  { "-lock", "off", "\033[=6l", (void(*)(const char*))0 },
  { "-bell", "visual", "\033[=9h", (void(*)(const char*))0 },
  { "-bell", "audio", "\033[=9l", (void(*)(const char*))0 },
  { "-init", "textport", "\033[2/y", (void(*)(const char*))0 },
  { "-init", "history", "\033[14/y", (void(*)(const char*))0 },
  { "-bindkey", "init", "\033[15/y", (void(*)(const char*))0 },
  { "-bindkey", "reset", "\033[16/y", (void(*)(const char*))0 },
  { "-push", (const char*)0, "\033[3/y", (void(*)(const char*))0 },
  { "-pop", (const char*)0, "\033[4/y", (void(*)(const char*))0 },
  { "-linewrap", "on", "\033[?7h", (void(*)(const char*))0 },
  { "-linewrap", "off", "\033[?7l", (void(*)(const char*))0 },
  { "-inversescreen", "on", "\033[?5h", (void(*)(const char*))0 },
  { "-inversescreen", "off", "\033[?5l", (void(*)(const char*))0 },
  { "-size", "ROWSxCOLS", (const char*)0, set_size },
  { "-pixelsize", "WIDTHxHEIGHT", (const char*)0, set_psize },
  { "-position", "X,Y", (const char*)0, set_position },
  { "-half", "color", (const char*)0, set_half },
  /* The following options are equivalent to the parameters to xwsh */
  { "-bg", "color", (const char*)0, set_bg },
  { "-bold", "color", (const char*)0, set_bold },
  { "-cursorbg", "color", (const char*)0, set_cursorbg },
  { "-cursorfg", "color", (const char*)0, set_cursorfg },
  { "-fg", "color", (const char*)0, set_fg },
  { "-selbg", "color", (const char*)0, set_selbg },
  { "-selfg", "color", (const char*)0, set_selfg },
  /* The following return information about the terminal */
  { "-getcursory", (const char*)0, (const char*)0, get_cursory },
  { "-getcursorx", (const char*)0, (const char*)0, get_cursorx },
  { "-getcolor", "bg|fg|selbg|selfg|cursorfg|cursorbg|half|bold", (const char*)0, get_color }, 
  { (const char*)0, (const char*)0, (const char*)0, (void(*)(const char*))0 },
};

Terminal xwsh = { names, opts };
