/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-19)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */
#include "glibctest.h"
#include "test_iswc.h"
#include "lcctype.h"
#include "charmap.h"
#include "boolean.h"

#define UNICODE_LEN 4

extern char *predefined_class[];
extern char *predefined_conv[];

typedef struct _test_apis_wc_data {
  char *symbol;
  char *code;
  char *value;
} test_apis_wc_data ;

static boolean
ispredefinedclass ( const char * classname )
{
  int i;
  boolean ret = false;

  for (i = 0; predefined_class[i] ; i++)
    {
      if (stringeq (classname, predefined_class[i]))
      {
	ret = true;
	break;
      }
    }

  return (ret);
}

static boolean
ispredefinedconv ( const char * classname )
{
  int i;
  boolean ret = false;

  for (i = 0; predefined_conv[i] ; i++)
    {
      if (stringeq (classname, predefined_conv[i]))
      {
	ret = true;
	break;
      }
    }

  return (ret);
}

static void
xstr2unicode ( const char *xstr, int * valuep)
{
  *valuep = (int) strtoul (xstr, NULL, 16);
}

static void
extract_unicode_value ( const char *buf, char *code )
{
  int i = 0, j = 0;

  while (buf[i] != '<')
    i++;
  i += 2;                       /* skip '<' and 'u(U)' */
  while (buf[i] != '>')
    {
      code[j++] = buf[i++];
    }
  code[j] = (char) NULL;
}

static int
symbol2unicode ( const char * symbol )
{
  char *unicode;
  int codevalue;

  unicode = (char *) mem_calloc (UNICODE_LEN + 1, sizeof (char));
  extract_unicode_value (symbol, unicode);
  xstr2unicode (unicode, &codevalue);
  mem_free ((void *) unicode);

  return codevalue;
}

static boolean
test_wctype_search ( lc_ctype *lcp,
		     char *classname,
		     char *symbol )
{
  boolean ret;
  boolean alnumflag, alphaflag;

  if (stringeq ("alpha", classname))
    {
      if (lc_ctype_defined_class (lcp, "alpha") == true)
	{
	  ret = lc_ctype_search_wctype (lcp, classname, symbol);
	}
      else
	{
	  ret = lc_ctype_search_wctype (lcp, "upper", symbol);
	  ret |= lc_ctype_search_wctype (lcp, "lower", symbol);
	}
    }
  else if (stringeq ("alnum", classname))
    {
      if (lc_ctype_defined_class (lcp, "alnum") == true)
	{
	  ret = lc_ctype_search_wctype (lcp, classname, symbol);
	}
      else
	{
	  if (lc_ctype_defined_class (lcp, "alpha") == true)
	    {
	      ret = lc_ctype_search_wctype (lcp, "alpha", symbol);
	    }
	  else
	    {
	      ret = lc_ctype_search_wctype (lcp, "upper", symbol);
	      ret |= lc_ctype_search_wctype (lcp, "lower", symbol);
	      fprintf (stderr, "sym = [%s], ret = %d\n", symbol, ret);
	    }
	  ret |= lc_ctype_search_wctype (lcp, "digit", symbol);
	}
    }
  else if (stringeq ("print", classname))
    {
      if (lc_ctype_defined_class (lcp, "print") == true)
	{
	  ret = lc_ctype_search_wctype (lcp, "print", symbol);
	}
      else
	{
	  ret = lc_ctype_search_wctype (lcp, "upper", symbol);
	  ret |= lc_ctype_search_wctype (lcp, "lower", symbol);
	  ret |= lc_ctype_search_wctype (lcp, "digit", symbol);
	  ret |= lc_ctype_search_wctype (lcp, "punct", symbol);
	  ret |= lc_ctype_search_wctype (lcp, "space", symbol);
	}
    }
  else
    {
      ret = lc_ctype_search_wctype (lcp, classname, symbol);
    }

  return (ret);
}

static boolean
test_wctype_func ( api_table *tp,
		   lc_ctype *lcp,
		   char *classname,
		   test_apis_wc_data *ptr,
		   boolean verbose_flag )
{
  int j;
  boolean result;
  size_t tmpsize;
  size_t padding = 15;
  boolean tmpbool;
  answer *ansp;
  wc_answer *wc_ansp;

  ansp = (answer *) mem_alloc (sizeof (answer));
  wc_ansp = (wc_answer *) mem_alloc (sizeof (wc_answer));

  ansp->api_name = (char *) mem_alloc (stringlen (classname) + 3);
  sprintf (ansp->api_name, "isw%s", classname);

  stringpcpy (&(ansp->param[0]), ptr->value);

  tmpbool = test_wctype_search (lcp, classname, ptr->symbol);
  if (tmpbool == true)
    {
      stringpcpy (&(ansp->result), "!0");
    }
  else
    {
      stringpcpy (&(ansp->result), "0");
    }

  stringpcpy (&(ansp->param[1]), "-");
  stringpcpy (&(ansp->param[2]), "-");

  convert_mbswcs (&(wc_ansp->api_name), ansp->api_name);

  wc_ansp->param[0] = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->param[0][0] = symbol2unicode (ptr->symbol);
  wc_ansp->param[0][1] = (wchar_t) NULL;

  for (j = 1; j < PARAM_NUM; j++)
    {
      convert_mbswcs (&(wc_ansp->param[j]), ansp->param[j]);
    }

  convert_mbswcs (&(wc_ansp->result), ansp->result);

  result = api_table_select (tp, ansp, wc_ansp, false, verbose_flag);

  free_ans (ansp, wc_ansp);

  return (result);
}

/* REVISED */
static boolean
test_wctype_func_w ( api_table *tp,
		     lc_ctype *lcp,
		     char *classname,
		     test_apis_wc_data *ptr,
		     boolean verbose_flag )
{
  int j;
  boolean result;
  size_t tmpsize;
  size_t padding = 15;
  boolean tmpbool;
  answer *ansp;
  wc_answer *wc_ansp;

  ansp = (answer *) mem_alloc (sizeof (answer));
  wc_ansp = (wc_answer *) mem_alloc (sizeof (wc_answer));

  stringpcpy (&(ansp->api_name), "iswctype");

  stringpcpy (&(ansp->param[0]), ptr->value);

  tmpbool = test_wctype_search (lcp, classname, ptr->symbol);
  if (tmpbool == true)
    {
      stringpcpy (&(ansp->result), "!0");
    }
  else
    {
      stringpcpy (&(ansp->result), "0");
    }

  stringpcpy (&(ansp->param[1]), classname);
  stringpcpy (&(ansp->param[2]), "-");

  convert_mbswcs (&(wc_ansp->api_name), ansp->api_name);

  wc_ansp->param[0] = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->param[0][0] = symbol2unicode (ptr->symbol);
  wc_ansp->param[0][1] = (wchar_t) NULL;

  for (j = 1; j < PARAM_NUM; j++)
    {
      convert_mbswcs (&(wc_ansp->param[j]), ansp->param[j]);
    }

  convert_mbswcs (&(wc_ansp->result), ansp->result);

  result = api_table_select (tp, ansp, wc_ansp, false, verbose_flag);

  free_ans (ansp, wc_ansp);

  return (result);
}

/* REVISED */
static boolean
test_wctrans_func ( api_table *tp,
		    lc_ctype *lcp,
		    charmap_hash *cmhash,
		    char *classname,
		    test_apis_wc_data *ptr,
		    boolean verbose_flag )
{
  int j;
  boolean result;
  size_t tmpsize;
  size_t padding = 15;
  answer *ansp;
  wc_answer *wc_ansp;
  char *unicode;
  int codevalue;
  char value[MB_LEN_MAX+1];
  char code[MB_LEN_MAX+1];
  char *converted_sym;

  ansp = (answer *) mem_alloc (sizeof (answer));
  wc_ansp = (wc_answer *) mem_alloc (sizeof (wc_answer));

  ansp->api_name = (char *) mem_alloc (stringlen (classname) + 3);
  sprintf (ansp->api_name, "tow%s", classname+2);

  stringpcpy (&(ansp->param[0]), ptr->value);
  stringpcpy (&(ansp->param[1]), "-");
  stringpcpy (&(ansp->param[2]), "-");

  converted_sym = lc_ctype_search_toxxx (lcp, classname, ptr->symbol);
  if (converted_sym == (char *)NULL)
    {
      return (false);
    }
  charmap_hash_resolv (cmhash, converted_sym, code);
  xstr2ul (code, value);

  stringpcpy (&(ansp->result), value);

  convert_mbswcs (&(wc_ansp->api_name), ansp->api_name);

  wc_ansp->param[0] = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->param[0][0] = symbol2unicode (ptr->symbol);
  wc_ansp->param[0][1] = (wchar_t) NULL;

  for (j = 1; j < PARAM_NUM; j++)
    {
      convert_mbswcs (&(wc_ansp->param[j]), ansp->param[j]);
    }

  wc_ansp->result = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->result[0] = symbol2unicode (converted_sym);
  wc_ansp->result[1] = (wchar_t) NULL;

  result = api_table_select (tp, ansp, wc_ansp, false, verbose_flag);

  free_ans (ansp, wc_ansp);

  return (result);
}

/* REVISED */
static boolean
test_wctrans_func_w ( api_table *tp,
		      lc_ctype *lcp,
		      charmap_hash *cmhash,
		      char *classname,
		      test_apis_wc_data *ptr,
		      boolean verbose_flag )
{
  int j;
  boolean result;
  size_t tmpsize;
  size_t padding = 15;
  answer *ansp;
  wc_answer *wc_ansp;
  char *unicode;
  int codevalue;
  char value[MB_LEN_MAX+1];
  char code[MB_LEN_MAX+1];
  char *converted_sym;
  ansp = (answer *) mem_alloc (sizeof (answer));
  wc_ansp = (wc_answer *) mem_alloc (sizeof (wc_answer));

  stringpcpy (&(ansp->api_name), "towctrans");

  stringpcpy (&(ansp->param[0]), ptr->value);
  stringpcpy (&(ansp->param[1]), classname);
  stringpcpy (&(ansp->param[2]), "-");

  converted_sym = lc_ctype_search_toxxx (lcp, classname, ptr->symbol);
  if (converted_sym == (char *)NULL)
    {
      return (false);
    }
  charmap_hash_resolv (cmhash, converted_sym, code);
  xstr2ul (code, value);

  stringpcpy (&(ansp->result), value);

  convert_mbswcs (&(wc_ansp->api_name), ansp->api_name);

  wc_ansp->param[0] = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->param[0][0] = symbol2unicode (ptr->symbol);
  wc_ansp->param[0][1] = (wchar_t) NULL;

  for (j = 1; j < PARAM_NUM; j++)
    {
      convert_mbswcs (&(wc_ansp->param[j]), ansp->param[j]);
    }

  wc_ansp->result = (wchar_t *) mem_alloc (sizeof (wchar_t) * 2);
  wc_ansp->result[0] = symbol2unicode (converted_sym);
  wc_ansp->result[1] = (wchar_t) NULL;

  result = api_table_select (tp, ansp, wc_ansp, false, verbose_flag);

  free_ans (ansp, wc_ansp);

  return (result);
}

/* REVISED */
static boolean
test_wctype ( api_table *tp,
	      lc_ctype *lcp,
	      boolean verbose_flag,
	      test_apis_wc_data *ptr )
{
  int i;
  wctype_t desc;
  boolean result;
  boolean tmpresult;

  result = true;

  /* iswalpha, ..., iswprint */
  for (i = 0; predefined_class[i] ; i++)
    {
      result &=
	test_wctype_func (tp, lcp, predefined_class[i], ptr, verbose_flag);
    }

  /* iswctype */
  for (i = 0; predefined_class[i] ; i++)
    {
      desc = wctype (predefined_class[i]);
      if (desc)
	{
	  result &=
	    test_wctype_func_w (tp, lcp, predefined_class[i], ptr, verbose_flag);
	}
      else
	{
	  /*
	     fprintf(stderr, "[ERROR] wctype: '%s' is not defined in system.\n", predefined_class[i]);
	   */
	}
    }

  for (i = 0; i < lcp->userdefined_class_num; i++)
    {
      desc = wctype (lcp->userdefined_class[i]);
      if (desc)
	{
	  result &=
	    test_wctype_func_w (tp, lcp, lcp->userdefined_class[i], ptr, verbose_flag);
	}
      else
	{
	  /*
	     fprintf(stderr, "[ERROR] wctype: '%s' is not defined in system.\n", lcp->userdefined_class[i]);
	   */
	}
    }

  return (result);
}

/* REVISED */
static boolean
test_wctrans ( api_table *tp,
	       lc_ctype *lcp,
	       charmap_hash *cmhash,
	       boolean verbose_flag,
	       test_apis_wc_data *ptr )
{
  int i;
  wctrans_t desc;
  boolean result;

  result = true;

  /* towlower, towupper */
  for (i = 0; predefined_conv[i] ; i++)
    {
      result &=
	test_wctrans_func (tp, lcp, cmhash, predefined_conv[i], ptr, verbose_flag);
    }

  /* wctrans */
  for (i = 0; predefined_conv[i] ; i++)
    {
      desc = wctrans (predefined_conv[i]);
      if (desc)
	{
	  result &=
	    test_wctrans_func_w (tp, lcp, cmhash, predefined_conv[i], ptr, verbose_flag);
	}
      else
	{
	  /*
	     fprintf(stderr, "[ERROR] wctrans: '%s' is not defined in system.\n", predefined_conv[i]);
	   */
	}
    }

  for (i = 0; i < lcp->userdefined_conv_num; i++)
    {
      desc = wctrans (lcp->userdefined_conv[i]);
      if (desc)
	{
	  result &=
	    test_wctrans_func_w (tp, lcp, cmhash, lcp->userdefined_conv[i], ptr, verbose_flag);
	}
      else
	{
	  /*
	     fprintf(stderr, "[ERROR] wctrans: '%s' is not defined in system.\n", lcp->userdefined_conv[i]);
	   */
	}
    }

  return (result);
}

/* REVISED */
int
test_apis_wc ( api_table *tp,
	       lc_ctype *lcp,
	       charmap *cmp,
	       charmap_hash *cmhash,
	       boolean verbose_flag )
{
  int i, num = 0;
  int j, k;
  boolean result;
  int error_cnt = 0;
  size_t tmpsize;
  test_apis_wc_data *ptr;
  int mb_cur_max;
  int total_num;
  char *converted;

  mb_cur_max = charmap_get_mb_cur_max (cmp);
  total_num = charmap_get_total_num (cmp);

  result = true;
  for (i = 0; i < total_num; i++)
    {
      num++;
      display_progress (num, total_num);
      ptr = (test_apis_wc_data *) mem_alloc (sizeof (test_apis_wc_data));
      charmap_get_data (cmp, i, &(ptr->symbol), &(ptr->code));

      tmpsize = mb_cur_max + 1;
      ptr->value = (char *) mem_calloc (tmpsize, sizeof (char));
      xstr2ul (ptr->code, ptr->value);

      display_string (ptr->symbol);
      display_string ("\n");
      result &= test_wctype (tp, lcp, verbose_flag, ptr);
      result &= test_wctrans (tp, lcp, cmhash, verbose_flag, ptr);

      error_cnt += (result == false) ? 1 : 0;
      /*
         if ( func(&(sd.input[i])) == SYMBOL_NOT_FOUND ) {
         fprintf(stderr, "! %s does not appear in locale.\n", sd.input[i].symbol);
         }
       */
      mem_free ((void *)ptr->symbol);
      mem_free ((void *)ptr->code);
      mem_free ((void *)ptr->value);
      mem_free ((void *)ptr);
    }

  result_report (num, error_cnt, "");

  return (0);
}
