/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-19)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */
#include "hash.h"

hashtable *
hashtable_init ( const int total_num, void (*data_display) ( void * ), void (*data_free) ( void * ) )
{
  int i;
  hashtable *tbl;

  tbl = (hashtable *) malloc (sizeof (hashtable));
  tbl->total_num = total_num;
  tbl->table_max = find_prime (tbl->total_num);
  tbl->tablep =
    (hashtable_data *) mem_alloc (tbl->table_max * sizeof (hashtable_data));
  tbl->data_display = data_display;
  tbl->data_free = data_free;

  for (i = 0; i < tbl->table_max; i++)
    {
      tbl->tablep[i].key = (char *) NULL;
      tbl->tablep[i].datap = (void *) NULL;
      tbl->tablep[i].nextp = (hashtable_data *) NULL;
    }

  return (tbl);
}

static int
hashkey ( const hashtable * tbl, const char *key )
{
  int value = 0;
  int tmp = 1;

  while (*key != (char) NULL)
    {
      tmp = (tmp * 255) % tbl->table_max;
      value += (((*key) * tmp) % tbl->table_max);
      key++;
    }

  return (value % (tbl->table_max));
}

void *
hashtable_search ( const hashtable * tbl, const char *key )
{
  hashtable_data *ptr;
  int keyvalue;

  keyvalue = hashkey (tbl, key);

  ptr = &(tbl->tablep[keyvalue]);
  while (ptr != NULL)
    {
      if ( ptr->key && stringeq (key, ptr->key))
	{
	  return (ptr->datap);
	}
      ptr = ptr->nextp;
    }

  return ((void *) NULL);
}

void
hashtable_insert ( const hashtable * tbl, const char *key, void *datap )
{
  int keyvalue;
  hashtable_data *ptr, *save;

  keyvalue = hashkey (tbl, key);

  ptr = &(tbl->tablep[keyvalue]);
  if (ptr->key)
    {
      save = ptr;
      while (save->nextp != NULL)
	{
	  save = save->nextp;
	}
      ptr = (hashtable_data *) mem_alloc (sizeof (hashtable_data));
      save->nextp = ptr;
    }
  stringpcpy (&(ptr->key), key);
  ptr->datap = datap;
  ptr->nextp = (hashtable_data *) NULL;
}

void
hashtable_free ( hashtable * tbl )
{
  int i;
  hashtable_data *ptr, *save;

  for (i = 0; i < tbl->table_max; i++)
    {
      ptr = &(tbl->tablep[i]);
      /*      fprintf (stderr, "ptr = %x key[%s] data[%x] next[%x]\n", ptr, ptr->key, ptr->datap, ptr->nextp); */
      save = ptr->nextp;
      mem_free (ptr->key);
      if (tbl->data_free)
	{
	  tbl->data_free (ptr->datap);
	}
      mem_free (ptr->datap);
      while (NULL != (ptr = save))
	{
	  /*	  fprintf (stderr, "ptr1 = %x key[%s] data[%x] next[%x]\n", ptr, ptr->key, ptr->datap, ptr->nextp); */
	  save = ptr->nextp;
	  mem_free (ptr->key);
	  if (tbl->data_free)
	    {
	      tbl->data_free (ptr->datap);
	    }
	  mem_free (ptr->datap);
	  mem_free (ptr);
	}
    }

  mem_free (tbl->tablep);
  mem_free (tbl);
}

void
hashtable_display ( const hashtable * tbl )
{
  int i, j;
  hashtable_data *ptr;

  for (i = 0; i < tbl->table_max; i++)
    {
      ptr = &(tbl->tablep[i]);
      if (ptr->key)
	{
	  printf ("key[%s]", ptr->key);
	  tbl->data_display (ptr->datap);
	  printf ("\n");
	  j = 1;
	  ptr = ptr->nextp;
	  while (ptr != NULL)
	    {
	      printf ("+- %d key[%s]", j, ptr->key);
	      tbl->data_display (ptr->datap);
	      printf ("\n");
	      ptr = ptr->nextp;
	      j++;
	    }
	}
      else
	{
	  printf ("\n");
	}
    }
}

int
end_of_hashtable (const hashtable * tbl, const int idx)
{
  return (idx == tbl->table_max ? 1 : 0);
}
