/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-12)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */
#include "glibctest.h"
#include "repertoiremap.h"

void
repertoiremap_data_display ( void *datap )
{
  if (datap)
    {
      printf ("sym[%s] unicode [%s]",
	      ((repertoiremapdata *) datap)->symbol,
	      ((repertoiremapdata *) datap)->code);
    }
}

void
repertoiremap_data_free ( void *datap )
{
  if (datap)
    {
      mem_free ((void *)((repertoiremapdata *) datap)->symbol);
      mem_free ((void *)((repertoiremapdata *) datap)->code);
    }
}

repertoiremap *
repertoiremap_init ( FILE *fp )
{
  int i, num, flag, total_num;
  int start, end;
  char tmpnum[10];
  fpos_t pos;
  char *bufp;
  repertoiremap *rp;

  rp = (repertoiremap *) mem_alloc (sizeof (repertoiremap));
  rp->fp = fp;

  flag = 0;
  rp->comment_char = '#';
  rp->escape_char = '\\';

  while (NULL != get_oneline (rp->fp, &bufp))
    {
      i=0;
      if (iscommentline (bufp, rp->comment_char))
	{
	  fgetpos (rp->fp, &pos);
	  mem_free ((void *)bufp);
	  continue;
	}

      if (stringeqn (bufp, "escape_char", 11))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "escape_char" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  rp->escape_char = *(bufp+i);
	  flag++;
	}
      else if (stringeqn (bufp, "comment_char", 12))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "comment_char" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  rp->comment_char = *(bufp+i);
	  flag++;
	}
      else if (stringeqn (bufp, "<", 1))
	{
	  mem_free ((void *)bufp);
	  break;
	}

      fgetpos (rp->fp, &pos);
      mem_free ((void *)bufp);
    }

  fsetpos (rp->fp, &pos);

  /*
   * Counting Number of Entries
   */

  total_num = 0;
  while (NULL != get_oneline (rp->fp, &bufp))
    {
      i=0;
      if (iscommentline (bufp, rp->comment_char))
	{
	  mem_free ((void *)bufp);
	  continue;
	}
      total_num++;
      mem_free ((void *)bufp);
    }

  rp->hashtable = hashtable_init (total_num, repertoiremap_data_display, repertoiremap_data_free);

  fsetpos (rp->fp, &pos);

  return (rp);
}

void
extract_unicode (char *buf, char *code)
{
  int i = 0, j = 0;

  while (buf[i] != '<')
    i++;
  i += 2;			/* skip '<' and 'u(U)' */
  while (buf[i] != '>')
    {
      code[j++] = buf[i++];
    }
  code[j] = (char) NULL;
}

void
xstr2unicode (char *xstr, unicode_t * valuep)
{
  unicode_t value = 0;
  int i, n;

  *valuep = (int) strtoul (xstr, NULL, 16);
}

static boolean
get_unicode_data (repertoiremap * rp, char *bufp, char **tmpsymp,
		  char *tmpunicode)
{
  int i, j, start, end;
  i = 0;

  while (bufp[i] != '<')
    {
      i++;
    }

  /* set symbol */
  start = i;
  do
    {
      if (bufp[i] == rp->escape_char)
	{
	  i += 2;
	}
      else
	{
	  i++;
	}
    }
  while (bufp[i] != '>');
  end = i;

  if (end == (start + 1)) /* means "<>" */
    {
      return (false);
    }

  *tmpsymp = (char *) mem_calloc (end - start + 2, sizeof (char));

  for (i=start, j=0; i <= end; i++, j++)
    {
      *((*tmpsymp)+j) = bufp[i];
    }
  *((*tmpsymp)+j) = (char) NULL;

  /* set ulvalue from charmap */
  extract_unicode ((bufp + end + 1), tmpunicode);

  return (true);
}

static void
insert_repertoiremap (repertoiremap * rp, char *symbol, char *unicode)
{
  repertoiremapdata *ptr;
  char *key;

  ptr = (repertoiremapdata *) mem_alloc (sizeof (repertoiremapdata));
  stringpcpy (&(ptr->symbol), symbol);
  stringpcpy (&(ptr->code), unicode);

  extract_realkey (&key, symbol, rp->escape_char);
  hashtable_insert (rp->hashtable, key, ptr);

  mem_free ((void *) key);
}

void
resolv_unicode (const repertoiremap * rp, const char *keysym, unicode_t * valuep)
{
  int hashkey;
  repertoiremapdata *ptr;
  char *key;

  extract_realkey (&key, keysym, rp->escape_char);
  ptr = hashtable_search (rp->hashtable, key);

  if (ptr == NULL)
    {
      *valuep = -1;
    }
  else
    {
      xstr2unicode (ptr->code, valuep);
    }
  mem_free ((void *) key);
}

void
repertoiremap_create (repertoiremap * rp)
{
  char *tmpsym;
  char *tmpcode;
  char *bufp;
  int i = 0;

  while (NULL != get_oneline (rp->fp, &bufp))
    {
      if (iscommentline (bufp, rp->comment_char))
	{
	  mem_free ((void *)bufp);
	  continue;
	}

      tmpcode = (char *) mem_calloc (UNICODE_LEN + 1, sizeof (char));
      if (get_unicode_data (rp, bufp, &tmpsym, tmpcode))
	{
	  insert_repertoiremap (rp, tmpsym, tmpcode);
	  mem_free ((void *) tmpsym);
      	}
      else
	{
	  fprintf (stderr, "Error: Invalid line %s\n", bufp);
	}
      mem_free ((void *) tmpcode);
      mem_free ((void *)bufp);
    }

  fclose (rp->fp);
}

void
repertoiremap_display ( repertoiremap *rp )
{
  hashtable_display (rp->hashtable);
  printf ("\n");
}

void
repertoiremap_free ( repertoiremap *rp )
{
  if (rp)
    {
      hashtable_free (rp->hashtable);
      mem_free ((void *) rp);
    }
}
