# -*- coding: UTF-8 -*-
#
#  kawari.py - a "華和梨" compatible Shiori module for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2004 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
#  $Id: kawari.py,v 1.23.4.1 2005/07/02 05:00:43 shy Exp $
#

import base64
import os
import re
import string
import time
import random
import sys
import StringIO

import codecs

builtin_open = open

###   READER   ###
charset = 'Shift_JIS' # default

def read_dict(path, debug=0):
    global charset
    file = builtin_open(path)
    lineno = 0
    buffer = []
    for line in file:
        lineno = lineno + 1
        position = (lineno, path)
        if line[:9] == "!KAWA0000":
            line = decrypt(line[9:])
        if not line.strip() or \
           line[0] =="#" or \
           line[:6] == ":crypt" or \
           line[:9] == ":endcrypt":
            continue
        pos = line.find(":")
        if pos < 0:
            if debug & 4:
                print "kawari.py: syntax error at line %d in %s:" % position
                print line.strip()
            continue
        entries = line[:pos].strip()
        phrases = line[pos+1:].strip()
        if not entries:
            if debug & 4:
                print "kawari.py: syntax error at line %d in %s:" % position
                print line.strip()
            continue
        if not phrases:
            continue
        if entries == "locale":
            try:
                codecs.lookup(phrases)
            except:
                print "kawari.py: unsupported charset %s" % phrases
            else:
                charset = phrases
        else:
            entries = unicode(entries, charset, 'ignore')
            phrases = unicode(phrases, charset, 'ignore')
        buffer.append((entries, phrases, position))
    return buffer

def decrypt(data):
    buffer = []
    for c in base64.decodestring(data):
        buffer.append(chr(ord(c) ^ 0xcc))
    return string.join(buffer, '')

def encrypt(data):
    buffer = []
    for c in data:
        buffer.append(chr(ord(c) ^ 0xcc))
    line = "!KAWA0000" + base64.encodestring(string.join(buffer, ''))
    return line.replace("\n", "")

def create_dict(buffer, debug=0):
    rdict = {} # rules
    kdict = {} # keywords
    for entries, phrases, position in buffer:
        parsed_entries = parse_entries(entries)
        parsed_phrases = parse_phrases(phrases)
        if parsed_phrases is None:
            if debug & 4:
                print "kawari.py: syntax error at line %d in %s:" % position
                print phrases.strip()
            continue
        if entries[0] == "[":
            add_phrases(kdict, tuple(parsed_entries), parsed_phrases)
            continue
        for entry in parsed_entries:
            add_phrases(rdict, entry, parsed_phrases)
    return rdict, kdict

def add_phrases(dict, entry, phrases):
    if not dict.has_key(entry):
        dict[entry] = []
    for phrase in phrases:
        if phrase:
            phrase[0]  = phrase[0].lstrip()
            phrase[-1] = phrase[-1].rstrip()
        dict[entry].append(tuple(phrase))

def parse_entries(data):
    if data[0] == "[" and data[-1] == "]":
        entries = []
        i = 0
        j = len(data)
        while i < j:
            if data[i] == '"':
                i, text = parse_quotes(data, i)
                entries.append(text)
            else:
                i += 1
    else:
        entries = [s.strip() for s in data.split(",")]
    return entries

re_comma = re.compile(",")

def parse_phrases(data):
    buffer = []
    i = 0
    j = len(data)
    while i < j:
        if data[i] == ',':
            i += 1
        i, phrase = parse(data, i, re_comma)
        if phrase:
            buffer.append(phrase)
    return buffer

def parse(data, start, stop_pattern=None):
    buffer = []
    i = start
    j = len(data)
    while i < j:
        if stop_pattern and stop_pattern.match(data, i):
            break
        elif data[i] == '"':
            i, text = parse_quotes(data, i)
            buffer.append('"%s"' % text)
        elif data[i:i+2] == '${':
            i, text = parse_reference(data, i)
            buffer.append(text)
        elif data[i:i+2] == '$(':
            i, text = parse_inline_script(data, i)
            buffer.append(text)
        elif data[i] == "$":
            buffer.append(data[i])
            i += 1
        elif data[i] == ";":
            buffer.append(data[i])
            i += 1
        else:
            i, text = parse_text(data, i, stop_pattern)
            buffer.append(text)
    if buffer:
        if is_space(buffer[0]):
            del buffer[0]
        else:
            buffer[0] = buffer[0].lstrip()
    if buffer:
        if is_space(buffer[-1]):
            del buffer[-1]
        else:
            buffer[-1] = buffer[-1].rstrip()
    return i, buffer

def parse_quotes(data, start):
    buffer = []
    i = start + 1
    j = len(data)
    while i < j:
        if data[i] == '"':
            i += 1
            break
        elif data[i] == '\\':
            i += 1
            if i < j and data[i] == '"':
                buffer.append('\\' + data[i])
                i += 1
            else:
                buffer.append('\\')
        else:
            buffer.append(data[i])
            i += 1
    return i, string.join(buffer, '')

def parse_reference(data, start):
    i = start
    j = len(data)
    while i < j:
        if data[i] == '}':
            i += 1
            break
        else:
            i += 1
    return i, data[start:i]

def parse_inline_script(data, start):
    buffer = ['$']
    i = start + 1
    j = len(data)
    npar = 0
    while i < j:
        #begin specification bug work-around (1/3)
        if data[i] == ')':
            buffer.append(data[i])
            i += 1
            break
        #end
        if data[i] == '"':
            i, text = parse_quotes(data, i)
            buffer.append('"%s"' % text)
        elif data[i:i+2] == '${':
            i, text = parse_reference(data, i)
            buffer.append(text)
        elif data[i:i+2] == '$(':
            i, text = parse_inline_script(data, i)
            buffer.append(text)
        else:
            if data[i] == '(':
                npar = npar + 1
            elif data[i] == ')':
                npar = npar - 1
            buffer.append(data[i])
            i += 1
        if npar == 0:
            break
    return i, string.join(buffer, '')

def is_space(s):
    return not s.strip()

def parse_text(data, start, stop_pattern=None):
    condition = is_space(data[start])
    i = start
    j = len(data)
    while i < j:
        if stop_pattern and stop_pattern.match(data, i):
            break
        elif data[i] in ['$', '"']:
            break
        elif is_space(data[i]) != condition:
            break
        elif data[i] == ";":
            if i == start:
                i += 1
            break
        else:
            i += 1
    return i, data[start:i]

def read_local_script(path):
    file = builtin_open(path)
    rdict = {}
    kdict = {}
    while 1:
        line = file.readline()
        if not line:
            break
        if line[0] == '#':
            rdict[unicode(line.strip(), charset, 'ignore')] = \
                 [unicode(file.readline().strip(), charset, 'ignore')]
    return rdict, kdict

###   KAWARI   ###

class Kawari:
    MAXDEPTH = 30
    def __init__(self, prefix, pathlist, rdictlist, kdictlist, debug=0):
        self.prefix = prefix
        self.pathlist = pathlist
        self.rdictlist = rdictlist
        self.kdictlist = kdictlist
        self.system_entries = {}
        self.debug = debug
        self.expr_parser = ExprParser()
        #begin specification bug work-around (2/3)
        self.expr_parser.kawari = self
        #end
        self.otherghost = {}
        self.get_system_entry("OnLoad")
    def finalize(self):
        self.get_system_entry("OnUnload")
    # SHIORI/1.0 API
    def getaistringrandom(self):
        return self.get("sentence").encode(charset, 'ignore')
    def getaistringfromtargetword(self, word):
        word = unicode(word, charset, 'ignore')
        return self.get("sentence").encode(charset, 'ignore') # XXX
    def getdms(self):
        return self.getword("dms")
    def getword(self, type):
        for delimiter in [".", "-"]:
            name = "compatible" + delimiter + type
            script = self.get(name, default=None)
            if script is not None:
                return script.strip().encode(charset, 'ignore')
        return ""
    def getstring(self, name):
        name = unicode(name, charset, 'ignore')
        return self.get("resource.%s" % name).encode(charset, 'ignore')
    # SHIORI/2.2 API
    def get_event_response(self, event,
                           ref0=None, ref1=None, ref2=None, ref3=None,
                           ref4=None, ref5=None, ref6=None, ref7=None): ## FIXME
        def proc(ref):
            if ref != None:
                ref = unicode(str(ref), charset, 'ignore')
            return ref
        ref = [proc(ref) for ref in [ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7]]
        for i in range(8):
            if ref[i] is not None:
                value = ref[i]
                self.system_entries["system.Reference%d" % i] = value
                self.system_entries["system-Reference%d" % i] = value
        script = None
        if event == "OnCommunicate":
            self.system_entries["system.Sender"] = ref[0]
            self.system_entries["system-Sender"] = ref[0]
            self.system_entries["system.Sender.Path"] = 'local' # (local/unknown/external)
            self.system_entries["system-Sender.Path"] = 'local' # (local/unknown/external)
            if not self.system_entries.has_key("system.Age"):
                self.system_entries["system.Age"] = '0'
                self.system_entries["system-Age"] = '0'
            self.system_entries["system.Sentence"] = ref[1]
            self.system_entries["system-Sentence"] = ref[1]
            if self.otherghost.has_key(ref[0]):
                s0, s1 = self.otherghost[ref[0]]
                self.system_entries["system.Surface"] = string.join((str(s0), str(s1)), ',')
                self.system_entries["system-Surface"] = string.join((str(s0), str(s1)), ',')
            script = self.get_system_entry("OnResponse")
            if not script:
                for dict in self.kdictlist:
                    for entry in dict.keys():
                        for word in entry:
                            if ref[1].find(word) < 0:
                                break
                        else:
                            script = self.expand(random.choice(dict[entry]))
                            break
            if not script:
                script = self.get_system_entry("OnResponseUnknown")
            if script is not None:
                script = script.strip()
        else:
            for delimiter in [".", "-"]:
                name = "event" + delimiter + event
                script = self.get(name, default=None)
                if script is not None:
                    script = script.strip()
                    break
        if script != None:
            script = script.encode(charset, 'ignore')
        return script
    # SHIORI/2.4 API
    def teach(self, word):
        word = unicode(word, charset, 'ignore')
        self.system_entries["system.Sentence"] = word
        self.system_entries["system-Sentence"] = word
        return self.get_system_entry("OnTeach").encode(charset, 'ignore')
    def get_system_entry(self, entry):
        for delimiter in [".", "-"]:
            name = "system" + delimiter + entry
            script = self.get(name, default=None)
            if script is not None:
                return script.strip()
        return None
    def otherghostname(self, list):
        ghosts = []
        for ghost in list:
            name, s0, s1 = ghost.split(chr(1))
            ghosts.append([unicode(name, charset, 'ignore'), s0, s1])
            self.otherghost[name] = [s0, s1]
        otherghost_name = []
        for ghost in ghosts:
            otherghost_name.append(ghost[0])
        self.system_entries["system.OtherGhost"] = otherghost_name
        self.system_entries["system-OtherGhost"] = otherghost_name
        otherghost_ex = []
        for ghost in ghosts:
            otherghost_ex.append(string.join(ghost, chr(1)))
        self.system_entries["system.OtherGhostEx"] = otherghost_ex
        self.system_entries["system-OtherGhostEx"] = otherghost_ex
        if len(ghosts) > 0:
            self.get_system_entry("OnNotifyOther")
        return ''
    def communicate_to(self):
        communicate = self.get_system_entry("communicate")
        if communicate:
            if communicate == 'stop':
                self.system_entries["system.Age"] = '0'
                self.system_entries["system-Age"] = '0'
                communicate = None
            else:
                if self.system_entries.has_key("system.Age"):
                    age = int(self.system_entries["system.Age"]) + 1
                    self.system_entries["system.Age"] = str(age)
                    self.system_entries["system-Age"] = str(age)
                else:
                    self.system_entries["system.Age"] = '0'
                    self.system_entries["system-Age"] = '0'
        self.clear("system.communicate")
        if communicate != None:
            communicate = communicate.encode(charset, 'ignore')
        return communicate
    # internal
    def clear(self, name):
        if self.debug & 128:
            print '*** clear("%s")' % name.encode('UTF-8', 'ignore')
        for dict in self.rdictlist:
            if dict.has_key(name):
                del dict[name]
    def get_internal_dict(self, name):
        for dict in self.rdictlist:
            if dict.has_key(name):
                break
        else:
            dict = self.rdictlist[0]
            dict[name] = []
        return dict
    def unshift(self, name, value):
        if self.debug & 128:
            print ('*** unshift("%s", "%s")' % (name, value)).encode('UTF-8', 'ignore')
        dict = self.get_internal_dict(name)
        i, segments = parse(value, 0)
        dict[name].insert(0, segments)
    def shift(self, name):
        dict = self.get_internal_dict(name)
        value = self.expand(dict[name].pop(0))
        if self.debug & 128:
            print ('*** shift("%s") => "%s"' % (name, value)).encode('UTF-8', 'ignore')
        return value
    def push(self, name, value):
        if self.debug & 128:
            print ('*** push("%s", "%s")' % (name, value)).encode('UTF-8', 'ignore')
        dict = self.get_internal_dict(name)
        i, segments = parse(value, 0)
        dict[name].append(segments)
    def pop(self, name):
        dict = self.get_internal_dict(name)
        value = self.expand(dict[name].pop())
        if self.debug & 128:
            print ('*** pop("%s") => "%s"' % (name, value)).encode('UTF-8', 'ignore')
        return value
    def set(self, name, value):
        self.clear(name)
        self.push(name, value)
    def get(self, name, context=None, depth=0, default=""):
        if depth == self.MAXDEPTH:
            return ""
        if name and name[0] == "@":
            segments = random.choice(context[name])
        else:
            if name.find("&") < 0:
                selection = self.select_simple_phrase(name)
            else:
                selection = self.select_compound_phrase(name)
            if selection is None:
                if self.debug & 8:
                    print "${%s} not found" % name.encode('UTF-8', 'ignore')
                return default
            segments, context = selection
        if self.debug & 16:
            print name.encode('UTF-8', 'ignore'), "=>", string.join(segments, '').encode('UTF-8', 'ignore')
        return self.expand(segments, context, depth)
    def parse_all(self, data, start=0):
        i, segments = parse(data, start)
        return self.expand(segments)
    def parse_sub(self, data, start=0, stop_pattern=None):
        i, segments = parse(data, start, stop_pattern)
        return i, self.expand(segments)
    def expand(self, segments, context=None, depth=0):
        buffer = []
        references = []
        i = 0
        j = len(segments)
        while i < j:
            segment = segments[i]
            if not segment:
                pass
            elif segment[:2] == "${" and segment[-1] == "}":
                newname = segment[2:-1]
                if self.is_number(newname):
                    try:
                        segment = references[int(newname)]
                    except IndexError:
                        pass
                elif self.is_system_entry(newname):
                    if newname in ["system.OtherGhost", "system-OtherGhost",
                                   "system.OtherGhostEx", "system-OtherGhostEx"]:
                        list = self.system_entries.get(newname)
                        if list:
                            segment = random.choice(list)
                        else:
                            segment = ''
                    elif newname == "system.communicate":
                        segment = self.get_system_entry("communicate")
                    else:
                        segment = self.system_entries.get(newname, segment)
                else:
                    segment = self.get(newname, context, depth + 1)
                references.append(segment)
            elif segment[:2] == "$(" and segment[-1] == ")":
                i, segment = self.eval_inline_script(segments, i)
            elif segment[0] == '"' and segment[-1] == '"':
                segment = segment[1:-1].replace(r"\"", "\"")
            buffer.append(segment)
            i += 1
        return string.join(buffer, '')
    def atoi(self, s):
        try:
            return int(s)
        except ValueError:
            return 0
    def is_number(self, s):
        try:
            int(s)
        except ValueError:
            return 0
        return 1
    def is_system_entry(self, s):
        return s[:7] in ["system-", "system."]
    def select_simple_phrase(self, name):
        n = 0
        buffer = []
        for d in self.rdictlist:
            c = d.get(name, [])
            n += len(c)
            buffer.append((c, d))
        if n == 0:
            return None
        n = random.randrange(0, n)
        for c, d in buffer:
            m = len(c)
            if n < m:
                break
            n -= m
        return c[n], d        
    def select_compound_phrase(self, name):
        buffer = []
        for name in [s.strip() for s in name.split("&")]:
            list = []
            for d in self.rdictlist:
                list.extend([tuple(e) for e in d.get(name, [])])
            buffer.append(list)
        buffer.sort(lambda x, y: cmp(len(x), len(y)))
        candidates = []
        for item in buffer.pop(0):
            for list in buffer:
                if item not in list:
                    break
            else:
                candidates.append(item)
        if not candidates:
            return None
        return random.choice(candidates), None
    def eval_inline_script(self, segments, i):
        # check old 'if' syntax
        if segments[i][:5] == "$(if " and i + 1 < len(segments) and \
           segments[i+1][:7] == "$(then ":
            if_block = segments[i][5:-1].strip()
            i += 1
            then_block = segments[i][7:-1].strip()
            if i + 1 < len(segments) and segments[i+1][:7] == "$(else ":
                i += 1
                else_block = segments[i][7:-1].strip()
            else:
                else_block = ""
            if i + 1 < len(segments) and segments[i+1] == "$(endif)":
                i += 1
            else:
                if self.debug & 32:
                    print "kawari.py: syntax error: $(endif) expected"
                return i, "" # syntax error
            return i, self.exec_old_if(if_block, then_block, else_block)
        # execute command(s)
        values = []
        for command in self.split_commands(segments[i][2:-1]):
            argv = self.parse_argument(command)
            argv[0] = self.expand(argv[0])
            if argv[0] == "silent":
                if len(argv) == 1:
                    values = []
                elif self.debug & 32:
                    print "kawari.py: syntax error:", segments[i].encode('UTF-8', 'ignore')
                continue
            handler = self.kis_commands.get(argv[0])
            try:
                if handler is None:
                    raise RuntimeError, "invalid command"
                values.append(handler(self, argv))
            except RuntimeError, message:
                if self.debug & 32:
                    print "kawari.py: %s: %s" % (message, segments[i].encode('UTF-8', 'ignore'))
        result = string.join(values, '')
        if self.debug & 64:
            print ">>>", segments[i].encode('UTF-8', 'ignore')
            print '"' + result.encode('UTF-8', 'ignore') + '"'
        return i, result
    def split_commands(self, data):
        i, segments = parse(data, 0)
        # find multiple commands separated by semicolons
        buffer = []
        command = []
        for segment in segments:
            if segment == ';':
                if command:
                    buffer.append(command)
                    command = []
            else:
                command.append(segment)
        if command:
            buffer.append(command)
        # strip white space before and after each command
        for command in buffer:
            if is_space(command[0]):
                del command[0]
            if is_space(command[-1]):
                del command[-1]
        return buffer
    def parse_argument(self, segments):
        buffer = [[]]
        for segment in segments:
            if is_space(segment):
                buffer.append([])
            else:
                buffer[-1].append(segment)
        return buffer
    def exec_new_if(self, argv):
        if len(argv) == 3:
            return self.exec_if(argv[1], argv[2], None)
        elif len(argv) == 4:
            return self.exec_if(argv[1], argv[2], argv[3])
        else:
            raise RuntimeError, "syntax error"
    def exec_old_if(self, if_block, then_block, else_block):
        # convert [...] into $([...])
        if if_block and if_block[0] == '[' and if_block[-1] == ']':
            if_block = "$(" + if_block + ")"
        # parse arguments
        i, if_block = parse(if_block, 0)
        i, then_block = parse(then_block, 0)
        if else_block:
            i, else_block = parse(else_block, 0)
        result = self.exec_if(if_block, then_block, else_block)
        if self.debug & 64:
            if else_block:
                print ">>> $(if %s)$(then %s)$(else %s)$(endif)" % (
                    string.join(if_block, "").encode('UTF-8', 'ignore'),
                    string.join(then_block, "").encode('UTF-8', 'ignore'),
                    string.join(else_block, "").encode('UTF-8', 'ignore'))
            else:
                print ">>> $(if %s)$(then %s)$(endif)" % (
                    string.join(if_block, "").encode('UTF-8', 'ignore'),
                    string.join(then_block, "").encode('UTF-8', 'ignore'))
            print '"' + result.encode('UTF-8', 'ignore') + '"'
        return result
    def exec_if(self, if_block, then_block, else_block):
        if self.expand(if_block) not in ["", "0", "false", "False"]:
            return self.expand(then_block)
        elif else_block:
            return self.expand(else_block)
        return ""
    def exec_foreach(self, argv):
        if len(argv) != 4:
            raise RuntimeError, "syntax error"
        temp = self.expand(argv[1])
        name = self.expand(argv[2])
        buffer = []
        for dict in self.rdictlist:
            for segments in dict.get(name, []):
                self.set(temp, self.expand(segments))
                buffer.append(self.expand(argv[3]))
        self.clear(temp)
        return string.join(buffer, '')
    def exec_loop(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        try:
            n = int(self.expand(argv[1]))
        except ValueError:
            raise RuntimeError, "invalid argument"
        buffer = []
        for i in range(n):
            buffer.append(self.expand(argv[2]))
        return string.join(buffer, '')
    def exec_while(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        buffer = []
        while self.expand(argv[1]) not in ["", "0", "false", "False"]:
            buffer.append(self.expand(argv[2]))
        return string.join(buffer, '')
    def exec_until(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        buffer = []
        while self.expand(argv[1]) in ["", "0", "false", "False"]:
            buffer.append(self.expand(argv[2]))
        return string.join(buffer, '')
    def exec_set(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        self.set(self.expand(argv[1]), self.expand(argv[2]))
        return ""
    def exec_adddict(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        self.push(self.expand(argv[1]), self.expand(argv[2]))
        return ""
    def exec_array(self, argv): # XXX experimental
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        n = self.atoi(self.expand(argv[2]))
        for d in self.rdictlist:
            c = d.get(name, [])
            if n < len(c):
                return string.join([self.expand(s) for s in c[n]], "")
            n -= len(c)
        else:
            raise RuntimeError, "invalid argument"
    def exec_clear(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        self.clear(self.expand(argv[1]))
        return ""
    def exec_enumerate(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        return string.join([self.expand(s) for s in self.enumerate(name)])
    def enumerate(self, name):
        buffer = []
        for dict in self.rdictlist:
            for segments in dict.get(name, []):
                buffer.append(segments)
        return buffer
    def exec_size(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        n = 0
        for d in self.rdictlist:
            c = d.get(name, [])
            n += len(c)
        return str(n)
    def exec_get(self, argv): # XXX experimental
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        n = self.atoi(self.expand(argv[2]))
        for d in self.rdictlist:
            c = d.get(name, [])
            if n < len(c):
                return string.join(c[n], "")
            n -= len(c)
        else:
            raise RuntimeError, "invalid argument"
    def exec_unshift(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        self.unshift(self.expand(argv[1]), self.expand(argv[2]))
        return ""
    def exec_shift(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        return self.shift(self.expand(argv[1]))
    def exec_push(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        self.push(self.expand(argv[1]), self.expand(argv[2]))
        return ""
    def exec_pop(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        return self.pop(self.expand(argv[1]))
    def exec_pirocall(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        selection = self.select_simple_phrase(self.expand(argv[1]))
        if selection is None:
            return ""
        return selection[0]
    def exec_split(self, argv):
        if len(argv) != 4:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        list = self.expand(argv[2]).split(self.expand(argv[3]))
        n = 0
        for word in list:
            n += 1
            entry = "%s.%d" % (name, n)
            self.set(entry, word)
        self.set(name + '.size', str(n))
        return ""
    def get_dict_path(self, path):
        path = path.replace("\\", "/").lower()
        if not path:
            raise RuntimeError, "invalid argument"
        if path[0] == '/':
            return path
        return os.path.join(self.prefix, path)
    def exec_load(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            rdict, kdict = create_dict(read_dict(path, self.debug))
        except IOError:
            raise RuntimeError, "cannot read file"
        if path in self.pathlist:
            i = self.pathlist.index(path)
            self.rdictlist[i].update(rdict)
            self.kdictlist[i].update(kdict)
        else:
            self.pathlist.insert(0, path)
            self.rdictlist.insert(0, rdict)
            self.kdictlist.insert(0, kdict)
        return ""
    def exec_save(self, argv, crypt=0):
        if len(argv) < 2:
            raise RuntimeError, "syntax error"
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            file = builtin_open(path, "w")
            file.write("#\r\n# Kawari save file\r\n#\r\n")
            for i in range(2, len(argv)):
                name = self.expand(argv[i])
                if not name.strip():
                    continue
                buffer = []
                for segments in self.enumerate(name):
                    buffer.append(string.join(segments, ''))
                name = name.encode(charset, 'ignore')
                line = name + " : " + \
                       string.join(buffer, " , ").encode(charset, 'ignore')
                if crypt:
                    line = encrypt(line)
                file.write("# Entry %s\r\n%s\r\n" % (name, line))
            file.close()
        except IOError:
            raise RuntimeError, "cannot write file"
        return ""
    def exec_savecrypt(self, argv):
        return self.exec_save(argv, 1)
    def exec_textload(self, argv):
        if len(argv) != 3:
            raise RuntimeError, "syntax error"
        path = self.get_dict_path(self.expand(argv[1]))
        try:
            linelist = builtin_open(path).readlines()
        except IOError:
            raise RuntimeError, "cannot read file"
        name = self.expand(argv[2])
        n = 0
        for line in linelist:
            n += 1
            entry = "%s.%d" % (name, n)
            if line[-2:] == "\r\n":
                line = line[:-2]
            elif line[-1] in "\r\n":
                line = line[:-1]
            if not line:
                self.clear(entry)
            else:
                self.set(entry, unicode(line, charset, "replace"))
        self.set(name + ".size", str(n))
        return ""
    def exec_escape(self, argv):
        data = string.join([self.expand(s) for s in argv[1:]])
        data = data.replace("\\", r"\\")
        data = data.replace("%", r"\%")
        return data
    def exec_echo(self, argv):
        return string.join([self.expand(s) for s in argv[1:]])
    def exec_tolower(self, argv):
        return self.exec_echo(argv).lower()
    def exec_toupper(self, argv):
        return self.exec_echo(argv).upper()
    def exec_eval(self, argv):
        return self.parse_all(self.exec_echo(argv))
    def exec_entry(self, argv):
        if len(argv) == 2:
            return self.get(self.expand(argv[1]))
        elif len(argv) == 3:
            return self.get(self.expand(argv[1])) or self.expand(argv[2])
        else:
            raise RuntimeError, "syntax error"
    def exec_null(self, argv):
        if len(argv) != 1:
            raise RuntimeError, "syntax error"
        return ""
    def exec_chr(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        num = self.atoi(self.expand(argv[1]))
        if num < 256:
            return chr(num)
        return chr((num >> 8) & 0xff) + chr(num & 0xff)
    def exec_choice(self, argv):
        if len(argv) == 1:
            return ""
        i = random.randrange(1, len(argv))        
        return self.expand(argv[i])
    def exec_rand(self, argv):
        if len(argv) != 2:
            raise RuntimeError, "syntax error"
        bound = self.atoi(self.expand(argv[1]))
        if bound == 0:
            return str(0)
        elif bound > 0:
            return str(random.randrange(0, bound))
        else:
            return str(random.randint(bound+1, 0))
    def exec_date(self, argv):
        if len(argv) == 1:
            format = "%y/%m/%d %H:%M:%S"
        else:
            format = string.join([self.expand(s) for s in argv[1:]])
        buffer = []
        i = 0
        j = len(format)
        now = time.localtime(time.time())
        while i < j:
            if format[i] == '%':
                i += 1
                if i < j:
                    c = format[i]
                    i += 1
                else:
                    break
                if c in ['y', 'Y']: # year (4 columns)
                    buffer.append("%04d" % now[0])
                elif c == 'm': # month (01 - 12)
                    buffer.append("%02d" % now[1])
                elif c == 'n': # month (1 - 12)
                    buffer.append(str(now[1]))
                elif c == 'd': # day (01 - 31)
                    buffer.append("%02d" % now[2])
                elif c == 'e': # day (1 - 31)
                    buffer.append(str(now[2]))
                elif c == 'H': # hour (00 - 23)
                    buffer.append("%02d" % now[3])
                elif c == 'k': # hour (0 - 23)
                    buffer.append(str(now[3]))
                elif c == 'M': # minute (00 - 59)
                    buffer.append("%02d" % now[4])
                elif c == 'N': # minute (0 - 59)
                    buffer.append(str(now[4]))
                elif c == 'S': # second (00 - 59)
                    buffer.append("%02d" % now[5])
                elif c == 'r': # second (0 - 59)
                    buffer.append(str(now[5]))
                elif c == 'w': # weekday (0 = Sunday)
                    buffer.append(str([1, 2, 3, 4, 5, 6, 0][now[6]]))
                elif c == 'j': # Julian day (001 - 366)
                    buffer.append("%03d" % now[7])
                elif c == 'J': # Julian day (1 - 366)
                    buffer.append(str(now[7]))
                elif c == '%':
                    buffer.append('%')
                else:
                    buffer.append('%')
                    i -= 1
            else:
                buffer.append(format[i])
                i += 1
        return string.join(buffer, '')
    def exec_inc(self, argv):
        def _inc(value, step, bound):
            value += step
            if bound is not None and value > bound:
                return bound
            return value
        self.apply_counter_op(_inc, argv)
        return ""
    def exec_dec(self, argv):
        def _dec(value, step, bound):
            value -= step
            if bound is not None and value < bound:
                return bound
            return value
        self.apply_counter_op(_dec, argv)
        return ""
    def apply_counter_op(self, func, argv):
        if len(argv) < 2 or len(argv) > 4:
            raise RuntimeError, "syntax error"
        name = self.expand(argv[1])
        value = self.atoi(self.get(name))
        if len(argv) >= 3:
            step = self.atoi(self.expand(argv[2]))
        else:
            step = 1
        if len(argv) == 4:
            bound = self.atoi(self.expand(argv[3]))
        else:
            bound = None
        self.set(name, str(func(value, step, bound)))
    def exec_test(self, argv):
        if argv[0] == "test" and len(argv) == 4 or \
           argv[0] == "[" and len(argv) == 5 and self.expand(argv[4]) == "]":
            op1 = self.expand(argv[1])
            op  = self.expand(argv[2])
            op2 = self.expand(argv[3])
        else:
            raise RuntimeError, "syntax error"
        if op in ["=", "=="]:
            return str(op1 == op2)
        elif op == "!=":
            return str(op1 != op2)
        elif op == "<=":
            return str(op1 <= op2)
        elif op == ">=":
            return str(op1 >= op2)
        elif op == "<":
            return str(op1 < op2)
        elif op == ">":
            return str(op1 > op2)
        elif op == "-eq":
            return str(self.atoi(op1) == self.atoi(op2))
        elif op == "-ne":
            return str(self.atoi(op1) != self.atoi(op2))
        elif op == "-le":
            return str(self.atoi(op1) <= self.atoi(op2))
        elif op == "-ge":
            return str(self.atoi(op1) >= self.atoi(op2))
        elif op == "-lt":
            return str(self.atoi(op1) < self.atoi(op2))
        elif op == "-gt":
            return str(self.atoi(op1) > self.atoi(op2))
        else:
            raise RuntimeError, "unknown operator"
    def exec_expr(self, argv):
        tree = self.expr_parser.parse(
            string.join([string.join(e, '') for e in argv[1:]]))
        if tree is None:
            raise RuntimeError, "syntax error"
        try:
            value = self.interp_expr(tree)
        except ExprError:
            raise RuntimeError, "runtime error"
        return value
    def interp_expr(self, tree):
        if tree[0] == ExprParser.OR_EXPR:
            for subtree in tree[1:]:
                value = self.interp_expr(subtree)
                if value and value not in ["0", "False"]:
                    break
            return value
        elif tree[0] == ExprParser.AND_EXPR:
            buffer = []
            for subtree in tree[1:]:
                value = self.interp_expr(subtree)
                if not value or value in ["0", "False"]:
                    return "0"
                buffer.append(value)
            return buffer[0]
        elif tree[0] == ExprParser.CMP_EXPR:
            op1 = self.interp_expr(tree[1])
            op2 = self.interp_expr(tree[3])
            if self.is_number(op1) and self.is_number(op2):
                op1 = int(op1)
                op2 = int(op2)
            if tree[2] in ["=", "=="]:
                return str(op1 == op2)
            elif tree[2] == "!=":
                return str(op1 != op2)
            elif tree[2] == "<=":
                return str(op1 <= op2)
            elif tree[2] == ">=":
                return str(op1 >= op2)
            elif tree[2] == "<":
                return str(op1 < op2)
            elif tree[2] == ">":
                return str(op1 > op2)
            else:
                raise RuntimeError, "unknown operator"
        elif tree[0] == ExprParser.ADD_EXPR:
            for i in range(1, len(tree), 2):
                tree[i] = self.interp_expr(tree[i])
                if not self.is_number(tree[i]):
                    raise ExprError
            value = int(tree[1])
            for i in range(2, len(tree), 2):
                if tree[i] == "+":
                    value += int(tree[i+1])
                elif tree[i] == "-":
                    value -= int(tree[i+1])
            return str(value)
        elif tree[0] == ExprParser.MUL_EXPR:
            for i in range(1, len(tree), 2):
                tree[i] = self.interp_expr(tree[i])
                if not self.is_number(tree[i]):
                    raise ExprError
            value = int(tree[1])
            for i in range(2, len(tree), 2):
                if tree[i] == "*":
                    value *= int(tree[i+1])
                elif tree[i] == "/":
                    try:
                        value /= int(tree[i+1])
                    except ZeroDivisionError:
                        raise ExprError
                elif tree[i] == "%":
                    try:
                        value %= int(tree[i+1])
                    except ZeroDivisionError:
                        raise ExprError
            return str(value)
        elif tree[0] == ExprParser.STR_EXPR:
            if tree[1] == "length":
                length = len(self.get_characters(self.interp_expr(tree[2])))
                return str(length)
            elif tree[1] == "index":
                s = self.get_characters(self.interp_expr(tree[2]))
                c = self.get_characters(self.interp_expr(tree[3]))
                for pos in range(len(s)):
                    if s[pos] in c:
                        break
                else:
                    pos = 0
                return str(pos)
            elif tree[1] == "match":
                try:
                    match = re.match(self.interp_expr(tree[3]),
                                     self.interp_expr(tree[2]))
                except re.error:
                    match = None
                if match:
                    length = match.end() - match.start()
                else:
                    length = 0
                return str(length)
            elif tree[1] == "find":
                s = self.interp_expr(tree[3])
                if self.interp_expr(tree[2]).find(s) < 0:
                    return ""
                return s
            elif tree[1] == "findpos":
                s = self.interp_expr(tree[3])
                pos = self.interp_expr(tree[2]).find(s);
                if pos < 0:
                    return ""
                return str(pos+1)
            elif tree[1] == "substr":
                s = self.interp_expr(tree[2])
                p = self.interp_expr(tree[3])
                n = self.interp_expr(tree[4])
                if self.is_number(p) and self.is_number(n):
                    p = int(p) - 1
                    n = p + int(n)
                    if 0 <= p <= n:
                        characters = self.get_characters(s)
                        return string.join(characters[p:n], '')
                return ""
        elif tree[0] == ExprParser.LITERAL:
            return self.expand(tree[1:])
    def get_characters(self, s):
        buffer = []
        i = 0
        j = len(s)
        while i < j:
            buffer.append(s[i])
            i += 1
        return buffer
    kis_commands = {
        # flow controls
        "if":          exec_new_if,
        "foreach":     exec_foreach,
        "loop":        exec_loop,
        "while":       exec_while,
        "until":       exec_until,
        # dictionary operators
        "adddict":     exec_adddict,
        "array":       exec_array,
        "clear":       exec_clear,
        "enumerate":   exec_enumerate,
        "set":         exec_set,
        "load":        exec_load,
        "save":        exec_save,
        "savecrypt":   exec_savecrypt,
        "textload":    exec_textload,
        "size":        exec_size,
        "get":         exec_get,
        # list operators
        "unshift":     exec_unshift,
        "shift":       exec_shift,
        "push":        exec_push,
        "pop":         exec_pop,
        # counter operators
        "inc":         exec_inc,
        "dec":         exec_dec,
        # expression evaluators
        "expr":        exec_expr,
        "test":        exec_test,
        "[":           exec_test,
        "entry":       exec_entry,
        "eval":        exec_eval,
        # utility functions
        "NULL":        exec_null,
        "?":           exec_choice,
        "date":        exec_date,
        "rand":        exec_rand,
        "echo":        exec_echo,
        "escape":      exec_escape,
        "tolower":     exec_tolower,
        "toupper":     exec_toupper,
        "pirocall":    exec_pirocall,
        "split":       exec_split,
        "urllist":     None,
        "chr":         exec_chr,
        "help":        None,
        "ver":         None,
        "searchghost": None,
        "saoriregist": None,
        "saorierase":  None,
        "callsaori":   None,
        "callsaorix":  None,
        }

###   EXPR PARSER   ###

class ExprError(ValueError):
    pass

class ExprParser:
    def __init__(self):
        self.debug = 0
        # bit 0 = trace get_expr()
        # bit 1 = trace all "get_" functions
    def show_progress(self, func, buffer):
        if buffer is None:
            print "%s() -> syntax error" % func
        else:
            print "%s() -> %s" % (func, buffer)
    re_token = re.compile('[():|&*/%+-]|[<>]=?|[!=]?=|match|index|findpos|find|substr|length|quote|(\\s+)')
    def tokenize(self, data):
        buffer = []
        i = 0
        j = len(data)
        while i < j:
            match = self.re_token.match(data, i)
            if match:
                buffer.append(match.group())
                i = match.end()
            else:
                i, segments = parse(data, i, self.re_token)
                buffer.extend(segments)
        return buffer
    def parse(self, data):
        self.tokens = self.tokenize(data)
        try:
            return self.get_expr()
        except ExprError:
            return None # syntax error
    # internal
    def done(self):
        return not self.tokens
    def pop(self):
        try:
            return self.tokens.pop(0)
        except IndexError:
            raise ExprError
    def look_ahead(self, index=0):
        try:
            return self.tokens[index]
        except IndexError:
            raise ExprError
    def match(self, s):
        if self.pop() != s:
            raise ExprError
    def match_space(self):
        if not is_space(self.pop()):
            raise ExprError
    def check(self, s, index=0):
        return self.look_ahead(index) == s
    def check_space(self, index=0):
        return is_space(self.look_ahead(index))
    # tree node types
    OR_EXPR  = 1
    AND_EXPR = 2
    CMP_EXPR = 3
    ADD_EXPR = 4
    MUL_EXPR = 5
    STR_EXPR = 6
    LITERAL  = 7
    def get_expr(self):
        buffer = self.get_or_expr()
        if not self.done():
            raise ExprError
        if self.debug & 1:
            self.show_progress("get_expr", buffer)
        return buffer
    def get_or_expr(self):
        buffer = [self.OR_EXPR]
        while 1:
            buffer.append(self.get_and_expr())
            if not self.done() and \
               self.check_space() and self.check('|', 1):
                self.pop() # space
                self.pop() # operator
                self.match_space()
            else:
                break
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_or_expr", buffer)
        return buffer
    def get_and_expr(self):
        buffer = [self.AND_EXPR]
        while 1:
            buffer.append(self.get_cmp_expr())
            if not self.done() and \
               self.check_space() and self.check('&', 1):
                self.pop() # space
                self.pop() # operator
                self.match_space()
            else:
                break
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_and_expr", buffer)
        return buffer
    def get_cmp_expr(self):
        buffer = [self.CMP_EXPR]
        buffer.append(self.get_add_expr())
        if not self.done() and \
           self.check_space() and \
           self.look_ahead(1) in ["<=", ">=", "<", ">", "=", "==", "!="]:
            self.pop() # space
            buffer.append(self.pop()) # operator
            self.match_space()
            buffer.append(self.get_add_expr())
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_cmp_expr", buffer)
        return buffer
    def get_add_expr(self):
        buffer = [self.ADD_EXPR]
        while 1:
            buffer.append(self.get_mul_expr())
            if not self.done() and \
               self.check_space() and self.look_ahead(1) in ["+", "-"]:
                self.pop() # space
                buffer.append(self.pop()) # operator
                self.match_space()
            else:
                break
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_add_expr", buffer)
        return buffer
    def get_mul_expr(self):
        buffer = [self.MUL_EXPR]
        while 1:
            buffer.append(self.get_mat_expr())
            if not self.done() and \
               self.check_space() and self.look_ahead(1) in ["*", "/", "%"]:
                self.pop() # space
                buffer.append(self.pop()) # operator
                self.match_space()
            else:
                break
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_mul_expr", buffer)
        return buffer
    def get_mat_expr(self):
        buffer = [self.STR_EXPR]
        buffer.append(self.get_str_expr())
        if not self.done() and \
           self.check_space() and self.check(':', 1):
            buffer.insert(1, "match")
            self.pop() # space
            self.pop() # ':'
            self.match_space()
            buffer.append(self.get_str_expr())
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_mat_expr", buffer)
        return buffer
    def get_str_expr(self):
        argc = 0
        if self.check("length"):
            argc = 1
        elif self.look_ahead() in ["match", "index", "find", "findpos"]:
            argc = 2
        elif self.check("substr"):
            argc = 3
        if argc > 0:
            buffer = [self.STR_EXPR, self.pop()] # fuction
            for i in range(argc):
                self.match_space()
                buffer.append(self.get_str_expr())
        elif self.check("quote"):
            buffer = [self.LITERAL]
            self.pop()
            self.match_space()
            if self.re_token.match(self.look_ahead()):
                buffer.append(self.pop())
            else:
                buffer.extend(self.get_str_seq())
        else:
            buffer = self.get_sub_expr()
        if self.debug & 2:
            self.show_progress("get_str_expr", buffer)
        return buffer
    def get_sub_expr(self):
        if self.check('('):
            self.pop()
            if self.check_space():
                self.pop()
            buffer = self.get_or_expr()
            if self.check_space():
                self.pop()
            #begin specification bug work-around (3/3)
            self.tokens[0] = self.kawari.parse_all(self.tokens[0])
            #end
            self.match(')')
        else:
            buffer = [self.LITERAL]
            buffer.extend(self.get_str_seq())
        if self.debug & 2:
            self.show_progress("get_sub_expr", buffer)
        return buffer
    def get_str_seq(self):
        buffer = []
        while not self.done() and \
              not self.re_token.match(self.look_ahead()):
            buffer.append(self.pop())
        if len(buffer) == 0:
            raise ExprError
        return buffer

# <<< EXPR SYNTAX >>>
# expr     := or-expr
# or-expr  := and-expr (sp or-op sp and-expr)*
# or-op    := '|'
# and-expr := cmp-expr (sp and-op sp cmp-expr)*
# and-op   := '&'
# cmp-expr := add-expr (sp cmp-op sp add-expr)?
# cmp-op   := <= | >= | < | > | == | = | !=
# add-expr := mul-expr (sp add-op sp mul-expr)*
# add-op   := '+' | '-'
# mul-expr := mat-expr (sp mul-op sp mat-expr)*
# mul-op   := '*' | '/' | '%'
# mat-expr := str-expr (sp ':' sp str-expr)?
# str-expr := 'quote' sp (OPERATOR | str-seq) |
#             'match' sp str-expr sp str-expr |
#             'index' sp str-expr sp str-expr |
#             'find' sp str-expr sp str-expr |
#             'findpos' sp str-expr sp str-expr |
#             'substr' sp str-expr sp str-expr sp str-expr |
#             'length' sp str-expr |
#             sub-expr
# sub-expr := '(' sp? or-expr sp? ')' | str-seq
# sp       := SPACE+ (white space)
# str-seq  := STRING+ (literal, "...", ${...}, and/or $(...))

###   API   ###

DICT_FILE, INI_FILE = range(2)

def list_dict(kawari_dir, saori_ini={}, debug=0):
    return scan_ini(kawari_dir, "kawari.ini", saori_ini, debug)

def scan_ini(kawari_dir, filename, saori_ini, debug=0):
    buffer = []
    ini_path = os.path.join(kawari_dir, filename)
    try:
        list = read_dict(ini_path)
    except IOError:
        list = []
    read_as_dict = 0
    for entry, value, position in list:
        if entry == "dict":
            file = value.replace("\\", "/").lower().encode('utf-8')
            path = os.path.join(kawari_dir, file)
            try:
                builtin_open(path).read(64)
            except IOError, (errno, message):
                if debug & 4:
                    print "kawari.py: read error:", path
                continue
            buffer.append((DICT_FILE, path))
        elif entry == "include":
            file = value.replace("\\", "/").lower().encode('utf-8')
            buffer.extend(scan_ini(kawari_dir, file, saori_ini, debug))
        elif entry == "set":
            read_as_dict = 1
        elif entry in ["randomseed", "debug", "security", "set"]:
            pass
        elif entry == "saori":
            list = value.split(',')
            path = list[0].strip().encode('utf-8')
            alias = list[1].strip()
            if len(list) == 3:
                option = list[2].strip()
            else:
                option = "loadoncall"
            saori_ini[alias] = [path, option]
        elif debug & 4:
            print "kawari.py: unknown entry:", entry
    if read_as_dict:
        buffer.append((INI_FILE, ini_path))
    return buffer

def read_ini(path):
    buffer = []
    try:
        list = read_dict(path)
    except IOError:
        list = []
    for entry, value, position in list:
        if entry == "set":
            try:
                entry, value = value.split(None, 1)
            except ValueError:
                continue
            buffer.append((entry.strip(), value.strip(), position))
    return buffer

class Shiori(Kawari):
    def __init__(self, dll_name, debug=0):
        self.debug = debug
        self.dll_name = dll_name
        self.saori_list = {}
        self.kis_commands["saoriregist"] = self.exec_saoriregist
        self.kis_commands["saorierase"] = self.exec_saorierase
        self.kis_commands["callsaori"] = self.exec_callsaori
        self.kis_commands["callsaorix"] = self.exec_callsaorix
    def use_saori(self, saori):
        self.saori = saori
    def load(self, kawari_dir):
        self.kawari_dir = kawari_dir
        pathlist = [None]
        rdictlist = [{}]
        kdictlist = [{}]
        self.saori_ini = {}
        for type, path in list_dict(kawari_dir, self.saori_ini, self.debug):
            pathlist.append(path)
            if type == INI_FILE:
                rdict, kdict = create_dict(read_ini(path), self.debug)
            elif is_local_script(path):
                rdict, kdict = read_local_script(path)
            else:
                rdict, kdict = create_dict(read_dict(path, self.debug), self.debug)
            rdictlist.append(rdict)
            kdictlist.append(kdict)
        Kawari.__init__(self, kawari_dir, pathlist, rdictlist, kdictlist, self.debug)
        for key in self.saori_ini.keys():
            if self.saori_ini[key][1] == "preload":
                head, tail = os.path.split(self.saori_ini[key][0].replace('\\', '/'))
                self.saori_load(self.saori_ini[key][0],
                                os.path.join(self.kawari_dir, head))
        return 1
    def unload(self):
        Kawari.finalize(self)
        for name in self.saori_list.keys():
            self.saori_list[name].unload()
            del self.saori_list[name]
        global charset
        charset = 'Shift_JIS' # reset
    def find(self, dir, dll_name):
        result = 0
        if list_dict(dir):
            result = 200
        global charset
        charset = 'Shift_JIS' # reset
        return result
    def show_description(self):
        sys.stdout.write('Shiori: KAWARI compatible module for ninix\n'
                         '        Copyright (C) 2001, 2002 by Tamito KAJIYAMA\n'
                         '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
                         '        Copyright (C) 2002-2004 by Shyouzou Sugitani\n'
                         '        Copyright (C) 2003 by Shun-ichi TAHARA\n')
    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        req_header = {}
        line = header.readline()
        if line:
            line = line.strip()
            list = line.split()
            if len(list) >= 2:
                command = list[0].strip()
                protocol = list[1].strip()
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                line = line.strip()
                if not line:
                    continue
                colon = line.find(':')
                if colon >= 0:
                    key = line[:colon].strip()
                    value = line[colon+1:].strip()
                    try:
                        value = int(value)
                    except:
                        value = str(value)
                    req_header[key] = value
                else:
                    continue
        result = ''
        to = None
        if req_header.has_key('ID'):
            if req_header['ID'] == 'charset':
                result = charset
            elif req_header['ID'] == 'dms':
                result = self.getdms()
            elif req_header['ID'] == 'OnAITalk':
                result = self.getaistringrandom()
            elif req_header['ID'] in ["ms", "mz", "ml", "mc", "mh", \
                                      "mt", "me", "mp"]:
                result = self.getword(req_header['ID'])
            elif req_header['ID'] == "m?":
                result = self.getword("m")
            elif req_header['ID'] == 'otherghostname':
                otherghost = []
                for n in range(128):
                    if req_header.has_key('Reference'+str(n)):
                        otherghost.append(req_header['Reference'+str(n)])
                result = self.otherghostname(otherghost)
            elif req_header['ID'] == 'OnTeach':
                if req_header.has_key('Reference0'):
                    self.teach(req_header['Reference0'])
            else:
                result = self.getstring(req_header['ID'])
                if not result:
                    ref = []
                    for n in range(8):
                        if req_header.has_key('Reference'+str(n)):
                            ref.append(req_header['Reference'+str(n)])
                        else:
                            ref.append(None)
                    ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7 = ref
                    result = self.get_event_response(
                        req_header['ID'], ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7)
            if result == None:
                result = ''
            to = self.communicate_to()
        result = 'SHIORI/3.0 200 OK\r\n' \
                 'Sender: Kawari\r\n' \
                 'Value: %s\r\n' % result
        if to != None:
            result += 'Reference0: %s\r\n' % to
        result += '\r\n'
        return result            
    def exec_saoriregist(self, kawari, argv):
        file = self.expand(argv[1])
        alias = self.expand(argv[2])
        if len(argv) == 4:
            option = self.expand(argv[3])
        else:
            option = "loadoncall"
        self.saori_ini[alias] = [file, option]
        if self.saori_ini[alias][1] == "preload":
            head, tail = os.path.split(self.saori_ini[alias][0].replace('\\', '/'))
            self.saori_load(self.saori_ini[alias][0],
                            os.path.join(self.kawari_dir, head))
        return ""
    def exec_saorierase(self, kawari, argv):
        alias = self.expand(argv[1])
        if self.saori_ini.has_key(alias):
            self.saori_unload(self.saori_ini[alias][0])
        return ""
    def exec_callsaori(self, kawari, argv):
        alias = self.expand(argv[1])
        if not self.saori_ini.has_key(alias):
            return ""
        if not self.saori_list.has_key(self.saori_ini[alias][0]):
            if self.saori_ini[alias][1] == "preload":
                return ""
            else:
                head, tail = os.path.split(self.saori_ini[alias][0].replace('\\', '/'))
                self.saori_load(self.saori_ini[alias][0],
                                os.path.join(self.kawari_dir, head))
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: KAWARI\r\n' \
              'SecurityLevel: local\r\n' \
              'Charset: %s\r\n' % charset
        for i in range(2, len(argv)):
            req += 'Argument%s: %s\r\n' % (i-2, self.expand(argv[i]))
        req += '\r\n'
        response = self.saori_request(self.saori_ini[alias][0], req.encode(charset, 'ignore'))
        header = StringIO.StringIO(response)
        line = header.readline()
        if line:
            if line[-1] == '\n':
                line = line[:-1]
            line = line.strip()
            pos_space = line.find(' ')
            if pos_space >= 0:
                saori_protocol = line[:pos_space].strip()
                saori_statuscode = line[pos_space:].strip()
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                if line[-1] == '\n':
                    line = line[:-1]
                line = line.strip()
                if not line:
                    continue
                colon = line.find(':')
                if colon >= 0:
                    key = line[:colon].strip()
                    value = line[colon+1:].strip()
                    if key:
                        saori_header.append(key)
                        saori_value[key] = value
        if saori_value.has_key('Result'):
            result = saori_value['Result']
        else:
            result =  ''
        if self.saori_ini[alias][1] == "noresident":
            self.saori_unload(self.saori_ini[alias][0])
        return result
    def exec_callsaorix(self, kawari, argv):
        alias = self.expand(argv[1])
        entry = self.expand(argv[2])
        if not self.saori_ini.has_key(alias):
            return ""
        if not self.saori_list.has_key(self.saori_ini[alias][0]):
            if self.saori_ini[alias][1] == 'preload':
                return ""
            else:
                head, tail = os.path.split(self.saori_ini[alias][0].replace('\\', '/'))
                self.saori_load(self.saori_ini[alias][0],
                                os.path.join(self.kawari_dir, head))
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: KAWARI\r\n' \
              'SecurityLevel: local\r\n'
        for i in range(3, len(argv)):
            req += 'Argument%s: %s\r\n' % (i-3, self.expand(argv[i]))
        req += '\r\n'
        response = self.saori_request(self.saori_ini[alias][0], req.encode(charset, 'ignore'))
        header = StringIO.StringIO(response)
        line = header.readline()
        if line:
            if line[-1] == '\n':
                line = line[:-1]
            line = line.strip()
            pos_space = line.find(' ')
            if pos_space >= 0:
                saori_protocol = line[:pos_space].strip()
                saori_statuscode = line[pos_space:].strip()
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                if line[-1] == '\n':
                    line = line[:-1]
                line = line.strip()
                if not line:
                    continue
                colon = line.find(':')
                if colon >= 0:
                    key = line[:colon].strip()
                    value = line[colon+1:].strip()
                    if key:
                        saori_header.append(key)
                        saori_value[key] = value
        result = {}
        for key in saori_value.keys():
            if key[:5] == 'Value':
                result[key] = saori_value[key]
        for key in result.keys():
            self.set(string.join((entry, '.', key), ''), result[key])
        if self.saori_ini[alias][1] == "noresident":
            self.saori_unload(self.saori_ini[alias][0])
        return len(result)
    def saori_load(self, saori, path):
        result = 0
        if self.saori_list.has_key(saori):
            result = self.saori_list[saori].load(path)
        else:
            module = self.saori.request(saori)
            if module:
                self.saori_list[saori] = module
                result = self.saori_list[saori].load(path)
        return result
    def saori_unload(self, saori):
        result = 0
        if self.saori_list.has_key(saori):
            result = self.saori_list[saori].unload()
        return result
    def saori_request(self, saori, req):
        result = 'SAORI/1.0 500 Internal Server Error'
        if self.saori_list.has_key(saori):
            result = self.saori_list[saori].request(req)
        return result

def open(kawari_dir, debug=0):
    pathlist = [None]
    rdictlist = [{}]
    kdictlist = [{}]
    for type, path in list_dict(kawari_dir, debug=debug):
        pathlist.append(path)
        if type == INI_FILE:
            rdict, kdict = create_dict(read_ini(path), debug)
        elif is_local_script(path):
            rdict, kdict = read_local_script(path)
        else:
            rdict, kdict = create_dict(read_dict(path, debug), debug)
        rdictlist.append(rdict)
        kdictlist.append(kdict)
    return Kawari(kawari_dir, pathlist, rdictlist, kdictlist, debug)

def is_local_script(path):
    line = builtin_open(path).readline()
    return line[:8] == "[SAKURA]"

###   TEST   ###

def test():
    import sys, getopt
    try:
        options, argv = getopt.getopt(sys.argv[1:], "d:", ["debug="])
    except getopt.error, e:
        sys.stderr.write(str(e) + "\n")
        sys.exit(1)
    if len(argv) > 0:
        kawari_dir = argv[0]
    else:
        kawari_dir = os.curdir
    debug = 4+8+16+32+64+128
    for opt, val in options:
        if opt in ["-d", "--debug"]:
            debug = int(val)
    print "reading kawari.ini..."
    kawari = open(kawari_dir, debug)
    for rdict in kawari.rdictlist:
        for k, v in rdict.items():
            print k.encode('UTF-8', 'ignore')
            for p in v:
                print "\t" + string.join(p, "").encode('UTF-8', 'ignore')
    for kdict in kawari.kdictlist:
        for k, v in kdict.items():
            print '[ "' + string.join(k, '", "').encode('UTF-8', 'ignore') + '" ]'
            for p in v:
                print "\t" + string.join(p, "").encode('UTF-8', 'ignore')
    while 1:
        print "=" * 40
        s = kawari.getaistringrandom()
        print "-" * 40
        print unicode(s, charset, 'ignore').encode('UTF-8', 'ignore')
        try:
            raw_input()
        except (EOFError, KeyboardInterrupt):
            break

if __name__ == "__main__":
    test()
