# -*- coding: UTF-8 -*-
#
#  niseshiori.py - a "偽栞" compatible Shiori module for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2004 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: niseshiori.py,v 1.20 2004/06/23 05:57:04 shy Exp $
#

import random
import time
import sys
import os
import re
import string
import StringIO
from types import *

import codecs

try:
    import _niseshiori
except ImportError:
    _niseshiori = None

REVISION = "$Revision: 1.20 $"

builtin_open = open

def list_dict(dir):
    buffer = []
    try:
        filelist = os.listdir(dir)
    except OSError:
        filelist = []
    re_dict_filename = re.compile(r"^ai.*\.(dtx|txt)$")
    for file in filelist:
        if re_dict_filename.match(file):
            buffer.append(os.path.join(dir, file))
    return buffer

class NiseShiori:
    def __init__(self, debug=0):
        self.debug = debug
        self.dict = {}
        self.type_chains = {}
        self.word_chains = {}
        self.keywords = {}
        self.responses = {}
        self.greetings = {}
        self.events = {}
        self.resources = {}
        self.variables = {}
        self.dbpath = None
        self.expr_parser = ExprParser()
        self.username = ""
        self.ai_talk_interval = 180
        self.ai_talk_count = 0
        self.surf0 = 0
        self.surf1 = 10
        self.event = None
        self.reference = None
        self.jump_entry = None
        self.motion_area = None
        self.motion_count = 0
        self.mikire = 0
        self.kasanari = 0
        self.otherghost = []
        self.to = ''
        self.sender = ''
    def load(self, dir):
        # read dictionaries
        dictlist = list_dict(dir)
        if not dictlist:
            return 1
        for path in dictlist:
            self.read_dict(path)
        # read variables
        self.dbpath = os.path.join(dir, "niseshiori.db")
        self.load_database(self.dbpath)
        return 0
    def load_database(self, path):
        try:
            file = builtin_open(path)
        except IOError:
            return
        dict = {}
        for line in file:
            if line[:9] == "# ns_st: ":
                try:
                    self.ai_talk_interval = int(line[9:])
                except ValueError:
                    pass
                continue
            elif line[:9] == "# ns_tn: ":
                self.username = line[9:].strip()
                continue
            try:
                name, value = [s.strip() for s in line.split("=")]
            except ValueError:
                print "niseshiori.py: malformed database (ignored)"
                return
            dict[name] = value
        self.variables = dict
    def save_database(self):
        if self.dbpath is None:
            return
        try:
            file = builtin_open(self.dbpath, "w")
        except IOError:
            print "niseshiori.py: cannot write database (ignored)"
            return
        file.write("# ns_st: %d\n" % self.ai_talk_interval)
        file.write("# ns_tn: %s\n" % self.username)
        for name, value in self.variables.items():
            if name[0] != "_":
                file.write("%s=%s\n" % (name, str(value)))
        file.close()
    def finalize(self):
        pass
    re_type = re.compile(r"\\(m[szlchtep?]|[dk])")
    re_user = re.compile(r"\\u[a-z]")
    re_category = re.compile(r"\\(m[szlchtep]|[dk])?\[([^\]]+)\]")
    def read_dict(self, path):
        # read dict file and decrypt if necessary
        file = builtin_open(path)
        if path[-4:] == ".dtx":
            buffer = self.decrypt(file.read())
        else:
            buffer = file.readlines()
        file.close()
        # omit empty lines and comments and merge continuous lines
        definitions = []
        decode = lambda line: unicode(line, 'Shift_JIS', "replace")
        in_comment = 0
        i, j = 0, len(buffer)
        while i < j:
            line = buffer[i].strip()
            i += 1
            if not line:
                continue
            elif i == 1 and line[:9] == "#Charset:":
                charset = line[9:].strip()
                if charset == "UTF-8":
                    decode = lambda line: unicode(line, 'utf-8')
                elif charset in ["EUC-JP", "EUC-KR"]:
                    decode = lambda line: unicode(line, charset)
                continue
            elif line[:2] == "/*":
                in_comment = 1
                continue
            elif line[:2] == "*/":
                in_comment = 0
                continue
            elif in_comment or line[0] == "#" or line[:2] == "//":
                continue
            lines = [line]
            while i < j and buffer[i] and buffer[i][0] in [" ", "\t"]:
                lines.append(buffer[i].strip())
                i += 1
            definitions.append(string.join(lines, ''))
        # parse each line
        for line in definitions:
            line = decode(line)
            # special case: words in a category
            match = self.re_category.match(line)
            if match:
                line = line[match.end():].strip()
                if not line or line[0] != ",":
                    self.syntax_error(path, line)
                    continue
                words = [s.strip() for s in self.split(line) if s.strip()]
                type, catlist = match.groups()
                for cat in [s.strip() for s in self.split(catlist)]:
                    if type is None:
                        keylist = [(None, cat)]
                    else:
                        keylist = [(None, cat), (type, cat)]
                    for key in keylist:
                        value = self.dict.get(key, [])
                        value.extend(words)
                        self.dict[key] = value
                if type is not None:
                    key = "\\" + type
                    value = self.dict.get(key, [])
                    value.extend(words)
                    self.dict[key] = value
                continue
            # other cases
            try:
                command, argv = [s.strip() for s in self.split(line, 1)]
            except ValueError:
                self.syntax_error(path, line)
                continue
            if command == r"\ch":
                argv = [s.strip() for s in self.split(argv)]
                if len(argv) == 5:
                    t1, w1, t2, w2, c = argv
                elif len(argv) == 4:
                    t1, w1, t2, w2 = argv
                    c = None
                elif len(argv) == 3:
                    t1, w1, c = argv
                    t2 = w2 = None
                else:
                    self.syntax_error(path, line)
                    continue
                if not self.re_type.match(t1) and not self.re_user.match(t1):
                    self.syntax_error(path, line)
                    continue
                if c is not None:
                    list = self.type_chains.get(t1, [])
                    list.append((c, w1))
                    self.type_chains[t1] = list
                if t2 is None:
                    continue
                if not self.re_type.match(t2) and not self.re_user.match(t2):
                    self.syntax_error(path, line)
                    continue
                if c is not None:
                    list = self.type_chains.get(t2, [])
                    list.append((c, w2))
                    self.type_chains[t2] = list
                m1 = "%" + t1[1:]
                m2 = "%" + t2[1:]
                key = (m1, w1)
                dict = self.word_chains.get(key, {})
                list = dict.get(m2, [])
                if (c, w2) not in list:
                    list.append((c, w2))
                    dict[m2] = list
                    self.word_chains[key] = dict
                key = (m2, w2)
                dict = self.word_chains.get(key, {})
                list = dict.get(m1, [])
                if (c, w1) not in list:
                    list.append((c, w1))
                    dict[m1] = list
                    self.word_chains[key] = dict
                list = self.dict.get(t1, [])
                if w1 not in list:
                    list.append(w1)
                    self.dict[t1] = list
                list = self.dict.get(t2, [])
                if w2 not in list:
                    list.append(w2)
                    self.dict[t2] = list
            elif self.re_type.match(command) or self.re_user.match(command):
                words = [s.strip() for s in self.split(argv) if s.strip()]
                value = self.dict.get(command, [])
                value.extend(words)
                self.dict[command] = value
            elif command in [r"\dms", r"\e"]:
                value = self.dict.get(command, [])
                value.append(argv)
                self.dict[command] = value
            elif command == r"\ft":
                argv = [s.strip() for s in self.split(argv, 2)]
                if len(argv) != 3:
                    self.syntax_error(path, line)
                    continue
                w, t, s = argv
                if not self.re_type.match(t):
                    self.syntax_error(path, line)
                    continue
                self.keywords[(w, t)] = s
            elif command == r"\re":
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    cond = self.parse_condition(argv[0])
                    list = self.responses.get(cond, [])
                    list.append(argv[1])
                    self.responses[cond] = list
            elif command == r"\hl":
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    list = self.greetings.get(argv[0], [])
                    list.append(argv[1])
                    self.greetings[argv[0]] = list
            elif command == r"\ev":
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    cond = self.parse_condition(argv[0])
                    list = self.events.get(cond, [])
                    list.append(argv[1])
                    self.events[cond] = list
            elif command == r"\id":
                argv = [s.strip() for s in self.split(argv, 1)]
                if len(argv) == 2:
                    if argv[0] in ["sakura.recommendsites",
                                   "kero.recommendsites",
                                   "sakura.portalsites"]:
                        list = self.resources.get(argv[0], "")
	                if list:
                            list += '\2'
	                list += argv[1].replace(' ', '\1')
                        self.resources[argv[0]] = list
                    else:
                        self.resources[argv[0]] = argv[1]
            elif command == r"\tc":
                pass
            else:
                self.syntax_error(path, line)
    def split(self, line, maxcount=None):
        buffer = []
        count = 0
        end = pos = 0
        while maxcount is None or count < maxcount:
            pos = line.find(",", pos)
            if pos < 0:
                break
            elif pos > 0 and line[pos-1] == "\\":
                pos += 1
            else:
                buffer.append(line[end:pos])
                count += 1
                end = pos = pos + 1
        buffer.append(line[end:])
        return [s.replace("\\,", ",") for s in  buffer]
    def syntax_error(self, path, line):
        if self.debug & 4:
            print "niseshiori.py: syntax error in", os.path.basename(path)
            print line
    re_comp_op = re.compile("<[>=]?|>=?|=")
    COND_COMPARISON = 1
    COND_STRING     = 2
    def parse_condition(self, condition):
        buffer = []
        for expr in [s.strip() for s in condition.split("&")]:
            match = self.re_comp_op.search(expr)
            if match:
                buffer.append((self.COND_COMPARISON, (
                    expr[:match.start()].strip(),
                    match.group(),
                    expr[match.end():].strip())))
            else:
                buffer.append((self.COND_STRING, expr))
        return tuple(buffer)
    def decrypt(self, data):
        if _niseshiori is not None:
            return _niseshiori.decrypt(data)
        buffer = []
        a = 0x61
        i = 0
        j = len(data)
        line = []
        while i < j:
            if data[i] == "@":
                i += 1
                buffer.append(string.join(line, ''))
                line = []
                continue
            y = ord(data[i])
            i += 1
            x = ord(data[i])
            i += 1
            x -= a
            a += 9
            y -= a
            a += 2
            if a > 0xdd:
                a = 0x61
            line.append(chr((x & 0x03) | ((y & 0x03) << 2) | ((y & 0x0c) << 2) | ((x & 0x0c) << 4)))
        return buffer

    # SHIORI/1.0 API
    def getaistringrandom(self):
        result = self.get_event_response("OnNSRandomTalk") or self.get(r"\e")
        return result.encode('utf-8', 'ignore')
    def getaistringfromtargetword(self, word):
        word = unicode(word, 'utf-8', 'ignore')
        return self.get(r"\e").encode('utf-8', 'ignore') # XXX
    def getdms(self):
        return self.get(r"\dms").encode('utf-8', 'ignore')
    def getword(self, type):
        return self.get("\\" + type).encode('utf-8', 'ignore')

    # SHIORI/2.4 API
    def teach(self, word):
        return None

    # SHIORI/2.5 API
    def getstring(self, name):
        name = unicode(name, 'utf-8', 'ignore')
        result = self.resources.get(name)
        if result != None:
            result = result.encode('utf-8', 'ignore')
        return result

    # SHIORI/2.2 API
    def get_event_response(self, event,
                           ref0=None, ref1=None, ref2=None, ref3=None,
                           ref4=None, ref5=None, ref6=None, ref7=None):
        def proc(ref):
            if ref != None and type(ref) not in [IntType, FloatType]:
                ref = unicode(ref, 'utf-8', 'ignore')
            return ref
        ref = [proc(ref) for ref in [ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7]]
        if event in ["OnSecondChange", "OnMinuteChange"]:
            if ref[1]:
                self.mikire += 1
                script = self.get_event_response("OnNSMikireHappen")
                if script is not None:
                    return script
            elif self.mikire > 0:
                self.mikire = 0
                return self.get_event_response("OnNSMikireSolve")
            if ref[2]:
                self.kasanari += 1
                script = self.get_event_response("OnNSKasanariHappen")
                if script is not None:
                    return script
            elif self.kasanari > 0:
                self.kasanari = 0
                return self.get_event_response("OnNSKasanariHappen")
            if event == "OnSecondChange" and self.ai_talk_interval > 0:
                self.ai_talk_count += 1
                if self.ai_talk_count == self.ai_talk_interval:
                    self.ai_talk_count = 0
                    if len(self.otherghost) > 0 and \
                       random.randint(0, 10) == 0:
                        target = []
                        for name, s0, s1 in self.otherghost:
                            if self.greetings.has_key(name):
                                target.append(name)
                        if len(target) > 0:
                            self.to = random.choice(target)
                        if self.to:
                            self.current_time = time.localtime(time.time())
                            s = random.choice(self.greetings[self.to])
                            if s:
                                s = self.replace_meta(s)
                                while 1:
                                    match = self.re_ns_tag.search(s)
                                    if not match:
                                        break
                                    value = self.eval_ns_tag(match.group())
                                    s = s[:match.start()] + str(value) + s[match.end():]
                            return s.encode('utf-8', 'ignore')
                    return self.getaistringrandom()
            return None
        elif event == "OnMouseMove":
            if self.motion_area != (ref[3], ref[4]):
                self.motion_area = (ref[3], ref[4])
                self.motion_count = 0
            else:
                self.motion_count += 5 # sensitivity
        elif event == "OnSurfaceChange":
            self.surf0 = ref[0]
            self.surf1 = ref[1]
        elif event == "OnUserInput" and ref[0] == "ninix.niseshiori.username":
            self.username = ref[1]
            self.save_database()
            return "\e"
        elif event == "OnCommunicate":
            self.event = ''
            self.reference = (ref[1], )
            if ref[0] == 'user':
                self.sender = 'User'
            else:
                self.sender = ref[0]
            candidate = []
            for cond in self.responses.keys():
                if self.eval_condition(cond):
                    candidate.append(cond)
            script = None
            self.to = ref[0]
            if len(candidate) > 0:
                cond = random.choice(candidate)
                script = random.choice(self.responses[cond])
            if not script and self.responses.has_key("nohit"):
                script = random.choice(self.responses["nohit"])
            if not script:
                self.to = ''
            else:
                script = self.replace_meta(script)
                while 1:
                    match = self.re_ns_tag.search(script)
                    if not match:
                        break
                    value = self.eval_ns_tag(match.group())
                    scipt = script[:match.start()] + str(value) + script[match.end():]
            self.sender = ''
            return script.encode('utf-8', 'ignore')
        key = "action"
        self.dict[key] = []
        self.event = event
        self.reference = tuple(ref)
        self.current_time = time.localtime(time.time())
        for condition, actions in self.events.items():
            if self.eval_condition(condition):
                self.dict[key].extend(actions)
        script = self.get(key, default=None)
        if script is not None:
            self.ai_talk_count = 0
        self.event = None
        self.reference = None
        if script != None:
            script = script.encode('utf-8', 'ignore')
        return script
    def otherghostname(self, list):
        self.otherghost = []
        for ghost in list:
            name, s0, s1 = ghost.split(chr(1))
            self.otherghost.append([unicode(name, 'utf-8', 'ignore'), s0, s1])
        return ''
    def communicate_to(self):
        to = self.to
        self.to = ''
        return to.encode('utf-8', 'ignore')
    def eval_condition(self, condition):
        for type, expr in condition:
            if not self.eval_conditional_expression(type, expr):
                return 0
        return 1
    def eval_conditional_expression(self, type, expr):
        if type == NiseShiori.COND_COMPARISON:
            value1 = self.expand_meta(expr[0])
            value2 = expr[2]
            try:
                op1 = int(value1)
                op2 = int(value2)
            except ValueError:
                op1 = str(value1)
                op2 = str(value2)
            if expr[1] == ">=":
                return op1 >= op2
            elif expr[1] == "<=":
                return op1 <= op2
            elif expr[1] == ">":
                return op1 > op2
            elif expr[1] == "<":
                return op1 < op2
            elif expr[1] == "=":
                return op1 == op2
            elif expr[1] == "<>":
                return op1 != op2
        elif type == NiseShiori.COND_STRING:
            if self.event.find(expr) >= 0:
                return 1
            for ref in self.reference:
                if ref is not None and str(ref).find(expr) >= 0:
                    return 1
            return 0

    re_ns_tag = re.compile(r"\\(ns_(st(\[[0-9]+\])?|cr|hl|rf\[[^\]]+\]|ce|tc\[[^\]]+\]|tn(\[[^\]]+\])?|jp\[[^\]]+\]|rt)|set\[[^\]]+\])")
    def get(self, key, default=""):
        self.current_time = time.localtime(time.time())
        s = self.expand(key, "", default)
        if s:
            while 1:
                match = self.re_ns_tag.search(s)
                if not match:
                    break
                value = self.eval_ns_tag(match.group())
                s = s[:match.start()] + str(value) + s[match.end():]
        return s
    def eval_ns_tag(self, tag):
        value = ""
        if tag == r"\ns_cr":
            self.ai_talk_count = 0
        elif tag == r"\ns_ce":
            self.to = ''
        elif tag == r"\ns_hl":
            if len(self.otherghost) > 0:
                self.to = random.choice(self.otherghost)[0]
        elif tag[:7] == r"\ns_st[" and tag[-1:] == "]":
            try:
                num = int(tag[7:-1])
            except ValueError:
                pass
            else:
                if num == 0:
                    self.ai_talk_interval = 0
                elif num == 1:
                    self.ai_talk_interval = 420
                elif num == 2:
                    self.ai_talk_interval = 180
                elif num == 3:
                    self.ai_talk_interval = 60
                else:
                    self.ai_talk_interval = min(max(num, 4), 999)
                self.save_database()
                self.ai_talk_count = 0
        elif tag[:7] == r"\ns_jp[" and tag[-1:] == "]":
            name = tag[7:-1]
            self.jump_entry = name
            value = self.get_event_response("OnNSJumpEntry") or ""
            self.jump_entry = None
        elif tag[:5] == r"\set[" and tag[-1:] == "]":
            statement = tag[5:-1]
            pos = statement.find("=")
            if pos > 0:
                name = statement[:pos].strip()
                expr = statement[pos+1:].strip()
                value = self.eval_expression(expr)
                if value is not None:
                    self.variables[name] = value
                    if name[0] != "_":
                        self.save_database()
                    if self.debug & 2048:
                        print 'set %s = "%s"' % (name, value)
                elif self.debug & 1024:
                    print "niseshiori.py: syntax error"
                    print tag
            elif self.debug & 1024:
                print "niseshiori.py: syntax error"
                print tag
        elif tag == r"\ns_rt":
            value = r"\a"
        elif tag == r"\ns_tn":
            value = r"\![open,inputbox,ninix.niseshiori.username,-1]"
        elif tag[:7] == r"\ns_tn[" and tag[-1:] == "]":
            self.username = tag[7:-1]
            self.save_database()
        return value
    re_meta = re.compile(r"%((rand([1-9]|\[[0-9]+\])|selfname2?|sakuraname|keroname|username|friendname|year|month|day|hour|minute|second|week|ghostname|sender|ref[0-7]|surf[01]|word|ns_st|get\[[^\]]+\]|jpentry|plathome|platform|move|mikire|kasanari|ver)|(dms|(m[szlchtep?]|[dk])(\[[^\]]*\])?|\[[^\]]*\]|u[a-z])([2-9]?))")
    def replace_meta(self, s):
        pos = 0
        buffer = []
        variables = {}
        variable_chains = []
        while 1:
            match = self.re_meta.search(s, pos)
            if not match:
                buffer.append(s[pos:])
                break
            buffer.append(s[pos:match.start()])
            meta = match.group()
            if match.group(4) is not None: # %ms, %dms, %ua, etc.
                if not variables.has_key(meta):
                    chained_meta = "%" + match.group(4)
                    for chains in variable_chains:
                        if chains.has_key(chained_meta):
                            candidates_A, candidates_B = chains[chained_meta]
                            if candidates_A:
                                word = random.choice(candidates_A)
                                candidates_A.remove(word)
                            else:
                                word = random.choice(candidates_B)
                                candidates_B.remove(word)
                            if not candidates_A and not candidates_B:
                                del chains[chained_meta]
                            if self.debug & 16:
                                print "chained: %s => %s" % (meta, word)
                            break
                    else:
                        if match.group(4) == 'm?':
                            word = self.expand("\\" + random.choice(["ms", "mz", "ml", "mc", "mh", "mt", "me", "mp"]), s)
                        else:
                            word = self.expand("\\" + match.group(4), s)
                    chains = self.find_chains((chained_meta, word), s)
                    if self.debug & 16:
                        prefix = "chain:"
                        for k in chains.keys():
                            candidates_A, candidates_B = chains[k]
                            for w in candidates_A:
                                print prefix, "%s, %s" % (k, w)
                                prefix = "      "
                            for w in candidates_B:
                                print prefix, "%s, %s" % (k, w)
                                prefix = "      "
                    variables[meta] = word
                    variable_chains.append(chains)
                buffer.append(variables[meta])
            else:
                buffer.append(str(self.expand_meta(meta)))
            pos = match.end()
        t = string.join(buffer, '')
        return t
    def expand(self, key, context, default=""):
        choices = []
        for keyword, word in self.type_chains.get(key, []):
            if not context.find(keyword) < 0:
                if self.debug & 16:
                    print "chain keyword:", keyword
                choices.append(word)
        if not choices:
            match = self.re_category.match(key)
            if match:
                key = match.groups()
            choices = self.dict.get(key)
        if not choices:
            if self.debug & 8:
                if type(key) == type(()):
                    key = "(%s, %s)" % key
                sys.stderr.write("%s not found\n" % key)
            return default
        s = random.choice(choices)
        t = self.replace_meta(s)
        if self.debug & 16:
            if type(key) == type(()):
                key = "\\%s[%s]" % (key[0] or "", key[1])
            print key.encode('UTF-8', 'ignore'), "=>", s.encode('UTF-8', 'ignore')
            print " " * len(key),
            print "=>", t.encode('UTF-8', 'ignore')
        return t
    def find_chains(self, key, context):
        chains = {}
        dict = self.word_chains.get(key, {})
        for chained_meta in dict.keys():
            candidates_A = []
            candidates_B = []
            for keyword, chained_word in dict[chained_meta]:
                if keyword and not context.find(keyword) < 0:
                    candidates_A.append(chained_word)
                else:
                    candidates_B.append(chained_word)
            chains[chained_meta] = (candidates_A, candidates_B)
        return chains
    WEEKDAY_NAMES = ["mon", "tue", "wed", "thu", "fri", "sat", "sun"]
    def expand_meta(self, name):
        if name in ["%selfname", "%selfname2", "%keroname", "%friendname"]:
            return name
        elif name == "%sakuraname":
            return "%selfname"
        elif name == "%username":
            return self.username or "%username"
        elif name[:5] == "%get[" and name[-1] == "]":
            value = self.variables.get(name[5:-1], "?")
            try:
                return int(value)
            except ValueError:
                return value
        elif name[:6] == "%rand[" and name[-1] == "]":
            n = int(name[6:-1])
            return random.randint(1, n)
        elif name[:5] == "%rand" and len(name) == 6:
            n = int(name[5])
            return random.randint(10**(n-1), 10**n-1)
        elif name[:4] == "%ref" and len(name) == 5 and name[4] in "01234567":
            if self.reference is None:
                return ""
            n = int(name[4])
            if self.reference[n] is None:
                return ""
            return self.reference[n]
        elif name == "%jpentry":
            if self.jump_entry is None:
                return ""
            return self.jump_entry
        elif name == "%year":
            return str(self.current_time[0])
        elif name == "%month":
            return str(self.current_time[1])
        elif name == "%day":
            return str(self.current_time[2])
        elif name == "%hour":
            return str(self.current_time[3])
        elif name == "%minute":
            return str(self.current_time[4])
        elif name == "%second":
            return str(self.current_time[5])
        elif name == "%week":
            return self.WEEKDAY_NAMES[self.current_time[6]]
        elif name == "%ns_st":
            return self.ai_talk_interval
        elif name == "%surf0":
            return str(self.surf0)
        elif name == "%surf1":
            return str(self.surf1)
        elif name in ["%plathome", "%platform"]:
            return "ninix"
        elif name == "%move":
            return str(self.motion_count)
        elif name == "%mikire":
            return str(self.mikire)
        elif name == "%kasanari":
            return str(self.kasanari)
        elif name == "%ver":
            if REVISION[1:11] == "Revision: ":
                return "偽栞 for ninix (rev.%s)" % REVISION[11:-2]
            else:
                return "偽栞 for ninix"
        elif name == "%sender":
            if self.sender:
                return self.sender
            else:
                return ''
        elif name == "%ghost":
            if self.to:
                return self.to
            elif len(self.otherghost) > 0:
                return random.choice(self.otherghost)[0]
            else:
                return ''
        return "\\" + name
    def eval_expression(self, expr):
        tree = self.expr_parser.parse(expr)
        if tree is None:
            return None
        return self.interp_expr(tree)
    def interp_expr(self, tree):
        if tree[0] == ExprParser.ADD_EXPR:
            value = self.interp_expr(tree[1])
            for i in range(2, len(tree), 2):
                operand = self.interp_expr(tree[i+1])
                try:
                    if tree[i] == "+":
                        value = int(value) + int(operand)
                    elif tree[i] == "-":
                        value = int(value) - int(operand)
                except ValueError:
                    value = str(value) + tree[i] + str(operand)
            return value
        elif tree[0] == ExprParser.MUL_EXPR:
            value = self.interp_expr(tree[1])
            for i in range(2, len(tree), 2):
                operand = self.interp_expr(tree[i+1])
                try:
                    if tree[i] == "*":
                        value = int(value) * int(operand)
                    elif tree[i] == "/":
                        value = int(value) / int(operand)
                    elif tree[i] == "\\":
                        value = int(value) % int(operand)
                except (ValueError, ZeroDivisionError):
                    value = str(value) + tree[i] + str(operand)
            return value
        elif tree[0] == ExprParser.UNARY_EXPR:
            operand = self.interp_expr(tree[2])
            try:
                if tree[1] == "+":
                    return + int(operand)
                elif tree[1] == "-":
                    return - int(operand)
            except ValueError:
                return tree[1] + str(operand)
        elif tree[0] == ExprParser.PRIMARY_EXPR:
            if self.is_number(tree[1]):
                return int(tree[1])
            elif tree[1][0] == "%":
                return self.expand_meta(tree[1])
            try:
                return self.variables[tree[1]]
            except KeyError:
                return "?"
    def is_number(self, s):
        return s and filter(lambda c: c in "0123456789", s) == s

class ExprError(ValueError):
    pass

class ExprParser:
    def __init__(self):
        self.debug = 0
        # bit 0 = trace get_expr()
        # bit 1 = trace all "get_" functions
    def show_progress(self, func, buffer):
        if buffer is None:
            print "%s() -> syntax error" % func
        else:
            print "%s() -> %s" % (func, buffer)
    re_token = re.compile(r'[()*/\+-]|\d+|\s+')
    def tokenize(self, data):
        buffer = []
        end = 0
        while 1:
            match = NiseShiori.re_meta.match(data, end)
            if match:
                buffer.append(match.group())
                end = match.end()
                continue
            match = self.re_token.match(data, end)
            if match:
                if match.group().strip():
                    buffer.append(match.group())
                end = match.end()
                continue
            match = self.re_token.search(data, end)
            if match:
                buffer.append(data[end:match.start()])
                if match.group().strip():
                    buffer.append(match.group())
                end = match.end()
            else:
                if end < len(data):
                    buffer.append(data[end:])
                break
        return buffer
    def parse(self, data):
        self.tokens = self.tokenize(data)
        try:
            return self.get_expr()
        except ExprError:
            return None # syntax error
    # internal
    def done(self):
        return not self.tokens
    def pop(self):
        try:
            return self.tokens.pop(0)
        except IndexError:
            raise ExprError
    def look_ahead(self, index=0):
        try:
            return self.tokens[index]
        except IndexError:
            raise ExprError
    def match(self, s):
        if self.pop() != s:
            raise ExprError
    # tree node types
    ADD_EXPR     = 1
    MUL_EXPR     = 2
    UNARY_EXPR   = 3
    PRIMARY_EXPR = 4
    def get_expr(self):
        buffer = self.get_add_expr()
        if not self.done():
            raise ExprError
        if self.debug & 3:
            self.show_progress("get_expr", buffer)
        return buffer
    def get_add_expr(self):
        buffer = [self.ADD_EXPR]
        while 1:
            buffer.append(self.get_mul_expr())
            if self.done() or self.look_ahead() not in ["+", "-"]:
                break
            buffer.append(self.pop()) # operator
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_add_expr", buffer)
        return buffer
    def get_mul_expr(self):
        buffer = [self.MUL_EXPR]
        while 1:
            buffer.append(self.get_unary_expr())
            if self.done() or self.look_ahead() not in ["*", "/", "\\"]:
                break
            buffer.append(self.pop()) # operator
        if len(buffer) == 2:
            buffer = buffer[1]
        if self.debug & 2:
            self.show_progress("get_mul_expr", buffer)
        return buffer
    def get_unary_expr(self):
        if self.look_ahead() in ["+", "-"]:
            buffer = [self.UNARY_EXPR, self.pop(), self.get_unary_expr()]
        else:
            buffer = self.get_primary_expr()
        if self.debug & 2:
            self.show_progress("get_unary_expr", buffer)
        return buffer
    def get_primary_expr(self):
        if self.look_ahead() == '(':
            self.match('(')
            buffer = self.get_add_expr()
            self.match(')')
        else:
            buffer = [self.PRIMARY_EXPR, self.pop()]
        if self.debug & 2:
            self.show_progress("get_primary_expr", buffer)
        return buffer

# <<< EXPRESSION SYNTAX >>>
# expr         := add-expr
# add-expr     := mul-expr (add-op mul-expr)*
# add-op       := '+' | '-'
# mul-expr     := unary-expr (mul-op unary-expr)*
# mul-op       := '*' | '/' | '\'
# unary-expr   := unary-op unary-expr | primary-expr
# unary-op     := '+' | '-'
# primary-expr := identifier | constant | '(' add-expr ')'

class Shiori(NiseShiori):
    def __init__(self, dll_name, debug=0):
        NiseShiori.__init__(self, debug)
        self.dll_name = dll_name
    def load(self, dir):
        NiseShiori.load(self, dir)
        return 1
    def unload(self):
        NiseShiori.finalize(self)
    def find(self, dir, dll_name):
        result = 0
        if list_dict(dir):
            result = 100
        return result
    def show_description(self):
        sys.stdout.write('Shiori: NiseShiori compatible module for ninix\n'
                         '        Copyright (C) 2001, 2002 by Tamito KAJIYAMA\n'
                         '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n'
                         '        Copyright (C) 2002-2004 by Shyouzou Sugitani\n'
                         '        Copyright (C) 2003 by Shun-ichi TAHARA\n')
    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        req_header = {}
        line = header.readline()
        if line:
            line = line.strip()
            list = line.split()
            if len(list) >= 2:
                command = list[0].strip()
                protocol = list[1].strip()
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                line = line.strip()
                if not line:
                    continue
                colon = line.find(':')
                if colon >= 0:
                    key = line[:colon].strip()
                    value = line[colon+1:].strip()
                    try:
                        value = int(value)
                    except:
                        value = str(value)
                    req_header[key] = value
                else:
                    continue
        result = ''
        to = None
        if req_header.has_key('ID'):
            if req_header['ID'] == 'charset':
                result = 'UTF-8'
            elif req_header['ID'] == 'dms':
                result = self.getdms()
            elif req_header['ID'] == 'OnAITalk':
                result = self.getaistringrandom()
            elif req_header['ID'] in ["ms", "mz", "ml", "mc", "mh", \
                                      "mt", "me", "mp"]:
                result = self.getword(req_header['ID'])
            elif req_header['ID'] == "m?":
                result = self.getword(random.choice(["ms", "mz", "ml", "mc", "mh","mt", "me", "mp"]))
            elif req_header['ID'] == 'otherghostname':
                otherghost = []
                for n in range(128):
                    if req_header.has_key('Reference'+str(n)):
                        otherghost.append(req_header['Reference'+str(n)])
                result = self.otherghostname(otherghost)
            elif req_header['ID'] == 'OnTeach':
                if req_header.has_key('Reference0'):
                    self.teach(req_header['Reference0'])
            else:
                result = self.getstring(req_header['ID'])
                if result == None:
                    ref = []
                    for n in range(8):
                        if req_header.has_key('Reference'+str(n)):
                            ref.append(req_header['Reference'+str(n)])
                        else:
                            ref.append(None)
                    ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7 = ref
                    result = self.get_event_response(
                        req_header['ID'], ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7)
            if result == None:
                result = ''
            to = self.communicate_to()
        result = 'SHIORI/3.0 200 OK\r\n' \
                 'Sender: Niseshiori\r\n' \
                 'Value: %s\r\n' % result
        if to != None:
            result = result + 'Reference0: %s\r\n' % to
        result += '\r\n'
        return result

def open(shiori_dir, debug=0):
    ns = NiseShiori(debug)
    ns.load(shiori_dir)
    return ns

def test():
    if len(sys.argv) == 2:
        dir = sys.argv[1]
    else:
        dir = os.curdir
    if not list_dict(dir):
        print "no dictionary"
        return
    ns = open(dir, 4+8+16)
    dump_dict(ns)
    while 1:
        print ns.getaistringrandom()
        try:
            raw_input()
        except:
            break

def dump_dict(ns):
    print "DICT"
    for k, v in ns.dict.items():
        if type(k) == type(()):
            k = "(%s, %s)" % k
        print k.encode('UTF-8', 'ignore')
        for e in v: print "\t", e.encode('UTF-8', 'ignore')
    print "*" * 50
    print "CHAINS"
    for t, list in ns.type_chains.items():
        print t.encode('UTF-8', 'ignore')
        for c, w in list:
            print ("-> %s, %s" % (c, w)).encode('UTF_8', 'ignore')
    for key, dict in ns.word_chains.items():
        print ("(%s, %s)" % key).encode('UTF-8', 'ignore')
        for t, list in dict.items():
            prefix = "-> %s" % t.encode('UTF-8', 'ignore')
            for c, w in list:
                print prefix, ("-> %s, %s" % (c, w)).encode('UTF-8', 'ignore')
                prefix = "   " + " " * len(t)
    print "*" * 50
    print "KEYWORDS"
    for (t, w), s in ns.keywords.items():
        print t.encode('UTF-8', 'ignore'), w.encode('UTF-8', 'ignore')
        print "->", s.encode('UTF-8', 'ignore')
    print "*" * 50
    print "RESPONSES"
    for k, v in ns.responses.items():
        print_condition(k)
        print "->", v
    print "*" * 50
    print "GREETINGS"
    for k, v in ns.greetings.items():
        print k.encode('UTF-8', 'ignore')
        print "->", v
    print "*" * 50
    print "EVENTS"
    for k, v in ns.events.items():
        print_condition(k)
        for e in v:
            print "->", e.encode('UTF-8', 'ignore')
    
def print_condition(condition):
    prefix = "Condition"
    for type, expr in condition:
        if type == NiseShiori.COND_COMPARISON:
            print prefix, ("'%s' '%s' '%s'" % expr).encode('UTF-8', 'ignore')
        elif type == NiseShiori.COND_STRING:
            print prefix, "'%s'" % expr.encode('UTF-8', 'ignore')
        prefix = "      and"

if __name__ == "__main__":
    test()
