/*******************************************************************************
 * Copyright (c) 2003, 2015 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Thibault Le Ouay <thibaultleouay@gmail.com> - Bug 457870
 *******************************************************************************/
package org.eclipse.ui.tests.navigator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeFalse;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.navigator.INavigatorContentDescriptor;
import org.eclipse.ui.navigator.INavigatorSorterService;
import org.eclipse.ui.tests.navigator.extension.TestLabelProviderBlue;
import org.eclipse.ui.tests.navigator.extension.TestLabelProviderCyan;
import org.eclipse.ui.tests.navigator.extension.TestLabelProviderStyledGreen;
import org.eclipse.ui.tests.navigator.extension.TestSimpleChildrenContentProvider;
import org.eclipse.ui.tests.navigator.extension.TestSorterData;
import org.junit.Test;

public class ProgrammaticOpenTest extends NavigatorTestBase {

	public ProgrammaticOpenTest() {

		_navigatorInstanceId = TEST_VIEWER_PROGRAMMATIC;
	}

	@Test
	public void testNavigatorRootContents() throws Exception {

		assertEquals("There should be no visible extensions for the programmatic viewer.", 0,
				_contentService.getVisibleExtensionIds().length);

		_contentService.bindExtensions(new String[] { COMMON_NAVIGATOR_RESOURCE_EXT }, false);

		assertEquals("There should be one visible extension for the programmatic viewer.", 1,
				_contentService.getVisibleExtensionIds().length);

		_contentService.getActivationService().activateExtensions(
				new String[] { COMMON_NAVIGATOR_RESOURCE_EXT }, true);

		refreshViewer();
		_viewer.expandAll();

		TreeItem[] items = _viewer.getTree().getItems();

		assertTrue("There should be some items.", items.length > 0); //$NON-NLS-1$

		assertEquals(_project, items[_projectInd].getData());

		// we do this to force the rendering of the children of items[0]
		_viewer.setSelection(new StructuredSelection(_project.getFile(".project"))); //$NON-NLS-1$

		TreeItem[] children = items[_projectInd].getItems();

		assertEquals(_expectedChildren.size(), children.length);
		for (TreeItem child : children) {
			assertTrue(_expectedChildren.contains(child.getData()));
		}

	}

	@Test
	public void testNavigatorExtensionEnablement() throws Exception {

		assertEquals("There should be no visible extensions for the programmatic viewer.", 0,
				_contentService.getVisibleExtensionIds().length);

		_contentService.bindExtensions(new String[] { COMMON_NAVIGATOR_RESOURCE_EXT }, false);

		assertEquals("There should be one visible extension for the programmatic viewer.", 1,
				_contentService.getVisibleExtensionIds().length);

		_contentService.getActivationService().activateExtensions(
				new String[] { COMMON_NAVIGATOR_RESOURCE_EXT }, true);

		_contentService.getActivationService().activateExtensions(new String[] {}, true);

		refreshViewer();
		_viewer.expandAll();

		TreeItem[] items = _viewer.getTree().getItems();

		assertTrue("There should be NO items.", items.length == 0); //$NON-NLS-1$

		_contentService.getActivationService().deactivateExtensions(new String[] {}, true);

		_viewer.expandToLevel(2);

		items = _viewer.getTree().getItems();

		assertTrue("There should be some items.", items.length > 0); //$NON-NLS-1$

	}

	@Test
	public void testNavigatorSorterAccess() throws Exception {

		INavigatorSorterService sorterService = _contentService.getSorterService();

		INavigatorContentDescriptor testDescriptor = _contentService
				.getContentDescriptorById(TEST_CONTENT);

		Map sorters = sorterService.findAvailableSorters(testDescriptor);

		assertEquals(1, sorters.size());

		for (Iterator iter = sorters.entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();
			assertTrue(entry.getValue() instanceof TestSorterData);
		}

	}

	// bug 299438 things are not right after the active NCEs are changed
	@Test
	public void testChangeNceActivation() throws Exception {
		_contentService.bindExtensions(new String[] { TEST_CONTENT_OVERRIDDEN1,
				TEST_CONTENT_OVERRIDE1, TEST_CONTENT_OVERRIDDEN2, TEST_CONTENT_OVERRIDE2 }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] { TEST_CONTENT_OVERRIDDEN2, TEST_CONTENT_OVERRIDE2 }, true);

		refreshViewer();

		TreeItem[] rootItems = _viewer.getTree().getItems();
		checkItems(rootItems, TestLabelProviderCyan.instance);

		_contentService.bindExtensions(new String[] { TEST_CONTENT_OVERRIDDEN1,
				TEST_CONTENT_OVERRIDE1 }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] { TEST_CONTENT_OVERRIDDEN1, TEST_CONTENT_OVERRIDE1 }, true);

		refreshViewer();

		_viewer.expandAll();

		rootItems = _viewer.getTree().getItems();
		checkItems(rootItems, TestLabelProviderStyledGreen.instance);
	}

	// bug 296728 expression evaluation does not support with
	@Test
	public void testEvaluateWith() throws Exception {
		assumeFalse("Test fails on Mac: Bug 537641", Platform.OS_MACOSX.equals(Platform.getOS()));

		_contentService.bindExtensions(new String[] { COMMON_NAVIGATOR_RESOURCE_EXT,
				TEST_CONTENT_WITH, }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] { COMMON_NAVIGATOR_RESOURCE_EXT, TEST_CONTENT_WITH }, true);

		refreshViewer();

		TreeItem[] rootItems = _viewer.getTree().getItems();
		checkItems(rootItems, TestLabelProviderBlue.instance);
	}

	// bug 228022 add ordering of content
	@Test
	public void testNceOrdering() throws Exception {
		_contentService.bindExtensions(new String[] {
				COMMON_NAVIGATOR_RESOURCE_EXT, TEST_SIMPLE_CHILDREN1 }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] {  COMMON_NAVIGATOR_RESOURCE_EXT, TEST_SIMPLE_CHILDREN1 }, true);

		TreeItem[] rootItems = _viewer.getTree().getItems();
		_viewer.expandAll();

		assertEquals("10", rootItems[0].getItems()[0].getText());

		_contentService.bindExtensions(new String[] { TEST_SIMPLE_CHILDREN3 }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] { TEST_SIMPLE_CHILDREN3 }, false);

		_viewer.expandAll();

		assertEquals("30", rootItems[0].getItems()[0].getText());
		assertEquals("10", rootItems[0].getItems()[TestSimpleChildrenContentProvider.NUM_ITEMS]
				.getText());

		_contentService.bindExtensions(new String[] { TEST_SIMPLE_CHILDREN2 }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] { TEST_SIMPLE_CHILDREN2 }, false);

		_viewer.expandAll();

		assertEquals("30", rootItems[0].getItems()[0].getText());
		assertEquals("20", rootItems[0].getItems()[TestSimpleChildrenContentProvider.NUM_ITEMS]
				.getText());
		assertEquals("10", rootItems[0].getItems()[TestSimpleChildrenContentProvider.NUM_ITEMS * 2]
				.getText());

	}

	// bug 349224 Navigator content provider "appearsBefore" creates hard reference to named id
	@Test
	public void testNceNotFound() throws Exception {
		_contentService.bindExtensions(new String[] {
				COMMON_NAVIGATOR_RESOURCE_EXT, TEST_SIMPLE_CHILDREN_NOT_FOUND }, false);
		_contentService.getActivationService().activateExtensions(
				new String[] {  COMMON_NAVIGATOR_RESOURCE_EXT, TEST_SIMPLE_CHILDREN_NOT_FOUND }, true);
	}



}
