/*
Jaxe - Editeur XML en Java

Copyright (C) 2008 Observatoire de Paris-Meudon

Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier conformment aux dispositions de la Licence Publique Gnrale GNU, telle que publie par la Free Software Foundation ; version 2 de la licence, ou encore ( votre choix) toute version ultrieure.

Ce programme est distribu dans l'espoir qu'il sera utile, mais SANS AUCUNE GARANTIE ; sans mme la garantie implicite de COMMERCIALISATION ou D'ADAPTATION A UN OBJET PARTICULIER. Pour plus de dtail, voir la Licence Publique Gnrale GNU .

Vous devez avoir reu un exemplaire de la Licence Publique Gnrale GNU en mme temps que ce programme ; si ce n'est pas le cas, crivez  la Free Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, Etats-Unis.
*/

package jaxe;


import java.util.ArrayList;
import java.util.Set;

import org.w3c.dom.Element;


/**
 * Interface pour un langage de schma, comme celui du W3C, Relax NG, ou les schmas simples de Jaxe.
 * Utilise la notion de "rfrence d'lment" qui correspond  l'lment du schma qui dfinit l'lment XML
 * (cela suppose que les schmas sont eux-mmes des arbres XML).
 */
public interface InterfaceSchema {
    
    /**
     * Renvoie true si la rfrence vient de ce schma
     */
    public boolean elementDansSchema(final Element refElement);
    
    /**
     * Renvoie la rfrence du premier lment du schma avec le nom donn.
     */
    public Element referenceElement(final String nom);
    
    /**
     * Renvoie la rfrence du premier lment du schma avec le nom et l'espace de noms de l'lment pass en paramtre.
     */
    public Element referenceElement(final Element el);
    
    /**
     * Renvoie la rfrence du premier lment du schma avec le nom et l'espace de noms de l'lment pass en paramtre,
     * et avec le parent dont la rfrence est passe en paramtre.
     */
    public Element referenceElement(final Element el, final Element refParent);
    
    /**
     * Renvoie le nom de l'lment dont la rfrence est donne.
     */
    public String nomElement(final Element refElement);
    
    /**
     * Renvoie l'espace de nom de l'lment dont la rfrence est passe en paramtre,
     * ou null si l'espace de noms n'est pas dfini.
     */
    public String espaceElement(final Element refElement);
    
    /**
     * Renvoie le prfixe  utiliser pour crer un lment dont on donne la rfrence,
     * ou null s'il n'y en a pas.
     */
    public String prefixeElement(final Element refElement);
    
    /**
     * Renvoie la documentation d'un lment dont on donne la rfrence
     * (sous forme de texte simple, avec des \n pour faire des sauts de lignes)
     */
    public String documentationElement(final Element refElement);
    
    /**
     * Renvoie la liste des valeurs possibles pour un lment,  partir de sa rfrence.
     * Renvoie null s'il y a un nombre infini de valeurs possibles ou si l'lment n'a pas un type simple.
     */
    public ArrayList<String> listeValeursElement(final Element refElement);
    
    /**
     * Renvoie true si la valeur donne est une valeur valide pour l'lment
     */
    public boolean valeurElementValide(final Element refElement, final String valeur);
    
    /**
     * Renvoie la liste des espaces de noms (String) utiliss par ce schma.
     */
    public ArrayList<String> listeEspaces();
    
    /**
     * Renvoie true si l'espace de nom est dfini dans le schma
     */
    public boolean aEspace(final String espace);
    
    /**
     * Renvoie un prfixe  utiliser pour l'espace de noms donn, ou null si aucune suggestion n'est possible
     */
    public String prefixeEspace(final String ns);
    
    /**
     * Renvoie l'espace de noms cible du schma (attribut targetNamespace avec WXS).
     * Attention: Le concept d'un espace cible unique pour un schma n'existe pas avec Relax NG.
     */
    public String espaceCible();
    
    /**
     * Renvoie les rfrences des lments qui ne sont pas dans l'espace de noms pass en paramtre
     */
    public ArrayList<Element> listeElementsHorsEspace(final String espace);
    
    /**
     * Renvoie les rfrences des lments qui sont dans les espaces de noms passs en paramtre
     */
    public ArrayList<Element> listeElementsDansEspaces(final Set<String> espaces);
    
    /**
     * Renvoie les rfrences de tous les lments du schma
     */
    public ArrayList<Element> listeTousElements();
    
    /**
     * Renvoit true si l'enfant est obligatoire sous le parent.
     */
    public boolean elementObligatoire(final Element refParent, final Element refEnfant);
    
    /**
     * Renvoit true si le parent peut avoir des enfants multiples avec la rfrence refEnfant.
     */
    public boolean enfantsMultiples(final Element refParent, final Element refEnfant);
    
    /**
     * Renvoie les rfrences des lments enfants de l'lment dont la rfrence est passe en paramtre
     */
    public ArrayList<Element> listeSousElements(final Element refParent);
    
    /**
     * Expression rgulire correspondant au schma pour un lment parent donn
     * @param modevisu  True si on cherche une expression rgulire  afficher pour l'utilisateur
     * @param modevalid  Pour obtenir une validation stricte au lieu de chercher si une insertion est possible
     */
    public String expressionReguliere(final Element refParent, final boolean modevisu, final boolean modevalid);
    
    /**
     * Renvoie la liste des rfrences des parents possibles pour un lment dont la rfrence est passe en paramtre
     */
    public ArrayList<Element> listeElementsParents(final Element refElement);
    
    /**
     * Renvoie la liste des rfrences des attributs possibles pour un lment dont
     * on donne la rfrence en paramtre
     */
    public ArrayList<Element> listeAttributs(final Element refElement);
    
    /**
     * Renvoie le nom d'un attribut  partir de sa rfrence
     */
    public String nomAttribut(final Element refAttribut);
    
    /**
     * Renvoie l'espace de noms d'un attribut  partir de sa rfrence, ou null si aucun n'est dfini
     */
    public String espaceAttribut(final Element refAttribut);
    
    /**
     * Renvoie la documentation d'un attribut  partir de sa rfrence
     */
    public String documentationAttribut(final Element refAttribut);
    
    /**
     * Renvoie l'espace de noms d'un attribut  partir de son nom complet (avec le prfixe s'il y en a un)
     */
    public String espaceAttribut(final String nomAttribut);
    
    /**
     * Renvoie true si un attribut est obligatoire,  partir de sa dfinition.
     * Attention: ce n'est pas possible  dterminer avec seulement la rfrence d'attribut avec Relax NG.
     * Il vaut mieux utiliser attributObligatoire.
     */
    @Deprecated
    public boolean estObligatoire(final Element refAttribut);
    
    /**
     * Renvoit true si l'attribut est obligatoire sous le parent.
     */
    public boolean attributObligatoire(final Element refParent, final Element refAttribut);
    
    /**
     * Renvoie la liste des valeurs possibles pour un attribut,  partir de sa rfrence.
     * Renvoie null s'il y a un nombre infini de valeurs possibles.
     */
    public ArrayList<String> listeValeursAttribut(final Element refAttribut);
    
    /**
     * Renvoie la valeur par dfaut d'un attribut dont la rfrence est donne en paramtre
     */
    public String valeurParDefaut(final Element refAttribut);
    
    /**
     * Renvoie true si la valeur donne est une valeur valide pour l'attribut
     */
    public boolean attributValide(final Element refAttribut, final String valeur);
    
    /**
     * Renvoie la rfrence du premier lment parent d'un attribut  partir de sa rfrence
     */
    public Element parentAttribut(final Element refAttribut);
    
    /**
     * Renvoie true si l'lment dont on donne la rfrence peut contenir du texte
     */
    public boolean contientDuTexte(final Element refElement);
    
}
