/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

/* eslint-disable no-restricted-syntax */

// MODULES //

var tape = require( 'tape' );
var abs = require( './../../../../../base/special/abs' );
var continuedFraction = require( './../../lib/' );


// TESTS //

tape( 'if provided a generator function, the function evaluates the corresponding continued fraction without the leading b_0 term', function test( t ) {
	// Continued fraction for (e-1)^(-1):
	var expected = 0.5819767068693261;
	var actual = continuedFraction( generator() );

	t.ok( abs( actual - expected ) < 1e-14, 'returned result is within tolerance. actual: ' + actual + '; expected: ' + expected + '.' );
	t.end();

	function* generator() {
		var i = 0;
		while ( true ) {
			i += 1;
			yield [ i, i ];
		}
	}
});

tape( 'if provided a generator function, the function evaluates the corresponding continued fraction with the leading b_0 term', function test( t ) {
	var expected = 0.6977746579640078;
	var actual = continuedFraction( generator1(), {
		'keep': true
	});

	t.ok( abs( actual - expected ) < 1e-14, 'returned result is within tolerance. actual: ' + actual + '; expected: ' + expected + '.' );

	actual = continuedFraction( generator2(), {
		'keep': true
	});
	expected = 1.525135276160983;

	t.ok( abs( actual - expected ) < 1e-14, 'returned result is within tolerance. actual: ' + actual + '; expected: ' + expected + '.' );

	t.end();

	// Continued fraction for (I_1(2))/(I_0(2)), see http://mathworld.wolfram.com/ContinuedFraction.html:
	function* generator1() {
		var b = 0;
		yield [ 1, b ];
		while ( true ) {
			b += 1;
			yield [ 1, b ];
		}
	}

	// Continued fraction for sqrt(2/(epi))[erfc(2^(-1/2))]^(-1), see http://mathworld.wolfram.com/ContinuedFraction.html:
	function* generator2() {
		var a = 0;
		yield [ a, 1 ];
		while ( true ) {
			a += 1;
			yield [ a, 1 ];
		}
	}
});
