/*
 * Refer AMD Am29F010 CMOS Sector Erase Flash Memory Data Sheet.
 */
#include <asm-i386/uaccess.h>
#include "wlapi.h"
#include "wlioctl.h"


/* Hack some linux functions */
#ifdef __FreeBSD__
   extern const char g_cpszVersion[];
   struct zwl_softc;
   extern int zwl_reset(struct zwl_softc *sc);
#  define WL_Reset(sc)      zwl_reset(sc)
#  define memcpy_tofs(pDest, pSrc, ulLen)       copyout(pSrc, pDest, ulLen)
#  define memcpy_fromfs(pDest, pSrc, ulLen)     copyin(pSrc, pDest, ulLen)
#  define suser()   (suser(curproc->p_ucred, &curproc->p_acflag) == 0)
#  include <sys/kernel.h>       /* for ticks */
#  define WL_DELTA_TIME(x)      ((x) = ticks - (x))

#  define VERIFY_READ   0
#  define VERIFY_WRITE  1

static int verify_area(int nOperation, const void *pData, unsigned long ulLen)
{
    /* always success */
    return 0;
}

#else /* For Linux */

#  include "wl24.h"
/* extern const char g_cpszVersion[]; */
/* extern int WL_Reset(struct device *dev); */

#  define WL_DELTA_TIME(x)      ((x) = jiffies - (x))
#  define VERIFY_READ   0
#  define VERIFY_WRITE  1
#  define memcpy_tofs           copy_to_user
#  define memcpy_fromfs         copy_from_user

/*
 * Perform IOCTL call functions here. Some are privileged operations and the
 * effective uid is checked in those cases.
 *
 * This part is optional. Needed only if you want to run wlu (unix version).
 *
 * CAUTION: To prevent interrupted by WL_Interrupt() and timer-based 
 * WL_StartXmit() from other interrupts, this should be run single-threaded.
 * This function is expected to be a rare operation, and it's 
 * simpler to just use cli() to disable ALL interrupts.
 */
int WL_Ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    WL_Adaptor      *pAdaptor = (WL_Adaptor *) dev->priv;
    WL_IOCTL_BLK    *pBlk = (WL_IOCTL_BLK *) &rq->ifr_data;
    unsigned long   ulFlags;
    int             nReturn;

    /* Keep flags first, these are inline functions */
    save_flags(ulFlags);
    cli();

    nReturn = WL_IoctlWla(pAdaptor, pBlk, dev);
    
    /* Restore CLI and others */
    restore_flags(ulFlags);

    return nReturn;
}

#endif /* Linux End */

/*
 * Common ioctl part for both Linux and FreeBSD
 * Return 0 if success.
 */
int WL_IoctlWla(WL_Adaptor *pAdaptor, WL_IOCTL_BLK *pBlk, void *pDevice)
{
    int  nReturn;
    long lParam;
    
    switch(pBlk->uCmd) {
      case WL_IOCTL_GET_VERSION:    /* Get driver's version */
        pBlk->uLen = strlen(g_cpszVersion);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, g_cpszVersion, pBlk->uLen);
        break;

      case WL_IOCTL_SET_RESET:      /* Reset driver (needed after set) */
        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = WL_Reset(pDevice);
        break;

      case WL_IOCTL_SET_FLASH:      /* Write current setting into Flash ROM */
        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = WL_SetMibInFlash(pAdaptor) ? 0 : -EIO;
        break;

#ifdef WLAPI_DEBUG            
      case WL_IOCTL_GET_DBGFLAG:    /* (Only if with debug enabled) */
        pBlk->uLen = sizeof(DWORD);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &WL_DbFlag, pBlk->uLen);
        break;

      case WL_IOCTL_SET_DBGFLAG:    /* (Only if with debug enabled) */
        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(WL_DbFlag));
        if (nReturn == 0) {
            memcpy_fromfs(&WL_DbFlag, pBlk->pData, sizeof(WL_DbFlag));
        }
        break;
#endif

      case WL_IOCTL_GET_CHANNEL: {  /* Get channel (1-12)  */
        BYTE    cChannel;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_CHANNEL, &cChannel, sizeof(cChannel))) {
            nReturn = -EIO;
            break;
        }
        
        lParam = cChannel;
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_CHANNEL: {  /* Set channel (1-12)  */
        BYTE    cChannel;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;
            
        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));
        
        cChannel = lParam;
        if (cChannel < 1 || cChannel > 12) {
            nReturn = -EINVAL;
            break;
        }
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_CHANNEL, &cChannel, sizeof(cChannel))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }


      case WL_IOCTL_GET_DOMAIN: {  /* Get freqDomain (0-2)  */
        BYTE    cDomain;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_FREQDOMAIN, &cDomain, sizeof(cDomain))) {
            nReturn = -EIO;
            break;
        }
        
        lParam = cDomain;
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_DOMAIN: {  /* Set freqDomain (0-2)  */
        BYTE    cDomain;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;
            
        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));
        
        cDomain = lParam;
        if ( cDomain > 2 ) {
            nReturn = -EINVAL;
            break;
        }
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_FREQDOMAIN, &cDomain, sizeof(cDomain))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }


      case WL_IOCTL_GET_SPEED: {    /* Get speed (1 Mbps / 2 Mbps)  */
        BYTE    cSpeed;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SPEED, &cSpeed, sizeof(cSpeed))) {
            nReturn = -EIO;
            break;
        }
        
        lParam = cSpeed;
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_SPEED: {     /* Set speed (1 Mbps / 2 Mbps)  */
        BYTE    cSpeed;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;

        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));
        
        cSpeed = lParam;
        if (cSpeed != 1 && cSpeed != 2) {
            nReturn = -EINVAL;
            break;
        }
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SPEED, &cSpeed, sizeof(cSpeed))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_MODE: {     /* Get mode (1=AdHoc / 2=infra.)*/
        BYTE    cCCR;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_CCR, &cCCR, sizeof(cCCR))) {
            nReturn = -EIO;
            break;
        }
        
        lParam = cCCR;
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_MODE: {     /* Set mode (1=AdHoc / 2=infra.)*/
        BYTE    cCCR;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;

        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));
        
        cCCR = lParam;
        if (cCCR > CCR_IAP) {
            nReturn = -EINVAL;
            break;
        }
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_CCR, &cCCR, sizeof(cCCR))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_ROAMING: {  /* Get roaming (0=off / 1=on) */
        BYTE    cRoamingEnable;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_ROAMINGENABLE, &cRoamingEnable, sizeof(cRoamingEnable))) {
            nReturn = -EIO;
            break;
        }
        
        lParam = cRoamingEnable;
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_ROAMING: {  /* Set roaming (0=off / 1=on) */
        BYTE    cRoamingEnable;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;
        
        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));
        
        cRoamingEnable = lParam;
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_ROAMINGENABLE, &cRoamingEnable, sizeof(cRoamingEnable))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_BSSID: {    /* Get BSSID (0<BSSID<FF FF FF FF FF FF */
        MADDR    BSSID;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_BSSID, &BSSID, sizeof(BSSID))) {
            nReturn = -EIO;
            break;
        }
        
        pBlk->uLen = sizeof(BSSID);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &BSSID, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_BSSID: {    /* Set BSSID (0<BSSID<FF FF FF FF FF FF */
        MADDR   BSSID;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(BSSID));
        if (nReturn != 0)
            break;

        memcpy_fromfs(&BSSID, pBlk->pData, sizeof(BSSID));
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_BSSID, &BSSID, sizeof(BSSID))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_ESSID: {    /* Get ESSID (0<ESSID<999999)   */
        MADDR    ESSID;

        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_ESSID, &ESSID, sizeof(ESSID))) {
            nReturn = -EIO;
            break;
        }
        
        pBlk->uLen = sizeof(ESSID);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &ESSID, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_ESSID: {    /* Set ESSID (0<ESSID<999999)   */
        MADDR   ESSID;

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(ESSID));
        if (nReturn != 0)
            break;

        memcpy_fromfs(&ESSID, pBlk->pData, sizeof(ESSID));
        
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_ESSID, &ESSID, sizeof(ESSID))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_SECURITY: { /* Get SECURITY (0-999999=on / -1=off) */
        BYTE    SecurityEnable;
        BYTE    SecurityKey[4];

        /* Only root could query security key */
        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SECURITYENABLE, &SecurityEnable, sizeof(SecurityEnable))) {
            nReturn = -EIO;
            break;
        }
        
        if (SecurityEnable) {
            if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SECURITYKEY, &SecurityKey, sizeof(SecurityKey))) {
                nReturn = -EIO;
                break;
            }
            
            lParam = *((long *) SecurityKey);
        }
        else
            lParam = -1;
        
        pBlk->uLen = sizeof(lParam);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &lParam, pBlk->uLen);
        break;
      }

      case WL_IOCTL_SET_SECURITY: { /* Get SECURITY (0-999999=on / -1=off) */
        BYTE    SecurityEnable;
        BYTE    SecurityKey[4];

        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(lParam));
        if (nReturn != 0)
            break;

        memcpy_fromfs(&lParam, pBlk->pData, sizeof(lParam));

        SecurityEnable = lParam == -1 ? 0 : 1;
        
        if (SecurityEnable) {
            if (lParam < 0 || lParam > 999999) {
                nReturn = -EINVAL;
                break;
            }
            
            *((long *) SecurityKey) = lParam;
            
            if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SECURITYKEY, &SecurityKey, sizeof(SecurityKey))) {
                nReturn = -EIO;
                break;
            }
        }
        
        /* Set SecurityEnable after SecurityKey prepared */
        if (!WL_SetMibValue(pAdaptor, TYPE_EXTRA_MIB, IDX_SECURITYENABLE, &SecurityEnable, sizeof(SecurityEnable))) {
            nReturn = -EIO;
            break;
        }
        
        /* Replace Extra MIB gotten when initializing API */
        if (!WL_GetMibValue(pAdaptor, TYPE_EXTRA_MIB, 0, 
            &pAdaptor->mibExtra, sizeof(pAdaptor->mibExtra))) {
            nReturn = -EIO;
            break;
        }
        
        nReturn = 0;
        break;
      }

      case WL_IOCTL_GET_MONITOR: {      /* Get Monitor Structure */
        WL_Monitor monitor = pAdaptor->Monitor;
        WL_DELTA_TIME(monitor.TimeStamp);
        pBlk->uLen = sizeof(monitor);
        nReturn = verify_area(VERIFY_WRITE, (void *)pBlk->pData, pBlk->uLen);
        if (nReturn == 0)
            memcpy_tofs(pBlk->pData, &monitor, pBlk->uLen);
        break;
      }
      
      case WL_IOCTL_SET_MONITOR:        /* Set Monitor.MAddr */
        if (!suser()) {
            nReturn = -EPERM;
            break;
        }
        
        nReturn = verify_area(VERIFY_READ, (void *)pBlk->pData, sizeof(MADDR));
        if (nReturn == 0)
            memcpy_fromfs(&pAdaptor->Monitor.MAddr, pBlk->pData, sizeof(MADDR));
        break;

      default:
    	nReturn = -EOPNOTSUPP;
    }
    
    return nReturn;
}

