/*
 *	Crystal SoundFusion CS46xx driver
 *
 *	Copyright 1999-2000 Jaroslav Kysela <perex@suse.cz>
 *	Copyright 2000 Alan Cox <alan@redhat.com>
 *
 *	The core of this code is taken from the ALSA project driver by 
 *	Jaroslav. Please send Jaroslav the credit for the driver and 
 *	report bugs in this port to <alan@redhat.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Changes:
 *	20000815	Updated driver to kernel 2.4, some cleanups/fixes
 *			Nils Faerber <nils@kernelconcepts.de>
 *
 */
 
#include <linux/module.h>
#include <linux/version.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/sound.h>
#include <linux/malloc.h>
#include <linux/soundcard.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/spinlock.h>
#include <linux/ac97_codec.h>
#include <linux/wrapper.h>
#include <asm/uaccess.h>
#include <asm/hardirq.h>

#include "cs461x.h"

#define ADC_RUNNING	1
#define DAC_RUNNING	2

#define CS_FMT_16BIT	1		/* These are fixed in fact */
#define CS_FMT_STEREO	2
#define CS_FMT_MASK	3

/*
 *	CS461x definitions
 */
 
#define CS461X_BA0_SIZE		0x2000
#define CS461X_BA1_DATA0_SIZE	0x3000
#define CS461X_BA1_DATA1_SIZE	0x3800
#define CS461X_BA1_PRG_SIZE	0x7000
#define CS461X_BA1_REG_SIZE	0x0100

#define GOF_PER_SEC	200

/*
 * Define this to enable recording,
 * this is curently broken and using it will cause data corruption
 * in kernel- and user-space!
 */
/* #define CS46XX_ENABLE_RECORD */

static int external_amp = 0;
static int thinkpad = 0;


/* an instance of the 4610 channel */

struct cs_channel 
{
	int used;
	int num;
	void *state;
};

#define DRIVER_VERSION "0.09"

/* magic numbers to protect our data structures */
#define CS_CARD_MAGIC		0x46524F4D /* "FROM" */
#define CS_STATE_MAGIC		0x414c5341 /* "ALSA" */
#define NR_HW_CH		3

/* maxinum number of AC97 codecs connected, AC97 2.0 defined 4 */
#define NR_AC97		2

/* minor number of /dev/dspW */
#define SND_DEV_DSP8	1

/* minor number of /dev/dspW */
#define SND_DEV_DSP16	1 

static const unsigned sample_size[] = { 1, 2, 2, 4 };
static const unsigned sample_shift[] = { 0, 1, 1, 2 };

/* "software" or virtual channel, an instance of opened /dev/dsp */
struct cs_state {
	unsigned int magic;
	struct cs_card *card;	/* Card info */

	/* single open lock mechanism, only used for recording */
	struct semaphore open_sem;
	wait_queue_head_t open_wait;

	/* file mode */
	mode_t open_mode;

	/* virtual channel number */
	int virt;
	
	struct dmabuf {
		/* wave sample stuff */
		unsigned int rate;
		unsigned char fmt, enable;

		/* hardware channel */
		struct cs_channel *channel;
		int pringbuf;		/* Software ring slot */
		int ppingbuf;		/* Hardware ring slot */
		void *pbuf;		/* 4K hardware DMA buffer */

		/* OSS buffer management stuff */
		void *rawbuf;
		dma_addr_t dma_handle;
		unsigned buforder;
		unsigned numfrag;
		unsigned fragshift;

		/* our buffer acts like a circular ring */
		unsigned hwptr;		/* where dma last started, updated by update_ptr */
		unsigned swptr;		/* where driver last clear/filled, updated by read/write */
		int count;		/* bytes to be comsumed or been generated by dma machine */
		unsigned total_bytes;	/* total bytes dmaed by hardware */

		unsigned error;		/* number of over/underruns */
		wait_queue_head_t wait;	/* put process on wait queue when no more space in buffer */

		/* redundant, but makes calculations easier */
		unsigned fragsize;
		unsigned dmasize;
		unsigned fragsamples;

		/* OSS stuff */
		unsigned mapped:1;
		unsigned ready:1;
		unsigned endcleared:1;
		unsigned update_flag;
		unsigned ossfragshift;
		int ossmaxfrags;
		unsigned subdivision;
	} dmabuf;
};


struct cs_card {
	struct cs_channel channel[2];
	unsigned int magic;

	/* We keep cs461x cards in a linked list */
	struct cs_card *next;

	/* The cs461x has a certain amount of cross channel interaction
	   so we use a single per card lock */
	spinlock_t lock;

	/* PCI device stuff */
	struct pci_dev * pci_dev;

	unsigned int pctl, cctl;	/* Hardware DMA flag sets */

	/* soundcore stuff */
	int dev_audio;

	/* structures for abstraction of hardware facilities, codecs, banks and channels*/
	struct ac97_codec *ac97_codec[NR_AC97];
	struct cs_state *states[NR_HW_CH];

	u16 ac97_features;

	int amplifier;			/* Amplifier control */
	void (*amplifier_ctrl)(struct cs_card *, int);

	int active;			/* Active clocking */
	void (*active_ctrl)(struct cs_card *, int);
	
	/* hardware resources */
	unsigned long ba0_addr;
	unsigned long ba1_addr;
	u32 irq;
	
	/* mappings */
	void *ba0;
	union
	{
		struct
		{
			u8 *data0;
			u8 *data1;
			u8 *pmem;
			u8 *reg;
		} name;
		u8 *idx[4];
	} ba1;
	
	/* Function support */
	struct cs_channel *(*alloc_pcm_channel)(struct cs_card *);
	struct cs_channel *(*alloc_rec_pcm_channel)(struct cs_card *);
	void (*free_pcm_channel)(struct cs_card *, int chan);
};

static struct cs_card *devs = NULL;

static int cs_open_mixdev(struct inode *inode, struct file *file);
static int cs_release_mixdev(struct inode *inode, struct file *file);
static int cs_ioctl_mixdev(struct inode *inode, struct file *file, unsigned int cmd,
				unsigned long arg);
static loff_t cs_llseek(struct file *file, loff_t offset, int origin);

extern __inline__ unsigned ld2(unsigned int x)
{
	unsigned r = 0;
	
	if (x >= 0x10000) {
		x >>= 16;
		r += 16;
	}
	if (x >= 0x100) {
		x >>= 8;
		r += 8;
	}
	if (x >= 0x10) {
		x >>= 4;
		r += 4;
	}
	if (x >= 4) {
		x >>= 2;
		r += 2;
	}
	if (x >= 2)
		r++;
	return r;
}


/*
 *  common I/O routines
 */

static void cs461x_poke(struct cs_card *codec, unsigned long reg, unsigned int val)
{
	writel(val, codec->ba1.idx[(reg >> 16) & 3]+(reg&0xffff));
}

static unsigned int cs461x_peek(struct cs_card *codec, unsigned long reg)
{
	return readl(codec->ba1.idx[(reg >> 16) & 3]+(reg&0xffff));
}

static void cs461x_pokeBA0(struct cs_card *codec, unsigned long reg, unsigned int val)
{
	writel(val, codec->ba0+reg);
}

static unsigned int cs461x_peekBA0(struct cs_card *codec, unsigned long reg)
{
	return readl(codec->ba0+reg);
}


static u16 cs_ac97_get(struct ac97_codec *dev, u8 reg);
static void cs_ac97_set(struct ac97_codec *dev, u8 reg, u16 data);

static struct cs_channel *cs_alloc_pcm_channel(struct cs_card *card)
{
	if(card->channel[1].used==1)
		return NULL;
	card->channel[1].used=1;
	card->channel[1].num=1;
	return &card->channel[1];
}

static struct cs_channel *cs_alloc_rec_pcm_channel(struct cs_card *card)
{
	if(card->channel[0].used==1)
		return NULL;
	card->channel[0].used=1;
	card->channel[0].num=0;
	return &card->channel[0];
}

static void cs_free_pcm_channel(struct cs_card *card, int channel)
{
	card->channel[channel].state = NULL;
	card->channel[channel].used=0;
}

/* set playback sample rate */
static unsigned int cs_set_dac_rate(struct cs_state * state, unsigned int rate)
{	
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned int tmp1, tmp2;
	unsigned int phiIncr;
	unsigned int correctionPerGOF, correctionPerSec;

	/*
	 *  Compute the values used to drive the actual sample rate conversion.
	 *  The following formulas are being computed, using inline assembly
	 *  since we need to use 64 bit arithmetic to compute the values:
	 *
	 *  phiIncr = floor((Fs,in * 2^26) / Fs,out)
	 *  correctionPerGOF = floor((Fs,in * 2^26 - Fs,out * phiIncr) /
         *                                   GOF_PER_SEC)
         *  ulCorrectionPerSec = Fs,in * 2^26 - Fs,out * phiIncr -M
         *                       GOF_PER_SEC * correctionPerGOF
	 *
	 *  i.e.
	 *
	 *  phiIncr:other = dividend:remainder((Fs,in * 2^26) / Fs,out)
	 *  correctionPerGOF:correctionPerSec =
	 *      dividend:remainder(ulOther / GOF_PER_SEC)
	 */
	tmp1 = rate << 16;
	phiIncr = tmp1 / 48000;
	tmp1 -= phiIncr * 48000;
	tmp1 <<= 10;
	phiIncr <<= 10;
	tmp2 = tmp1 / 48000;
	phiIncr += tmp2;
	tmp1 -= tmp2 * 48000;
	correctionPerGOF = tmp1 / GOF_PER_SEC;
	tmp1 -= correctionPerGOF * GOF_PER_SEC;
	correctionPerSec = tmp1;

	/*
	 *  Fill in the SampleRateConverter control block.
	 */
	 
	spin_lock_irq(&state->card->lock);
	cs461x_poke(state->card, BA1_PSRC,
	  ((correctionPerSec << 16) & 0xFFFF0000) | (correctionPerGOF & 0xFFFF));
	cs461x_poke(state->card, BA1_PPI, phiIncr);
	spin_unlock_irq(&state->card->lock);
	dmabuf->rate = rate;
	
	return rate;
}

/* set recording sample rate */
static unsigned int cs_set_adc_rate(struct cs_state * state, unsigned int rate)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned int phiIncr, coeffIncr, tmp1, tmp2;
	unsigned int correctionPerGOF, correctionPerSec, initialDelay;
	unsigned int frameGroupLength, cnt;

	/*
	 *  We can only decimate by up to a factor of 1/9th the hardware rate.
	 *  Correct the value if an attempt is made to stray outside that limit.
	 */
	if ((rate * 9) < 48000)
		rate = 48000 / 9;

	/*
	 *  We can not capture at at rate greater than the Input Rate (48000).
	 *  Return an error if an attempt is made to stray outside that limit.
	 */
	if (rate > 48000)
		rate = 48000;

	/*
	 *  Compute the values used to drive the actual sample rate conversion.
	 *  The following formulas are being computed, using inline assembly
	 *  since we need to use 64 bit arithmetic to compute the values:
	 *
	 *     coeffIncr = -floor((Fs,out * 2^23) / Fs,in)
	 *     phiIncr = floor((Fs,in * 2^26) / Fs,out)
	 *     correctionPerGOF = floor((Fs,in * 2^26 - Fs,out * phiIncr) /
	 *                                GOF_PER_SEC)
	 *     correctionPerSec = Fs,in * 2^26 - Fs,out * phiIncr -
	 *                          GOF_PER_SEC * correctionPerGOF
	 *     initialDelay = ceil((24 * Fs,in) / Fs,out)
	 *
	 * i.e.
	 *
	 *     coeffIncr = neg(dividend((Fs,out * 2^23) / Fs,in))
	 *     phiIncr:ulOther = dividend:remainder((Fs,in * 2^26) / Fs,out)
	 *     correctionPerGOF:correctionPerSec =
	 * 	    dividend:remainder(ulOther / GOF_PER_SEC)
	 *     initialDelay = dividend(((24 * Fs,in) + Fs,out - 1) / Fs,out)
	 */

	tmp1 = rate << 16;
	coeffIncr = tmp1 / 48000;
	tmp1 -= coeffIncr * 48000;
	tmp1 <<= 7;
	coeffIncr <<= 7;
	coeffIncr += tmp1 / 48000;
	coeffIncr ^= 0xFFFFFFFF;
	coeffIncr++;
	tmp1 = 48000 << 16;
	phiIncr = tmp1 / rate;
	tmp1 -= phiIncr * rate;
	tmp1 <<= 10;
	phiIncr <<= 10;
	tmp2 = tmp1 / rate;
	phiIncr += tmp2;
	tmp1 -= tmp2 * rate;
	correctionPerGOF = tmp1 / GOF_PER_SEC;
	tmp1 -= correctionPerGOF * GOF_PER_SEC;
	correctionPerSec = tmp1;
	initialDelay = ((48000 * 24) + rate - 1) / rate;

	/*
	 *  Fill in the VariDecimate control block.
	 */
	spin_lock_irq(&card->lock);
	cs461x_poke(card, BA1_CSRC,
		((correctionPerSec << 16) & 0xFFFF0000) | (correctionPerGOF & 0xFFFF));
	cs461x_poke(card, BA1_CCI, coeffIncr);
	cs461x_poke(card, BA1_CD,
		(((BA1_VARIDEC_BUF_1 + (initialDelay << 2)) << 16) & 0xFFFF0000) | 0x80);
	cs461x_poke(card, BA1_CPI, phiIncr);
	spin_unlock_irq(&card->lock);

	/*
	 *  Figure out the frame group length for the write back task.  Basically,
	 *  this is just the factors of 24000 (2^6*3*5^3) that are not present in
	 *  the output sample rate.
	 */
	frameGroupLength = 1;
	for (cnt = 2; cnt <= 64; cnt *= 2) {
		if (((rate / cnt) * cnt) != rate)
			frameGroupLength *= 2;
	}
	if (((rate / 3) * 3) != rate) {
		frameGroupLength *= 3;
	}
	for (cnt = 5; cnt <= 125; cnt *= 5) {
		if (((rate / cnt) * cnt) != rate) 
			frameGroupLength *= 5;
        }

	/*
	 * Fill in the WriteBack control block.
	 */
	spin_lock_irq(&card->lock);
	cs461x_poke(card, BA1_CFG1, frameGroupLength);
	cs461x_poke(card, BA1_CFG2, (0x00800000 | frameGroupLength));
	cs461x_poke(card, BA1_CCST, 0x0000FFFF);
	cs461x_poke(card, BA1_CSPB, ((65536 * rate) / 24000));
	cs461x_poke(card, (BA1_CSPB + 4), 0x0000FFFF);
	spin_unlock_irq(&card->lock);
	dmabuf->rate = rate;
	return rate;
}

/* prepare channel attributes for playback */ 
static void cs_play_setup(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned int tmp, tmp1;

	tmp1=16;
	if (!(dmabuf->fmt & CS_FMT_STEREO))
		tmp1>>=1;
	cs461x_poke(card, BA1_PVOL, 0x80008000);
	cs461x_poke(card, BA1_PBA, virt_to_bus(dmabuf->pbuf));
	
	tmp=cs461x_peek(card, BA1_PDTC);
	tmp&=~0x000003FF;
	tmp|=tmp1-1;
	cs461x_poke(card, BA1_PDTC, tmp);
	
	tmp=cs461x_peek(card, BA1_PFIE);
	tmp&=~0x0000F03F;
	if(!(dmabuf->fmt & CS_FMT_STEREO))
	{
		tmp|=0x00002000;
	}
	cs461x_poke(card, BA1_PFIE, tmp);

}

/* prepare channel attributes for recording */
static void cs_rec_setup(struct cs_state *state)
{
	struct cs_card *card = state->card;
	struct dmabuf *dmabuf = &state->dmabuf;
	/* set the attenuation to 0dB */
	cs461x_poke(card, BA1_CVOL, 0x80008000);
	cs461x_poke(card, BA1_CBA, virt_to_bus(dmabuf->pbuf));
}


/* get current playback/recording dma buffer pointer (byte offset from LBA),
   called with spinlock held! */
   
extern __inline__ unsigned cs_get_dma_addr(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	u32 offset;
	
	if (!dmabuf->enable)
		return 0;
		
	offset = dmabuf->pringbuf * 2048;
	return offset;
}

static void resync_dma_ptrs(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	int offset;
	
	offset = 0;
	dmabuf->hwptr=dmabuf->swptr = 0;
	dmabuf->ppingbuf = dmabuf->pringbuf = 0;
	dmabuf->ppingbuf = 1;
	if(dmabuf->fmt&CS_FMT_16BIT)
		memset(dmabuf->pbuf, 0, PAGE_SIZE);
	else
		memset(dmabuf->pbuf, 0x80, PAGE_SIZE);
}
	
/* Stop recording (lock held) */
extern __inline__ void __stop_adc(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned int tmp;
	
	dmabuf->enable &= ~ADC_RUNNING;
	
	tmp=cs461x_peek(card, BA1_CCTL);
	tmp&=0xFFFF;
	cs461x_poke(card, BA1_CCTL, tmp);
	
}

static void stop_adc(struct cs_state *state)
{
	struct cs_card *card = state->card;
	unsigned long flags;

	spin_lock_irqsave(&card->lock, flags);
	__stop_adc(state);
	spin_unlock_irqrestore(&card->lock, flags);
}

static void start_adc(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned long flags;
	unsigned int tmp;

	spin_lock_irqsave(&card->lock, flags);
	if ((dmabuf->mapped || dmabuf->count < (signed)dmabuf->dmasize) && dmabuf->ready) {
		dmabuf->enable |= ADC_RUNNING;
		tmp=cs461x_peek(card, BA1_CCTL);
		tmp&=0xFFFF;
		tmp|=card->cctl;
		cs461x_poke(card, BA1_CCTL, tmp);
	}
	spin_unlock_irqrestore(&card->lock, flags);
}

/* stop playback (lock held) */
extern __inline__ void __stop_dac(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned int tmp;

	dmabuf->enable &= ~DAC_RUNNING;
	
	tmp=cs461x_peek(card, BA1_PCTL);
	tmp&=0xFFFF;
	cs461x_poke(card, BA1_PCTL, tmp);
}

static void stop_dac(struct cs_state *state)
{
	struct cs_card *card = state->card;
	unsigned long flags;

	spin_lock_irqsave(&card->lock, flags);
	__stop_dac(state);
	spin_unlock_irqrestore(&card->lock, flags);
}	

static void start_dac(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct cs_card *card = state->card;
	unsigned long flags;
	int tmp;

	spin_lock_irqsave(&card->lock, flags);
	if ((dmabuf->mapped || dmabuf->count > 0) && dmabuf->ready) {
		if(!(dmabuf->enable&DAC_RUNNING))
		{
			dmabuf->enable |= DAC_RUNNING;
			tmp = cs461x_peek(card, BA1_PCTL);
			tmp &= 0xFFFF;
			tmp |= card->pctl;
			cs461x_poke(card, BA1_PCTL, tmp);
		}
	}
	spin_unlock_irqrestore(&card->lock, flags);
}

#define DMABUF_DEFAULTORDER (15-PAGE_SHIFT)
#define DMABUF_MINORDER 1

/* allocate DMA buffer, playback and recording buffer should be allocated seperately */
static int alloc_dmabuf(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	void *rawbuf = NULL;
	int order;
	struct page *page, *pend;

	/* alloc as big a chunk as we can */
	for (order = DMABUF_DEFAULTORDER; order >= DMABUF_MINORDER; order--)
		if((rawbuf = (void *)__get_free_pages(GFP_KERNEL|GFP_DMA, order)))
			break;

	if (!rawbuf)
		return -ENOMEM;

#ifdef DEBUG
	printk("cs461x: allocated %ld (order = %d) bytes at %p\n",
	       PAGE_SIZE << order, order, rawbuf);
#endif

	dmabuf->ready  = dmabuf->mapped = 0;
	dmabuf->rawbuf = rawbuf;
	dmabuf->buforder = order;
	
	/* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */
	pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
	for (page = virt_to_page(rawbuf); page <= pend; page++)
		mem_map_reserve(page);

	return 0;
}

/* free DMA buffer */
static void dealloc_dmabuf(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	struct page *page, *pend;

	if (dmabuf->rawbuf) {
		/* undo marking the pages as reserved */
		pend = virt_to_page(dmabuf->rawbuf + (PAGE_SIZE << dmabuf->buforder) - 1);
		for (page = virt_to_page(dmabuf->rawbuf); page <= pend; page++)
			mem_map_unreserve(page);
		pci_free_consistent(state->card->pci_dev, PAGE_SIZE << dmabuf->buforder,
		                    dmabuf->rawbuf, dmabuf->dma_handle);
	}
	dmabuf->rawbuf = NULL;
	dmabuf->mapped = dmabuf->ready = 0;
}

static int prog_dmabuf(struct cs_state *state, unsigned rec)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned bytepersec;
	unsigned bufsize;
	unsigned long flags;
	int ret;
	
	spin_lock_irqsave(&state->card->lock, flags);
	resync_dma_ptrs(state);
	dmabuf->total_bytes = 0;
	dmabuf->count = dmabuf->error = 0;
	spin_unlock_irqrestore(&state->card->lock, flags);

	/* allocate DMA buffer if not allocated yet */
	if (!dmabuf->rawbuf)
		if ((ret = alloc_dmabuf(state)))
			return ret;

	/* FIXME: figure out all this OSS fragment stuff */
	bytepersec = dmabuf->rate << sample_shift[dmabuf->fmt];
	bufsize = PAGE_SIZE << dmabuf->buforder;
	if (dmabuf->ossfragshift) {
		if ((1000 << dmabuf->ossfragshift) < bytepersec)
			dmabuf->fragshift = ld2(bytepersec/1000);
		else
			dmabuf->fragshift = dmabuf->ossfragshift;
	} else {
		/* lets hand out reasonable big ass buffers by default */
		dmabuf->fragshift = (dmabuf->buforder + PAGE_SHIFT -2);
	}
	dmabuf->numfrag = bufsize >> dmabuf->fragshift;
	while (dmabuf->numfrag < 4 && dmabuf->fragshift > 3) {
		dmabuf->fragshift--;
		dmabuf->numfrag = bufsize >> dmabuf->fragshift;
	}
	dmabuf->fragsize = 1 << dmabuf->fragshift;
	if (dmabuf->ossmaxfrags >= 4 && dmabuf->ossmaxfrags < dmabuf->numfrag)
		dmabuf->numfrag = dmabuf->ossmaxfrags;
	dmabuf->fragsamples = dmabuf->fragsize >> sample_shift[dmabuf->fmt];
	dmabuf->dmasize = dmabuf->numfrag << dmabuf->fragshift;

	memset(dmabuf->rawbuf, (dmabuf->fmt & CS_FMT_16BIT) ? 0 : 0x80,
	       dmabuf->dmasize);

	/*
	 *	Now set up the ring 
	 */

	spin_lock_irqsave(&state->card->lock, flags);
	if (rec) {
		cs_rec_setup(state);
	} else {
		cs_play_setup(state);
	}
	spin_unlock_irqrestore(&state->card->lock, flags);

	/* set the ready flag for the dma buffer */
	dmabuf->ready = 1;

#ifdef DEBUG
	printk("cs461x: prog_dmabuf, sample rate = %d, format = %d, numfrag = %d, "
	       "fragsize = %d dmasize = %d\n",
	       dmabuf->rate, dmabuf->fmt, dmabuf->numfrag,
	       dmabuf->fragsize, dmabuf->dmasize);
#endif

	return 0;
}

static void cs_clear_tail(struct cs_state *state)
{
}

static int drain_dac(struct cs_state *state, int nonblock)
{
	DECLARE_WAITQUEUE(wait, current);
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned long flags;
	unsigned long tmo;
	int count;

	if (dmabuf->mapped || !dmabuf->ready)
		return 0;

	add_wait_queue(&dmabuf->wait, &wait);
	for (;;) {
		/* It seems that we have to set the current state to TASK_INTERRUPTIBLE
		   every time to make the process really go to sleep */
		current->state = TASK_INTERRUPTIBLE;

		spin_lock_irqsave(&state->card->lock, flags);
		count = dmabuf->count;
		spin_unlock_irqrestore(&state->card->lock, flags);

		if (count <= 0)
			break;

		if (signal_pending(current))
			break;

		if (nonblock) {
			remove_wait_queue(&dmabuf->wait, &wait);
			current->state = TASK_RUNNING;
			return -EBUSY;
		}

		tmo = (dmabuf->dmasize * HZ) / dmabuf->rate;
		tmo >>= sample_shift[dmabuf->fmt];
		tmo += (4096*HZ)/dmabuf->rate;

		if (!schedule_timeout(tmo ? tmo : 1) && tmo){
			printk(KERN_ERR "cs461x: drain_dac, dma timeout? %d\n", count);
			break;
		}
	}
	remove_wait_queue(&dmabuf->wait, &wait);
	current->state = TASK_RUNNING;
	if (signal_pending(current))
		return -ERESTARTSYS;

	return 0;
}

/* update buffer manangement pointers, especially, dmabuf->count and dmabuf->hwptr */
static void cs_update_ptr(struct cs_state *state)
{
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned hwptr, swptr;
	int clear_cnt = 0;
	int diff;
	unsigned char silence;

	/* update hardware pointer */
	hwptr = cs_get_dma_addr(state);
	diff = (dmabuf->dmasize + hwptr - dmabuf->hwptr) % dmabuf->dmasize;
	dmabuf->hwptr = hwptr;
	dmabuf->total_bytes += diff;

	/* error handling and process wake up for DAC */
	if (dmabuf->enable == ADC_RUNNING) {
		if (dmabuf->mapped) {
			dmabuf->count -= diff;
			if (dmabuf->count >= (signed)dmabuf->fragsize)
				wake_up(&dmabuf->wait);
		} else {
			dmabuf->count += diff;

			if (dmabuf->count < 0 || dmabuf->count > dmabuf->dmasize) {
				/* buffer underrun or buffer overrun, we have no way to recover
				   it here, just stop the machine and let the process force hwptr
				   and swptr to sync */
				__stop_adc(state);
				dmabuf->error++;
			}
			else if (!dmabuf->endcleared) {
				swptr = dmabuf->swptr;
				silence = (dmabuf->fmt & CS_FMT_16BIT ? 0 : 0x80);
				if (dmabuf->count < (signed) dmabuf->fragsize) 
				{
					clear_cnt = dmabuf->fragsize;
					if ((swptr + clear_cnt) > dmabuf->dmasize)
						clear_cnt = dmabuf->dmasize - swptr;
					memset (dmabuf->rawbuf + swptr, silence, clear_cnt);
					dmabuf->endcleared = 1;
				}
			}			
			wake_up(&dmabuf->wait);
		}
	}
	/* error handling and process wake up for DAC */
	if (dmabuf->enable == DAC_RUNNING) {
		if (dmabuf->mapped) {
			dmabuf->count += diff;
			if (dmabuf->count >= (signed)dmabuf->fragsize)
				wake_up(&dmabuf->wait);
		} else {
			dmabuf->count -= diff;

			if (dmabuf->count < 0 || dmabuf->count > dmabuf->dmasize) {
				/* buffer underrun or buffer overrun, we have no way to recover
				   it here, just stop the machine and let the process force hwptr
				   and swptr to sync */
				__stop_dac(state);
				dmabuf->error++;
			}
			wake_up(&dmabuf->wait);
		}
	}
}

static void cs_record_interrupt(struct cs_state *state)
{
	memcpy(state->dmabuf.rawbuf + (2048*state->dmabuf.pringbuf++),
		state->dmabuf.pbuf+2048*state->dmabuf.ppingbuf++, 2048);
	state->dmabuf.ppingbuf&=1;
	if(state->dmabuf.pringbuf > (PAGE_SIZE<<state->dmabuf.buforder)/2048)
		state->dmabuf.pringbuf=0;
	cs_update_ptr(state);
}

static void cs_play_interrupt(struct cs_state *state)
{
	memcpy(state->dmabuf.pbuf+2048*state->dmabuf.ppingbuf++,
		state->dmabuf.rawbuf + (2048*state->dmabuf.pringbuf++), 2048);
	state->dmabuf.ppingbuf&=1;
	if(state->dmabuf.pringbuf >= (PAGE_SIZE<<state->dmabuf.buforder)/2048)
		state->dmabuf.pringbuf=0;
	cs_update_ptr(state);
}

static void cs_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct cs_card *card = (struct cs_card *)dev_id;
	/* Single channel card */
	struct cs_state *recstate = card->channel[0].state;
	struct cs_state *playstate = card->channel[1].state;
	u32 status;

	spin_lock(&card->lock);

	status = cs461x_peekBA0(card, BA0_HISR);
	
	if((status&0x7fffffff)==0)
	{
		cs461x_pokeBA0(card, BA0_HICR, HICR_CHGM|HICR_IEV);
		spin_unlock(&card->lock);
		return;
	}
	
	if((status & HISR_VC0) && playstate && playstate->dmabuf.ready)
		cs_play_interrupt(playstate);
	if((status & HISR_VC1) && recstate && recstate->dmabuf.ready)
		cs_record_interrupt(recstate);
	
 	/* clear 'em */
	cs461x_pokeBA0(card, BA0_HICR, HICR_CHGM|HICR_IEV);
	spin_unlock(&card->lock);
}

static loff_t cs_llseek(struct file *file, loff_t offset, int origin)
{
	return -ESPIPE;
}

/* in this loop, dmabuf.count signifies the amount of data that is waiting to be copied to
   the user's buffer.  it is filled by the dma machine and drained by this loop. */
static ssize_t cs_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;
	ssize_t ret;
	unsigned long flags;
	unsigned swptr;
	int cnt;

#ifdef DEBUG
	printk("cs461x: cs_read called, count = %d\n", count);
#endif

	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (dmabuf->mapped)
		return -ENXIO;
	if (!dmabuf->ready && (ret = prog_dmabuf(state, 1)))
		return ret;
	if (!access_ok(VERIFY_WRITE, buffer, count))
		return -EFAULT;
	ret = 0;

	while (count > 0) {
		spin_lock_irqsave(&state->card->lock, flags);
		if (dmabuf->count > (signed) dmabuf->dmasize) {
			/* buffer overrun, we are recovering from sleep_on_timeout,
			   resync hwptr and swptr, make process flush the buffer */
			dmabuf->count = dmabuf->dmasize;
			dmabuf->swptr = dmabuf->hwptr;
		}
		swptr = dmabuf->swptr;
		cnt = dmabuf->dmasize - swptr;
		if (dmabuf->count < cnt)
			cnt = dmabuf->count;
		spin_unlock_irqrestore(&state->card->lock, flags);

		if (cnt > count)
			cnt = count;
		if (cnt <= 0) {
			unsigned long tmo;
			/* buffer is empty, start the dma machine and wait for data to be
			   recorded */
			start_adc(state);
			if (file->f_flags & O_NONBLOCK) {
				if (!ret) ret = -EAGAIN;
				return ret;
			}
			/* This isnt strictly right for the 810  but it'll do */
			tmo = (dmabuf->dmasize * HZ) / (dmabuf->rate * 2);
			tmo >>= sample_shift[dmabuf->fmt];
			/* There are two situations when sleep_on_timeout returns, one is when
			   the interrupt is serviced correctly and the process is waked up by
			   ISR ON TIME. Another is when timeout is expired, which means that
			   either interrupt is NOT serviced correctly (pending interrupt) or it
			   is TOO LATE for the process to be scheduled to run (scheduler latency)
			   which results in a (potential) buffer overrun. And worse, there is
			   NOTHING we can do to prevent it. */
			if (!interruptible_sleep_on_timeout(&dmabuf->wait, tmo)) {
#ifdef DEBUG
				printk(KERN_ERR "cs461x: recording schedule timeout, "
				       "dmasz %u fragsz %u count %i hwptr %u swptr %u\n",
				       dmabuf->dmasize, dmabuf->fragsize, dmabuf->count,
				       dmabuf->hwptr, dmabuf->swptr);
#endif
				/* a buffer overrun, we delay the recovery untill next time the
				   while loop begin and we REALLY have space to record */
			}
			if (signal_pending(current)) {
				ret = ret ? ret : -ERESTARTSYS;
				return ret;
			}
			continue;
		}

		if (copy_to_user(buffer, dmabuf->rawbuf + swptr, cnt)) {
			if (!ret) ret = -EFAULT;
			return ret;
		}

		swptr = (swptr + cnt) % dmabuf->dmasize;

		spin_lock_irqsave(&state->card->lock, flags);
		dmabuf->swptr = swptr;
		dmabuf->count -= cnt;
		spin_unlock_irqrestore(&state->card->lock, flags);

		count -= cnt;
		buffer += cnt;
		ret += cnt;
		start_adc(state);
	}
	return ret;
}

/* in this loop, dmabuf.count signifies the amount of data that is waiting to be dma to
   the soundcard.  it is drained by the dma machine and filled by this loop. */
static ssize_t cs_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;
	ssize_t ret;
	unsigned long flags;
	unsigned swptr;
	int cnt;

#ifdef DEBUG
	printk("cs461x: cs_write called, count = %d\n", count);
#endif

	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (dmabuf->mapped)
		return -ENXIO;
	if (!dmabuf->ready && (ret = prog_dmabuf(state, 0)))
		return ret;
	if (!access_ok(VERIFY_READ, buffer, count))
		return -EFAULT;
	ret = 0;

	while (count > 0) {
		spin_lock_irqsave(&state->card->lock, flags);
		if (dmabuf->count < 0) {
			/* buffer underrun, we are recovering from sleep_on_timeout,
			   resync hwptr and swptr */
			dmabuf->count = 0;
			dmabuf->swptr = dmabuf->hwptr;
		}
		swptr = dmabuf->swptr;
		cnt = dmabuf->dmasize - swptr;
		if (dmabuf->count + cnt > dmabuf->dmasize)
			cnt = dmabuf->dmasize - dmabuf->count;
		spin_unlock_irqrestore(&state->card->lock, flags);

		if (cnt > count)
			cnt = count;
		if (cnt <= 0) {
			unsigned long tmo;
			/* buffer is full, start the dma machine and wait for data to be
			   played */
			start_dac(state);
			if (file->f_flags & O_NONBLOCK) {
				if (!ret) ret = -EAGAIN;
				return ret;
			}
			/* Not strictly correct but works */
			tmo = (dmabuf->dmasize * HZ) / (dmabuf->rate * 2);
			tmo >>= sample_shift[dmabuf->fmt];
			/* There are two situations when sleep_on_timeout returns, one is when
			   the interrupt is serviced correctly and the process is waked up by
			   ISR ON TIME. Another is when timeout is expired, which means that
			   either interrupt is NOT serviced correctly (pending interrupt) or it
			   is TOO LATE for the process to be scheduled to run (scheduler latency)
			   which results in a (potential) buffer underrun. And worse, there is
			   NOTHING we can do to prevent it. */
			if (!interruptible_sleep_on_timeout(&dmabuf->wait, tmo)) {
#ifdef DEBUG
				printk(KERN_ERR "cs461x: playback schedule timeout, "
				       "dmasz %u fragsz %u count %i hwptr %u swptr %u\n",
				       dmabuf->dmasize, dmabuf->fragsize, dmabuf->count,
				       dmabuf->hwptr, dmabuf->swptr);
#endif
				/* a buffer underrun, we delay the recovery untill next time the
				   while loop begin and we REALLY have data to play */
			}
			if (signal_pending(current)) {
				if (!ret) ret = -ERESTARTSYS;
				return ret;
			}
			continue;
		}
		if (copy_from_user(dmabuf->rawbuf + swptr, buffer, cnt)) {
			if (!ret) ret = -EFAULT;
			return ret;
		}

		swptr = (swptr + cnt) % dmabuf->dmasize;

		spin_lock_irqsave(&state->card->lock, flags);
		dmabuf->swptr = swptr;
		dmabuf->count += cnt;
		dmabuf->endcleared = 0;
		spin_unlock_irqrestore(&state->card->lock, flags);

		count -= cnt;
		buffer += cnt;
		ret += cnt;
		start_dac(state);
	}
	return ret;
}

static unsigned int cs_poll(struct file *file, struct poll_table_struct *wait)
{
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned long flags;
	unsigned int mask = 0;

	if (file->f_mode & FMODE_WRITE)
		poll_wait(file, &dmabuf->wait, wait);
	if (file->f_mode & FMODE_READ)
		poll_wait(file, &dmabuf->wait, wait);

	spin_lock_irqsave(&state->card->lock, flags);
	cs_update_ptr(state);
	if (file->f_mode & FMODE_READ) {
		if (dmabuf->count >= (signed)dmabuf->fragsize)
			mask |= POLLIN | POLLRDNORM;
	}
	if (file->f_mode & FMODE_WRITE) {
		if (dmabuf->mapped) {
			if (dmabuf->count >= (signed)dmabuf->fragsize)
				mask |= POLLOUT | POLLWRNORM;
		} else {
			if ((signed)dmabuf->dmasize >= dmabuf->count + (signed)dmabuf->fragsize)
				mask |= POLLOUT | POLLWRNORM;
		}
	}
	spin_unlock_irqrestore(&state->card->lock, flags);

	return mask;
}

static int cs_mmap(struct file *file, struct vm_area_struct *vma)
{
	return -EINVAL;
#if 0	
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;
	int ret;
	unsigned long size;
	

	if (vma->vm_flags & VM_WRITE) {
		if ((ret = prog_dmabuf(state, 0)) != 0)
			return ret;
	} else if (vma->vm_flags & VM_READ) {
		if ((ret = prog_dmabuf(state, 1)) != 0)
			return ret;
	} else 
		return -EINVAL;

	if (vma->vm_offset != 0)
		return -EINVAL;
	size = vma->vm_end - vma->vm_start;
	if (size > (PAGE_SIZE << dmabuf->buforder))
		return -EINVAL;
	if (remap_page_range(vma->vm_start, virt_to_phys(dmabuf->rawbuf),
			     size, vma->vm_page_prot))
		return -EAGAIN;
	dmabuf->mapped = 1;

	return 0;
#endif	
}

static int cs_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;
	unsigned long flags;
	audio_buf_info abinfo;
	count_info cinfo;
	int val, mapped, ret;

	mapped = ((file->f_mode & FMODE_WRITE) && dmabuf->mapped) ||
		((file->f_mode & FMODE_READ) && dmabuf->mapped);
#ifdef DEBUG
	printk("cs461x: cs_ioctl, command = %2d, arg = 0x%08x\n",
	       _IOC_NR(cmd), arg ? *(int *)arg : 0);
#endif

	switch (cmd) 
	{
	case OSS_GETVERSION:
		return put_user(SOUND_VERSION, (int *)arg);

	case SNDCTL_DSP_RESET:
		/* FIXME: spin_lock ? */
		if (file->f_mode & FMODE_WRITE) {
			stop_dac(state);
			synchronize_irq();
			dmabuf->ready = 0;
			resync_dma_ptrs(state);
			dmabuf->swptr = dmabuf->hwptr = 0;
			dmabuf->count = dmabuf->total_bytes = 0;
		}
		if (file->f_mode & FMODE_READ) {
			stop_adc(state);
			synchronize_irq();
			resync_dma_ptrs(state);
			dmabuf->ready = 0;
			dmabuf->swptr = dmabuf->hwptr = 0;
			dmabuf->count = dmabuf->total_bytes = 0;
		}
		return 0;

	case SNDCTL_DSP_SYNC:
		if (file->f_mode & FMODE_WRITE)
			return drain_dac(state, file->f_flags & O_NONBLOCK);
		return 0;

	case SNDCTL_DSP_SPEED: /* set smaple rate */
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (val >= 0) {
			if (file->f_mode & FMODE_WRITE) {
				stop_dac(state);
				dmabuf->ready = 0;
				cs_set_dac_rate(state, val);
			}
			if (file->f_mode & FMODE_READ) {
				stop_adc(state);
				dmabuf->ready = 0;
				cs_set_adc_rate(state, val);
			}
		}
		return put_user(dmabuf->rate, (int *)arg);

	case SNDCTL_DSP_STEREO: /* set stereo or mono channel */
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (file->f_mode & FMODE_WRITE) {
			stop_dac(state);
			dmabuf->ready = 0;
			if(val)
				dmabuf->fmt |= CS_FMT_STEREO;
			else
				dmabuf->fmt &= ~CS_FMT_STEREO;
		}
		if (file->f_mode & FMODE_READ) {
			stop_adc(state);
			dmabuf->ready = 0;
			if(val)
			{
				dmabuf->fmt |= CS_FMT_STEREO;
				return put_user(1, (int *)arg);
			}
#if 0
			/* Needs extra work to support this */

			else
				dmabuf->fmt &= ~CS_FMT_STEREO;
#endif
		}
		return 0;

	case SNDCTL_DSP_GETBLKSIZE:
		if (file->f_mode & FMODE_WRITE) {
			if ((val = prog_dmabuf(state, 0)))
				return val;
			return put_user(dmabuf->fragsize, (int *)arg);
		}
		if (file->f_mode & FMODE_READ) {
			if ((val = prog_dmabuf(state, 1)))
				return val;
			return put_user(dmabuf->fragsize, (int *)arg);
		}

	case SNDCTL_DSP_GETFMTS: /* Returns a mask of supported sample format*/
		return put_user(AFMT_S16_LE, (int *)arg);

	case SNDCTL_DSP_SETFMT: /* Select sample format */
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (val != AFMT_QUERY) {
			if(val==AFMT_S16_LE/* || val==AFMT_U8*/)
			{
				if (file->f_mode & FMODE_WRITE) {
					stop_dac(state);
					dmabuf->ready = 0;
				}
				if (file->f_mode & FMODE_READ) {
					stop_adc(state);
					dmabuf->ready = 0;
				}
				if(val==AFMT_S16_LE)
					dmabuf->fmt |= CS_FMT_16BIT;
				else
					dmabuf->fmt &= ~CS_FMT_16BIT;
			}
		}
		if(dmabuf->fmt&CS_FMT_16BIT)
			return put_user(AFMT_S16_LE, (int *)arg);
		else
			return put_user(AFMT_U8, (int *)arg);

	case SNDCTL_DSP_CHANNELS:
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (val != 0) {
			if (file->f_mode & FMODE_WRITE) {
				stop_dac(state);
				dmabuf->ready = 0;
				if (val > 1)
					dmabuf->fmt |= CS_FMT_STEREO;
				else
					dmabuf->fmt &= ~CS_FMT_STEREO;
			}
			if (file->f_mode & FMODE_READ) {
				stop_adc(state);
				dmabuf->ready = 0;
			}
		}
		return put_user((dmabuf->fmt & CS_FMT_STEREO) ? 2 : 1,
				(int *)arg);

	case SNDCTL_DSP_POST:
		/* FIXME: the same as RESET ?? */
		return 0;

	case SNDCTL_DSP_SUBDIVIDE:
		if (dmabuf->subdivision)
			return -EINVAL;
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (val != 1 && val != 2)
			return -EINVAL;
		dmabuf->subdivision = val;
		return 0;

	case SNDCTL_DSP_SETFRAGMENT:
		if (get_user(val, (int *)arg))
			return -EFAULT;

		dmabuf->ossfragshift = val & 0xffff;
		dmabuf->ossmaxfrags = (val >> 16) & 0xffff;
		switch(dmabuf->ossmaxfrags)
		{
			case 1:
				dmabuf->ossfragshift=12;
				return 0;
			default:
				/* Fragments must be 2K long */
				dmabuf->ossfragshift = 11;
				dmabuf->ossmaxfrags=2;
		}
		return 0;

	case SNDCTL_DSP_GETOSPACE:
		if (!(file->f_mode & FMODE_WRITE))
			return -EINVAL;
		if (!dmabuf->enable && (val = prog_dmabuf(state, 0)) != 0)
			return val;
		spin_lock_irqsave(&state->card->lock, flags);
		cs_update_ptr(state);
		abinfo.fragsize = dmabuf->fragsize;
		abinfo.bytes = dmabuf->dmasize - dmabuf->count;
		abinfo.fragstotal = dmabuf->numfrag;
		abinfo.fragments = abinfo.bytes >> dmabuf->fragshift;
		spin_unlock_irqrestore(&state->card->lock, flags);
		return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

	case SNDCTL_DSP_GETISPACE:
		if (!(file->f_mode & FMODE_READ))
			return -EINVAL;
		if (!dmabuf->enable && (val = prog_dmabuf(state, 1)) != 0)
			return val;
		spin_lock_irqsave(&state->card->lock, flags);
		cs_update_ptr(state);
		abinfo.fragsize = dmabuf->fragsize;
		abinfo.bytes = dmabuf->count;
		abinfo.fragstotal = dmabuf->numfrag;
		abinfo.fragments = abinfo.bytes >> dmabuf->fragshift;
		spin_unlock_irqrestore(&state->card->lock, flags);
		return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

	case SNDCTL_DSP_NONBLOCK:
		file->f_flags |= O_NONBLOCK;
		return 0;

	case SNDCTL_DSP_GETCAPS:
		return put_user(DSP_CAP_REALTIME|DSP_CAP_TRIGGER|DSP_CAP_MMAP,
			    (int *)arg);

	case SNDCTL_DSP_GETTRIGGER:
		val = 0;
		if (file->f_mode & FMODE_READ && dmabuf->enable)
			val |= PCM_ENABLE_INPUT;
		if (file->f_mode & FMODE_WRITE && dmabuf->enable)
			val |= PCM_ENABLE_OUTPUT;
		return put_user(val, (int *)arg);

	case SNDCTL_DSP_SETTRIGGER:
		if (get_user(val, (int *)arg))
			return -EFAULT;
		if (file->f_mode & FMODE_READ) {
			if (val & PCM_ENABLE_INPUT) {
				if (!dmabuf->ready && (ret = prog_dmabuf(state, 1)))
					return ret;
				start_adc(state);
			} else
				stop_adc(state);
		}
		if (file->f_mode & FMODE_WRITE) {
			if (val & PCM_ENABLE_OUTPUT) {
				if (!dmabuf->ready && (ret = prog_dmabuf(state, 0)))
					return ret;
				start_dac(state);
			} else
				stop_dac(state);
		}
		return 0;

	case SNDCTL_DSP_GETIPTR:
		if (!(file->f_mode & FMODE_READ))
			return -EINVAL;
		spin_lock_irqsave(&state->card->lock, flags);
		cs_update_ptr(state);
		cinfo.bytes = dmabuf->total_bytes;
		cinfo.blocks = dmabuf->count >> dmabuf->fragshift;
		cinfo.ptr = dmabuf->hwptr;
		if (dmabuf->mapped)
			dmabuf->count &= dmabuf->fragsize-1;
		spin_unlock_irqrestore(&state->card->lock, flags);
		return copy_to_user((void *)arg, &cinfo, sizeof(cinfo));

	case SNDCTL_DSP_GETOPTR:
		if (!(file->f_mode & FMODE_WRITE))
			return -EINVAL;
		spin_lock_irqsave(&state->card->lock, flags);
		cs_update_ptr(state);
		cinfo.bytes = dmabuf->total_bytes;
		cinfo.blocks = dmabuf->count >> dmabuf->fragshift;
		cinfo.ptr = dmabuf->hwptr;
		if (dmabuf->mapped)
			dmabuf->count &= dmabuf->fragsize-1;
		spin_unlock_irqrestore(&state->card->lock, flags);
		return copy_to_user((void *)arg, &cinfo, sizeof(cinfo));

	case SNDCTL_DSP_SETDUPLEX:
		return -EINVAL;

	case SNDCTL_DSP_GETODELAY:
		if (!(file->f_mode & FMODE_WRITE))
			return -EINVAL;
		spin_lock_irqsave(&state->card->lock, flags);
		cs_update_ptr(state);
		val = dmabuf->count;
		spin_unlock_irqrestore(&state->card->lock, flags);
		return put_user(val, (int *)arg);

	case SOUND_PCM_READ_RATE:
		return put_user(dmabuf->rate, (int *)arg);

	case SOUND_PCM_READ_CHANNELS:
		return put_user((dmabuf->fmt & CS_FMT_STEREO) ? 2 : 1,
				(int *)arg);

	case SOUND_PCM_READ_BITS:
		return put_user(AFMT_S16_LE, (int *)arg);

	case SNDCTL_DSP_MAPINBUF:
	case SNDCTL_DSP_MAPOUTBUF:
	case SNDCTL_DSP_SETSYNCRO:
	case SOUND_PCM_WRITE_FILTER:
	case SOUND_PCM_READ_FILTER:
		return -EINVAL;
	}
	return -EINVAL;
}


/*
 *	AMP control - null AMP
 */

static void amp_none(struct cs_card *card, int change)
{      
}

/*
 *	Crystal EAPD mode
 */

static void amp_voyetra(struct cs_card *card, int change)
{
	/* Manage the EAPD bit on the Crystal 4297 */
	int old=card->amplifier;

	card->amplifier+=change;
	if(card->amplifier && !old)
	{
		/* Turn the EAPD amp on */
		cs_ac97_set(card->ac97_codec[0],  AC97_POWER_CONTROL, 
			cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) |
				0x8000);
	}
	else if(old && !card->amplifier)
	{
		/* Turn the EAPD amp off */
		cs_ac97_set(card->ac97_codec[0],  AC97_POWER_CONTROL, 
			cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) &
				~0x8000);
	}
}



/*
 *	Untested
 */
#if 0
static void amp_voyetra_4294(struct cs_card *card, int change)
{
	struct ac97_codec *c=card->ac97_codec[0];
	int old = card->amplifier;

	card->amplifier+=change;

	if(card->amplifier)
	{
		/* Switch the GPIO pins 7 and 8 to open drain */
		cs_ac97_set(c, 0x4C, cs_ac97_get(c, 0x4C) & 0xFE7F);
		cs_ac97_set(c, 0x4E, cs_ac97_get(c, 0x4E) | 0x0180);
		/* Now wake the AMP (this might be backwards) */
		cs_ac97_set(c, 0x54, cs_ac97_get(c, 0x54) & ~0x0180);
	}
	else
	{
		cs_ac97_set(c, 0x54, cs_ac97_get(c, 0x54) | 0x0180);
	}
}
#endif

/*
 *	Handle the CLKRUN on a thinkpad. We must disable CLKRUN support
 *	whenever we need to beat on the chip.
 *
 *	The original idea and code for this hack comes from David Kaiser at
 *	Linuxcare. Perhaps one day Crystal will document their chips well
 *	enough to make them useful.
 */

static void clkrun_hack(struct cs_card *card, int change)
{
	struct pci_dev *acpi_dev;
	u16 control;
	u8 pp;
	unsigned long port;
	int old=card->amplifier;

	card->amplifier += change;

	acpi_dev = pci_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82371AB_3, NULL);
	if(acpi_dev == NULL)
		return;		/* Not a thinkpad thats for sure */


	/* Find the control port */             
	pci_read_config_byte(acpi_dev, 0x41, &pp);
	port = pp<<8;

	/* Read ACPI port */    
	control = inw(port+0x10);

	/* Flip CLKRUN off while running */
	if(!card->amplifier && old)
		outw(control|0x2000, port+0x10);
	else if(card->amplifier && !old)
		outw(control&~0x2000, port+0x10);
}


static int cs_open(struct inode *inode, struct file *file)
{
	int i = 0;
	struct cs_card *card = devs;
	struct cs_state *state = NULL;
	struct dmabuf *dmabuf = NULL;

#ifndef CS46XX_ENABLE_RECORD
	if (file->f_mode & FMODE_READ)
		return -ENODEV;
#endif

	/* find an avaiable virtual channel (instance of /dev/dsp) */
	while (card != NULL) {
		for (i = 0; i < NR_HW_CH; i++) {
			if (card->states[i] == NULL) {
				state = card->states[i] = (struct cs_state *)
					kmalloc(sizeof(struct cs_state), GFP_KERNEL);
				if (state == NULL)
					return -ENOMEM;
				memset(state, 0, sizeof(struct cs_state));
				dmabuf = &state->dmabuf;
				dmabuf->pbuf = (void *)get_free_page(GFP_KERNEL);
				if(dmabuf->pbuf==NULL)
				{
					kfree(state);
					card->states[i]=NULL;
					return -ENOMEM;
				}
				goto found_virt;
			}
		}
		card = card->next;
	}
	/* no more virtual channel avaiable */
	if (!state)
		return -ENODEV;

 found_virt:
	/* found a free virtual channel, allocate hardware channels */
	if(file->f_mode & FMODE_READ)
		dmabuf->channel = card->alloc_rec_pcm_channel(card);
	else
		dmabuf->channel = card->alloc_pcm_channel(card);
		
	if (dmabuf->channel == NULL) {
		kfree (card->states[i]);
		card->states[i] = NULL;;
		return -ENODEV;
	}

	/* Now turn on external AMP if needed */
	state->card = card;
	state->card->active_ctrl(state->card,1);
	state->card->amplifier_ctrl(state->card,1);

	dmabuf->channel->state = state;
	/* initialize the virtual channel */
	state->virt = i;
	state->magic = CS_STATE_MAGIC;
	init_waitqueue_head(&dmabuf->wait);
	init_MUTEX(&state->open_sem);
	file->private_data = state;

	down(&state->open_sem);

	/* set default sample format. According to OSS Programmer's Guide  /dev/dsp
	   should be default to unsigned 8-bits, mono, with sample rate 8kHz and
	   /dev/dspW will accept 16-bits sample */
	if (file->f_mode & FMODE_WRITE) {
		/* Output is 16bit only mono or stereo */
		dmabuf->fmt &= ~CS_FMT_MASK;
		dmabuf->fmt |= CS_FMT_16BIT;
		dmabuf->ossfragshift = 0;
		dmabuf->ossmaxfrags  = 0;
		dmabuf->subdivision  = 0;
		cs_set_dac_rate(state, 8000);
	}

	if (file->f_mode & FMODE_READ) {
		/* Input is 16bit stereo only */
		dmabuf->fmt &= ~CS_FMT_MASK;
		dmabuf->fmt |= CS_FMT_16BIT|CS_FMT_STEREO;
		dmabuf->ossfragshift = 0;
		dmabuf->ossmaxfrags  = 0;
		dmabuf->subdivision  = 0;
		cs_set_adc_rate(state, 8000);
	}

	state->open_mode |= file->f_mode & (FMODE_READ | FMODE_WRITE);
	up(&state->open_sem);


	MOD_INC_USE_COUNT;
	return 0;
}

static int cs_release(struct inode *inode, struct file *file)
{
	struct cs_state *state = (struct cs_state *)file->private_data;
	struct dmabuf *dmabuf = &state->dmabuf;

	if (file->f_mode & FMODE_WRITE) {
		/* FIXME :.. */
		cs_clear_tail(state);
		drain_dac(state, file->f_flags & O_NONBLOCK);
	}

	/* stop DMA state machine and free DMA buffers/channels */
	down(&state->open_sem);

	if (file->f_mode & FMODE_WRITE) {
		stop_dac(state);
		dealloc_dmabuf(state);
		state->card->free_pcm_channel(state->card, dmabuf->channel->num);
	}
	if (file->f_mode & FMODE_READ) {
		stop_adc(state);
		dealloc_dmabuf(state);
		state->card->free_pcm_channel(state->card, dmabuf->channel->num);
	}

	free_page((unsigned long)state->dmabuf.pbuf);

	/* we're covered by the open_sem */
	up(&state->open_sem);
	state->card->states[state->virt] = NULL;
	state->open_mode &= (~file->f_mode) & (FMODE_READ|FMODE_WRITE);

	/* Now turn off external AMP if needed */
	state->card->amplifier_ctrl(state->card, -1);
	state->card->active_ctrl(state->card, -1);

	kfree(state);
	MOD_DEC_USE_COUNT;
	return 0;
}

static /*const*/ struct file_operations cs461x_fops = {
	llseek:		cs_llseek,
	read:		cs_read,
	write:		cs_write,
	poll:		cs_poll,
	ioctl:		cs_ioctl,
	mmap:		cs_mmap,
	open:		cs_open,
	release:	cs_release,
};

/* Write AC97 codec registers */


static u16 cs_ac97_get(struct ac97_codec *dev, u8 reg)
{
	struct cs_card *card = dev->private_data;
	int count;
	
	/*
	 *  1. Write ACCAD = Command Address Register = 46Ch for AC97 register address
	 *  2. Write ACCDA = Command Data Register = 470h    for data to write to AC97 
	 *  3. Write ACCTL = Control Register = 460h for initiating the write
	 *  4. Read ACCTL = 460h, DCV should be reset by now and 460h = 17h
	 *  5. if DCV not cleared, break and return error
	 *  6. Read ACSTS = Status Register = 464h, check VSTS bit
	 */


	cs461x_peekBA0(card, BA0_ACSDA);

	/*
	 *  Setup the AC97 control registers on the CS461x to send the
	 *  appropriate command to the AC97 to perform the read.
	 *  ACCAD = Command Address Register = 46Ch
	 *  ACCDA = Command Data Register = 470h
	 *  ACCTL = Control Register = 460h
	 *  set DCV - will clear when process completed
	 *  set CRW - Read command
	 *  set VFRM - valid frame enabled
	 *  set ESYN - ASYNC generation enabled
	 *  set RSTN - ARST# inactive, AC97 codec not reset
	 */

	cs461x_pokeBA0(card, BA0_ACCAD, reg);
	cs461x_pokeBA0(card, BA0_ACCDA, 0);
	cs461x_pokeBA0(card, BA0_ACCTL, ACCTL_DCV | ACCTL_CRW |
					     ACCTL_VFRM | ACCTL_ESYN |
					     ACCTL_RSTN);


	/*
	 *  Wait for the read to occur.
	 */
	for (count = 0; count < 500; count++) {
		/*
		 *  First, we want to wait for a short time.
	 	 */
		udelay(10);
		/*
		 *  Now, check to see if the read has completed.
		 *  ACCTL = 460h, DCV should be reset by now and 460h = 17h
		 */
		if (!(cs461x_peekBA0(card, BA0_ACCTL) & ACCTL_DCV))
			break;
	}

	/*
	 *  Make sure the read completed.
	 */
	if (cs461x_peekBA0(card, BA0_ACCTL) & ACCTL_DCV) {
		printk(KERN_WARNING "cs461x: AC'97 read problem (ACCTL_DCV), reg = 0x%x\n", reg);
		return 0xffff;
	}

	/*
	 *  Wait for the valid status bit to go active.
	 */
	for (count = 0; count < 100; count++) {
		/*
		 *  Read the AC97 status register.
		 *  ACSTS = Status Register = 464h
		 *  VSTS - Valid Status
		 */
		if (cs461x_peekBA0(card, BA0_ACSTS) & ACSTS_VSTS)
			break;
		udelay(10);
	}
	
	/*
	 *  Make sure we got valid status.
	 */
	if (!(cs461x_peekBA0(card, BA0_ACSTS) & ACSTS_VSTS)) {
		printk(KERN_WARNING "cs461x: AC'97 read problem (ACSTS_VSTS), reg = 0x%x\n", reg);
		return 0xffff;
	}

	/*
	 *  Read the data returned from the AC97 register.
	 *  ACSDA = Status Data Register = 474h
	 */
#if 0
	printk("e) reg = 0x%x, val = 0x%x, BA0_ACCAD = 0x%x\n", reg,
			cs461x_peekBA0(card, BA0_ACSDA),
			cs461x_peekBA0(card, BA0_ACCAD));
#endif
	return cs461x_peekBA0(card, BA0_ACSDA);
}

static void cs_ac97_set(struct ac97_codec *dev, u8 reg, u16 val)
{
	struct cs_card *card = dev->private_data;
	int count;
	
	/*
	 *  1. Write ACCAD = Command Address Register = 46Ch for AC97 register address
	 *  2. Write ACCDA = Command Data Register = 470h    for data to write to AC97
	 *  3. Write ACCTL = Control Register = 460h for initiating the write
	 *  4. Read ACCTL = 460h, DCV should be reset by now and 460h = 07h
	 *  5. if DCV not cleared, break and return error
	 */

	/*
	 *  Setup the AC97 control registers on the CS461x to send the
	 *  appropriate command to the AC97 to perform the read.
	 *  ACCAD = Command Address Register = 46Ch
	 *  ACCDA = Command Data Register = 470h
	 *  ACCTL = Control Register = 460h
	 *  set DCV - will clear when process completed
	 *  reset CRW - Write command
	 *  set VFRM - valid frame enabled
	 *  set ESYN - ASYNC generation enabled
	 *  set RSTN - ARST# inactive, AC97 codec not reset
         */
	cs461x_pokeBA0(card, BA0_ACCAD, reg);
	cs461x_pokeBA0(card, BA0_ACCDA, val);
	cs461x_pokeBA0(card, BA0_ACCTL, ACCTL_DCV | ACCTL_VFRM |
				             ACCTL_ESYN | ACCTL_RSTN);
	for (count = 0; count < 1000; count++) {
		/*
		 *  First, we want to wait for a short time.
		 */
		udelay(10);
		/*
		 *  Now, check to see if the write has completed.
		 *  ACCTL = 460h, DCV should be reset by now and 460h = 07h
		 */
		if (!(cs461x_peekBA0(card, BA0_ACCTL) & ACCTL_DCV))
			break;
	}
	/*
	 *  Make sure the write completed.
	 */
	if (cs461x_peekBA0(card, BA0_ACCTL) & ACCTL_DCV)
		printk(KERN_WARNING "cs461x: AC'97 write problem, reg = 0x%x, val = 0x%x\n", reg, val);
}


/* OSS /dev/mixer file operation methods */

static int cs_open_mixdev(struct inode *inode, struct file *file)
{
	int i;
	int minor = MINOR(inode->i_rdev);
	struct cs_card *card = devs;

	for (card = devs; card != NULL; card = card->next)
		for (i = 0; i < NR_AC97; i++)
			if (card->ac97_codec[i] != NULL &&
			    card->ac97_codec[i]->dev_mixer == minor)
				goto match;

	if (!card)
		return -ENODEV;

 match:
	file->private_data = card->ac97_codec[i];

	card->active_ctrl(card,1);
	MOD_INC_USE_COUNT;
	return 0;
}

static int cs_release_mixdev(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	struct cs_card *card = devs;
	int i;

	for (card = devs; card != NULL; card = card->next)
		for (i = 0; i < NR_AC97; i++)
			if (card->ac97_codec[i] != NULL &&
			    card->ac97_codec[i]->dev_mixer == minor)
				goto match;

	if (!card)
		return -ENODEV;
match:
	card->active_ctrl(card, -1);

	MOD_DEC_USE_COUNT;
	return 0;
}

static int cs_ioctl_mixdev(struct inode *inode, struct file *file, unsigned int cmd,
				unsigned long arg)
{
	struct ac97_codec *codec = (struct ac97_codec *)file->private_data;

	return codec->mixer_ioctl(codec, cmd, arg);
}

static /*const*/ struct file_operations cs_mixer_fops = {
	llseek:		cs_llseek,
	ioctl:		cs_ioctl_mixdev,
	open:		cs_open_mixdev,
	release:	cs_release_mixdev,
};

/* AC97 codec initialisation. */
static int __init cs_ac97_init(struct cs_card *card)
{
	int num_ac97 = 0;
	int ready_2nd = 0;
	struct ac97_codec *codec;
	u16 eid;

	for (num_ac97 = 0; num_ac97 < NR_AC97; num_ac97++) {
		if ((codec = kmalloc(sizeof(struct ac97_codec), GFP_KERNEL)) == NULL)
			return -ENOMEM;
		memset(codec, 0, sizeof(struct ac97_codec));

		/* initialize some basic codec information, other fields will be filled
		   in ac97_probe_codec */
		codec->private_data = card;
		codec->id = num_ac97;

		codec->codec_read = cs_ac97_get;
		codec->codec_write = cs_ac97_set;
	
		if (ac97_probe_codec(codec) == 0)
			break;

		eid = cs_ac97_get(codec, AC97_EXTENDED_ID);
		
		if(eid==0xFFFFFF)
		{
			printk(KERN_WARNING "cs461x: no codec attached ?\n");
			kfree(codec);
			break;
		}
		
		card->ac97_features = eid;
				
			
		if ((codec->dev_mixer = register_sound_mixer(&cs_mixer_fops, -1)) < 0) {
			printk(KERN_ERR "cs461x: couldn't register mixer!\n");
			kfree(codec);
			break;
		}

		card->ac97_codec[num_ac97] = codec;

		/* if there is no secondary codec at all, don't probe any more */
		if (!ready_2nd)
			return num_ac97+1;
	}
	return num_ac97;
}

/* Boot the card
 */
 
static void cs461x_download(struct cs_card *card, u32 *src, unsigned long offset, unsigned long len)
{
	unsigned long counter;
	void *dst;
	
	dst = card->ba1.idx[(offset>>16)&3];
	dst += (offset&0xFFFF)<<2;
	for(counter=0;counter<len;counter+=4)
		writel(*src++, dst+counter);
}

/* 3*1024 parameter, 3.5*1024 sample, 2*3.5*1024 code */
#define BA1_DWORD_SIZE		(13 * 1024 + 512)
#define BA1_MEMORY_COUNT	3

struct BA1struct {
	struct {
		unsigned long offset;
		unsigned long size;
	} memory[BA1_MEMORY_COUNT];
	unsigned int map[BA1_DWORD_SIZE];
};

#include "cs461x_image.h"

static void cs461x_download_image(struct cs_card *card)
{
	int idx;
	unsigned long offset = 0;
	
	for (idx = 0; idx < BA1_MEMORY_COUNT; idx++) {
		cs461x_download(card,&BA1Struct.map[offset],
			       BA1Struct.memory[idx].offset,
			       BA1Struct.memory[idx].size);
		offset += BA1Struct.memory[idx].size >> 2;
	}	
}

/*
 *  Chip reset
 */

static void cs461x_reset(struct cs_card *card)
{
	int idx;

	/*
	 *  Write the reset bit of the SP control register.
	 */
	cs461x_poke(card, BA1_SPCR, SPCR_RSTSP);

	/*
	 *  Write the control register.
	 */
	cs461x_poke(card, BA1_SPCR, SPCR_DRQEN);

	/*
	 *  Clear the trap registers.
	 */
	for (idx = 0; idx < 8; idx++) {
		cs461x_poke(card, BA1_DREG, DREG_REGID_TRAP_SELECT + idx);
		cs461x_poke(card, BA1_TWPR, 0xFFFF);
	}
	cs461x_poke(card, BA1_DREG, 0);

	/*
	 *  Set the frame timer to reflect the number of cycles per frame.
	 */
	cs461x_poke(card, BA1_FRMT, 0xadf);
}

static void cs461x_clear_serial_FIFOs(struct cs_card *card)
{
	int idx, loop, powerdown = 0;
	unsigned int tmp;

	/*
	 *  See if the devices are powered down.  If so, we must power them up first
	 *  or they will not respond.
	 */
	if (!((tmp = cs461x_peekBA0(card, BA0_CLKCR1)) & CLKCR1_SWCE)) {
		cs461x_pokeBA0(card, BA0_CLKCR1, tmp | CLKCR1_SWCE);
		powerdown = 1;
	}

	/*
	 *  We want to clear out the serial port FIFOs so we don't end up playing
	 *  whatever random garbage happens to be in them.  We fill the sample FIFOS
	 *  with zero (silence).
         */
	cs461x_pokeBA0(card, BA0_SERBWP, 0);

	/*
	 *  Fill all 256 sample FIFO locations.
	 */
	for (idx = 0; idx < 256; idx++) {
		/*
		 *  Make sure the previous FIFO write operation has completed.
		 */
		for (loop = 0; loop < 5; loop++) {
			udelay(50);
			if (!(cs461x_peekBA0(card, BA0_SERBST) & SERBST_WBSY))
				break;
		}
		if (cs461x_peekBA0(card, BA0_SERBST) & SERBST_WBSY) {
			if (powerdown)
				cs461x_pokeBA0(card, BA0_CLKCR1, tmp);
		}
		/*
		 *  Write the serial port FIFO index.
		 */
		cs461x_pokeBA0(card, BA0_SERBAD, idx);
		/*
		 *  Tell the serial port to load the new value into the FIFO location.
		 */
		cs461x_pokeBA0(card, BA0_SERBCM, SERBCM_WRC);
	}
	/*
	 *  Now, if we powered up the devices, then power them back down again.
	 *  This is kinda ugly, but should never happen.
	 */
	if (powerdown)
		cs461x_pokeBA0(card, BA0_CLKCR1, tmp);
}

static void cs461x_powerup_dac(struct cs_card *card)
{
	int count;
	unsigned int tmp;

	/*
	 *  Power on the DACs on the AC97 card.  We turn off the DAC
	 *  powerdown bit and write the new value of the power control
	 *  register.
	 */
	tmp = cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL);
	if (tmp & 2)	/* already */
		return;
	cs_ac97_set(card->ac97_codec[0], AC97_POWER_CONTROL, tmp & 0xfdff);

	/*
	 *  Now, we wait until we sample a DAC ready state.
	 */
	for (count = 0; count < 32; count++) {
		/*
		 *  First, lets wait a short while to let things settle out a
		 *  bit, and to prevent retrying the read too quickly.
		 */
		udelay(50);

		/*
		 *  Read the current state of the power control register.
		 */
		if (cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) & 2)
			break;
	}
	
	/*
	 *  Check the status..
	 */
	if (!(cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) & 2))
		printk(KERN_WARNING "cs461x: powerup DAC failed\n");
}

static void cs461x_powerup_adc(struct cs_card *card)
{
	int count;
	unsigned int tmp;

	/*
	 *  Power on the ADCs on the AC97 card.  We turn off the DAC
	 *  powerdown bit and write the new value of the power control
	 *  register.
	 */
	tmp = cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL);
	if (tmp & 1)	/* already */
		return;
	cs_ac97_set(card->ac97_codec[0], AC97_POWER_CONTROL, tmp & 0xfeff);

	/*
	 *  Now, we wait until we sample a ADC ready state.
	 */
	for (count = 0; count < 32; count++) {
		/*
		 *  First, lets wait a short while to let things settle out a
		 *  bit, and to prevent retrying the read too quickly.
		 */
		udelay(50);

		/*
		 *  Read the current state of the power control register.
		 */
		if (cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) & 1)
			break;
	}

	/*
	 *  Check the status..
	 */
	if (!(cs_ac97_get(card->ac97_codec[0], AC97_POWER_CONTROL) & 1))
		printk(KERN_WARNING "cs461x: powerup ADC failed\n");
}

static void cs461x_proc_start(struct cs_card *card)
{
	int cnt;

	/*
	 *  Set the frame timer to reflect the number of cycles per frame.
	 */
	cs461x_poke(card, BA1_FRMT, 0xadf);
	/*
	 *  Turn on the run, run at frame, and DMA enable bits in the local copy of
	 *  the SP control register.
	 */
	cs461x_poke(card, BA1_SPCR, SPCR_RUN | SPCR_RUNFR | SPCR_DRQEN);
	/*
	 *  Wait until the run at frame bit resets itself in the SP control
	 *  register.
	 */
	for (cnt = 0; cnt < 25; cnt++) {
		udelay(50);
		if (!(cs461x_peek(card, BA1_SPCR) & SPCR_RUNFR))
			break;
	}

	if (cs461x_peek(card, BA1_SPCR) & SPCR_RUNFR)
		printk(KERN_WARNING "cs461x: SPCR_RUNFR never reset\n");
}

static void cs461x_proc_stop(struct cs_card *card)
{
	/*
	 *  Turn off the run, run at frame, and DMA enable bits in the local copy of
	 *  the SP control register.
	 */
	cs461x_poke(card, BA1_SPCR, 0);
}



static int cs_hardware_init(struct cs_card *card)
{
	unsigned long end_time;
	unsigned int tmp;
	
	/* 
	 *  First, blast the clock control register to zero so that the PLL starts
         *  out in a known state, and blast the master serial port control register
         *  to zero so that the serial ports also start out in a known state.
         */
        cs461x_pokeBA0(card, BA0_CLKCR1, 0);
        cs461x_pokeBA0(card, BA0_SERMC1, 0);

	/*
	 *  If we are in AC97 mode, then we must set the part to a host controlled
         *  AC-link.  Otherwise, we won't be able to bring up the link.
         */        
        cs461x_pokeBA0(card, BA0_SERACC, SERACC_HSP | SERACC_CODEC_TYPE_1_03);	/* 1.03 card */
        /* cs461x_pokeBA0(card, BA0_SERACC, SERACC_HSP | SERACC_CODEC_TYPE_2_0); */ /* 2.00 card */

        /*
         *  Drive the ARST# pin low for a minimum of 1uS (as defined in the AC97
         *  spec) and then drive it high.  This is done for non AC97 modes since
         *  there might be logic external to the CS461x that uses the ARST# line
         *  for a reset.
         */
        cs461x_pokeBA0(card, BA0_ACCTL, 0);
        udelay(500);
        cs461x_pokeBA0(card, BA0_ACCTL, ACCTL_RSTN);

	/*
	 *  The first thing we do here is to enable sync generation.  As soon
	 *  as we start receiving bit clock, we'll start producing the SYNC
	 *  signal.
	 */
	cs461x_pokeBA0(card, BA0_ACCTL, ACCTL_ESYN | ACCTL_RSTN);

	/*
	 *  Now wait for a short while to allow the AC97 part to start
	 *  generating bit clock (so we don't try to start the PLL without an
	 *  input clock).
	 */
	mdelay(10);		/* 1 should be enough ?? */

	/*
	 *  Set the serial port timing configuration, so that
	 *  the clock control circuit gets its clock from the correct place.
	 */
	cs461x_pokeBA0(card, BA0_SERMC1, SERMC1_PTC_AC97);

	/*
	 *  Write the selected clock control setup to the hardware.  Do not turn on
	 *  SWCE yet (if requested), so that the devices clocked by the output of
	 *  PLL are not clocked until the PLL is stable.
	 */
	cs461x_pokeBA0(card, BA0_PLLCC, PLLCC_LPF_1050_2780_KHZ | PLLCC_CDR_73_104_MHZ);
	cs461x_pokeBA0(card, BA0_PLLM, 0x3a);
	cs461x_pokeBA0(card, BA0_CLKCR2, CLKCR2_PDIVS_8);

	/*
	 *  Power up the PLL.
	 */
	cs461x_pokeBA0(card, BA0_CLKCR1, CLKCR1_PLLP);

	/*
         *  Wait until the PLL has stabilized.
	 */
	mdelay(100);		/* Again 1 should be enough ?? */

	/*
	 *  Turn on clocking of the core so that we can setup the serial ports.
	 */
	tmp = cs461x_peekBA0(card, BA0_CLKCR1) | CLKCR1_SWCE;
	cs461x_pokeBA0(card, BA0_CLKCR1, tmp);

	/*
	 *  Fill the serial port FIFOs with silence.
	 */
	cs461x_clear_serial_FIFOs(card);

	/*
	 *  Set the serial port FIFO pointer to the first sample in the FIFO.
	 */
	/* cs461x_pokeBA0(card, BA0_SERBSP, 0); */

	/*
	 *  Write the serial port configuration to the part.  The master
	 *  enable bit is not set until all other values have been written.
	 */
	cs461x_pokeBA0(card, BA0_SERC1, SERC1_SO1F_AC97 | SERC1_SO1EN);
	cs461x_pokeBA0(card, BA0_SERC2, SERC2_SI1F_AC97 | SERC1_SO1EN);
	cs461x_pokeBA0(card, BA0_SERMC1, SERMC1_PTC_AC97 | SERMC1_MSPE);


	mdelay(5);		/* Shouldnt be needed ?? */
	
	/*
	 * Wait for the card ready signal from the AC97 card.
	 */
	end_time = jiffies + 3 * (HZ >> 2);
	do {
		/*
		 *  Read the AC97 status register to see if we've seen a CODEC READY
		 *  signal from the AC97 card.
		 */
		if (cs461x_peekBA0(card, BA0_ACSTS) & ACSTS_CRDY)
			break;
		current->state = TASK_UNINTERRUPTIBLE;
		schedule_timeout(1);
	} while (time_before(jiffies, end_time));

	/*
	 *  Make sure CODEC is READY.
	 */
	if (!(cs461x_peekBA0(card, BA0_ACSTS) & ACSTS_CRDY)) {
		printk(KERN_WARNING "cs461x: create - never read card ready from AC'97\n");
		printk(KERN_WARNING "cs461x: it is probably not a bug, try using the CS4232 driver\n");
		return -EIO;
	}

	/*
	 *  Assert the vaid frame signal so that we can start sending commands
	 *  to the AC97 card.
	 */
	cs461x_pokeBA0(card, BA0_ACCTL, ACCTL_VFRM | ACCTL_ESYN | ACCTL_RSTN);

	/*
	 *  Wait until we've sampled input slots 3 and 4 as valid, meaning that
	 *  the card is pumping ADC data across the AC-link.
	 */
	end_time = jiffies + 3 * (HZ >> 2);
	do {
		/*
		 *  Read the input slot valid register and see if input slots 3 and
		 *  4 are valid yet.
		 */
		if ((cs461x_peekBA0(card, BA0_ACISV) & (ACISV_ISV3 | ACISV_ISV4)) == (ACISV_ISV3 | ACISV_ISV4))
			break;
		current->state = TASK_UNINTERRUPTIBLE;
		schedule_timeout(1);
	} while (time_before(jiffies, end_time));

	/*
	 *  Make sure input slots 3 and 4 are valid.  If not, then return
	 *  an error.
	 */
	if ((cs461x_peekBA0(card, BA0_ACISV) & (ACISV_ISV3 | ACISV_ISV4)) != (ACISV_ISV3 | ACISV_ISV4)) {
		printk(KERN_WARNING "cs461x: create - never read ISV3 & ISV4 from AC'97\n");
		return -EIO;
	}

	/*
	 *  Now, assert valid frame and the slot 3 and 4 valid bits.  This will
	 *  commense the transfer of digital audio data to the AC97 card.
	 */
	cs461x_pokeBA0(card, BA0_ACOSV, ACOSV_SLV3 | ACOSV_SLV4);

	/*
	 *  Power down the DAC and ADC.  We will power them up (if) when we need
	 *  them.
	 */
	/* cs461x_pokeBA0(card, BA0_AC97_POWERDOWN, 0x300); */

	/*
	 *  Turn off the Processor by turning off the software clock enable flag in 
	 *  the clock control register.
	 */
	/* tmp = cs461x_peekBA0(card, BA0_CLKCR1) & ~CLKCR1_SWCE; */
	/* cs461x_pokeBA0(card, BA0_CLKCR1, tmp); */

	/*
         *  Reset the processor.
         */
	cs461x_reset(card);

	/*
         *  Download the image to the processor.
	 */
	
	cs461x_download_image(card);

	/*
         *  Stop playback DMA.
	 */
	tmp = cs461x_peek(card, BA1_PCTL);
	card->pctl = tmp & 0xffff0000;
	cs461x_poke(card, BA1_PCTL, tmp & 0x0000ffff);

	/*
         *  Stop capture DMA.
	 */
	tmp = cs461x_peek(card, BA1_CCTL);
	card->cctl = tmp & 0x0000ffff;
	cs461x_poke(card, BA1_CCTL, tmp & 0xffff0000);

	/* initialize AC97 codec and register /dev/mixer */
	if (cs_ac97_init(card) <= 0)
		return -EIO;
		
	mdelay(5);		/* Do we need this ?? */
	
	cs461x_powerup_adc(card);
	cs461x_powerup_dac(card);

	cs461x_proc_start(card);

	/*
	 *  Enable interrupts on the part.
	 */
	cs461x_pokeBA0(card, BA0_HICR, HICR_IEV | HICR_CHGM);

	tmp = cs461x_peek(card, BA1_PFIE);
	tmp &= ~0x0000f03f;
	cs461x_poke(card, BA1_PFIE, tmp);	/* playback interrupt enable */

	tmp = cs461x_peek(card, BA1_CIE);
	tmp &= ~0x0000003f;
	tmp |=  0x00000001;
	cs461x_poke(card, BA1_CIE, tmp);	/* capture interrupt enable */	
	return 0;
}

/* install the driver, we do not allocate hardware channel nor DMA buffer now, they are defered 
   until "ACCESS" time (in prog_dmabuf called by open/read/write/ioctl/mmap) */


/*
 *	Card subid table
 */

struct cs_card_type
{
	u16 vendor;
	u16 id;
	char *name;
	void (*amp)(struct cs_card *, int);
	void (*active)(struct cs_card *, int);
};

static struct cs_card_type __init cards[]={
	{0x1489, 0x7001, "Genius Soundmaker 128 value", amp_none, NULL},
	{0x5053, 0x3357, "Voyetra", amp_voyetra, NULL},
	/* MI6020/21 use the same chipset as the Thinkpads, maybe needed */
	{0x1071, 0x6003, "Mitac MI6020/21", amp_none, clkrun_hack},
	/* Not sure if the 570 needs the clkrun hack */
	{PCI_VENDOR_ID_IBM, 0x0132, "Thinkpad 570", amp_none, clkrun_hack},
	{PCI_VENDOR_ID_IBM, 0x0153, "Thinkpad 600X/A20/T20", amp_none, clkrun_hack},
	{PCI_VENDOR_ID_IBM, 0x1010, "Thinkpad 600E (unsupported)", NULL, NULL},
	{0, 0, NULL, NULL}
};

static int __init cs_install(struct pci_dev *pci_dev)
{
	struct cs_card *card;
	struct cs_card_type *cp = &cards[0];
	u16 ss_card, ss_vendor;


	pci_read_config_word(pci_dev, PCI_SUBSYSTEM_VENDOR_ID, &ss_vendor);
	pci_read_config_word(pci_dev, PCI_SUBSYSTEM_ID, &ss_card);

	if ((card = kmalloc(sizeof(struct cs_card), GFP_KERNEL)) == NULL) {
		printk(KERN_ERR "cs461x: out of memory\n");
		return -ENOMEM;
	}
	memset(card, 0, sizeof(*card));

	card->ba0_addr = pci_dev->resource[0].start&PCI_BASE_ADDRESS_MEM_MASK;
	card->ba1_addr = pci_dev->resource[1].start&PCI_BASE_ADDRESS_MEM_MASK;
	card->pci_dev = pci_dev;
	card->irq = pci_dev->irq;
	card->magic = CS_CARD_MAGIC;
	spin_lock_init(&card->lock);

	pci_set_master(pci_dev);

	printk(KERN_INFO "cs461x: Card found at 0x%08lx and 0x%08lx, IRQ %d\n",
	       card->ba0_addr, card->ba1_addr, card->irq);

	card->alloc_pcm_channel = cs_alloc_pcm_channel;
	card->alloc_rec_pcm_channel = cs_alloc_rec_pcm_channel;
	card->free_pcm_channel = cs_free_pcm_channel;
	card->amplifier_ctrl = amp_none;
	card->active_ctrl = amp_none;

	while(cp->name)
	{
		if(cp->vendor == ss_vendor && cp->id == ss_card)
		{
			card->amplifier_ctrl = cp->amp;
			if(cp->active)
				card->active_ctrl = cp->active;
			break;
		}
		cp++;
	}
	if(cp->name==NULL)
	{
	printk(KERN_INFO "cs461x: Unknown card (%04X:%04X) at 0x%08lx/0x%08lx, IRQ %d\n",
		ss_vendor, ss_card, card->ba0_addr, card->ba1_addr,  card->irq);
	}
	else
	{
		printk(KERN_INFO "cs461x: %s at 0x%08lx/0x%08lx, IRQ %d\n",
			cp->name, card->ba0_addr, card->ba1_addr, card->irq);
	}

	if(card->amplifier_ctrl==NULL)
	{
		printk(KERN_ERR "cs461x: Unsupported configuration due to lack of documentation.\n");
		kfree(card);
		return -EINVAL;
	}

	if(external_amp == 1)
	{
		printk(KERN_INFO "cs461x: Crystal EAPD support forced on.\n");
		card->amplifier_ctrl = amp_voyetra;
	}

	if(thinkpad == 1)
	{
		card->active_ctrl = clkrun_hack;
		printk(KERN_INFO "cs461x: Activating CLKRUN hack for Thinkpad.\n");
	}

	card->active_ctrl(card, 1);

	/* claim our iospace and irq */
	
	card->ba0 = ioremap(card->ba0_addr, CS461X_BA0_SIZE);
	card->ba1.name.data0 = ioremap(card->ba1_addr + BA1_SP_DMEM0, CS461X_BA1_DATA0_SIZE);
	card->ba1.name.data1 = ioremap(card->ba1_addr + BA1_SP_DMEM1, CS461X_BA1_DATA1_SIZE);
	card->ba1.name.pmem = ioremap(card->ba1_addr + BA1_SP_PMEM, CS461X_BA1_PRG_SIZE);
	card->ba1.name.reg = ioremap(card->ba1_addr + BA1_SP_REG, CS461X_BA1_REG_SIZE);
	
	if(card->ba0 == 0 || card->ba1.name.data0 == 0 ||
		card->ba1.name.data1 == 0 || card->ba1.name.pmem == 0 ||
		card->ba1.name.reg == 0)
		goto fail2;
		
	if (request_irq(card->irq, &cs_interrupt, SA_SHIRQ, "cs461x", card)) {
		printk(KERN_ERR "cs461x: unable to allocate irq %d\n", card->irq);
		goto fail2;
	}
	/* register /dev/dsp */
	if ((card->dev_audio = register_sound_dsp(&cs461x_fops, -1)) < 0) {
		printk(KERN_ERR "cs461x: unable to register dsp\n");
		goto fail;
	}

	if (cs_hardware_init(card)<0)
	{
		unregister_sound_dsp(card->dev_audio);
		goto fail;
	}
	card->next = devs;
	devs = card;
	
	card->active_ctrl(card, -1);
	return 0;
	
fail:
	free_irq(card->irq, card);
fail2:
	if(card->ba0)
		iounmap(card->ba0);
	if(card->ba1.name.data0)
		iounmap(card->ba1.name.data0);
	if(card->ba1.name.data1)
		iounmap(card->ba1.name.data1);
	if(card->ba1.name.pmem)
		iounmap(card->ba1.name.pmem);
	if(card->ba1.name.reg)
		iounmap(card->ba1.name.reg);
	kfree(card);
	return -ENODEV;

}

static void cs_remove(struct cs_card *card)
{
	int i;
	unsigned int tmp;
	
	card->active_ctrl(card,1);

	tmp = cs461x_peek(card, BA1_PFIE);
	tmp &= ~0x0000f03f;
	tmp |=  0x00000010;
	cs461x_poke(card, BA1_PFIE, tmp);	/* playback interrupt disable */

	tmp = cs461x_peek(card, BA1_CIE);
	tmp &= ~0x0000003f;
	tmp |=  0x00000011;
	cs461x_poke(card, BA1_CIE, tmp);	/* capture interrupt disable */

	/*
         *  Stop playback DMA.
	 */
	tmp = cs461x_peek(card, BA1_PCTL);
	cs461x_poke(card, BA1_PCTL, tmp & 0x0000ffff);

	/*
         *  Stop capture DMA.
	 */
	tmp = cs461x_peek(card, BA1_CCTL);
	cs461x_poke(card, BA1_CCTL, tmp & 0xffff0000);

	/*
         *  Reset the processor.
         */
	cs461x_reset(card);

	cs461x_proc_stop(card);

	/*
	 *  Power down the DAC and ADC.  We will power them up (if) when we need
	 *  them.
	 */
	cs_ac97_set(card->ac97_codec[0], AC97_POWER_CONTROL, 0x300);

	/*
	 *  Power down the PLL.
	 */
	cs461x_pokeBA0(card, BA0_CLKCR1, 0);

	/*
	 *  Turn off the Processor by turning off the software clock enable flag in 
	 *  the clock control register.
	 */
	tmp = cs461x_peekBA0(card, BA0_CLKCR1) & ~CLKCR1_SWCE;
	cs461x_pokeBA0(card, BA0_CLKCR1, tmp);

	card->active_ctrl(card,-1);

	/* free hardware resources */
	free_irq(card->irq, card);
	iounmap(card->ba0);
	iounmap(card->ba1.name.data0);
	iounmap(card->ba1.name.data1);
	iounmap(card->ba1.name.pmem);
	iounmap(card->ba1.name.reg);
	
	/* unregister audio devices */
	for (i = 0; i < NR_AC97; i++)
		if (card->ac97_codec[i] != NULL) {
			unregister_sound_mixer(card->ac97_codec[i]->dev_mixer);
			kfree (card->ac97_codec[i]);
		}
	unregister_sound_dsp(card->dev_audio);
	kfree(card);
}

MODULE_AUTHOR("Alan Cox <alan@redhat.com>, Jaroslav Kysela");
MODULE_DESCRIPTION("Crystal SoundFusion Audio Support");

int __init cs_probe(void)
{
	struct pci_dev *pcidev = NULL;
	int foundone=0;
	
	if (!pci_present())   /* No PCI bus in this machine! */
		return -ENODEV;
		
	printk(KERN_INFO "Crystal 4280/461x + AC97 Audio, version "
	       DRIVER_VERSION ", " __TIME__ " " __DATE__ "\n");

	while( (pcidev = pci_find_device(PCI_VENDOR_ID_CIRRUS, 0x6001 , pcidev))!=NULL ) {
		if (cs_install(pcidev)==0)
			foundone++;
	}
	while( (pcidev = pci_find_device(PCI_VENDOR_ID_CIRRUS, 0x6003 , pcidev))!=NULL ) {
		if (cs_install(pcidev)==0)
			foundone++;
	}
	while( (pcidev = pci_find_device(PCI_VENDOR_ID_CIRRUS, 0x6004 , pcidev))!=NULL ) {
		if (cs_install(pcidev)==0)
			foundone++;
	}

	printk(KERN_INFO "cs461x: Found %d audio device(s).\n",
		foundone);
	return foundone;
}

#ifdef MODULE

int init_module(void)
{
	if(cs_probe()==0)
		printk(KERN_ERR "cs461x: No devices found.\n");
	return 0;
}

void cleanup_module (void)
{
	struct cs_card *next;
	while(devs)
	{
		next=devs->next;
		cs_remove(devs);
		devs=next;
	}
}

MODULE_PARM(external_amp, "i");
MODULE_PARM(thinkpad, "i");

#endif
