/* rserv.c
 * Support functions for eRServer 1.2 replication.
 * (c) 2000 Vadim Mikheev, PostgreSQL Inc.
 */

#include "executor/spi.h"		/* this is what you need to work with SPI */
#include "commands/trigger.h"	/* -"- and triggers */
#include "utils/tqual.h"		/* -"- and SnapshotData */
#include <ctype.h>				/* tolower () */

#ifdef PG_FUNCTION_INFO_V1
#define CurrentTriggerData ((TriggerData *) fcinfo->context)
#endif

#ifdef PG_FUNCTION_INFO_V1
PG_FUNCTION_INFO_V1(_rserv_log_);
PG_FUNCTION_INFO_V1(_rserv_sync_);
PG_FUNCTION_INFO_V1(_rserv_debug_);
PG_FUNCTION_INFO_V1(_rserv_lock_altid_);
PG_FUNCTION_INFO_V1(_rserv_unlock_altid_);
Datum		_rserv_log_(PG_FUNCTION_ARGS);
Datum		_rserv_sync_(PG_FUNCTION_ARGS);
Datum		_rserv_debug_(PG_FUNCTION_ARGS);
Datum		_rserv_lock_altid_(PG_FUNCTION_ARGS);
Datum		_rserv_unlock_altid_(PG_FUNCTION_ARGS);

#else

HeapTuple	_rserv_log_(void);
int32		_rserv_sync_(int32);
int32		_rserv_debug_(int32);
int32		_rserv_lock_altid_(int32);
int32		_rserv_unlock_altid_(int32);

#endif

static int	debug = 0;

static char *OutputValue(char *key, char *buf, int size);

/* To support 2 log tables */
static TransactionId	lastxid = InvalidTransactionId;
static int32			ActiveLogTableId = 0;

extern LOCKMETHOD		LockTableId;

static void	GetActiveLogTableId(void);

#ifdef PG_FUNCTION_INFO_V1
Datum
_rserv_log_(PG_FUNCTION_ARGS)
#else
HeapTuple
_rserv_log_()
#endif
{
	Trigger    *trigger;		/* to get trigger name */
	int			nargs;			/* # of args specified in CREATE TRIGGER */
	char	  **args;			/* argument: argnum */
	Relation	rel;			/* triggered relation */
	HeapTuple	tuple;			/* tuple to return */
	HeapTuple	newtuple = NULL;/* tuple to return */
	TupleDesc	tupdesc;		/* tuple description */
	int			keynum;
	char	   *key;
	char	   *okey;
	char	   *newkey = NULL;
	int			deleted = 0;
	int			inserted = 0;
	char		sql[8192];
	char		outbuf[8192];
	char		oidbuf[64];
	int			ret;

	/* Called by trigger manager ? */
	if (!CurrentTriggerData)
		elog(ERROR, "_rserv_log_: triggers are not initialized");

	/* Should be called for ROW trigger */
	if (TRIGGER_FIRED_FOR_STATEMENT(CurrentTriggerData->tg_event))
		elog(ERROR, "_rserv_log_: can't process STATEMENT events");

	tuple = CurrentTriggerData->tg_trigtuple;

	trigger = CurrentTriggerData->tg_trigger;
	nargs = trigger->tgnargs;
	args = trigger->tgargs;

	if (nargs != 1)				/* odd number of arguments! */
		elog(ERROR, "_rserv_log_: need in *one* argument");

	keynum = atoi(args[0]);

	if (keynum < 0 && keynum != ObjectIdAttributeNumber)
		elog(ERROR, "_rserv_log_: invalid keynum %d", keynum);

	rel = CurrentTriggerData->tg_relation;
	tupdesc = rel->rd_att;

	if (TRIGGER_FIRED_BY_DELETE(CurrentTriggerData->tg_event))
		deleted = 1;
	else if (TRIGGER_FIRED_BY_UPDATE(CurrentTriggerData->tg_event))
		newtuple = CurrentTriggerData->tg_newtuple;
	else if (TRIGGER_FIRED_BY_INSERT(CurrentTriggerData->tg_event))
		inserted = 1;

	/*
	 * Setting CurrentTriggerData to NULL prevents direct calls to trigger
	 * functions in queries. Normally, trigger functions have to be called
	 * by trigger manager code only.
	 */
	CurrentTriggerData = NULL;

	/* Connect to SPI manager */
	if ((ret = SPI_connect()) < 0)
		elog(ERROR, "_rserv_log_: SPI_connect returned %d", ret);

	/*
	 * Check if we called first time in this transaction
	 */
	if (lastxid != GetCurrentTransactionId())
	{
		GetActiveLogTableId();
		lastxid = GetCurrentTransactionId();
	}

	if (keynum == ObjectIdAttributeNumber)
	{
#ifndef USE_T_XMIN
		sprintf(oidbuf, "%u", tuple->t_data->t_oid);
#else
		sprintf(oidbuf, "%u", tuple->t_data->t_xmin);
#endif
		key = oidbuf;
	}
	else
		key = SPI_getvalue(tuple, tupdesc, keynum);

	if (key == NULL)
		elog(ERROR, "_rserv_log_: key must be not null");

	if (newtuple && keynum != ObjectIdAttributeNumber)
	{
		newkey = SPI_getvalue(newtuple, tupdesc, keynum);
		if (newkey == NULL)
			elog(ERROR, "_rserv_log_: key must be not null");
		if (strcmp(newkey, key) == 0)
			newkey = NULL;
		else
			deleted = 1;		/* old key was deleted */
	}

	if (strpbrk(key, "\\	\n'"))
		okey = OutputValue(key, outbuf, sizeof(outbuf));
	else
		okey = key;

	sprintf(sql, "update _RSERV_LOG_%d_ set logid = '%u', logtime = now(), "
			"deleted = %d where reloid = %u and key = '%s'",
			ActiveLogTableId, 
			GetCurrentTransactionId(), deleted, rel->rd_id, okey);

	if (debug)
		elog(NOTICE, sql);

	ret = SPI_exec(sql, 0);

	if (ret < 0)
		elog(ERROR, "_rserv_log_: SPI_exec(update) returned %d", ret);

	/*
	 * If no tuple was UPDATEd then do INSERT...
	 */
	if (SPI_processed > 1)
		elog(ERROR, "_rserv_log_: duplicate tuples");
	else if (SPI_processed == 0)
	{
		sprintf(sql, "insert into _RSERV_LOG_%d_ "
				"(reloid, logid, logtime, deleted, key) "
				"values (%u, '%u', now(), %d, '%s')",
				ActiveLogTableId,
				rel->rd_id, GetCurrentTransactionId(),
				((inserted) ? -1 : deleted), okey);

		if (debug)
			elog(NOTICE, sql);

		ret = SPI_exec(sql, 0);

		if (ret < 0)
			elog(ERROR, "_rserv_log_: SPI_exec(insert) returned %d", ret);
	}

	if (okey != key && okey != outbuf)
		pfree(okey);

	/* key was updated */
	if (newkey)
	{
		if (strpbrk(newkey, "\\	\n'"))
			okey = OutputValue(newkey, outbuf, sizeof(outbuf));
		else
			okey = newkey;

		/*
		 * What if new key was deleted before and "delete" log
		 * is still in log ? - have to handle this...
		 */
		sprintf(sql, "update _RSERV_LOG_%d_ set logid = '%u', logtime = now(), "
				"deleted = 0 where reloid = %u and key = '%s'",
				ActiveLogTableId,
				GetCurrentTransactionId(), rel->rd_id, okey);

		if (debug)
			elog(NOTICE, sql);

		ret = SPI_exec(sql, 0);

		if (ret < 0)
			elog(ERROR, "_rserv_log_: SPI_exec(update newkey) returned %d", ret);

		if (SPI_processed > 1)
			elog(ERROR, "_rserv_log_: duplicate tuples (newkey)");
		else if (SPI_processed == 0)
		{
			/*
			 * Either there was never newkey or it was deleted and
			 * this delete was replicated already to all slaves and
			 * "delete" record was removed from log after that.
			 * So, it's Ok to put -1 in "deleted" column.
			 */
			sprintf(sql, "insert into _RSERV_LOG_%d_ "
					"(reloid, logid, logtime, deleted, key) "
					"values (%u, '%u', now(), -1, '%s')",
					ActiveLogTableId, 
					rel->rd_id, GetCurrentTransactionId(), okey);

			if (debug)
				elog(NOTICE, sql);

			ret = SPI_exec(sql, 0);

			if (ret < 0)
				elog(ERROR, "_rserv_log_: SPI_exec(insert newkey) returned %d", ret);
		}

		if (okey != newkey && okey != outbuf)
			pfree(okey);
	}

	SPI_finish();

#ifdef PG_FUNCTION_INFO_V1
	return (PointerGetDatum(tuple));
#else
	return (tuple);
#endif
}

#ifdef PG_FUNCTION_INFO_V1
Datum
_rserv_sync_(PG_FUNCTION_ARGS)
#else
int32
_rserv_sync_(int32 server)
#endif
{
#ifdef PG_FUNCTION_INFO_V1
	int32		server = PG_GETARG_INT32(0);

#endif
	char		sql[8192];
	char		buf[8192];
	char	   *active = buf;
	uint32		xcnt;
	int			ret;

	if (SerializableSnapshot == NULL)
		elog(ERROR, "_rserv_sync_: SerializableSnapshot is NULL");

	buf[0] = 0;
	for (xcnt = 0; xcnt < SerializableSnapshot->xcnt; xcnt++)
	{
		sprintf(buf + strlen(buf), "%s\\'%u\\'::xxid", (xcnt) ? ", " : "",
				SerializableSnapshot->xip[xcnt]);
	}

	if ((ret = SPI_connect()) < 0)
		elog(ERROR, "_rserv_sync_: SPI_connect returned %d", ret);

	sprintf(sql, "insert into _RSERV_SYNC_ "
			"(server, syncid, synctime, status, minid, maxid, active) "
	  "values (%u, currval('_rserv_sync_seq_'), now(), 0, '%d', '%d', '%s')",
			server, SerializableSnapshot->xmin, SerializableSnapshot->xmax, active);

	ret = SPI_exec(sql, 0);

	if (ret < 0)
		elog(ERROR, "_rserv_sync_: SPI_exec returned %d", ret);

	SPI_finish();

	return (0);
}

#ifdef PG_FUNCTION_INFO_V1
Datum
_rserv_debug_(PG_FUNCTION_ARGS)
#else
int32
_rserv_debug_(int32 newval)
#endif
{
#ifdef PG_FUNCTION_INFO_V1
	int32		newval = PG_GETARG_INT32(0);

#endif
	int32		oldval = debug;

	debug = newval;

	return (oldval);
}

#define ExtendBy	1024

static char *
OutputValue(char *key, char *buf, int size)
{
	int			i = 0;
	char	   *out = buf;
	char	   *subst = NULL;
	int			slen = 0;

	size--;
	for (;;)
	{
		switch (*key)
		{
			case '\\':
				subst = "\\\\";
				slen = 2;
				break;
			case '	':
				subst = "\\011";
				slen = 4;
				break;
			case '\n':
				subst = "\\012";
				slen = 4;
				break;
			case '\'':
				subst = "\\047";
				slen = 4;
				break;
			case '\0':
				out[i] = 0;
				return (out);
			default:
				slen = 1;
				break;
		}

		if (i + slen >= size)
		{
			if (out == buf)
			{
				out = (char *) palloc(size + ExtendBy);
				strncpy(out, buf, i);
				size += ExtendBy;
			}
			else
			{
				out = (char *) repalloc(out, size + ExtendBy);
				size += ExtendBy;
			}
		}

		if (slen == 1)
			out[i++] = *key;
		else
		{
			memcpy(out + i, subst, slen);
			i += slen;
		}
		key++;
	}

	return (out);

}

static void	GetActiveLogTableId(void)
{
	LOCKTAG		tag;
	char		lockq[NAMEDATALEN + 64];
	int			ret;
	int32		altid;
	bool		isnull;

	/* Lock "pseudo" object */
	MemSet(&tag, 0, sizeof(tag));
	tag.relId = XactLockTableId;
	tag.dbId = 1;			/* InvalidOid is used in XactLockTableInsert */
	tag.objId.xid = InvalidTransactionId;

	if (!LockAcquire(LockTableId, &tag,
			GetCurrentTransactionId(), ShareLock, false))
		elog(ERROR, "_rserv_log_: failed to acquire ActiveLogTableIdLock");

	ret = SPI_exec("SELECT last_value::int4 FROM _rserv_active_log_id_", 2);

	if (ret < 0 || SPI_processed != 1)
		elog(ERROR, "_rserv_log_: failed to read ActiveLogTableId");

	altid = DatumGetInt32(
		SPI_getbinval(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1, &isnull)
		);

	if (altid != 1 && altid != 2)
		elog(ERROR, "_rserv_log_: bad ActiveLogTableId");

	ActiveLogTableId = altid;

	sprintf(lockq, "LOCK TABLE _RSERV_LOG_%d_ IN ROW EXCLUSIVE MODE", ActiveLogTableId);

	if (SPI_exec(lockq, 0) < 0)
		elog(ERROR, "_rserv_log_: failed to lock ActiveLogTable %d", ActiveLogTableId);

	MemSet(&tag, 0, sizeof(tag));
	tag.relId = XactLockTableId;
	tag.dbId = 1;			/* InvalidOid is used in XactLockTableInsert */
	tag.objId.xid = InvalidTransactionId;

	if (!LockRelease(LockTableId, &tag, GetCurrentTransactionId(), ShareLock))
		elog(ERROR, "_rserv_log_: failed to release ActiveLogTableIdLock");
}

#ifdef PG_FUNCTION_INFO_V1
Datum
_rserv_lock_altid_(PG_FUNCTION_ARGS)
#else
int32
_rserv_lock_altid_(int32 exclusive)
#endif
{
#ifdef PG_FUNCTION_INFO_V1
	int32		exclusive = PG_GETARG_INT32(0);
#endif
	LOCKTAG		tag;

	/* Lock "pseudo" object */
	MemSet(&tag, 0, sizeof(tag));
	tag.relId = XactLockTableId;
	tag.dbId = 1;			/* InvalidOid is used in XactLockTableInsert */
	tag.objId.xid = InvalidTransactionId;

	if (!LockAcquire(LockTableId, &tag, GetCurrentTransactionId(),
						(exclusive) ? ExclusiveLock : ShareLock, false))
		elog(ERROR, "_rserv_lock_altid_: failed to acquire %s ActiveLogTableIdLock",
			(exclusive) ? "Exclusive" : "Share");

	return(0);
}

#ifdef PG_FUNCTION_INFO_V1
Datum
_rserv_unlock_altid_(PG_FUNCTION_ARGS)
#else
int32
_rserv_unlock_altid_(int32 exclusive)
#endif
{
#ifdef PG_FUNCTION_INFO_V1
	int32		exclusive = PG_GETARG_INT32(0);
#endif
	LOCKTAG		tag;

	/* Lock "pseudo" object */
	MemSet(&tag, 0, sizeof(tag));
	tag.relId = XactLockTableId;
	tag.dbId = 1;			/* InvalidOid is used in XactLockTableInsert */
	tag.objId.xid = InvalidTransactionId;

	if (!LockRelease(LockTableId, &tag, GetCurrentTransactionId(),
						(exclusive) ? ExclusiveLock : ShareLock))
		elog(ERROR, "_rserv_unlock_altid_: failed to release %s ActiveLogTableIdLock",
			(exclusive) ? "Exclusive" : "Share");

	return(0);
}
