/*-------------------------------------------------------------------------
 *
 * utils.c		- utils for the PL/pgPSM
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/utils.c
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"
#include "lib/stringinfo.h"

/*
 * Returns true, when statement has nested statements
 *
 */
bool
plpgpsm_allow_cast_to_eb(PLpgPSM_astnode *astnode)
{
	switch (astnode->type)
	{
		case PLPGPSM_STMT_IF:
		case PLPGPSM_STMT_CASE:
		case PLPGPSM_CONDITIONAL_BLOCK:
		case PLPGPSM_SIMPLE_BLOCK:
		case PLPGPSM_STMT_LOOP:
		case PLPGPSM_STMT_WHILE:
		case PLPGPSM_STMT_REPEAT_UNTIL:
		case PLPGPSM_STMT_COMPOUND:
		case PLPGPSM_STMT_FOR:
			return true;
		default:
			return false;
	}
}

/*
 * Returns tag of any PLpgPSM AST node
 *
 */
char *
plpgpsm_astnode_tag(PLpgPSM_astnode *astnode)
{
	switch (astnode->type)
	{
		case PLPGPSM_STMT_IF:
			return "IF";
		case PLPGPSM_STMT_CASE:
			return "CASE";
		case PLPGPSM_CONDITIONAL_BLOCK:
			return "BLOCK WITH CONDITION";
		case PLPGPSM_SIMPLE_BLOCK:
			return "SIMPLE BLOCK";
		case PLPGPSM_STMT_RETURN:
			return "RETURN";
		case PLPGPSM_STMT_LOOP:
			return "LOOP";
		case PLPGPSM_STMT_WHILE:
			return "WHILE";
		case PLPGPSM_STMT_REPEAT_UNTIL:
			return "REPEAT UNTIL";
		case PLPGPSM_STMT_FOR:
			return "FOR";
		case PLPGPSM_STMT_PRINT:
			return "PRINT";
		case PLPGPSM_STMT_ASSIGN:
			return "ASSIGN";
		case PLPGPSM_STMT_LEAVE:
			return "LEAVE";
		case PLPGPSM_STMT_ITERATE:
			return "ITERATE";
		case PLPGPSM_STMT_OPEN:
			return "OPEN";
		case PLPGPSM_STMT_CLOSE:
			return "CLOSE";
		case PLPGPSM_STMT_FETCH:
			return "FETCH";
		case PLPGPSM_STMT_DIAGNOSTICS:
			return "GET DIAGNOSTICS";
		case PLPGPSM_CONDITION:
			return "CONDITION";
		case PLPGPSM_DECLARE_VARIABLE:
			return "DECLARE VARIABLE";
		case PLPGPSM_DECLARE_PARAM:
			return "DECLARE PARAM";
		case PLPGPSM_DECLARE_CURSOR:
			return "DECLARE CURSOR";
		case PLPGPSM_DECLARE_HANDLER:
			return "DECLARE HANDLER";
		case PLPGPSM_DECLARE_CONDITION:
			return "DECLARE_CONDITION";
		case PLPGPSM_STMT_COMPOUND:
			return "COMPOUND STATEMENT";
		case PLPGPSM_IDENT_VARIABLE:
			return "VARIABLE IDENTIFIER";
		case PLPGPSM_IDENT_TYPE:
			return "TYPE IDENTIFIER";
		case PLPGPSM_IDENT_CURSOR:
			return "CURSOR IDENTIFIER";
		case PLPGPSM_REFERER:
			return "REFERER";
		case PLPGPSM_VARIABLE:
			return "VARIABLE";
		case PLPGPSM_CURSOR:
			return "CURSOR";
		case PLPGPSM_ESQL:
			return "ESQL";
		case PLPGPSM_SIGNAL_INFO:
			return "SIGNAL INFO";
		case PLPGPSM_STMT_SIGNAL:
			return "SIGNAL";
		case PLPGPSM_STMT_RESIGNAL:
			return "RESIGNAL";
		case PLPGPSM_DIAG_INFO:
			return "DIAG INFO";
		default:
			elog(ERROR, "unknown astnode type %u %u", astnode->type, PLPGPSM_CURSOR);
			return false;		/* off warning */
	}
}

/*
 * returns PLpgPSM_ESQL pointers from any single statement without PRINT
 *
 */
PLpgPSM_ESQL *
plpgpsm_inner_esql(PLpgPSM_astnode *astnode)
{
	switch (astnode->type)
	{
		case PLPGPSM_STMT_IF:
		case PLPGPSM_STMT_CASE:
		case PLPGPSM_CONDITIONAL_BLOCK:
		case PLPGPSM_SIMPLE_BLOCK:
		case PLPGPSM_STMT_LOOP:
		case PLPGPSM_STMT_WHILE:
		case PLPGPSM_STMT_REPEAT_UNTIL:
		case PLPGPSM_STMT_FOR:
			return ((PLpgPSM_eb_stmt *) astnode)->expr;

		case PLPGPSM_DECLARE_VARIABLE:
			return ((PLpgPSM_declare_variable *) astnode)->expr;
		case PLPGPSM_STMT_RETURN:
			return ((PLpgPSM_stmt_return *) astnode)->expr;
		case PLPGPSM_STMT_ASSIGN:
			return ((PLpgPSM_stmt_assign *) astnode)->expr;
		case PLPGPSM_DECLARE_CURSOR:
			return ((PLpgPSM_declare_cursor *) astnode)->esql;
		case PLPGPSM_ESQL:
			return (PLpgPSM_ESQL *) astnode;

		case PLPGPSM_VARIABLE:
		case PLPGPSM_CURSOR:
		case PLPGPSM_REFERER:
		case PLPGPSM_STMT_COMPOUND:
		case PLPGPSM_STMT_OPEN:
		case PLPGPSM_STMT_CLOSE:
		case PLPGPSM_STMT_FETCH:
		case PLPGPSM_STMT_LEAVE:
		case PLPGPSM_STMT_ITERATE:
		case PLPGPSM_DECLARE_CONDITION:
		case PLPGPSM_DECLARE_HANDLER:
		case PLPGPSM_IDENT_VARIABLE:
		case PLPGPSM_IDENT_CURSOR:
		case PLPGPSM_IDENT_TYPE:
		case PLPGPSM_DIAG_INFO:
		case PLPGPSM_STMT_DIAGNOSTICS:
			return NULL;

		default:
			elog(ERROR, "unknown astnode type %u", astnode->type);
			return NULL;		/* off warning */
	}
}

/*
 * Dump ast node
 *
 */
static void
dump_astnode(PLpgPSM_astnode *astnode, int level, StringInfo str)
{
	if (astnode == NULL)
		return;

	appendStringInfoSpaces(str, (level * 2));

	if (astnode->is_visible)
		appendStringInfo(str, "%s", plpgpsm_astnode_tag(astnode));
	else
		appendStringInfo(str, "|%s", plpgpsm_astnode_tag(astnode));

	if (plpgpsm_allow_cast_to_eb(astnode))
	{
		ListCell *lc;

		PLpgPSM_eb_stmt *ebstmt = (PLpgPSM_eb_stmt *) astnode;

		if (ebstmt->label != NULL)
			appendStringInfo(str, " label: %s", ebstmt->label);

		if (ebstmt->option)
			appendStringInfo(str, " option: true");

		if (ebstmt->namespace != NULL)
			appendStringInfo(str, " namespace: %s", ebstmt->namespace);

		if (ebstmt->expr != NULL)
			appendStringInfo(str, " expr: \"%s\"", ebstmt->expr->sqlstr);

		appendStringInfoChar(str, '\n');

		if (ebstmt->cursor != NULL)
			dump_astnode((PLpgPSM_astnode *) ebstmt->cursor, level + 3, str);

		foreach(lc, ebstmt->stmts)
		{
			dump_astnode((PLpgPSM_astnode *) lfirst(lc), level + 1, str);
		}

	}
	else if (astnode->type == PLPGPSM_CONDITION)
	{
		PLpgPSM_condition *c = (PLpgPSM_condition *) astnode;

		switch (c->type)
		{
			case PLPGPSM_CONDITION_SQLSTATE:
				appendStringInfo(str, " %s\n",
							    unpack_sql_state(c->sqlstate));
				break;
			case PLPGPSM_CONDITION_NOTFOUND:
				appendStringInfoString(str, " NOT FOUND\n");
				break;
			case PLPGPSM_CONDITION_SQLWARNING:
				appendStringInfoString(str, " SQLWARNING\n");
				break;
			case PLPGPSM_CONDITION_SQLEXCEPTION:
				appendStringInfoString(str, " SQLEXCEPTION\n");
				break;
			case PLPGPSM_CONDITION_NAMED:
				appendStringInfo(str, " %s", c->name);
				if (c->sqlstate != 0)
					appendStringInfo(str, " %s",
								    unpack_sql_state(c->sqlstate));
				appendStringInfoChar(str, '\n');
				break;
		}
	}
	else if (astnode->type == PLPGPSM_STMT_RETURN)
	{
		PLpgPSM_stmt_return *stmt = (PLpgPSM_stmt_return *) astnode;

		if (stmt->expr != NULL)
			appendStringInfo(str, " expr: \"%s\"", stmt->expr->sqlstr);

		appendStringInfoChar(str, '\n');
	}
	else if (astnode->type == PLPGPSM_STMT_PRINT)
	{
		ListCell	*lc;
		PLpgPSM_stmt_print *stmt = (PLpgPSM_stmt_print *) astnode;

		appendStringInfoChar(str, '\n');

		foreach (lc, stmt->expr_list)
		{
			PLpgPSM_ESQL	*esql = (PLpgPSM_ESQL *) lfirst(lc);

			appendStringInfoSpaces(str, level * 2);
			appendStringInfo(str, "   expr: \"%s\"\n", esql->sqlstr);
		}
	}
	else if (astnode->type == PLPGPSM_ESQL)
	{
		appendStringInfo(str, "  expr: \"%s\"\n", ((PLpgPSM_ESQL *) astnode)->sqlstr);
	}
	else if (astnode->type == PLPGPSM_STMT_ASSIGN)
	{
		PLpgPSM_stmt_assign *stmt = (PLpgPSM_stmt_assign *) astnode;

		appendStringInfo(str, " expr: \"%s\"\n", stmt->expr->sqlstr);

		if (stmt->target != NULL)
		{
			dump_astnode((PLpgPSM_astnode *) stmt->target, level + 1, str);
		}
		else
		{
			ListCell	*lc;

			Assert(stmt->target_list != NULL);

			foreach(lc, stmt->target_list)
			{
				PLpgPSM_referer *ref = (PLpgPSM_referer *) lfirst(lc);
				dump_astnode((PLpgPSM_astnode *) ref, level + 1, str);
			}
		}
	}
	else if (astnode->type == PLPGPSM_STMT_LEAVE || astnode->type == PLPGPSM_STMT_ITERATE)
	{
		PLpgPSM_stmt_leave_iterate *stmt = (PLpgPSM_stmt_leave_iterate *) astnode;

		appendStringInfo(str, " %s\n", stmt->label);
	}
	else if (astnode->type == PLPGPSM_IDENT_VARIABLE || astnode->type == PLPGPSM_IDENT_CURSOR)
	{
		PLpgPSM_ident *i = (PLpgPSM_ident *) astnode;

		if (i->scope != NULL)
			appendStringInfo(str, " \"%s\".\"%s\"\n", i->scope, i->name);
		else
			appendStringInfo(str, " \"%s\"\n", i->name);
	}
	else if (astnode->type == PLPGPSM_DECLARE_VARIABLE)
	{
		PLpgPSM_declare_variable *dv = (PLpgPSM_declare_variable *) astnode;
		ListCell	*lc;

		/* type can be undefined - implicit DECLARE in CASE statement */
		if (dv->typename != NULL)
			appendStringInfo(str, " type: \"%s\"", dv->typename->name);

		if (dv->expr != NULL)
			appendStringInfo(str, " expr: \"%s\"", dv->expr->sqlstr);

		appendStringInfoChar(str, '\n');

		foreach(lc, dv->variables)
		{
			dump_astnode((PLpgPSM_astnode *) lfirst(lc), level + 1, str);
		}
	}
	else if (astnode->type == PLPGPSM_DECLARE_PARAM)
	{
		PLpgPSM_declare_param *dp = (PLpgPSM_declare_param *) astnode;

		appendStringInfo(str, " parameter: %d typeid: %d mode: %c\n", dp->param_offset, dp->typeid, dp->param_mode);
		dump_astnode((PLpgPSM_astnode *) dp->variable, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_DECLARE_CURSOR)
	{
		PLpgPSM_declare_cursor *declcursor = (PLpgPSM_declare_cursor *) astnode;

		if (declcursor->option == PLPGPSM_CURSOR_SCROLL)
			appendStringInfo(str, " SCROLL");
		appendStringInfoChar(str, '\n');
		dump_astnode((PLpgPSM_astnode *) declcursor->cursor, level + 1, str);
		dump_astnode((PLpgPSM_astnode *) declcursor->esql, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_DECLARE_CONDITION)
	{
		PLpgPSM_declare_condition *declcond = (PLpgPSM_declare_condition *) astnode;

		appendStringInfoChar(str, '\n');
		dump_astnode((PLpgPSM_astnode *) declcond->condition, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_DECLARE_HANDLER)
	{
		PLpgPSM_declare_handler *declhandler = (PLpgPSM_declare_handler *) astnode;
		ListCell	*lc;

		switch (declhandler->handler_type)
		{
			case PLPGPSM_HANDLER_CONTINUE:
				appendStringInfoString(str, " CONTINUE\n");
				break;
			case PLPGPSM_HANDLER_EXIT:
				appendStringInfoString(str, "EXIT\n");
				break;
			case PLPGPSM_HANDLER_UNDO:
				appendStringInfoString(str, "UNDO\n");
				break;
		}

		foreach (lc, declhandler->condition_list)
		{
			dump_astnode((PLpgPSM_astnode *) lfirst(lc), level + 1, str);
		}

		dump_astnode((PLpgPSM_astnode *) declhandler->stmt, level + 2, str);
	}
	else if (astnode->type == PLPGPSM_REFERER)
	{
		PLpgPSM_referer *r = (PLpgPSM_referer *) astnode;

		appendStringInfoChar(str, '\n');

		dump_astnode((PLpgPSM_astnode *) r->ident, level + 1, str);
		dump_astnode((PLpgPSM_astnode *) r->ref, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_VARIABLE)
	{
		PLpgPSM_variable *v = (PLpgPSM_variable *) astnode;

		appendStringInfo(str, " \"%s\" offset: %d", v->name, v->offset);

		if (v->special == PLPGPSM_SPECIAL_SQLCODE)
			appendStringInfo(str, " special: SQLCODE");
		else if (v->special == PLPGPSM_SPECIAL_SQLSTATE)
			appendStringInfo(str, " special: SQLSTATE");

		appendStringInfoChar(str, '\n');
	}
	else if (astnode->type == PLPGPSM_CURSOR)
	{
		PLpgPSM_cursor *c = (PLpgPSM_cursor *) astnode;

		appendStringInfo(str, " \"%s\" offset: %d\n", c->name, c->offset);
	}
	else if (astnode->type == PLPGPSM_STMT_OPEN)
	{
		appendStringInfoChar(str, '\n');
		dump_astnode((PLpgPSM_astnode *)((PLpgPSM_stmt_open *) astnode)->cursor, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_STMT_CLOSE)
	{
		appendStringInfoChar(str, '\n');
		dump_astnode((PLpgPSM_astnode *)((PLpgPSM_stmt_open *) astnode)->cursor, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_STMT_FETCH)
	{
		PLpgPSM_stmt_fetch *f = (PLpgPSM_stmt_fetch *) astnode;
		ListCell	*lc;

		appendStringInfoChar(str, '\n');
		dump_astnode((PLpgPSM_astnode *) f->cursor, level + 1, str);

		Assert(f->target_list != NULL);

		foreach(lc, f->target_list)
		{
			PLpgPSM_referer *ref = (PLpgPSM_referer *) lfirst(lc);
			dump_astnode((PLpgPSM_astnode *) ref, level + 1, str);
		}
	}
	else if (astnode->type == PLPGPSM_SIGNAL_INFO)
	{
		PLpgPSM_signal_info *sinfo = (PLpgPSM_signal_info *) astnode;

		switch (sinfo->type)
		{
			case PLPGPSM_SIGNAL_INFO_MESSAGE:
				appendStringInfoString(str, " MESSAGE_TEXT\n");
				break;
			case PLPGPSM_SIGNAL_INFO_DETAIL:
				appendStringInfoString(str, " DETAIL_TEXT\n");
				break;
			case PLPGPSM_SIGNAL_INFO_HINT:
				appendStringInfoString(str, " HINT_TEXT\n");
				break;
		}
		dump_astnode((PLpgPSM_astnode *) sinfo->expr, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_STMT_SIGNAL || astnode->type == PLPGPSM_STMT_RESIGNAL)
	{
		PLpgPSM_stmt_signal *stmt = (PLpgPSM_stmt_signal *) astnode;
		ListCell	*l;

		if (stmt->handler != NULL)
			appendStringInfoString(str, " WITH STATIC HANDLING");

		appendStringInfoChar(str, '\n');

		dump_astnode((PLpgPSM_astnode *) stmt->condition, level + 1, str);
		foreach(l, stmt->signal_info_list)
		{
			dump_astnode((PLpgPSM_astnode *) lfirst(l), level + 1, str);
		}
	}
	else if (astnode->type == PLPGPSM_DIAG_INFO)
	{
		PLpgPSM_diag_info *dinfo = (PLpgPSM_diag_info *) astnode;

		switch (dinfo->type)
		{
			case PLPGPSM_DIAGINFO_RETURNED_SQLSTATE:
				appendStringInfoString(str, " RETURNED_SQLSTATE\n");
				break;
			case PLPGPSM_DIAGINFO_RETURNED_SQLCODE:
				appendStringInfoString(str, " RETURNED_SQLCODE\n");
				break;
			case PLPGPSM_DIAGINFO_MESSAGE_TEXT:
				appendStringInfoString(str, " MESSAGE_TEXT\n");
				break;
			case PLPGPSM_DIAGINFO_DETAIL_TEXT:
				appendStringInfoString(str, " DETAIL_TEXT\n");
				break;
			case PLPGPSM_DIAGINFO_HINT_TEXT:
				appendStringInfoString(str, " HINT_TEXT\n");
				break;
			case PLPGPSM_DIAGINFO_ROW_COUNT:
				appendStringInfoString(str, " ROW_COUNT\n");
				break;
			case PLPGPSM_DIAGINFO_CONDITION_IDENTIFIER:
				appendStringInfoString(str, " CONDITION_IDENTIFIER\n");
				break;
		}

		dump_astnode((PLpgPSM_astnode *) dinfo->target, level + 1, str);
	}
	else if (astnode->type == PLPGPSM_STMT_DIAGNOSTICS)
	{
		PLpgPSM_stmt_diagnostics *stmt = (PLpgPSM_stmt_diagnostics *) astnode;
		ListCell	*l;

		switch (stmt->diag_area)
		{
			case PLPGPSM_DIAGAREA_STACKED:
				appendStringInfoString(str, " STACKED\n");
				break;
			case PLPGPSM_DIAGAREA_CURRENT:
				appendStringInfoString(str, " CURRENT\n");
				break;
		}

		foreach(l, stmt->diag_info_list)
		{
			PLpgPSM_astnode *iter = (PLpgPSM_astnode *) lfirst(l);

			Assert(iter->type == PLPGPSM_DIAG_INFO);
			dump_astnode(iter, level + 1, str);
		}
	}
	else
		elog(ERROR, "unknown astnode type %u", astnode->type);
}

/*
 * Helps with parser debugging
 *
 */
void
plpgpsm_elog_astnode(int level, PLpgPSM_astnode *astnode)
{
	StringInfoData		str;

	initStringInfo(&str);
	dump_astnode(astnode, 0, &str);

	elog(level, "Parser Result\n%s", str.data);
}

