%{	
/*-------------------------------------------------------------------------
 *
 * gram.y				- Parser for the PL/pgSQL
 *						  procedural language
 *
 * Portions Copyright (c) 1996-2006, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/pl/plpgpsm/src/gram.y,v 1.95 2006/08/14 21:14:41 tgl Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include "postgres.h"
#include "parser/parser.h"
#include "executor/spi_priv.h"
#include "utils/elog.h"
#include "funcapi.h"
#include "utils/builtins.h"

/*
 * Bison doesn't allocate anything that needs to live across parser calls,
 * so we can easily have it use palloc instead of malloc.  This prevents
 * memory leaks if we error out during parsing.  Note this only works with
 * bison >= 2.0.  However, in bison 1.875 the default is to use alloca()
 * if possible, so there's not really much problem anyhow, at least if
 * you're building with gcc.
 */
#define YYMALLOC palloc
#define YYFREE   pfree

typedef struct                                                                                                   
{                                                                                                                
    const char *label;                                                                                       
    int                     sqlerrstate;                                                                     
} ExceptionLabelMap; 

extern const ExceptionLabelMap exception_label_map[];

extern PLpgPSM_datum **plpgpsm_Datums;

static int sqlerrstate_to_elevel(int errstate);

static int assign_expr_param(int dno, int *params, int *nparams);

static int assign_init_expr_param(int dno, PLpgPSM_init_params **params);

static bool is_error(int errstate);

static void check_uniq_condition(PLpgPSM_condition *c1, PLpgPSM_condition *c2, const char *hint);

static void check_uniq_handlers(List *exceptions, PLpgPSM_condition *cond);


static PLpgPSM_stmt_fors *register_from_expr(char *label, 
							    int lno, 
							    char *inner_label, 
							    PLpgPSM_expr *expr);



static PLpgPSM_expr *read_sql_stmt(char *sqlstart);

static PLpgPSM_expr		*read_sql_construct(int until,
											int until2,
											const char *expected,
											char *sqlstart,
											bool isexpression,
											bool valid_sql,
											int *endtoken,
											const char *sqlfinish);
static	PLpgPSM_type	*read_datatype(int tok);
static	PLpgPSM_stmt	*make_execsql_stmt(const char *sqlstart, int lineno);
static  PLpgPSM_stmt_fetch *read_fetch_direction(void);
static	PLpgPSM_stmt	*make_return_stmt(int lineno, bool is_tblexpr);
static  PLpgPSM_stmt 	*make_case(int lineno, PLpgPSM_expr *t_expr,
								   List *case_when_list, List *else_stmts);

static	void			 check_assignable(PLpgPSM_datum *datum);
static	void			 read_into_target(PLpgPSM_rec **rec, PLpgPSM_row **row,
										  bool *strict);
static	PLpgPSM_row		*read_into_scalar_list(const char *initial_name,
											   PLpgPSM_datum *initial_datum);
static	void			 check_sql_expr(const char *stmt);
static	void			 plpgpsm_sql_error_callback(void *arg);
static	void			 check_labels(const char *start_label,
									  const char *end_label);
static	char *check_label(const char *yytxt);
static void set_signal_stmt(PLpgPSM_stmt_signal *stmt, PLpgPSM_signal_info *info);


static void merge_init_expr_param(PLpgPSM_init_params **dest, PLpgPSM_init_params **src);


#define MAX_EXPR_PARAMS  1024
                                                                                                                       

%}

%expect 0
%name-prefix="plpgpsm_yy"

%union {
		int32					ival;
		bool					boolean;
		char					*str;
		struct
		{
			char *name;
			int  lineno;
		}						varname;
		struct
		{
			char *name;
			int  lineno;
			PLpgPSM_datum   *scalar;
			PLpgPSM_rec     *rec;
			PLpgPSM_row     *row;
		}						forvariable;
		struct
		{
			char *label;
			int  n_initvars;
			int  *initvarnos;
		}						declhdr;
		struct
		{
			char *end_label;
			List *stmts;
		}						loop_body;
		struct
		{
			int sqlerrstate;
			bool defined;
		}						sqlerrstate;
		struct
		{	
			List	*expr_list;
			PLpgPSM_expr	*expr;	
		}						when_expr;
		struct 
		{
			int deep;
			int varno;
			bool defined;
		}						exception_deep;
		struct
		{
			int value;
			int kind;
		}						diag_identif;
		struct
		{
			int	decl_type;
			int	lineno;
			char	*name;
			union  
			{
			    PLpgPSM_init_params 	*params;
			    struct
			    {
				int	hdl_type;
				PLpgPSM_exception	*exception;
			    } handler;
			} result;
		}						declaration;
		struct
		{
			int  dno;
			bool scrollable;
			char *prepname;				/* prepared statement's name for dynamic cursor */
		}						cursor_var;
		List					*list;
		PLpgPSM_type			*dtype;
		PLpgPSM_datum			*scalar;	/* a VAR, RECFIELD, or TRIGARG */
		PLpgPSM_variable		*variable;	/* a VAR, REC, or ROW */
		PLpgPSM_var				*var;
		PLpgPSM_row				*row;
		PLpgPSM_rec				*rec;
		PLpgPSM_expr			*expr;
		PLpgPSM_stmt			*stmt;
		PLpgPSM_program			*program;
		PLpgPSM_condition		*condition;
		PLpgPSM_exception		*exception;
		PLpgPSM_exception_block	*exception_block;
		PLpgPSM_nsitem			*nsitem;
		PLpgPSM_diag_item		*diagitem;
		PLpgPSM_case_when		*whenclause;
		PLpgPSM_stmt_signal		*signal;
		PLpgPSM_signal_info		signalinfo;
		PLpgPSM_cond_var		*condvar;
		PLpgPSM_init_params		*initpar;
		PLpgPSM_assign_item		*assign_item;
		PLpgPSM_stmt_fors		*for_header;
		PLpgPSM_decl_block		*decl_block;
		PLpgPSM_stmt_fetch      	*fetch;
}


%type <ival>	lno
%type <program>	pl_function

%type <stmt> stmt_compound_stmt stmt_sqlpsm stmt_return 
%type <stmt> stmt_loop stmt_while 
%type <stmt> stmt_repeat stmt_signal stmt_resignal stmt_assign stmt_if
%type <stmt> stmt_print stmt_leave stmt_case stmt_sql stmt_for
%type <stmt> stmt_execute_imm stmt_call
%type <stmt> stmt_execute stmt_prepare
%type <stmt> stmt_getdiag 
%type <stmt> stmt_fetch stmt_move stmt_open stmt_close

%type <list> sql_stmt_list 

%type <str>  sql_start 
%type <fetch> opt_fetch_direction

%type <str>  sqlname varname

%type <ival> assign_var assign_scalar_list

%type <expr>    expr_until_semi expr_until_rightbracket expr_until_end expr_until_do
%type <expr>	opt_expr_until_when_r expr_until_then 
%type <str>     block_lblname opt_block_label opt_label_semi lblname opt_semi
%type <str>	sqlstate_value_opt sqlstate_value_str

%type <sqlerrstate> opt_sqlstate_value


%type <list>	assign_list
%type <dtype>	decl_datatype

%type <list> getdiag_cond_list
%type <diagitem> getdiag_cond_item
%type <ival>  getdiag_target

%type <boolean>	opt_atomic leave_type 

%type <decl_block> local_decl_list 
%type <ival>  sqlstate_value 
%type <str> signal_info_item_name

%type <signal> signal_info_item_list set_signal_info
%type <signalinfo> signal_info_item

%type <whenclause> when_clause
%type <list> when_clause_list opt_case_default stmt_else proc_sect
%type <condition> signal_value condition_value condition_value_list opt_signal_value

%type <list> opt_datatype_list datatype_list

%type <cursor_var> cursor_variable

%type <assign_item> assign_item


%type <loop_body> for_body

%type <for_header> for_header

%type <exception_deep> exception_deep opt_exception_deep

%type <ival>  opt_diag_area
%type <diag_identif> signal_info_identif_excpt

%type <declaration> decl_head local_decl 



		/*
		 * Keyword tokens
		 */
%token  K_AS
%token  K_ATOMIC
%token	K_BEGIN
%token  K_CALL
%token  K_CASE
%token  K_CLOSE
%token  K_CONDITION
%token  K_CONTINUE
%token  K_CURSOR
%token	K_DECLARE
%token  K_DIAGNOSTICS
%token	K_DEFAULT
%token  K_DO
%token	K_ELSE
%token	K_ELSEIF
%token	K_END
%token  K_EXECUTE
%token	K_EXCEPTION
%token	K_EXIT
%token  K_FETCH
%token	K_FOR
%token  K_FOUND
%token  K_FROM
%token  K_GET
%token  K_HANDLER
%token	K_HOLDABLE
%token	K_IF
%token  K_INSERT
%token	K_INTO
%token  K_IMMEDIATE
%token  K_ITERATE
%token  K_LEAVE
%token	K_LOOP
%token	K_NOT
%token  K_NEXT
%token  K_NULL
%token  K_OPEN
%token  K_PRINT
%token  K_REPEAT
%token  K_RESIGNAL
%token  K_RETURN
%token	K_SCROLL
%token	K_MOVE
%token  K_NOSCROLL
%token	K_NOHOLDABLE
%token  K_SET
%token  K_SIGNAL
%token  K_STRICT
%token  K_SQLWARNING
%token  K_SQLEXCEPTION
%token  K_TABLE
%token	K_THEN
%token	K_WHEN
%token	K_WHILE
%token  K_UNDO
%token  K_UNTIL
%token  K_VALUE
%token	K_VALUES
%token  K_SQLSTATE
%token	K_PREPARE
%token  K_USING


%token  K_CREATE
%token  K_SELECT
%token  K_DROP
%token  K_DELETE
%token  K_UPDATE
%token  K_GRANT
%token  K_REVOKE
%token  K_TRUNCATE

		/*
		 * Other tokens
		 */
%token	T_FUNCTION
%token	T_TRIGGER
%token	T_STRING
%token	T_NUMBER
%token	T_SCALAR				/* a VAR, RECFIELD, or TRIGARG */
%token	T_ROW
%token	T_RECORD
%token	T_DTYPE
%token	T_WORD
%token	T_ERROR
%token  T_CONDITION

%token	O_OPTION
%token	O_DUMP

%%

pl_function			: T_FUNCTION comp_optsect stmt_sqlpsm opt_semi
					{
						PLpgPSM_program *new = palloc0(sizeof(PLpgPSM_program));
					
						if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
							new->sqlstate_varno = yylval.scalar->dno;
						else 	
							new->sqlstate_varno = PLPGPSM_INVALID_VARNO;
							
						new->body = $3;

						yylval.program = new;


					}
				| T_TRIGGER comp_optsect stmt_sqlpsm opt_semi
					{
						PLpgPSM_program *new = palloc0(sizeof(PLpgPSM_program));
					
						new->sqlstate_varno = PLPGPSM_INVALID_VARNO;
						new->body = $3;
					
						yylval.program = new;
					}
				;

comp_optsect	:
				| O_OPTION comp_options
				;

comp_options	: comp_options comp_option
				| comp_option
				;

comp_option		: O_DUMP
					{
						plpgpsm_DumpExecTree = true;
					}
				;

stmt_compound_stmt		: opt_block_label K_BEGIN lno opt_atomic local_decl_list sql_stmt_list K_END opt_label_semi
					{
						PLpgPSM_stmt_block *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_block));
						
						new->cmd_type = PLPGPSM_STMT_BLOCK;
						new->lineno		= $3;
						
						new->label		= $1;
						new->atomic		= $4;
						new->generated 	= false;

						/* because is possible declare next variables inside
						 * error handler, isn't possible use older method for
						 * building list of initialised variables
						 */
						
						if ($5->params)
						{
							new->n_initvars		= $5->params->nparams;
							new->initvarnos		= (int *) palloc(sizeof(int)*$5->params->nparams);
							memcpy(new->initvarnos, $5->params->params, sizeof(int)*$5->params->nparams);
						}
						
						/* check sqlstate variable */
						if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
							new->sqlstate_varno = yylval.scalar->dno;
						else 	
							new->sqlstate_varno = PLPGPSM_INVALID_VARNO;
						
						plpgpsm_add_initdatums(NULL);
						
						new->body 		= $6;
						new->exceptions		= $5->handlers;
						check_labels($1, $8);

						if ($5->handlers && $5->handlers->has_undo_handler && !$4)
							ereport(ERROR,
									(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
									 errmsg("UNDO handler can be only in ATOMIC block")));

						plpgpsm_ns_pop();

						$$ = (PLpgPSM_stmt *) new;
					}
				| opt_block_label K_BEGIN lno opt_atomic sql_stmt_list K_END opt_label_semi
					{
						PLpgPSM_stmt_block *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_block));
						
						new->cmd_type = PLPGPSM_STMT_BLOCK;
						new->lineno		= $3;
						
						new->label		= $1;
						new->atomic		= $4;
						new->generated 	= false;

						/* check sqlstate variable */
						if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
							new->sqlstate_varno = yylval.scalar->dno;
						else 	
							new->sqlstate_varno = PLPGPSM_INVALID_VARNO;

						plpgpsm_add_initdatums(NULL);

						new->body 		= $5;
						new->exceptions		= NULL;
						check_labels($1, $7);
						
						plpgpsm_ns_pop();
						$$ = (PLpgPSM_stmt *) new;

					}
				| opt_block_label K_BEGIN lno opt_atomic  K_END opt_label_semi
					{
						PLpgPSM_stmt_block *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_block));
						
						new->cmd_type = PLPGPSM_STMT_BLOCK;
						new->lineno		= $3;
						
						new->label		= $1;
						new->atomic		= $4;
						new->generated 	= false;

						/* check sqlstate variable */
						if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
							new->sqlstate_varno = yylval.scalar->dno;
						else 	
							new->sqlstate_varno = PLPGPSM_INVALID_VARNO;
						
						plpgpsm_add_initdatums(NULL);

						new->body 		= NULL;
						new->exceptions		= NULL;
						check_labels($1, $6);
						
						plpgpsm_ns_pop();
						$$ = (PLpgPSM_stmt *) new;
					}
				;

opt_atomic		:
					{ $$ = false; }
				| K_ATOMIC
					{ $$ = true; }
				| K_NOT K_ATOMIC
					{ $$ = false; }
				;
			
							
local_decl_list			: local_decl_list local_decl
					{
						if ($1->last_decl_type > $2.decl_type)
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("syntax error"),
									 errhint("check order of declaration variables and conditions, cursors and signal's handlers")));
						
						$1->last_decl_type = $2.decl_type;
						
						/* small fix */
						if ($2.decl_type == PLPGPSM_DECL_VARIABLES)
							$1->last_decl_type = PLPGPSM_DECL_CONDITION;
						
						switch ($2.decl_type)
						{
							case PLPGPSM_DECL_VARIABLES:
							case PLPGPSM_DECL_CURSOR:
								merge_init_expr_param(&($1->params), &($2.result.params));
								break;
							
							case PLPGPSM_DECL_HANDLER:
								if (!$1->handlers)
									$1->handlers = palloc0(sizeof(PLpgPSM_exception_block));
								

								/* have to detect CONTINUE and EXIT handler type only for 
								 * exceptions. For warning, I need do nothing. UNDO is controlled
								 * BEGIN ATOMIC flag.
								 */
								if ($2.result.handler.exception->handle_errors)
									switch($2.result.handler.hdl_type)
									{
										case PLPGPSM_HDL_CONTINUE:
											$1->handlers->has_continue_handler = true;
											break;
										
										case PLPGPSM_HDL_EXIT:
											$1->handlers->has_exit_handler = true;
											break;
									}
									
								if ($2.result.handler.hdl_type == PLPGPSM_HDL_UNDO)
									$1->handlers->has_undo_handler = true;

								if ($2.result.handler.exception->handle_errors)
									$1->handlers->has_exception_handler = true;
									
								check_uniq_handlers($1->handlers->exc_list, $2.result.handler.exception->conditions);

								$1->handlers->exc_list = lappend($1->handlers->exc_list, $2.result.handler.exception);
								break;	
						}
						
						$$ = $1;
					}
				| local_decl
					{
						/* transform from declaration to decl_block */
						
						PLpgPSM_decl_block	*new = palloc0(sizeof(PLpgPSM_decl_block));
						
						switch($1.decl_type)
						{
							case PLPGPSM_DECL_CONDITION:
								new->last_decl_type = PLPGPSM_DECL_CONDITION;
								new->handlers = NULL;
								new->params = NULL;
								break;
								
							case PLPGPSM_DECL_CURSOR:
								new->last_decl_type = PLPGPSM_DECL_CURSOR;
								new->handlers = NULL;
								new->params = $1.result.params;
								break;

							case PLPGPSM_DECL_VARIABLES:
								new->last_decl_type = PLPGPSM_DECL_CONDITION;
								new->handlers = NULL;
								new->params = $1.result.params;
								break;

							case PLPGPSM_DECL_HANDLER:
								new->last_decl_type = PLPGPSM_DECL_HANDLER;
								new->params = NULL;
								
								new->handlers = palloc0(sizeof(PLpgPSM_exception_block));
								
								/* have to detect CONTINUE and EXIT handler type only for 
								 * exceptions. For warning, I need do nothing. UNDO is controlled
								 * BEGIN ATOMIC flag.
								 */
								if ($1.result.handler.exception->handle_errors)
									switch($1.result.handler.hdl_type)
									{
										case PLPGPSM_HDL_CONTINUE:
											new->handlers->has_continue_handler = true;
											break;
										
										case PLPGPSM_HDL_EXIT:
											new->handlers->has_exit_handler = true;
											break;
									}
									
								if ($1.result.handler.hdl_type == PLPGPSM_HDL_UNDO)
									new->handlers->has_undo_handler = true;
								
								new->handlers->has_exception_handler = $1.result.handler.exception->handle_errors;
								new->handlers->sqlstate_varno = PLPGPSM_INVALID_VARNO;
								new->handlers->exc_list = list_make1($1.result.handler.exception);
								break;
								
							default:
								yyerror("unknown decl type");			
	    					}
	    					
	    					$$ = new;
					}
						
				    ;
				    

local_decl			: decl_head K_CONDITION opt_sqlstate_value ';'
					{
						PLpgPSM_cond_var *con;
						PLpgPSM_type	*dtype;
						
						if ($1.decl_type != PLPGPSM_DECL_CONDITION)
							yyerror("syntax error");
						
						dtype = (PLpgPSM_type *) palloc0(sizeof(PLpgPSM_type));
						dtype->typname = "condition";
						dtype->ttype = PLPGPSM_TTYPE_CONDITION;
						
						con = (PLpgPSM_cond_var *)plpgpsm_build_variable($1.name, $1.lineno,
													dtype, true, NULL);
													
						/* SQLSTATE P1001 means, don't use SQLSTATE, check only condition_name */
						if ($3.defined)
						{
							elog(WARNING, "It is not recommended to set SQLSTATE for CONDITION.");
							con->sqlerrstate = $3.sqlerrstate;
						}
						else
							con->sqlerrstate = MAKE_SQLSTATE('P','1','0','0','1');
						
						$$.decl_type = $1.decl_type;						
					}
				| decl_head K_HANDLER K_FOR condition_value_list stmt_sqlpsm
					{
						PLpgPSM_exception *new;
						
						if ($1.decl_type != PLPGPSM_DECL_HANDLER)
							yyerror("syntax error");
						
						new = palloc0(sizeof(PLpgPSM_exception));
						
						new->lineno	= $1.lineno;
						new->handler_type 	= $1.result.handler.hdl_type;
						new->conditions = $4;
						new->action	= $5;
						new->handle_errors = $4->handle_errors;
						
						$$.decl_type 	= $1.decl_type;
						$$.lineno 	= $1.lineno;
						$$.result.handler.hdl_type 	= $1.result.handler.hdl_type;
						$$.result.handler.exception 	= new;
					}		
				| decl_head 
					{
						if ($1.decl_type == PLPGPSM_DECL_HANDLER ||
						    $1.decl_type == PLPGPSM_DECL_CONDITION)
							yyerror("syntax error");

						$$.decl_type = $1.decl_type;
						$$.lineno = $1.lineno;
						$$.result.params = $1.result.params;
					}
				    ;
					

decl_head			: K_DECLARE lno
					{
						int tok;
						PLpgPSM_type	*dtype;
						PLpgPSM_expr	*expr;
						bool	scrollable = false;
						bool	holdable = false;
						List	*vars = NIL;
						char 	*name = NULL;
						PLpgPSM_init_params 	*params = NULL;
						ListCell	*l;
						
						$$.name = NULL;
						$$.lineno = $2;
						
						plpgpsm_ns_setlocal(true);
						
						tok = yylex();
						
						if (tok == T_WORD || tok == K_SQLSTATE)
						{
							/* read variable name */
							plpgpsm_convert_ident(yytext, &name, 1);
							
							/* read variable list if any */
							while ((tok = yylex()) == ',')
							{
								tok = yylex();
								
								switch (tok)
								{
									case T_WORD:
									case K_SQLSTATE:
										if (!vars)
											vars = list_make1(name);
										plpgpsm_convert_ident(yytext, &name, 1);
										vars = lappend(vars, name);
										break;
										
									default:
										ereport(ERROR,
												(errcode(ERRCODE_SYNTAX_ERROR),
												 errmsg("\"%s\" is not identifier", yytext)));
								}
							}
							
							/* check datatype and decl type */
							
							plpgpsm_ns_setlocal(false);
							
							switch (tok)
							{
								case K_CONDITION:
									if (vars)
										ereport(ERROR,
												(errcode(ERRCODE_SYNTAX_ERROR),
												 errmsg("syntax error"),
												 errhint("in this case, you cannot declare variable list")));
									
									$$.decl_type = PLPGPSM_DECL_CONDITION;
									$$.name = name;
									plpgpsm_push_back_token(tok);
									break;
									
								case K_CURSOR:
								case K_NOSCROLL:
								case K_SCROLL:
									if (vars)
										ereport(ERROR,
												(errcode(ERRCODE_SYNTAX_ERROR),
												 errmsg("syntax error"),
												 errhint("in this case, you cannot declare variable list")));
									
									scrollable = (tok == K_SCROLL);
																			
									if ((tok == K_SCROLL || tok == K_NOSCROLL) && yylex() != K_CURSOR)
										yyerror("expected CURSOR");
												 
									$$.decl_type = PLPGPSM_DECL_CURSOR;
									
									tok = yylex();
									if (tok  == K_HOLDABLE || tok == K_NOHOLDABLE)
									{
										holdable = (tok == K_HOLDABLE ? true : false);

										tok = yylex();
									}
									if (tok != K_FOR)
										yyerror("expected FOR");
									else
									{
										/* read cursor's params */
										PLpgPSM_var *new;
										PLpgPSM_expr *curname_def;
										char		buf[1024];
										char		*cp1;
										char		*cp2;

										new = (PLpgPSM_var *)
										plpgpsm_build_variable(name, $2,
												   plpgpsm_build_datatype(REFCURSOROID,
																  -1),	   true, NULL);												   
										assign_init_expr_param(new->dno, &params);
						
										curname_def = palloc0(sizeof(PLpgPSM_expr));
										curname_def->dtype = PLPGPSM_DTYPE_EXPR;
						
										strcpy(buf, "SELECT ");
										cp1 = new->refname;
										cp2 = buf + strlen(buf);
										/*
										 * Don't trust standard_conforming_strings here;
										 * it might change before we use the string.
										 */
										if (strchr(cp1, '\\') != NULL)
											*cp2++ = ESCAPE_STRING_SYNTAX;
										*cp2++ = '\'';
								    		while (*cp1)
				    						{
											if (SQL_STR_DOUBLE(*cp1, true))
												*cp2++ = *cp1;
											*cp2++ = *cp1++;
										}
										strcpy(cp2, "'::refcursor");
										curname_def->query = pstrdup(buf);
										new->default_val = curname_def;
										
										/* dynamic cursor support, use prepared statement */
										tok = yylex();
										switch (tok)
										{
											case T_WORD:
											case T_ERROR:
											case T_SCALAR:
											case T_ROW:
											case T_RECORD:
												plpgpsm_convert_ident(yytext, &name, 1);
												new->prepname = name;
												if (yylex() != ';')
													yyerror("expected ';'");
												break;

											default:
												plpgpsm_push_back_token(tok);
												new->cursor_explicit_expr = read_sql_stmt("");
										}
										new->cursor_explicit_argrow = -1;
										new->cursor_options = CURSOR_OPT_FAST_PLAN | 
													    (scrollable ? CURSOR_OPT_SCROLL : CURSOR_OPT_NO_SCROLL);
										new->holdable = holdable;
					
										/* init list I can use for info about opened or closed cursors */
						
										$$.result.params = params;
									}
									break;
									
								default:
									/* read datatype and default value */
									dtype = read_datatype(tok);
									yyclearin;
									
									expr = NULL;
									
									if ((tok = yylex()) == K_DEFAULT)
										expr = plpgpsm_read_expression(';',";");
									else
									
										if (tok != ';')		
											yyerror("expected ';'");
												 
									/* register variables */
									if (vars == NIL)
										vars = list_make1(name);
									
									foreach(l, vars)
									{
										char *varname = (char*)lfirst(l);
										List *dno_lst = NIL;
										ListCell *l;
									
										/* 
										 * because plpgpsm_build_variables can create others
										 * invisible, but initialisable variables, I have to
										 * local list of initialised variables.
										 */ 
										PLpgPSM_variable *var = plpgpsm_build_variable(varname, $2, 
															     dtype, true, &dno_lst);
										foreach(l, dno_lst)
										{
											int i = (int) lfirst(l);
											assign_init_expr_param(i, &params);
										}
										    
										if (expr != NULL)
										{
											if (var->dtype == PLPGPSM_DTYPE_VAR)
												((PLpgPSM_var *) var)->default_val = expr;
											else
												ereport(ERROR,
													(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
													 errmsg("default value for row or record variable is not supported")));
										}
									}
									
									$$.decl_type = PLPGPSM_DECL_VARIABLES;
									$$.result.params = params;
							}
							
						}
						else
						{
							/* read any signal handler head */
							
							plpgpsm_ns_setlocal(false);
							
							switch (tok)
							{
								case K_CONTINUE:
									$$.result.handler.hdl_type = PLPGPSM_HDL_CONTINUE;
									break;
								
								case K_EXIT:
									$$.result.handler.hdl_type = PLPGPSM_HDL_EXIT;
									break;

								case K_UNDO:
									$$.result.handler.hdl_type = PLPGPSM_HDL_UNDO;
									break;
						
								default:
									yyerror("expected CONTINUE, EXIT OR UNDO");
							}
							
							
							$$.decl_type = PLPGPSM_DECL_HANDLER;
							$$.name = name;
						}
					
					}
				    ;


opt_sqlstate_value		: 
					{
					        $$.defined = false;  
					}
				| K_FOR sqlstate_value
					{
						$$.sqlerrstate = $2;
						$$.defined = true;
					}
				    ;
				

sqlstate_value			: K_SQLSTATE sqlstate_value_opt sqlstate_value_str
					{ 
						int errstate;
					
						if (strlen($3) != 5)
							yyerror("Sqlstate has five chars");
							
						errstate = MAKE_SQLSTATE($3[0],$3[1],$3[2],$3[3],$3[4]);
							
						if (errstate == MAKE_SQLSTATE('0','0','0','0','0'))
							ereport(ERROR,
									(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
									 errmsg("shall not use  SQLSTATE value for the condition successful completion")));

						if (errstate == MAKE_SQLSTATE('P','1','0','0','1'))
							ereport(ERROR,
									(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
									 errmsg("SQLSTATE 'P1000' is reserved"),
									 errhint("you cannot signal this sqlstate")));
							
						$$ = errstate;
					}
				;				

sqlstate_value_opt		:
					{}
				| K_VALUE
					{}
				    ;

sqlstate_value_str		: T_STRING
					{
						$$ = plpgpsm_get_string_value();
					}
				    ;



decl_datatype			:
					{
						$$ = read_datatype(yychar);
						yyclearin;
					}
				;
				
			
condition_value_list		: condition_value_list ',' condition_value
					{
						PLpgPSM_condition  *old;
						
						/* 
						 * if a <condition value> specifies SQLEXCEPTION, SQLWARNING, or NOT FOUND, 
						 * then neither <sqlstate  value> nor <condition value> shall be specified.
						 */
						 
						 if ($1->general_handler || $3->general_handler)
						    	    ereport(ERROR,
						    			(errcode(ERRCODE_INVALID_PARAMETER_VALUE),
									 errmsg("state class can be defined only once"),
									 errhint("you cannot use SQLEXCEPTION, SQLWARNING or NOT FOUND together with others")));

						
						/* 
						 * Checking if condition is unique in condition list
						 * and finding end of condition list
						 */
						for (old = $1; old->next != NULL; old = old->next)
							check_uniq_condition(old, $3, "Condition list contains same conditions");

						check_uniq_condition(old, $3, "Condition list contains same conditions");
						old->next = $3;
						
						if ($3->handle_errors)
							$1->handle_errors = true;
						
						$$ = $1;
					}
				| condition_value
					{
						$$ = $1;
						
					}
				    ;			


condition_value			: T_CONDITION
					{
						char *name;

						$$ = palloc0(sizeof(PLpgPSM_condition));
						
						plpgpsm_convert_ident(yytext, &name, 1);
						
						$$->sqlerrstate = yylval.condvar->sqlerrstate;
						$$->condname = name;
						$$->handle_errors = is_error($$->sqlerrstate);
						$$->general_handler = false;
					}
				| sqlstate_value
					{
						char unpacked_sqlstate[17];
					
						$$ = palloc0(sizeof(PLpgPSM_condition));
						$$->sqlerrstate = $1;						
							
						sprintf(unpacked_sqlstate, "SQLSTATE '%s'",
											    unpack_sql_state($1));
						$$->condname = pstrdup(unpacked_sqlstate);
						$$->handle_errors = is_error($$->sqlerrstate);
						$$->general_handler = false;
					}
				| T_WORD
					{
						char *name;
						
						plpgpsm_convert_ident(yytext, &name, 1);
					
						$$ = plpgpsm_parse_err_condition(name);
						$$->handle_errors = is_error($$->sqlerrstate);
						$$->general_handler = false;
								
					}
				| K_SQLEXCEPTION
					{
						$$ = palloc0(sizeof(PLpgPSM_condition));
						
						$$->sqlerrstate = MAKE_SQLSTATE('0','0','0','0','0');
						$$->condname = "*sqlexception*";
						$$->handle_errors = true;
						$$->general_handler = true;
					}
				| K_SQLWARNING
					{
						$$ = palloc0(sizeof(PLpgPSM_condition));
						
						$$->sqlerrstate = MAKE_SQLSTATE('0','1','0','0','0');
						$$->condname = "*sqlwarning*";
						$$->handle_errors = false;
						$$->general_handler = true;
					}
				| K_NOT K_FOUND
					{
						$$ = palloc0(sizeof(PLpgPSM_condition));
						
						$$->sqlerrstate = MAKE_SQLSTATE('0','2','0','0','0');
						$$->condname = "*not found*";
						$$->handle_errors = false;
						$$->general_handler = true;
					}
				;
				

sql_stmt_list			: sql_stmt_list stmt_sqlpsm
					{ $$ = lappend($1, $2); }
				| stmt_sqlpsm
					{ $$ = list_make1($1); }
				;


					
stmt_sqlpsm			: stmt_compound_stmt
					{ $$ = $1; }
				| stmt_return
					{ $$ = $1; }
				| stmt_assign
					{ $$ = $1; }
				| stmt_signal
					{ $$ = $1; }
				| stmt_resignal
					{ $$ = $1; }
				| stmt_print
					{ $$ = $1; }
				| stmt_case
					{ $$ = $1; }
				| stmt_execute_imm
					{ $$ = $1; }
				| stmt_sql
					{ $$ = $1; }
				| stmt_call
					{ $$ = $1; }
				| stmt_if
					{ $$ = $1; }
				| stmt_leave
					{ $$ = $1; }
				| stmt_loop
					{ $$ = $1; }
				| stmt_while
					{ $$ = $1; }
				| stmt_repeat
					{ $$ = $1; }
				| stmt_prepare
					{ $$ = $1; }
				| stmt_execute
					{ $$ = $1; }
				| stmt_fetch
					{ $$ = $1; }
				| stmt_move
					{ $$ = $1; }
				| stmt_open
					{ $$ = $1; }
				| stmt_close
					{ $$ = $1; }
				| stmt_for
					{ $$ = $1; }
				| stmt_getdiag
					{ $$ = $1; }
				    ;
			

stmt_return			: K_RETURN lno 
					{
				    		int tok;
					
						tok = yylex();
						if (tok == K_TABLE)
						{
							$$ = (PLpgPSM_stmt *)make_return_stmt($2, true);
						}
						else
						{
							plpgpsm_push_back_token(tok);
							$$ = (PLpgPSM_stmt *)make_return_stmt($2, false);
						}
					}
				;
				
				
stmt_print			: K_PRINT lno
					{
						PLpgPSM_stmt_print	*new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_print));
						
						new->cmd_type	= PLPGPSM_STMT_PRINT;
						new->lineno		= $2;
						new->params		= NIL;
						
						for (;;)
						{
							PLpgPSM_expr	*expr;
							int 	term;
						
							expr = read_sql_construct(',',';', ", or ;", "SELECT ",
											true, true, &term, NULL);
							new->params = lappend(new->params, expr);				
							if (term == ';')
								    break;
						}
						
						$$ = (PLpgPSM_stmt*) new;
					}
				;
				
stmt_assign			: K_SET lno assign_list ';'
					{
						PLpgPSM_stmt_assign *new;
					
						new = palloc0(sizeof(PLpgPSM_stmt_assign));
						new->cmd_type = PLPGPSM_STMT_ASSIGN;
						new->lineno	= $2;
						new->items	= $3;
						
						$$ = (PLpgPSM_stmt *)new;
					}	
				| K_SET lno '(' assign_scalar_list ')' '='
					{
						int tok;
						PLpgPSM_stmt_assign *new;
						PLpgPSM_assign_item *item;
						
						new = palloc0(sizeof(PLpgPSM_stmt_assign));
						new->cmd_type = PLPGPSM_STMT_ASSIGN;
						new->lineno	= $2;

						item = (PLpgPSM_assign_item *) palloc0(sizeof(PLpgPSM_assign_item));
						item->lineno 	= $2;
						item->varno	= $4;

						
						/* we can modify read_sql_construct, so if it's necessery
						 * cut subselect, which is simpler and faster then
						 * derived table (little bit, there is flattening)
						 */

						if ((tok = yylex()) != '(')
						{
							plpgpsm_push_back_token(tok);
							item->expr = read_sql_construct(';', 0, ";", "SELECT ", true, true, NULL, NULL);
						} 
						else
						{
							if((tok = yylex()) != K_SELECT)
							{
								plpgpsm_push_back_token(tok);
								item->expr = read_sql_construct(';', 0, ";", "SELECT (", true, true, NULL, NULL);
							}
							else
							{
								/* 
								 * transformation to derivated table,
								 * without this transformation I can't detect not found event 
								 */
								item->expr = read_sql_construct(';', 0, ";", "SELECT (__sqlpsm.*,1) FROM (SELECT ", true, true, NULL, " __sqlpsm");
							}							
						}

						new->items = list_make1(item);
						$$ = (PLpgPSM_stmt *)new;
					}
				| K_SET lno varname '=' expr_until_semi
					{
						PLpgPSM_stmt_assign *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_assign));
						new->cmd_type = PLPGPSM_STMT_ASSIGN;
						new->lineno	= $2;
						new->varname	= $3;
						new->expr 	= $5;
						
						DirectFunctionCall1(show_config_by_name,
							    DirectFunctionCall1(textin, CStringGetDatum($3)));
						
						$$ = (PLpgPSM_stmt *)new;
					}
				    ;
				
assign_list			: assign_list ',' assign_item
					{
						$$ = lappend($1, $3);
					}
				| assign_item
					{
						$$ = list_make1($1);
					}
				    ;
				    
assign_item			: assign_var '=' lno 
					{
						int tok;
						PLpgPSM_assign_item *new;
						
						new = (PLpgPSM_assign_item *) palloc0(sizeof(PLpgPSM_assign_item));
						
						new->lineno = $3;
						new->varno = $1;

						if ((tok = yylex()) != '(')
						{
							plpgpsm_push_back_token(tok);
							new->expr = read_sql_construct(';', ',', "[,;]", "SELECT ", true, true, &tok, NULL);
						} 
						else
						{
							if((tok = yylex()) != K_SELECT)
							{
								plpgpsm_push_back_token(tok);
								new->expr = read_sql_construct(';', ',', "[,;]", "SELECT (", true, true, &tok, NULL);
							}
							else
							{ 
								/* 
								 * transformation to derivated table,
								 * without this transformation I can't detect not found event 
								 */
								new->expr = read_sql_construct(';', ',', "[,;]", "SELECT * FROM (SELECT ", true, true, &tok, " __sqlpsm");
							} 
							
						} 
						
						plpgpsm_push_back_token(tok);

						$$ = new;
					}
				    ;
				    
				
assign_scalar_list		: T_SCALAR
					{
						char *name = pstrdup(yytext);
						PLpgPSM_datum *var = yylval.scalar;
						
						PLpgPSM_row *row = read_into_scalar_list(name, var);
						$$ = row->dno;
					}
				    ;


				
assign_var			: T_SCALAR
					{
						check_assignable(yylval.scalar);
						$$ = yylval.scalar->dno;
					}
				| T_ROW
					{
						check_assignable((PLpgPSM_datum *) yylval.row);
						$$ = yylval.row->dno;
					}
				| T_RECORD
					{
						check_assignable((PLpgPSM_datum *) yylval.rec);
						$$ = yylval.rec->dno;
					}
				| assign_var '[' expr_until_rightbracket
					{
						PLpgPSM_arrayelem	*new;

						new = palloc0(sizeof(PLpgPSM_arrayelem));
						new->dtype		= PLPGPSM_DTYPE_ARRAYELEM;
						new->subscript	= $3;
						new->arrayparentno = $1;

						plpgpsm_adddatum((PLpgPSM_datum *)new);

						$$ = new->dno;
					}
				| K_SQLSTATE 
					{
						if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
						{
							check_assignable(yylval.scalar);
							$$ = yylval.scalar->dno;
						}
						else
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("undefined variable 'SQLSTATE'"),
									 errhint("declare SQLSTATE char(5) variable before using")));
					}
				;


stmt_getdiag			: K_GET opt_diag_area K_DIAGNOSTICS opt_exception_deep lno getdiag_cond_list ';'
					{
						PLpgPSM_stmt_getdiag	*new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_getdiag));
						new->cmd_type = PLPGPSM_STMT_GETDIAG;
						new->lineno = $5;
						new->diag_items = $6;

						if ($2 == PLPGPSM_GETDIAG_CURRENT && $4.defined)
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("diagnostic level is allowed only for stacked diagnostic area")));

						new->stacked = ($4.defined || $2 == PLPGPSM_GETDIAG_STACKED);
						new->deep = $4.deep;
						new->deep_varno = $4.varno;

						$$ = (PLpgPSM_stmt *) new;					
					}
				    ;


opt_exception_deep:			
					{
						$$.deep = 1;
						$$.varno = PLPGPSM_INVALID_VARNO;
						$$.defined = false;
					}
				| K_EXCEPTION exception_deep
					{
						$$ = $2;
						$$.defined = true;
					}
				| K_CONDITION exception_deep
					{
						$$ = $2;
						$$.defined = true;
					}


exception_deep			: T_NUMBER
					{
						$$.deep = yylval.ival;
						$$.varno = PLPGPSM_INVALID_VARNO;
						
						if (yylval.ival < 1 || yylval.ival > 255)
							ereport(ERROR,
									(errcode(ERRCODE_NUMERIC_VALUE_OUT_OF_RANGE),
									 errmsg("exception's deep is out of range"),
									 errhint("check deep is between 1 and 255")));
					}
				| T_SCALAR
					{
						$$.deep = PLPGPSM_EXCEPTION_DEEP_INVALID;
						$$.varno = yylval.scalar->dno;					
					}
				    ;


opt_diag_area			:
					{
						int tok;
						
						if ((tok = yylex()) == K_DIAGNOSTICS)
						{
							plpgpsm_push_back_token(tok);
							$$ = PLPGPSM_GETDIAG_UNDEF;
						}
						else
						{
							char *name;
							plpgpsm_convert_ident(yytext, &name, 1);
							
							if (strcmp(name, "current") == 0)
								$$ = PLPGPSM_GETDIAG_CURRENT;
							else if (strcmp(name, "stacked") == 0)
								$$ = PLPGPSM_GETDIAG_STACKED;
							else
								yyerror("expected CURRENT or STACKED");
						}
					}


getdiag_cond_list		: getdiag_cond_list ',' getdiag_cond_item
					{
						$$ = lappend($1, $3);
					}
				| getdiag_cond_item
					{
						$$ = list_make1($1);
					}
				    ;
				

getdiag_cond_item		: getdiag_target '=' signal_info_identif_excpt
					{
						PLpgPSM_diag_item	*new;
						
						new = palloc0(sizeof(PLpgPSM_diag_item));
						new->target = $1;
						new->kind = $3.value;
						
						$$ = new;						
					}
				    ;
				    				    
getdiag_target			: T_SCALAR
					{
						check_assignable(yylval.scalar);
						$$ = yylval.scalar->dno;
					}
				    | T_ROW
					{
						yyerror("expected an integer variable");
					}
				    | T_RECORD
					{
						yyerror("expected an integer variable");
					}
				    | T_WORD
					{
						yyerror("expected an integer variable");
					}
				    ;


stmt_if				: K_IF lno expr_until_then proc_sect stmt_else K_END K_IF ';'
					{
						PLpgPSM_stmt_if *new;

						new = palloc0(sizeof(PLpgPSM_stmt_if));
						new->cmd_type	= PLPGPSM_STMT_IF;
						new->lineno		= $2;
						new->cond		= $3;
						new->true_body	= $4;
						new->false_body = $5;

						$$ = (PLpgPSM_stmt *)new;
					}
				;

proc_sect			:	
					{
						$$ = NIL;
					}
				| sql_stmt_list
					{
						$$ = $1;
					}
    

stmt_else			:
					{
						$$ = NIL;
					}
				| K_ELSEIF lno expr_until_then proc_sect stmt_else
					{
						/*
						 * Translate the structure:	   into:
						 *
						 * IF c1 THEN				   IF c1 THEN
						 *	 ...						   ...
						 * ELSIF c2 THEN			   ELSE
						 *								   IF c2 THEN
						 *	 ...							   ...
						 * ELSE							   ELSE
						 *	 ...							   ...
						 * END IF						   END IF
						 *							   END IF
						 */
						PLpgPSM_stmt_if *new_if;

						/* first create a new if-statement */
						new_if = palloc0(sizeof(PLpgPSM_stmt_if));
						new_if->cmd_type	= PLPGPSM_STMT_IF;
						new_if->lineno		= $2;
						new_if->cond		= $3;
						new_if->true_body	= $4;
						new_if->false_body	= $5;

						/* wrap the if-statement in a "container" list */
						$$ = list_make1(new_if);
					}

				| K_ELSE proc_sect
					{
						$$ = $2;
					}
				;


stmt_case			: K_CASE lno opt_expr_until_when_r when_clause_list opt_case_default K_END K_CASE ';'
					{
						$$ = make_case($2, $3, $4, $5);
					}
				    ;

opt_case_default			:  
					{
						$$ = NIL;
					}
				| K_ELSE sql_stmt_list
					{
						$$ = $2;
					}
				    ;


when_clause_list		: when_clause_list when_clause
					{
						$$ = lappend($1, $2);
					}
				| when_clause
					{	
						$$ = list_make1($1);
					}
				    ;


when_clause			: K_WHEN lno expr_until_then sql_stmt_list 
					{
						PLpgPSM_case_when *new = palloc0(sizeof(PLpgPSM_case_when));

						new->lineno	= $2;
						new->expr	= $3;
						new->stmts 	= $4;
						$$ = new;
					}
				    ;
				    

stmt_leave			: leave_type lno lblname ';' 
					{ 
						PLpgPSM_stmt_leave *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_leave));
						new->cmd_type = PLPGPSM_STMT_LEAVE;
						new->is_leave = $1;
						new->lineno 		= $2;
						new->label		= $3;
						
						$$ = (PLpgPSM_stmt *) new;
					}
				;	


leave_type			: K_ITERATE
					{
						$$ = false;
					}
				| K_LEAVE
					{
						$$ = true;
					}
				;
				
stmt_loop			: opt_block_label K_LOOP lno sql_stmt_list K_END K_LOOP opt_label_semi
					{
						PLpgPSM_stmt_loop *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_loop));
						new->cmd_type = PLPGPSM_STMT_LOOP;
						new->ctype    = PLPGPSM_CYCLE_LOOP;
						new->lineno	= $3;
						new->label 	= $1;
						new->body	= $4;
						
						check_labels($1, $7);
						plpgpsm_ns_pop();
						
						$$ = (PLpgPSM_stmt *)new;
					}		    
				;
				
stmt_while			: opt_block_label K_WHILE lno expr_until_do sql_stmt_list K_END K_WHILE opt_label_semi
					{
						PLpgPSM_stmt_loop *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_loop));
						new->cmd_type = PLPGPSM_STMT_LOOP;
						new->ctype    = PLPGPSM_CYCLE_WHILE;
						new->lineno	= $3;
						new->label 	= $1;
						new->cond	= $4;
						new->body	= $5;
						
						check_labels($1, $8);
						plpgpsm_ns_pop();
						
						$$ = (PLpgPSM_stmt *)new;
					}
				    ;
				
stmt_repeat			: opt_block_label K_REPEAT lno sql_stmt_list K_UNTIL expr_until_end K_REPEAT opt_label_semi
					{
						PLpgPSM_stmt_loop *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_loop));
						new->cmd_type = PLPGPSM_STMT_LOOP;
						new->ctype    = PLPGPSM_CYCLE_REPEAT;
						new->lineno	= $3;
						new->label 	= $1;
						new->cond	= $6;
						new->body	= $4;
						
						check_labels($1, $8);
						plpgpsm_ns_pop();
						
						$$ = (PLpgPSM_stmt *)new;
					}
				    ;
				    
stmt_for			: for_header for_body
					{
						
						PLpgPSM_stmt_fors	*new;
						
						new = (PLpgPSM_stmt_fors *) $1;
						
						check_labels(new->label, $2.end_label);
						
						new->body = $2.stmts;	
		
						/* close namespace started for_header */
						plpgpsm_ns_pop();
						/* close namespace started in opt_label */ 	
						plpgpsm_ns_pop();
						
						$$ = (PLpgPSM_stmt *) new;
					}	
				    ;			    
				    
				    
for_header			: opt_block_label K_FOR lno 
					{
						int tok;
						PLpgPSM_stmt_fors *new;
						PLpgPSM_expr 	  *expr;
						char *cursor_name = NULL;
						char *for_loop_varname = NULL;
						char *identif = NULL;
						
						/* 
						 * set two possible identifiers for_loop_varname
						 * and cursor_name. Isn't possible using gramatics
						 * because keywords K_AS and K_CURSOR K_FOR are
						 * used over identifiers
						 */
						if ((tok = yylex()) == T_WORD)
						{
							plpgpsm_convert_ident(yytext, &identif, 1);
							if ((tok = yylex()) == K_AS)
							{
								for_loop_varname = identif;
								identif = NULL;
								/* next possible is cursor name */
								
								if ((tok = yylex()) == T_WORD)
								{
									plpgpsm_convert_ident(yytext, &identif, 1);
									tok = yylex();
								}
							}
						}
						
						if (identif)
						{
							if (tok == K_CURSOR)
							{
								cursor_name = identif;
								if (yylex() != K_FOR)
									yyerror("syntax error, expected FOR"); 
								
								tok = yylex();
							}
							else
								yyerror("syntax error, expected CURSOR FOR"); 
						}
						
						expr = read_sql_construct(K_DO, 0, "DO", yytext,
									    true, true, NULL, NULL);
									    
						new = register_from_expr($1, $3, for_loop_varname, expr);
						new->cursor_name = cursor_name;
						
						$$ = new;
					}
				    ;


for_body			: sql_stmt_list K_END K_FOR opt_label_semi
					{
						$$.stmts = $1;
						$$.end_label = $4;
					}
				    ;

				    
stmt_execute_imm		: K_EXECUTE K_IMMEDIATE lno
					{
						PLpgPSM_stmt_execute *new;
						PLpgPSM_expr *expr;
						int endtoken;

						expr = read_sql_construct(K_INTO, ';', "INTO|;",
												  "SELECT ",
												  true, true, &endtoken, NULL);

						new = palloc0(sizeof(PLpgPSM_stmt_execute));
						new->cmd_type = PLPGPSM_STMT_EXECUTE;
						new->lineno = $3;
						new->query = expr;
						new->into = false;
						new->strict = false;
						new->rec = NULL;
						new->row = NULL;
						
						new->dtype = PLPGPSM_DSQLTYPE_EXEC_IMMEDIATE;

						if (endtoken == K_INTO)
						{
							new->into = true;
							read_into_target(&new->rec, &new->row, &new->strict);
							if (yylex() != ';')
								yyerror("syntax error");
						}

						$$ = (PLpgPSM_stmt *)new;
					}
				;
				
stmt_execute			: K_EXECUTE lno sqlname 
					{
						int tok;
						int nparams = 0;
						int params[MAX_EXPR_PARAMS];
												
						PLpgPSM_stmt_execute *new;
						PLpgPSM_expr	*expr;
						
						new = palloc0(sizeof(PLpgPSM_stmt_execute));
						new->cmd_type = PLPGPSM_STMT_EXECUTE;
						new->lineno = $2;
						new->name = $3;
						new->strict = false;
						new->dtype = PLPGPSM_DSQLTYPE_EXEC_PREPARE;
						
						
						if ((tok = yylex()) != ';')
						{
							if (tok == K_INTO)
							{
								new->into = true;
								read_into_target(&new->rec, &new->row, &new->strict);
								tok = yylex();
							}
							if (tok == K_USING)
							{
								for (;;)
								{
									tok = yylex();
									
									switch (tok)
									{
										case T_SCALAR:
											assign_expr_param(yylval.scalar->dno,
													    params, &nparams);
											break;
											
										case T_ROW:
											assign_expr_param(yylval.row->dno,
													    params, &nparams);
											break;
													    
										case T_RECORD:
											assign_expr_param(yylval.rec->dno,
													    params, &nparams);
											break;
											
										case K_SQLSTATE:
											if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
											{
												assign_expr_param(yylval.scalar->dno,
														    params, &nparams);
	    											break;
											} 
											
											
										default:
											plpgpsm_error_lineno = plpgpsm_scanner_lineno();
											ereport(ERROR,
												    (errcode(ERRCODE_SYNTAX_ERROR),
												     errmsg("syntax error at \"%s\"", yytext),
												     errdetail("Expected scalar variable, record variable, row variable, "
												    	       "or list of variables following USING.")));
									}
									tok = yylex();
									
									if (tok == ';')
										break;
									else if (tok != ',')
										yyerror("syntax error");									
								}
								
								expr = palloc0(sizeof(PLpgPSM_expr) + sizeof(int) * nparams - sizeof(int));
								expr->dtype 	= PLPGPSM_DTYPE_EXPR;
								expr->nparams	= nparams;
								while(nparams-- > 0)
									expr->params[nparams] = params[nparams]; 
									
								new->query = expr;
							}
						}
						
						if (tok != ';')
							yyerror("syntax error");
					
						$$ = (PLpgPSM_stmt *) new;
					}
				    ;

				
stmt_prepare			: K_PREPARE lno sqlname opt_datatype_list K_FROM expr_until_semi
					{
						PLpgPSM_stmt_prepare *new;

						int nparams = 0;
						int params[MAX_EXPR_PARAMS];
						ListCell *c;
						
						foreach(c, $4)
						{
							Oid oid = (Oid) lfirst(c);
							params[nparams++] = oid;
						}
						
						new = palloc0(sizeof(PLpgPSM_stmt_prepare) + sizeof(int)*nparams);
						new->cmd_type = PLPGPSM_STMT_PREPARE;
						new->lineno 	= $2;
						new->name 	= $3;
						new->expr	= $6;

						new->nparams	= nparams;
						memcpy(new->params, params, sizeof(int)*nparams);
						
						$$ = (PLpgPSM_stmt *) new;
					}
				    ;
				    
opt_datatype_list		:
					{
						$$ = NIL;
					}
				| '(' datatype_list ')'
					{
						$$ = $2;
					}
				    ;
				    
datatype_list			: datatype_list ',' decl_datatype
					{
						$$ = lappend($1, (void*)$3->typoid);
					}
				| decl_datatype
					{
						$$ = list_make1((void*)$1->typoid);
					}
				    ;
				    
				
stmt_signal			: K_SIGNAL lno signal_value set_signal_info ';'
					{
						$4->lineno = $2;
						$4->sqlerrstate = $3->sqlerrstate;
						$4->condname	= $3->condname;
						$4->elog_level = sqlerrstate_to_elevel($3->sqlerrstate);
						$4->is_resignal = false;
						
						$$ = (PLpgPSM_stmt *) $4;
					}
    				| K_SIGNAL lno signal_value ';'
					{
						PLpgPSM_stmt_signal *new = palloc0(sizeof(PLpgPSM_stmt_signal));
					    
						new->cmd_type = PLPGPSM_STMT_SIGNAL;
						new->lineno = $2;
						new->sqlerrstate 	= $3->sqlerrstate;
						new->elog_level = sqlerrstate_to_elevel($3->sqlerrstate);
						new->condname	= $3->condname;
						new->is_resignal = false;
					    
						$$ = (PLpgPSM_stmt *) new;
					}
				    ;

stmt_resignal			: K_RESIGNAL lno opt_signal_value set_signal_info ';'
					{ 
						$4->lineno = $2;
						if ($3)
						{
							$4->sqlerrstate = $3->sqlerrstate;
							$4->condname	= $3->condname;
							$4->elog_level = sqlerrstate_to_elevel($3->sqlerrstate);
						}
						$4->is_resignal = true;
						
						$$ = (PLpgPSM_stmt *) $4;
					}
				| K_RESIGNAL lno opt_signal_value ';'
					{
						PLpgPSM_stmt_signal *new = palloc0(sizeof(PLpgPSM_stmt_signal));
					    
						new->cmd_type = PLPGPSM_STMT_SIGNAL;
					
						new->lineno = $2;
						if ($3)
						{
							new->sqlerrstate 	= $3->sqlerrstate;
							new->elog_level = sqlerrstate_to_elevel($3->sqlerrstate);
							new->condname	= $3->condname;
						}
						new->is_resignal = true;
					    
						$$ = (PLpgPSM_stmt *) new;
					}
				;

signal_value			: sqlstate_value
					{
						int i;
						
						PLpgPSM_condition *c;
						
						c = palloc0(sizeof(PLpgPSM_condition));
						
						c->sqlerrstate = $1;
						
						for (i = 0; exception_label_map[i].label != NULL; i++)
						{
							if (exception_label_map[i].sqlerrstate == $1)
							{
								c->condname = pstrdup(exception_label_map[i].label);
							}
						}
				
						/* SQLSTATE to condname */
						if (!c->condname)
						{
							char unpacked_sqlstate[17];
							
							sprintf(unpacked_sqlstate, "SQLSTATE '%s'",
												    unpack_sql_state($1));
							c->condname = pstrdup(unpacked_sqlstate);
						}
						$$ = c;		
					}
				| T_CONDITION
					{
						char *name;
						PLpgPSM_condition *c;

						c = palloc0(sizeof(PLpgPSM_condition));
						
						plpgpsm_convert_ident(yytext, &name, 1);
						
						c->sqlerrstate = yylval.condvar->sqlerrstate;
						c->condname = name;
						
				    		$$ = c;
				       }
				    ;
				
set_signal_info			: K_SET signal_info_item_list
					{ 
							
						$$ = $2;
					}
    				    ;
				
signal_info_item_list		: signal_info_item_list ',' signal_info_item
					{
						set_signal_stmt($1, &$3);
						$$ = $1;
					}
				| signal_info_item
					{
						PLpgPSM_stmt_signal *new = palloc0(sizeof(PLpgPSM_stmt_signal));
						new->cmd_type = PLPGPSM_STMT_SIGNAL;

						set_signal_stmt(new, &$1);
						
						$$ = new;
					}
				;
				
signal_info_item		: signal_info_item_name '=' 
				    {
				    	    int tok;
				    	    char *name;
					    plpgpsm_convert_ident($1, &name, 1);
					
					    $$.ival = PLPGPSM_ERRLEVEL_UNKNOWN;
					    $$.expr = NULL;
					
					    if (strcmp(name, "message_text") == 0)
					    {
						    $$.item_type = PLPGPSM_SINFO_MESSAGE;
						    $$.expr = read_sql_construct(',', ';', "[,;]", 
									"SELECT", false, true, &tok, NULL);
						
						    plpgpsm_push_back_token(tok);
					    }
					    else if (strcmp(name, "pg_message_hint") == 0)
					    {
						    $$.item_type = PLPGPSM_SINFO_HINT;
						    $$.expr = read_sql_construct(',', ';', "[,;]", 
									"SELECT", false, true, &tok, NULL);
						    plpgpsm_push_back_token(tok);
					    }
					    else if (strcmp(name, "pg_message_detail") == 0)
					    {
						    $$.item_type = PLPGPSM_SINFO_DETAIL;
						    $$.expr = read_sql_construct(',', ';', "[,;]", 
									"SELECT", false, true, &tok, NULL);
						    plpgpsm_push_back_token(tok);
					    }

					    else
						    yyerror("unknown identifier [message_text, level, hint, detail]");	
				    }
			    ;

signal_info_identif_excpt	:
					{
						char *name;
						
						yylex();
						
						plpgpsm_convert_ident(yytext, &name, 1);

						if (strcmp(name, "row_count") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_ROW_COUNT;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "result_oid") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_RESULT_OID;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "returned_sqlstate") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_RETURNED_SQLSTATE;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "message_text") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_MESSAGE;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "message_length") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_MESSAGE_LENGTH;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "pg_message_hint") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_HINT;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "pg_message_detail") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_DETAIL;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RW;
						}
						else if (strcmp(name, "class_origin") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_CLASS_ORIGIN;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RO;
						}
						else if (strcmp(name, "subclass_origin") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_SUBCLASS_ORIGIN;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RO;
						}
						else if (strcmp(name, "constraint_name") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_CONSTRAINT_NAME;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RO;
						}
						else if (strcmp(name, "constraint_schema") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_CONSTRAINT_SCHEMA;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RO;
						}
						else if (strcmp(name, "condition_identifier") == 0)
						{
							$$.value = PLPGPSM_GETDIAG_CONDITION;
							$$.kind  = PLPGPSM_GETDIAG_EXCEPTION_RO;
						}
						else
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("unknown identifier %s", yytext)));
					}
				    ;
				
signal_info_item_name		: T_WORD
					{ $$ = pstrdup(yytext);	}
				| T_ERROR
					{ $$ = pstrdup(yytext); }
				| T_SCALAR
					{ $$ = pstrdup(yytext);	}
				| T_ROW
					{ $$ = pstrdup(yytext);	}
				| T_RECORD
					{ $$ = pstrdup(yytext);	}
				    ;
				    
		
sqlname				:  T_WORD
					{ 	
						plpgpsm_convert_ident(yytext, &($$), 1);
					}
				| T_ERROR
					{
						plpgpsm_convert_ident(yytext, &($$), 1); 
					}
				| T_SCALAR
					{
						plpgpsm_convert_ident(yytext, &($$), 1);	
					}
				| T_ROW
					{ 
						plpgpsm_convert_ident(yytext, &($$), 1);
					}
				| T_RECORD
					{ 
						plpgpsm_convert_ident(yytext, &($$), 1);
					}
				    ;		    
			
varname				:  T_WORD
					{ 	
						plpgpsm_convert_ident(yytext, &($$), 1);
					}
				| T_ERROR
					{
						plpgpsm_convert_ident(yytext, &($$), 1); 
					}
				    ;		    
				    				
opt_signal_value	:
				{ $$ =  NULL; }
			| signal_value
				{ $$ = $1; }
				;

stmt_sql		: sql_start lno
					{
						$$ = make_execsql_stmt($1, $2);
					}
				    ;

/*
 * opt label do problems, eat first T_WORD. so isn't possible universal T_WORD.
 * Every first word of possible SQL have to be declared as keyword 
 *
 */
				    
sql_start			: K_INSERT K_INTO
					{ $$ = "INSERT INTO"; 	}
				| K_SELECT lno
					{ $$ = "SELECT";	}
				| K_UPDATE
					{ $$ = "UPDATE";	}
				| K_CREATE
					{ $$ = "CREATE";	}
				| K_DROP
					{ $$ = "DROP";		}
				| K_DELETE
					{ $$ = "DELETE";	}
				| K_TRUNCATE
					{ $$ = "TRUNCATE";	}
				| K_VALUES
					{ $$ = "VALUES";	}
				| T_ERROR
					{
						$$ = pstrdup(yytext);
					}
				    ;
				    	    
stmt_call			: K_CALL lno expr_until_semi
					{
						PLpgPSM_stmt_call *new;
						
						new = palloc0(sizeof(PLpgPSM_stmt_call));
						new->cmd_type = PLPGPSM_STMT_CALL;
						new->lineno	= $2;
						new->expr	= $3;
						
						$$ = (PLpgPSM_stmt *)new;
					}
				    ;
				
expr_until_semi :
					{ $$ = plpgpsm_read_expression(';', ";"); }
				;

expr_until_then :    
                                        { $$ = plpgpsm_read_expression(K_THEN, "THEN"); }    
                                ;                                                            

expr_until_do :    
                                        { $$ = plpgpsm_read_expression(K_DO, "DO"); }    
                                ;                                                            

expr_until_end :    
                                        { $$ = plpgpsm_read_expression(K_END, "END"); }    
                                ;                                       

stmt_open		: K_OPEN lno cursor_variable
					{
						PLpgPSM_stmt_open *new;
						int				  tok;

						new = palloc0(sizeof(PLpgPSM_stmt_open));
						new->cmd_type = PLPGPSM_STMT_OPEN;
						new->lineno = $2;
						new->curvar = $3.dno;

						tok = yylex();
						if (tok == K_USING)
						{
							int nparams = 0;
							int params[MAX_EXPR_PARAMS];
							PLpgPSM_expr *expr;

							if (!$3.prepname)
								ereport(ERROR,
									    (errcode(ERRCODE_SYNTAX_ERROR),
									     errmsg("USING isn't allowed for static cursors"),
									     errdetail("Clause USING is allowed only for dynamic cursors defined by PREPARE statement.")));
							for (;;)
							{
								tok = yylex();
									
								switch (tok)
								{
									case T_SCALAR:
										assign_expr_param(yylval.scalar->dno,
												    params, &nparams);
										break;
											
									case T_ROW:
										assign_expr_param(yylval.row->dno,
												    params, &nparams);
										break;
													    
									case T_RECORD:
										assign_expr_param(yylval.rec->dno,
												    params, &nparams);
										break;
											
									case K_SQLSTATE:
										if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
										{
											assign_expr_param(yylval.scalar->dno,
													    params, &nparams);
    											break;
										} 
											
											
									default:
										plpgpsm_error_lineno = plpgpsm_scanner_lineno();
										ereport(ERROR,
											    (errcode(ERRCODE_SYNTAX_ERROR),
											     errmsg("syntax error at \"%s\"", yytext),
											     errdetail("Expected scalar variable, record variable, row variable, "
											    	       "or list of variables following USING.")));
								}
								tok = yylex();
									
								if (tok == ';')
									break;
								else if (tok != ',')
									yyerror("syntax error");									
							}
								
							expr = palloc0(sizeof(PLpgPSM_expr) + sizeof(int) * nparams - sizeof(int));
							expr->dtype 	= PLPGPSM_DTYPE_EXPR;
							expr->nparams	= nparams;
							while(nparams-- > 0)
								expr->params[nparams] = params[nparams]; 
									
							new->args = expr;
						}

						if (tok != ';')
						{
							plpgpsm_error_lineno = plpgpsm_scanner_lineno();
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("syntax error at \"%s\"",
											yytext)));
						}

						$$ = (PLpgPSM_stmt *)new;
					}
				;

stmt_fetch			: K_FETCH lno opt_fetch_direction cursor_variable K_INTO
					{
						PLpgPSM_stmt_fetch *fetch = $3;
						PLpgPSM_rec	   *rec;
						PLpgPSM_row	   *row;

						/* We have already parsed everything through the INTO keyword */
						read_into_target(&rec, &row, NULL);

						/* early check of scrollable cursor */
						if (!$3->expr)
							if (($3->direction != FETCH_FORWARD || $3->how_many < 0) && !$4.scrollable)
								ereport(ERROR,
										(errcode(ERRCODE_INVALID_CURSOR_STATE),
										 errmsg("cursor can scan only forward"),
										 errhint("DECLARE cursor with SCROLL option to enable scrollable scan.")));										

						if (yylex() != ';')
							yyerror("syntax error");

						fetch->lineno = $2;
						fetch->rec		= rec;
						fetch->row		= row;
						fetch->curvar	= $4.dno;
						fetch->is_move	= false;

						$$ = (PLpgPSM_stmt *)fetch;
					}										
				    ;

stmt_move		: K_MOVE lno opt_fetch_direction cursor_variable ';'
					{
						PLpgPSM_stmt_fetch *fetch = $3;

						fetch->lineno = $2;
						fetch->curvar	= $4.dno;
						fetch->is_move	= true;

						/* early check of scrollable cursor */
						if (!$3->expr)
							if (($3->direction != FETCH_FORWARD || $3->how_many < 0) && !$4.scrollable)
								ereport(ERROR,
										(errcode(ERRCODE_INVALID_CURSOR_STATE),
										 errmsg("cursor can scan only forward"),
										 errhint("DECLARE cursor with SCROLL option to enable scrollable scan.")));

						$$ = (PLpgPSM_stmt *)fetch;
					}
				;

opt_fetch_direction		: 
					{
						$$ = read_fetch_direction();
					}
				    ;

stmt_close			: K_CLOSE lno cursor_variable ';'
					{
						PLpgPSM_stmt_close *new;

						new = palloc0(sizeof(PLpgPSM_stmt_close));
						new->cmd_type = PLPGPSM_STMT_CLOSE;
						new->lineno = $2;
						new->curvar = $3.dno;

						$$ = (PLpgPSM_stmt *)new;
					}
				    ;
    
cursor_variable			: T_SCALAR
					{
						if (yylval.scalar->dtype != PLPGPSM_DTYPE_VAR)
							yyerror("cursor variable must be a simple variable");

						if (((PLpgPSM_var *) yylval.scalar)->datatype->typoid != REFCURSOROID)
						{
							plpgpsm_error_lineno = plpgpsm_scanner_lineno();
							ereport(ERROR,
									(errcode(ERRCODE_DATATYPE_MISMATCH),
									 errmsg("\"%s\" must be of type refcursor",
											((PLpgPSM_var *) yylval.scalar)->refname)));
						}
						$$.dno = yylval.scalar->dno;
						$$.scrollable = (((PLpgPSM_var *) yylval.scalar)->cursor_options & CURSOR_OPT_SCROLL) != 0;
						$$.prepname = ((PLpgPSM_var *) yylval.scalar)->prepname;
					}
				    | T_ROW
					{
						yyerror("expected a cursor or refcursor variable");
					}
				    | T_RECORD
					{
						yyerror("expected a cursor or refcursor variable");
					}
				    | T_WORD
					{
						yyerror("expected a cursor or refcursor variable");
					}
				;

opt_expr_until_when_r :    
                                        {
                                    		PLpgPSM_expr *expr = NULL;
                                    		int tok = yylex();
                                    		
                                    		if (tok != K_WHEN)
                                    		{
                                    			plpgpsm_push_back_token(tok);
                                    			expr = plpgpsm_read_expression(K_WHEN, "WHEN");
                                    		}

                                    		plpgpsm_push_back_token(K_WHEN);
                                    		$$ = expr;
                                    	}    
                                ;                                       
                                
expr_until_rightbracket		: 
					{
						$$ = plpgpsm_read_expression(']', "]");
					}
					
				;
                                
opt_block_label			:	
					{
						plpgpsm_ns_push(NULL);
						$$ = NULL;
					}
				|  block_lblname ':'
					{
						plpgpsm_ns_push($1);
						$$ = $1;
					} 
				;

			    
opt_label_semi			:  ';'
					{ 
						$$ = NULL; 
					}
				| lblname ';'
					{
						$$ = $1;
					}
				;
				
opt_semi			: 
					{
						$$ = NULL; 
					}
				| ';'
					{ 
						$$ = NULL;
					}

block_lblname			: T_WORD
					{
						char	*name;

						plpgpsm_convert_ident(yytext, &name, 1);
						$$ = name;
					}
				;


lblname				: T_WORD
					{
						$$ = check_label(yytext);
					}
				| T_SCALAR
					{
						$$ = check_label(yytext);
					}
				| T_RECORD
					{
						$$ = check_label(yytext);
					}
				| T_ROW
					{
						$$ = check_label(yytext);
					}
				;



lno				:
					{
						$$ = plpgpsm_error_lineno = plpgpsm_scanner_lineno();
					}
				;

%%

static int 
sqlerrstate_to_elevel(int errstate)
{
	int elog_level;

	if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_SUCCESSFUL_COMPLETION)
		elog_level = INFO;
	else if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_WARNING)
		elog_level = WARNING;
	else if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_NO_DATA)
		elog_level = WARNING;
	else 
		elog_level = ERROR;
		
	return elog_level;
}

static void 
check_uniq_condition(PLpgPSM_condition *c1, PLpgPSM_condition *c2, const char *hint)
{
	if (strcmp(c1->condname, c2->condname) == 0)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("duplicit condition name \"%s\" in condition list", c1->condname),
				 errhint(hint)));
	
	
	if ((c1->sqlerrstate == c2->sqlerrstate) && (c1->sqlerrstate != MAKE_SQLSTATE('P','1','0','0','1')))
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("duplicit sqlstate \"%s\" in condition list", unpack_sql_state(c1->sqlerrstate)),
				 errhint(hint)));
}


/* Slow procedure, but generally procedure has not lot of handlers */
static void 
check_uniq_handlers(List *exceptions, PLpgPSM_condition *cond)
{
	while (cond)
	{
		ListCell *e;
		
		foreach(e, exceptions)
		{
			PLpgPSM_exception *excpt = (PLpgPSM_exception *) lfirst(e);
			PLpgPSM_condition *c = excpt->conditions;
			
			while (c)
			{
				check_uniq_condition(c, cond, "Condition is handled in any preview handler");
	    			c = c->next;
			}
		}
	
		cond = cond->next;
	}
}


static bool
is_error(int errstate)
{
	bool result;

	if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_SUCCESSFUL_COMPLETION)
		result = false;
	else if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_WARNING)
		result = false;
	else if (ERRCODE_TO_CATEGORY(errstate) == ERRCODE_NO_DATA)
		result = false;
	else 
		result = true;
		
	return result;
};



static void 
set_signal_stmt(PLpgPSM_stmt_signal *stmt, PLpgPSM_signal_info *info)
{
	switch (info->item_type)
	{
		case PLPGPSM_SINFO_MESSAGE:
			if (stmt->message)
				yyerror("field is used two times");
			stmt->message = info->expr;
			break;

		case PLPGPSM_SINFO_HINT:
			if (stmt->hint)
				yyerror("field is used two times");
			stmt->hint = info->expr;
			break;

		case PLPGPSM_SINFO_DETAIL:
			if (stmt->detail)
				yyerror("field is used two times");
			stmt->detail = info->expr;
			break;
			
		default:
			yyerror("unknown signal info type");
	}
}





/*
 * determine the expression parameter position to use for a plpgpsm datum
 *
 * It is important that any given plpgpsm datum map to just one parameter.
 * We used to be sloppy and assign a separate parameter for each occurrence
 * of a datum reference, but that fails for situations such as "select DATUM
 * from ... group by DATUM".
 *
 * The params[] array must be of size MAX_EXPR_PARAMS.
 */
static int
assign_expr_param(int dno, int *params, int *nparams)
{
	int		i;

	/* already have an instance of this dno? */
	for (i = 0; i < *nparams; i++)
	{
		if (params[i] == dno)
			return i+1;
	}
	/* check for array overflow */
	if (*nparams >= MAX_EXPR_PARAMS)
	{
		plpgpsm_error_lineno = plpgpsm_scanner_lineno();
		ereport(ERROR,
				(errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
				 errmsg("too many variables specified in SQL statement")));
	}
	/* add new parameter dno to array */
	params[*nparams] = dno;
	(*nparams)++;
	return *nparams;
}


static int
assign_init_expr_param(int dno, PLpgPSM_init_params **params)
{
	int	i;
	PLpgPSM_init_params *par, *new;
	
	if (*params)
	    par = *params;
	else
	{
	    new = palloc0(sizeof(PLpgPSM_init_params) + 32*sizeof(int));
	    new->nparams = 0;
	    new->mparams = 25;
	    par = new;
	    *params = new; 
	}

	for (i = 0; i < par->nparams; i++)
	{
		if (par->params[i] == dno)
			return i + 1;
	}
	
	if (par->nparams == par->mparams)
	{
	    new = palloc0(sizeof(PLpgPSM_init_params) + 2*par->mparams*sizeof(int));
	    new->mparams = 2*par->mparams;
	    new->nparams = par->nparams;
	    while ((par->nparams)-- > 0)
		    new->params[par->nparams] = par->params[par->nparams];
	    pfree(par);
	}
	
	par->params[par->nparams] = dno;
	(par->nparams)++;
	
	return par->nparams;
}


static void
merge_init_expr_param(PLpgPSM_init_params **dest, PLpgPSM_init_params **src)
{
	
	PLpgPSM_init_params *new, *l, *i;
	
	int gs, j;

	if (*src == NULL)
	{
		return;
	}
		
	if (*dest == NULL)
	{
		int size = sizeof(PLpgPSM_init_params) + sizeof(int)*((*src)->mparams);
		new = palloc0(size);
		memcpy(new, (*src), size);
		*dest = new;
		return;
	}
	
	l = *dest; i = *src;
	
	gs = l->mparams;

	while (gs < (l->nparams + i->nparams))
		gs = gs * 2;
		
	if (gs != l->mparams)
	{
		new = palloc0(sizeof(PLpgPSM_init_params) + sizeof(int)*gs);
		new->mparams = gs;
		memcpy(new->params, l->params, sizeof(int)*l->nparams);
		new->nparams = l->nparams;
		*dest = new;
		pfree (l);
		l = new;
	}
	
	for (j = 0; j < i->nparams; j++)
		l->params[l->nparams++] = i->params[j];		
}


PLpgPSM_expr *
plpgpsm_read_expression(int until, const char *expected)
{
	return read_sql_construct(until, 0, expected, "SELECT ", true, true, NULL, NULL);
}


static PLpgPSM_expr *
read_sql_stmt(char *sqlstart)
{
	return read_sql_construct(';', 0, ";", sqlstart, false, true, NULL, NULL);
}


/*
 * Read a SQL construct and build a PLpgPSM_expr for it.
 *
 * until:		token code for expected terminator
 * until2:		token code for alternate terminator (pass 0 if none)
 * expected:	text to use in complaining that terminator was not found
 * sqlstart:	text to prefix to the accumulated SQL text
 * isexpression: whether to say we're reading an "expression" or a "statement"
 * valid_sql:   whether to check the syntax of the expr (prefixed with sqlstart)
 * endtoken:	if not NULL, ending token is stored at *endtoken
 *				(this is only interesting if until2 isn't zero)
 * sqlfinish	we can generate derivated tables
 */
static PLpgPSM_expr *
read_sql_construct(int until,
				   int until2,
				   const char *expected,
				   char *sqlstart,
				   bool isexpression,
				   bool valid_sql,
				   int *endtoken,
				   const char *sqlfinish)
{
	int					tok;
	int					lno;
	PLpgPSM_dstring		ds;
	int					parenlevel = 0;
	int					nparams = 0;
	int					params[MAX_EXPR_PARAMS];
	char				buf[32];
	PLpgPSM_expr		*expr;
	char *aux = sqlstart;

	lno = plpgpsm_scanner_lineno();
	plpgpsm_dstring_init(&ds);
	plpgpsm_dstring_append(&ds, sqlstart);
	
	
	if (aux)
		while (*aux)
			{
				if (*aux == '(' || *aux == '[' )
					parenlevel++;
				else if (*aux == ')' || *aux == ']' )
					if (--parenlevel < 0)
						yyerror("mismatched parentheses");
						
				aux++;
			}	

	for (;;)
	{
		tok = yylex();
		if (tok == until && parenlevel == 0)
			break;
		if (tok == until2 && parenlevel == 0)
			break;
		if (tok == '(' || tok == '[')
			parenlevel++;
		else if (tok == ')' || tok == ']')
		{
			parenlevel--;
			if (parenlevel < 0)
				yyerror("mismatched parentheses");
		}
		/*
		 * End of function definition is an error, and we don't expect to
		 * hit a semicolon either (unless it's the until symbol, in which
		 * case we should have fallen out above).
		 */
		if (tok == 0 || tok == ';')
		{
			plpgpsm_error_lineno = lno;
			if (parenlevel != 0)
				yyerror("mismatched parentheses");

			if (isexpression)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("missing \"%s\" at end of SQL expression",
								expected)));
			else
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("missing \"%s\" at end of SQL statement",
								expected)));
		}

		if (plpgpsm_SpaceScanned)
			plpgpsm_dstring_append(&ds, " ");

		switch (tok)
		{
			case T_SCALAR:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.scalar->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;
				
			case K_SQLSTATE:
				/* we have to find defined SQLSTATE variable */
				
				if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
				{
					snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.scalar->dno,
										   params, &nparams));
					plpgpsm_dstring_append(&ds, buf);
				}
				else
					plpgpsm_dstring_append(&ds, yytext);

				break;

			case T_ROW:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.row->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;

			case T_RECORD:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.rec->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;

			default:
				plpgpsm_dstring_append(&ds, yytext);
				break;
		}
	}

	if (endtoken)
		*endtoken = tok;

	if (sqlfinish)
		plpgpsm_dstring_append(&ds, sqlfinish);

	expr = palloc0(sizeof(PLpgPSM_expr) + sizeof(int) * nparams - sizeof(int));
	expr->dtype			= PLPGPSM_DTYPE_EXPR;
	expr->query			= pstrdup(plpgpsm_dstring_get(&ds));
	expr->plan			= NULL;
	expr->nparams		= nparams;
	while(nparams-- > 0)
		expr->params[nparams] = params[nparams];
	plpgpsm_dstring_free(&ds);

	if (valid_sql)
		check_sql_expr(expr->query);

	return expr;
}

/*
 * Read the argument of an INTO clause.  On entry, we have just read the
 * INTO keyword.
 */
static void
read_into_target(PLpgPSM_rec **rec, PLpgPSM_row **row, bool *strict)
{
	int			tok;

	/* Set default results */
	*rec = NULL;
	*row = NULL;
	if (strict)
		*strict = false;

	tok = yylex();
	if (strict && tok == K_STRICT)
	{
		*strict = true;
		tok = yylex();
	}

	switch (tok)
	{
		case T_ROW:
			*row = yylval.row;
			check_assignable((PLpgPSM_datum *) *row);
			break;

		case T_RECORD:
			*rec = yylval.rec;
			check_assignable((PLpgPSM_datum *) *rec);
			break;

		case T_SCALAR:
			*row = read_into_scalar_list(yytext, yylval.scalar);
			break;
			
		case K_SQLSTATE:
			*row = read_into_scalar_list("SQLSTATE", yylval.scalar);
			break;

		default:
			plpgpsm_error_lineno = plpgpsm_scanner_lineno();
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("syntax error at \"%s\"", yytext),
					 errdetail("Expected record variable, row variable, "
							   "or list of scalar variables following INTO.")));
	}
}


static PLpgPSM_type *
read_datatype(int tok)
{
	int					lno;
	PLpgPSM_dstring		ds;
	char				*type_name;
	PLpgPSM_type		*result;
	bool				needspace = false;
	int					parenlevel = 0;

	lno = plpgpsm_scanner_lineno();

	/* Often there will be a lookahead token, but if not, get one */
	if (tok == YYEMPTY)
		tok = yylex();

	if (tok == T_DTYPE)
	{
		/* lexer found word%TYPE and did its thing already */
		return yylval.dtype;
	}

	plpgpsm_dstring_init(&ds);

	while (tok != ';')
	{
		if (tok == 0)
		{
			if (parenlevel != 0)
				yyerror("mismatched parentheses");
			else
				yyerror("incomplete datatype declaration");
		}

		/* Possible followers for datatype in a declaration */
		if (tok == K_DEFAULT)
			break;

		/* Possible followers for datatype in a cursor_arg list */
		if ((tok == ',' || tok == ')') && parenlevel == 0)
			break;

		if (tok == '(')
			parenlevel++;
		else if (tok == ')')
			parenlevel--;

		if (needspace)
			plpgpsm_dstring_append(&ds, " ");

		needspace = true;
		plpgpsm_dstring_append(&ds, yytext);

		tok = yylex();
	}

	plpgpsm_push_back_token(tok);
	
	type_name = plpgpsm_dstring_get(&ds);

	if (type_name[0] == '\0')
		yyerror("missing datatype declaration");

	
	plpgpsm_error_lineno = lno;	/* in case of error in parse_datatype */

	result = plpgpsm_parse_datatype(type_name);

	plpgpsm_dstring_free(&ds);

	return result;
}


static PLpgPSM_stmt_fetch *
read_fetch_direction(void)
{
	PLpgPSM_stmt_fetch *fetch;
	int			tok;
	bool		check_FROM = true;

	/*
	 * We create the PLpgPSM_stmt_fetch struct here, but only fill in
	 * the fields arising from the optional direction clause
	 */
	fetch = (PLpgPSM_stmt_fetch *) palloc0(sizeof(PLpgPSM_stmt_fetch));
	fetch->cmd_type = PLPGPSM_STMT_FETCH;
	/* set direction defaults: */
	fetch->direction = FETCH_FORWARD;
	fetch->how_many  = 1;
	fetch->expr      = NULL;

	/*
	 * Most of the direction keywords are not plpgpsm keywords, so we
	 * rely on examining yytext ...
	 */
	tok = yylex();
	if (tok == 0)
		yyerror("unexpected end of function definition");

	if (pg_strcasecmp(yytext, "next") == 0)
	{
		/* use defaults */
	}
	else if (pg_strcasecmp(yytext, "prior") == 0)
	{
		fetch->direction = FETCH_BACKWARD;
	}
	else if (pg_strcasecmp(yytext, "first") == 0)
	{
		fetch->direction = FETCH_ABSOLUTE;
	}
	else if (pg_strcasecmp(yytext, "last") == 0)
	{
		fetch->direction = FETCH_ABSOLUTE;
		fetch->how_many  = -1;
	}
	else if (pg_strcasecmp(yytext, "absolute") == 0)
	{
		fetch->direction = FETCH_ABSOLUTE;
		fetch->expr = plpgpsm_read_expression(K_FROM, "FROM");
		check_FROM = false;
	}
	else if (pg_strcasecmp(yytext, "relative") == 0)
	{
		fetch->direction = FETCH_RELATIVE;
		fetch->expr = plpgpsm_read_expression(K_FROM, "FROM");
		check_FROM = false;
	}
	else if (tok == K_FROM)
	{
		check_FROM = false;
	}
	else
	{
		/* Assume there's no direction clause */
		plpgpsm_push_back_token(tok);
		check_FROM = false;
	}

	/* check FROM keyword after direction's specification */
	if (check_FROM && yylex() != K_FROM)
		yyerror("expected \"FROM\"");

	return fetch;
}


static PLpgPSM_stmt *
make_execsql_stmt(const char *sqlstart, int lineno)
{
	PLpgPSM_dstring		ds;
	int					nparams = 0;
	int					params[MAX_EXPR_PARAMS];
	char				buf[32];
	PLpgPSM_stmt_sql *execsql;
	PLpgPSM_expr		*expr;
	PLpgPSM_row			*row = NULL;
	PLpgPSM_rec			*rec = NULL;
	int					tok;
	bool				have_into = false;
	bool				have_strict = false;

	plpgpsm_dstring_init(&ds);
	plpgpsm_dstring_append(&ds, sqlstart);

	for (;;)
	{
		tok = yylex();
		if (tok == ';')
			break;
		if (tok == 0)
			yyerror("unexpected end of function definition");

		if (tok == K_INTO)
		{
			if (have_into)
				yyerror("INTO specified more than once");

			have_into = true;
			read_into_target(&rec, &row, &have_strict);
			continue;
		}

		if (plpgpsm_SpaceScanned)
			plpgpsm_dstring_append(&ds, " ");

		switch (tok)
		{
			case T_SCALAR:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.scalar->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;
				
			case K_SQLSTATE:
				if (plpgpsm_parse_word("sqlstate") == T_SCALAR)
				{
					snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.scalar->dno,
										   params, &nparams));
					plpgpsm_dstring_append(&ds, buf);
				}
				else
					plpgpsm_dstring_append(&ds, yytext);
				break;

			case T_ROW:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.row->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;

			case T_RECORD:
				snprintf(buf, sizeof(buf), " $%d ",
						 assign_expr_param(yylval.rec->dno,
										   params, &nparams));
				plpgpsm_dstring_append(&ds, buf);
				break;

			default:
				plpgpsm_dstring_append(&ds, yytext);
				break;
		}
	}

	expr = palloc0(sizeof(PLpgPSM_expr) + sizeof(int) * nparams - sizeof(int));
	expr->dtype			= PLPGPSM_DTYPE_EXPR;
	expr->query			= pstrdup(plpgpsm_dstring_get(&ds));
	expr->plan			= NULL;
	expr->nparams		= nparams;
	while(nparams-- > 0)
		expr->params[nparams] = params[nparams];
	plpgpsm_dstring_free(&ds);

	check_sql_expr(expr->query);

	execsql = palloc0(sizeof(PLpgPSM_stmt_sql));
	execsql->cmd_type = PLPGPSM_STMT_SQL;
	execsql->lineno  = lineno;
	execsql->sqlstmt = expr;
	execsql->into	 = have_into;
	execsql->strict	 = have_strict;
	execsql->rec	 = rec;
	execsql->row	 = row;

	return (PLpgPSM_stmt *) execsql;
}


static PLpgPSM_stmt *
make_return_stmt(int lineno, bool is_tblexpr)
{
	PLpgPSM_stmt_return *new;

	new = palloc0(sizeof(PLpgPSM_stmt_return));
	new->cmd_type = PLPGPSM_STMT_RETURN;
	new->lineno   = lineno;
	new->expr	  = NULL;
	new->retvarno = -1;
	new->is_tblexpr = is_tblexpr;

	if (is_tblexpr)
	{
		if (!plpgpsm_curr_compile->fn_retset)
			yyerror("table expression can be used only if table returns SETOF or TABLE");

		/* here can be test for cursor in future */

		if (yylex() != '(')
			yyerror("syntax error");

		new->expr = read_sql_construct(')', 0, ")", "", false, true, NULL, NULL);
		
		if (yylex() != ';')
			yyerror("syntax error");
	}
	else if (plpgpsm_curr_compile->fn_retset)
	{
		int tok;
		/* RETURN can contains any SQL statement which returning set */
		if ((tok = yylex()) != ';')
		{
			plpgpsm_push_back_token(tok);
			new->expr = read_sql_construct(';', 0, ";", "", false, true, NULL, NULL);
		}
	}
	else if (plpgpsm_curr_compile->out_param_varno >= 0)
	{
		if (yylex() != ';')
			yyerror("RETURN cannot have a parameter in function with OUT parameters");
		new->retvarno = plpgpsm_curr_compile->out_param_varno;
	}
	else if (plpgpsm_curr_compile->fn_rettype == VOIDOID)
	{
		if (yylex() != ';')
			yyerror("RETURN cannot have a parameter in function returning void");
	}
	else if (plpgpsm_curr_compile->fn_retistuple)
	{
		switch (yylex())
		{
			case K_NULL:
				/* we allow this to support RETURN NULL in triggers */
				break;

			case T_ROW:
				new->retvarno = yylval.row->dno;
				break;

			case T_RECORD:
				new->retvarno = yylval.rec->dno;
				break;

			default:
				yyerror("RETURN must specify a record or row variable in function returning tuple");
				break;
		}
		if (yylex() != ';')
			yyerror("RETURN must specify a record or row variable in function returning tuple");
	}
	else
	{
		/*
		 * Note that a well-formed expression is
		 * _required_ here; anything else is a
		 * compile-time error.
		 */
		new->expr = plpgpsm_read_expression(';', ";");
	}

	return (PLpgPSM_stmt *) new;
}


static void
check_assignable(PLpgPSM_datum *datum)
{
	switch (datum->dtype)
	{
		case PLPGPSM_DTYPE_VAR:
			if (((PLpgPSM_var *) datum)->isconst)
			{
				plpgpsm_error_lineno = plpgpsm_scanner_lineno();
				ereport(ERROR,
						(errcode(ERRCODE_ERROR_IN_ASSIGNMENT),
						 errmsg("\"%s\" is declared CONSTANT",
								((PLpgPSM_var *) datum)->refname)));
			}
			break;
		case PLPGPSM_DTYPE_ROW:
			/* always assignable? */
			break;
		case PLPGPSM_DTYPE_REC:
			/* always assignable?  What about NEW/OLD? */
			break;
		case PLPGPSM_DTYPE_RECFIELD:
			/* always assignable? */
			break;
		case PLPGPSM_DTYPE_ARRAYELEM:
			/* always assignable? */
			break;
		case PLPGPSM_DTYPE_TRIGARG:
			yyerror("cannot assign to tg_argv");
			break;
		default:
			elog(ERROR, "unrecognized dtype: %d", datum->dtype);
			break;
	}
}

/*
 * Given the first datum and name in the INTO list, continue to read
 * comma-separated scalar variables until we run out. Then construct
 * and return a fake "row" variable that represents the list of
 * scalars.
 */
static PLpgPSM_row *
read_into_scalar_list(const char *initial_name,
					  PLpgPSM_datum *initial_datum)
{
	int				 nfields;
	char			*fieldnames[1024];
	int				 varnos[1024];
	PLpgPSM_row		*row;
	int				 tok;

	check_assignable(initial_datum);
	fieldnames[0] = pstrdup(initial_name);
	varnos[0]	  = initial_datum->dno;
	nfields		  = 1;

	while ((tok = yylex()) == ',')
	{
		/* Check for array overflow */
		if (nfields >= 1024)
		{
			plpgpsm_error_lineno = plpgpsm_scanner_lineno();
			ereport(ERROR,
					(errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
					 errmsg("too many INTO variables specified")));
		}

		tok = yylex();
		switch(tok)
		{
			case T_SCALAR:
				check_assignable(yylval.scalar);
				fieldnames[nfields] = pstrdup(yytext);
				varnos[nfields++]	= yylval.scalar->dno;
				break;

			default:
				plpgpsm_error_lineno = plpgpsm_scanner_lineno();
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("\"%s\" is not a scalar variable",
								yytext)));
		}
	}

	/*
	 * We read an extra, non-comma token from yylex(), so push it
	 * back onto the input stream
	 */
	plpgpsm_push_back_token(tok);

	row = palloc0(sizeof(PLpgPSM_row));
	row->dtype = PLPGPSM_DTYPE_ROW;
	row->refname = pstrdup("*internal*");
	row->lineno = plpgpsm_scanner_lineno();
	row->rowtupdesc = NULL;
	row->nfields = nfields;
	row->fieldnames = palloc(sizeof(char *) * nfields);
	row->varnos = palloc(sizeof(int) * nfields);
	while (--nfields >= 0)
	{
		row->fieldnames[nfields] = fieldnames[nfields];
		row->varnos[nfields] = varnos[nfields];
	}

	plpgpsm_adddatum((PLpgPSM_datum *)row);

	return row;
}

/*
 * When the PL/PgSQL parser expects to see a SQL statement, it is very
 * liberal in what it accepts; for example, we often assume an
 * unrecognized keyword is the beginning of a SQL statement. This
 * avoids the need to duplicate parts of the SQL grammar in the
 * PL/PgSQL grammar, but it means we can accept wildly malformed
 * input. To try and catch some of the more obviously invalid input,
 * we run the strings we expect to be SQL statements through the main
 * SQL parser.
 *
 * We only invoke the raw parser (not the analyzer); this doesn't do
 * any database access and does not check any semantic rules, it just
 * checks for basic syntactic correctness. We do this here, rather
 * than after parsing has finished, because a malformed SQL statement
 * may cause the PL/PgSQL parser to become confused about statement
 * borders. So it is best to bail out as early as we can.
 */
static void
check_sql_expr(const char *stmt)
{
	ErrorContextCallback  syntax_errcontext;
	ErrorContextCallback *previous_errcontext;
	MemoryContext oldCxt;

	if (!plpgpsm_check_syntax)
		return;

	/*
	 * Setup error traceback support for ereport(). The previous
	 * ereport callback is installed by pl_comp.c, but we don't want
	 * that to be invoked (since it will try to transpose the syntax
	 * error to be relative to the CREATE FUNCTION), so temporarily
	 * remove it from the list of callbacks.
	 */
	Assert(error_context_stack->callback == plpgpsm_compile_error_callback);

	previous_errcontext = error_context_stack;
	syntax_errcontext.callback = plpgpsm_sql_error_callback;
	syntax_errcontext.arg = (char *) stmt;
	syntax_errcontext.previous = error_context_stack->previous;
	error_context_stack = &syntax_errcontext;

	oldCxt = MemoryContextSwitchTo(compile_tmp_cxt);
	(void) raw_parser(stmt);
	MemoryContextSwitchTo(oldCxt);

	/* Restore former ereport callback */
	error_context_stack = previous_errcontext;
}

static void
plpgpsm_sql_error_callback(void *arg)
{
	char *sql_stmt = (char *) arg;

	Assert(plpgpsm_error_funcname);

	errcontext("SQL statement in PL/PgSQL function \"%s\" near line %d",
			   plpgpsm_error_funcname, plpgpsm_error_lineno);
	internalerrquery(sql_stmt);
	internalerrposition(geterrposition());
	errposition(0);
}


static char *
check_label(const char *yytxt)
{
        char       *label_name;

        plpgpsm_convert_ident(yytxt, &label_name, 1);
        if (plpgpsm_ns_lookup_label(label_name) == NULL)
                yyerror("no such label");
        return label_name;
}


static void
check_labels(const char *start_label, const char *end_label)
{
	if (end_label)
	{
		if (!start_label)
		{
			plpgpsm_error_lineno = plpgpsm_scanner_lineno();
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("end label \"%s\" specified for unlabelled block",
							end_label)));
		}

		if (strcmp(start_label, end_label) != 0)
		{
			plpgpsm_error_lineno = plpgpsm_scanner_lineno();
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("end label \"%s\" differs from block's label \"%s\"",
							end_label, start_label)));
		}
	}
}

static PLpgPSM_stmt_fors *
register_from_expr(char *label, int lno, char *inner_label, PLpgPSM_expr *query)
{
	_SPI_plan 	*spi_plan;
	TupleDesc	tupDesc;
	int i;
	PLpgPSM_stmt_fors	*new;
	
	int 	nfields = 0;
	char	*fieldnames[1024];
	int	varnos[1024];
	PLpgPSM_row 	*row;
	Oid 	*argtypes;
	MemoryContext oldCxt;
	CachedPlanSource *plansource;
	
	int dummycount	= 0;	/* counter of dummy variables */

	argtypes = (Oid *) palloc(query->nparams * sizeof(Oid));
	
	/*
	 * Read typeid from Datum items
	 *
	 */
	for (i = 0; i < query->nparams; i++)
	{
		PLpgPSM_datum *datum = plpgpsm_Datums[i];
	
		
		switch (datum->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				{
					PLpgPSM_var *var = (PLpgPSM_var *) datum;
					argtypes[i] = var->datatype->typoid;
					
					break;
				}
			
			case PLPGPSM_DTYPE_ROW:
				{
					PLpgPSM_row *row = (PLpgPSM_row *) datum;
					if (!row->rowtupdesc)
						elog(ERROR, "row variable has no tupdesc");
						
					BlessTupleDesc(row->rowtupdesc);
					argtypes[i] = row->rowtupdesc->tdtypeid;
					
					break;
				}
				
			case PLPGPSM_DTYPE_REC:
				{
					PLpgPSM_rec *rec = (PLpgPSM_rec *) datum;
					if (!rec->tupdesc)
						elog(ERROR, "rec variable has no tupdesc");
						
					BlessTupleDesc(rec->tupdesc);
					argtypes[i] = rec->tupdesc->tdtypeid;
					
					break;
				}
				
			case PLPGPSM_DTYPE_RECFIELD:
				{
					PLpgPSM_recfield *recfield = (PLpgPSM_recfield *) datum;                                              
					PLpgPSM_rec *rec;                                                                                     
					int                     fno;                                                                          
                                                                                                        
					rec = (PLpgPSM_rec *) (plpgpsm_Datums[recfield->recparentno]);                                        
					if (!HeapTupleIsValid(rec->tup))                                                                      
						ereport(ERROR,                                                                                
							(errcode(ERRCODE_OBJECT_NOT_IN_PREREQUISITE_STATE),                                 
                					 errmsg("record \"%s\" is not assigned yet",                                        
							 rec->refname),                                                      
							 errdetail("The tuple structure of a not-yet-assigned record is indeterminate."))); 
					
					fno = SPI_fnumber(rec->tupdesc, recfield->fieldname);                                                 
					if (fno == SPI_ERROR_NOATTRIBUTE)                                                                     
						ereport(ERROR,                                                                                
				                              (errcode(ERRCODE_UNDEFINED_COLUMN),                                           
				                               errmsg("record \"%s\" has no field \"%s\"",                                  
                                				      rec->refname, recfield->fieldname)));                         
					argtypes[i] = SPI_gettypeid(rec->tupdesc, fno);                                                           
					
					break;				
				}
			
			case PLPGPSM_DTYPE_TRIGARG:
				{
					argtypes[i] = INT4OID;
					
					break;
				}

			default:
				elog(ERROR, "unrecognized dtype: %d", datum->dtype);
		}
	}
	
	oldCxt = MemoryContextSwitchTo(compile_tmp_cxt);		
	spi_plan = (_SPI_plan *) SPI_prepare(query->query, query->nparams, argtypes); 	
	MemoryContextSwitchTo(oldCxt);

	if (!spi_plan->plancache_list)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("can't detect attributes from SQL query")));

	plansource = (CachedPlanSource *) linitial(spi_plan->plancache_list);	
	tupDesc = plansource->resultDesc;

	if (!tupDesc)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
				 errmsg("can't detect attributes from SQL query")));	

	new = (PLpgPSM_stmt_fors*) palloc0(sizeof(PLpgPSM_stmt_fors));
	new->cmd_type = PLPGPSM_STMT_FORS;
	
	new->query = query;
	new->label = label;
	new->lineno = lno;
	
	plpgpsm_ns_push(inner_label);
	
	/* register local variables relevant to result */
	for (i = 0; i < tupDesc->natts; i++)
	{
		PLpgPSM_variable 	*var;
	
		if (nfields >= 1023)
		{
			plpgpsm_error_lineno = plpgpsm_scanner_lineno();
			ereport(ERROR,	
					(errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
					 errmsg("too many columns specified")));
		}
	
		if (strcmp(NameStr(tupDesc->attrs[i]->attname), "?column?") != 0)
		{
			var = plpgpsm_build_variable(NameStr(tupDesc->attrs[i]->attname),
							    lno, 
								    plpgpsm_build_datatype(tupDesc->attrs[i]->atttypid, -1), 
								    true, NULL);
			fieldnames[nfields] = pstrdup(NameStr(tupDesc->attrs[i]->attname));
			varnos[nfields++]	= var->dno;
		}
		else
		{
			char varname[10];
		    
			sprintf(varname, 
					    "__%d____", dummycount++);

			var = plpgpsm_build_variable(pstrdup(varname),
							    lno, 
								    plpgpsm_build_datatype(tupDesc->attrs[i]->atttypid, -1), 
								    true, NULL);
			fieldnames[nfields] = pstrdup(varname);
			varnos[nfields++]	= var->dno;								    
		}	
	}
	
	/* create scalar list from for's variables */
	SPI_freeplan((void*) spi_plan);
	
	row = palloc0(sizeof(PLpgPSM_row));
	row->dtype = PLPGPSM_DTYPE_ROW;
	row->refname = pstrdup("**internal**");
	row->lineno = lno;
	row->rowtupdesc = NULL;
	row->nfields = nfields;
	row->fieldnames = palloc(sizeof(char*) * nfields);
	row->varnos = palloc(sizeof(int) * nfields);
	while (--nfields >= 0)
	{
		row->fieldnames[nfields] = fieldnames[nfields];
		row->varnos[nfields] = varnos[nfields];
	} 
		
	plpgpsm_adddatum((PLpgPSM_datum *) row);	
	new->row = row;
	
	return new;
}

/*
 * Fix up CASE statement
 */
static PLpgPSM_stmt *
make_case(int lineno, PLpgPSM_expr *t_expr,
		  List *case_when_list, List *else_stmts)
{
	PLpgPSM_stmt_case 	*new;

	new = palloc(sizeof(PLpgPSM_stmt_case));
	new->cmd_type = PLPGPSM_STMT_CASE;
	new->lineno = lineno;
	new->t_expr = t_expr;
	new->t_varno = 0;
	new->case_when_list = case_when_list;
	new->have_else = (else_stmts != NIL);
	/* Get rid of list-with-NULL hack */
	if (list_length(else_stmts) == 1 && linitial(else_stmts) == NULL)
		new->else_stmts = NIL;
	else
		new->else_stmts = else_stmts;

	/*
	 * When test expression is present, we create a var for it and then
	 * convert all the WHEN expressions to "VAR IN (original_expression)".
	 * This is a bit klugy, but okay since we haven't yet done more than
	 * read the expressions as text.  (Note that previous parsing won't
	 * have complained if the WHEN ... THEN expression contained multiple
	 * comma-separated values.)
	 */
	if (t_expr)
	{
		ListCell *l;
		PLpgPSM_var *t_var;
		int		t_varno;

		/*
		 * We don't yet know the result datatype of t_expr.  Build the
		 * variable as if it were INT4; we'll fix this at runtime if needed.
		 */
		t_var = (PLpgPSM_var *)
			plpgpsm_build_variable("*case*", lineno,
								   plpgpsm_build_datatype(INT4OID, -1),
								   false, NULL);
		t_varno = t_var->dno;
		new->t_varno = t_varno;

		foreach(l, case_when_list)
		{
			PLpgPSM_case_when *cwt = (PLpgPSM_case_when *) lfirst(l);
			PLpgPSM_expr *expr = cwt->expr;
			int		nparams = expr->nparams;
			PLpgPSM_expr *new_expr;
			PLpgPSM_dstring ds;
			char	buff[32];

			/* Must add the CASE variable as an extra param to expression */
			if (nparams >= MAX_EXPR_PARAMS)
			{
				plpgpsm_error_lineno = cwt->lineno;
				ereport(ERROR,
					    (errcode(ERRCODE_PROGRAM_LIMIT_EXCEEDED),
					     errmsg("too many variables specified in SQL statement")));
			}

			new_expr = palloc(sizeof(PLpgPSM_expr) + sizeof(int) * (nparams + 1) - sizeof(int));
			memcpy(new_expr, expr,
				   sizeof(PLpgPSM_expr) + sizeof(int) * nparams - sizeof(int));
			new_expr->nparams = nparams + 1;
			new_expr->params[nparams] = t_varno;

			/* And do the string hacking */
			plpgpsm_dstring_init(&ds);

			plpgpsm_dstring_append(&ds, "SELECT $");
			snprintf(buff, sizeof(buff), "%d", nparams + 1);
			plpgpsm_dstring_append(&ds, buff);
			plpgpsm_dstring_append(&ds, " IN (");

			/* copy expression query without SELECT keyword */
			Assert(strncmp(expr->query, "SELECT ", 7) == 0);
			plpgpsm_dstring_append(&ds, expr->query + 7);
			plpgpsm_dstring_append_char(&ds, ')');

			new_expr->query = pstrdup(plpgpsm_dstring_get(&ds));

			plpgpsm_dstring_free(&ds);
			pfree(expr->query);
			pfree(expr);

			cwt->expr = new_expr;
		}
	}

	return (PLpgPSM_stmt *) new;
}




/* Needed to avoid conflict between different prefix settings: */
#undef yylex

#include "pl_scan.c"


