/*
 * textsearch_senna.c
 *
 *	Copyright (c) 2009, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */
#include "postgres.h"

#include <senna.h>

#include "access/reloptions.h"
#include "access/relscan.h"
#include "access/xact.h"
#include "catalog/catalog.h"
#include "catalog/index.h"
#include "catalog/pg_type.h"
#include "commands/tablecmds.h"
#include "mb/pg_wchar.h"
#include "miscadmin.h"
#include "nodes/parsenodes.h"
#include "nodes/relation.h"
#include "postmaster/syslogger.h"
#include "storage/fd.h"
#include "storage/lmgr.h"
#include "storage/bufmgr.h"
#include "utils/acl.h"
#include "utils/builtins.h"
#include "utils/inval.h"
#include "utils/lsyscache.h"
#include "utils/memutils.h"
#include "utils/selfuncs.h"

PG_MODULE_MAGIC;

/*
#define DEBUG_LEVEL		LOG
*/

#ifdef DEBUG_LEVEL
#define DEBUG_INDEX_CREATE(rnode) \
	elog(DEBUG_LEVEL, "senna: sen_index_create(%u/%u/%u)", (rnode).spcNode, (rnode).dbNode, (rnode).relNode)
#define DEBUG_INDEX_OPEN(rnode) \
	elog(DEBUG_LEVEL, "senna: sen_index_open(%u/%u/%u)", (rnode).spcNode, (rnode).dbNode, (rnode).relNode)
#define DEBUG_INDEX_CLOSE(rnode) \
	elog(DEBUG_LEVEL, "senna: sen_index_close(%u/%u/%u)", (rnode).spcNode, (rnode).dbNode, (rnode).relNode)
#define DEBUG_DELETE(message, ctid) \
	elog(DEBUG_LEVEL, "senna: delete with %s (%u, %u)", \
		message, \
		ItemPointerGetBlockNumber(ctid), \
		ItemPointerGetOffsetNumber(ctid))
#else
#define DEBUG_INDEX_CREATE(rnode)		((void) 0)
#define DEBUG_INDEX_OPEN(rnode)			((void) 0)
#define DEBUG_INDEX_CLOSE(rnode)		((void) 0)
#define DEBUG_DELETE(message, ctid)		((void) 0)
#endif

#define INDEX_CACHE_SIZE	4
#define QUERY_CACHE_SIZE	8
#define SENNA_MAX_N_EXPR	32
#define REMOVE_RETRY		300	/* 30s */

/*
 * VACUUM_USES_HEAP - read heap tuples during vacuum if defined.
 */
#define VACUUM_USES_HEAP

typedef struct SennaOptions
{
	int32       vl_len_;	/* varlena header (do not touch directly!) */
	int32		initial_n_segments;
} SennaOptions;

typedef struct SennaIndexCache
{
	Oid				relid;
	RelFileNode		rnode;
	sen_index	   *index;
} SennaIndexCache;

typedef struct SennaQueryCache
{
	char		   *key;
	sen_query	   *query;
} SennaQueryCache;

typedef struct SennaScanDesc
{
	sen_records	   *records;
#if NOT_USED
	sen_sort_optarg	sorter;
#endif
} SennaScanDesc;

typedef struct SennaBuildState
{
	sen_index	   *index;
} SennaBuildState;

PG_FUNCTION_INFO_V1(pg_sync_file);
PG_FUNCTION_INFO_V1(senna_drop_index);
PG_FUNCTION_INFO_V1(senna_reindex_index);
PG_FUNCTION_INFO_V1(senna_contains);
PG_FUNCTION_INFO_V1(senna_contained);
PG_FUNCTION_INFO_V1(senna_insert);
PG_FUNCTION_INFO_V1(senna_beginscan);
PG_FUNCTION_INFO_V1(senna_gettuple);
PG_FUNCTION_INFO_V1(senna_getbitmap);
PG_FUNCTION_INFO_V1(senna_rescan);
PG_FUNCTION_INFO_V1(senna_endscan);
PG_FUNCTION_INFO_V1(senna_build);
PG_FUNCTION_INFO_V1(senna_bulkdelete);
PG_FUNCTION_INFO_V1(senna_vacuumcleanup);
PG_FUNCTION_INFO_V1(senna_costestimate);
PG_FUNCTION_INFO_V1(senna_options);

extern void		_PG_init(void);
extern void		_PG_fini(void);
extern Datum pg_sync_file(PG_FUNCTION_ARGS);
extern Datum senna_drop_index(PG_FUNCTION_ARGS);
extern Datum senna_reindex_index(PG_FUNCTION_ARGS);
extern Datum senna_contains(PG_FUNCTION_ARGS);
extern Datum senna_contained(PG_FUNCTION_ARGS);
extern Datum senna_insert(PG_FUNCTION_ARGS);
extern Datum senna_beginscan(PG_FUNCTION_ARGS);
extern Datum senna_gettuple(PG_FUNCTION_ARGS);
extern Datum senna_getbitmap(PG_FUNCTION_ARGS);
extern Datum senna_rescan(PG_FUNCTION_ARGS);
extern Datum senna_endscan(PG_FUNCTION_ARGS);
extern Datum senna_build(PG_FUNCTION_ARGS);
extern Datum senna_bulkdelete(PG_FUNCTION_ARGS);
extern Datum senna_vacuumcleanup(PG_FUNCTION_ARGS);
extern Datum senna_costestimate(PG_FUNCTION_ARGS);
extern Datum senna_options(PG_FUNCTION_ARGS);

static char *SennaPath(const RelFileNode *rnode);
static sen_encoding SennaEncoding(void);
static sen_index *SennaIndexOpen(Relation index, bool create);
static void SennaIndexClose(Oid relid, const RelFileNode *rnode);
static bool SennaRemove(const RelFileNode *rnode);
static sen_query *SennaQuery(const char *key, size_t len);
static sen_records *SennaRecordsOpen(void);
static void SennaRecordsClose(sen_records *records);
static void SennaLock(Relation index, LOCKMODE lockmode);
static void SennaUnlock(Relation index, LOCKMODE lockmode);
static void SennaScanOpen(SennaScanDesc *desc, IndexScanDesc scan, Datum datum);
static bool SennaScanNext(SennaScanDesc *desc, ItemPointer ctid);
static void SennaScanClose(SennaScanDesc *desc);
static void SennaInsert(sen_index *index, ItemPointer ctid, Datum value);
static void SennaDelete(sen_index *index, ItemPointer key, text *value);
static void SennaBuildCallback(Relation index, HeapTuple htup, Datum *values, bool *isnull, bool tupleIsAlive, void *context);
static size_t SennaIndexSize(sen_index *index);
static IndexBulkDeleteResult *SennaBulkDeleteResult(sen_index *index);
static void SennaXactCallback(XactEvent event, void *arg);
static void SennaCacheRelCallback(Datum arg, Oid relid);

static int RELOPT_KIND_SENNA;
static SennaIndexCache	index_cache[INDEX_CACHE_SIZE];
static SennaQueryCache	query_cache[QUERY_CACHE_SIZE];
static List	*records_list;

#define SennaInitialSegments(relation) \
	((relation)->rd_options ? \
	 ((SennaOptions *) (relation)->rd_options)->initial_n_segments : 8)

/* copied from adt/genfile.c */
static char *
convert_and_check_filename(text *arg)
{
	char	   *filename;

	filename = text_to_cstring(arg);
	canonicalize_path(filename);	/* filename can change length here */

	/* Disallow ".." in the path */
	if (path_contains_parent_reference(filename))
		ereport(ERROR,
				(errcode(ERRCODE_INSUFFICIENT_PRIVILEGE),
			(errmsg("reference to parent directory (\"..\") not allowed"))));

	if (is_absolute_path(filename))
	{
		/* Allow absolute references within DataDir */
		if (path_is_prefix_of_path(DataDir, filename))
			return filename;
		/* The log directory might be outside our datadir, but allow it */
		if (is_absolute_path(Log_directory) &&
			path_is_prefix_of_path(Log_directory, filename))
			return filename;

		ereport(ERROR,
				(errcode(ERRCODE_INSUFFICIENT_PRIVILEGE),
				 (errmsg("absolute path not allowed"))));
		return NULL;			/* keep compiler quiet */
	}
	else
	{
		return filename;
	}
}

Datum
pg_sync_file(PG_FUNCTION_ARGS)
{
	text	   *filename_t = PG_GETARG_TEXT_P(0);
	char	   *filename;
	int			fd;

	if (!superuser())
		ereport(ERROR,
				(errcode(ERRCODE_INSUFFICIENT_PRIVILEGE),
				 (errmsg("must be superuser to read files"))));

	filename = convert_and_check_filename(filename_t);

	if ((fd = BasicOpenFile(filename, O_RDWR | PG_BINARY, 0)) < 0)
		ereport(ERROR,
			(errcode_for_file_access(),
			 errmsg("could not open file \"%s\" for reading: %m",
					filename)));

	if (pg_fsync(fd))
		ereport(WARNING,
				(errcode_for_file_access(),
				 errmsg("could not fsync file \"%s\": %m", filename)));

	if (close(fd))
		ereport(WARNING,
				(errcode_for_file_access(),
				 errmsg("could not close file \"%s\": %m", filename)));

	pfree(filename);

	PG_RETURN_VOID();
}

/*
 * Open senna index, or raise error if the index is not a senna.
 */
static Relation
senna_index_open(Oid relid, LOCKMODE lockmode)
{
	Relation	index;
	const char *type;

	index = index_open(relid, lockmode);

	/* Check permissions */
	if (!pg_class_ownercheck(relid, GetUserId()))
		aclcheck_error(ACLCHECK_NOT_OWNER, ACL_KIND_CLASS,
					   RelationGetRelationName(index));

	type = NameStr(index->rd_am->amname);

	if (strcmp(type, "senna") != 0)
		ereport(ERROR,
				(errcode(ERRCODE_WRONG_OBJECT_TYPE),
				 errmsg("senna: not a senna index (%s)", type)));

	return index;
}

static void
IndexCacheDispose(SennaIndexCache *cache)
{
	sen_rc	rc;

	DEBUG_INDEX_CLOSE(cache->rnode);
	rc = sen_index_close(cache->index);
	if (rc != sen_success)
		elog(WARNING, "senna: sen_index_close() : code=%d", rc);
}

Datum
senna_drop_index(PG_FUNCTION_ARGS)
{
	Oid			relid = PG_GETARG_OID(0);
	Relation	index;
	RelFileNode	rnode;
	DropStmt   *stmt;
	List	   *obj;

	index = senna_index_open(relid, NoLock);

	obj = list_make2(
		makeString(get_namespace_name(RelationGetNamespace(index))),
		makeString(pstrdup(RelationGetRelationName(index))));

	stmt = makeNode(DropStmt);
	stmt->removeType = OBJECT_INDEX;
	stmt->missing_ok = true;
	stmt->objects = list_make1(obj);
	stmt->behavior = DROP_CASCADE;

	rnode = index->rd_node;
	index_close(index, NoLock);

	RemoveRelations(stmt);

	SennaRemove(&rnode);

	PG_RETURN_VOID();
}

Datum
senna_reindex_index(PG_FUNCTION_ARGS)
{
	Oid			relid = PG_GETARG_OID(0);
	Relation	index;
	RelFileNode	rnode;

	index = senna_index_open(relid, NoLock);
	rnode = index->rd_node;
	index_close(index, NoLock);

	reindex_index(relid);

	SennaRemove(&rnode);

	PG_RETURN_VOID();
}

static bool
SennaContains(text *doc, text *query)
{
	const char *str = VARDATA_ANY(doc);
	int			len = VARSIZE_ANY_EXHDR(doc);
	sen_query  *q;
	sen_rc		rc;
	int			found;
	int			score;

	q = SennaQuery(VARDATA_ANY(query), VARSIZE_ANY_EXHDR(query));

	rc = sen_query_scan(q, &str, &len, 1, SEN_QUERY_SCAN_NORMALIZE, &found, &score);
	if (rc != sen_success)
		elog(ERROR, "senna: sen_query_scan() : code=%d", rc);

	return found && score;
}

Datum
senna_contains(PG_FUNCTION_ARGS)
{
	PG_RETURN_BOOL(SennaContains(
		PG_GETARG_TEXT_PP(0), PG_GETARG_TEXT_PP(1)));
}

Datum
senna_contained(PG_FUNCTION_ARGS)
{
	PG_RETURN_BOOL(SennaContains(
		PG_GETARG_TEXT_PP(1), PG_GETARG_TEXT_PP(0)));
}

Datum
senna_insert(PG_FUNCTION_ARGS)
{
	Relation	index = (Relation) PG_GETARG_POINTER(0);
	Datum	   *values = (Datum *) PG_GETARG_POINTER(1);
	bool	   *isnull = (bool *) PG_GETARG_POINTER(2);
	ItemPointer	ctid = (ItemPointer) PG_GETARG_POINTER(3);
#ifdef NOT_USED
	Relation	heap = (Relation) PG_GETARG_POINTER(4);
	bool		checkUnique = PG_GETARG_BOOL(5);
#endif

	sen_index  *i;

	/* cannot index nulls */
	if (isnull[0])
		PG_RETURN_BOOL(false);

	i = SennaIndexOpen(index, false);

	SennaLock(index, ExclusiveLock);
	SennaInsert(i, ctid, values[0]);
	SennaUnlock(index, ExclusiveLock);

	PG_RETURN_BOOL(true);
}

Datum
senna_beginscan(PG_FUNCTION_ARGS)
{
	Relation		index = (Relation) PG_GETARG_POINTER(0);
	int				keysz = PG_GETARG_INT32(1);
	ScanKey			scankey = (ScanKey) PG_GETARG_POINTER(2);
	IndexScanDesc	scan;

	if (keysz < 1)
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("senna: do not support whole-index scans")));

	scan = RelationGetIndexScan(index, keysz, scankey);

	PG_RETURN_POINTER(scan);
}

static text *
extract_value(HeapTuple tuple, Relation heap, Relation index)
{
	TupleDesc	tupdesc;
	int			attnum;
	Datum		value;
	bool		isnull;

	if (tuple->t_data == NULL)
		return NULL;

	tupdesc = RelationGetDescr(heap);
	attnum = index->rd_index->indkey.values[0];
	value = heap_getattr(tuple, attnum, tupdesc, &isnull);

	if (isnull)
		return NULL;
	else
		return DatumGetTextPP(value);
}

#ifdef NOT_USED
static int
sort_by_score(sen_records *r1, const sen_recordh *rh1,
			  sen_records *r2, const sen_recordh *rh2, void *arg)
{
	sen_rc	rc;
	int		score1;
	int		score2;
	
	rc = sen_record_info(r1, rh1, NULL, 0, NULL, NULL, NULL, &score1, NULL);
	rc = sen_record_info(r2, rh2, NULL, 0, NULL, NULL, NULL, &score2, NULL);

	if (score1 < score2)
		return -1;
	else if (score1 > score2)
		return +1;
	else
		return 0;
}
#endif

Datum
senna_gettuple(PG_FUNCTION_ARGS)
{
	IndexScanDesc	scan = (IndexScanDesc) PG_GETARG_POINTER(0);
	ScanDirection	dir = (ScanDirection) PG_GETARG_INT32(1);

	SennaScanDesc  *desc = (SennaScanDesc *) scan->opaque;
	ItemPointerData ctid;

	if (dir != ForwardScanDirection)
		elog(ERROR, "senna: only supports forward scans");

	if (scan->kill_prior_tuple)
	{
		sen_index  *index = SennaIndexOpen(scan->indexRelation, false);
		text	   *value = extract_value(&scan->xs_ctup,
								scan->heapRelation, scan->indexRelation);

		SennaLock(scan->indexRelation, ExclusiveLock);
		SennaDelete(index, &scan->xs_ctup.t_self, value);
		SennaUnlock(scan->indexRelation, ExclusiveLock);
	}

#if NOT_USED
	/*
	 * sort is performed in gettuple but not in rescan because it is
	 * not useful for bitmap scans.
	 */
	if (desc->sorter.compar != NULL)
	{
		int	limit = sen_records_nhits(desc->records);
		sen_records_sort(desc->records, limit, &desc->sorter);
		desc->sorter.compar = NULL;
	}
#endif

	scan->xs_recheck = false;
	if (!SennaScanNext(desc, &ctid))
		PG_RETURN_BOOL(false);

	scan->xs_ctup.t_self = ctid;
	PG_RETURN_BOOL(true);
}

Datum
senna_getbitmap(PG_FUNCTION_ARGS)
{
	IndexScanDesc	scan = (IndexScanDesc) PG_GETARG_POINTER(0);
	TIDBitmap	   *tbm = (TIDBitmap *) PG_GETARG_POINTER(1);

	SennaScanDesc  *desc = (SennaScanDesc *) scan->opaque;
	int64			ntids;
	ItemPointerData ctid;

	for (ntids = 0; SennaScanNext(desc, &ctid); ntids++)
	{
		CHECK_FOR_INTERRUPTS();
		tbm_add_tuples(tbm, &ctid, 1, false);
	}

	PG_RETURN_INT64(ntids);
}

Datum
senna_rescan(PG_FUNCTION_ARGS)
{
	IndexScanDesc	scan = (IndexScanDesc) PG_GETARG_POINTER(0);
	ScanKey			scankey = (ScanKey) PG_GETARG_POINTER(1);

	SennaScanDesc  *desc = (SennaScanDesc  *) scan->opaque;

	if (desc == NULL)
	{
		scan->opaque = desc = palloc0(sizeof(SennaScanDesc));
#ifdef NOT_USED
		desc->sorter.mode = sen_sort_descending;
		desc->sorter.compar = sort_by_score;
#endif
	}
	else
		SennaScanClose(desc);

	if (scankey && scan->numberOfKeys > 0)
		SennaScanOpen(desc, scan, scankey[0].sk_argument);

	PG_RETURN_VOID();
}

Datum
senna_endscan(PG_FUNCTION_ARGS)
{
	IndexScanDesc	scan = (IndexScanDesc) PG_GETARG_POINTER(0);

	SennaScanDesc  *desc = (SennaScanDesc *) scan->opaque;

	if (desc != NULL)
	{
		SennaScanClose(desc);
		pfree(desc);
	}

	PG_RETURN_VOID();
}

Datum
senna_build(PG_FUNCTION_ARGS)
{
	Relation	heap = (Relation) PG_GETARG_POINTER(0);
	Relation	index = (Relation) PG_GETARG_POINTER(1);
	IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
	TupleDesc	tupdesc;

	IndexBuildResult   *result;
	SennaBuildState		state;

	if (indexInfo->ii_Unique)
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("senna: do not support unique index")));

	if (RelationGetNumberOfAttributes(index) != 1)
		ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("senna: do not support multi columns")));

	tupdesc = RelationGetDescr(index);
	switch (tupdesc->attrs[0]->atttypid)
	{
		case BPCHAROID:
		case VARCHAROID:
		case TEXTOID:
			break;
		default:
			ereport(ERROR,
				(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
				 errmsg("senna: only supports char, varchar and text")));
	}

	state.index = SennaIndexOpen(index, true);

	PG_TRY();
	{
		result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
		SennaLock(index, ExclusiveLock);
		result->heap_tuples = IndexBuildHeapScan(heap, index, indexInfo, true, SennaBuildCallback, &state);
		SennaUnlock(index, ExclusiveLock);
		result->index_tuples = sen_sym_size(state.index->keys);
	}
	PG_CATCH();
	{
		SennaRemove(&index->rd_node);
		PG_RE_THROW();
	}
	PG_END_TRY();

	PG_RETURN_POINTER(result);
}

Datum
senna_bulkdelete(PG_FUNCTION_ARGS)
{
	IndexVacuumInfo		   *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0);
	IndexBulkDeleteResult  *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);
	IndexBulkDeleteCallback	callback = (IndexBulkDeleteCallback) PG_GETARG_POINTER(2);
	void				   *callback_state = (void *) PG_GETARG_POINTER(3);

	double			tuples_removed;
	sen_id			id;
	sen_index	   *index;

#ifdef VACUUM_USES_HEAP
	Relation		heap;
	HeapTupleData	tuple;
#endif

	index = SennaIndexOpen(info->index, false);

	if (stats == NULL)
		stats = SennaBulkDeleteResult(index);

	if (callback == NULL)
		PG_RETURN_POINTER(stats);

#ifdef VACUUM_USES_HEAP
	heap = heap_open(info->index->rd_index->indrelid, NoLock);
	tuple.t_tableOid = RelationGetRelid(heap);
#endif

	tuples_removed = 0;
	id = 0;
	for (;;)
	{
		ItemPointerData	ctid;
		sen_id			next_id;
		
		CHECK_FOR_INTERRUPTS();

		next_id = sen_sym_next(index->keys, id);
		if (next_id == SEN_SYM_NIL || next_id == id)
			break;
		id = next_id;

		if (sen_sym_key(index->keys, id, &ctid, SizeOfIptrData) != SizeOfIptrData)
		{
			elog(WARNING, "senna: sen_sym_key()");
			continue;
		}

		if (callback(&ctid, callback_state))
		{
			text		   *value = NULL;

#ifdef VACUUM_USES_HEAP
			BlockNumber		blknum;
			BlockNumber		offnum;
			Buffer			buffer;
			Page			page;
			ItemId			itemid;

			blknum = ItemPointerGetBlockNumber(&ctid);
			offnum = ItemPointerGetOffsetNumber(&ctid);
			buffer = ReadBuffer(heap, blknum);

			LockBuffer(buffer, BUFFER_LOCK_SHARE);
			page = BufferGetPage(buffer);
			itemid = PageGetItemId(page, offnum);
			tuple.t_data = ItemIdIsNormal(itemid)
				? (HeapTupleHeader) PageGetItem(page, itemid)
				: NULL;
			LockBuffer(buffer, BUFFER_LOCK_UNLOCK);

			if (tuple.t_data != NULL)
			{
				tuple.t_len = ItemIdGetLength(itemid);
				tuple.t_self = ctid;
				value = extract_value(&tuple, heap, info->index);
			}
#endif
			SennaLock(info->index, ExclusiveLock);
			SennaDelete(index, &ctid, value);
			SennaUnlock(info->index, ExclusiveLock);

#ifdef VACUUM_USES_HEAP
			ReleaseBuffer(buffer);
#endif

			tuples_removed += 1;
		}
	}

#ifdef VACUUM_USES_HEAP
	heap_close(heap, NoLock);
#endif

	stats->tuples_removed = tuples_removed;

	PG_RETURN_POINTER(stats);
}

Datum
senna_vacuumcleanup(PG_FUNCTION_ARGS)
{
	IndexVacuumInfo *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0);
	IndexBulkDeleteResult *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);

	if (stats == NULL)
		stats = SennaBulkDeleteResult(SennaIndexOpen(info->index, false));

	PG_RETURN_POINTER(stats);
}

Datum
senna_costestimate(PG_FUNCTION_ARGS)
{
	/*
	 * We cannot use genericcostestimate because it is a static funciton.
	 * Use gincostestimate instead, which just calls genericcostestimate.
	 */
	return gincostestimate(fcinfo);
}

Datum
senna_options(PG_FUNCTION_ARGS)
{
	Datum			reloptions = PG_GETARG_DATUM(0);
	bool			validate = PG_GETARG_BOOL(1);
	relopt_value   *options;
	SennaOptions   *rdopts;
	int				numoptions;
	static const relopt_parse_elt tab[] = {
		{"initial_n_segments", RELOPT_TYPE_INT, offsetof(SennaOptions, initial_n_segments)}
	};

	options = parseRelOptions(reloptions, validate, RELOPT_KIND_SENNA,
							  &numoptions);

	/* if none set, we're done */
	if (numoptions == 0)
		PG_RETURN_NULL();

	rdopts = allocateReloptStruct(sizeof(SennaOptions), options, numoptions);

	fillRelOptions(rdopts, sizeof(SennaOptions), options, numoptions,
					validate, tab, lengthof(tab));

	pfree(options);

	PG_RETURN_BYTEA_P(rdopts);
}

static char *
SennaPath(const RelFileNode *rnode)
{
	return relpath(*rnode, MAIN_FORKNUM);
}

static sen_encoding	sen_encoding_cache = sen_enc_default;

static sen_encoding
SennaEncoding(void)
{
	if (sen_encoding_cache == sen_enc_default)
	{
		if (pg_database_encoding_max_length() == 1)
			sen_encoding_cache = sen_enc_none;
		else switch (GetDatabaseEncoding())
		{
			case PG_EUC_JP:
			case PG_EUC_JIS_2004:
				sen_encoding_cache = sen_enc_euc_jp;
				break;
			case PG_UTF8:
				sen_encoding_cache = sen_enc_utf8;
				break;
			default:
				ereport(ERROR,
						(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
						 errmsg("senna: do not support %s encoding", GetDatabaseEncodingName())));
		}
	}
	return sen_encoding_cache;
}

static void
SennaBuildCallback(Relation index,
				   HeapTuple htup,
				   Datum *values,
				   bool *isnull,
				   bool tupleIsAlive,
				   void *context)
{
	SennaBuildState	*state = (SennaBuildState *) context;

	if (!isnull[0])
		SennaInsert(state->index, &htup->t_self, values[0]);
}

static sen_index *
SennaIndexOpen(Relation indexRelation, bool create)
{
	int					i;
	SennaIndexCache    *cache;
	sen_index		   *index;
	char			   *path;

	/* search query cache */
	for (i = 0; i < lengthof(index_cache); i++)
	{
		cache = &index_cache[i];

		if (cache->index == NULL)
			break;
		if (RelFileNodeEquals(cache->rnode, indexRelation->rd_node))
			return cache->index;
	}

	/* Be careful not to leak resources on error. */
	path = SennaPath(&indexRelation->rd_node);
	if (create)
	{
		DEBUG_INDEX_CREATE(indexRelation->rd_node);
		index = sen_index_create(
			path,
			SizeOfIptrData,
			SEN_INDEX_NGRAM | SEN_INDEX_NORMALIZE,
			SennaInitialSegments(indexRelation),
			SennaEncoding());
		if (index == NULL)
			elog(ERROR, "senna: sen_index_create(%s)", path);
	}
	else
	{
		DEBUG_INDEX_OPEN(indexRelation->rd_node);
		index = sen_index_open(path);
		if (index == NULL)
			elog(ERROR, "senna: sen_index_open(%s)", path);
	}

	if (cache->index != NULL)
	{
		/* release the last */
		cache = &index_cache[lengthof(index_cache) - 1];
		IndexCacheDispose(cache);
		/* pop the last */
		memmove(&index_cache[1], &index_cache[0],
			(lengthof(index_cache) - 1) * sizeof(SennaIndexCache));
		/* first one */
		cache = &index_cache[0];
	}

	cache->relid = RelationGetRelid(indexRelation);
	cache->rnode = indexRelation->rd_node;
	cache->index = index;

	pfree(path);

	return index;
}

static void
SennaIndexClose(Oid relid, const RelFileNode *rnode)
{
	int		i;

	if (relid == InvalidOid && rnode == NULL)
	{
		/* close all indexes */
		for (i = lengthof(index_cache) - 1; i >= 0; i--)
		{
			SennaIndexCache *cache = &index_cache[i];
			if (cache->index != NULL)
			{
				IndexCacheDispose(cache);
				cache->index = NULL;
			}
		}
	}
	else
	{
		/* search index cache */
		for (i = 0; i < lengthof(index_cache); i++)
		{
			SennaIndexCache *cache = &index_cache[i];

			if (cache->index == NULL)
				break;
			if ((rnode != NULL && RelFileNodeEquals(cache->rnode, *rnode)) ||
				cache->relid == relid)
			{
				IndexCacheDispose(cache);
				memmove(&index_cache[i], &index_cache[i + 1],
					(lengthof(index_cache) - i - 1) * sizeof(SennaIndexCache));
				index_cache[lengthof(index_cache) - 1].index = NULL;
				return;
			}
		}
		/* not found */
	}
}

static bool
SennaRemove(const RelFileNode *rnode)
{
	int			loops = 0;
	sen_rc		rc;
	char	   *path;

	SennaIndexClose(InvalidOid, rnode);
	path = SennaPath(rnode);

	while ((rc = sen_index_remove(path)) != sen_success)
	{
		if (rc != sen_file_operation_error || ++loops > REMOVE_RETRY)
		{
			elog(WARNING, "senna: sen_index_remove(%s) : code=%d", path, rc);
			break;
		}

		pg_usleep(100000);		/* us */

		CHECK_FOR_INTERRUPTS();
	}

	pfree(path);

	return rc == sen_success;
}

static sen_query *
SennaQuery(const char *sql, size_t len)
{
	int					i;
	SennaQueryCache    *cache;
	sen_query		   *query;
	char			   *key;

	/* search query cache */
	for (i = 0; i < lengthof(query_cache); i++)
	{
		cache = &query_cache[i];

		if (cache->query == NULL)
			break;
		if (strncmp(cache->key, sql, len) == 0)
			return cache->query;
	}

	/* Be careful not to leak resources on error. */
	key = malloc(len + 1);
	if (key == NULL)
		ereport(ERROR,
				(errcode(ERRCODE_OUT_OF_MEMORY),
				 errmsg("out of memory")));
	strlcpy(key, sql, len + 1);
	query = sen_query_open(sql, len, sen_sel_or, SENNA_MAX_N_EXPR, SennaEncoding());
	if (query == NULL)
		elog(ERROR, "senna: sen_query_open()");

	if (cache->query != NULL)
	{
		/* release the last */
		cache = &query_cache[lengthof(query_cache) - 1];
		sen_query_close(cache->query);
		free(cache->key);
		/* pop the last */
		memmove(&query_cache[1], &query_cache[0],
			(lengthof(query_cache) - 1) * sizeof(SennaQueryCache));
		/* first one */
		cache = &query_cache[0];
	}

	cache->key = key;
	cache->query = query;
	return query;
}

static sen_records *
SennaRecordsOpen(void)
{
	MemoryContext	oldctx;
	sen_records	   *records;

	records = sen_records_open(sen_rec_document, sen_rec_none, 0);
	if (records == NULL)
		elog(ERROR, "senna: sen_records_open()");

	oldctx = MemoryContextSwitchTo(TopMemoryContext);
	records_list = lappend(records_list, records);
	MemoryContextSwitchTo(oldctx);

	return records;
}

static void
SennaRecordsClose(sen_records *records)
{
	sen_rc rc;

	rc = sen_records_close(records);
	if (rc != sen_success)
		elog(WARNING, "senna: sen_records_close() : code=%d", rc);

	records_list = list_delete_ptr(records_list, records);
}

static void
SennaLock(Relation index, LOCKMODE lockmode)
{
	const RelFileNode *rnode = &index->rd_node;
	LockDatabaseObject(rnode->spcNode,
					   rnode->dbNode,
					   rnode->relNode,
					   lockmode);
}

static void
SennaUnlock(Relation index, LOCKMODE lockmode)
{
	const RelFileNode *rnode = &index->rd_node;
	UnlockDatabaseObject(rnode->spcNode,
						 rnode->dbNode,
						 rnode->relNode,
						 lockmode);
}

static void
SennaScanOpen(SennaScanDesc *desc, IndexScanDesc scan, Datum datum)
{
	text		   *key;
	sen_index	   *index;
	sen_query	   *query;
	sen_records	   *records = NULL;
	sen_rc			rc;

	key = DatumGetTextPP(datum);
	query = SennaQuery(VARDATA_ANY(key), VARSIZE_ANY_EXHDR(key));

	records = SennaRecordsOpen();

	index = SennaIndexOpen(scan->indexRelation, false);

	/*
	 * Use sen_query_exec() instead of sen_index_select() in order to
	 * minimize locking duration. Compile queries before locking.
	 */
	SennaLock(scan->indexRelation, ShareLock);
	rc = sen_query_exec(index, query, records, sen_sel_or);
	SennaUnlock(scan->indexRelation, ShareLock);

	if (rc != sen_success)
		elog(ERROR, "senna: sen_query_exec() : code=%d", rc);

	desc->records = records;
}

static bool
SennaScanNext(SennaScanDesc *desc, ItemPointer ctid)
{
	return sen_records_next(desc->records, ctid, SizeOfIptrData, NULL) == SizeOfIptrData;
}

static void
SennaScanClose(SennaScanDesc *desc)
{
	if (desc->records)
	{
		SennaRecordsClose(desc->records);
		desc->records = NULL;
	}
}

static void
SennaInsert(sen_index *index, ItemPointer ctid, Datum value)
{
	text	   *doc = DatumGetTextPP(value);
	const char *str = VARDATA_ANY(doc);
	int			len = VARSIZE_ANY_EXHDR(doc);
	sen_rc		rc;

	rc = sen_index_upd(index, ctid, NULL, 0, str, len);

	if (rc != sen_success)
		elog(ERROR, "senna: sen_index_upd(insert) : code=%d", rc);
}

static void
SennaDelete(sen_index *index, ItemPointer ctid, text *value)
{
	sen_rc	rc;

	if (!sen_sym_at(index->keys, ctid))
		return;	/* deletion will be failed in this case. */

	if (value != NULL)
	{
		DEBUG_DELETE("upd", ctid);

		rc = sen_index_upd(
				index, ctid,
				VARDATA_ANY(value), VARSIZE_ANY_EXHDR(value),
				NULL, 0);
		if (rc != sen_success)
			elog(WARNING, "senna: sen_index_upd(delete) : code=%d", rc);
	}
	else
	{
		/* The value could be unavailable in 8.3 or later. */
		DEBUG_DELETE("del", ctid);

		rc = sen_index_del(index, ctid);
		if (rc != sen_success)
			elog(WARNING, "senna: sen_index_del() : code=%d", rc);
		rc = sen_sym_del(index->keys, ctid);
		if (rc != sen_success)
			elog(WARNING, "senna: sen_sym_del() : code=%d", rc);
	}
}

/*
 * Total index size in bytes.
 */
static size_t
SennaIndexSize(sen_index *index)
{
	unsigned file_size_keys;
	unsigned file_size_lexicon;
	unsigned long long inv_seg_size;
	unsigned long long inv_chunk_size;
	sen_rc rc;

	rc = sen_index_info(index,
						NULL,	/* key_size */
						NULL,	/* flags */
						NULL,	/* initial_n_segments */
						NULL,	/* encoding */
						NULL,	/* nrecords_keys */
						&file_size_keys,
						NULL,	/* nrecords_lexicon */
						&file_size_lexicon,
						&inv_seg_size,
						&inv_chunk_size);

	if (rc != sen_success)
		return 0;

	return file_size_keys + file_size_lexicon +
		   inv_seg_size + inv_chunk_size;
}

static IndexBulkDeleteResult *
SennaBulkDeleteResult(sen_index *index)
{
	IndexBulkDeleteResult *stats;

	stats = (IndexBulkDeleteResult *) palloc0(sizeof(IndexBulkDeleteResult));
	stats->num_pages = (BlockNumber) (SennaIndexSize(index) / BLCKSZ);
	stats->num_index_tuples = sen_sym_size(index->keys);

	return stats;
}

/*
 * Release senna objects at the end of transactions.
 */
static void
SennaXactCallback(XactEvent event, void *arg)
{
	sen_rc		rc;
	int			i;

	/* cleanup queries */
	for (i = 0; i < lengthof(query_cache); i++)
	{
		SennaQueryCache *cache = &query_cache[i];

		if (cache->query != NULL)
		{
			rc = sen_query_close(cache->query);
			if (rc != sen_success)
				elog(WARNING, "senna: sen_query_close() : code=%d", rc);
			cache->query = NULL;
		}
		if (cache->key)
		{
			free(cache->key);
			cache->key = NULL;
		}
	}

	/* cleanup records */
	if (records_list != NIL)
	{
		ListCell   *cell;

		foreach(cell, records_list)
		{
			rc = sen_records_close((sen_records *) lfirst(cell));
			if (rc != sen_success)
				elog(WARNING, "senna: sen_records_close() : code=%d", rc);
		}
		list_free(records_list);
		records_list = NIL;
	}
}

static void
SennaCacheRelCallback(Datum arg, Oid relid)
{
	SennaIndexClose(relid, NULL);
}

static void
SennaLogger(int level, const char *time, const char *title,
			const char *msg, const char *location, void *func_arg)
{
	int	elevel;

	const char *log_level[] =
	{
		"",			/* sen_log_none */
		"PANIC",	/* sen_log_emerg */
		"FATAL",	/* sen_log_alert */
		"CRITICAL",	/* sen_log_crit */
		"ERROR",	/* sen_log_error */
		"WARNING",	/* sen_log_warning */
		"NOTICE",	/* sen_log_notice */
		"INFO",		/* sen_log_info */
		"DEBUG",	/* sen_log_debug */
		"NOTICE"	/* sen_log_dump */
	};

	if (level <= sen_log_crit)
		elevel = WARNING;
	else
		elevel = LOG;

	elog(elevel, "senna: [%s]%s %s %s", log_level[level], title, msg, location);
}

static sen_logger_info senna_logger =
{
	sen_log_warning,
	SEN_LOG_TIME | SEN_LOG_MESSAGE,
	SennaLogger
};

void
_PG_init(void)
{
	sen_rc	rc;

	RELOPT_KIND_SENNA = add_reloption_kind();

	sen_logger_info_set(&senna_logger);
	rc = sen_init();
	if (rc != sen_success)
		elog(ERROR, "senna: sen_init() : code=%d", rc);

	RegisterXactCallback(SennaXactCallback, NULL);
	CacheRegisterRelcacheCallback(SennaCacheRelCallback, (Datum) 0);
}

void
_PG_fini(void)
{
#ifdef NOT_USED
	/* Somehow postgres doesn't support unregistration. */
	CacheUnregisterRelcacheCallback(SennaCacheRelCallback, (Datum) 0);
#endif
	UnregisterXactCallback(SennaXactCallback, NULL);
	sen_fin();
}
