/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.client;

import bk-shade.com.google.common.cache.Cache;
import bk-shade.com.google.common.cache.CacheBuilder;
import bk-shade.com.google.common.cache.RemovalListener;
import bk-shade.com.google.common.cache.RemovalNotification;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.BookiesListener;
import org.apache.bookkeeper.client.EnsemblePlacementPolicy;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.util.SafeRunnable;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.ACL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BookieWatcher
implements Watcher,
AsyncCallback.ChildrenCallback {
    static final Logger logger = LoggerFactory.getLogger(BookieWatcher.class);
    public static int ZK_CONNECT_BACKOFF_SEC = 1;
    private static final Set<BookieSocketAddress> EMPTY_SET = new HashSet<BookieSocketAddress>();
    private final String bookieRegistrationPath;
    final BookKeeper bk;
    final ScheduledExecutorService scheduler;
    final EnsemblePlacementPolicy placementPolicy;
    final Cache<BookieSocketAddress, Boolean> quarantinedBookies;
    SafeRunnable reReadTask = new SafeRunnable(){

        @Override
        public void safeRun() {
            BookieWatcher.this.readBookies();
        }
    };
    private ReadOnlyBookieWatcher readOnlyBookieWatcher;

    public BookieWatcher(ClientConfiguration conf, ScheduledExecutorService scheduler, EnsemblePlacementPolicy placementPolicy, BookKeeper bk) throws KeeperException, InterruptedException {
        this.bk = bk;
        this.bookieRegistrationPath = conf.getZkAvailableBookiesPath();
        this.scheduler = scheduler;
        this.placementPolicy = placementPolicy;
        this.readOnlyBookieWatcher = new ReadOnlyBookieWatcher(conf, bk);
        this.quarantinedBookies = CacheBuilder.newBuilder().expireAfterWrite(conf.getBookieQuarantineTimeSeconds(), TimeUnit.SECONDS).removalListener(new RemovalListener<BookieSocketAddress, Boolean>(){

            @Override
            public void onRemoval(RemovalNotification<BookieSocketAddress, Boolean> bookie) {
                logger.info("Bookie {} is no longer quarantined", bookie.getKey());
            }
        }).build();
    }

    void notifyBookiesChanged(final BookiesListener listener) throws BKException {
        try {
            this.bk.getZkHandle().getChildren(this.bookieRegistrationPath, new Watcher(){

                public void process(WatchedEvent event) {
                    if (event.getType() == Watcher.Event.EventType.NodeChildrenChanged) {
                        listener.availableBookiesChanged();
                    }
                }
            });
        }
        catch (KeeperException ke) {
            logger.error("Error registering watcher with zookeeper", (Throwable)ke);
            throw new BKException.ZKException();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            logger.error("Interrupted registering watcher with zookeeper", (Throwable)ie);
            throw new BKException.BKInterruptedException();
        }
    }

    void notifyReadOnlyBookiesChanged(BookiesListener listener) throws BKException {
        this.readOnlyBookieWatcher.notifyBookiesChanged(listener);
    }

    public Collection<BookieSocketAddress> getBookies() throws BKException {
        try {
            List children = this.bk.getZkHandle().getChildren(this.bookieRegistrationPath, false);
            children.remove("readonly");
            return BookieWatcher.convertToBookieAddresses(children);
        }
        catch (KeeperException ke) {
            logger.error("Failed to get bookie list : ", (Throwable)ke);
            throw new BKException.ZKException();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            logger.error("Interrupted reading bookie list", (Throwable)ie);
            throw new BKException.BKInterruptedException();
        }
    }

    Collection<BookieSocketAddress> getReadOnlyBookies() {
        return new HashSet<BookieSocketAddress>(this.readOnlyBookieWatcher.getReadOnlyBookies());
    }

    public void readBookies() {
        this.readBookies(this);
    }

    public void readBookies(AsyncCallback.ChildrenCallback callback) {
        this.bk.getZkHandle().getChildren(this.bookieRegistrationPath, (Watcher)this, callback, null);
    }

    public void process(WatchedEvent event) {
        this.readBookies();
    }

    public void processResult(int rc, String path, Object ctx, List<String> children) {
        if (rc != KeeperException.Code.OK.intValue()) {
            try {
                this.scheduler.schedule(this.reReadTask, (long)ZK_CONNECT_BACKOFF_SEC, TimeUnit.SECONDS);
            }
            catch (RejectedExecutionException ree) {
                logger.warn("Failed to schedule reading bookies task : ", (Throwable)ree);
            }
            return;
        }
        children.remove("readonly");
        HashSet<BookieSocketAddress> newBookieAddrs = BookieWatcher.convertToBookieAddresses(children);
        this.bk.mainWorkerPool.submitOrdered((Object)path, SafeRunnable.safeRun(() -> {
            BookieWatcher bookieWatcher = this;
            synchronized (bookieWatcher) {
                HashSet<BookieSocketAddress> readonlyBookies = this.readOnlyBookieWatcher.getReadOnlyBookies();
                this.placementPolicy.onClusterChanged(newBookieAddrs, readonlyBookies);
                if (this.bk.conf.getDiskWeightBasedPlacementEnabled()) {
                    this.bk.bookieInfoReader.availableBookiesChanged(newBookieAddrs);
                }
            }
        }));
    }

    private static HashSet<BookieSocketAddress> convertToBookieAddresses(List<String> children) {
        HashSet<BookieSocketAddress> newBookieAddrs = new HashSet<BookieSocketAddress>();
        for (String bookieAddrString : children) {
            BookieSocketAddress bookieAddr;
            try {
                bookieAddr = new BookieSocketAddress(bookieAddrString);
            }
            catch (IOException e) {
                logger.error("Could not parse bookie address: " + bookieAddrString + ", ignoring this bookie");
                continue;
            }
            newBookieAddrs.add(bookieAddr);
        }
        return newBookieAddrs;
    }

    public void readBookiesBlocking() throws InterruptedException, KeeperException {
        this.readOnlyBookieWatcher.readROBookiesBlocking();
        final LinkedBlockingQueue queue = new LinkedBlockingQueue();
        this.readBookies(new AsyncCallback.ChildrenCallback(){

            public void processResult(int rc, String path, Object ctx, List<String> children) {
                BookieWatcher.this.bk.mainWorkerPool.submitOrdered((Object)path, SafeRunnable.safeRun(() -> {
                    BookieWatcher.this.processResult(rc, path, ctx, children);
                    queue.add(rc);
                }));
            }
        });
        int rc = (Integer)queue.take();
        if (rc != KeeperException.Code.OK.intValue()) {
            throw KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc));
        }
    }

    public ArrayList<BookieSocketAddress> newEnsemble(int ensembleSize, int writeQuorumSize, int ackQuorumSize, Map<String, byte[]> customMetadata) throws BKException.BKNotEnoughBookiesException {
        try {
            return this.placementPolicy.newEnsemble(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, new HashSet<BookieSocketAddress>(this.quarantinedBookies.asMap().keySet()));
        }
        catch (BKException.BKNotEnoughBookiesException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("Not enough healthy bookies available, using quarantined bookies");
            }
            return this.placementPolicy.newEnsemble(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, EMPTY_SET);
        }
    }

    public BookieSocketAddress replaceBookie(int ensembleSize, int writeQuorumSize, int ackQuorumSize, Map<String, byte[]> customMetadata, List<BookieSocketAddress> existingBookies, int bookieIdx, Set<BookieSocketAddress> excludeBookies) throws BKException.BKNotEnoughBookiesException {
        BookieSocketAddress addr = existingBookies.get(bookieIdx);
        try {
            HashSet<BookieSocketAddress> existingAndQuarantinedBookies = new HashSet<BookieSocketAddress>(existingBookies);
            existingAndQuarantinedBookies.addAll(this.quarantinedBookies.asMap().keySet());
            return this.placementPolicy.replaceBookie(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, existingAndQuarantinedBookies, addr, excludeBookies);
        }
        catch (BKException.BKNotEnoughBookiesException e) {
            if (logger.isDebugEnabled()) {
                logger.debug("Not enough healthy bookies available, using quarantined bookies");
            }
            return this.placementPolicy.replaceBookie(ensembleSize, writeQuorumSize, ackQuorumSize, customMetadata, new HashSet<BookieSocketAddress>(existingBookies), addr, excludeBookies);
        }
    }

    public void quarantineBookie(BookieSocketAddress bookie) {
        if (this.quarantinedBookies.getIfPresent(bookie) == null) {
            this.quarantinedBookies.put(bookie, Boolean.TRUE);
            logger.warn("Bookie {} has been quarantined because of read/write errors.", (Object)bookie);
        }
    }

    private static class ReadOnlyBookieWatcher
    implements Watcher,
    AsyncCallback.ChildrenCallback {
        private static final Logger LOG = LoggerFactory.getLogger(ReadOnlyBookieWatcher.class);
        private volatile HashSet<BookieSocketAddress> readOnlyBookies = new HashSet();
        private BookKeeper bk;
        private String readOnlyBookieRegPath;

        public ReadOnlyBookieWatcher(ClientConfiguration conf, BookKeeper bk) throws KeeperException, InterruptedException {
            this.bk = bk;
            this.readOnlyBookieRegPath = conf.getZkAvailableBookiesPath() + "/" + "readonly";
            if (null == bk.getZkHandle().exists(this.readOnlyBookieRegPath, false)) {
                try {
                    List<ACL> zkAcls = ZkUtils.getACLs(conf);
                    bk.getZkHandle().create(this.readOnlyBookieRegPath, new byte[0], zkAcls, CreateMode.PERSISTENT);
                }
                catch (KeeperException.NodeExistsException nodeExistsException) {
                    // empty catch block
                }
            }
        }

        public void process(WatchedEvent event) {
            this.readROBookies();
        }

        void readROBookiesBlocking() throws InterruptedException, KeeperException {
            final LinkedBlockingQueue queue = new LinkedBlockingQueue();
            this.readROBookies(new AsyncCallback.ChildrenCallback(){

                public void processResult(int rc, String path, Object ctx, List<String> children) {
                    try {
                        this.processResult(rc, path, ctx, children);
                        queue.put(rc);
                    }
                    catch (InterruptedException e) {
                        logger.error("Interruped when trying to read readonly bookies in a blocking fashion");
                        throw new RuntimeException(e);
                    }
                }
            });
            int rc = (Integer)queue.take();
            if (rc != KeeperException.Code.OK.intValue()) {
                throw KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc));
            }
        }

        void notifyBookiesChanged(final BookiesListener listener) throws BKException {
            try {
                HashSet newReadOnlyBookies;
                List children = this.bk.getZkHandle().getChildren(this.readOnlyBookieRegPath, new Watcher(){

                    public void process(WatchedEvent event) {
                        if (event.getType() == Watcher.Event.EventType.NodeChildrenChanged) {
                            listener.availableBookiesChanged();
                        }
                    }
                });
                this.readOnlyBookies = newReadOnlyBookies = BookieWatcher.convertToBookieAddresses(children);
            }
            catch (KeeperException ke) {
                logger.error("Error registering watcher with zookeeper", (Throwable)ke);
                throw new BKException.ZKException();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                logger.error("Interrupted registering watcher with zookeeper", (Throwable)ie);
                throw new BKException.BKInterruptedException();
            }
        }

        void readROBookies(AsyncCallback.ChildrenCallback callback) {
            this.bk.getZkHandle().getChildren(this.readOnlyBookieRegPath, (Watcher)this, callback, null);
        }

        void readROBookies() {
            this.readROBookies(this);
        }

        public void processResult(int rc, String path, Object ctx, List<String> children) {
            HashSet newReadOnlyBookies;
            if (rc != KeeperException.Code.OK.intValue()) {
                LOG.error("Not able to read readonly bookies : ", (Throwable)KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc)));
                return;
            }
            this.readOnlyBookies = newReadOnlyBookies = BookieWatcher.convertToBookieAddresses(children);
        }

        public HashSet<BookieSocketAddress> getReadOnlyBookies() {
            return this.readOnlyBookies;
        }
    }
}

