/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package cmd

import (
	"context"
	"errors"
	"fmt"
	"strconv"

	v1 "github.com/apache/camel-k/v2/pkg/apis/camel/v1"
	"github.com/apache/camel-k/v2/pkg/apis/camel/v1alpha1"

	"github.com/apache/camel-k/v2/pkg/client"
	"github.com/apache/camel-k/v2/pkg/util/kubernetes"
	"github.com/spf13/cobra"
	k8errors "k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	k8sclient "sigs.k8s.io/controller-runtime/pkg/client"
)

// newCmdDelete --.
func newCmdDelete(rootCmdOptions *RootCmdOptions) (*cobra.Command, *deleteCmdOptions) {
	options := deleteCmdOptions{
		RootCmdOptions: rootCmdOptions,
	}
	cmd := cobra.Command{
		Use:     "delete [integration1] [integration2] ...",
		Short:   "Delete integrations deployed on Kubernetes",
		PreRunE: decode(&options, options.Flags),
		RunE: func(cmd *cobra.Command, args []string) error {
			if err := options.validate(args); err != nil {
				return err
			}
			return options.run(cmd, args)
		},
	}

	cmd.Flags().Bool("all", false, "Delete all integrations")

	return &cmd, &options
}

type deleteCmdOptions struct {
	*RootCmdOptions
	DeleteAll bool `mapstructure:"all"`
}

func (command *deleteCmdOptions) validate(args []string) error {
	if command.DeleteAll && len(args) > 0 {
		return errors.New("invalid combination: --all flag is set and at least one integration name is provided")
	}
	if !command.DeleteAll && len(args) == 0 {
		return errors.New("invalid combination: provide one or several integration names or set --all flag for all integrations")
	}

	return nil
}

func (command *deleteCmdOptions) run(cmd *cobra.Command, args []string) error {
	c, err := command.GetCmdClient()
	if err != nil {
		return err
	}
	if len(args) != 0 && !command.DeleteAll {
		for _, arg := range args {
			name := kubernetes.SanitizeName(arg)
			integration, err := getIntegration(command.Context, c, name, command.Namespace)
			if err != nil {
				if k8errors.IsNotFound(err) {
					fmt.Fprintln(cmd.OutOrStdout(), "Integration "+name+" not found. Skipped.")
				} else {
					return err
				}
			} else {
				err := deleteIntegration(command.Context, cmd, c, integration)
				if err != nil {
					return err
				}
				fmt.Fprintln(cmd.OutOrStdout(), "Integration "+name+" deleted")
			}
		}
	} else if command.DeleteAll {
		integrationList := v1.IntegrationList{
			TypeMeta: metav1.TypeMeta{
				APIVersion: v1.SchemeGroupVersion.String(),
				Kind:       v1.IntegrationKind,
			},
		}

		err := c.List(command.Context, &integrationList, k8sclient.InNamespace(command.Namespace))
		if err != nil {
			return err
		}
		for i := range integrationList.Items {
			integration := integrationList.Items[i]
			err := deleteIntegration(command.Context, cmd, c, &integration)
			if err != nil {
				return err
			}
		}
		if len(integrationList.Items) == 0 {
			fmt.Fprintln(cmd.OutOrStdout(), "Nothing to delete")
		} else {
			fmt.Fprintln(cmd.OutOrStdout(), strconv.Itoa(len(integrationList.Items))+" integration(s) deleted")
		}
	}

	return nil
}

func getIntegration(ctx context.Context, c client.Client, name string, namespace string) (*v1.Integration, error) {
	key := k8sclient.ObjectKey{
		Name:      name,
		Namespace: namespace,
	}
	answer := v1.NewIntegration(namespace, name)
	if err := c.Get(ctx, key, &answer); err != nil {
		return nil, err
	}
	return &answer, nil
}

func deleteIntegration(ctx context.Context, cmd *cobra.Command, c client.Client, integration *v1.Integration) error {
	deletedPipes, pipe, err := deletePipeIfExists(ctx, c, integration)
	if err != nil {
		return err
	}
	if deletedPipes {
		// Deleting Pipe will automatically clean up the integration
		fmt.Fprintln(cmd.OutOrStdout(), "Pipe "+pipe+" deleted")
		return nil
	}
	deletedKameletBindings, klb, err := deleteKameletBindingIfExists(ctx, c, integration)
	if err != nil {
		return err
	}
	if deletedKameletBindings {
		// Deleting KameletBinding will automatically clean up the integration
		fmt.Fprintln(cmd.OutOrStdout(), "KameletBinding "+klb+" deleted")
		return nil
	}
	return c.Delete(ctx, integration)
}

func deletePipeIfExists(ctx context.Context, c client.Client, integration *v1.Integration) (bool, string, error) {
	kind, name := findCreator(integration)
	if kind != v1.PipeKind || name == "" {
		return false, "", nil
	}

	binding := v1.Pipe{
		TypeMeta: metav1.TypeMeta{
			Kind:       kind,
			APIVersion: v1.SchemeGroupVersion.String(),
		},
		ObjectMeta: metav1.ObjectMeta{
			Namespace: integration.Namespace,
			Name:      name,
		},
	}
	err := c.Delete(ctx, &binding)
	if k8errors.IsNotFound(err) {
		// Simply skip if binding doesn't exist (could be deleted already)
		return false, name, nil
	}
	return err == nil, name, err
}

func deleteKameletBindingIfExists(ctx context.Context, c client.Client, integration *v1.Integration) (bool, string, error) {
	kind, name := findCreator(integration)
	if kind != v1alpha1.KameletBindingKind || name == "" {
		return false, "", nil
	}

	binding := v1alpha1.KameletBinding{
		TypeMeta: metav1.TypeMeta{
			Kind:       kind,
			APIVersion: v1.SchemeGroupVersion.String(),
		},
		ObjectMeta: metav1.ObjectMeta{
			Namespace: integration.Namespace,
			Name:      name,
		},
	}
	err := c.Delete(ctx, &binding)
	if k8errors.IsNotFound(err) {
		// Simply skip if binding doesn't exist (could be deleted already)
		return false, name, nil
	}
	return err == nil, name, err
}

func findCreator(integration *v1.Integration) (string, string) {
	kind := integration.GetLabels()[kubernetes.CamelCreatorLabelKind]
	name := integration.GetLabels()[kubernetes.CamelCreatorLabelName]
	if kind == "" && name == "" {
		// Look up in OwnerReferences in case creator labels are absent
		for _, owner := range integration.GetOwnerReferences() {
			if owner.Kind == v1.PipeKind {
				return owner.Kind, owner.Name
			}
		}
	}
	return kind, name
}
