/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.ingestion.google.webmaster;

import com.google.api.client.googleapis.batch.BatchRequest;
import com.google.api.client.googleapis.batch.json.JsonBatchCallback;
import com.google.api.client.repackaged.com.google.common.base.Preconditions;
import com.google.api.services.webmasters.model.ApiDimensionFilter;
import com.google.api.services.webmasters.model.SearchAnalyticsQueryResponse;
import com.google.common.base.Optional;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.ingestion.google.webmaster.GoogleWebmasterClient;
import org.apache.gobblin.ingestion.google.webmaster.GoogleWebmasterDataFetcher;
import org.apache.gobblin.ingestion.google.webmaster.GoogleWebmasterFilter;
import org.apache.gobblin.ingestion.google.webmaster.ProducerJob;
import org.apache.gobblin.ingestion.google.webmaster.SimpleProducerJob;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.gobblin.util.limiter.RateBasedLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GoogleWebmasterDataFetcherImpl
extends GoogleWebmasterDataFetcher {
    private static final Logger log = LoggerFactory.getLogger(GoogleWebmasterDataFetcherImpl.class);
    private final double API_REQUESTS_PER_SECOND;
    private final RateBasedLimiter LIMITER;
    private final int GET_PAGE_SIZE_TIME_OUT;
    private final int GET_PAGES_RETRIES;
    private final String _siteProperty;
    private final GoogleWebmasterClient _client;
    private final List<ProducerJob> _jobs;

    GoogleWebmasterDataFetcherImpl(String siteProperty, GoogleWebmasterClient client, State wuState) throws IOException {
        this._siteProperty = siteProperty;
        Preconditions.checkArgument((boolean)this._siteProperty.endsWith("/"), (Object)"The site property must end in \"/\"");
        this._client = client;
        this._jobs = GoogleWebmasterDataFetcherImpl.getHotStartJobs(wuState);
        this.API_REQUESTS_PER_SECOND = wuState.getPropAsDouble("source.google_webmasters.request.tuning.get_pages.requests_per_second", 4.5);
        this.GET_PAGE_SIZE_TIME_OUT = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_pages.time_out", 2);
        this.LIMITER = new RateBasedLimiter(this.API_REQUESTS_PER_SECOND, TimeUnit.SECONDS);
        this.GET_PAGES_RETRIES = wuState.getPropAsInt("source.google_webmasters.request.tuning.get_pages.max_retries", 120);
    }

    private static List<ProducerJob> getHotStartJobs(State wuState) {
        String hotStartString = wuState.getProp("source.google_webmasters.request.hot_start", "");
        if (!hotStartString.isEmpty()) {
            return SimpleProducerJob.deserialize(hotStartString);
        }
        return new ArrayList<ProducerJob>();
    }

    @Override
    public Collection<ProducerJob> getAllPages(String startDate, String endDate, String country, int rowLimit) throws IOException {
        log.info("Requested row limit: " + rowLimit);
        if (!this._jobs.isEmpty()) {
            log.info("Service got hot started.");
            return this._jobs;
        }
        ApiDimensionFilter countryFilter = GoogleWebmasterFilter.countryEqFilter(country);
        ArrayList<GoogleWebmasterFilter.Dimension> requestedDimensions = new ArrayList<GoogleWebmasterFilter.Dimension>();
        requestedDimensions.add(GoogleWebmasterFilter.Dimension.PAGE);
        int expectedSize = -1;
        if (rowLimit >= 5000) {
            expectedSize = this.getPagesSize(startDate, endDate, country, requestedDimensions, Arrays.asList(countryFilter));
            log.info(String.format("Expected number of pages is %d for market-%s from %s to %s", expectedSize, GoogleWebmasterFilter.countryFilterToString(countryFilter), startDate, endDate));
        }
        ArrayDeque<Pair<String, GoogleWebmasterFilter.FilterOperator>> jobs = new ArrayDeque<Pair<String, GoogleWebmasterFilter.FilterOperator>>();
        jobs.add(Pair.of((Object)this._siteProperty, (Object)((Object)GoogleWebmasterFilter.FilterOperator.CONTAINS)));
        Collection<String> allPages = this.getPages(startDate, endDate, requestedDimensions, countryFilter, jobs, Math.min(rowLimit, 5000));
        int actualSize = allPages.size();
        log.info(String.format("A total of %d pages fetched for property %s at country-%s from %s to %s", actualSize, this._siteProperty, country, startDate, endDate));
        if (expectedSize != -1 && actualSize != expectedSize) {
            log.warn(String.format("Expected page size is %d, but only able to get %d", expectedSize, actualSize));
        }
        ArrayDeque<ProducerJob> producerJobs = new ArrayDeque<ProducerJob>(actualSize);
        for (String page : allPages) {
            producerJobs.add(new SimpleProducerJob(page, startDate, endDate));
        }
        return producerJobs;
    }

    private int getPagesSize(final String startDate, final String endDate, final String country, final List<GoogleWebmasterFilter.Dimension> requestedDimensions, final List<ApiDimensionFilter> apiDimensionFilters) throws IOException {
        ExecutorService es = Executors.newCachedThreadPool(ExecutorsUtils.newDaemonThreadFactory((Optional)Optional.of((Object)log), (Optional)Optional.of((Object)this.getClass().getSimpleName())));
        int startRow = 0;
        long groupSize = Math.max(1L, Math.round(this.API_REQUESTS_PER_SECOND));
        ArrayList<Future<Integer>> results = new ArrayList<Future<Integer>>((int)groupSize);
        while (true) {
            int i = 0;
            while ((long)i < groupSize) {
                final int n = startRow;
                startRow += 5000;
                Future<Integer> submit = es.submit(new Callable<Integer>(){

                    @Override
                    public Integer call() {
                        log.info(String.format("Getting page size from %s...", n));
                        String interruptedMsg = String.format("Interrupted while trying to get the size of all pages for %s. Current start row is %d.", country, n);
                        while (true) {
                            try {
                                GoogleWebmasterDataFetcherImpl.this.LIMITER.acquirePermits(1L);
                            }
                            catch (InterruptedException e) {
                                log.error("RateBasedLimiter: " + interruptedMsg, (Throwable)e);
                                return -1;
                            }
                            if (Thread.interrupted()) {
                                log.error(interruptedMsg);
                                return -1;
                            }
                            try {
                                List<String> pages = GoogleWebmasterDataFetcherImpl.this._client.getPages(GoogleWebmasterDataFetcherImpl.this._siteProperty, startDate, endDate, country, 5000, requestedDimensions, apiDimensionFilters, n);
                                if (pages.size() < 5000) {
                                    return pages.size() + n;
                                }
                                return -1;
                            }
                            catch (IOException e) {
                                log.info(String.format("Getting page size from %s failed. Retrying...", n));
                                continue;
                            }
                            break;
                        }
                    }
                });
                results.add(submit);
                ++i;
            }
            for (Future future : results) {
                try {
                    Integer integer = (Integer)future.get(this.GET_PAGE_SIZE_TIME_OUT, TimeUnit.MINUTES);
                    if (integer < 0) continue;
                    es.shutdownNow();
                    return integer;
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new RuntimeException(e);
                }
                catch (TimeoutException e) {
                    throw new RuntimeException(String.format("Exceeding the timeout of %d minutes while getting the total size of all pages.", this.GET_PAGE_SIZE_TIME_OUT), e);
                }
            }
            results.clear();
        }
    }

    private Collection<String> getPages(String startDate, String endDate, List<GoogleWebmasterFilter.Dimension> dimensions, ApiDimensionFilter countryFilter, Queue<Pair<String, GoogleWebmasterFilter.FilterOperator>> toProcess, int rowLimit) throws IOException {
        String country = GoogleWebmasterFilter.countryFilterToString(countryFilter);
        ConcurrentLinkedDeque<String> allPages = new ConcurrentLinkedDeque<String>();
        int r = 0;
        while (r <= this.GET_PAGES_RETRIES) {
            log.info(String.format("Get pages at round %d with size %d.", ++r, toProcess.size()));
            ConcurrentLinkedDeque<Pair<String, GoogleWebmasterFilter.FilterOperator>> nextRound = new ConcurrentLinkedDeque<Pair<String, GoogleWebmasterFilter.FilterOperator>>();
            ExecutorService es = Executors.newFixedThreadPool(10, ExecutorsUtils.newDaemonThreadFactory((Optional)Optional.of((Object)log), (Optional)Optional.of((Object)this.getClass().getSimpleName())));
            while (!toProcess.isEmpty()) {
                this.submitJob(toProcess.poll(), countryFilter, startDate, endDate, dimensions, es, allPages, nextRound, rowLimit);
            }
            try {
                es.shutdown();
                boolean terminated = es.awaitTermination(5L, TimeUnit.MINUTES);
                if (!terminated) {
                    es.shutdownNow();
                    log.warn(String.format("Timed out while getting all pages for country-%s at round %d. Next round now has size %d.", country, r, nextRound.size()));
                }
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            if (nextRound.isEmpty()) break;
            toProcess = nextRound;
        }
        if (r == this.GET_PAGES_RETRIES) {
            throw new RuntimeException(String.format("Getting all pages reaches the maximum number of retires %d. Date range: %s ~ %s. Country: %s.", this.GET_PAGES_RETRIES, startDate, endDate, country));
        }
        return allPages;
    }

    private void submitJob(final Pair<String, GoogleWebmasterFilter.FilterOperator> job, final ApiDimensionFilter countryFilter, final String startDate, final String endDate, final List<GoogleWebmasterFilter.Dimension> dimensions, ExecutorService es, final ConcurrentLinkedDeque<String> allPages, final ConcurrentLinkedDeque<Pair<String, GoogleWebmasterFilter.FilterOperator>> nextRound, final int rowLimit) {
        es.submit(new Runnable(){

            @Override
            public void run() {
                List<String> pages;
                try {
                    GoogleWebmasterDataFetcherImpl.this.LIMITER.acquirePermits(1L);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException("RateBasedLimiter got interrupted.", e);
                }
                String countryString = GoogleWebmasterFilter.countryFilterToString(countryFilter);
                LinkedList<ApiDimensionFilter> filters = new LinkedList<ApiDimensionFilter>();
                filters.add(countryFilter);
                String prefix = (String)job.getLeft();
                GoogleWebmasterFilter.FilterOperator operator = (GoogleWebmasterFilter.FilterOperator)((Object)job.getRight());
                String jobString = String.format("job(prefix: %s, operator: %s)", new Object[]{prefix, operator});
                filters.add(GoogleWebmasterFilter.pageFilter(operator, prefix));
                try {
                    pages = GoogleWebmasterDataFetcherImpl.this._client.getPages(GoogleWebmasterDataFetcherImpl.this._siteProperty, startDate, endDate, countryString, rowLimit, dimensions, filters, 0);
                    log.debug(String.format("%d pages fetched for %s market-%s from %s to %s.", pages.size(), jobString, countryString, startDate, endDate));
                }
                catch (IOException e) {
                    log.debug(String.format("%s failed due to %s. Retrying...", jobString, e.getMessage()));
                    nextRound.add(job);
                    return;
                }
                if (pages.size() == 5000) {
                    log.info(String.format("Expanding the prefix '%s'", prefix));
                    nextRound.add(Pair.of((Object)prefix, (Object)((Object)GoogleWebmasterFilter.FilterOperator.EQUALS)));
                    for (String expanded : GoogleWebmasterDataFetcherImpl.this.getUrlPartitions(prefix)) {
                        nextRound.add(Pair.of((Object)expanded, (Object)((Object)GoogleWebmasterFilter.FilterOperator.CONTAINS)));
                    }
                } else {
                    allPages.addAll(pages);
                }
            }
        });
    }

    private ArrayList<String> getUrlPartitions(String prefix) {
        ArrayList<String> expanded = new ArrayList<String>();
        for (char c = 'a'; c <= 'z'; c = (char)(c + '\u0001')) {
            expanded.add(prefix + c);
        }
        for (int num = 0; num <= 9; ++num) {
            expanded.add(prefix + num);
        }
        expanded.add(prefix + "-");
        expanded.add(prefix + ".");
        expanded.add(prefix + "_");
        expanded.add(prefix + "~");
        expanded.add(prefix + "/");
        expanded.add(prefix + "%");
        expanded.add(prefix + ":");
        expanded.add(prefix + "?");
        expanded.add(prefix + "#");
        expanded.add(prefix + "@");
        expanded.add(prefix + "!");
        expanded.add(prefix + "$");
        expanded.add(prefix + "&");
        expanded.add(prefix + "+");
        expanded.add(prefix + "*");
        expanded.add(prefix + "'");
        expanded.add(prefix + "=");
        return expanded;
    }

    @Override
    public List<String[]> performSearchAnalyticsQuery(String startDate, String endDate, int rowLimit, List<GoogleWebmasterFilter.Dimension> requestedDimensions, List<GoogleWebmasterDataFetcher.Metric> requestedMetrics, Collection<ApiDimensionFilter> filters) throws IOException {
        SearchAnalyticsQueryResponse response = (SearchAnalyticsQueryResponse)this._client.createSearchAnalyticsQuery(this._siteProperty, startDate, endDate, requestedDimensions, GoogleWebmasterFilter.andGroupFilters(filters), rowLimit, 0).execute();
        return GoogleWebmasterDataFetcherImpl.convertResponse(requestedMetrics, response);
    }

    @Override
    public void performSearchAnalyticsQueryInBatch(List<ProducerJob> jobs, List<ArrayList<ApiDimensionFilter>> filterList, List<JsonBatchCallback<SearchAnalyticsQueryResponse>> callbackList, List<GoogleWebmasterFilter.Dimension> requestedDimensions, int rowLimit) throws IOException {
        BatchRequest batchRequest = this._client.createBatch();
        for (int i = 0; i < jobs.size(); ++i) {
            ProducerJob job = jobs.get(i);
            ArrayList<ApiDimensionFilter> filters = filterList.get(i);
            JsonBatchCallback<SearchAnalyticsQueryResponse> callback = callbackList.get(i);
            this._client.createSearchAnalyticsQuery(this._siteProperty, job.getStartDate(), job.getEndDate(), requestedDimensions, GoogleWebmasterFilter.andGroupFilters(filters), rowLimit, 0).queue(batchRequest, callback);
        }
        batchRequest.execute();
    }

    @Override
    public String getSiteProperty() {
        return this._siteProperty;
    }
}

