/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.consensus.ratis;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import org.apache.iotdb.common.rpc.thrift.TConsensusGroupType;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.service.metric.PerformanceOverviewMetrics;
import org.apache.iotdb.consensus.IStateMachine;
import org.apache.iotdb.consensus.common.DataSet;
import org.apache.iotdb.consensus.common.request.ByteBufferConsensusRequest;
import org.apache.iotdb.consensus.common.request.IConsensusRequest;
import org.apache.iotdb.consensus.ratis.RequestMessage;
import org.apache.iotdb.consensus.ratis.ResponseMessage;
import org.apache.iotdb.consensus.ratis.SnapshotStorage;
import org.apache.iotdb.consensus.ratis.metrics.RatisMetricsManager;
import org.apache.iotdb.consensus.ratis.utils.Retriable;
import org.apache.iotdb.consensus.ratis.utils.Utils;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.StateMachineStorage;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.TimeDuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApplicationStateMachineProxy
extends BaseStateMachine {
    private static final Logger logger = LoggerFactory.getLogger(ApplicationStateMachineProxy.class);
    private static final PerformanceOverviewMetrics PERFORMANCE_OVERVIEW_METRICS = PerformanceOverviewMetrics.getInstance();
    private final IStateMachine applicationStateMachine;
    private final SnapshotStorage snapshotStorage;
    private final RaftGroupId groupId;
    private final TConsensusGroupType consensusGroupType;
    private final BiConsumer<RaftGroupMemberId, RaftPeerId> leaderChangeListener;

    ApplicationStateMachineProxy(IStateMachine stateMachine, RaftGroupId id) {
        this(stateMachine, id, null);
    }

    ApplicationStateMachineProxy(IStateMachine stateMachine, RaftGroupId id, BiConsumer<RaftGroupMemberId, RaftPeerId> onLeaderChanged) {
        this.applicationStateMachine = stateMachine;
        this.leaderChangeListener = onLeaderChanged;
        this.groupId = id;
        this.snapshotStorage = new SnapshotStorage(this.applicationStateMachine, this.groupId);
        this.consensusGroupType = Utils.getConsensusGroupTypeFromPrefix(this.groupId.toString());
        this.applicationStateMachine.start();
    }

    public void initialize(RaftServer raftServer, RaftGroupId raftGroupId, RaftStorage storage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            this.snapshotStorage.init(storage);
            this.loadSnapshot(this.snapshotStorage.findLatestSnapshotDir());
        }, new Class[0]);
    }

    public void reinitialize() {
        this.setLastAppliedTermIndex(null);
        this.loadSnapshot(this.snapshotStorage.findLatestSnapshotDir());
        if (this.getLifeCycleState() == LifeCycle.State.PAUSED) {
            this.getLifeCycle().transition(LifeCycle.State.STARTING);
            this.getLifeCycle().transition(LifeCycle.State.RUNNING);
        }
    }

    public void pause() {
        if (this.getLifeCycleState() == LifeCycle.State.RUNNING) {
            this.getLifeCycle().transition(LifeCycle.State.PAUSING);
            this.getLifeCycle().transition(LifeCycle.State.PAUSED);
        }
    }

    public void close() throws IOException {
        this.getLifeCycle().checkStateAndClose(this.applicationStateMachine::stop);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        ResponseMessage ret;
        IConsensusRequest applicationRequest;
        boolean isLeader = false;
        long writeToStateMachineStartTime = System.nanoTime();
        RaftProtos.LogEntryProto log = trx.getLogEntry();
        this.updateLastAppliedTermIndex(log.getTerm(), log.getIndex());
        if (trx.getClientRequest() != null && trx.getClientRequest().getMessage() instanceof RequestMessage) {
            RequestMessage requestMessage = (RequestMessage)trx.getClientRequest().getMessage();
            applicationRequest = requestMessage.getActualRequest();
            isLeader = true;
        } else {
            applicationRequest = new ByteBufferConsensusRequest(log.getStateMachineLogEntry().getLogData().asReadOnlyByteBuffer());
        }
        IConsensusRequest deserializedRequest = this.applicationStateMachine.deserializeRequest(applicationRequest);
        this.waitUntilSystemAllowApply();
        while (true) {
            try {
                if (!isLeader) {
                    deserializedRequest.markAsGeneratedByRemoteConsensusLeader();
                }
                TSStatus result = this.applicationStateMachine.write(deserializedRequest);
                ret = new ResponseMessage(result);
            }
            catch (Throwable rte) {
                logger.error("application statemachine throws a runtime exception: ", rte);
                ret = new ResponseMessage(new TSStatus(TSStatusCode.INTERNAL_SERVER_ERROR.getStatusCode()).setMessage("internal error. statemachine throws a runtime exception: " + rte));
                if (!Utils.stallApply(this.consensusGroupType)) break;
                this.waitUntilSystemAllowApply();
                if (Utils.stallApply(this.consensusGroupType)) continue;
            }
            break;
        }
        if (isLeader) {
            if (this.consensusGroupType == TConsensusGroupType.DataRegion) {
                PERFORMANCE_OVERVIEW_METRICS.recordEngineCost(System.nanoTime() - writeToStateMachineStartTime);
            }
            RatisMetricsManager.getInstance().recordWriteStateMachineCost(System.nanoTime() - writeToStateMachineStartTime, this.consensusGroupType);
        }
        return CompletableFuture.completedFuture(ret);
    }

    private void waitUntilSystemAllowApply() {
        try {
            Retriable.attemptUntilTrue(() -> !Utils.stallApply(this.consensusGroupType), TimeDuration.ONE_MINUTE, "waitUntilSystemAllowApply", logger);
        }
        catch (InterruptedException e) {
            logger.warn("{}: interrupted when waiting until system ready: ", (Object)this, (Object)e);
            Thread.currentThread().interrupt();
        }
    }

    public CompletableFuture<Message> query(Message request) {
        if (!(request instanceof RequestMessage)) {
            logger.error("An RequestMessage is required but got {}", (Object)request);
            return CompletableFuture.completedFuture(new ResponseMessage(null));
        }
        RequestMessage requestMessage = (RequestMessage)request;
        DataSet result = this.applicationStateMachine.read(requestMessage.getActualRequest());
        return CompletableFuture.completedFuture(new ResponseMessage(result));
    }

    public long takeSnapshot() throws IOException {
        TermIndex lastApplied = this.getLastAppliedTermIndex();
        if (lastApplied.getTerm() <= 0L || lastApplied.getIndex() <= 0L) {
            return -1L;
        }
        String metadata = Utils.getMetadataFromTermIndex(lastApplied);
        File snapshotTmpDir = this.snapshotStorage.getSnapshotTmpDir(metadata);
        FileUtils.deleteFully((File)snapshotTmpDir);
        snapshotTmpDir.mkdirs();
        if (!snapshotTmpDir.isDirectory()) {
            logger.error("Unable to create temp snapshotDir at {}", (Object)snapshotTmpDir);
            return -1L;
        }
        boolean applicationTakeSnapshotSuccess = this.applicationStateMachine.takeSnapshot(snapshotTmpDir, this.snapshotStorage.getSnapshotTmpId(metadata), metadata);
        if (!applicationTakeSnapshotSuccess) {
            this.deleteIncompleteSnapshot(snapshotTmpDir);
            return -1L;
        }
        File snapshotDir = this.snapshotStorage.getSnapshotDir(metadata);
        FileUtils.deleteFully((File)snapshotDir);
        try {
            Files.move(snapshotTmpDir.toPath(), snapshotDir.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            logger.error("{} atomic rename {} to {} failed with exception {}", new Object[]{this, snapshotTmpDir, snapshotDir, e});
            this.deleteIncompleteSnapshot(snapshotTmpDir);
            return -1L;
        }
        this.snapshotStorage.updateSnapshotCache();
        return lastApplied.getIndex();
    }

    private void deleteIncompleteSnapshot(File snapshotDir) throws IOException {
        boolean isEmpty = snapshotDir.delete();
        if (!isEmpty) {
            logger.info("Snapshot directory is incomplete, deleting {}", (Object)snapshotDir.getAbsolutePath());
            FileUtils.deleteFully((File)snapshotDir);
        }
    }

    private void loadSnapshot(File latestSnapshotDir) {
        this.snapshotStorage.updateSnapshotCache();
        if (latestSnapshotDir == null) {
            return;
        }
        this.applicationStateMachine.loadSnapshot(latestSnapshotDir);
        TermIndex snapshotTermIndex = Utils.getTermIndexFromDir(latestSnapshotDir);
        this.updateLastAppliedTermIndex(snapshotTermIndex.getTerm(), snapshotTermIndex.getIndex());
    }

    public StateMachineStorage getStateMachineStorage() {
        return this.snapshotStorage;
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId newLeaderId) {
        this.leaderChangeListener.accept(groupMemberId, newLeaderId);
        this.applicationStateMachine.event().notifyLeaderChanged(Utils.fromRaftGroupIdToConsensusGroupId(groupMemberId.getGroupId()), Utils.fromRaftPeerIdToNodeId(newLeaderId));
    }

    public void notifyLeaderReady() {
        this.applicationStateMachine.event().notifyLeaderReady();
    }

    public void notifyNotLeader(Collection<TransactionContext> pendingEntries) throws IOException {
        this.applicationStateMachine.event().notifyNotLeader();
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
        this.applicationStateMachine.event().notifyConfigurationChanged(term, index, Utils.fromRaftProtoListAndRaftGroupIdToPeers(newRaftConfiguration.getPeersList(), this.groupId));
    }
}

