/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.portlet.faces.util.map;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.portlet.ClientDataRequest;
import javax.portlet.PortletRequest;
import javax.portlet.faces.Bridge;

public class PortletRequestHeaders
{
  private PortletRequest mPortletRequest = null;
  private List<String> mHeaderNames = null;
  private Map<String, List<String>> mHeaders = null;

  public PortletRequestHeaders(PortletRequest request)
  {
    mPortletRequest = request;
  }

  @SuppressWarnings("unchecked")
  public String getHeader(String name)
  {
    // initialize the header map if it hasn't been already
    initHeaderMap();

    List<String> headerVals = mHeaders.get(name.toUpperCase());
    return headerVals == null? null: (String) headerVals.get(0);
  }

  public Enumeration<String> getHeaders(String name)
  {
    // initialize the header map if it hasn't been already
    initHeaderMap();

    List<String> headerVals = mHeaders.get(name.toUpperCase());

    if (headerVals == null)
    {
      headerVals = Collections.emptyList();
    }

    return Collections.enumeration(headerVals);
  }

  public Enumeration<String> getHeaderNames()
  {
    // initialize the header map if it hasn't been already
    initHeaderMap();

    return Collections.enumeration(mHeaderNames);
  }

  /**
   * Does 'lazy' initialization of Map of 'properties', i.e. mime headers.
   */
   @SuppressWarnings("unchecked")
   protected boolean initHeaderMap()
   {
     if (mHeaders != null)
     {
       return false;
     }

     mHeaders = Collections.emptyMap();
     mHeaderNames = Collections.emptyList();

     Enumeration<String> props = mPortletRequest.getPropertyNames();
     while (props.hasMoreElements())
     {
       String name = props.nextElement();
       Enumeration<String> values = mPortletRequest.getProperties(name);
       while (values != null && values.hasMoreElements())
       {
         addProperty(name, values.nextElement());
       }
     }

     StringBuilder property = null;

     // can't assume portlet container overrides these headers to reflect portlet constraints -- so do so
     ensurePortletAcceptHeader();
     ensurePortletAcceptLanguage();
     
     switch ((Bridge.PortletPhase) mPortletRequest.getAttribute(Bridge.PORTLET_LIFECYCLE_PHASE))
     {
     case ACTION_PHASE:
     case RESOURCE_PHASE:
        ensurePortletContentType();
        ensurePortletContentLength();
        break;
     case RENDER_PHASE:
     case EVENT_PHASE:
       // its the RENDER_PHASE -- spec says we must remove the CONTENT_TYPE and CONTENT-LENGTH if
       // came in the request -- so it matches null return from
       // EC.getRequestContentType/CharacterSetEncoding
       mHeaders.remove("CONTENT-TYPE");
       mHeaderNames.remove("CONTENT-TYPE");
       mHeaders.remove("CONTENT-LENGTH");
       mHeaderNames.remove("CONTENT-LENGTH");
     default:
       // shouldn't get here
     }

     return true;
   }

   private boolean containsHeader(List<String> headerNames, String key)
   {
     for (String name : headerNames)
     {
       if (key.toUpperCase().equals(name.toUpperCase()))
       {
         return true;
       }
     }
     
     return false;
   }
   
   private void ensurePortletAcceptHeader()
   {
     // Make sure its not there
     mHeaders.remove("ACCEPT");
     mHeaderNames.remove("ACCEPT");
     
     Enumeration<String> contentTypes =
       mPortletRequest.getResponseContentTypes();
     StringBuilder property = new StringBuilder(64);

     boolean addComma = false;
     while (contentTypes.hasMoreElements())
     {
       String type = contentTypes.nextElement();
       if (type != null)
       {
         if (addComma)
         {
           property = property.append(',');
         }
         else
         {
           addComma = true;
         }

         property = property.append(type);
       }
     }

     if (addComma)
     {
       addProperty("ACCEPT", property.toString());
     }
   }

   private void ensurePortletAcceptLanguage()
   {
     // Make sure its not there
     mHeaders.remove("ACCEPT-LANGUAGE");
     mHeaderNames.remove("ACCEPT-LANGUAGE");
     
     StringBuilder property = new StringBuilder(64);
     Enumeration<Locale> locales = mPortletRequest.getLocales();

     boolean addComma = false;
     while (locales.hasMoreElements())
     {
       Locale l = locales.nextElement();
       if (l != null)
       {
         if (addComma)
         {
           property = property.append(',');
         }
         else
         {
           addComma = true;
         }

         String s = l.getLanguage();
         // only add if language not empty
         if (s.length() > 0)
         {
           property = property.append(s);
           s = l.getCountry();
           if (s.length() > 0)
           {
             property = property.append('-');
             property = property.append(s);
           }
         }
       }
     }

     if (addComma)
     {
       addProperty("ACCEPT-LANGUAGE", property.toString());
     }
   }
   
   private void ensurePortletContentType()
   {
     // Make sure its not there
     mHeaders.remove("CONTENT-TYPE");
     mHeaderNames.remove("CONTENT-TYPE");
     
     StringBuilder property = new StringBuilder(64);
     String contentType =
       ((ClientDataRequest) mPortletRequest).getContentType();
     String charset =
       ((ClientDataRequest) mPortletRequest).getCharacterEncoding();

     if (contentType != null)
     {
       if (charset != null)
       {
         // remove existing charset if its there -- it might be incorrect in a wsrp world
         int index = contentType.indexOf(";");
         if (index < 0)
         {
            property = property.append(contentType);
         }
         else
         {
           property = property.append(contentType, 0, index);
         }
         property = property.append("; charset=");
         property = property.append(charset);
       }
       
       addProperty("CONTENT-TYPE", property.toString());
     }
   }

   private void ensurePortletContentLength()
   {
     // Make sure its not there
     mHeaders.remove("CONTENT-LENGTH");
     mHeaderNames.remove("CONTENT-LENGTH");
     
     int contentLength =
       ((ClientDataRequest) mPortletRequest).getContentLength();

     if (contentLength != -1)
     {
       addProperty("CONTENT-LENGTH", String.valueOf(contentLength));
     }
   }
   
   
  protected final void addProperty(String name, String value)
  {
    if (mHeaders == Collections.EMPTY_MAP)
    {
      mHeaders = new HashMap<String, List<String>>(40);
      mHeaderNames = new ArrayList<String>(30);
    }
    // Store against UPPER CASE key to make case-insensitive
    String upperName = name.toUpperCase();
    List<String> propertyList = mHeaders.get(upperName);
    if (propertyList == null)
    {
      propertyList = new ArrayList<String>(4);
      mHeaders.put(upperName, propertyList);
      mHeaderNames.add(upperName);
    }
    propertyList.add(value);
  }
}
